
#include "xplore.h"
#include "dialogs.h"

#include "callbacks.h"
#include "devmount.h"
#include "icons.h"
#include "interface.h"
#include "util.h"

#include <Xm/XmAll.h>

/* dialog types */

enum {
  QUESTION, INFO, WARNING, ERRORMSG,
};

/* message strings */

#define CONFIRM_DROP		"Drop %d file(s)\nfrom %s\nonto %s?"
#define CONFIRM_MOVE		"Move %d file(s)\nfrom %s\nto %s?"
#define CONFIRM_COPY		"Copy %d file(s)\nfrom %s\nto %s?"
#define CONFIRM_LINK		"Link %d file(s)\nfrom %s\nto %s?"
#define CONFIRM_DELETE		"Delete %d file(s)\nfrom %s?"
#define CONFIRM_OVERWRITE_DIR	"Directory %s\nalready exists. Overwrite?"
#define CONFIRM_OVERWRITE_FILE	"File %s\nalready exists. Overwrite?"
#define CONFIRM_RENAME_DIR	"Directory %s\nalready exists. Rename?"
#define CONFIRM_RENAME_FILE	"File %s\nalready exists. Rename?"
#define CONFIRM_DELETE_DIR	"Delete directory\n%s\n" \
                                "and all files contained in it?"
#define CONFIRM_DELETE_FILE	"Delete file\n%s?"
#define CONFIRM_QUIT		"Quit xplore?"
#define MOUNT_RETRY		"Directory %s\ncould not be mounted. Retry?"
#define UMOUNT_RETRY		"Directory %s\ncould not be unmounted. Retry?"
#define MKDIR_MSG		"Make a new directory"
#define MKDIR_PROMPT		"Directory name:"
#define CHDIR_MSG		"Change to directory"
#define CHDIR_PROMPT		"Directory name:"
#define RENAME_MSG		"Rename file"
#define RENAME_PROMPT		"New name:"
#define MOVE_MSG		"Move files"
#define MOVE_PROMPT		"Target:"
#define COPY_MSG		"Copy files"
#define COPY_PROMPT		"Target:"
#define LINK_MSG		"Link files"
#define LINK_PROMPT		"Target:"
#define FILTER_MSG		"Filter files"
#define FILTER_PROMPT		"Pattern:"
#define SELECT_MSG		"Select files"
#define SELECT_PROMPT		"Pattern:"
#define PARAM_MSG		"Enter parameters"
#define PROPS_MSG		"Properties"
#define PROPS_FILE_INFO		"File info"
#define PROPS_PERMS		"Permissions"
#define PROPS_SUMMARY		"%s byte%s in %s item%s"
#define PROPS_DIR		"Directory:"
#define PROPS_FILE		"Filename:"
#define PROPS_COMMENT		"Comment:"
#define PROPS_DESC		"Type:"
#define PROPS_LINK		"Link to:"
#define PROPS_LAST_ACC		"Last access:"
#define PROPS_LAST_MOD		"Last modification:"
#define PROPS_LAST_CHG		"Last status change:"
#define PROPS_USER		"User:"
#define PROPS_GROUP		"Group:"
#define PROPS_OTHERS		"Others:"

/* button masks */

#define Ok		(1<<(OK-1))
#define Cancel		(1<<(CANCEL-1))
#define Yes		(1<<(YES-1))
#define No		(1<<(NO-1))
#define Doall		(1<<(DOALL-1))
#define Ignore		(1<<(IGNORE-1))
#define Replace		(1<<(REPLACE-1))
#define Add		(1<<(ADD-1))
#define Remove		(1<<(REMOVE-1))
#define Clear		(1<<(CLEAR-1))

/* dialog button return value */

static int button_return;

/* generic dialog creation routine */

static void DestroyPixmapCB(Widget w, Pixmap pixmap)
{
  XFreePixmap(display, pixmap);
}

static Widget create_dlg(String name, IconPtr icon, String msg, int type)
{
  XmString message = msg?XmStringCreateLtoR(msg, XmFONTLIST_DEFAULT_TAG):NULL;
  Arg args[3];
  int i = 0;
  Widget dlg;

  XtSetArg(args[i], XmNautoUnmanage, True); i++;
  XtSetArg(args[i], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); i++;
  if (message) {
    XtSetArg(args[i], XmNmessageString, message); i++;
  }
  switch (type) {
  case QUESTION:
    dlg = XmCreateQuestionDialog(app_shell, name, args, i);
    break;
  case INFO:
    dlg = XmCreateInformationDialog(app_shell, name, args, i);
    break;
  case WARNING:
    dlg = XmCreateWarningDialog(app_shell, name, args, i);
    break;
  case ERRORMSG:
    dlg = XmCreateErrorDialog(app_shell, name, args, i);
    break;
  default:
    /* should not happen */
    fatal("unknown dialog type (create_dlg)");
  }
  if (icon) {
    Pixel background;
    Pixmap pixmap;
    Widget symbol;

    XtVaGetValues(dlg, XmNbackground, &background, NULL);
    pixmap = combine_pixmap(icon, background);
    symbol = XmMessageBoxGetChild(dlg, XmDIALOG_SYMBOL_LABEL);
    XtVaSetValues(symbol, XmNlabelPixmap, pixmap, NULL);
    XtAddCallback(symbol, XmNdestroyCallback, (XtCallbackProc)DestroyPixmapCB,
		  (XtPointer)pixmap);
  }
  return dlg;
}

/* create the buttons */

static Widget button_widgets[N_BUTTONS];
static XmString button_labels[N_BUTTONS];
static String button_strings[N_BUTTONS] = {
  NULL, NULL, NULL, "Yes", "No", "All", "Ignore", "Replace", "Add",
  "Remove", "Clear",
};

static void button_cb(Widget w, int button, XtPointer widget_data)
{
  button_return = button;
}

static void create_buttons(Widget dlg, int type, int buttons)
{
  int i;

  /* when invoked for the first time, initialize the button labels */
  if (button_labels[YES] == NULL)
    for (i = YES; i < N_BUTTONS; i++)
      button_labels[i] = XmStringCreateLocalized(button_strings[i]);
  /* now initialize the requested button widgets */
  /* standard buttons */
  button_widgets[OK] = XmMessageBoxGetChild(dlg, XmDIALOG_OK_BUTTON);
  button_widgets[CANCEL] = XmMessageBoxGetChild(dlg, XmDIALOG_CANCEL_BUTTON);
  XtUnmanageChild(XmMessageBoxGetChild(dlg, XmDIALOG_HELP_BUTTON));
  if (buttons & Ok)
    XtAddCallback(dlg, XmNokCallback, (XtCallbackProc) button_cb,
		  (XtPointer)OK);
  else
    XtUnmanageChild(button_widgets[OK]);
  if (buttons & Cancel)
    XtAddCallback(dlg, XmNcancelCallback, (XtCallbackProc) button_cb,
		  (XtPointer)CANCEL);
  else
    XtUnmanageChild(button_widgets[CANCEL]);
  /* our own special buttons */
  for (i = YES; i < N_BUTTONS; i++)
    if (buttons & (1<<(i-1))) {
      button_widgets[i] =
	XtVaCreateManagedWidget(button_strings[i], xmPushButtonGadgetClass,
				dlg, XmNlabelString, button_labels[i], NULL);
      XtAddCallback(button_widgets[i], XmNactivateCallback,
		    (XtCallbackProc) button_cb, (XtPointer)i);
    } else
      button_widgets[i] = NULL;
  if (!(buttons & Ok))
    for (i = YES; i < N_BUTTONS; i++)
      if (button_widgets[i]) {
	XtVaSetValues(dlg, XmNdefaultButton, button_widgets[i], NULL);
	break;
      }
}

/* field descriptions */

typedef struct _FieldDescRec {
  int id;
  String label, value;
  String *buttons;
  unsigned long flags, mask;
  int n;
} FieldDescRec;

enum {
  ID_SEPARATOR, ID_LABEL, ID_CONST_FIELD, ID_EDIT_FIELD, ID_FLAGS_FIELD,
};

#define SEPARATOR { ID_SEPARATOR, NULL, NULL, NULL, 0L, 0L, 0 }
#define LABEL(label) { ID_LABEL, label, NULL, NULL, 0L, 0L, 0 }
#define CONST_FIELD(label,value) \
  { ID_CONST_FIELD, label, value, NULL, 0L, 0L, 0 }
#define EDIT_FIELD(label,value) \
  { ID_EDIT_FIELD, label, value, NULL, 0L, 0L, 0 }
#define FLAGS_FIELD(label,buttons,flags,mask,n) \
  { ID_FLAGS_FIELD, label, NULL, buttons, flags, mask, n }

FieldDescRec mkdir_fld_desc[] = {
  EDIT_FIELD(MKDIR_PROMPT, NULL),
};

FieldDescRec chdir_fld_desc[] = {
  EDIT_FIELD(CHDIR_PROMPT, NULL),
};

FieldDescRec rename_fld_desc[] = {
  EDIT_FIELD(RENAME_PROMPT, NULL),
};

FieldDescRec move_fld_desc[] = {
  EDIT_FIELD(MOVE_PROMPT, NULL),
};

FieldDescRec copy_fld_desc[] = {
  EDIT_FIELD(COPY_PROMPT, NULL),
};

FieldDescRec link_fld_desc[] = {
  EDIT_FIELD(LINK_PROMPT, NULL),
};

FieldDescRec filter_fld_desc[] = {
  EDIT_FIELD(FILTER_PROMPT, NULL),
};

FieldDescRec select_fld_desc[] = {
  EDIT_FIELD(SELECT_PROMPT, NULL),
};

String perm_buttons[] = { "read", "write", "execute" };

FieldDescRec props_fld_desc[] = {
  SEPARATOR,
  LABEL(PROPS_FILE_INFO),
  SEPARATOR,
  LABEL(NULL),
  CONST_FIELD(PROPS_DIR, NULL),
  CONST_FIELD(PROPS_FILE, NULL),
  CONST_FIELD(PROPS_DESC, NULL),
  CONST_FIELD(PROPS_LINK, NULL),
  CONST_FIELD(PROPS_COMMENT, NULL),
  CONST_FIELD(PROPS_LAST_ACC, NULL),
  CONST_FIELD(PROPS_LAST_MOD, NULL),
  CONST_FIELD(PROPS_LAST_CHG, NULL),
  EDIT_FIELD(PROPS_USER, NULL),
  EDIT_FIELD(PROPS_GROUP, NULL),
  SEPARATOR,
  LABEL(PROPS_PERMS),
  SEPARATOR,
  FLAGS_FIELD(PROPS_USER, perm_buttons, 0L, 0L, XtNumber(perm_buttons)),
  FLAGS_FIELD(PROPS_GROUP, perm_buttons, 0L, 0L, XtNumber(perm_buttons)),
  FLAGS_FIELD(PROPS_OTHERS, perm_buttons, 0L, 0L, XtNumber(perm_buttons)),
};

/* create a collection of dialog fields */

static WidgetList label_widgets, value_widgets, *flag_widgets;

static void create_fields(Widget dlg, FieldDescRec *fld_desc, int n)
{
  if (n > 0) {
    int i;
    Dimension max_wd = 0;
    Widget rowcol =
      XtVaCreateManagedWidget("Fields", xmRowColumnWidgetClass, dlg, NULL);

    label_widgets = (WidgetList)MALLOC(n * sizeof(Widget));
    value_widgets = (WidgetList)MALLOC(n * sizeof(Widget));
    flag_widgets = (WidgetList*)MALLOC(n * sizeof(WidgetList));
    for (i = 0; i < n; i++)
      switch (fld_desc[i].id) {
      case ID_SEPARATOR:
	XtManageChild(XmCreateSeparator(rowcol, "Separator", NULL, 0));
	label_widgets[i] = value_widgets[i] = NULL;
	flag_widgets[i] = NULL;
	break;
      case ID_LABEL:
	if (fld_desc[i].label) {
	  XmString labelstr = XmStringCreateLocalized(fld_desc[i].label);
	  label_widgets[i] =
	    XtVaCreateManagedWidget("Label", xmLabelGadgetClass,
				    rowcol, XmNlabelType, XmSTRING,
				    XmNlabelString, labelstr,
				    NULL);
	  XmStringFree(labelstr);
	} else
	  label_widgets[i] = NULL;
	value_widgets[i] = NULL;
	flag_widgets[i] = NULL;
	break;
      case ID_CONST_FIELD:
      case ID_EDIT_FIELD:
      case ID_FLAGS_FIELD:
	if (fld_desc[i].label &&
	    ((fld_desc[i].id == ID_FLAGS_FIELD) ? (fld_desc[i].buttons &&
	      fld_desc[i].n) : (fld_desc[i].value != NULL))) {
	  Widget form = XtVaCreateManagedWidget("Line", xmFormWidgetClass,
						rowcol, NULL);
	  Dimension wd;
	  XmString labelstr = XmStringCreateLocalized(fld_desc[i].label);

	  label_widgets[i] =
	    XtVaCreateManagedWidget("Label", xmLabelGadgetClass,
				    form, XmNlabelType, XmSTRING,
				    XmNlabelString, labelstr,
				    XmNtopAttachment, XmATTACH_FORM,
				    XmNbottomAttachment, XmATTACH_FORM,
				    XmNleftAttachment, XmATTACH_FORM,
				    NULL);
	  XmStringFree(labelstr);
	  XtVaGetValues(label_widgets[i], XmNwidth, &wd, NULL);
	  if (wd > max_wd) max_wd = wd;
	  if (fld_desc[i].id == ID_EDIT_FIELD) {
	    value_widgets[i] =
	      XtVaCreateManagedWidget("Input", xmTextFieldWidgetClass, form,
				      XmNtopAttachment, XmATTACH_FORM,
				      XmNbottomAttachment, XmATTACH_FORM,
				      XmNleftAttachment, XmATTACH_WIDGET,
				      XmNleftWidget, label_widgets[i], NULL);
	    XmTextFieldSetString(value_widgets[i], fld_desc[i].value);
	    flag_widgets[i] = NULL;
	  } else if (fld_desc[i].id == ID_CONST_FIELD) {
	    labelstr = XmStringCreateLocalized(fld_desc[i].value);
	    value_widgets[i] =
	      XtVaCreateManagedWidget("Value", xmLabelGadgetClass,
				      form, XmNlabelType, XmSTRING,
				      XmNlabelString, labelstr,
				      XmNtopAttachment, XmATTACH_FORM,
				      XmNbottomAttachment, XmATTACH_FORM,
				      XmNleftAttachment, XmATTACH_WIDGET,
				      XmNleftWidget, label_widgets[i], NULL);
	    XmStringFree(labelstr);
	    flag_widgets[i] = NULL;
	  } else {
	    Widget prev = label_widgets[i];
	    int j;

	    flag_widgets[i] =
	      (WidgetList) MALLOC(fld_desc[i].n * sizeof(Widget));
	    for (j = 0; j < fld_desc[i].n; j++) {
	      XmString labelstr =
		XmStringCreateLocalized(fld_desc[i].buttons[j]);

	      flag_widgets[i][j] =
		XtVaCreateManagedWidget("Button", xmToggleButtonGadgetClass,
					form,
					XmNlabelString, labelstr,
					XmNtoggleMode, XmTOGGLE_INDETERMINATE,
					XmNindicatorType, XmN_OF_MANY,
					XmNset,
					(!(fld_desc[i].mask & (1L<<j)))?
					XmINDETERMINATE:
					(fld_desc[i].flags & (1L<<j))?
					XmSET:XmUNSET,
					XmNtopAttachment, XmATTACH_FORM,
					XmNbottomAttachment, XmATTACH_FORM,
					XmNleftAttachment, XmATTACH_WIDGET,
					XmNleftWidget, prev,
 					NULL);
	      XmStringFree(labelstr);
	      prev = flag_widgets[i][j];
	    }
	    value_widgets[i] = NULL;
	  }
	} else {
	  label_widgets[i] = value_widgets[i] = NULL;
	  flag_widgets[i] = NULL;
	}
	break;
      default:
	/* shouldn't happen */
	fatal("invalid dialog description");
      }
    for (i = 0; i < n; i++)
      if (value_widgets[i] || flag_widgets[i])
	XtVaSetValues(label_widgets[i], XmNwidth, max_wd, NULL);
    XtVaSetValues(dlg, XmNinitialFocus, rowcol, NULL);
  } else {
    label_widgets = value_widgets = NULL;
    flag_widgets = NULL;
  }
}

static void get_fields(FieldDescRec *fld_desc, int n, int maxlen)
{
  int i;

  for (i = 0; i < n; i++)
    if (fld_desc[i].id == ID_EDIT_FIELD) {
      if (fld_desc[i].value && value_widgets[i]) {
	String s = XmTextFieldGetString(value_widgets[i]);
      
	strncpy(fld_desc[i].value, s, maxlen);
	fld_desc[i].value[maxlen-1] = '\0';
	FREE(s);
      }
    } else if (fld_desc[i].id == ID_FLAGS_FIELD) {
      int j;

      for (j = 0; j < fld_desc[i].n; j++) {
	unsigned char set;

	XtVaGetValues(flag_widgets[i][j], XmNset, &set, NULL);
	fld_desc[i].mask |= (1L<<j);
	if (set == XmSET)
	  fld_desc[i].flags |= (1L<<j);
	else if (set == XmUNSET)
	  fld_desc[i].flags &= ~(1L<<j);
	else
	  fld_desc[i].mask &= ~(1L<<j);
      }
    }
}

/* process a dialog */

static void process_dlg(Widget dlg)
{
  XtManageChild(dlg);
  while (XtIsManaged(dlg)) {
    XEvent event;

    XtAppNextEvent(app, &event);
    XtDispatchEvent(&event);
  }
}

/* create and manage a dialog box */

static int dlg(String name, IconPtr icon, String msg,
	       FieldDescRec *fld_desc, int n, int maxlen,
	       int type, int buttons, int default_button)
{
  Boolean SaveWait = WaitFlag;
  Widget dlg = create_dlg(name, icon, msg, type);
  int i;

  create_buttons(dlg, type, buttons);
  create_fields(dlg, fld_desc, n);
  button_return = default_button;
  /* make sure we don't get an update during processing of the dialog */
  lock = True;
  Done();
  process_dlg(dlg);
  if (SaveWait) Wait();
  lock = False;
  get_fields(fld_desc, n, maxlen);
  FREE(label_widgets);
  FREE(value_widgets);
  if (flag_widgets)
    for (i = 0; i < n; i++)
      FREE(flag_widgets[i]);
  FREE(flag_widgets);
  XtDestroyWidget(dlg);
  XmUpdateDisplay(app_shell);
  return button_return;
}

#define msg_dlg(name,icon,msg,type,buttons,button) \
  dlg(name, icon, msg, NULL, 0, 0, type, buttons, button)
#define input_dlg(name,icon,msg,fld_desc,n,maxlen,buttons,button) \
  dlg(name, icon, msg, fld_desc, n, maxlen, QUESTION, buttons, \
      button)

static char s[3000];

Boolean force_overwrt = False, force_rename = False, force_delete = False,
  force_deldir = False, force_delfile = False;

int confirm_drop_dlg(IconPtr icon, int nFiles, String dirname, String target)
{
  if (confirm_drop) {
    sprintf(s, CONFIRM_DROP, nFiles, dirname, target);
    return msg_dlg("confirm_drop_dlg", icon, s, QUESTION, Ok|Cancel,
		   CANCEL);
  } else
    return OK;
}

int confirm_move_dlg(IconPtr icon, int nFiles, String dirname, String target)
{
  if (confirm_move) {
    sprintf(s, CONFIRM_MOVE, nFiles, dirname, target);
    return msg_dlg("confirm_move_dlg", icon, s, QUESTION, Ok|Cancel,
		   CANCEL);
  } else
    return OK;
}

int confirm_copy_dlg(IconPtr icon, int nFiles, String dirname, String target)
{
  if (confirm_copy) {
    sprintf(s, CONFIRM_COPY, nFiles, dirname, target);
    return msg_dlg("confirm_copy_dlg", icon, s, QUESTION, Ok|Cancel,
		   CANCEL);
  } else
    return OK;
}

int confirm_link_dlg(IconPtr icon, int nFiles, String dirname, String target)
{
  if (confirm_link) {
    sprintf(s, CONFIRM_LINK, nFiles, dirname, target);
    return msg_dlg("confirm_link_dlg", icon, s, QUESTION, Ok|Cancel,
		   CANCEL);
  } else
    return OK;
}

int confirm_delete_dlg(IconPtr icon, int nFiles, String dirname)
{
  if (confirm_delete) {
    sprintf(s, CONFIRM_DELETE, nFiles, dirname);
    return msg_dlg("confirm_delete_dlg", icon, s, QUESTION, Ok|Cancel,
		   CANCEL);
  } else
    return OK;
}

int confirm_overwrite_dir_dlg(IconPtr icon, String target)
{
  if (confirm_overwrt && !force_overwrt) {
    sprintf(s, CONFIRM_OVERWRITE_DIR, target);
    return msg_dlg("confirm_overwrite_dir_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_overwrite_file_dlg(IconPtr icon, String target)
{
  if (confirm_overwrt && !force_overwrt) {
    sprintf(s, CONFIRM_OVERWRITE_FILE, target);
    return msg_dlg("confirm_overwrite_file_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_rename_dir_dlg(IconPtr icon, String target)
{
  if (confirm_overwrt && !force_overwrt) {
    sprintf(s, CONFIRM_RENAME_DIR, target);
    return msg_dlg("confirm_rename_dir_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_rename_file_dlg(IconPtr icon, String target)
{
  if (confirm_overwrt && !force_overwrt) {
    sprintf(s, CONFIRM_RENAME_FILE, target);
    return msg_dlg("confirm_rename_file_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_delete_dir_dlg(IconPtr icon, String name)
{
  if (confirm_deldir && !force_deldir) {
    sprintf(s, CONFIRM_DELETE_DIR, name);
    return msg_dlg("confirm_delete_dir_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_delete_file_dlg(IconPtr icon, String name)
{
  if (confirm_delfile && !force_delfile) {
    sprintf(s, CONFIRM_DELETE_FILE, name);
    return msg_dlg("confirm_delete_file_dlg", icon, s, WARNING,
	       Yes|No|Doall|Cancel, CANCEL);
  } else
    return YES;
}

int confirm_quit_dlg(IconPtr icon)
{
  if (confirm_quit)
    return msg_dlg("confirm_quit_dlg", icon, CONFIRM_QUIT, QUESTION, Ok|Cancel,
	       CANCEL);
  else
    return OK;
}

static int mount_dlg1(IconPtr icon, String name)
{
  sprintf(s, MOUNT_RETRY, name);
  return msg_dlg("mount_dlg", icon, s, QUESTION, Ok|Ignore|Cancel, CANCEL);
}

int mount_dlg(IconPtr icon, String name, Boolean verify)
{
  if (!check || !verify)
    return devmount(name);
  else {
    int d;
    while ((d = devmount(name)) == NONE)
      switch (mount_dlg1(icon, name)) {
      case OK:
	break;
      case IGNORE:
	d = devmount_ignore(name);
      case CANCEL:
	return d;
      }
    return d;
  }
}

static int umount_dlg1(IconPtr icon, int d)
{
  sprintf(s, UMOUNT_RETRY, devname(d));
  return msg_dlg("umount_dlg", icon, s, QUESTION, Ok|Ignore, IGNORE);
}

void umount_dlg(IconPtr icon, int d)
{
  while (!devumount(d))
    switch (umount_dlg1(icon, d)) {
    case OK:
      break;
    case IGNORE:
      devumount_ignore(d);
      return;
    }
  return;
}

int mkdir_dlg(IconPtr icon, String s, int maxlen)
{
  mkdir_fld_desc[0].value = s;
  return input_dlg("mkdir_dlg", icon, MKDIR_MSG,
		   mkdir_fld_desc, XtNumber(mkdir_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

int chdir_dlg(IconPtr icon, String s, int maxlen)
{
  chdir_fld_desc[0].value = s;
  return input_dlg("chdir_dlg", icon, CHDIR_MSG,
		   chdir_fld_desc, XtNumber(chdir_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

int rename_dlg(IconPtr icon, String s, int maxlen)
{
  rename_fld_desc[0].value = s;
  return input_dlg("rename_dlg", icon, RENAME_MSG,
		   rename_fld_desc, XtNumber(rename_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

int move_dlg(IconPtr icon, String s, int maxlen)
{
  move_fld_desc[0].value = s;
  return input_dlg("move_dlg", icon, MOVE_MSG,
		   move_fld_desc, XtNumber(move_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

int copy_dlg(IconPtr icon, String s, int maxlen)
{
  copy_fld_desc[0].value = s;
  return input_dlg("copy_dlg", icon, COPY_MSG,
		   copy_fld_desc, XtNumber(copy_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

int link_dlg(IconPtr icon, String s, int maxlen)
{
  link_fld_desc[0].value = s;
  return input_dlg("link_dlg", icon, LINK_MSG,
		   link_fld_desc, XtNumber(link_fld_desc), maxlen,
		   Ok|Cancel, CANCEL);
}

/* unfortunately, the rwx permission flags in the stat structure are just the
   wrong way round; they are reversed with the following routine */

static unsigned long perm_flags(unsigned long flags)
{
  return ((flags & 1) << 2) | (flags & 2) | ((flags & 4) >> 2);
}

int props_dlg(IconPtr icon, int nFiles, int nBytes,
	      String dirname, String filename, String linkname,
	      String desc, String comment,
	      String last_acc, String last_mod, String last_chg,
	      String user, String group, int maxlen,
	      umode_t *mode, umode_t *mask)
{
  int ret;
  umode_t mode1;
  char *summary, size[10], num[10];

  sprintf(size, "%d", nBytes);
  sprintf(num, "%d", nFiles);
  summary = alloca(strlen(PROPS_SUMMARY)+strlen(size)+strlen(num)+1);
  sprintf(summary, PROPS_SUMMARY, size, (nBytes==1)?"":"s", num,
	  (nFiles==1)?"":"s");
  props_fld_desc[3].label = summary;
  props_fld_desc[4].value = dirname;
  props_fld_desc[5].value = filename;
  props_fld_desc[6].value = desc;
  props_fld_desc[7].value = linkname;
  props_fld_desc[8].value = comment;
  props_fld_desc[9].value = last_acc;
  props_fld_desc[10].value = last_mod;
  props_fld_desc[11].value = last_chg;
  props_fld_desc[12].value = user;
  props_fld_desc[13].value = group;
  props_fld_desc[17].flags = perm_flags((((unsigned long)*mode) &
					S_IRWXU)>>6);
  props_fld_desc[18].flags = perm_flags((((unsigned long)*mode) &
					 S_IRWXG)>>3);
  props_fld_desc[19].flags = perm_flags((((unsigned long)*mode) &
					 S_IRWXO)>>0);
  props_fld_desc[17].mask = perm_flags((((unsigned long)*mask) &
				       S_IRWXU)>>6);
  props_fld_desc[18].mask = perm_flags((((unsigned long)*mask) &
					S_IRWXG)>>3);
  props_fld_desc[19].mask = perm_flags((((unsigned long)*mask) &
					S_IRWXO)>>0);
  ret = input_dlg("props_dlg", icon, PROPS_MSG,
		  props_fld_desc, XtNumber(props_fld_desc), maxlen,
		  Ok|Cancel, CANCEL);
  mode1 = (umode_t) ((perm_flags(props_fld_desc[17].flags)<<6) |
		     (perm_flags(props_fld_desc[18].flags)<<3) |
		     (perm_flags(props_fld_desc[19].flags)<<0));
  *mask = (umode_t) ((perm_flags(props_fld_desc[17].mask)<<6) |
		     (perm_flags(props_fld_desc[18].mask)<<3) |
		     (perm_flags(props_fld_desc[19].mask)<<0));
  *mode = (*mode & ~(*mask)) | (mode1 & (*mask));
  return ret;
}

int filter_dlg(IconPtr icon, String s, int maxlen)
{
  filter_fld_desc[0].value = s;
  return input_dlg("filter_dlg", icon, FILTER_MSG,
		   filter_fld_desc, XtNumber(filter_fld_desc), maxlen,
		   Ok|Clear|Cancel, CANCEL);
}

int select_dlg(IconPtr icon, String s, int maxlen)
{
  select_fld_desc[0].value = s;
  return input_dlg("select_dlg", icon, SELECT_MSG,
		   select_fld_desc, XtNumber(select_fld_desc), maxlen,
		   Replace|Add|Remove|Cancel, CANCEL);
}

/* this comes from xfm/moxfm -- thanks to Brian King for his default value
   patch */

#define kDefaultValueMarker "--" /* Marker to denote default value */
#define kDefaultValue ""         /* Default Value to use if none specified */

int param_dlg(IconPtr icon, String msg, String action, String buf, int bufsz)
{
  char *act1 = (char *)alloca(strlen(action)+1), *s, *t;
  char *str = (char *)alloca(strlen(action)+1);
  char **acts = NULL, **vars = NULL;
  int n_acts = 0, n_vars = 0;
  char *def_val;
  char **vals = NULL;

  if (!action) return CANCEL;

  strcpy(act1, action);

  for (s = split2(act1, '%'); s; s = split2(NULL, '%')) {
    acts = (char **)REALLOC(acts, (n_acts+1)*sizeof(char *));
    acts[n_acts++] = XtNewString(unquote(str, s));
    if ((t = split2(NULL, '%'))) {
      vars = (char **)REALLOC(vars, (n_vars+1)*sizeof(char *));
      vals = (char **)REALLOC(vals, (n_vars+1)*sizeof(char *));
      if (!strcmp(t, "--")) {
	/* separator */
	vars[n_vars] = NULL;
	vals[n_vars++] = NULL;
      } else if (*t == '-') {
	/* label */
	vars[n_vars] = XtNewString(unquote(str, t+1));
	vals[n_vars++] = NULL;
      } else {
	vars[n_vars] = XtNewString(unquote(str, t));
	vals[n_vars] = (char *)MALLOC(bufsz+1);
	/* check string for default value character */
	if ((def_val = strstr(vars[n_vars], kDefaultValueMarker)) == NULL) {
	  strcpy(vals[n_vars++], kDefaultValue);
	} else {
	  def_val[0] = '\0'; /* Separate label and default value */
	  strcpy(vals[n_vars++], def_val + strlen(kDefaultValueMarker));
	}
      }
    } else
      break;
  }

  if (n_vars) {
    FieldDescRec *param_fld_desc =
      (FieldDescRec*) MALLOC(n_vars * sizeof(FieldDescRec));
    int i, ret;

    for (i = 0; i < n_vars; i++) {
      param_fld_desc[i].id = vars[i]?(vals[i]?ID_EDIT_FIELD:ID_LABEL):
	ID_SEPARATOR;
      param_fld_desc[i].label = vars[i];
      param_fld_desc[i].value = vals[i];
      param_fld_desc[i].buttons = NULL;
      param_fld_desc[i].flags = param_fld_desc[i].mask = 0L;
      param_fld_desc[i].n = 0;
    }
    ret  = input_dlg("param_dlg", icon, msg?msg:PARAM_MSG, param_fld_desc,
		     n_vars, bufsz, Ok|Cancel, CANCEL);
    FREE(param_fld_desc);
    if (ret == OK) {
      int i, l;
      for (l = i = 0; i < n_acts; i++) {
	int l1 = strlen(acts[i]), l2 = (i<n_vars && vals[i])?strlen(vals[i]):0;
	if (l+l1+l2 >= bufsz) {
	  ret = ERROR;
	  break;
	}
	strcpy(buf+l, acts[i]);
	if (l2) strcpy(buf+l+l1, vals[i]);
	l += l1+l2;
      }
      for (i = 0; i < n_acts; i++)
	FREE(acts[i]);
      for (i = 0; i < n_vars; i++)
	if (vars[i]) FREE(vars[i]);
      for (i = 0; i < n_vars; i++)
	if (vals[i]) FREE(vals[i]);
      FREE(acts); FREE(vars); FREE(vals);
    }
    return ret;
  } else if (strlen(action) > bufsz)
    return ERROR;
  else {
    strcpy(buf, action);
    return OK;
  }
}

int about_dlg(IconPtr icon)
{
  static String msg = "This is xplore version %s\n\n"
"Copyright (c) 1996, 1997 by Albert Graef\n\n"
"Parts of this work are derived from xfm 1.3.2 and moxfm 0.99,\n\n"
"Copyright (c) 1990-1996 by Simon Marlow, Albert Graef, Oliver Mai\n"
"Copyright (c) 1994 by Robert Vogelgesang (shell autodetection)\n"
"Copyright (c) 1995 by Juan D. Martin (magic headers)\n\n"
"and from xfsm 1.89,\n\n"
"Copyright (c) 1993-1996 by Robert Gasch\n\n"
"Copying policy: GNU General Public License\n\n"
"For further documentation please refer to the xplore(1) manual page.";

  sprintf(s, msg, version);
  return msg_dlg("about_dlg", icon, s, INFO, Ok, OK);
}

int error_dlg(IconPtr icon, String msg)
{
  return msg_dlg("error_dlg", icon, msg, ERRORMSG, Ok, OK);
}
