/* Cut/copy and paste functions
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static EDITORBUFFER
	*currentClipboard;							/* pointer to the current clipboard buffer, or NULL if none */

EDITORBUFFER *EditorGetCurrentClipboard()
/* return the current clipboard, or NULL if there is none
 */
{
	if(currentClipboard)
		{
		if(!currentClipboard->contentName)		/* see if the buffer went inactive when we were not looking */
			{
			EditorReleaseBuffer(currentClipboard);	/* release our grip on it */
			currentClipboard=NULL;				/* no more current clipboard */
			}
		}
	return(currentClipboard);
}

void EditorSetCurrentClipboard(EDITORBUFFER *theClipboard)
/* set the current clipboard
 */
{
	if(currentClipboard)
		{
		EditorReleaseBuffer(currentClipboard);	/* release our grip on it */
		}
	if(currentClipboard=theClipboard)
		{
		EditorHoldBuffer(currentClipboard);		/* hang on to this one */
		}
}

void EditorCopy(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard,BOOLEAN append)
/* copy the current selection if there is one from theEditorUniverse into theClipboard
 * if append is TRUE, the old contents of the clipboard remain, and the new
 * data is appended
 * a "selection" is made in the clipboard text for every selection that is
 * copied from the source text. These selections are used when columnar pasting.
 * NOTE: for speed, this deals directly with the text universe, and invalidates
 * all views at the end
 */
{
	CHUNKHEADER
		*sourceChunk;
	UINT32
		sourceChunkOffset;
	EDITORUNIVERSE
		*sourceEditorUniverse,
		*destEditorUniverse;
	SELECTIONUNIVERSE
		*sourceSelectionUniverse,
		*destSelectionUniverse;
	ARRAYCHUNKHEADER
		*sourceSelectionChunk,
		*destSelectionChunk;
	UINT32
		sourceSelectionOffset,
		destSelectionOffset;
	UINT32
		sourceCurrentPosition,
		sourceStartPosition,
		sourceEndPosition,
		destCurrentPosition,
		destStartPosition,
		destEndPosition;
	UINT32
		oldEndOffset;
	BOOLEAN
		fail;

	fail=FALSE;
	if(theClipboard)
		{
		destEditorUniverse=theClipboard->editorUniverse;
		destSelectionUniverse=destEditorUniverse->selectionUniverse;
		sourceEditorUniverse=theEditorUniverse;
		sourceSelectionUniverse=sourceEditorUniverse->selectionUniverse;
		sourceSelectionChunk=sourceSelectionUniverse->selectionChunks.firstChunkHeader;
		EditorStartReplace(destEditorUniverse);
		if(!append)				/* if not appending, then delete what was in the clipboard */
			{
			if(RegisterUndoDelete(destEditorUniverse,0,destEditorUniverse->textUniverse->totalBytes))
				{
				oldEndOffset=destEditorUniverse->textUniverse->totalBytes;			/* remember these so we can fix selections later */
				if(DeleteUniverseText(destEditorUniverse->textUniverse,0,destEditorUniverse->textUniverse->totalBytes))
					{
					DeleteUniverseSelection(destSelectionUniverse,destSelectionUniverse->selectionChunks.firstChunkHeader,0,destSelectionUniverse->selectionChunks.totalElements,&destSelectionChunk,&destSelectionOffset);	/* get rid of selection list for clipboard */
					FixEditorUniverseSelections(destEditorUniverse,0,oldEndOffset,0);	/* fix any remaining selections/marks (0 them) */
					}
				else
					{
					fail=TRUE;
					}
				}
			else
				{
				fail=TRUE;
				}
			}

		/* find out the position of the last selection in the clipboard */
		destCurrentPosition=0;
		destSelectionChunk=destSelectionUniverse->selectionChunks.firstChunkHeader;
		destSelectionOffset=0;
		while(destSelectionChunk)
			{
			destCurrentPosition+=((SELECTIONELEMENT *)destSelectionChunk->data)[destSelectionOffset].startOffset+((SELECTIONELEMENT *)destSelectionChunk->data)[destSelectionOffset].endOffset;
			destSelectionOffset++;
			if(destSelectionOffset>=destSelectionChunk->totalElements)
				{
				destSelectionChunk=destSelectionChunk->nextHeader;
				destSelectionOffset=0;
				}
			}

		sourceSelectionOffset=0;
		sourceCurrentPosition=0;
		while(sourceSelectionChunk&&!fail)
			{
			sourceStartPosition=sourceCurrentPosition+((SELECTIONELEMENT *)sourceSelectionChunk->data)[sourceSelectionOffset].startOffset;
			sourceEndPosition=sourceStartPosition+((SELECTIONELEMENT *)sourceSelectionChunk->data)[sourceSelectionOffset].endOffset;
			sourceCurrentPosition=sourceEndPosition;
			PositionToChunkPosition(sourceEditorUniverse->textUniverse,sourceStartPosition,&sourceChunk,&sourceChunkOffset);	/* point to the start of the selection in the source data */
			destStartPosition=destEditorUniverse->textUniverse->totalBytes;
			if(InsertUniverseChunks(destEditorUniverse->textUniverse,destEditorUniverse->textUniverse->totalBytes,sourceChunk,sourceChunkOffset,sourceEndPosition-sourceStartPosition))			/* insert the text at the end of the destination */
				{
				destEndPosition=destEditorUniverse->textUniverse->totalBytes;
				if(destEndPosition>destStartPosition)
					{
					if(InsertUniverseSelection(destSelectionUniverse,NULL,0,1,&destSelectionChunk,&destSelectionOffset))		/* create new selection element */
						{
						((SELECTIONELEMENT *)destSelectionChunk->data)[destSelectionOffset].startOffset=destStartPosition-destCurrentPosition;
						((SELECTIONELEMENT *)destSelectionChunk->data)[destSelectionOffset].endOffset=destEndPosition-destStartPosition;
						destCurrentPosition=destEndPosition;
						}
					else
						{
						fail=TRUE;
						}
					}
				sourceSelectionOffset++;
				if(sourceSelectionOffset>=sourceSelectionChunk->totalElements)
					{
					sourceSelectionChunk=sourceSelectionChunk->nextHeader;
					sourceSelectionOffset=0;
					}
				if(!fail&&sourceSelectionChunk)				/* if not at end of selection list, insert newlines between into buffer */
					{
					fail=!InsertUniverseText(destEditorUniverse->textUniverse,destEditorUniverse->textUniverse->totalBytes,(UINT8 *)"\n",1);
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		if(!fail)
			{
			fail=!RegisterUndoInsert(destEditorUniverse,0,destEditorUniverse->textUniverse->totalBytes);
			}
		destEditorUniverse->dirty=TRUE;					/* this buffer is dirty, since we just wrote into it */
		EditorInvalidateViews(destEditorUniverse);
		EditorEndReplace(destEditorUniverse);
		}
	if(!fail)
		{
		if(theClipboard==EditorGetCurrentClipboard())		/* only export the clipboard if it is the current one, and we did not fail */
			{
			if(!ExportClipboard())
				{
				ReportMessage("Failed to export clipboard\n");
				}
			}
		}
	else
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to copy: %.256s\n",errorDescription);
		}
}

void EditorCut(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard,BOOLEAN append)
/* cut the current selection if there is one from theEditorUniverse into theClipboard
 * if append is TRUE, the old contents of the clipboard remain, and the new
 * data is appended
 */
{
	EditorCopy(theEditorUniverse,theClipboard,append);
	EditorStartReplace(theEditorUniverse);
	BeginUndoGroup(theEditorUniverse);
	DeleteAllSelectedText(theEditorUniverse);
	StrictEndUndoGroup(theEditorUniverse);
	EditorEndReplace(theEditorUniverse);
}

void EditorPaste(EDITORUNIVERSE *theEditorUniverse,EDITORBUFFER *theClipboard)
/* paste the contents of theClipboard into theEditorUniverse in the simplest way
 * (overwrite any selection with the contents of the clipboard)
 */
{
	EDITORUNIVERSE
		*sourceEditorUniverse,
		*destEditorUniverse;
	SELECTIONUNIVERSE
		*destSelectionUniverse;

	if(theClipboard)
		{
		if(theClipboard==EditorGetCurrentClipboard())		/* only import the clipboard if it is the current one */
			{
			if(!ImportClipboard())							/* give external world a chance to override clipboard */
				{
				ReportMessage("Failed to import clipboard\n");
				}
			}
		sourceEditorUniverse=theClipboard->editorUniverse;
		destEditorUniverse=theEditorUniverse;
		destSelectionUniverse=destEditorUniverse->selectionUniverse;
		EditorStartReplace(destEditorUniverse);
		BeginUndoGroup(destEditorUniverse);
		DeleteAllSelectedText(destEditorUniverse);			/* kill all selections in the passed view */
		ReplaceEditorChunks(destEditorUniverse,destSelectionUniverse->cursorPosition,destSelectionUniverse->cursorPosition,sourceEditorUniverse->textUniverse->firstChunkHeader,0,sourceEditorUniverse->textUniverse->totalBytes);
		StrictEndUndoGroup(destEditorUniverse);
		EditorEndReplace(destEditorUniverse);
		}
}

void EditorColumnarPaste(EDITORVIEW *theView,EDITORBUFFER *theClipboard)
/* paste the contents of theClipboard into theView in columnar format
 * this means that each "selection" of the clipboard will be placed into theView at
 * the current cursor position, then the position is moved to the next line, and
 * the process is repeated until all of the selections have been placed
 * NOTE: if there are selections in theView, they are deleted first
 * NOTE: if there are no selections in the clipboard, then it is just pasted as one
 * large blob
 */
{
	CHUNKHEADER
		*sourceChunk,
		*destChunk;
	UINT32
		sourceChunkOffset,
		destChunkOffset;
	EDITORUNIVERSE
		*sourceEditorUniverse,
		*destEditorUniverse;
	SELECTIONUNIVERSE
		*sourceSelectionUniverse,
		*destSelectionUniverse;
	ARRAYCHUNKHEADER
		*sourceSelectionChunk;
	UINT32
		sourceSelectionOffset;
	UINT32
		sourceCurrentPosition,
		sourceStartPosition,
		sourceEndPosition,
		destStartPosition;
	UINT32
		temp,
		lastNumLines,
		newPosition,
		theLine,
		theLineOffset;
	INT32
		desiredX;
	BOOLEAN
		fail;

	if(theClipboard)
		{
		sourceEditorUniverse=theClipboard->editorUniverse;
		if(ImportClipboard())							/* give external world a chance to override clipboard */
			{
			fail=FALSE;

			destEditorUniverse=theView->editorUniverse;
			destSelectionUniverse=destEditorUniverse->selectionUniverse;
			sourceSelectionUniverse=sourceEditorUniverse->selectionUniverse;
			EditorStartReplace(destEditorUniverse);
			BeginUndoGroup(destEditorUniverse);
			DeleteAllSelectedText(destEditorUniverse);	/* kill all selections in the passed view */

			if(sourceSelectionChunk=sourceSelectionUniverse->selectionChunks.firstChunkHeader)	/* see if selections in the clipboard */
				{
				PositionToLinePosition(destEditorUniverse->textUniverse,destSelectionUniverse->cursorPosition,&theLine,&theLineOffset,&destChunk,&destChunkOffset);
				GetEditorViewTextToGraphicPosition(theView,destChunk,destChunkOffset,theLineOffset,&desiredX);	/* find the X offset we will attempt to insert at each time */

				sourceSelectionOffset=0;
				sourceCurrentPosition=0;

				while(sourceSelectionChunk&&!fail)
					{
					sourceStartPosition=sourceCurrentPosition+((SELECTIONELEMENT *)sourceSelectionChunk->data)[sourceSelectionOffset].startOffset;
					sourceEndPosition=sourceStartPosition+((SELECTIONELEMENT *)sourceSelectionChunk->data)[sourceSelectionOffset].endOffset;
					sourceCurrentPosition=sourceEndPosition;
					PositionToChunkPosition(sourceEditorUniverse->textUniverse,sourceStartPosition,&sourceChunk,&sourceChunkOffset);	/* point to the start of the selection in the source data */

					LineToChunkPosition(destEditorUniverse->textUniverse,theLine,&destChunk,&destChunkOffset,&destStartPosition);

					GetEditorViewGraphicToTextPosition(theView,destChunk,destChunkOffset,desiredX,&newPosition,&temp);
					destStartPosition+=newPosition;			/* this is where the insertion should occur */

					if(theLine>destEditorUniverse->textUniverse->totalLines)	/* attempting to place on non-existent line? */
						{
						if(ReplaceEditorText(destEditorUniverse,destStartPosition,destStartPosition,(UINT8 *)"\n",1))	/* stick in a newline at the end */
							{
							destStartPosition++;					/* then move past it */
							}
						else
							{
							fail=TRUE;
							}
						}
					lastNumLines=destEditorUniverse->textUniverse->totalLines;			/* remember how many lines there were, so we can see how many were inserted */
					if(!fail)
						{
						if(ReplaceEditorChunks(destEditorUniverse,destStartPosition,destStartPosition,sourceChunk,sourceChunkOffset,sourceEndPosition-sourceStartPosition))
							{
							sourceSelectionOffset++;
							if(sourceSelectionOffset>=sourceSelectionChunk->totalElements)
								{
								sourceSelectionChunk=sourceSelectionChunk->nextHeader;
								sourceSelectionOffset=0;
								}
							theLine+=destEditorUniverse->textUniverse->totalLines-lastNumLines+1;	/* skip to next line, plus any inserted */
							}
						else
							{
							fail=TRUE;
							}
						}
					}
				}
			else
				{
				fail=!ReplaceEditorChunks(destEditorUniverse,destSelectionUniverse->cursorPosition,destSelectionUniverse->cursorPosition,sourceEditorUniverse->textUniverse->firstChunkHeader,0,sourceEditorUniverse->textUniverse->totalBytes);
				}
			StrictEndUndoGroup(destEditorUniverse);
			EditorEndReplace(destEditorUniverse);

			if(fail)
				{
				GetError(&errorFamily,&errorFamilyMember,&errorDescription);
				ReportMessage("Failed to paste: %.256s\n",errorDescription);
				}
			}
		else
			{
			ReportMessage("Failed to import clipboard\n");
			}
		}
}

EDITORUNIVERSE *EditorStartImportClipboard()
/* Call this from the gui to get the editor universe that is
 * the current clipboard, when something external to the editor has modified
 * the system clipboard, and it is desired to get that data into the editor
 * it is legal for this routine to return NULL
 * in which case you should not attempt to place anything into the clipboard
 * and you should not call EditorEndImportClipboard
 */
{
	EDITORBUFFER
		*theClipboard;
	EDITORUNIVERSE
		*destEditorUniverse;

	destEditorUniverse=NULL;
	if(theClipboard=EditorGetCurrentClipboard())
		{
		destEditorUniverse=theClipboard->editorUniverse;
		}
	return(destEditorUniverse);
}

void EditorEndImportClipboard()
/* Call this when you are done modifying the clipboard that was returned
 * from EditorStartImportClipboard
 */
{
}

EDITORUNIVERSE *EditorStartExportClipboard()
/* Call this from the gui to get the editor universe that is
 * the current clipboard, when you wish to export the current clipboard
 * if this returns NULL, you should not call EditorEndExportClipboard
 */
{
	EDITORBUFFER
		*theClipboard;

	if(theClipboard=EditorGetCurrentClipboard())
		{
		return(theClipboard->editorUniverse);
		}
	return(NULL);
}

void EditorEndExportClipboard()
/* Call this when you are done exporting the clipboard that was returned
 * from EditorStartExportClipboard
 */
{
}

BOOLEAN InitClipboard()
/* initialize clipboard handling
 * if there is a problem, SetError, and return FALSE
 */
{
	currentClipboard=NULL;				/* no start buffer as of yet */
	return(TRUE);
}

void UnInitClipboard()
/* undo whatever InitClipboard did
 */
{
	EditorSetCurrentClipboard(NULL);			/* clear the current clipboard, releasing any buffer it held */
}
