/* Shell interface for e93 (currently using John Ousterhout's Tcl)
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static Tcl_AsyncHandler
	abortHandler;				/* this allows us to check if the user has aborted during a Tcl script */

void InsertBufferTaskData(EDITORBUFFER *theBuffer,UINT8 *theData,UINT32 numBytes)
/* insert numBytes of theData into theBuffer, update any
 * views of theBuffer that may need it
 * if there is a problem, report it here
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	EDITORVIEW
		*currentView;
	UINT32
		startPosition,
		endPosition;
	UINT32
		startLine;
	CHUNKHEADER
		*startLineChunk;
	UINT32
		startLineOffset;

/* run through the views, see which ones will need to be homed after the insert:
 * The ones to home, are those that had the cursor in view at the time of the insert
 */

	currentView=theBuffer->editorUniverse->firstView;	/* get first view of this buffer (if any) */
	while(currentView)									/* walk through all views, update each one as needed */
		{
		GetEditorViewTextInfo(currentView,&topLine,&numLines,&leftPixel,&numPixels);
		GetSelectionEndPositions(currentView->editorUniverse->selectionUniverse,&startPosition,&endPosition);
		PositionToLinePosition(currentView->editorUniverse->textUniverse,startPosition,&startLine,&startPosition,&startLineChunk,&startLineOffset);
		currentView->wantHome=(startLine>=topLine&&startLine<topLine+numLines);	/* see if the cursor is on the view */
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}

	EditorAuxInsert(theBuffer->editorUniverse,theData,numBytes);	/* drop the text into theBuffer */

/* now home all of those views that need it */
	currentView=theBuffer->editorUniverse->firstView;	/* get first view of this buffer (if any) */
	while(currentView)									/* walk through all views, update each one as needed */
		{
		if(currentView->wantHome)
			{
			GetSelectionEndPositions(currentView->editorUniverse->selectionUniverse,&startPosition,&endPosition);
			EditorHomeViewToPositionLenient(currentView,startPosition);
			}
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

BOOLEAN HandleBoundKeyEvent(UINT32 keyCode,UINT32 modifierValue)
/* see if there is a bound key that matches the given code, and modifiers
 * if there is, perform the script given by the bound key, and return TRUE
 * if there is not, return FALSE
 */
{
	EDITORKEYBINDING
		*theEntry;
	int
		tclResult;

	if(theEntry=LocateKeyBindingMatch(keyCode,modifierValue))
		{
		ClearAbort();
		tclResult=Tcl_Eval(theTclInterpreter,&(theEntry->dataText[0]));
		if((tclResult!=TCL_OK)&&*(theTclInterpreter->result))
			{
			ReportMessage("Bound key execution error:\n%.256s\n",theTclInterpreter->result);
			}
		return(TRUE);
		}
	return(FALSE);
}

void HandleMenuEvent(EDITORMENU *theMenu)
/* when a menu event arrives, this is called (by the menu manager of the GUI) to handle the menu
 */
{
	int
		tclResult;

	ClearAbort();
	tclResult=Tcl_Eval(theTclInterpreter,GetEditorMenuDataText(theMenu));
	if((tclResult!=TCL_OK)&&*(theTclInterpreter->result))
		{
		ReportMessage("Menu execution error:\n%.256s\n",theTclInterpreter->result);
		}
}

#define MAXSCROLLSPEED	2

void HandleViewEvent(VIEWEVENT *theEvent)
/* high level view events are handled here
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	UINT8
		theKey;
	static UINT32
		scrollTimeout;
	static UINT32
		scrollSpeed;
	UINT16
		trackMode;							/* tells editor how to place cursor when cursor position keys arrive */
	EDITORKEY
		*keyEventData;
	VIEWPOSEVENTDATA
		*posEventData;
	VIEWCLICKEVENTDATA
		*clickEventData;

	switch(theEvent->eventType)
		{
		case VET_KEYDOWN:
		case VET_KEYREPEAT:
			keyEventData=(EDITORKEY *)theEvent->eventData;
			if(keyEventData->isVirtual)
				{
				switch(keyEventData->keyCode)
					{
					case VVK_SCROLLUP:
						if(theEvent->eventType==VET_KEYDOWN)
							{
							scrollSpeed=1;
							scrollTimeout=0;
							}
						EditorVerticalScroll(theEvent->theView,-((INT32)scrollSpeed));
						if(scrollTimeout++>3)
							{
							if(scrollSpeed<MAXSCROLLSPEED)
								{
								scrollSpeed++;
								}
							scrollTimeout=0;
							}
						break;
					case VVK_SCROLLDOWN:
						if(theEvent->eventType==VET_KEYDOWN)
							{
							scrollSpeed=1;
							scrollTimeout=0;
							}
						EditorVerticalScroll(theEvent->theView,scrollSpeed);
						if(scrollTimeout++>3)
							{
							if(scrollSpeed<MAXSCROLLSPEED)
								{
								scrollSpeed++;
								}
							scrollTimeout=0;
							}
						break;
					case VVK_DOCUMENTPAGEUP:
						EditorVerticalScrollByPages(theEvent->theView,-1);
						break;
					case VVK_DOCUMENTPAGEDOWN:
						EditorVerticalScrollByPages(theEvent->theView,1);
						break;
					case VVK_SCROLLLEFT:
						EditorHorizontalScroll(theEvent->theView,-HORIZONTALSCROLLTHRESHOLD);
						break;
					case VVK_SCROLLRIGHT:
						EditorHorizontalScroll(theEvent->theView,HORIZONTALSCROLLTHRESHOLD);
						break;
					case VVK_DOCUMENTPAGELEFT:
						EditorHorizontalScrollByPages(theEvent->theView,-1);
						break;
					case VVK_DOCUMENTPAGERIGHT:
						EditorHorizontalScrollByPages(theEvent->theView,1);
						break;
					case VVK_RETURN:
						EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
						EditorAutoIndent(theEvent->theView->editorUniverse);
						EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
						break;
					case VVK_TAB:
						theKey='\t';				/* insert a tab */
						EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
						EditorInsert(theEvent->theView->editorUniverse,&theKey,1);
						EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
						break;
					case VVK_LEFTARROW:
						if(!(keyEventData->modifiers&EEM_CTL))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_CHAR);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_CHAR);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									}
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_WORD);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_MOD1))
									{
									EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorReduceNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								}
							}
						break;
					case VVK_RIGHTARROW:
						if(!(keyEventData->modifiers&EEM_CTL))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_CHAR);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_CHAR);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_LINEEDGE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_LINEEDGE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									}
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_WORD);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_MOD1))
									{
									EditorExpandNormalSelection(theEvent->theView,RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								else
									{
									EditorReduceNormalSelection(theEvent->theView,RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							}
						break;
					case VVK_UPARROW:
						if(!(keyEventData->modifiers&EEM_CTL))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_LINE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_LINE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_LINE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_PAGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_PAGE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_PAGE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									}
								}
							}
						else
							{
							EditorVerticalScroll(theEvent->theView,-1);
							}
						break;
					case VVK_DOWNARROW:
						if(!(keyEventData->modifiers&EEM_CTL))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_LINE);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_LINE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_LINE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorMoveCursor(theEvent->theView,RPM_PAGE);
									EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
									}
								else
									{
									if(!(keyEventData->modifiers&EEM_MOD1))
										{
										EditorExpandNormalSelection(theEvent->theView,RPM_PAGE);
										EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
										}
									else
										{
										EditorReduceNormalSelection(theEvent->theView,RPM_PAGE);
										EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
										}
									}
								}
							}
						else
							{
							EditorVerticalScroll(theEvent->theView,1);
							}
						break;
					case VVK_BACKSPACE:
						if(!(keyEventData->modifiers&EEM_CTL)&&!(keyEventData->modifiers&EEM_MOD1))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_DOCEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_DOCEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							}
						break;
					case VVK_DELETE:
						if(!(keyEventData->modifiers&EEM_CTL)&&!(keyEventData->modifiers&EEM_MOD1))
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_CHAR);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_LINEEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_MOD0))
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_WORD);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							else
								{
								if(!(keyEventData->modifiers&EEM_SHIFT))
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_mBACKWARD|RPM_DOCEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								else
									{
									EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
									EditorDelete(theEvent->theView,RPM_DOCEDGE);
									EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
									}
								}
							}
						break;
					case VVK_HOME:
						if(!(keyEventData->modifiers&EEM_MOD0))
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_LINEEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_DOCEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_DOCEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							}
						break;
					case VVK_END:
						if(!(keyEventData->modifiers&EEM_MOD0))
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_LINEEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_LINEEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_DOCEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_DOCEDGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							}
						break;
					case VVK_PAGEUP:
						if(keyEventData->modifiers&EEM_CTL)
							{
							EditorVerticalScrollByPages(theEvent->theView,-1);
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_mBACKWARD|RPM_PAGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_mBACKWARD|RPM_PAGE);
								EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
								}
							}
						break;
					case VVK_PAGEDOWN:
						if(keyEventData->modifiers&EEM_CTL)
							{
							EditorVerticalScrollByPages(theEvent->theView,1);
							}
						else
							{
							if(!(keyEventData->modifiers&EEM_SHIFT))
								{
								EditorMoveCursor(theEvent->theView,RPM_PAGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							else
								{
								EditorExpandNormalSelection(theEvent->theView,RPM_PAGE);
								EditorHomeViewToCursorLenient(theEvent->theView,TRUE);
								}
							}
						break;
					case VVK_UNDO:
						if(EditorUndo(theEvent->theView->editorUniverse))
							{
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							}
						else
							{
							EditorBeep();
							}
						break;
					case VVK_REDO:
						if(EditorRedo(theEvent->theView->editorUniverse))
							{
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							}
						else
							{
							EditorBeep();
							}
						break;
					case VVK_UNDOTOGGLE:
						if(EditorToggleUndo(theEvent->theView->editorUniverse))
							{
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							}
						else
							{
							EditorBeep();
							}
						break;
					case VVK_CUT:
						EditorCut(theEvent->theView->editorUniverse,EditorGetCurrentClipboard(),FALSE);
						EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
						break;
					case VVK_COPY:
						EditorCopy(theEvent->theView->editorUniverse,EditorGetCurrentClipboard(),FALSE);
						break;
					case VVK_PASTE:
						EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
						EditorColumnarPaste(theEvent->theView,EditorGetCurrentClipboard());
						EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
						break;
					}
				}
			else
				{
				theKey=(UINT8)keyEventData->keyCode;
				if(keyEventData->modifiers&EEM_MOD0)		/* check for bound keys */
					{
					switch(theKey)
						{
						case 'j':
							UniverseSanityCheck(theEvent->theView->editorUniverse->textUniverse);
							break;
						case 'a':
							EditorSelectAll(theEvent->theView->editorUniverse);
							break;
						case 'y':
							if(EditorRedo(theEvent->theView->editorUniverse))
								{
								EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
								}
							else
								{
								EditorBeep();
								}
							break;
						case 'u':
							if(EditorUndo(theEvent->theView->editorUniverse))
								{
								EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
								}
							else
								{
								EditorBeep();
								}
							break;
						case 'z':
							if(EditorToggleUndo(theEvent->theView->editorUniverse))
								{
								EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
								}
							else
								{
								EditorBeep();
								}
							break;
						case 'x':
							EditorCut(theEvent->theView->editorUniverse,EditorGetCurrentClipboard(),FALSE);
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							break;
						case 'c':
							EditorCopy(theEvent->theView->editorUniverse,EditorGetCurrentClipboard(),FALSE);
							break;
						case 'v':
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							EditorColumnarPaste(theEvent->theView,EditorGetCurrentClipboard());
							EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
							break;
						}
					}
				else
					{
					EditorHomeViewToCursorSemiStrict(theEvent->theView,FALSE);
					EditorInsert(theEvent->theView->editorUniverse,&theKey,1);
					EditorHomeViewToCursorLenient(theEvent->theView,FALSE);
					}
				}
			break;
		case VET_POSITIONVERTICAL:
			posEventData=(VIEWPOSEVENTDATA *)theEvent->eventData;
			GetEditorViewTextInfo(theEvent->theView,&topLine,&numLines,&leftPixel,&numPixels);
			SetViewTopLeft(theEvent->theView,posEventData->position,leftPixel);
			break;
		case VET_POSITIONHORIZONTAL:
			posEventData=(VIEWPOSEVENTDATA *)theEvent->eventData;
			GetEditorViewTextInfo(theEvent->theView,&topLine,&numLines,&leftPixel,&numPixels);
			SetViewTopLeft(theEvent->theView,topLine,posEventData->position);
			break;
		case VET_CLICK:
		case VET_CLICKHOLD:
			clickEventData=(VIEWCLICKEVENTDATA *)theEvent->eventData;
			trackMode=0;								/* default place mode */
			if(theEvent->eventType==VET_CLICKHOLD)		/* see if still tracking */
				{
				trackMode|=TM_mREPEAT;
				}
			if(clickEventData->modifiers&EEM_SHIFT)		/* attempting to add to current selection? */
				{
				trackMode|=TM_mCONTINUE;
				}
			if((clickEventData->modifiers&EEM_MOD0)||(clickEventData->keyCode>0))
				{
				trackMode|=TM_mCOLUMNAR;
				}
			switch((clickEventData->modifiers&EEM_STATE0)>>EES_STATE0)
				{
				case 0:
					trackMode|=TM_CHAR;
					break;
				case 1:
					trackMode|=TM_WORD;
					break;
				case 2:
					trackMode|=TM_LINE;
					break;
				default:
					trackMode|=TM_ALL;
					break;
				}
			EditorTrackViewPointer(theEvent->theView,clickEventData->xClick,clickEventData->yClick,trackMode);
			break;
		}
	ResetEditorViewCursorBlink(theEvent->theView);	/* reset cursor blinking (if we even have a cursor that blinks) */
}

BOOLEAN HandleShellCommand(char *theCommand,int argc,char *argv[])
/* sometimes things in the gui request that the shell should perform
 * certain operations. this is the interface to allow that to happen
 * If the operation has some sort of failure (defined by the operation)
 * this will report it (if there is a message) and return FALSE
 */
{
	char
		*theList;
	int
		tclResult;

	theList=Tcl_Merge(argc,argv);
	ClearAbort();
	tclResult=Tcl_VarEval(theTclInterpreter,"ShellCommand ",theCommand," ",theList,(char *)NULL);
	ckfree(theList);
	if(tclResult!=TCL_OK)
		{
		if(*(theTclInterpreter->result))				/* see if something to report */
			{
			ReportMessage("Shell command execution error:\n%.256s\n",theTclInterpreter->result);
			}
		return(FALSE);
		}
	return(TRUE);
}

static BOOLEAN ExecuteStartupScript(Tcl_Interp *theInterpreter)
/* Locate the startup script file, open, and execute it
 * if there is any problem, report it to stderr, and return FALSE
 */
{
	BOOLEAN
		foundFile,
		fail;
	char
		**scriptPaths;
	UINT32
		currentIndex;

	fail=FALSE;
	if(scriptPaths=GenerateStartupScriptLocationList())
		{
		currentIndex=0;
		foundFile=FALSE;
		ClearAbort();
		while(!foundFile&&scriptPaths[currentIndex])
			{
			Tcl_SetVar(theInterpreter,"SCRIPTPATH",scriptPaths[currentIndex],TCL_LEAVE_ERR_MSG);
			if(Tcl_EvalFile(theInterpreter,scriptPaths[currentIndex])==TCL_OK)
				{
				foundFile=TRUE;
				}
			else
				{
				if(theInterpreter->errorLine)
					{
					fprintf(stderr,"Startup script ERROR:\n%s:%d:%s\n",scriptPaths[currentIndex],theInterpreter->errorLine,theInterpreter->result);
					foundFile=TRUE;			/* let the editor try to run if it got even a little way through the startup script */
					}
				}
			currentIndex++;
			}
		if(!foundFile)
			{
			fprintf(stderr,"Could not locate readable startup script in:\n");
			currentIndex=0;
			while(scriptPaths[currentIndex])
				{
				fprintf(stderr,"  %s\n",scriptPaths[currentIndex]);
				currentIndex++;
				}
			fail=TRUE;
			}
		FreeStartupScriptLocationList(scriptPaths);
		}
	else
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		fprintf(stderr,"Failed to locate startup script: %s\n",errorDescription);
		fail=TRUE;
		}
	return(!fail);
}

static int Tcl_AbortProc(ClientData theClientData,Tcl_Interp *theInterpreter,int theCode)
/* this procedure will be enabled if we discover that an abort attempt has been made
 * while executing Tcl script, it will change the return code of the previously
 * executed Tcl command, so that we can abort
 */
{
	Tcl_ResetResult(theInterpreter);
	Tcl_AppendResult(theInterpreter,"User Aborted",NULL);
	return(TCL_ERROR);
}

static void Tcl_TraceCheckAbortProc(ClientData theClientData,Tcl_Interp *theInterpreter,int theLevel,char *theCommand,Tcl_CmdProc *theProc,ClientData cmdClientData,int argc,char *argv[])
/* This is a small trick on Tcl. We tell it we want to trace, but really, we want
 * to check to see if the user is trying to abort the execution of script.
 * so, every time we are called, we check to see if the user is aborting, and
 * if so, mark the AbortProc so that it will run
 */
{
	if(CheckAbort())
		{
		Tcl_AsyncMark((Tcl_AsyncHandler)theClientData);
		}
}

static void UnSetUpTclAbortHandling(Tcl_Interp *theInterpreter)
/* undo what SetUpTclAbortHandling did
 */
{
	Tcl_AsyncDelete(abortHandler);
}

static BOOLEAN SetUpTclAbortHandling(Tcl_Interp *theInterpreter)
/* set up tcl so that it manages to call CheckAbort, and if CheckAbort
 * returns TRUE, then we force an error in to the Tcl control Flow
 * with a result string of "Abort"
 */
{
	if(abortHandler=Tcl_AsyncCreate(Tcl_AbortProc,NULL))
		{
		if(Tcl_CreateTrace(theInterpreter,INT_MAX,Tcl_TraceCheckAbortProc,(ClientData)abortHandler))
			{
			return(TRUE);
			}
		Tcl_AsyncDelete(abortHandler);
		}
	return(FALSE);
}

void ShellLoop(int argc,char *argv[])
/* this is where the editor shell gets and handles events
 * it actually defers to the gui level to call us back with events
 */
{
	if(InitKeyBindingTable())
		{
		if(InitBuffers())
			{
			if(InitClipboard())
				{
				if(theTclInterpreter=Tcl_CreateInterp())
					{
					if(SetUpTclAbortHandling(theTclInterpreter))
						{
						if(CreateEditorShellCommands(theTclInterpreter))
							{
							if(AddSupplementalShellCommands(theTclInterpreter))
								{
								if(ExecuteStartupScript(theTclInterpreter))			/* deal with the start-up script, leave if there is a problem */
									{
									EditorEventLoop(argc,argv);						/* pass initial parameters */
									}
								}
							else
								{
								fprintf(stderr,"Failed to add supplemental shell commands\n");
								}
							}
						else
							{
							fprintf(stderr,"Failed to create editor shell commands\n");
							}
						UnSetUpTclAbortHandling(theTclInterpreter);
						}
					else
						{
						fprintf(stderr,"Failed to set up abort handling in Tcl\n");
						}
					Tcl_DeleteInterp(theTclInterpreter);						/* have TCL clean up */
					}
				else
					{
					fprintf(stderr,"Failed to create TCL interpreter\n");
					}
				UnInitClipboard();
				}
			else
				{
				fprintf(stderr,"Failed to initialize clipboard\n");
				}
			UnInitBuffers();
			}
		else
			{
			fprintf(stderr,"Failed to initialize buffers\n");
			}
		UnInitKeyBindingTable();
		}
	else
		{
		fprintf(stderr,"Failed to initialize key bindings\n");
		}
}
