/* The edit module handles the non-visible parts of editing not handled
   by text.c
   These include:
    cursor position/selections
    insertion and deletion of text
    load/save
    marks

   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

#define	LOADSAVEBUFFERSIZE		65536

BOOLEAN EditorAddSelection(SELECTIONUNIVERSE *theSelectionUniverse,UINT32 thePosition,UINT32 numBytes)
/* add the selection given by thePosition/numBytes into theSelectionUniverse
 * if the new selection overlaps previous ones, the previous ones are absorbed
 * NOTE: EditorStartSelectionChange, and EditorEndSelectionChange should be called externally
 * if theSelectionUniverse is visible
 * if there is a problem, set the error, and return FALSE
 */
{
	UINT32
		lastPosition;
	UINT32
		startPosition,
		endPosition;
	ARRAYCHUNKHEADER
		*modifiedChunk,
		*theChunk;
	UINT32
		modifiedOffset,
		theOffset;
	UINT32
		elementsToDelete;
	BOOLEAN
		done,
		added,
		fail;
	SELECTIONELEMENT
		*theElement;

	fail=FALSE;
	if(numBytes)												/* do not add empty selections */
		{
		theChunk=theSelectionUniverse->selectionChunks.firstChunkHeader;
		theOffset=0;
		done=added=FALSE;
		endPosition=startPosition=lastPosition=0;
		while(theChunk&&!done)
			{
			theElement=(SELECTIONELEMENT *)theChunk->data;
			theOffset=0;
			while((theOffset<theChunk->totalElements)&&!done)
				{
				if((endPosition+=theElement->startOffset+theElement->endOffset)>thePosition)
					{
					startPosition=endPosition-theElement->endOffset;
					lastPosition=startPosition-theElement->startOffset;

					if(startPosition<thePosition+numBytes)			/* if this is TRUE, the selection intersects the one we wish to add, so combine them */
						{
						lastPosition=endPosition;					/* remember old end position */
						if(startPosition<thePosition)
							{
							if(endPosition<thePosition+numBytes)
								{
								theElement->endOffset+=thePosition+numBytes-endPosition;	/* move end forward */
								endPosition=thePosition+numBytes;
								}
							}
						else
							{
							theElement->startOffset-=(startPosition-thePosition);	/* move start back */
							theElement->endOffset+=(startPosition-thePosition);		/* move end forward */
							if(endPosition<thePosition+numBytes)
								{
								theElement->endOffset+=(thePosition+numBytes)-endPosition;
								endPosition=thePosition+numBytes;
								}
							startPosition=thePosition;
							}
						thePosition=startPosition;							/* update these */
						numBytes=endPosition-startPosition;
						added=TRUE;
						}
					done=TRUE;
					}
				else
					{
					theElement++;								/* move to the next element */
					theOffset++;
					}
				}
			if(!done)
				{
				theChunk=theChunk->nextHeader;
				}
			}
		if(!done)
			{
			lastPosition=endPosition;
			}
		/* now theChunk/theOffset point to the modified element, or to the one just beyond the area we wish to add to */
		/* lastPosition is the ending position of the last chunk looked at */
		if(!added)												/* see if we need to add the selection */
			{
			if(InsertUniverseSelection(theSelectionUniverse,theChunk,theOffset,1,&theChunk,&theOffset))		/* make a new selection */
				{
				((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=thePosition-lastPosition;
				((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset=numBytes;
				}
			else
				{
				fail=TRUE;
				}
			}
		/* now theChunk/theOffset point to the inserted/modified selection element */
		/* run through remaining selections absorbing them until we pass the end */
		if(!fail&&theChunk)
			{
			modifiedChunk=theChunk;
			modifiedOffset=theOffset;
			theOffset++;									/* move to the next one */
			if(theOffset>=theChunk->totalElements)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			done=FALSE;
			elementsToDelete=0;
			while(theChunk&&!done)
				{
				startPosition=lastPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
				endPosition=startPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
				lastPosition=endPosition;
				if(startPosition<thePosition+numBytes)				/* see if this selection begins in the added selection */
					{
					if(endPosition>thePosition+numBytes)
						{
						((SELECTIONELEMENT *)modifiedChunk->data)[modifiedOffset].endOffset=endPosition-thePosition;
						done=TRUE;
						}
					elementsToDelete++;
					}
				else
					{
					((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=startPosition-(thePosition+numBytes);
					done=TRUE;
					}
				theOffset++;									/* move to the next one */
				if(theOffset>=theChunk->totalElements)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			if(elementsToDelete)
				{
				modifiedOffset++;									/* move to the next one */
				if(modifiedOffset>=modifiedChunk->totalElements)
					{
					modifiedChunk=modifiedChunk->nextHeader;
					modifiedOffset=0;
					}
				DeleteUniverseSelection(theSelectionUniverse,modifiedChunk,modifiedOffset,elementsToDelete,&theChunk,&theOffset);
				}
			}
		}
	return(!fail);
}

void EditorFixSelections(SELECTIONUNIVERSE *theUniverse,UINT32 startOffset,UINT32 oldEndOffset,UINT32 newEndOffset)
/* given a change at startOffset, run through the given selection list, and fix it up
 */
{
	UINT32
		lastPosition;
	UINT32
		movingPosition;
	UINT32
		startPosition,
		endPosition;
	ARRAYCHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	INT32
		numBytesAdded;											/* relative distance to move */
	BOOLEAN
		done;

	theOffset=0;
	theChunk=theUniverse->selectionChunks.firstChunkHeader;
	lastPosition=0;											/* this is used to keep track of the absolute selection positions */
	done=FALSE;
	while(theChunk&&!done)
		{
		startPosition=lastPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
		endPosition=startPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
		if(endPosition>=startOffset)						/* find a selection that is in, or after the replacement */
			{
			numBytesAdded=newEndOffset-oldEndOffset;		/* start by trying to move the amount of bytes that were added/deleted */
			movingPosition=lastPosition;					/* this keeps track of where we are while deleting selections */
			while(theChunk&&!done)
				{
				startPosition=movingPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
				endPosition=startPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
				if(startPosition<oldEndOffset)				/* see if it starts before, or during the replacement */
					{
					if(startPosition>=startOffset)			/* selection starts in replacement, ends in or after */
						{
						if(endPosition<=oldEndOffset)		/* selection begins and ends in replacement (so delete it) */
							{
							DeleteUniverseSelection(theUniverse,theChunk,theOffset,1,&theChunk,&theOffset);
							}
						else								/* selection begins in replacement, ends after, so reduce it, and be done */
							{
							((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=newEndOffset-lastPosition;	/* selection starts in replacement, ends after, so move over start, adjust end, and be done */
							((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset-=(oldEndOffset-startPosition);
							done=TRUE;
							}
						}
					else									/* selection starts before replacement, ends in or after */
						{
						if(endPosition<=oldEndOffset)		/* selection begins before, and ends during replacement, so move end back to start of replacement */
							{
							((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset=startOffset-(lastPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset);
							theOffset++;
							if(theOffset>=theChunk->totalElements)
								{
								theChunk=theChunk->nextHeader;
								theOffset=0;
								}
							lastPosition=startOffset;		/* update last known position in selection list */
							}
						else								/* selection begins before, and ends after replacement, so reduce span, and be done */
							{
							((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset+=numBytesAdded;
							done=TRUE;
							}
						}
					movingPosition=endPosition;				/* update current position in selection list */
					}
				else										/* selection starts after the replacement, so move it over, and be done */
					{
					((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset=startPosition+numBytesAdded-lastPosition;
					done=TRUE;
					}
				}
			}
		else
			{
			lastPosition=endPosition;
			theOffset++;
			if(theOffset>=theChunk->totalElements)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}

	if(theUniverse->cursorPosition>=startOffset)			/* if cursor was before the change, then do not move it */
		{
		if(theUniverse->cursorPosition>oldEndOffset)		/* see if cursor was past changed area (move it relatively) */
			{
			theUniverse->cursorPosition=(theUniverse->cursorPosition+newEndOffset)-oldEndOffset;
			}
		else
			{
			theUniverse->cursorPosition=newEndOffset;		/* if cursor in middle of change, put it at the end */
			}
		}
}

void FixEditorUniverseSelections(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 oldEndOffset,UINT32 newEndOffset)
/* given a change at startOffset, run through the given selection lists for theUniverse,
 * and fix them up
 */
{
	MARKLIST
		*theMark;

	EditorFixSelections(theUniverse->selectionUniverse,startOffset,oldEndOffset,newEndOffset);
	EditorFixSelections(theUniverse->auxSelectionUniverse,startOffset,oldEndOffset,newEndOffset);
	theMark=theUniverse->theMarks;
	while(theMark)
		{
		EditorFixSelections(theMark->selectionUniverse,startOffset,oldEndOffset,newEndOffset);	/* keep all the marks updated too */
		theMark=theMark->nextMark;
		}
}

void EditorHomeViewToPositionLenient(EDITORVIEW *theView,UINT32 thePosition)
/* move theView so that thePosition is displayed
 * attempt to move theView as little as possible to get thePosition displayed
 */
{
	UINT32
		theLine,
		newTopLine,
		topLine,
		numLines,
		numPixels;
	INT32
		theLeftPixel,
		newLeftPixel,
		leftPixel;
	CHUNKHEADER
		*theChunk;
	UINT32
		theLineOffset,
		theChunkOffset;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theChunkOffset);
	GetEditorViewTextToGraphicPosition(theView,theChunk,theChunkOffset,theLineOffset,&theLeftPixel);
	if(theLine<topLine||!numLines)								/* line is off the top, or no lines in display */
		{
		newTopLine=theLine;										/* so set top to new line */
		}
	else
		{
		if(theLine>=topLine+numLines)							/* line is off the bottom, so place it at the bottom */
			{
			newTopLine=theLine-numLines+1;
			}
		else
			{
			newTopLine=topLine;
			}
		}

/* we do more stuff with the left pixel to make scrolling better */

	if(theLeftPixel<leftPixel)
		{
		newLeftPixel=theLeftPixel-(theLeftPixel%HORIZONTALSCROLLTHRESHOLD);
		}
	else
		{
		if(theLeftPixel>=leftPixel+numPixels)
			{
			newLeftPixel=((theLeftPixel-numPixels)/HORIZONTALSCROLLTHRESHOLD+1)*HORIZONTALSCROLLTHRESHOLD;
			}
		else
			{
			newLeftPixel=leftPixel;
			}
		}
	if((newTopLine!=topLine)||(newLeftPixel!=leftPixel))
		{
		SetViewTopLeft(theView,newTopLine,newLeftPixel);
		}
}

void EditorHomeViewToPositionStrict(EDITORVIEW *theView,UINT32 thePosition)
/* move theView so that thePosition is displayed as close to 1/4 the way down
 * the screen as possible.
 * theLeft margin is made 0 if possible, othersize it is set so that thePosition
 * is as far right as possible
 */
{
	UINT32
		theLine,
		newTopLine,
		topLine,
		numLines,
		numPixels;
	INT32
		theLeftPixel,
		newLeftPixel,
		leftPixel;
	CHUNKHEADER
		*theChunk;
	UINT32
		theLineOffset,
		theChunkOffset;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theChunkOffset);
	GetEditorViewTextToGraphicPosition(theView,theChunk,theChunkOffset,theLineOffset,&theLeftPixel);

	newTopLine=numLines/4;
	if(theLine<newTopLine)
		{
		newTopLine=0;
		}
	else
		{
		newTopLine=theLine-newTopLine;
		if((newTopLine+numLines)>(theView->editorUniverse->textUniverse->totalLines+1))	/* try not to display empty space at end */
			{
			if((theView->editorUniverse->textUniverse->totalLines+1)>numLines)
				{
				newTopLine=(theView->editorUniverse->textUniverse->totalLines+1)-numLines;
				}
			else
				{
				newTopLine=0;
				}
			}
		}

/* we do more stuff with the left pixel to make scrolling better */

	if(theLeftPixel<leftPixel)
		{
		newLeftPixel=theLeftPixel-(theLeftPixel%HORIZONTALSCROLLTHRESHOLD);
		}
	else
		{
		if(theLeftPixel>=leftPixel+numPixels)
			{
			newLeftPixel=((theLeftPixel-numPixels)/HORIZONTALSCROLLTHRESHOLD+1)*HORIZONTALSCROLLTHRESHOLD;
			}
		else
			{
			newLeftPixel=leftPixel;
			}
		}
	if((newTopLine!=topLine)||(newLeftPixel!=leftPixel))
		{
		SetViewTopLeft(theView,newTopLine,newLeftPixel);
		}
}

void EditorHomeViewToPositionSemiStrict(EDITORVIEW *theView,UINT32 thePosition)
/* if thePosition is displayed, do nothing, however if it is not displayed,
 * do strict movement to try to place thePosition 1/4 the way down the screen
 */
{
	UINT32
		theLine,
		newTopLine,
		topLine,
		numLines,
		numPixels;
	INT32
		theLeftPixel,
		newLeftPixel,
		leftPixel;
	CHUNKHEADER
		*theChunk;
	UINT32
		theLineOffset,
		theChunkOffset;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theChunkOffset);
	GetEditorViewTextToGraphicPosition(theView,theChunk,theChunkOffset,theLineOffset,&theLeftPixel);

	if(theLine<topLine||theLine>=topLine+numLines)				/* line is off the view */
		{
		newTopLine=numLines/4;
		if(theLine<newTopLine)
			{
			newTopLine=0;
			}
		else
			{
			newTopLine=theLine-newTopLine;
			if((newTopLine+numLines)>(theView->editorUniverse->textUniverse->totalLines+1))	/* try not to display empty space at end */
				{
				if((theView->editorUniverse->textUniverse->totalLines+1)>numLines)
					{
					newTopLine=(theView->editorUniverse->textUniverse->totalLines+1)-numLines;
					}
				else
					{
					newTopLine=0;
					}
				}
			}
		}
	else
		{
		newTopLine=topLine;
		}

/* we do more stuff with the left pixel to make scrolling better */

	if(theLeftPixel<leftPixel)
		{
		newLeftPixel=theLeftPixel-(theLeftPixel%HORIZONTALSCROLLTHRESHOLD);
		}
	else
		{
		if(theLeftPixel>=leftPixel+numPixels)
			{
			newLeftPixel=((theLeftPixel-numPixels)/HORIZONTALSCROLLTHRESHOLD+1)*HORIZONTALSCROLLTHRESHOLD;
			}
		else
			{
			newLeftPixel=leftPixel;
			}
		}
	if((newTopLine!=topLine)||(newLeftPixel!=leftPixel))
		{
		SetViewTopLeft(theView,newTopLine,newLeftPixel);
		}
}

void EditorHomeViewToCursorLenient(EDITORVIEW *theView,BOOLEAN useEnd)
/* move theView so that the start (or end) of the selection is displayed
 * attempt to move theView as little as possible to get the position displayed
 */
{
	UINT32
		startPosition,
		endPosition;

	GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);
	EditorHomeViewToPositionLenient(theView,useEnd?endPosition:startPosition);
}

void EditorHomeViewToCursorStrict(EDITORVIEW *theView,BOOLEAN useEnd)
/* strictly home the view to the selection start (or end)
 */
{
	UINT32
		startPosition,
		endPosition;

	GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);
	EditorHomeViewToPositionStrict(theView,useEnd?endPosition:startPosition);
}

void EditorHomeViewToCursorSemiStrict(EDITORVIEW *theView,BOOLEAN useEnd)
/* semi strictly home the view to the selection start (or end)
 */
{
	UINT32
		startPosition,
		endPosition;

	GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);
	EditorHomeViewToPositionSemiStrict(theView,useEnd?endPosition:startPosition);
}

void EditorVerticalScroll(EDITORVIEW *theView,INT32 amountToScroll)
/* scroll the view vertically by amountToScroll
 * amountToScroll<0, then scroll towards lower line numbers
 * the scroll will pin to the edges of the view
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	if(amountToScroll>=0)
		{
		if((topLine+amountToScroll)<theView->editorUniverse->textUniverse->totalLines)
			{
			topLine+=amountToScroll;
			}
		else
			{
			topLine=theView->editorUniverse->textUniverse->totalLines;
			}
		}
	else
		{
		amountToScroll=-amountToScroll;
		if(topLine>amountToScroll)
			{
			topLine-=amountToScroll;
			}
		else
			{
			topLine=0;
			}
		}
	SetViewTopLeft(theView,topLine,leftPixel);
}

void EditorVerticalScrollByPages(EDITORVIEW *theView,INT32 pagesToScroll)
/* page theView vertically by pagesToScroll
 * if pagesToScroll<0, scroll towards lower line numbers in the view
 * the scroll will pin to the edges of the view
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	EditorVerticalScroll(theView,pagesToScroll*(INT32)numLines);
}

void EditorHorizontalScroll(EDITORVIEW *theView,INT32 amountToScroll)
/* scroll the view horizontally by amountToScroll pixels
 * amountToScroll<0, then scroll towards the left
 * the scroll will pin to the edges of the view
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	if(amountToScroll>=0)
		{
		if(leftPixel+amountToScroll<HORIZONTALMAXPIXELS)
			{
			leftPixel+=amountToScroll;
			}
		else
			{
			leftPixel=HORIZONTALMAXPIXELS;
			}
		}
	else
		{
		if(leftPixel+amountToScroll>0)
			{
			leftPixel+=amountToScroll;
			}
		else
			{
			leftPixel=0;
			}
		}
	SetViewTopLeft(theView,topLine,leftPixel);
}

void EditorHorizontalScrollByPages(EDITORVIEW *theView,INT32 pagesToScroll)
/* page theView horizontally by pagesToScroll
 * if pagesToScroll<0, scroll towards the left
 * the scroll will pin to the edges of the view
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	EditorHorizontalScroll(theView,pagesToScroll*(INT32)numPixels);
}

void EditorStartSelectionChange(EDITORUNIVERSE *theUniverse)
/* the selection in theUniverse is about to be changed in some way, do whatever is needed
 * NOTE: a change in the selection could mean something as simple as a cursor position change
 * NOTE also: selection changes cannot be nested, and nothing except the selection is allowed
 * to be altered during a selection change
 */
{
	ViewsStartSelectionChange(theUniverse);			/* tell all the views of this universe that the selection is changing */
}

void EditorEndSelectionChange(EDITORUNIVERSE *theUniverse)
/* the selection in theUniverse has finished being changed
 * do whatever needs to be done
 */
{
	ViewsEndSelectionChange(theUniverse);			/* tell all the views of this universe that the selection is changed */
	theUniverse->haveStartX=theUniverse->haveEndX=FALSE;	/* after selection changed, X offsets are no longer valid */
}

void EditorStartTextChange(EDITORUNIVERSE *theUniverse)
/* the text in theUniverse is about to be changed in some way, do whatever is needed
 * NOTE: when the text changes, it is possible that the selection information
 * will also change, so that updates will be handled correctly
 */
{
	ViewsStartTextChange(theUniverse);
}

void EditorEndTextChange(EDITORUNIVERSE *theUniverse)
/* the text in theUniverse has finished being changed, and the views have been
 * updated, do whatever needs to be done
 */
{
	ViewsEndTextChange(theUniverse);
	theUniverse->haveStartX=theUniverse->haveEndX=FALSE;	/* after text has been replaced, X offsets are no longer valid */
}

void EditorInvalidateViews(EDITORUNIVERSE *theUniverse)
/* when something happens that requires an update of the entire editor universe, call this, and
 * all the views will be invalidated, and homed to line 0
 */
{
	EDITORVIEW
		*currentView;
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, invalidating each one */
		{
		GetEditorViewTextInfo(currentView,&topLine,&numLines,&leftPixel,&numPixels);
		InvalidateViewPortion(currentView,0,numLines,0,numPixels);
		SetEditorViewTopLine(currentView,0);
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

static BOOLEAN
	rte_Multiple;														/* tells if in the middle of more than one replacement at a time */

static void AdjustViewForSingleChange(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine1,UINT32 endLine2,INT32 updateStartX)
/* scroll, and invalidate theView based on the given update locations
 * the update locations are given relative to the text, not the current view
 * NOTE: this attempts to leave the view pointed to the same lines of
 * text, even if those lines may have changed absolute position because of the
 * insertion, or deletion of lines above them
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	UINT32
		invalToLine;
	UINT32
		invalToPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	if(startLine<topLine+numLines)										/* if change happened in text after the view, we need do nothing to adjust the view */
		{
		if(endLine1>=topLine)											/* make sure change did not happen entirely before the view */
			{
			/* make start and end lines relative to the view */

			if(startLine<topLine)										/* if start is before the view, then move the view to the start line */
				{
				SetEditorViewTopLine(theView,startLine);				/* actually adjust the top line of the view */
				endLine1-=topLine;										/* fake this, so we have an idea of what WAS displayed */
				endLine2-=startLine;									/* this is the view relative place where we end now */
				startLine=0;											/* start line was forced to top of view */
				updateStartX=0;											/* start at upper left edge of view */
				}
			else
				{
				startLine-=topLine;										/* make line numbers relative to the view */
				endLine1-=topLine;
				endLine2-=topLine;
				}
			if((endLine1<numLines)&&(endLine2<numLines))				/* only scroll if both ends are on the view */
				{
				if(endLine1==endLine2)									/* net number of lines is the same, so invalidate to the end */
					{
					invalToLine=endLine1+1;								/* invalidate to end of line */
					invalToPixel=0;
					}
				else													/* start and end lines were not the same, so scroll vertically */
					{
					invalToLine=Min(endLine1,endLine2)+1;				/* invalidate from start to just before this line */
					invalToPixel=0;
					ScrollViewPortion(theView,invalToLine,numLines,endLine2-endLine1,0,numPixels,0);
					}
				}
			else
				{
				invalToLine=numLines;									/* invalidate to the very bottom */
				invalToPixel=0;
				}
			if(updateStartX>leftPixel)
				{
				updateStartX-=leftPixel;
				}
			else
				{
				updateStartX=0;
				}
			if(startLine<invalToLine)									/* see if we can inval to end of first line */
				{
				if(updateStartX<numPixels)								/* if off the edge, then do not invalidate */
					{
					InvalidateViewPortion(theView,startLine,startLine+1,updateStartX,numPixels);
					}
				startLine++;
				updateStartX=0;
				if(startLine<invalToLine)								/* find out how many complete lines to update */
					{
					InvalidateViewPortion(theView,startLine,invalToLine,0,numPixels);
					}
				}
			if(updateStartX<invalToPixel)								/* do invalidations on final line */
				{
				InvalidateViewPortion(theView,invalToLine,invalToLine+1,updateStartX,invalToPixel);
				}
			}
		else
			{
			SetEditorViewTopLine(theView,topLine+endLine2-endLine1);		/* adjust what we believe the top line is (change does not affect view) */
			}
		}
}

static void AdjustViewForMultipleChange(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine1,UINT32 endLine2,INT32 updateStartX)
/* invalidate theView based on the given update locations
 * the update locations are given relative to the text, not the current view
 * NOTE: this attempts to leave the view pointed to the same lines of
 * text, even if those lines may have changed absolute position because of the
 * insertion, or deletion of lines above them
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	if(startLine<topLine+numLines)					/* make sure all the changes did not happen completely after the view */
		{
		if(endLine1>=topLine)
			{
			if(startLine<topLine)										/* if start is before the view, then move the view to the start line */
				{
				SetEditorViewTopLine(theView,startLine);				/* actually adjust the top line of the view to the new start line */
				InvalidateViewPortion(theView,0,numLines,0,numPixels);	/* invalidate the whole thing */
				}
			else
				{
				if((updateStartX-=leftPixel)<0)							/* make X view relative */
					{
					updateStartX=0;										/* if before left edge of view, place at left edge of view */
					}
				InvalidateViewPortion(theView,startLine-topLine,startLine-topLine+1,updateStartX,numPixels);	/* invalidate to the end of the start line */
				if((endLine1==endLine2)&&(endLine1<topLine+numLines))
					{
					if(endLine1>startLine)								/* make sure start and end not on same line */
						{
						InvalidateViewPortion(theView,startLine-topLine+1,endLine1+1,0,numPixels);	/* invalidate all lines below up to and including the end line */
						}
					}
				else
					{
					if(startLine-topLine+1<numLines)
						{
						InvalidateViewPortion(theView,startLine-topLine+1,numLines,0,numPixels);	/* invalidate to the end of the view */
						}
					}
				}
			}
		else
			{
			SetEditorViewTopLine(theView,topLine+endLine2-endLine1);	/* adjust what we believe the top line is (change does not affect view) */
			}
		}
}

static void AdjustViewForChange(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine1,UINT32 endLine2,INT32 updateStartX)
/* See what is changing in theView, and update it accordingly
 * NOTE: this attempts to leave the view pointed to the same lines of
 * text, even if those lines may have changed absolute position because of the
 * insertion, or deletion of lines above them
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;

	if(!rte_Multiple)
		{
		AdjustViewForSingleChange(theView,startLine,endLine1,endLine2,updateStartX);
		}
	else
		{
		AdjustViewForMultipleChange(theView,startLine,endLine1,endLine2,updateStartX);
		}
	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	LineToChunkPosition(theView->editorUniverse->textUniverse,topLine,&theChunk,&theOffset,&(theView->startPosition));			/* adjust these after the view changes */
	LineToChunkPosition(theView->editorUniverse->textUniverse,topLine+numLines,&theChunk,&theOffset,&(theView->endPosition));
	rte_Multiple=TRUE;
}

void EditorStartReplace(EDITORUNIVERSE *theUniverse)
/* call this when about to begin replacing editor text
 * it will begin the process of remembering what has changed, so that the update
 * can happen at the end
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	EDITORVIEW
		*currentView;

	EditorStartTextChange(theUniverse);					/* we are going to change text in an edit universe (turn off cursors, and selections, and the like) */

	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, and collect information about them that will help speed the replace */
		{
		GetEditorViewTextInfo(currentView,&topLine,&numLines,&leftPixel,&numPixels);
		LineToChunkPosition(theUniverse->textUniverse,topLine,&theChunk,&theOffset,&(currentView->startPosition));	/* get the text position of the first character in the view */
		LineToChunkPosition(theUniverse->textUniverse,topLine+numLines,&theChunk,&theOffset,&(currentView->endPosition));	/* get the text position of the first character just off the view */
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
	rte_Multiple=FALSE;
}

BOOLEAN ReplaceEditorChunks(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 endOffset,CHUNKHEADER *textChunk,UINT32 textOffset,UINT32 numBytes)
/* replace the text between startOffset and endOffset with numBytes of text at textChunk/textOffset
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		*startLineChunk,
		*unusedChunk;
	UINT32
		startLineOffset,
		unusedOffset;
	EDITORVIEW
		*currentView;
	UINT32
		unused,
		startLine,
		endLine,
		startCharOffset,
		oldNumLines;
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	BOOLEAN
		haveViewNeedingUpdate,
		fail;

	fail=FALSE;
	haveViewNeedingUpdate=FALSE;
	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, see if any need attention */
		{
		if(endOffset>=currentView->startPosition)		/* make sure change is not completely before the start, if it is, we can ignore it */
			{
			if(startOffset<=currentView->endPosition)	/* make sure change is not completely after the end, if it is, we can ignore it */
				{
				if(!haveViewNeedingUpdate)
					{
					PositionToLinePosition(theUniverse->textUniverse,startOffset,&startLine,&startCharOffset,&startLineChunk,&startLineOffset);
					PositionToLinePosition(theUniverse->textUniverse,endOffset,&endLine,&unused,&unusedChunk,&unusedOffset);
					haveViewNeedingUpdate=TRUE;			/* remember this stuff */
					}
				GetEditorViewTextInfo(currentView,&topLine,&numLines,&leftPixel,&numPixels);
				if(startLine>=topLine)
					{
					GetEditorViewTextToLimitedGraphicPosition(currentView,startLineChunk,startLineOffset,startCharOffset,leftPixel+numPixels,&(currentView->updateStartX));
					}
				else
					{
					currentView->updateStartX=0;		/* change begins before start line, so X is 0 */
					}
				}
			}
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}

	if(RegisterUndoDelete(theUniverse,startOffset,endOffset-startOffset))		/* remember what we are about to delete */
		{
		oldNumLines=theUniverse->textUniverse->totalLines;	/* use this to find out how many lines were added/removed */
		if(DeleteUniverseText(theUniverse->textUniverse,startOffset,endOffset-startOffset))
			{
			theUniverse->dirty=TRUE;									/* this makes it dirty */
			if(InsertUniverseChunks(theUniverse->textUniverse,startOffset,textChunk,textOffset,numBytes))
				{
				if(!RegisterUndoInsert(theUniverse,startOffset,numBytes))
					{
					fail=TRUE;											/* we do not cope well if this fails, just keep trying */
					}
				}
			else
				{
				fail=TRUE;
				numBytes=0;												/* we have failed to insert any bytes */
				}

			currentView=theUniverse->firstView;
			while(currentView)										/* walk through all views, see if any need attention */
				{
				if(endOffset>=currentView->startPosition)			/* make sure change is not completely before the start, if it is, we just have to update the topline, and the start and end positions */
					{
					if(startOffset<=currentView->endPosition)		/* make sure change is not completely after the end, if it is, we can ignore it */
						{
						AdjustViewForChange(currentView,startLine,endLine,(endLine+theUniverse->textUniverse->totalLines)-oldNumLines,currentView->updateStartX);
						}
					}
				else
					{
					currentView->startPosition-=(endOffset-startOffset);	/* subtract for bytes deleted */
					currentView->startPosition+=numBytes;					/* add for bytes inserted */
					currentView->endPosition-=(endOffset-startOffset);		/* subtract for bytes deleted */
					currentView->endPosition+=numBytes;						/* add for bytes inserted */
					GetEditorViewTextInfo(currentView,&topLine,&numLines,&leftPixel,&numPixels);
					SetEditorViewTopLine(currentView,topLine+theUniverse->textUniverse->totalLines-oldNumLines);	/* adjust what we believe the top line is (change does not affect view) */
					}
				currentView=currentView->nextUniverseView;			/* locate next view of this edit universe */
				}

			FixEditorUniverseSelections(theUniverse,startOffset,endOffset,startOffset+numBytes);
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}

BOOLEAN ReplaceEditorText(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 endOffset,UINT8 *theText,UINT32 numBytes)
/* replace the text between startOffset and endOffset with numBytes of theText
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		psuedoChunk;														/* used to create a chunk that points to theText */

	psuedoChunk.previousHeader=psuedoChunk.nextHeader=NULL;					/* set up fake chunk header, so we can call replace chunk routine */
	psuedoChunk.data=theText;
	psuedoChunk.totalBytes=numBytes;
	psuedoChunk.totalLines=0;												/* this does not have to be set accurately */
	return(ReplaceEditorChunks(theUniverse,startOffset,endOffset,&psuedoChunk,0,numBytes));
}

BOOLEAN ReplaceEditorFile(EDITORUNIVERSE *theUniverse,UINT32 startOffset,UINT32 endOffset,char *thePath)
/* replace the text between startOffset and endOffset with the text at thePath
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		psuedoChunk;														/* used to create a chunk that points to theText */
	UINT8
		*theBuffer;
	EDITORFILE
		*theFile;
	UINT32
		numRead;
	BOOLEAN
		done,
		fail;

	fail=FALSE;
	if(theFile=OpenEditorReadFile(thePath))								/* open the file to load from */
		{
		if(theBuffer=(UINT8 *)MNewPtr(LOADSAVEBUFFERSIZE))				/* create a buffer to read into */
			{
			psuedoChunk.previousHeader=psuedoChunk.nextHeader=NULL;		/* set up fake chunk header, so we can call replace chunk routine */
			psuedoChunk.data=theBuffer;
			psuedoChunk.totalLines=0;									/* this does not have to be set accurately */
			done=FALSE;
			while(!done&&!fail)											/* read until we reach the end of the file */
				{
				if(ReadEditorFile(theFile,theBuffer,LOADSAVEBUFFERSIZE,&numRead))	/* inhale as many bytes as we can fit into the buffer */
					{
					if(numRead)											/* if no bytes were read, but we did not fail, then we are at the end of the file */
						{
						psuedoChunk.totalBytes=numRead;
						if(ReplaceEditorChunks(theUniverse,startOffset,endOffset,&psuedoChunk,0,numRead))	/* take the bytes we got, and insert them into the universe */
							{
							startOffset+=numRead;						/* move these for next time */
							endOffset=startOffset;
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						done=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				}
			MDisposePtr(theBuffer);
			}
		else
			{
			fail=TRUE;
			}
		CloseEditorFile(theFile);
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}

void EditorEndReplace(EDITORUNIVERSE *theUniverse)
/* when a replacement has been completed, call this, and it will
 * update all the views as needed
 */
{
	UpdateEditorWindows();													/* update the windows, making sure to update any view that has been invalidated */
	EditorEndTextChange(theUniverse);
}

/* --------------------------------------------------------------------------------------------------------------------------
 * editor medium level functions (used by high level)
 */

void DeleteAllSelectedText(EDITORUNIVERSE *theEditorUniverse)
/* delete all the selected text from the passed editor universe, place the
 * cursor at the start of the first selection deleted
 * EditorStartReplace and EditorEndReplace should be called externally
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	ARRAYCHUNKHEADER
		*theChunk;
	UINT32
		finalCursorPosition,
		startOffset,
		endOffset;
	BOOLEAN
		fail;

	theSelectionUniverse=theEditorUniverse->selectionUniverse;
	fail=FALSE;
	if(theSelectionUniverse->selectionChunks.totalElements)
		{
		finalCursorPosition=((SELECTIONELEMENT *)theSelectionUniverse->selectionChunks.firstChunkHeader->data)[0].startOffset;
		while((theChunk=theSelectionUniverse->selectionChunks.firstChunkHeader)&&!fail)
			{
			startOffset=((SELECTIONELEMENT *)theChunk->data)[0].startOffset;
			endOffset=startOffset+((SELECTIONELEMENT *)theChunk->data)[0].endOffset;
			fail=!ReplaceEditorText(theEditorUniverse,startOffset,endOffset,NULL,0);
			}
		theSelectionUniverse->cursorPosition=finalCursorPosition;		/* move the cursor to the start of the deleted text (even if we failed) */
		}

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to delete: %.256s\n",errorDescription);
		}
}

void GetSelectionEndPositions(SELECTIONUNIVERSE *theUniverse,UINT32 *startPosition,UINT32 *endPosition)
/* return the start and end absolute text positions of the passed selection universe
 * if there are no selections, then the cursor position is returned as both the start and the end
 */
{
	ARRAYCHUNKHEADER
		*theChunk;
	UINT32
		theOffset,
		currentOffset;

	if(theUniverse->selectionChunks.totalElements)
		{
		if(theChunk=theUniverse->selectionChunks.firstChunkHeader)
			{
			theOffset=0;
			currentOffset=0;
			(*startPosition)=((SELECTIONELEMENT *)theChunk->data)[0].startOffset;
			while(theChunk)
				{
				currentOffset+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
				theOffset++;
				if(theOffset>=theChunk->totalElements)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			(*endPosition)=currentOffset;
			}
		}
	else
		{
		(*startPosition)=(*endPosition)=theUniverse->cursorPosition;
		}
}

void EditorGetSelectionInfo(EDITORUNIVERSE *theEditorUniverse,UINT32 *startPosition,UINT32 *endPosition,UINT32 *startLine,UINT32 *endLine,UINT32 *startLinePosition,UINT32 *endLinePosition,UINT32 *totalSegments,UINT32 *totalSpan)
/* get useful information about the current selection
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	CHUNKHEADER
		*textChunk;
	ARRAYCHUNKHEADER
		*theChunk;
	UINT32
		theOffset,
		textOffset,
		currentOffset;

	theSelectionUniverse=theEditorUniverse->selectionUniverse;
	(*totalSpan)=0;
	if((*totalSegments)=theSelectionUniverse->selectionChunks.totalElements)
		{
		if(theChunk=theSelectionUniverse->selectionChunks.firstChunkHeader)
			{
			theOffset=0;
			currentOffset=0;
			(*startPosition)=((SELECTIONELEMENT *)theChunk->data)[0].startOffset;
			while(theChunk)
				{
				(*totalSpan)+=((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
				currentOffset+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
				theOffset++;
				if(theOffset>=theChunk->totalElements)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			(*endPosition)=currentOffset;
			}
		}
	else
		{
		(*startPosition)=(*endPosition)=theSelectionUniverse->cursorPosition;
		}
	PositionToLinePosition(theEditorUniverse->textUniverse,*startPosition,startLine,startLinePosition,&textChunk,&textOffset);
	PositionToLinePosition(theEditorUniverse->textUniverse,*endPosition,endLine,endLinePosition,&textChunk,&textOffset);
}

UINT8 *EditorNextSelectionToBuffer(EDITORUNIVERSE *theEditorUniverse,UINT32 *currentPosition,ARRAYCHUNKHEADER **currentChunk,UINT32 *currentOffset,UINT32 additionalLength,UINT32 *actualLength,BOOLEAN *atEnd)
/* create a buffer the size of the current selection in theEditorUniverse + additionalLength
 * then read the bytes of the selection into the buffer, returning a pointer
 * to the buffer. The buffer must be disposed of by the caller using MDisposePtr.
 * If there is a problem, set the error, and return NULL
 * If currentChunk is passed in pointing to NULL, then use the first selection of
 * theEditorUniverse, and reset currentPosition
 * If atEnd is TRUE, there are no more selections in the list
 * If there are no selections, return NULL, and set atEnd to TRUE (do not SetError)
 * NOTE: if NULL comes back, and atEnd is FALSE, it indicates an error
 */
{
	CHUNKHEADER
		*sourceChunk;
	UINT32
		sourceChunkOffset;
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	UINT32
		startPosition,
		endPosition;
	UINT32
		numBytes;
	UINT8
		*theBuffer;

	theSelectionUniverse=theEditorUniverse->selectionUniverse;
	(*atEnd)=FALSE;
	if(!(*currentChunk))
		{
		*currentPosition=0;
		*currentOffset=0;
		*currentChunk=theSelectionUniverse->selectionChunks.firstChunkHeader;
		}
	if(*currentChunk)	/* see if there are selections */
		{
		startPosition=(*currentPosition)+((SELECTIONELEMENT *)(*currentChunk)->data)[*currentOffset].startOffset;
		endPosition=startPosition+((SELECTIONELEMENT *)(*currentChunk)->data)[*currentOffset].endOffset;

		PositionToChunkPosition(theEditorUniverse->textUniverse,startPosition,&sourceChunk,&sourceChunkOffset);	/* point to the start of the selection in the source data */
		numBytes=endPosition-startPosition;
		(*actualLength)=numBytes+additionalLength;
		if(theBuffer=(UINT8 *)MNewPtr(*actualLength))
			{
			if(ExtractUniverseText(theEditorUniverse->textUniverse,sourceChunk,sourceChunkOffset,theBuffer,numBytes,&sourceChunk,&sourceChunkOffset))
				{
				(*currentPosition)=endPosition;				/* update position */

				(*currentOffset)++;
				if((*currentOffset)>=(*currentChunk)->totalElements)
					{
					(*currentChunk)=(*currentChunk)->nextHeader;
					*currentOffset=0;
					}
				if(!(*currentChunk))						/* see if at the end now */
					{
					(*atEnd)=TRUE;
					}
				return(theBuffer);
				}
			MDisposePtr(theBuffer);
			}
		}
	else
		{
		(*atEnd)=TRUE;
		}
	return(NULL);
}

/* cursor position/movement routines
 */

void EditorDeleteSelection(EDITORUNIVERSE *theEditorUniverse)
/* delete all the selected text
 */
{
	EditorStartReplace(theEditorUniverse);
	BeginUndoGroup(theEditorUniverse);
	DeleteAllSelectedText(theEditorUniverse);
	StrictEndUndoGroup(theEditorUniverse);
	EditorEndReplace(theEditorUniverse);
}

static BOOLEAN IsWordSpace(UINT8 theChar)
/* returns TRUE if theChar is considered to be the space between words
 */
{
	return((BOOLEAN)wordSpaceTable[theChar]);
}

static UINT32 NextCharLeft(EDITORVIEW *theView,UINT32 thePosition)
/* return the position one character to the left of thePosition
 */
{
	if(thePosition)
		{
		return(thePosition-1);
		}
	else
		{
		return(thePosition);
		}
}

static UINT32 NextCharRight(EDITORVIEW *theView,UINT32 thePosition)
/* return the position one character to the right of thePosition
 */
{
	if(thePosition<theView->editorUniverse->textUniverse->totalBytes)
		{
		return(thePosition+1);
		}
	else
		{
		return(thePosition);
		}
}

static UINT32 NextWordLeft(EDITORVIEW *theView,UINT32 thePosition)
/* return the position one word to the left of thePosition
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	if(thePosition)
		{
		PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition-1,&theChunk,&theOffset);		/* look at the character in question */
		found=FALSE;
		if(theChunk)
			{
			thePosition--;
			if(IsWordSpace(theChunk->data[theOffset]))
				{
				found=TRUE;
				}
			}
		while(theChunk&&!found)
			{
			if(theOffset)
				{
				theOffset--;
				}
			else
				{
				if(theChunk=theChunk->previousHeader)
					{
					theOffset=theChunk->totalBytes-1;
					}
				}
			if(theChunk)
				{
				if(IsWordSpace(theChunk->data[theOffset]))
					{
					found=TRUE;
					}
				else
					{
					thePosition--;
					}
				}
			}
		}
	return(thePosition);
}

static UINT32 NextWordRight(EDITORVIEW *theView,UINT32 thePosition)
/* return the position one word to the right of thePosition
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition,&theChunk,&theOffset);
	found=FALSE;
	if(theChunk)
		{
		thePosition++;
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	while(theChunk&&!found)
		{
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			thePosition++;
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(thePosition);
}

static UINT32 NextLineUp(EDITORVIEW *theView,UINT32 thePosition,BOOLEAN *haveX,INT32 *desiredX)
/* return the position above thePosition
 */
{
	UINT32
		newPosition;
	UINT32
		theLine,
		theLineOffset,
		temp;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;

	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theOffset);
	if(theLine)
		{
		if(!(*haveX))
			{
			GetEditorViewTextToGraphicPosition(theView,theChunk,theOffset,theLineOffset,desiredX);
			*haveX=TRUE;
			}
		LineToChunkPosition(theView->editorUniverse->textUniverse,theLine-1,&theChunk,&theOffset,&newPosition);
		GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,*desiredX,&theLineOffset,&temp);
		return(newPosition+theLineOffset);
		}
	else
		{
		return(thePosition);			/* there is no line above this one */
		}
}

static UINT32 NextLineDown(EDITORVIEW *theView,UINT32 thePosition,BOOLEAN *haveX,INT32 *desiredX)
/* return the position below thePosition
 */
{
	UINT32
		newPosition;
	UINT32
		theLine,
		theLineOffset,
		temp;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;

	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theOffset);
	if(theLine<theView->editorUniverse->textUniverse->totalLines)
		{
		if(!(*haveX))
			{
			GetEditorViewTextToGraphicPosition(theView,theChunk,theOffset,theLineOffset,desiredX);
			*haveX=TRUE;
			}
		LineToChunkPosition(theView->editorUniverse->textUniverse,theLine+1,&theChunk,&theOffset,&newPosition);
		GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,*desiredX,&theLineOffset,&temp);
		return(newPosition+theLineOffset);
		}
	else
		{
		return(thePosition);			/* there is no line below this one */
		}
}

static UINT32 NextLineLeft(EDITORVIEW *theView,UINT32 thePosition)
/* return the position of the start of the line of thePosition
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	if(thePosition)
		{
		PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition-1,&theChunk,&theOffset);		/* ignore starting character */
		found=FALSE;
		while(theChunk&&!found)
			{
			if(theChunk->data[theOffset]=='\n')
				{
				found=TRUE;
				}
			else
				{
				thePosition--;
				if(theOffset)
					{
					theOffset--;
					}
				else
					{
					if(theChunk=theChunk->previousHeader)
						{
						theOffset=theChunk->totalBytes-1;
						}
					}
				}
			}
		}
	return(thePosition);
}

static UINT32 NextLineRight(EDITORVIEW *theView,UINT32 thePosition)
/* return the position of the end of the line of thePosition
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition,&theChunk,&theOffset);
	found=FALSE;
	while(theChunk&&!found)
		{
		if(theChunk->data[theOffset]=='\n')
			{
			found=TRUE;
			}
		else
			{
			thePosition++;
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(thePosition);
}

static UINT32 NextParagraphLeft(EDITORVIEW *theView,UINT32 thePosition)
/* return the position of the start of the paragraph of thePosition
 * NOTE: a paragraph is considered to be the text between
 * empty lines
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		locatedPosition;
	BOOLEAN
		foundPrevious,
		found;

	if(locatedPosition=thePosition)
		{
		found=foundPrevious=FALSE;
		if(thePosition>=theView->editorUniverse->textUniverse->totalBytes)	/* if searching back from the end, then step back */
			{
			thePosition=theView->editorUniverse->textUniverse->totalBytes-1;
			foundPrevious=TRUE;
			}
		PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition,&theChunk,&theOffset);
		while(theChunk&&!found)
			{
			if(theChunk->data[theOffset]=='\n'&&foundPrevious)
				{
				found=TRUE;
				}
			else
				{
				if(!(foundPrevious=(theChunk->data[theOffset]=='\n')))
					{
					locatedPosition=thePosition;		/* move to this place */
					}
				thePosition--;
				if(theOffset)
					{
					theOffset--;
					}
				else
					{
					if(theChunk=theChunk->previousHeader)
						{
						theOffset=theChunk->totalBytes-1;
						}
					}
				}
			}
		}
	return(locatedPosition);
}

static UINT32 NextParagraphRight(EDITORVIEW *theView,UINT32 thePosition)
/* return the position of the end of the paragraph of thePosition
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		locatedPosition;
	BOOLEAN
		foundPrevious,
		found;

	found=foundPrevious=FALSE;
	locatedPosition=thePosition;
	if(thePosition)
		{
		thePosition--;								/* step back one */
		}
	else
		{
		foundPrevious=TRUE;							/* if at start, then it is like having a newline previously */
		}
	PositionToChunkPosition(theView->editorUniverse->textUniverse,thePosition,&theChunk,&theOffset);
	while(theChunk&&!found)
		{
		if(theChunk->data[theOffset]=='\n'&&foundPrevious)
			{
			found=TRUE;
			}
		else
			{
			foundPrevious=(theChunk->data[theOffset]=='\n');
			thePosition++;
			if(!foundPrevious)
				{
				locatedPosition=thePosition;		/* move to this place */
				}
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(locatedPosition);
}

static UINT32 NextPageUp(EDITORVIEW *theView,UINT32 thePosition,BOOLEAN *haveX,INT32 *desiredX)
/* return the position that is one page up from thePosition
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	UINT32
		newPosition;
	UINT32
		theLine,
		theLineOffset,
		temp;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theOffset);
	if(numLines)
		{
		if(theLine>numLines-1)
			{
			theLine-=numLines-1;
			}
		else
			{
			theLine=0;
			}
		}
	if(!(*haveX))
		{
		GetEditorViewTextToGraphicPosition(theView,theChunk,theOffset,theLineOffset,desiredX);
		*haveX=TRUE;
		}
	LineToChunkPosition(theView->editorUniverse->textUniverse,theLine,&theChunk,&theOffset,&newPosition);
	GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,*desiredX,&theLineOffset,&temp);
	return(newPosition+theLineOffset);
}

static UINT32 NextPageDown(EDITORVIEW *theView,UINT32 thePosition,BOOLEAN *haveX,INT32 *desiredX)
/* return the position that is one page down from thePosition
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	INT32
		leftPixel;
	UINT32
		newPosition;
	UINT32
		theLine,
		theLineOffset,
		temp;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&theLine,&theLineOffset,&theChunk,&theOffset);
	if(numLines)
		{
		theLine+=numLines-1;
		}
	if(theLine>theView->editorUniverse->textUniverse->totalLines)
		{
		theLine=theView->editorUniverse->textUniverse->totalLines;
		}
	if(!(*haveX))
		{
		GetEditorViewTextToGraphicPosition(theView,theChunk,theOffset,theLineOffset,desiredX);
		*haveX=TRUE;
		}
	LineToChunkPosition(theView->editorUniverse->textUniverse,theLine,&theChunk,&theOffset,&newPosition);
	GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,*desiredX,&theLineOffset,&temp);
	return(newPosition+theLineOffset);
}

static UINT32 GetRelativePosition(EDITORVIEW *theView,UINT32 thePosition,UINT16 relativeMode,BOOLEAN *haveX,INT32 *desiredX)
/* return the position that relates to thePosition in theView by relativeMode
 * if haveX is TRUE, desiredX is valid for moving up or down
 * this will return an updated haveX/desiredX
 */
{
	if(relativeMode&RPM_mBACKWARD)
		{
		switch(relativeMode&RPM_mMODE)
			{
			case RPM_CHAR:
				*haveX=FALSE;
				return(NextCharLeft(theView,thePosition));
				break;
			case RPM_WORD:
				*haveX=FALSE;
				return(NextWordLeft(theView,thePosition));
				break;
			case RPM_LINE:
				return(NextLineUp(theView,thePosition,haveX,desiredX));
				break;
			case RPM_LINEEDGE:
				*haveX=FALSE;
				return(NextLineLeft(theView,thePosition));
				break;
			case RPM_PARAGRAPHEDGE:
				*haveX=FALSE;
				return(NextParagraphLeft(theView,thePosition));
				break;
			case RPM_PAGE:
				return(NextPageUp(theView,thePosition,haveX,desiredX));
				break;
			case RPM_DOCEDGE:
				*haveX=FALSE;
				return(0);
				break;
			default:
				*haveX=FALSE;
				return(thePosition);
				break;
			}
		}
	else
		{
		switch(relativeMode&RPM_mMODE)
			{
			case RPM_CHAR:
				*haveX=FALSE;
				return(NextCharRight(theView,thePosition));
				break;
			case RPM_WORD:
				*haveX=FALSE;
				return(NextWordRight(theView,thePosition));
				break;
			case RPM_LINE:
				return(NextLineDown(theView,thePosition,haveX,desiredX));
				break;
			case RPM_LINEEDGE:
				*haveX=FALSE;
				return(NextLineRight(theView,thePosition));
				break;
			case RPM_PARAGRAPHEDGE:
				*haveX=FALSE;
				return(NextParagraphRight(theView,thePosition));
				break;
			case RPM_PAGE:
				return(NextPageDown(theView,thePosition,haveX,desiredX));
				break;
			case RPM_DOCEDGE:
				*haveX=FALSE;
				return(theView->editorUniverse->textUniverse->totalBytes);
				break;
			default:
				*haveX=FALSE;
				return(thePosition);
				break;
			}
		}
}

/* selection boundary location routines
 */

static UINT32 NoMove(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* used to not move on the given side of the selection
 */
{
	return(betweenPosition);
}

static void ValueLeft(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT8 leftValue,UINT8 rightValue,UINT32 *newPosition)
/* search to the left for the first unmatched leftValue
 * adjust newPosition
 * if leftValue and rightValue are the same, this will stop at the first occurrence
 */
{
	UINT8
		theChar;
	UINT32
		matchCount;
	BOOLEAN
		found;
	UINT32
		skippedBytes;

	matchCount=0;
	skippedBytes=0;
	found=FALSE;
	while(theChunk&&!found)
		{
		if(theOffset)
			{
			theOffset--;
			}
		else
			{
			if(theChunk=theChunk->previousHeader)
				{
				theOffset=theChunk->totalBytes-1;
				}
			}
		if(theChunk)
			{
			theChar=theChunk->data[theOffset];
			if(theChar==leftValue)
				{
				if(matchCount)
					{
					matchCount--;
					skippedBytes++;
					}
				else
					{
					found=TRUE;
					}
				}
			else
				{
				if(theChar==rightValue)
					{
					matchCount++;
					}
				skippedBytes++;
				}
			}
		}
	if(found)
		{
		(*newPosition)-=skippedBytes;
		}
}

static void ValueRight(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT8 leftValue,UINT8 rightValue,UINT32 *newPosition)
/* search to the right for the first unmatched rightValue
 * adjust newPosition
 * if leftValue and rightValue are the same, this will stop at the first occurrence
 */
{
	UINT8
		theChar;
	UINT32
		matchCount;
	BOOLEAN
		found;
	UINT32
		skippedBytes;

	matchCount=0;
	skippedBytes=0;
	found=FALSE;
	if(theChunk)				/* always skip over the start */
		{
		(*newPosition)++;
		theOffset++;
		if(theOffset>=theChunk->totalBytes)
			{
			theChunk=theChunk->nextHeader;
			theOffset=0;
			}
		}
	while(theChunk&&!found)
		{
		theChar=theChunk->data[theOffset];
		if(theChar==rightValue)
			{
			if(matchCount)
				{
				matchCount--;
				skippedBytes++;
				}
			else
				{
				found=TRUE;
				}
			}
		else
			{
			if(theChar==leftValue)
				{
				matchCount++;
				}
			skippedBytes++;
			}
		theOffset++;
		if(theOffset>=theChunk->totalBytes)
			{
			theChunk=theChunk->nextHeader;
			theOffset=0;
			}
		}
	if(found)
		{
		(*newPosition)+=skippedBytes;
		}
}

static BOOLEAN LocateSpecialWordMatchLeft(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 *newPosition)
/* see if the character being selected is special, if so, do special matching on it
 * this is used to match parenthesis, and brackets, and quotes, etc during the initial word
 * select, subsequent tracking is done normally
 */
{
	BOOLEAN
		found;

	found=FALSE;
	switch(theChunk->data[theOffset])
		{
		case '{':
			(*newPosition)++;
			found=TRUE;
			break;
		case '}':
			ValueLeft(theUniverse,theChunk,theOffset,'{','}',newPosition);
			found=TRUE;
			break;
		case '(':
			(*newPosition)++;
			found=TRUE;
			break;
		case ')':
			ValueLeft(theUniverse,theChunk,theOffset,'(',')',newPosition);
			found=TRUE;
			break;
		case '[':
			(*newPosition)++;
			found=TRUE;
			break;
		case ']':
			ValueLeft(theUniverse,theChunk,theOffset,'[',']',newPosition);
			found=TRUE;
			break;
		case '"':
			(*newPosition)++;
			found=TRUE;
			break;
		case '/':																	/* c style comment */
			(*newPosition)++;
			found=TRUE;
			break;
		case '\'':
			(*newPosition)++;
			found=TRUE;
			break;
		}
	return(found);
}

static BOOLEAN LocateSpecialWordMatchRight(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 *newPosition)
/* see if the character being selected is special, if so, do special matching on it
 * this is used to match parenthesis, and brackets, and quotes, etc during the initial word
 * select, subsequent tracking is done normally
 */
{
	BOOLEAN
		found;

	found=FALSE;
	switch(theChunk->data[theOffset])
		{
		case '}':
			found=TRUE;
			break;
		case '{':
			ValueRight(theUniverse,theChunk,theOffset,'{','}',newPosition);
			found=TRUE;
			break;
		case ')':
			found=TRUE;
			break;
		case '(':
			ValueRight(theUniverse,theChunk,theOffset,'(',')',newPosition);
			found=TRUE;
			break;
		case ']':
			found=TRUE;
			break;
		case '[':
			ValueRight(theUniverse,theChunk,theOffset,'[',']',newPosition);
			found=TRUE;
			break;
		case '"':
			ValueRight(theUniverse,theChunk,theOffset,'"','"',newPosition);
			found=TRUE;
			break;
		case '/':																	/* c style comment */
			ValueRight(theUniverse,theChunk,theOffset,'/','/',newPosition);
			found=TRUE;
			break;
		case '\'':
			ValueRight(theUniverse,theChunk,theOffset,'\'','\'',newPosition);
			found=TRUE;
			break;
		}
	return(found);
}

static UINT32 WordLeft(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* locate next word boundary to the left
 * return the new between position
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	PositionToChunkPosition(theUniverse,charPosition,&theChunk,&theOffset);		/* look at the character in question */
	found=FALSE;
	if(theChunk)
		{
		if(!(trackMode&TM_mREPEAT))									/* if not in repeat, then see if clicked on something we want to match */
			{
			found=LocateSpecialWordMatchLeft(theUniverse,theChunk,theOffset,&charPosition);
			}
		if(!found)
			{
			if(IsWordSpace(theChunk->data[theOffset]))
				{
				found=TRUE;
				}
			else
				{
				if(theOffset)
					{
					theOffset--;
					}
				else
					{
					if(theChunk=theChunk->previousHeader)
						{
						theOffset=theChunk->totalBytes-1;
						}
					}
				}
			}
		}
	while(theChunk&&!found)
		{
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			charPosition--;
			if(theOffset)
				{
				theOffset--;
				}
			else
				{
				if(theChunk=theChunk->previousHeader)
					{
					theOffset=theChunk->totalBytes-1;
					}
				}
			}
		}
	return(charPosition);
}

static UINT32 WordRight(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* locate next word boundary to the right
 * return the new between position
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	PositionToChunkPosition(theUniverse,charPosition,&theChunk,&theOffset);
	found=FALSE;
	if(theChunk)
		{
		if(!(trackMode&TM_mREPEAT))									/* if not in repeat, then see if clicked on something we want to match */
			{
			found=LocateSpecialWordMatchRight(theUniverse,theChunk,theOffset,&charPosition);
			}
		if(!found)
			{
			charPosition++;
			if(IsWordSpace(theChunk->data[theOffset]))
				{
				found=TRUE;
				}
			else
				{
				theOffset++;
				if(theOffset>=theChunk->totalBytes)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			}
		}
	while(theChunk&&!found)
		{
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			charPosition++;
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(charPosition);
}

static UINT32 LineLeft(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* locate next line boundary to the left, and including charPosition
 * return the new between position
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	if(charPosition)
		{
		PositionToChunkPosition(theUniverse,charPosition-1,&theChunk,&theOffset);		/* ignore starting character */
		found=FALSE;
		while(theChunk&&!found)
			{
			if(theChunk->data[theOffset]=='\n')
				{
				found=TRUE;
				}
			else
				{
				charPosition--;
				if(theOffset)
					{
					theOffset--;
					}
				else
					{
					if(theChunk=theChunk->previousHeader)
						{
						theOffset=theChunk->totalBytes-1;
						}
					}
				}
			}
		}
	return(charPosition);
}

static UINT32 LineRight(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* locate next line boundary to the right, and including charPosition
 * return the new between position
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		found;

	PositionToChunkPosition(theUniverse,charPosition,&theChunk,&theOffset);
	found=FALSE;
	while(theChunk&&!found)
		{
		charPosition++;
		if(theChunk->data[theOffset]=='\n')
			{
			found=TRUE;
			}
		else
			{
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(charPosition);
}

static UINT32 AllLeft(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* move back to the start
 * return the new between position
 */
{
	return(0);
}

static UINT32 AllRight(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode)
/* locate the end
 * return the new between position
 */
{
	return(theUniverse->totalBytes);
}

/* columnar selection functions
 * NOTE: these are different from the functions above which return absolute positions in the
 * text universe. These return offsets relative to the start of the line in question.
 */

static UINT32 CNoMove(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition)
/* used to not move on the given side of the selection
 */
{
	return(betweenPosition);
}

static UINT32 ColumnarWordLeft(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition)
/* locate next word boundary to the left
 * return the new between position
 */
{
	BOOLEAN
		found;

	AddToChunkPosition(theUniverse,theChunk,theOffset,&theChunk,&theOffset,charPosition);		/* look at the character in question */
	found=FALSE;
	if(theChunk&&charPosition)
		{
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			if(theOffset)
				{
				theOffset--;
				}
			else
				{
				if(theChunk=theChunk->previousHeader)
					{
					theOffset=theChunk->totalBytes-1;
					}
				}
			}
		}
	while(theChunk&&!found&&charPosition)
		{
		if(IsWordSpace(theChunk->data[theOffset]))
			{
			found=TRUE;
			}
		else
			{
			charPosition--;
			if(theOffset)
				{
				theOffset--;
				}
			else
				{
				if(theChunk=theChunk->previousHeader)
					{
					theOffset=theChunk->totalBytes-1;
					}
				}
			}
		}
	return(charPosition);
}

static UINT32 ColumnarWordRight(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition)
/* locate next word boundary to the right
 * return the new between position
 */
{
	BOOLEAN
		found;

	AddToChunkPosition(theUniverse,theChunk,theOffset,&theChunk,&theOffset,charPosition);		/* look at the character in question */
	found=FALSE;
	if(theChunk)
		{
		if(theChunk->data[theOffset]!='\n')					/* if over the CR, do not select it in columnar mode */
			{
			charPosition++;
			if(IsWordSpace(theChunk->data[theOffset]))
				{
				found=TRUE;
				}
			else
				{
				theOffset++;
				if(theOffset>=theChunk->totalBytes)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			}
		else
			{
			found=TRUE;
			}
		}
	while(theChunk&&!found)
		{
		if(IsWordSpace(theChunk->data[theOffset])||theChunk->data[theOffset]=='\n')
			{
			found=TRUE;
			}
		else
			{
			charPosition++;
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(charPosition);
}

static UINT32 ColumnarLineLeft(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition)
/* locate next line boundary to the left, and including charPosition
 * return the new between position
 */
{
	return(0);
}

static UINT32 ColumnarLineRight(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition)
/* locate next line boundary to the right, and including charPosition
 * return the new between position
 */
{
	BOOLEAN
		found;

	AddToChunkPosition(theUniverse,theChunk,theOffset,&theChunk,&theOffset,charPosition);		/* look at the character in question */
	found=FALSE;
	while(theChunk&&!found)
		{
		if(theChunk->data[theOffset]=='\n')
			{
			found=TRUE;
			}
		else
			{
			charPosition++;
			theOffset++;
			if(theOffset>=theChunk->totalBytes)
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	return(charPosition);
}

static void AdjustColumnarRect(EDITORUNIVERSE *theUniverse,UINT32 theLine,INT32 theXPosition,UINT16 trackMode)
/* adjust the columnar selection rectangle bounds based on the new position, and trackMode
 */
{
	if(!(trackMode&TM_mREPEAT))				/* during repeat, nothing is ever done to the anchor position */
		{
		if(trackMode&TM_mCONTINUE)			/* to continue find out which side of the rect is closest to the point, and set the anchor to opposite end */
			{
			if(theLine<theUniverse->columnarTopLine)
				{
				theUniverse->anchorLine=theUniverse->columnarBottomLine;
				}
			else
				{
				if(theLine>theUniverse->columnarBottomLine)
					{
					theUniverse->anchorLine=theUniverse->columnarTopLine;
					}
				else
					{
					if(theUniverse->columnarBottomLine-theLine>=theLine-theUniverse->columnarTopLine)
						{
						theUniverse->anchorLine=theUniverse->columnarBottomLine;
						}
					else
						{
						theUniverse->anchorLine=theUniverse->columnarTopLine;
						}
					}
				}
			if(theXPosition<theUniverse->columnarLeftX)
				{
				theUniverse->anchorX=theUniverse->columnarRightX;
				}
			else
				{
				if(theXPosition>theUniverse->columnarRightX)
					{
					theUniverse->anchorX=theUniverse->columnarLeftX;
					}
				else
					{
					if(theUniverse->columnarRightX-theXPosition>=theXPosition-theUniverse->columnarLeftX)
						{
						theUniverse->anchorX=theUniverse->columnarRightX;
						}
					else
						{
						theUniverse->anchorX=theUniverse->columnarLeftX;
						}
					}
				}
			}
		else
			{
			theUniverse->anchorLine=theLine;		/* set the columnar rectangle to empty rect centered at the given point, anchor tracking to this point */
			theUniverse->anchorX=theXPosition;
			}
		}
	/* now, use the anchor and the point to determine the top, left, bottom, and right */
	if(theUniverse->anchorLine>=theLine)
		{
		theUniverse->columnarTopLine=theLine;
		theUniverse->columnarBottomLine=theUniverse->anchorLine;
		}
	else
		{
		theUniverse->columnarTopLine=theUniverse->anchorLine;
		theUniverse->columnarBottomLine=theLine;
		}
	if(theUniverse->anchorX>=theXPosition)
		{
		theUniverse->columnarLeftX=theXPosition;
		theUniverse->columnarRightX=theUniverse->anchorX;
		}
	else
		{
		theUniverse->columnarLeftX=theUniverse->anchorX;
		theUniverse->columnarRightX=theXPosition;
		}
}

void EditorSetNormalSelection(EDITORUNIVERSE *theEditorUniverse,UINT32 startPosition,UINT32 endPosition)
/* clear the current selection(s) if any, and set a new one
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	ARRAYCHUNKHEADER
		*selectionChunk;
	UINT32
		selectionOffset;
	BOOLEAN
		fail;

	fail=FALSE;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;

	EditorStartSelectionChange(theEditorUniverse);
	DeleteUniverseSelection(theSelectionUniverse,theSelectionUniverse->selectionChunks.firstChunkHeader,0,theSelectionUniverse->selectionChunks.totalElements,&selectionChunk,&selectionOffset);	/* remove old selection(s) */
	theSelectionUniverse->cursorPosition=startPosition;												/* set the cursor here */
	if(startPosition<endPosition)
		{
		if(InsertUniverseSelection(theSelectionUniverse,NULL,0,1,&selectionChunk,&selectionOffset))	/* make a new selection */
			{
			((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=startPosition;
			((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=endPosition-startPosition;
			}
		else
			{
			fail=TRUE;
			}
		}
	EditorEndSelectionChange(theEditorUniverse);

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to set selection: %.256s\n",errorDescription);
		}
}

void EditorSetColumnarSelection(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine,INT32 leftX,INT32 rightX,UINT16 trackMode)
/* clear the current selection(s) if any, and set a new columnar one
 */
{
	EDITORUNIVERSE
		*theEditorUniverse;
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	ARRAYCHUNKHEADER
		*selectionChunk;
	UINT32
		selectionOffset;
	UINT32
		distance,
		basePosition;
	UINT32
		currentPosition,
		startPosition,
		endPosition,
		startBetween,
		startChar,
		endBetween,
		endChar;
	UINT32
		currentLine;
	UINT32
		(*leftFunction)(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition),
		(*rightFunction)(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 betweenPosition,UINT32 charPosition);
	BOOLEAN
		fail;

	theEditorUniverse=theView->editorUniverse;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;

	switch(trackMode&TM_mMODE)
		{
		case TM_CHAR:
			leftFunction=rightFunction=CNoMove;
			break;
		case TM_WORD:
			leftFunction=ColumnarWordLeft;
			rightFunction=ColumnarWordRight;
			break;
		case TM_LINE:
			leftFunction=ColumnarLineLeft;
			rightFunction=ColumnarLineRight;
			break;
		case TM_ALL:
			leftFunction=ColumnarLineLeft;
			rightFunction=ColumnarLineRight;
			startLine=0;
			endLine=theEditorUniverse->textUniverse->totalLines;
			break;
		default:
			leftFunction=rightFunction=CNoMove;
			break;
		}

	EditorStartSelectionChange(theEditorUniverse);

	DeleteUniverseSelection(theSelectionUniverse,theSelectionUniverse->selectionChunks.firstChunkHeader,0,theSelectionUniverse->selectionChunks.totalElements,&selectionChunk,&selectionOffset);

	currentPosition=0;
	LineToChunkPosition(theEditorUniverse->textUniverse,startLine,&theChunk,&theOffset,&basePosition);	/* find start of top line */

	fail=FALSE;
	for(currentLine=startLine;!fail&&(currentLine<=endLine);currentLine++)
		{
		GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,leftX,&startBetween,&startChar);
		GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,rightX,&endBetween,&endChar);

		startPosition=basePosition+leftFunction(theEditorUniverse->textUniverse,theChunk,theOffset,startBetween,startChar);
		endPosition=basePosition+rightFunction(theEditorUniverse->textUniverse,theChunk,theOffset,endBetween,endChar);

		if(currentLine==startLine)				/* place cursor at start of selection */
			{
			theSelectionUniverse->cursorPosition=startPosition;
			}

		if(startPosition<currentPosition)		/* if this extends back before a previous selection, then start immediately following it */
			{
			startPosition=currentPosition;
			}
		if(endPosition<currentPosition)			/* if this is too far back, then pull forward */
			{
			endPosition=currentPosition;
			}
		if(startPosition<endPosition)			/* make sure there is something to select */
			{
			if(InsertUniverseSelection(theSelectionUniverse,NULL,0,1,&selectionChunk,&selectionOffset))		/* create new selection element */
				{
				((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=startPosition-currentPosition;
				((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=endPosition-startPosition;
				currentPosition=endPosition;
				}
			else
				{
				fail=TRUE;
				}
			}
		ChunkPositionToNextLine(theEditorUniverse->textUniverse,theChunk,theOffset,&theChunk,&theOffset,&distance);
		basePosition+=distance;
		}
	EditorEndSelectionChange(theEditorUniverse);

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to set selection: %.256s\n",errorDescription);
		}
}

static void EditorAdjustNormalSelection(EDITORVIEW *theView,UINT32 theLine,INT32 theXPosition,UINT16 trackMode)
/* adjust/create the selection based on theLine, theXPosition, and trackMode
 * if nothing would be selected, then place the cursor at the correct position
 */
{
	EDITORUNIVERSE
		*theEditorUniverse;
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		base,
		betweenPosition,
		charPosition,
		startPosition,
		endPosition;
	UINT32
		(*leftFunction)(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode),
		(*rightFunction)(TEXTUNIVERSE *theUniverse,UINT32 betweenPosition,UINT32 charPosition,UINT16 trackMode);

	theEditorUniverse=theView->editorUniverse;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;

	switch(trackMode&TM_mMODE)
		{
		case TM_CHAR:
			leftFunction=rightFunction=NoMove;
			break;
		case TM_WORD:
			leftFunction=WordLeft;
			rightFunction=WordRight;
			break;
		case TM_LINE:
			leftFunction=LineLeft;
			rightFunction=LineRight;
			break;
		case TM_ALL:
			leftFunction=AllLeft;
			rightFunction=AllRight;
			break;
		default:
			leftFunction=rightFunction=NoMove;
			break;
		}

	AdjustColumnarRect(theEditorUniverse,theLine,theXPosition,trackMode);
	LineToChunkPosition(theEditorUniverse->textUniverse,theLine,&theChunk,&theOffset,&base);
	GetEditorViewGraphicToTextPosition(theView,theChunk,theOffset,theXPosition,&betweenPosition,&charPosition);

	betweenPosition+=base;
	charPosition+=base;

	/* adjust the anchor */
	if(!(trackMode&TM_mREPEAT))				/* during repeat, nothing is ever done to the anchor position */
		{
		if(trackMode&TM_mCONTINUE)
			{
			GetSelectionEndPositions(theSelectionUniverse,&startPosition,&endPosition);		/* find ends of old selection */
			if(betweenPosition<startPosition)
				{
				startPosition=leftFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
				theEditorUniverse->anchorStartPosition=theEditorUniverse->anchorEndPosition=endPosition;
				}
			else
				{
				if(betweenPosition>endPosition)
					{
					endPosition=rightFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
					theEditorUniverse->anchorStartPosition=theEditorUniverse->anchorEndPosition=startPosition;
					}
				else
					{
					if(endPosition-betweenPosition>=betweenPosition-startPosition)
						{
						startPosition=leftFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
						theEditorUniverse->anchorStartPosition=theEditorUniverse->anchorEndPosition=endPosition;
						}
					else
						{
						endPosition=rightFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
						theEditorUniverse->anchorStartPosition=theEditorUniverse->anchorEndPosition=startPosition;
						}
					}
				}
			}
		else
			{
			startPosition=leftFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
			endPosition=rightFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
			theEditorUniverse->anchorStartPosition=startPosition;	/* if starting a new selection, and not continuing from an old one, then drop anchor on the selection */
			theEditorUniverse->anchorEndPosition=endPosition;
			}
		}
	else
		{
		startPosition=leftFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
		endPosition=rightFunction(theEditorUniverse->textUniverse,betweenPosition,charPosition,trackMode);
		if(startPosition>theEditorUniverse->anchorStartPosition)
			{
			startPosition=theEditorUniverse->anchorStartPosition;
			}
		if(endPosition<theEditorUniverse->anchorEndPosition)
			{
			endPosition=theEditorUniverse->anchorEndPosition;
			}
		}
	EditorSetNormalSelection(theEditorUniverse,startPosition,endPosition);
}

static void EditorAdjustColumnarSelection(EDITORVIEW *theView,UINT32 theLine,INT32 theXPosition,UINT16 trackMode)
/* adjust the columnar selection in the view based on the given global coordinates, and
 * trackMode
 */
{
	EDITORUNIVERSE
		*theEditorUniverse;

	theEditorUniverse=theView->editorUniverse;

	AdjustColumnarRect(theEditorUniverse,theLine,theXPosition,trackMode);
	EditorSetColumnarSelection(theView,theEditorUniverse->columnarTopLine,theEditorUniverse->columnarBottomLine,theEditorUniverse->columnarLeftX,theEditorUniverse->columnarRightX,trackMode);
}

static void ScrollViewToFollowPointer(EDITORVIEW *theView,INT32 viewXPosition,INT32 viewLine,UINT16 trackMode)
/* given a view relative line and X offset and the current tracking mode, scroll the view
 */
{
	UINT32
		topLine,
		numLines,
		numPixels,
		newTopLine;
	INT32
		leftPixel,
		newLeftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	if(viewLine<0)
		{
		if(topLine)
			{
			newTopLine=topLine-1;
			}
		else
			{
			newTopLine=0;
			}
		}
	else
		{
		if(viewLine>=numLines)
			{
			if(topLine<theView->editorUniverse->textUniverse->totalLines)
				{
				newTopLine=topLine+1;
				}
			else
				{
				newTopLine=theView->editorUniverse->textUniverse->totalLines;
				}
			}
		else
			{
			newTopLine=topLine;
			}
		}
	if(viewXPosition<0)
		{
		if(leftPixel-HORIZONTALSCROLLTHRESHOLD>0)
			{
			newLeftPixel=leftPixel-HORIZONTALSCROLLTHRESHOLD;
			}
		else
			{
			newLeftPixel=0;
			}
		}
	else
		{
		if(viewXPosition>=numPixels)
			{
			newLeftPixel=leftPixel+HORIZONTALSCROLLTHRESHOLD;
			}
		else
			{
			newLeftPixel=leftPixel;
			}
		}
	if((newTopLine!=topLine)||(newLeftPixel!=leftPixel))
		{
		SetViewTopLeft(theView,newTopLine,newLeftPixel);
		}
}

static void EditorTrackViewPointerNormal(EDITORVIEW *theView,INT32 viewXPosition,INT32 viewLine,UINT16 trackMode)
/* track the pointer in non-columnar mode
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	UINT32
		newLine;
	INT32
		leftPixel,
		newLeftPixel;
	static UINT32
		lastLine;
	static INT32
		lastLeftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	newLeftPixel=leftPixel+viewXPosition;
	if(viewLine>=0)
		{
		newLine=topLine+viewLine;
		}
	else
		{
		newLine=-viewLine;
		if(newLine<topLine)
			{
			newLine=topLine-newLine;
			}
		else
			{
			newLine=0;
			newLeftPixel=0;
			}
		}

	if((!(trackMode&TM_mREPEAT))||(newLine!=lastLine)||(newLeftPixel!=lastLeftPixel))	/* if first time, or point has moved, then track it */
		{
		lastLine=newLine;
		lastLeftPixel=newLeftPixel;
		EditorAdjustNormalSelection(theView,newLine,newLeftPixel,trackMode);
		}
}

static void EditorTrackViewPointerColumnar(EDITORVIEW *theView,INT32 viewXPosition,INT32 viewLine,UINT16 trackMode)
/* track the pointer in columnar mode
 */
{
	UINT32
		topLine,
		numLines,
		numPixels;
	UINT32
		newLine;
	INT32
		leftPixel,
		newLeftPixel;
	static UINT32
		lastLine;
	static INT32
		lastLeftPixel;

	GetEditorViewTextInfo(theView,&topLine,&numLines,&leftPixel,&numPixels);
	newLeftPixel=leftPixel+viewXPosition;
	if(viewLine>=0)
		{
		newLine=topLine+viewLine;
		}
	else
		{
		newLine=-viewLine;
		if(newLine<topLine)
			{
			newLine=topLine-newLine;
			}
		else
			{
			newLine=0;
			}
		}

	if((!(trackMode&TM_mREPEAT))||(newLine!=lastLine)||(newLeftPixel!=lastLeftPixel))	/* if first time, or point has moved, then track it */
		{
		lastLine=newLine;
		lastLeftPixel=newLeftPixel;
		EditorAdjustColumnarSelection(theView,newLine,newLeftPixel,trackMode);
		}
}

void EditorTrackViewPointer(EDITORVIEW *theView,INT32 viewXPosition,INT32 viewLine,UINT16 trackMode)
/* track the pointer in theView, according to the current cursor/selection in theView
 * and trackMode
 */
{
	if(!(trackMode&TM_mCOLUMNAR))
		{
		EditorTrackViewPointerNormal(theView,viewXPosition,viewLine,trackMode);
		}
	else
		{
		EditorTrackViewPointerColumnar(theView,viewXPosition,viewLine,trackMode);
		}
	ScrollViewToFollowPointer(theView,viewXPosition,viewLine,trackMode);
}

void EditorMoveCursor(EDITORVIEW *theView,UINT16 relativeMode)
/* move the cursor according to relativeMode, unless there is a selection
 * if there is a selection, send the cursor to the top if relativeMode is backward,
 * send the cursor to the bottom of the selection if relativeMode is forward
 */
{
	UINT32
		startPosition,
		endPosition;
	BOOLEAN
		haveX;
	INT32
		desiredX;

	if(theView->editorUniverse->selectionUniverse->selectionChunks.totalElements)
		{
		GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection */
		if(relativeMode&RPM_mBACKWARD)
			{
			EditorSetNormalSelection(theView->editorUniverse,startPosition,startPosition);
			}
		else
			{
			EditorSetNormalSelection(theView->editorUniverse,endPosition,endPosition);
			}
		}
	else
		{
		haveX=theView->editorUniverse->haveStartX;			/* pass these to relative position routine */
		desiredX=theView->editorUniverse->desiredStartX;
		startPosition=GetRelativePosition(theView,theView->editorUniverse->selectionUniverse->cursorPosition,relativeMode,&haveX,&desiredX);
		EditorSetNormalSelection(theView->editorUniverse,startPosition,startPosition);
		theView->editorUniverse->haveStartX=haveX;			/* update these */
		theView->editorUniverse->desiredStartX=desiredX;
		}
}

void EditorExpandNormalSelection(EDITORVIEW *theView,UINT16 relativeMode)
/* expand the selection using relativeMode
 */
{
	UINT32
		startPosition,
		endPosition;
	BOOLEAN
		haveStartX,
		haveEndX;
	INT32
		desiredStartX,
		desiredEndX;

	GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection */

	haveStartX=theView->editorUniverse->haveStartX;
	desiredStartX=theView->editorUniverse->desiredStartX;
	haveEndX=theView->editorUniverse->haveEndX;
	desiredEndX=theView->editorUniverse->desiredEndX;

	if(relativeMode&RPM_mBACKWARD)
		{
		startPosition=GetRelativePosition(theView,startPosition,relativeMode,&haveStartX,&desiredStartX);
		}
	else
		{
		endPosition=GetRelativePosition(theView,endPosition,relativeMode,&haveEndX,&desiredEndX);
		}
	EditorSetNormalSelection(theView->editorUniverse,startPosition,endPosition);

	theView->editorUniverse->haveStartX=haveStartX;				/* update these */
	theView->editorUniverse->desiredStartX=desiredStartX;
	theView->editorUniverse->haveEndX=haveEndX;
	theView->editorUniverse->desiredEndX=desiredEndX;
}

void EditorReduceNormalSelection(EDITORVIEW *theView,UINT16 relativeMode)
/* reduce the selection using relativeMode
 */
{
	UINT32
		startPosition,
		endPosition;
	BOOLEAN
		haveStartX,
		haveEndX;
	INT32
		desiredStartX,
		desiredEndX;

	GetSelectionEndPositions(theView->editorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection */

	haveStartX=theView->editorUniverse->haveStartX;
	desiredStartX=theView->editorUniverse->desiredStartX;
	haveEndX=theView->editorUniverse->haveEndX;
	desiredEndX=theView->editorUniverse->desiredEndX;

	if(relativeMode&RPM_mBACKWARD)
		{
		endPosition=GetRelativePosition(theView,endPosition,relativeMode,&haveEndX,&desiredEndX);
		if(endPosition<=startPosition)
			{
			haveStartX=haveEndX=FALSE;
			endPosition=startPosition;
			}
		}
	else
		{
		startPosition=GetRelativePosition(theView,startPosition,relativeMode,&haveStartX,&desiredStartX);
		if(startPosition>=endPosition)
			{
			haveStartX=haveEndX=FALSE;
			startPosition=endPosition;
			}
		}
	EditorSetNormalSelection(theView->editorUniverse,startPosition,endPosition);

	theView->editorUniverse->haveStartX=haveStartX;				/* update these */
	theView->editorUniverse->desiredStartX=desiredStartX;
	theView->editorUniverse->haveEndX=haveEndX;
	theView->editorUniverse->desiredEndX=desiredEndX;
}

void EditorLocateLine(EDITORUNIVERSE *theEditorUniverse,UINT32 theLine)
/* find the given line, and select it
 * if the line is past the end of the text, select the end of the text
 * NOTE: this thinks of lines as starting at 0, not 1!!!!
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		startPosition,
		endPosition;

	LineToChunkPosition(theEditorUniverse->textUniverse,theLine,&theChunk,&theOffset,&startPosition);	/* find start of given line line */
	ChunkPositionToNextLine(theEditorUniverse->textUniverse,theChunk,theOffset,&theChunk,&theOffset,&endPosition);
	endPosition+=startPosition;
	EditorSetNormalSelection(theEditorUniverse,startPosition,endPosition);
}

void EditorSelectAll(EDITORUNIVERSE *theEditorUniverse)
/* Select all of the text in theEditorUniverse
 */
{
	EditorSetNormalSelection(theEditorUniverse,0,theEditorUniverse->textUniverse->totalBytes);
}

void EditorDelete(EDITORVIEW *theView,UINT16 relativeMode)
/* delete the character(s) between the current cursor position, and relativeMode away from it
 */
{
	EDITORUNIVERSE
		*theEditorUniverse;
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	UINT32
		relativePosition,
		startDelete,
		endDelete;
	BOOLEAN
		haveX;
	INT32
		desiredX;
	BOOLEAN
		fail;

	fail=FALSE;
	theEditorUniverse=theView->editorUniverse;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;
	EditorStartReplace(theEditorUniverse);

	if(theSelectionUniverse->selectionChunks.totalElements)
		{
		BeginUndoGroup(theEditorUniverse);
		DeleteAllSelectedText(theEditorUniverse);
		StrictEndUndoGroup(theEditorUniverse);
		haveX=FALSE;
		}
	else
		{
		haveX=theEditorUniverse->haveStartX;			/* pass these to relative position routine */
		desiredX=theEditorUniverse->desiredStartX;
		relativePosition=GetRelativePosition(theView,theSelectionUniverse->cursorPosition,relativeMode,&haveX,&desiredX);

		if(relativePosition<theSelectionUniverse->cursorPosition)
			{
			startDelete=relativePosition;
			endDelete=theSelectionUniverse->cursorPosition;
			}
		else
			{
			startDelete=theSelectionUniverse->cursorPosition;
			endDelete=relativePosition;
			}
		if(startDelete<endDelete)
			{
			if(!ReplaceEditorText(theEditorUniverse,startDelete,endDelete,NULL,0))
				{
				fail=TRUE;
				}
			}
		}
	EditorEndReplace(theEditorUniverse);
	theEditorUniverse->haveStartX=haveX;			/* update these */
	theEditorUniverse->desiredStartX=desiredX;

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to delete: %.256s\n",errorDescription);
		}
}

static void EditorInsertAtSelection(EDITORUNIVERSE *theEditorUniverse,SELECTIONUNIVERSE *theSelectionUniverse,UINT8 *theText,UINT32 textLength)
/* insert text after the cursor or over the selection in the given selection universe
 * if there is a problem, it will be reported here
 */
{
	BOOLEAN
		hadGroup;
	BOOLEAN
		fail;

	fail=hadGroup=FALSE;
	EditorStartReplace(theEditorUniverse);
	if(theSelectionUniverse->selectionChunks.totalElements)
		{
		BeginUndoGroup(theEditorUniverse);
		DeleteAllSelectedText(theEditorUniverse);	/* NOTE: we do not end the group here, so that the replace is in the same group, otherwise, it may or may not be depending on if the selection had more than one element */
		hadGroup=TRUE;
		}
	fail=!ReplaceEditorText(theEditorUniverse,theSelectionUniverse->cursorPosition,theSelectionUniverse->cursorPosition,theText,textLength);
	if(hadGroup)
		{
		EndUndoGroup(theEditorUniverse);
		}
	EditorEndReplace(theEditorUniverse);

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to insert: %.256s\n",errorDescription);
		}
}

void EditorInsert(EDITORUNIVERSE *theEditorUniverse,UINT8 *theText,UINT32 textLength)
/* insert the text after the cursor, or over the selection if there is one
 * if there is a problem, it will be reported here
 */
{
	EditorInsertAtSelection(theEditorUniverse,theEditorUniverse->selectionUniverse,theText,textLength);
}

void EditorAuxInsert(EDITORUNIVERSE *theEditorUniverse,UINT8 *theText,UINT32 textLength)
/* insert the text after the aux cursor, or over the aux selection if there is one
 * if there is a problem, it will be reported here
 */
{
	EditorInsertAtSelection(theEditorUniverse,theEditorUniverse->auxSelectionUniverse,theText,textLength);
}

BOOLEAN EditorInsertFile(EDITORUNIVERSE *theEditorUniverse,char *thePath)
/* insert the data in the file at thePath after the cursor or over the selection
 * in the selection universe
 * if there is a SetError, and return FALSE, it will be reported here
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	BOOLEAN
		fail;

	fail=FALSE;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;
	EditorStartReplace(theEditorUniverse);
	BeginUndoGroup(theEditorUniverse);
	if(theSelectionUniverse->selectionChunks.totalElements)
		{
		DeleteAllSelectedText(theEditorUniverse);	/* NOTE: we do not end the group here, so that the replace is in the same group, otherwise, it may or may not be depending on if the selection had more than one element */
		}
	fail=!ReplaceEditorFile(theEditorUniverse,theSelectionUniverse->cursorPosition,theSelectionUniverse->cursorPosition,thePath);
	StrictEndUndoGroup(theEditorUniverse);
	EditorEndReplace(theEditorUniverse);
	return(!fail);
}

void EditorAutoIndent(EDITORUNIVERSE *theEditorUniverse)
/* drop a return into the text, and auto indent the next line
 * if there is a problem, it will be reported here
 */
{
	SELECTIONUNIVERSE
		*theSelectionUniverse;
	UINT32
		startReplace;
	CHUNKHEADER
		*startChunk,
		*theChunk;
	UINT32
		startOffset,
		theOffset;
	UINT32
		theLine,
		lineOffset;
	UINT32
		numWhite;
	BOOLEAN
		fail,
		done;
	UINT8
		*insertBuffer;
	BOOLEAN
		hadGroup;

	fail=hadGroup=FALSE;
	theSelectionUniverse=theEditorUniverse->selectionUniverse;

	EditorStartReplace(theEditorUniverse);
	if(theSelectionUniverse->selectionChunks.totalElements)
		{
		BeginUndoGroup(theEditorUniverse);
		DeleteAllSelectedText(theEditorUniverse);
		hadGroup=TRUE;
		}
	startReplace=theSelectionUniverse->cursorPosition;
	PositionToLinePosition(theEditorUniverse->textUniverse,startReplace,&theLine,&lineOffset,&theChunk,&theOffset);	/* find the line that we are replacing in */
	startChunk=theChunk;
	startOffset=theOffset;
	numWhite=0;
	done=FALSE;
	while(numWhite<lineOffset&&!done&&theChunk)							/* run through counting the number of white space characters in this line, until the first non-white is seen, or until we reach the place where we will insert the return */
		{
		switch(theChunk->data[theOffset])
			{
			case ' ':
			case '\t':
				numWhite++;
				break;
			default:
				done=TRUE;
				break;
			}
		theOffset++;
		if(theOffset>=theChunk->totalBytes)								/* push through end of chunk */
			{
			theChunk=theChunk->nextHeader;
			theOffset=0;
			}
		}
	if(insertBuffer=(UINT8 *)MNewPtr(numWhite+1))						/* make room for all the white space, and the CR */
		{
		insertBuffer[0]='\n';											/* jam in the CR */
		if(ExtractUniverseText(theEditorUniverse->textUniverse,startChunk,startOffset,&(insertBuffer[1]),numWhite,&theChunk,&theOffset))	/* pull the white space from the line */
			{
			if(ReplaceEditorText(theEditorUniverse,startReplace,startReplace,insertBuffer,numWhite+1))
				{
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		MDisposePtr(insertBuffer);
		}
	else
		{
		fail=TRUE;
		}
	if(hadGroup)
		{
		EndUndoGroup(theEditorUniverse);
		}
	EditorEndReplace(theEditorUniverse);

	if(fail)
		{
		GetError(&errorFamily,&errorFamilyMember,&errorDescription);
		ReportMessage("Failed to auto indent: %.256s\n",errorDescription);
		}
}

/* --------------------------------------------------------------------------------------------------------------------------
 * clear/load/save functions
 */

BOOLEAN ClearEditorUniverse(EDITORUNIVERSE *theUniverse)
/* clear the editor universe, wiping out any text that was previously there
 * also kill any selection, and reset the cursor
 * and set the text non-dirty
 * if there is a problem, set the error, and return FALSE
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	EditorStartReplace(theUniverse);									/* we are going to change text in an edit universe */
	fail=!ReplaceEditorText(theUniverse,0,theUniverse->textUniverse->totalBytes,NULL,0);	/* remove all that was there */
	EditorEndReplace(theUniverse);										/* text change is complete */
	EditorClearUndo(theUniverse);
	theUniverse->dirty=FALSE;
	return(!fail);
}

BOOLEAN LoadEditorUniverse(EDITORUNIVERSE *theUniverse,char *thePath)
/* load the editor universe from thePath, wiping out
 * any text that was previously there
 * also kill any selection, and reset the cursor
 * and set the text non-dirty
 * if there is a problem, set the error, and return FALSE
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	EditorStartReplace(theUniverse);									/* we are going to change text in an edit universe */
	fail=!ReplaceEditorFile(theUniverse,0,theUniverse->textUniverse->totalBytes,thePath);	/* replace all that was there */
	EditorEndReplace(theUniverse);										/* text change is complete */
	EditorSetNormalSelection(theUniverse,0,0);							/* move the cursor to the top after the change */
	EditorClearUndo(theUniverse);
	theUniverse->dirty=FALSE;
	return(!fail);
}

BOOLEAN SaveEditorUniverse(EDITORUNIVERSE *theUniverse,char *thePath,BOOLEAN clearDirty)
/* save theUniverse to thePath
 * if clearDirty is TRUE, then clear the buffer's dirty flag
 * if there is a problem, set the error, and return FALSE
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT8
		*theBuffer;
	EDITORFILE
		*theFile;
	UINT32
		numWritten;
	UINT32
		bytesLeft,
		bytesToWrite;
	BOOLEAN
		fail;

	fail=FALSE;
	EditorStartTextChange(theUniverse);									/* start text change, so that status bars can update if needed */
	if(theFile=OpenEditorWriteFile(thePath))							/* open the file to write to */
		{
		if(theBuffer=(UINT8 *)MNewPtr(LOADSAVEBUFFERSIZE))				/* create an intermediate buffer to speed the save */
			{
			bytesLeft=theUniverse->textUniverse->totalBytes;
			theChunk=theUniverse->textUniverse->firstChunkHeader;
			theOffset=0;
			while(bytesLeft&&!fail)										/* read until we reach the end of the file */
				{
				if(bytesLeft>LOADSAVEBUFFERSIZE)						/* see how many bytes to extract this time */
					{
					bytesToWrite=LOADSAVEBUFFERSIZE;
					}
				else
					{
					bytesToWrite=bytesLeft;
					}
				if(ExtractUniverseText(theUniverse->textUniverse,theChunk,theOffset,theBuffer,bytesToWrite,&theChunk,&theOffset))	/* read data from universe into buffer */
					{
					if(WriteEditorFile(theFile,theBuffer,bytesToWrite,&numWritten))	/* write the buffer contents to the file */
						{
						bytesLeft-=bytesToWrite;						/* decrement bytes remaining to be written */
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				}
			MDisposePtr(theBuffer);
			}
		else
			{
			fail=TRUE;
			}
		CloseEditorFile(theFile);
		}
	else
		{
		fail=TRUE;
		}
	if(!fail&&clearDirty)
		{
		theUniverse->dirty=FALSE;					/* make it clean again if asked to */
		}
	EditorEndTextChange(theUniverse);				/* text change is complete */
	return(!fail);
}

void LinkViewToEditorUniverse(EDITORVIEW *theView,EDITORUNIVERSE *theUniverse)
/* connect theView to theUniverse (at the start)
 */
{
	theView->editorUniverse=theUniverse;
	theView->nextUniverseView=theUniverse->firstView;
	theUniverse->firstView=theView;
}

void UnlinkViewFromEditorUniverse(EDITORVIEW *theView)
/* disconnect theView from its universe
 */
{
	EDITORVIEW
		*previousView,
		*currentView;

	previousView=NULL;
	currentView=theView->editorUniverse->firstView;
	while(currentView&&currentView!=theView)
		{
		previousView=currentView;
		currentView=currentView->nextUniverseView;
		}
	if(currentView)
		{
		if(previousView)
			{
			previousView->nextUniverseView=theView->nextUniverseView;
			}
		else
			{
			theView->editorUniverse->firstView=theView->nextUniverseView;
			}
		}
}

BOOLEAN GotoEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark)
/* set the editor's selection to be the same as theMark
 * if there is a problem here, SetError, return FALSE, and leave the selection
 * unchanged
 */
{
	SELECTIONUNIVERSE
		*newUniverse;

	if(newUniverse=CopySelectionUniverse(theMark->selectionUniverse))
		{
		EditorStartSelectionChange(theUniverse);
		CloseSelectionUniverse(theUniverse->selectionUniverse);
		theUniverse->selectionUniverse=newUniverse;
		EditorEndSelectionChange(theUniverse);
		}
	return(FALSE);
}

/* --------------------------------------------------------------------------------------------------------------------------
 * creation/deletion functions
 */

MARKLIST *LocateEditorMark(EDITORUNIVERSE *theUniverse,char *markName)
/* attempt to locate a mark with the given name on theUniverse
 * if one is found, return it, else, return NULL
 */
{
	BOOLEAN
		found;
	MARKLIST
		*currentMark;

	currentMark=theUniverse->theMarks;
	found=FALSE;
	while(currentMark&&!found)
		{
		if(currentMark->markName&&(strcmp(currentMark->markName,markName)==0))
			{
			found=TRUE;
			}
		else
			{
			currentMark=currentMark->nextMark;			/* locate the mark to delete */
			}
		}
	return(currentMark);
}

static void UnlinkEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark)
/* unlink a mark from the editor universe
 */
{
	MARKLIST
		*previousMark,
		*currentMark;

	previousMark=NULL;
	currentMark=theUniverse->theMarks;
	while(currentMark&&(currentMark!=theMark))
		{
		previousMark=currentMark;
		currentMark=currentMark->nextMark;			/* locate the mark to delete */
		}
	if(currentMark)									/* make sure we located it */
		{
		if(previousMark)							/* see if it was at the top */
			{
			previousMark->nextMark=currentMark->nextMark;	/* link across */
			}
		else
			{
			theUniverse->theMarks=currentMark->nextMark;	/* set new top of list */
			}
		}
}

static void LinkEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark)
/* link a mark to the editor universe
 */
{
	theMark->nextMark=theUniverse->theMarks;				/* link onto start of list */
	theUniverse->theMarks=theMark;
}

MARKLIST *SetEditorMark(EDITORUNIVERSE *theUniverse,char *markName)
/* add a mark to the editor universe with markName
 * if a mark already exists with the given name, reset it to the current
 * selection
 * if there is a problem, SetError, and return NULL (if the mark existed, and we have
 * a problem, it gets deleted)
 */
{
	MARKLIST
		*theMark;

	if(theMark=LocateEditorMark(theUniverse,markName))
		{
		CloseSelectionUniverse(theMark->selectionUniverse);	/* get rid of the old universe */
		if(theMark->selectionUniverse=CopySelectionUniverse(theUniverse->selectionUniverse))
			{
			return(theMark);
			}
		else
			{
			UnlinkEditorMark(theUniverse,theMark);
			MDisposePtr(theMark->markName);
			MDisposePtr(theMark);
			}
		}
	else
		{
		if(theMark=(MARKLIST *)MNewPtr(sizeof(MARKLIST)))
			{
			if(theMark->selectionUniverse=CopySelectionUniverse(theUniverse->selectionUniverse))	/* copy the current selection as the new mark */
				{
				if(theMark->markName=(char *)MNewPtr(strlen(markName)+1))
					{
					strcpy(theMark->markName,markName);			/* copy the name */
					LinkEditorMark(theUniverse,theMark);
					return(theMark);
					}
				CloseSelectionUniverse(theMark->selectionUniverse);
				}
			MDisposePtr(theMark);
			}
		}
	return(NULL);
}

void ClearEditorMark(EDITORUNIVERSE *theUniverse,MARKLIST *theMark)
/* remove a mark to the editor universe
 */
{
	UnlinkEditorMark(theUniverse,theMark);
	MDisposePtr(theMark->markName);
	CloseSelectionUniverse(theMark->selectionUniverse);		/* close the selections */
	MDisposePtr(theMark);									/* get rid of the mark */
}

EDITORUNIVERSE *OpenEditorUniverse()
/* open an editor universe (editor universe contains both text and selection
 * universes (which are created here))
 * if there is a problem, set the error, and return NULL
 */
{
	EDITORUNIVERSE
		*theEditorUniverse;

	if(theEditorUniverse=(EDITORUNIVERSE *)MNewPtr(sizeof(EDITORUNIVERSE)))
		{
		if(theEditorUniverse->textUniverse=OpenTextUniverse())						/* create the text universe */
			{
			if(theEditorUniverse->selectionUniverse=OpenSelectionUniverse())		/* create the selection universe */
				{
				if(theEditorUniverse->auxSelectionUniverse=OpenSelectionUniverse())		/* create the aux selection universe (used for tasks, so they can write data into a place other than at the cursor) */
					{
					if(theEditorUniverse->undoUniverse=OpenUndoUniverse())				/* create the undo universe */
						{
						theEditorUniverse->theMarks=NULL;								/* no marks on this universe yet */

						theEditorUniverse->anchorStartPosition=0;						/* clear selection handling variables */
						theEditorUniverse->anchorEndPosition=0;
						theEditorUniverse->anchorLine=0;
						theEditorUniverse->anchorX=0;
						theEditorUniverse->columnarTopLine=0;
						theEditorUniverse->columnarBottomLine=0;
						theEditorUniverse->columnarLeftX=0;
						theEditorUniverse->columnarRightX=0;
						theEditorUniverse->haveStartX=FALSE;
						theEditorUniverse->haveEndX=FALSE;
						theEditorUniverse->desiredStartX=0;
						theEditorUniverse->desiredEndX=0;
						theEditorUniverse->dirty=FALSE;									/* not dirty yet */
						theEditorUniverse->firstView=NULL;								/* no views onto this universe yet */
						return(theEditorUniverse);
						}
					CloseSelectionUniverse(theEditorUniverse->auxSelectionUniverse);
					}
				CloseSelectionUniverse(theEditorUniverse->selectionUniverse);
				}
			CloseTextUniverse(theEditorUniverse->textUniverse);
			}
		MDisposePtr(theEditorUniverse);
		}
	return(NULL);
}

void CloseEditorUniverse(EDITORUNIVERSE *theEditorUniverse)
/* dispose of an editor universe
 * it is a mistake to try to close an editor universe that still
 * has views
 */
{
	while(theEditorUniverse->theMarks)
		{
		ClearEditorMark(theEditorUniverse,theEditorUniverse->theMarks);				/* get rid of all marks on this universe */
		}
	CloseUndoUniverse(theEditorUniverse->undoUniverse);
	CloseSelectionUniverse(theEditorUniverse->auxSelectionUniverse);
	CloseSelectionUniverse(theEditorUniverse->selectionUniverse);
	CloseTextUniverse(theEditorUniverse->textUniverse);
	MDisposePtr(theEditorUniverse);
}
