#include "Radio.h++"
#include "uLaw2short.h"
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/time.h>

#if	defined(sun)
extern "C" char *strerror(int);
#endif

Interface::Interface(char *n)
{
    memset((void *)&sin, 0, sizeof(sockaddr_in));
    memset((void *)&netmask, 0, sizeof(in_addr));
    strncpy(name, n, IFNAMSIZ);
}

int
Interface::Send(int fd, void *data, int length)
{
    return (sendto(fd, (char *)data, length, 0,
		   (struct sockaddr *)&sin, sizeof(sin)));
}


RadioInfo::RadioInfo()
{
    sendtype = RADIO_SUN_ULAW;
    sendchannel = DEFAULT_AUDIO_CHANNEL;
    eprint = _ErrorPrint;
    dprint = _DebugPrint;

    source = 0;
    sourcefd = -1;
    icnt = 0;
    port = htons('TK');
    noise = 50;

    memset((void *)&recv, 0, sizeof(recv));
    recv.sin_family = AF_INET;
    recv.sin_port = htons(port);
    recv.sin_addr.s_addr = htonl(INADDR_ANY);

    if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("socket");
    }

    struct ifconf ifc;
    struct ifreq *ifr;
    char ifbuf[8192];

    ifc.ifc_len = sizeof(ifbuf);
    ifc.ifc_buf = ifbuf;

    if (ioctl(s, SIOCGIFCONF, (char *)&ifc) < 0) {
	perror("SIOCGIFCONF");
    }

    ifr = ifc.ifc_req;

    for (int n = ifc.ifc_len / sizeof(struct ifreq); n-- > 0; ++ifr) {
	if (ifr->ifr_addr.sa_family != AF_INET)
	    continue;

	if (ioctl(s, SIOCGIFFLAGS, (char *)ifr) < 0)
		continue;
	if (((ifr->ifr_flags & IFF_UP) == 0) ||
	     (ifr->ifr_flags & IFF_LOOPBACK) ||
	     (ifr->ifr_flags & IFF_POINTOPOINT) ||
	     (ifr->ifr_flags & IFF_BROADCAST) == 0) {
	    continue;
	}

	if (ioctl(s, SIOCGIFBRDADDR, (caddr_t)ifr) < 0) {
	    perror("SIOCGIFBRDADDR");
	}

	if (icnt < MAX_INTERFACES) {
	    interfaces[icnt] = new Interface(ifr->ifr_name);
	    interfaces[icnt]->SetFamily(AF_INET);
	    interfaces[icnt]->SetPort(port);
	    interfaces[icnt]->SetAddress(ifr->ifr_broadaddr);

            if (ioctl(s, SIOCGIFNETMASK, (caddr_t)ifr) < 0) {
                perror("SIOCGIFNETMASK");
            }
	    interfaces[icnt]->SetNetMask(ifr->ifr_addr);
	    ++icnt;
	}
    }

    int on = 1;
    if (setsockopt(s, SOL_SOCKET, SO_BROADCAST, (char *)&on, sizeof(on)) < 0) {
	perror("SO_BROADCAST");
    }
}

void
RadioInfo::perror(char *msg)
{
    sprintf(printbuf, "%s: %s", msg, strerror(errno));
    (*eprint)(printbuf);
}

void
RadioInfo::PrintDebug()
{
    for (int i = 0; i < icnt; ++i) {
	sprintf(printbuf, "%-16s Broadcasts on %s",
	    interfaces[i]->Name(),
	    (char *)(*interfaces[i]));
	(*dprint)(printbuf);
    }
    if (source) {
	sprintf(printbuf, "%-16s Broadcasts on %s",
	    source->Name(), (char *)(*source));
	(*dprint)(printbuf);
    }
}

void
RadioInfo::SetPort(u_short p)
{
    port = p;
    for (int i = 0; i < icnt; ++i) {
	interfaces[i]->SetPort(port);
    }
    if (source)
	source->SetPort(port);
    recv.sin_port = port;
}

void
RadioInfo::SetSource(char *name)
{
    if (!name)
	name = interfaces[0]->Name();
    if (source || sourcefd >= 0) {
	sprintf(printbuf, "%s: Source already set to %s", name, source->Name());
	(*eprint)(printbuf);
    }
    if (!strcmp(name, "-")) {
	sourcefd = 0;
	return;
    }
    for (int i = 0; i < icnt; ++i) {
	if (!strncmp(interfaces[i]->Name(), name, IFNAMSIZ)) {
	    source = interfaces[i];
	    --icnt;
	    while (i < icnt) {
		interfaces[i] = interfaces[i+1];
		++i;
	    }
	    return;
	}
    }
    sprintf(printbuf, "%s: no such interface", name);
    (*eprint)(printbuf);
}

void
RadioInfo::Exclude(char *name)
{
    for (int i = 0; i < icnt; ++i) {
	if (!strncmp(interfaces[i]->Name(), name, IFNAMSIZ)) {
	    --icnt;
	    while (i < icnt) {
		interfaces[i] = interfaces[i+1];
		++i;
	    }
	    return;
	}
    }
}

void
RadioInfo::Bind()
{
    if (bind(s, (struct sockaddr *)&recv, sizeof(recv)) < 0) {
	perror("bind");
    }
}

int
RadioInfo::ReadPacket()
{
    for (;;) {
	int r = PollPacket();
	if (r)
	    return(r);
    }
}

int
RadioInfo::PollPacket()
{
    if (sourcefd >= 0) {
	plen = read(sourcefd, (char *)dp.data, sizeof(dp.data));

	if (plen <= 0)
	    return(-1);

	dp.header.type = sendtype;
	dp.header.channel = sendchannel;
    } else {
	sockaddr_in from;
	int len = sizeof(from);

	plen = recvfrom(s, (char *)&dp, sizeof(dp), 0,
			    (struct sockaddr *)&from, &len);

	if (plen <= 0) {
	    if (plen < 0 && errno != EINTR) {
		perror("recvfrom");
	    }
	    return(0);
	}

	if (plen < sizeof(dp.header))
	    return(0);

	if (source && !source->Match(from.sin_addr))
	    return(0);

	if (debug) {
	    sprintf(printbuf, "%s packet came in on channel %d from %s", 
		    typetostr(dp.header.type), dp.header.channel,
		    inet_ntoa(from.sin_addr));
	    (*dprint)(printbuf);
	}
	plen -= sizeof(dp.header);
    }
    /*
     * filter out "silent" packets
     */
    if (dp.header.type == RADIO_SUN_ULAW) {
	for (int i = 0; i < plen; ++i) {
	    if (abs(uLaw2short[dp.data[i]]) > noise)
		break;
	}
	if (i >= plen)
	    return(0);
    }
    return(plen);
}

void
RadioInfo::SendMessage(void *msg, int len)
{
    memcpy(dp.data, msg, len);
    BroadcastPacket();
}

void
RadioInfo::BroadcastPacket()
{
    for (int i = 0; i < icnt; ++i) {
	if (interfaces[i]->Send(s, &dp, plen + sizeof(dp.header)) < 0) {
	    sprintf(printbuf, "%s: sendto: %s", interfaces[i]->Name(),
		    strerror(errno));
	    (*eprint)(printbuf);
	}
	if (debug) {
	    sprintf(printbuf, "Sent packet to %s", interfaces[i]->Name());
	    (*dprint)(printbuf);
	}
    }
}

void
RadioInfo::WriteData(int fd)
{
    write(fd, (char *)dp.data, plen);
}

struct ChannelTypes {
    char        type;
    char        channel;
    char        *name;
    char        *description;
} types[] = {
    { 0,                0,                      "-NULL-",       "Null", },
    { RADIO_SUN_ULAW,   RADIO_AUDIO_CHANNEL,    "audio",        "Sun Audio", },
    { RADIO_TEXT,       RADIO_LISTING_CHANNEL,  "text",         "Radio Text", },
    { 0, },
};

char *
typetostr(int type)
{
    struct ChannelTypes *ct = types;

    while (ct->name) {
	if (type == ct->type)
	    return(ct->description);
	++ct;
    }
    return("unknown");
}

int
strtotype(char *type)
{
    struct ChannelTypes *ct = types;

    while (ct->name) {
	if (!strcasecmp(type, ct->name))
	    return(ct->type);
	++ct;
    }
    return(-1);
}

int
strtochannel(char *type)
{
    struct ChannelTypes *ct = types;

    while (ct->name) {
	if (!strcasecmp(type, ct->name))
	    return(ct->channel);
	++ct;
    }
    return(-1);
}

void 
_ErrorPrint(char *msg)
{
    fprintf(stderr, "%s\n", msg);
    exit(1);
}

void 
_DebugPrint(char *msg)
{
    fprintf(stderr, "%s\n", msg);
}
