//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%	%I%	%G% %U%"); }
static void RSCID() { void("$Id: CRT.cc,v 1.12 1995/06/04 07:14:49 prb Exp $"); }
#include <Cvo/CRT.h++>
#include <X11/Xatom.h>

static Cvo_Default defaults[] = {
        "*CvoCRT*Sunken:True",
        "*CvoCRT*Pad:2",
        "*CvoCRT.Cursor:xterm",
	"*CvoCRT*FontFamily:Helvetica",
	"*CvoCRT*TabStop:8",
};

static char *trans = "\
    <Btn1Down>:select-start() \n\
    <Btn1Down>(2):select-start(WORD) \n\
    <Btn1Down>(3):select-start(LINE) \n\
    <Btn1Motion>:select-extend() \n\
    <Btn3Down>:start-extend() \n\
    <Btn3Motion>:select-extend() \n\
    <Btn1Up>:select-end(PRIMARY, CUT_BUFFER0) \n\
    <Btn1Up>(2):select-end(PRIMARY, CUT_BUFFER0) \n\
    <Btn1Up>(3):select-end(PRIMARY, CUT_BUFFER0) \n\
    <Btn3Up>:select-end(PRIMARY, CUT_BUFFER0) \n\
";
//  <Btn2Down>:ignore() \n\

CONSTRUCTORS(Cvo_CRT, Cvo_Window, "CvoCRT")
CVO_CREATE_REGISTER_FUNCTIONS(Cvo_CRT)

CVO_BEGIN_BIND(Cvo_CRT)
CVO_DOBIND(Cvo_CRT, "select-start",     SelectStart)
CVO_DOBIND(Cvo_CRT, "select-extend",    SelectExtend)
CVO_DOBIND(Cvo_CRT, "start-extend",     StartExtend)
CVO_DOBIND(Cvo_CRT, "select-end",       SelectEnd)
CVO_END_BIND(Cvo_CRT, Cvo_Window)

inline int ComputeWchars(int x)
{
    return((x * sizeof(wchar_t) + sizeof(Cvo_CRT_LogType) - 1)
	    / sizeof(Cvo_CRT_LogType));
}

inline int ComputeAttrs(int x)
{
    return((x * sizeof(Cvo_TextAttribute *) + sizeof(Cvo_CRT_LogType) - 1)
	    / sizeof(Cvo_CRT_LogType));
}

void
Cvo_CRT::_Init()
{   CVO_ENTER

    type = CvoT_Text;
    LoadFont();
    ClearLog();
    cursorx = 0;
    cursory = 0;
    showcursor = False;
    dontdraw = False;
    currentstate = TextAttribute();
    highlights = 0;
    highlightattr = 0;
    anchor_row = -1;
    anchor_col = -1;
    charmode = wordmode = linemode = False;

    jumpscroll = GetResourceTruth(_QjumpScroll, _QJumpScroll, True);

    maxlines = rows = 1;
    extrapad = 0;
    lines = new Cvo_TextLine *[maxlines];
    for (int j = 0; j < maxlines; ++j) {
	lines[j] = new Cvo_TextLine(this, width/TextAttribute()->MWidth());
	lines[j]->SetState(currentstate);
	lines[j]->ChangeY(ComputeY(j), j);
	lines[j]->MakeDirty();
    }
    tabsize = 0;
    origin = 0;
    Register(CvoResizeEvent, &Cvo_CRT::DoResize);
    Register(NoExpose, &Cvo_CRT::CopyDoneEvent);
    Register(GraphicsExpose, &Cvo_CRT::DoGraphicsExpose);

    Register(SelectionClear, &Cvo_CRT::SelectClear);
    Cvo_Object::Register(SelectionRequest, &Cvo_Object::SelectRequest);

    AddTranslations(trans);
    CVO_VOID_RETURN
}

void
Cvo_CRT::_Create()
{   CVO_ENTER
    if (Object())
        CVO_VOID_RETURN

    Cvo_Window::_Create();
    SetGraphicsExposures(True);

    Cvo_TextAttribute *ta = TextAttribute();

    for (int j = 0; j < maxlines; ++j) {
  	lines[j]->SetState(TextAttribute());
  	lines[j]->ChangeY(ComputeY(j), j);
  	lines[j]->MakeDirty();
    }
    if (!tabstop)
	InitTabs();
    CVO_VOID_RETURN
}

void
Cvo_CRT::ComputeUserSizes(Cvo_FrameSize &mymin, Cvo_FrameSize &mydes)
{   CVO_ENTER
    int fh = 1;
    int fw = 1;

    Cvo_Window *win = ToWindow();

    if (win) {
        fh = FH();
        fw = win->TextAttribute()->MWidth();
    }

    mymin.width = fw * minSize.wchars + minSize.wpixel;
    mymin.height = fh * minSize.hchars + extrapad + minSize.hpixel;

    mydes.width = fw * defSize.wchars + defSize.wpixel;
    mydes.height = fh * defSize.hchars + extrapad + defSize.hpixel;

    layout.maximum.width = fw * maxSize.wchars + maxSize.wpixel;
    layout.maximum.height = fh * maxSize.hchars + maxSize.hpixel;
    if (fh * maxSize.hchars)
	layout.maximum.height += extrapad;


    if (!mymin.width)
        mymin.width = CalculateWidth();
    if (!mymin.height)
        mymin.height = CalculateHeight();

    if (!mydes.width)
        mydes.width = mymin.width;
    if (!mydes.height)
        mydes.height = mymin.height;

    CVO_VOID_RETURN
}

void
Cvo_CRT::InitTabs()
{   CVO_ENTER
    tabstop = GetResourceInt(_QtabStop, _QTabStop, 8);
    if (!tabstop)
	tabstop = 8;
    tabsize = TextAttribute()->MWidth() * tabstop;
    CVO_VOID_RETURN
}

void
Cvo_CRT::AllocateLines(int nm)
{   CVO_ENTER
    if (!Object())
	rows = nm;

    if (nm <= maxlines)
	CVO_VOID_RETURN

    Cvo_TextLine **n = new Cvo_TextLine *[nm];

    for (int j = 0; j < maxlines; ++j)
	n[j] = lines[j];

    delete lines;
    lines = n;
    for (j = maxlines; j < nm; ++j) {
	lines[j] = new Cvo_TextLine(this, width/TextAttribute()->MWidth());
	lines[j]->SetState(TextAttribute());
  	lines[j]->ChangeY(ComputeY(j), j);
  	lines[j]->MakeDirty();
    }

    maxlines = nm;
    CVO_VOID_RETURN
}

void
Cvo_CRT::DoResize(XEvent *ev, void *)
{   CVO_ENTER
    Cvo_ResizeEvent *re = (Cvo_ResizeEvent *)ev;

    Cvo_TextChangedSizeEvent tcse;

    tcse.oldrows = rows;
    tcse.oldwidth = (re->old_width - 2 * extrapad) / TextAttribute()->MWidth();

    int orows = rows;
    int nrows = ComputeRows();

    AllocateLines(nrows);

    while (orows > nrows)
	lines[--orows]->Clear();
    for (orows = 0; orows < nrows; ++orows) {
	lines[orows]->ChangeY(ComputeY(orows), orows);
	lines[orows]->MakeDirty();
    }

    rows = nrows;
    if (Object() && lines[rows-1]->NextY() < height)
	ClearArea(0, lines[rows-1]->NextY(),
		     width, height - lines[rows-1]->NextY());

    tcse.rows = rows;
    tcse.width = (Width() - 2 * extrapad) / TextAttribute()->MWidth();
    SendEvent(CvoTextChangedSizeEvent, &tcse);
    CVO_VOID_RETURN
}


void
Cvo_CRT::SetExtraPad(int p)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2);
    if (log) {
	log[0] = Cvo_CRT_SetExtraPad;
	log[1] = p;
	CVO_VOID_RETURN
    }

    if (p == extrapad)
	CVO_VOID_RETURN

    Cvo_TextChangedSizeEvent tcse;

    tcse.oldrows = rows;
    tcse.oldwidth = (Width() - 2 * extrapad) / TextAttribute()->MWidth();

    extrapad = p;

    //
    // If we have no size yet, we cannot calculate the new number of lines
    // in the object
    //
    if (!Height())
	CVO_VOID_RETURN

    int nrows = ComputeRows();
    int orows = rows;

    AllocateLines(nrows);

    while (orows > nrows)
	lines[--orows]->Clear();

    for (orows = 0; orows < nrows; ++orows) {
	lines[orows]->ChangeY(ComputeY(orows), orows);
	lines[orows]->MakeDirty();
    }

    rows = nrows;
    if (Object())
	Redraw();

    tcse.rows = rows;
    tcse.width = (Width() - 2 * extrapad) / TextAttribute()->MWidth();
    SendEvent(CvoTextChangedSizeEvent, &tcse);
    CVO_VOID_RETURN
}

void
Cvo_CRT::ShowCursor()
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(1);
    if (log) {
	log[0] = Cvo_CRT_ShowCursor;
	CVO_VOID_RETURN
    }

    if (showcursor)
	CVO_VOID_RETURN

    if (!registeredfocus) {
        registeredfocus = True;
        Register(CvoFocusInEvent, &Cvo_CRT::FocusInEvent);
        Register(CvoFocusOutEvent, &Cvo_CRT::FocusOutEvent);
    }

    showcursor = True;
    if (int(cursory) < rows)
	lines[cursory]->SetCursor(cursorx, fixedwidth);
    CVO_VOID_RETURN
}

void
Cvo_CRT::FocusInEvent(XEvent *, void *)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(1);
    if (log) {
	log[0] = Cvo_CRT_FocusInEvent;
	CVO_VOID_RETURN
    }
#if 0
    if (showcursor && int(cursory) < rows) {
	lines[cursory]->MakeDirty();
	Flush();
    }
#endif
    CVO_VOID_RETURN
}

void
Cvo_CRT::FocusOutEvent(XEvent *, void *)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(1);
    if (log) {
	log[0] = Cvo_CRT_FocusOutEvent;
	CVO_VOID_RETURN
    }
#if 0
    if (showcursor && int(cursory) < rows) {
	lines[cursory]->MakeDirty();
	Flush();
    }
#endif
    CVO_VOID_RETURN
}

void
Cvo_CRT::HideCursor()
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(1);
    if (log) {
	log[0] = Cvo_CRT_HideCursor;
	CVO_VOID_RETURN
    }
    if (!showcursor)
	CVO_VOID_RETURN
    showcursor = False;
    if (int(cursory) < rows)
	lines[cursory]->UnsetCursor();
    CVO_VOID_RETURN
}

void
Cvo_CRT::MoveCursor(int line, int col)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(3);
    if (log) {
	log[0] = Cvo_CRT_MoveCursor;
	log[1] = line;
	log[2] = col;
	CVO_VOID_RETURN
    }
    if (showcursor && int(cursory) < rows)
	lines[cursory]->UnsetCursor();

    if (line < 0)
	line = 0;
    if (line >= rows)
	line = rows - 1;
    if (col < 0)
	col = 0;
    cursory = line;
    cursorx = col;
    if (showcursor && int(cursory) < rows)
	lines[cursory]->SetCursor(cursorx, fixedwidth);
    CVO_VOID_RETURN
}

void
Cvo_CRT::Exposure(XExposeEvent *ev)
{
    Cvo_Window::Exposure(ev);
    DoExpose((XEvent *)ev, 0);
}

void
Cvo_CRT::DoExpose(XEvent *ev, void *)
{   CVO_ENTER
    if (dontdraw)
	CVO_VOID_RETURN
    int from = 0;
    int to = 0;

    if (rows > 1) {
	from = ev ? FindLine(ev->xexpose.y) : 0;
	to = ev ? FindLine(ev->xexpose.y + ev->xexpose.height) : rows - 1;
    }
    for (int i = from; i <= to; ++i) {
	lines[i]->MakeDirty();
	lines[i]->Draw(extrapad, Width() - 2 * extrapad);
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT::DoGraphicsExpose(XEvent *ev, void *d)
{   CVO_ENTER
    DoExpose(ev, d);
    CopyDoneEvent(ev, d);
    CVO_VOID_RETURN
}

Cvo_TextAttribute *
Cvo_CRT::SetState(Cvo_TextAttribute *s)
{
    Cvo_TextAttribute *os = currentstate;
    currentstate = s;

#define	STA	sizeof(Cvo_TextAttribute *)
#define	SCL	sizeof(Cvo_CRT_LogType)

    Cvo_CRT_LogType *log = NeedLog(1 + (STA + SCL - 1)/SCL);
    if (log) { 
        log[0] = Cvo_CRT_SetState;
    	*(Cvo_TextAttribute **)&log[1] = currentstate;
    }                

    return(os);
}

void
Cvo_CRT::InsertLine(int line, int cnt)
{   CVO_ENTER

    if (!jumpscroll)
	WaitForCopy();

    Cvo_CRT_LogType *log = NeedLog(3);
    if (log) {
	log[0] = Cvo_CRT_InsertLine;
	log[1] = line;
	log[2] = cnt;
	CVO_VOID_RETURN
    }

    if (line < 0 || line >= rows || cnt < 1)
	CVO_VOID_RETURN
    if (line + cnt > rows)
	cnt = rows - line;

    Cvo_CRT_Highlight *hl = highlights;

    while (hl) {
	hl->InsertLines(line, cnt);
        hl = hl->Next();
    }

    //
    // Calculate the height of a new line to determine how much space will
    // actually be scrolled
    //
    int nlh = TextAttribute()->Font()->Ascent() +
	      TextAttribute()->Font()->Descent() + extrapad;
    nlh *= cnt;

    if (!dontdraw) {
	int ep = extrapad >> 1;
	if (!Object())
	    ;
	else if (line + cnt == rows) {
	    ClearArea(0, ComputeY(line) - ep,
		      width, height - ComputeY(line) + ep);
	} else {
	    CopyArea(0, ComputeY(line) - ep,
		     width, height - (ComputeY(line) - ep + nlh),
		     0, ComputeY(line) - ep + nlh);
	}
    } else
	scrolled = True;

    while (cnt-- > 0) {
	Cvo_TextLine *save = lines[rows-1];
	for (int x = rows-1; x > line; --x) {
	    lines[x] = lines[x-1];
	}

	lines[line] = save;
	lines[line]->Clear();
	lines[line]->SetState(currentstate);
    }
    RecalculateYs(False);
    if (lines[rows-1]->Raised()) {
	WaitForCopy();
	lines[rows-1]->MakeDirty();
	lines[rows-1]->ReDraw();
    }

    CVO_VOID_RETURN
}

void
Cvo_CRT::DeleteLine(int line, int cnt)
{   CVO_ENTER

    if (!jumpscroll)
	WaitForCopy();

    Cvo_CRT_LogType *log = NeedLog(3);
    if (log) {
	log[0] = Cvo_CRT_DeleteLine;
	log[1] = line;
	log[2] = cnt;
	CVO_VOID_RETURN
    }

    if (line < 0 || line >= rows || cnt < 1)
	CVO_VOID_RETURN

    if (line + cnt > rows)
	cnt = rows - line;

    Cvo_CRT_Highlight *hl = highlights;

    while (hl) {
	hl->DeleteLines(line, cnt);
        hl = hl->Next();
    }

    if (!dontdraw) {
	int ep = extrapad>>1;
	if (!Object())
	    ;
	else if (line + cnt == rows) {
	    ClearArea(0, ComputeY(line)-ep, width, height - ComputeY(line)-ep);
	} else {
	    CopyArea(0,		ComputeY(line+cnt)-ep,
		     width,	height - (ComputeY(line+cnt)-(ep<<1)),
		     0,		ComputeY(line)-ep);
	}
    } else
	scrolled = True;
    while (cnt-- > 0) {
	Cvo_TextLine *save = lines[line];

	for (int x = line; x < rows-1; ++x) {
	    lines[x] = lines[x+1];
	}

	lines[rows-1] = save;
	lines[rows-1]->Clear();
	lines[line]->SetState(currentstate);
    }

    RecalculateYs(False);
    CVO_VOID_RETURN
}

void
Cvo_CRT::InsertAt(int line, int col, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(4);
    if (log) {
	log[0] = Cvo_CRT_InsertBlanksAt;
	log[1] = line;
	log[2] = col;
	log[3] = cnt;
	CVO_VOID_RETURN
    }

    if (!SetStateFor(line))
	CVO_VOID_RETURN
    lines[line]->Insert(col, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::InsertAt(int line, int col, wchar_t *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(3, str, cnt);
    if (log) {
	log[0] = Cvo_CRT_InsertAt;
	log[1] = line;
	log[2] = col;
	CVO_VOID_RETURN
    }

    if (!SetStateFor(line))
	CVO_VOID_RETURN
    lines[line]->Insert(col, str, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::InsertAt(int line, int col, char *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(3, str, cnt);
    if (log) {
	log[0] = Cvo_CRT_InsertAt;
	log[1] = line;
	log[2] = col;
	CVO_VOID_RETURN
    }

    if (line < 0 || line >= rows)
	CVO_VOID_RETURN
    tbuf.Set(str, cnt);
    if (!SetStateFor(line))
	CVO_VOID_RETURN
    lines[line]->Insert(col, tbuf.wcValue(), tbuf.wcLength());
    CVO_VOID_RETURN
}

void
Cvo_CRT::ReplaceAt(int line, int col, wchar_t *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(3, str, cnt);
    if (log) {
        log[0] = Cvo_CRT_ReplaceAt;
        log[1] = line;
        log[2] = col;
        CVO_VOID_RETURN
    }

    if (!SetStateFor(line))
	CVO_VOID_RETURN
    lines[line]->Replace(col, str, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::ReplaceAt(int line, int col, char *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(3, str, cnt);
    if (log) {
        log[0] = Cvo_CRT_ReplaceAt;
        log[1] = line;
        log[2] = col;
        CVO_VOID_RETURN
    }
    if (line < 0 || line >= rows)
	CVO_VOID_RETURN
    tbuf.Set(str, cnt);
    if (!SetStateFor(line))
	CVO_VOID_RETURN
    lines[line]->Replace(col, tbuf.wcValue(), tbuf.wcLength());
    CVO_VOID_RETURN
}

void
Cvo_CRT::DeleteAt(int line, int col, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(4);
    if (log) {
        log[0] = Cvo_CRT_DeleteAt;
        log[1] = line;
        log[2] = col;
        log[3] = cnt;
        CVO_VOID_RETURN
    }
    if (line < 0 || line >= rows)
	CVO_VOID_RETURN
    lines[line]->Delete(col, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::AppendTo(int line, char *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2, str, cnt);
    if (log) {
        log[0] = Cvo_CRT_AppendTo;
        log[1] = line;
        CVO_VOID_RETURN
    }
    if (line < 0 || line >= rows)
	CVO_VOID_RETURN
    tbuf.Set(str, cnt);
    if (!SetStateFor(line)) {
	CVO_VOID_RETURN
    }
    lines[line]->Append(tbuf.wcValue(), cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::AppendTo(int line, char *str, Cvo_TextAttribute **attrs, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2, str, attrs, cnt);
    if (log) {
        log[0] = Cvo_CRT_AppendToAttr;
        log[1] = line;
        CVO_VOID_RETURN
    }
    if (line < 0 || line >= rows)
	CVO_VOID_RETURN
    tbuf.Set(str, cnt);
    if (!SetStateFor(line)) {
	CVO_VOID_RETURN
    }
    lines[line]->Append(tbuf.wcValue(), attrs, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::AppendTo(int line, wchar_t *str, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2, str, cnt);
    if (log) {
        log[0] = Cvo_CRT_AppendTo;
        log[1] = line;
        CVO_VOID_RETURN
    }
    if (!SetStateFor(line)) {
	CVO_VOID_RETURN
    }
    lines[line]->Append(str, cnt);
    CVO_VOID_RETURN
}

void
Cvo_CRT::AppendTo(int line, wchar_t *str, Cvo_TextAttribute **attrs, int cnt)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2, str, attrs, cnt);
    if (log) {
        log[0] = Cvo_CRT_AppendToAttr;
        log[1] = line;
        CVO_VOID_RETURN
    }
    if (!SetStateFor(line)) {
	CVO_VOID_RETURN
    }
    int olh = lines[line]->LineHeight();
    lines[line]->Append(str, attrs, cnt);
    if (olh != lines[line]->LineHeight())
	RecalculateYs();
    CVO_VOID_RETURN
}

BOOL
Cvo_CRT::OverFlowed(int line)
{   CVO_ENTER
    WaitForCopy();
    CVO_RETURN(lines[line]->Extra(Width() - 2 * extrapad) ? True : False)
}

void
Cvo_CRT::ChangeOrigin(int org)
{   CVO_ENTER
    Cvo_CRT_LogType *log = NeedLog(2);
    if (log) {
        log[0] = Cvo_CRT_ChangeOrigin;
        log[1] = org;
        CVO_VOID_RETURN
    }

    if (origin == org)
	CVO_VOID_RETURN
    origin = org;
    for (int j = 0; j < rows; ++j) {
	lines[j]->MakeDirty();
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT::Flush(int r)
{   CVO_ENTER
    if (!dontdraw && Object()) {
	for (int j = 0; j < rows; ++j)
	    lines[j]->Draw(extrapad, Width() - 2 * extrapad);
    }
    Cvo_Window::Flush(r);
    CVO_VOID_RETURN
}

void
Cvo_CRT::ClearWindow()
{   CVO_ENTER
    ClearLog();
    ClearArea(0, 0, width, height);
    for (int j = 0; j < rows; ++j)
	lines[j]->Clear();
    CVO_VOID_RETURN
}

CARD16
Cvo_CRT::CalculateHeight()
{   CVO_ENTER
    CVO_RETURN(FH() + extrapad)
}

CARD16
Cvo_CRT::CalculateWidth()
{   CVO_ENTER
    CVO_RETURN(TextAttribute()->MWidth() + 2 * extrapad)
}

int
Cvo_CRT::ComputeRows()
{   CVO_ENTER
    //
    // Be careful not to look at non-exsistant lines as this function
    // is called on a reconfig to determine how many lines we can actually
    // hold.  Also, always make sure we say there is at least 1 line.
    //
    if (lines[rows-1]->NextY() + extrapad <  Height()) {
	int h = Height() - (lines[rows-1]->NextY() + extrapad);
	CVO_RETURN(rows + (h / (TextAttribute()->MHeight() + extrapad)))
    } else {
	int n = rows - 1;
	while (n > 1 && lines[n]->NextY() + extrapad > Height()) {
	    --n;
	}
	CVO_RETURN(n < 1 ? 1 : n+1)
    }
}

void
Cvo_CRT::RecalculateYs(BOOL markdirty)
{   CVO_ENTER

    if (!Object())
	return;
    Cvo_TextChangedSizeEvent tcse;
    tcse.oldrows = rows;
    tcse.oldwidth = (Width() - 2 * extrapad) / TextAttribute()->MWidth();

    for (int j = 0; j < rows; ++j) {
	if (lines[j]->ChangeY(ComputeY(j), j) && markdirty)
	    lines[j]->MakeDirty();
    }

    int nrows = ComputeRows();
    if (nrows < rows)
	rows = nrows;
    else if (nrows > rows) {
	AllocateLines(nrows);
	while (rows < nrows) {
	    lines[rows]->Clear();
	    lines[rows]->SetState(TextAttribute());
	    lines[rows]->ChangeY(ComputeY(rows), rows);
	    ++rows;
	}
    }

    if (Object() && lines[rows-1]->NextY() < height)
	ClearArea(0, lines[rows-1]->NextY(),
		     width, height - lines[rows-1]->NextY());

    tcse.rows = rows;
    tcse.width = tcse.oldwidth;
    if (tcse.rows != tcse.oldrows)
	SendEvent(CvoTextChangedSizeEvent, &tcse);
    CVO_VOID_RETURN
}

BOOL
Cvo_CRT::WillFit(int line, Cvo_TextAttribute *at)
{   CVO_ENTER
    if (line < 0 || line >= rows)
	CVO_RETURN(False)
    if (at && lines[line]->Y() + at->MHeight()+ extrapad > height)
	CVO_RETURN(False)
    CVO_RETURN(True)
}

BOOL
Cvo_CRT::WillFit(int line, Cvo_TextAttribute **ats, int cnt)
{   CVO_ENTER
    if (line < 0 || line >= rows)
	CVO_RETURN(False)

    for (int j = 0; j < cnt; ++j)
	if (ats[j] && lines[line]->Y() + ats[j]->MHeight() + extrapad > height)
	    CVO_RETURN(False)
    CVO_RETURN(True)
}

int
Cvo_CRT::FindLine(int y)
{   CVO_ENTER
    WaitForCopy();
    for (int j = 0; j < rows; ++j)
	if (y < lines[j]->NextY())
	    CVO_RETURN(j)

    CVO_RETURN(rows-1)
}

int
Cvo_CRT::FindColumn(int line, int x)
{   CVO_ENTER
    WaitForCopy();
    if (line >= 0 && line < rows) {
	x = lines[line]->PixelIndex(x + origin) - 1;
	if (x < 0)
	    x = 0;
	CVO_RETURN(x)
    }
    CVO_RETURN(-1)
}

Cvo_TextAttribute *
Cvo_CRT::InHighlight(int row, int col, Cvo_TextAttribute *old)
{   CVO_ENTER
    Cvo_CRT_Highlight *hl = highlights;

    while (hl) {
	old = hl->Modify(row, col, old);
	hl = hl->Next();
    }
    CVO_RETURN(old)
}

Cvo_TextAttribute *
Cvo_CRT_Highlight::Modify(int row, int col, Cvo_TextAttribute *old)
{   CVO_ENTER
    if (!unset &&
	ToIndex(row, col) >= ToIndex(start_row, start_col) &&
        ToIndex(row, col) < ToIndex(end_row, end_col)) {
	    CVO_RETURN(Highlight(old))
    } else {
	    CVO_RETURN(old)
    }
}

Cvo_CRT_Highlight::Cvo_CRT_Highlight(Cvo_CRT_Highlight **root, Cvo_CRT *me)
              : Cvo_RootedList((Cvo_RootedList **)root)
{   CVO_ENTER
    myroot = (Cvo_RootedList **)root;
    win = me;
    start_row = 0;
    start_col = 0;
    end_row = 0;
    end_col = 0;
    unset = True;
    reverse = True;
    CVO_VOID_RETURN
}

Cvo_TextAttribute *
Cvo_CRT_Highlight::_Highlight(Cvo_TextAttribute *old)
{   CVO_ENTER
    Cvo_Font fn = old->Font();
    Cvo_Color fg = foreground;
    Cvo_Color bg = background;
    BOOL ul = underline ? True : old->IsUnderline();
    BOOL en = enbolden ? True : old->IsEnbolden();

    if (bg.Empty())
	bg = old->Background();
    if (fg.Empty())
	fg = old->Foreground();

    if (fg.Empty() || bg.Empty() ||
        (bg == old->Background() && fg == old->Foreground() &&
	 en == old->IsEnbolden() && ul == old->IsUnderline()))
	    CVO_RETURN(old)

    if (fg == old->Background() && bg == old->Foreground() &&
	en == old->IsEnbolden() && ul == old->IsUnderline())
	    CVO_RETURN(old->Reverse())

    Cvo_TextAttribute *hl = win->highlightattr;

    while (hl) {
	if (hl->Background() == bg && hl->Foreground() == fg
	    && hl->Font() == fn 
	    && hl->IsUnderline() == ul && hl->IsEnbolden() == en)
	    CVO_RETURN(hl)
	if (hl->Background() == fg && hl->Foreground() == bg
	    && hl->Font() == fn 
	    && hl->IsUnderline() == ul && hl->IsEnbolden() == en)
	    CVO_RETURN(hl->Reverse())
	hl = hl->Next();
    }

    hl = new Cvo_TextAttribute(&(win->highlightattr), win, fn);
    hl->ModifyGC(bg, fg);
    hl->Enbolden(en);
    hl->Underline(ul);
    CVO_RETURN(hl)
}

void
Cvo_CRT_Highlight::ResetRange(int sr, int sc, int er, int ec)
{   CVO_ENTER
    for (int j = start_row; j <= end_row; ++j) {
	win->DirtyLine(j);
    }

    if (ToIndex(sr, sc) > ToIndex(er, ec)) {
	int t = sr;
	sr = er;
	er = t;

	t = sc;
	sc = ec;
	ec = t;
    }

    start_row = sr;
    start_col = sc;
    end_row = er;
    end_col = ec;
    unset = False;

    for (j = start_row; j <= end_row; ++j) {
	win->DirtyLine(j);
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::Redraw()
{   CVO_ENTER
    for (int j = start_row; j <= end_row; ++j) {
	win->DirtyLine(j);
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::ExtendRange(int nr, int nc)
{   CVO_ENTER
    for (int j = start_row; j <= end_row; ++j) {
	win->DirtyLine(j);
    }

    if (unset) {
	start_row = nr;
	start_col = nc;
	end_row = nr;
	end_col = nc;
	unset = False;
    } else if (ToIndex(start_row, start_col) >= ToIndex(nr, nc)) {
	start_row = nr;
	start_col = nc;
    } else {
	end_row = nr;
	end_col = nc;
    }

    for (j = start_row; j <= end_row; ++j) {
	win->DirtyLine(j);
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::GetText()
{   CVO_ENTER
    int n;
    wchar_t *w;
    static wchar_t newline[2] = { '\n', 0 };

    win->WaitForCopy();

    text.Set("", 0);
    if (unset)
	CVO_VOID_RETURN
    if (start_row >= 0) {
	if (start_row == end_row) {
	    if (start_row < win->rows) {
		n = win->lines[start_row]->GetText(&w, start_col, end_col);
		text.Append(w, n);
	    }
    	    if (end_col > start_col + n)
		text.Append(newline, 1);
	} else {
	    n = win->lines[start_row]->GetText(&w, start_col);
	    text.Append(w, n);
	    text.Append(newline, 1);
	    for (int j = start_row + 1; j < end_row && j < win->rows; ++j) {
		n = win->lines[j]->GetText(&w);
		text.Append(w, n);
		text.Append(newline, 1);
	    }
	    if (end_row < win->rows) {
		n = win->lines[end_row]->GetText(&w, 0, end_col);
		text.Append(w, n);
		if (end_col > n)
		    text.Append(newline, 1);
	    }
	}
    }
    CVO_VOID_RETURN
}

int
Cvo_CRT_Highlight::GetText(char **cp)
{   CVO_ENTER
    GetText();
    *cp = text.mbValue();
    CVO_RETURN(text.mbLength())
}

int
Cvo_CRT_Highlight::GetText(wchar_t **wp)
{   CVO_ENTER
    GetText();
    *wp = text.wcValue();
    CVO_RETURN(text.wcLength())
}

void
Cvo_CRT_Highlight::DeleteAt(int line, int col, int cnt)
{   CVO_ENTER
    if (cnt <= 0 || unset)
	CVO_VOID_RETURN

    if (line == start_row && start_col > col && (start_col -= cnt) < col)
	start_col = col;

    if (line == end_row && end_col >= col) {
	if (end_col - cnt < col)
	    end_col = col;
	else if (end_col != Cvo_HL_MAX_COLUMN)
	    end_col -= cnt;
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::InsertAt(int line, int col, int cnt)
{   CVO_ENTER
    if (cnt <= 0 || unset)
	CVO_VOID_RETURN

    if (line == start_row && start_col > col) {
	if (start_col >= Cvo_HL_MAX_COLUMN - cnt)
	    start_col = Cvo_HL_MAX_COLUMN;
	else
	    start_col += cnt;

	if (start_col < col)
	    start_col = col;
    }
    if (line == end_row && end_row >= col) {
	if (end_col >= Cvo_HL_MAX_COLUMN - cnt)
	    end_col = Cvo_HL_MAX_COLUMN;
	else
	    end_col += cnt;

	if (end_col < col)
	    end_col = col;
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::InsertLines(int line, int cnt)
{   CVO_ENTER
    if (cnt <= 0 || unset || line + cnt > end_row)
	CVO_VOID_RETURN

    if (start_row >= line)
	start_row += cnt;
    if (end_row >= line)
	end_row += cnt;
    CVO_VOID_RETURN
}

void
Cvo_CRT_Highlight::DeleteLines(int line, int cnt)
{   CVO_ENTER
    if (cnt <= 0 || unset || line > end_row)
	CVO_VOID_RETURN

    if (line <= start_row && line + cnt > end_row) {
	// Note that this also takes care of selections on a single line
	unset = True;
    } else if (line + cnt <= start_row) {
	// If the deletion is before the selection
	start_row -= cnt;
	end_row -= cnt;
    } else if (line <= start_row) {
	// Only true if the deletion does not delete the end of the selection
	start_row = line;
	start_col = 0;
	end_row -= cnt;
    } else if (line + cnt > end_row) {
	// Only true if the deletion does not delete the start of the selection
	end_row = line - 1;
	end_col = Cvo_HL_MAX_COLUMN;
    } else {
	end_row -= cnt;
    }
    CVO_VOID_RETURN
}

Cvo_CRT_LogType *
Cvo_CRT::NeedLog(int n)
{   CVO_ENTER
    if (!InCopy())
    	CVO_RETURN(0)

#if defined(real_slow_scroll)
    if (!jumpscroll) {
	WaitForCopy();
    	CVO_RETURN(0)
    }
#endif

    Cvo_CRT_LogType *log = ProduceLog(n);

    if (!log)
	CVO_RETURN(0)

    CVO_RETURN(log)
}

Cvo_CRT_LogType *
Cvo_CRT::NeedLog(int n, char *str, int len)
{   CVO_ENTER
    if (!InCopy())
	CVO_RETURN(0)

#if defined(real_slow_scroll)
    if (!jumpscroll) {
        WaitForCopy();
        CVO_RETURN(0)
    }
#endif


    tbuf.Set(str, len);

    len = tbuf.wcLength();
    int nlen = ((len * sizeof(wchar_t)) + sizeof(Cvo_CRT_LogType) - 1)
		/ sizeof(Cvo_CRT_LogType);

    Cvo_CRT_LogType *log = ProduceLog(n + 1 + nlen);

    if (!log) {
	WaitForCopy();
	CVO_RETURN(0)
    }

    wchar_t *wstr = tbuf.wcValue();
    wchar_t *nstr = (wchar_t *)&log[n+1];

    log[n] = tbuf.wcLength();
    while (len-- > 0)
	*nstr++ = *wstr++;
    CVO_RETURN(log)
}

Cvo_CRT_LogType *
Cvo_CRT::NeedLog(int n, wchar_t *str, int len)
{   CVO_ENTER
    if (!InCopy())
	CVO_RETURN(0)

#if defined(real_slow_scroll)
    if (!jumpscroll) {
        WaitForCopy();
        CVO_RETURN(0)
    }
#endif


    if (len < 0) {
	len = 0;
	wchar_t *wstr;
	if (wstr = str) {
	    while (*wstr++)
		++len;
	}
    }

    int nlen = ((len * sizeof(wchar_t)) + sizeof(Cvo_CRT_LogType) - 1)
		/ sizeof(Cvo_CRT_LogType);

    Cvo_CRT_LogType *log = ProduceLog(n + 1 + nlen);

    if (!log) {
	WaitForCopy();
	CVO_RETURN(0)
    }

    log[n] = len;
    memcpy(&log[n+1], str, len * sizeof(wchar_t));
    CVO_RETURN(log)
}

Cvo_CRT_LogType *
Cvo_CRT::NeedLog(int n, char *str, Cvo_TextAttribute **attrs, int len)
{   CVO_ENTER
    if (!InCopy())
	CVO_RETURN(0)

#if defined(real_slow_scroll)
    if (!jumpscroll) {
        WaitForCopy();
        CVO_RETURN(0)
    }
#endif


    tbuf.Set(str, len);

    len = tbuf.wcLength();
    int nlen = ((len * sizeof(wchar_t)) + sizeof(Cvo_CRT_LogType) - 1)
		/ sizeof(Cvo_CRT_LogType);
    int alen = ((len * sizeof(Cvo_TextAttribute *)) + sizeof(Cvo_CRT_LogType)-1)
		/ sizeof(Cvo_CRT_LogType);

    Cvo_CRT_LogType *log = ProduceLog(n + 1 + nlen + alen);

    if (!log) {
	WaitForCopy();
	CVO_RETURN(0)
    }

    log[n] = len;

    wchar_t *wstr = tbuf.wcValue();
    wchar_t *nstr = (wchar_t *)&log[n+1];
    Cvo_TextAttribute **nattrs = (Cvo_TextAttribute **)&log[n+1+nlen];


    while (len-- > 0) {
	*nstr++ = *wstr++;
	*nattrs++ = *attrs++;
    }
    CVO_RETURN(log)
}

Cvo_CRT_LogType *
Cvo_CRT::NeedLog(int n, wchar_t *str, Cvo_TextAttribute **attrs, int len)
{   CVO_ENTER
    if (!InCopy())
	CVO_RETURN(0)

#if defined(real_slow_scroll)
    if (!jumpscroll) {
        WaitForCopy();
        CVO_RETURN(0)
    }
#endif


    if (len < 0) {
	len = 0;
	wchar_t *wstr;
	if (wstr = str) {
	    while (*wstr++)
		++len;
	}
    }

    int nlen = ((len * sizeof(wchar_t)) + sizeof(Cvo_CRT_LogType) - 1)
		/ sizeof(Cvo_CRT_LogType);
    int alen = ((len * sizeof(Cvo_TextAttribute *)) + sizeof(Cvo_CRT_LogType)-1)
		/ sizeof(Cvo_CRT_LogType);
    Cvo_CRT_LogType *log = ProduceLog(n + 1 + nlen + alen);

    if (!log) {
	WaitForCopy();
	CVO_RETURN(0)
    }

    log[n] = len;
    memcpy(&log[n+1], str, len * sizeof(wchar_t));
    Cvo_TextAttribute **nattrs = (Cvo_TextAttribute **)&log[n+1+nlen];


    while (len-- > 0)
	*nattrs++ = *attrs++;
    CVO_RETURN(log)
}

Cvo_CRT_LogType *
Cvo_CRT::ProduceLog(int n)
{   CVO_ENTER
    if (logTail == logHead)
	ClearLog();

    if (logTail <= logHead) {
	if (LogSize() - logHead > n) {
	    logHead += n;
	    CVO_RETURN(&datalog[logHead - n])
	}
	if (logTail > n) {
	    datalog[logHead] = Cvo_CRT_CycleLog;
	    logHead = n;
	    CVO_RETURN(datalog)
	}
    } else {
	if (logTail - logHead > n) {
	    logHead += n;
	    CVO_RETURN(&datalog[logHead - n])
	}
    }
    CVO_RETURN(0)
}

void
Cvo_CRT::CopyDoneEvent(XEvent *, void *)
{   CVO_ENTER
    int pad;
    int line;
    int column;
    int count;
    wchar_t *string;
    Cvo_TextAttribute **attrs;
    Cvo_TextAttribute *cs = currentstate;
    int nlen, alen;

    dontdraw = True;
    scrolled = False;

    while (logTail != logHead && !InCopy()) {
	switch (int(datalog[logTail])) {
	case Cvo_CRT_CycleLog:           //
	    logTail = 0;
	    break;
	case Cvo_CRT_SetExtraPad:        // pad
	    pad = datalog[logTail+1];
	    logTail += 2;
	    SetExtraPad(pad);
	    break;
	case Cvo_CRT_ShowCursor:         //
	    logTail += 1;
	    ShowCursor();
	    break;
	case Cvo_CRT_FocusInEvent:       //
	    logTail += 1;
	    FocusInEvent(0, 0);
	    break;
	case Cvo_CRT_FocusOutEvent:      //
	    logTail += 1;
	    FocusOutEvent(0, 0);
	    break;
	case Cvo_CRT_HideCursor:         //
	    logTail += 1;
	    HideCursor();
	    break;
	case Cvo_CRT_MoveCursor:         // line, column
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    logTail += 3;
	    MoveCursor(line, column);
	    break;
	case Cvo_CRT_InsertLine:         // line, count
	    line = datalog[logTail+1];
	    count = datalog[logTail+2];
	    logTail += 3;
	    InsertLine(line, count);
	    break;
	case Cvo_CRT_DeleteLine:         // line, count
	    line = datalog[logTail+1];
	    count = datalog[logTail+2];
	    logTail += 3;
	    DeleteLine(line, count);
	    break;
	case Cvo_CRT_InsertBlanksAt:     // line, column, count
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    logTail += 4;
	    InsertAt(line, column, count);
	    break;
	case Cvo_CRT_DeleteAt:           // line, column, count
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    logTail += 4;
	    DeleteAt(line, column, count);
	    break;
	case Cvo_CRT_ChangeOrigin:       // origin
	    pad = datalog[logTail+1];
	    logTail += 2;
	    ChangeOrigin(pad);
	    break;
	case Cvo_CRT_InsertAt:           // line, column, count, string
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    string = (wchar_t *)&datalog[logTail+4];
	    nlen = ComputeWchars(count);
	    logTail += 4 + nlen;
	    InsertAt(line, column, string, count);
	    break;
	case Cvo_CRT_ReplaceAt:          // line, column, count, string
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    string = (wchar_t *)&datalog[logTail+4];
	    nlen = ComputeWchars(count);
	    logTail += 4 + nlen;
	    ReplaceAt(line, column, string, count);
	    break;
	case Cvo_CRT_AppendTo:           // line, count, string
	    line = datalog[logTail+1];
	    count = datalog[logTail+2];
	    string = (wchar_t *)&datalog[logTail+3];
	    nlen = ComputeWchars(count);
	    logTail += 3 + nlen;
	    AppendTo(line, string, count);
	    break;
#if	defined(future)
	case Cvo_CRT_InsertAtAttr:       // line, column, count, string, attr
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    string = (wchar_t *)&datalog[logTail+4];
	    nlen = ComputeWchars(count);
	    alen = ComputeAttrs(count);
	    attrs = (Cvo_TextAttribute **)&datalog[logTail+4+nlen];
	    logTail += 4 + nlen + alen;
	    InsertAt(line, column, string, attr, count);
	    break;
	case Cvo_CRT_ReplaceAtAttr:      // line, column, count, string, attr
	    line = datalog[logTail+1];
	    column = datalog[logTail+2];
	    count = datalog[logTail+3];
	    string = (wchar_t *)&datalog[logTail+4];
	    nlen = ComputeWchars(count);
	    alen = ComputeAttrs(count);
	    attrs = (Cvo_TextAttribute **)&datalog[logTail+4+nlen];
	    logTail += 4 + nlen + alen;
	    ReplaceAt(line, column, string, attr, count);
	    break;
#endif
	case Cvo_CRT_AppendToAttr:       // line, count, string, attr
	    line = datalog[logTail+1];
	    count = datalog[logTail+2];
	    string = (wchar_t *)&datalog[logTail+3];
	    nlen = ComputeWchars(count);
	    alen = ComputeAttrs(count);
	    attrs = (Cvo_TextAttribute **)&datalog[logTail+3+nlen];
	    logTail += 3 + nlen + alen;
	    AppendTo(line, string, attrs, count);
	    break;
    	case Cvo_CRT_SetState:
	    currentstate = *(Cvo_TextAttribute **)&datalog[logTail+1];
	    logTail += 1 + (STA + SCL - 1)/SCL;
	    break;
	default:
	    ClearLog();
	    break;
	}
    }
    dontdraw = False;
    if (scrolled) {
	//
	// If we scrolled then we clear the whole screen and mark everything
	// as dirty.  This is basically jump scroll.
	//
	ClearArea(0, 0, width, height);
	for (int j = 0; j < rows; ++j)
	    lines[j]->MakeDirty();
    }
    Flush();
    currentstate = cs;
    CVO_VOID_RETURN
}

void
Cvo_CRT::FindCoordinates(XEvent *ev, int *x, int *y)
{   CVO_ENTER
    if (!selection) {
	selection = NewHighlight();
    	if (Color()) {
	    char *fg = GetResource(_QselectionForeground,_QSelectionForeground);
	    char *bg = GetResource(_QselectionBackground,_QSelectionBackground);
	
	    if (fg)
		selection->SetForeground(Cvo_Color(this, fg));
	    if (bg)
		selection->SetBackground(Cvo_Color(this, bg));
	    else
		selection->SetBackground(CurrentBackground()->Raised());
	}
    }

    if (ev->type == ButtonPress || ev->type == ButtonRelease) {
	*x = ev->xbutton.x;
	*y = ev->xbutton.y;

	PixelToCharacter(y, x);
    } else if (ev->type == MotionNotify) {
	*x = ev->xmotion.x;
	*y = ev->xmotion.y;

	PixelToCharacter(y, x);
    } else {
	*x = cursorx;
	*y = cursory;
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT::SelectStart(XEvent *ev, int, char **av)
{   CVO_ENTER
    int x, y;

    BOOL cont;

    if (av && !strcasecmp(av[0], "Word")) {
	cont = charmode;
	charmode = linemode = False;
	wordmode = True;
    } else if (av && !strcasecmp(av[0], "Line")) {
	cont = wordmode;
	charmode = wordmode = False;
	linemode = True;
    } else {
    	cont = linemode;
	linemode = wordmode = False;
	charmode = True;
    }

    if (anchor_row < 0)
	cont = False;

    FindCoordinates(ev, &x, &y);

    if (!cont) {
	anchor_row = y;
	anchor_col = x;
    }

    int ay = anchor_row;
    int ax = anchor_col;

    AdjustSelection(ay, ax, y, x);

    selection->ResetRange(ay, ax, y, x);
    Flush(2);
    CVO_VOID_RETURN
}

void
Cvo_CRT::AdjustSelection(int &ay, int &ax, int &y, int &x)
{
    if (linemode) {
	if (ay > y) {
	    x = 0;
	    ax = Cvo_HL_MAX_COLUMN;
    	} else {
	    ax = 0;
	    x = Cvo_HL_MAX_COLUMN;
    	}
    } else if (wordmode) {
	if (ToIndex(y,x) > ToIndex(ay, ax)) {
	    if (ay >= 0 && ay < rows)
		lines[ay]->BeginWord(ax);
	    if (y >= 0 && y < rows)
		lines[y]->EndWord(x);
    	} else {
	    if (ay >= 0 && ay < rows)
		lines[ay]->EndWord(ax);
	    if (y >= 0 && y < rows)
		lines[y]->BeginWord(x);
    	}
    }
}

void
Cvo_CRT::SelectExtend(XEvent *ev, int, char **)
{   CVO_ENTER
    int x, y;

    FindCoordinates(ev, &x, &y);

    int ay = anchor_row;
    int ax = anchor_col;

    AdjustSelection(ay, ax, y, x);

    selection->ResetRange(ay, ax, y, x);
    Flush(2);
    CVO_VOID_RETURN
}

void
Cvo_CRT::StartExtend(XEvent *ev, int, char **)
{   CVO_ENTER
    int x, y;

    FindCoordinates(ev, &x, &y);
    if (anchor_row < 0) {
	anchor_row = y;
	anchor_col = x;
    }
    int ay = anchor_row;
    int ax = anchor_col;

    AdjustSelection(ay, ax, y, x);

    selection->ExtendRange(y, x);
    Flush(2);
    CVO_VOID_RETURN
}

void
Cvo_CRT::SelectEnd(XEvent *, int, char **)
{   CVO_ENTER

    charmode = False;

    if (selection) {
    	if (selection->Empty()) {
	    MoveCursor(anchor_row, anchor_col);
	    Flush(2);
	} else {
	    selection->GetText();
	    SetSelection(selection->GetTextBuffer());
    	}
    }
    CVO_VOID_RETURN
}

void
Cvo_CRT::SelectClear(XEvent *e, void *d)
{   CVO_ENTER
    if (selection) {
	Cvo_Object::SelectClear(e, d);
	selection->TurnOff();
	Flush(2);
    }
    CVO_VOID_RETURN
}
