//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%	%I%	%G% %U%"); }
static void RSCID() { void("$Id: ChartVertical.cc,v 1.6 1994/09/21 18:18:28 prb Exp $"); }

#include <Cvo/Chart.h++>
#include <Cvo/Frame.h++>
#include <Cvo/TextBoard.h++>
#include <Cvo/Paint.h++>
#include <math.h>

CONSTRUCTORS_1ARG_PASS(Cvo_VerticalGraph,Cvo_GraphObject,"CvoVerticalGraph",Cvo_Chart*)
CONSTRUCTORS_1ARG_PASS(Cvo_VerticalSideGraph,Cvo_VerticalGraph,"CvoVerticalSideGraph",Cvo_Chart*)
CVO_CREATE_REGISTER_FUNCTIONS(Cvo_VerticalGraph)

void
Cvo_VerticalSideGraph::_Init(Cvo_Chart*)
{
    return;
}

void
Cvo_VerticalGraph::_Init(Cvo_Chart*)
{
    colwidth = 30;
    fudge = 0;
    coldepth = GetResourceInt("depth","Depth",15);
    colseparation = GetResourceInt("separation","Separation",7);
    mincolwidth = 1;
    mincolheight = 0.0;

    VerticalChildren();

    frame1 = new Cvo_Frame("frame1", this);
    frame1->ExpandFrame();
    frame1->HorizontalChildren();

    frame2 = new Cvo_Frame("frame2", this);
    frame2->HorizontalChildren();

    if (pixmapbacking==True)
        graph = new Cvo_Paint("grafik", frame1);
    else
        graph = new Cvo_Window("grafik", frame1);

    graph->Flatten();
    graph->ExpandFrame();
    graph->LayOpposite();

	graph->Register(Expose, _Cvo_GraphObject_Expose_Handler, this);
    if (pixmapbacking==True)
        graph->Register(CvoPaintChangedSizeEvent, _Cvo_GraphObject_Expose_Handler, this);

    ticks = new Cvo_TextBoard("ticks", frame1);
    ticks->LayOpposite();
    ticks->SetMinSize(1, 0);

    cticks = new Cvo_TextBoard("ticks", frame2);
    cticks->SetMinSize(0, 1);
    cticks->MatchWidth(graph);
    cticks->MatchX(graph);

    marklist = NULL;
    cmarklist = NULL;
}

Cvo_GraphObject*
Cvo_VerticalGraph::CloneMe(Cvo_Object *o,Cvo_Chart *c)
{
    Cvo_VerticalGraph *ret = new Cvo_VerticalGraph("grafik",o,c);
    ret->colwidth = colwidth;
    ret->fudge = fudge;
    ret->coldepth = coldepth;
    ret->colseparation = colseparation;
    ret->mincolwidth = mincolwidth;
    ret->mincolheight = mincolheight;
    ret->dirty = dirty;

    return(ret);
}


Cvo_GraphObject*
Cvo_VerticalSideGraph::CloneMe(Cvo_Object *o, Cvo_Chart *c)
{
    Cvo_VerticalSideGraph *ret = new Cvo_VerticalSideGraph("grafik",o,c);
    ret->colwidth = colwidth;
    ret->fudge = fudge;
    ret->coldepth = coldepth;
    ret->colseparation = colseparation;
    ret->mincolwidth = mincolwidth;
    ret->mincolheight = mincolheight;
    ret->dirty = dirty;

    return(ret);
}

void
Cvo_VerticalGraph::SetText(wchar_t *s)
{
    int len=0;
    wchar_t *p = s;

    while(*p++)
	    len++;

    wchar_t *buf = new wchar_t[len*2];
    for(int i = 0;i < len; i++) {
	    buf[i*2] = s[i];
	    buf[i*2+1] = '\n';
    }
    buf[len*2-1] = '\0';

    if (marklabel) {
	marklabel->SetText(buf);
	RootObject()->ToLayoutWindow()->ReLayout(1);
    } else {
	marklabel = new Cvo_Label("marklabel",frame1,buf);
	marklabel->PlaceBefore(ticks);
	marklabel->MatchY(graph);
	marklabel->MatchHeight(graph);
	if (Mapped())
	    marklabel->Map();
    }

    delete[] buf;
}

void
Cvo_VerticalGraph::SetColumnText(wchar_t *s)
{
    if (cmarklabel) {
	cmarklabel->SetText(s);
	RootObject()->ToLayoutWindow()->ReLayout(1);
    } else {
	cmarklabel = new Cvo_Label("cmarklabel", this, s);
	cmarklabel->PlaceAfter(frame2);
	cmarklabel->MatchX(graph);
	cmarklabel->MatchWidth(graph);
	if (Mapped())
	    cmarklabel->Map();
    }
}

Cvo_GraphMark*
Cvo_VerticalGraph::AddMark(Cvo_ChartMark *m)
{
    Cvo_GraphMark *ret = new Cvo_GraphMark(&marklist, m);
                                           
    ret->item = ticks->AddString(m->string);
    dirty = True;
    return(ret);
}

Cvo_GraphMark*
Cvo_VerticalGraph::AddColumnMark(Cvo_ChartMark *m)
{
    Cvo_GraphMark *ret = new Cvo_GraphMark(&cmarklist, m);
                                           
    ret->item = cticks->AddString(m->string);
    dirty = True;
    return(ret);
}

void
Cvo_VerticalGraph::ClearMarks()
{
    while(marklist)
        RemoveMark(marklist);
    ticks->Redraw();
}

void
Cvo_VerticalGraph::ClearColumnMarks()
{
    while(cmarklist)
        RemoveColumnMark(cmarklist);
    cticks->Redraw();
}

int
Cvo_VerticalGraph::ComputeX(int i)
{
    return((colwidth + colseparation) * i + colseparation + 1
        + ((i<fudge)?i:fudge) );
}

void
Cvo_VerticalGraph::DrawMarks()
{
    if (!scale)
        return;

    int max = graph->Height() - coldepth - 2;

    graph->SetForeground(Linecolor());

    for (Cvo_GraphMark *sm = marklist; sm; sm = sm->Next())
    {
        int x = ticks->Width() - ticks->StringWidth(sm->mark->string);
        int v = (int)floor(sm->mark->value * scale + 0.5);
        int y = graph->Height() - v;
        int h = ticks->StringHeight(sm->mark->string);

        if (v < 0 || v > max)
        {
            x = -1; y = -1;
        }
        else if (v == 0)
        {
            y = graph->Height() - h;
        }
        else if (v < (h / 2))
        {
            // Diagonal
            if (coldepth>1)
                graph->DrawLine(1, y-2, coldepth, y - coldepth - 1);
            else if (coldepth==1)
                graph->DrawPoint(1, y-1);

            // Horizontal
            graph->DrawLine(coldepth + 1, y - coldepth - 1,
                         graph->Width()-1, y - coldepth - 1);

            y = graph->Height() - h + 1;
        }
        else if (v!=max)
        {
            // Diagonal
            if (coldepth>1)
                graph->DrawLine(1, y-2, coldepth, y - coldepth - 1);
            else if (coldepth==1)
                graph->DrawPoint(1, y-1);

            // Horizontal
            graph->DrawLine(coldepth+1, y - coldepth - 1,
                         graph->Width()-1, y - coldepth - 1);

            y -= h/2;
        }
        else
        {
            y -= h/2;
        }

        if ((y<0) && (x!=-1))
            y = 0;

        sm->item->SetLocation(x, y);
    }

    for (sm = cmarklist; sm; sm = sm->Next())
    {
        int x = ComputeX(int(sm->mark->value));
        x += (colwidth - ticks->StringWidth(sm->mark->string))/2;
        sm->item->SetLocation(x, 0);
    }

    ticks->Redraw();
    cticks->Redraw();
}


void
Cvo_VerticalGraph::EvaluateMarks()
{
    if (!ticks)
        return;

    int max = 1;

    for (Cvo_GraphMark *sm = marklist; sm; sm = sm->Next())
    {
        int x = ticks->StringWidth(sm->mark->string);

        if (x > max)
            max = x;
    }

    ticks->SetMinSize(0, 0);
    ticks->SetMinPixelSize(max, 0);
	ticks->Redraw();
}

void
Cvo_VerticalGraph::EvaluateColumnMarks()
{
    if (!cticks)
        return;

    int max = 1;

    for (Cvo_GraphMark *sm = cmarklist; sm; sm = sm->Next())
    {
        int x = cticks->StringWidth(sm->mark->string);

        if (x > max)
            max = x;
    }
    SetMinColumnWidth(max);
}


CARD16
Cvo_VerticalGraph::CalculateHeight()
{
    int width = coldepth + colseparation + Ncolumns() * (mincolwidth+colseparation);
    int height = 20 + coldepth;

    int theight = (!marklabel)?0:marklabel->CalculateHeight();
    height = (height<theight)?theight:height;
    int twidth = (!cmarklabel)?0:cmarklabel->CalculateWidth();
    width = (width<twidth)?twidth:width;

    graph->SetMinSize(0,0);
    graph->SetMinPixelSize(width,height);

    return(0);
}


CARD16
Cvo_VerticalGraph::CalculateWidth()
{
    return(0);
}


void
Cvo_VerticalGraph::FillRhombus(Cvo_Window *obj, int x, int y, int w, int h, int xoff, int yoff)
{
    XPoint poly[5];

    if (!pixmapbacking)
        obj->ToXCoord(&x, &y);

    poly[0].x = x;
    poly[0].y = y;

    poly[1].x = x + w;
    poly[1].y = y + yoff;

    poly[2].x = x + w + xoff;
    poly[2].y = y + yoff + h;

    poly[3].x = x + xoff;
    poly[3].y = y + h;
    
    poly[4] = poly[0];

    if (pixmapbacking)
    {
        XFillPolygon(obj->Dpy(), ((Cvo_Paint*)obj)->GetPixmap()->Object(), obj->Gc(),
             poly, 5, Convex, CoordModeOrigin);
        ((Cvo_Paint*)obj)->MakeDirty();
    }
    else
        XFillPolygon(obj->Dpy(), obj->Object(), obj->Gc(),
			poly, 5, Convex, CoordModeOrigin);
}

void
Cvo_VerticalGraph::DrawIt(XEvent *, void *)
{
    if (!Object())
        return;

    if (graph->Mapped())
        graph->Erase();

    int ncolors = Ncolors();
    int ncolumns = Ncolumns();
    double** columns = Columns();

    double fh = 0.0;

    for (int i = 0; i < ncolumns; i++) {
        double th = 0.0;
        for (int j = 0; j < ncolors; j++)
            th += columns[i][j];

        if (th > fh)
            fh = th;
    }

    if (fh < mincolheight)
        fh = mincolheight;

    int x = 0;
    int y = graph->Height() - 1;
    int h = y - coldepth;
    int w = graph->Width() - coldepth - 2;

    if (fh == 0.0)
        scale = (h-1);
    else
        scale = (h-1) / fh;

    if (ncolumns == 0)
        return;

    colwidth = (w - colseparation) / ncolumns;

    fudge = (w - colseparation) - colwidth * ncolumns;

    colwidth -= colseparation;

    graph->SetForeground(Backcolor());
    graph->FillRectangle(coldepth+1, 0, w, h);
    if (coldepth) {
        FillRhombus(graph, 0, coldepth, coldepth+1, h, 0, -coldepth);
        FillRhombus(graph, coldepth, h, w+1, coldepth, -coldepth, 0);
    }

    /*            
             _____________________
            /|        7          |
          6/ |                   |
          /  |                   |
          |  |4                  |8
          |  |                   |
         1|  |___________________|
          |  /        5         /
          | /2                 /9
          |/__________________/
                    3
    */

    graph->SetForeground(Linecolor());

    if (coldepth) {
        graph->DrawLine(0, coldepth+1, 0, y);                       // Line 1
        graph->DrawLine(0, y, coldepth+1, h-1);                     // Line 2
        graph->DrawLine(0, y, w, y);                                // Line 3
        graph->DrawLine(coldepth+1, 0, coldepth+1, h-1);            // Line 4
        graph->DrawLine(coldepth+1, h-1, w+coldepth+1, h-1);        // Line 5
        graph->DrawLine(0, coldepth, coldepth, 0);                  // Line 6
        graph->DrawLine(coldepth+1, 0, w+coldepth+1, 0);            // Line 7
        graph->DrawLine(w+coldepth+1, 0, w+coldepth+1, h-1);        // Line 8
        graph->DrawLine(w+1, y, w+coldepth+1, h);                   // Line 9
    } else {
        graph->DrawLine(0, 0, 0, h-1);                              // Line 4
        graph->DrawLine(0, h, w+1, h);                              // Line 5
        graph->DrawLine(0, 0, w+1, 0);                              // Line 7
        graph->DrawLine(w+1, 0, w+1, h-1);                          // Line 8
    }

    DrawMarks();

    for (i = 0; i < ncolumns; ++i) {
        double y = 0.0;
        int yy = 1;
        for (int j = 0; j < ncolors; ++j) {
            double h = columns[i][j];
            if (h) {
                int v = (int)floor((y+h)*scale);
                int he = (int)floor(y*scale);

                DrawBox(ComputeX(i), yy, (v==he)?1:v-he, Colors()[j]);
                y += h;
                yy += v-he;
            }
        }
    }
    dirty = False;
}


void
Cvo_VerticalSideGraph::DrawIt(XEvent *, void *)
{
    if (!Object())
        return;

    if (graph->Mapped())
        graph->Erase();

    int ncolors=Ncolors();
    int ncolumns=Ncolumns();
    double** columns=Columns();

    double fh = 0.0;

    for (int i = 0; i < ncolumns; i++) {
        for (int j = 0; j < ncolors; j++) {
            double th = columns[i][j];
            if (th>fh)
                fh = th;
        }
    }

    if (fh < mincolheight)
        fh = mincolheight;

    int x = 0;
    int y = graph->Height() - 1;
    int h = y - coldepth;
    int w = graph->Width() - coldepth - 2;

    if (fh == 0.0)
        scale = (h-1);
    else
        scale = (h-1) / fh;

    if (ncolumns == 0)
        return;

    colwidth = (w - colseparation) / ncolumns;

    fudge = (w - colseparation) - colwidth * ncolumns;

    colwidth -= colseparation;

    graph->SetForeground(Backcolor());
    graph->FillRectangle(coldepth+1, 0, w, h);
    if (coldepth) {
        FillRhombus(graph, 0, coldepth, coldepth+1, h, 0, -coldepth);
        FillRhombus(graph, coldepth, h, w+1, coldepth, -coldepth, 0);
    }

    /*            
             _____________________
            /|        7          |
          6/ |                   |
          /  |                   |
          |  |4                  |8
          |  |                   |
         1|  |___________________|
          |  /        5         /
          | /2                 /9
          |/__________________/
                    3
    */

    graph->SetForeground(Linecolor());

    if (coldepth) {
        graph->DrawLine(0, coldepth+1, 0, y);                       // Line 1
        graph->DrawLine(0, y, coldepth+1, h-1);                     // Line 2
        graph->DrawLine(0, y, w, y);                                // Line 3
        graph->DrawLine(coldepth+1, 0, coldepth+1, h-1);            // Line 4
        graph->DrawLine(coldepth+1, h-1, w+coldepth+1, h-1);        // Line 5
        graph->DrawLine(0, coldepth, coldepth, 0);                  // Line 6
        graph->DrawLine(coldepth+1, 0, w+coldepth+1, 0);            // Line 7
        graph->DrawLine(w+coldepth+1, 0, w+coldepth+1, h-1);        // Line 8
        graph->DrawLine(w+1, y, w+coldepth+1, h);                   // Line 9
    } else {
        graph->DrawLine(0, 0, 0, h-1);                              // Line 4
        graph->DrawLine(0, h, w+1, h);                              // Line 5
        graph->DrawLine(0, 0, w+1, 0);                              // Line 7
        graph->DrawLine(w+1, 0, w+1, h-1);                          // Line 8
    }

    DrawMarks();

    int ndraw=0;

    Bool *draw = new Bool[ncolors];
    for(i=0;i<ncolors;i++)
        draw[i]=False;
    for (i=0; i<ncolors; i++) {
        for (int j=0; j<ncolumns; j++) {
            if (!draw[i] && columns[j][i]) {
                draw[i]=True;
                ndraw++;
            }
	}
    }

    for (i = 0; i < ncolumns; i++) {
        int pos = 0;
        for (int j = 0; j < ncolors; ++j) {
            double h = columns[i][j];
            if (h) {
                int v = (int)floor(h*scale);

                DrawBox(ComputeX(i), 1, v, colwidth/ndraw, pos, Colors()[j]);
            }
            if (draw[j])
                pos++;
        }
    }

    delete[] draw;

    dirty = False;
}


void
Cvo_VerticalGraph::DrawBox (int x, int y, int h, const Cvo_Color &c)
{
    graph->SetForeground(c);

    y = graph->Height() - y - h;

    graph->FillRectangle(x, y, colwidth, h);

    if (coldepth) {
        graph->SetForeground(c->Lower());

        if (coldepth>1)
            FillRhombus(graph, x+colwidth, y,
                coldepth, h, 0, -coldepth);
        else
            graph->DrawLine(x+colwidth, y, x+colwidth, y+h-2);


        graph->SetForeground(c->Upper());

        if (coldepth>1)
            FillRhombus(graph, x+coldepth-1, y-coldepth,
                colwidth, coldepth, -coldepth, 0);
        else
            graph->DrawLine(x+1,y-1, x+colwidth-1,y-1);

        graph->SetForeground(c);
        
        if (coldepth>1)
            graph->DrawLine(x+colwidth, y-1,
                 x+colwidth+coldepth-1, y-coldepth);
        else
            graph->DrawPoint(x+colwidth, y-1);
    }
}


void
Cvo_VerticalSideGraph::DrawBox(int x, int y, int h, int width, int pos, const Cvo_Color &c)
{
    graph->SetForeground(c);

    y = graph->Height() - y - h;

    graph->FillRectangle(x + width*pos, y, width, h);

    if (coldepth) {
        graph->SetForeground(c->Lower());

        if (coldepth>1)
            FillRhombus(graph, x+width+width*pos, y,
                coldepth, h, 0, -coldepth);
        else
            graph->DrawLine(x+width+width*pos, y, x+width+width*pos, y+h-2);


        graph->SetForeground(c->Upper());

        if (coldepth>1)
            FillRhombus(graph, x+width*pos+coldepth-1, y-coldepth,
                width, coldepth, -coldepth, 0);
        else
            graph->DrawLine(x+1,y-1, x+width+width*pos-1,y-1);

        graph->SetForeground(c);
        
        if (coldepth>1)
            graph->DrawLine(x+width+width*pos, y-1,
                 x+width+width*pos+coldepth-1, y-coldepth);
        else
            graph->DrawPoint(x+width+width*pos, y-1);
    }
}
    
void Cvo_VerticalGraph::HandleEvent(XEvent* ev, void*)
{
    static unsigned cnt=0;

    if (ev->type==ButtonPress) {
        cnt++;
        return;
    }

    if (ev->type==ButtonRelease) {
        if (--cnt==0) {
            int xo,yo;
            graph->XOrigin(&xo,&yo);
            int x = ev->xbutton.x - xo;
            int y = ev->xbutton.y - yo;
            graph->FromXCoord(&x,&y);

            if ((x>=0) && (y>=0) && (x<graph->XWidth()) && (y<graph->XHeight()))
            {
                Cvo_ChartEvent cev;

                cev.button = ev->xbutton.button;
                cev.state = ev->xbutton.state;

                int ncolumns = Ncolumns();
                int ncolors = Ncolors();
                double** columns = Columns();

                for (int i = ncolumns-1; i>=0; i--)
                {
                    for (int j = ncolors-1; j>=0; j--)
                    {
                        double h = columns[i][j];
                        if (h)
                        {
                            double yv=0.0;
                            for(int k=0;k<j;k++)
                                yv+=columns[i][k];
                                
                            int v = (int)floor((yv+h)*scale);
                            int he = (int)floor(yv*scale);

                            int left = ComputeX(i);
                            int right = left + colwidth - 1;
                            int top = graph->Height() - ((v==he)?1:v-he) - he - 1;
                            int bottom = graph->Height() - he - 2;

                            cev.bar = i;
                            cev.color = j;
                            cev.value = h;

                            if ((x>=left) && (x<=right) && (y>=top) && (y<=bottom))
                            {
                                chartob->SendEvent(CvoChartEvent, &cev);
                                return;
                            }

                            if (coldepth)
                            {
                                if ((y>=(top-coldepth)) && (y<top) &&
                                        ((x+y)>(left+top-2)) && ((x+y)<=(right+top)))
                                {
                                    chartob->SendEvent(CvoChartEvent, &cev);
                                    return;
                                }

                                if ((x>=right) && (x<=(right+coldepth)) &&
                                        ((x+y)>(right+top)) && ((x+y)<=(right+bottom+1)))
                                {
                                    chartob->SendEvent(CvoChartEvent, &cev);
                                    return;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

void
Cvo_VerticalSideGraph::HandleEvent(XEvent* ev, void*)
{
    static unsigned cnt=0;

    if (ev->type==ButtonPress)
    {
        cnt++;
        return;
    }

    if (ev->type==ButtonRelease)
    {
        if (--cnt==0)
        {
            int xo,yo;
            graph->XOrigin(&xo,&yo);
            int x = ev->xbutton.x - xo;
            int y = ev->xbutton.y - yo;
            graph->FromXCoord(&x,&y);

            if ((x>=0) && (y>=0) && (x<graph->XWidth()) && (y<graph->XHeight()))
            {
                Cvo_ChartEvent cev;

                cev.button = ev->xbutton.button;
                cev.state = ev->xbutton.state;

                int ncolumns = Ncolumns();
                int ncolors = Ncolors();
                double** columns = Columns();
				int ndraw = 0;
				Bool *draw = new Bool[ncolors];
				for (int i=0;i<ncolors;i++)
					draw[i]=False;
				for(i=0;i<ncolors;i++)
				{
					for(int j=0;j<ncolumns;j++)
					{
						if (!draw[i] && columns[j][i])
						{
							draw[i]=True;
							ndraw++;
						}
					}
				}

                for (i = ncolumns-1; i>=0; i--)
                {
					int pos=ndraw-1;
                    for (int j = ncolors-1; j>=0; j--)
                    {
                        double h = columns[i][j];
                        if (h)
                        {
                            int v = (int)floor(h*scale);

                            int left = ComputeX(i) + (colwidth/ndraw)*pos;
                            int right = left + (colwidth/ndraw) - 1;
                            int top = graph->Height() - v - 1;
                            int bottom = graph->Height() - 1;

                            cev.bar = i;
                            cev.color = j;
                            cev.value = h;

                            if ((x>=left) && (x<=right) && (y>=top) && (y<=bottom))
                            {
                                chartob->SendEvent(CvoChartEvent, &cev);
                                return;
                            }

                            if (coldepth)
                            {
                                if ((y>=(top-coldepth)) && (y<top) &&
                                        ((x+y)>(left+top-2)) && ((x+y)<=(right+top)))
                                {
                                    chartob->SendEvent(CvoChartEvent, &cev);
                                    return;
                                }

                                if ((x>=right) && (x<=(right+coldepth)) &&
                                        ((x+y)>(right+top)) && ((x+y)<=(right+bottom+1)))
                                {
                                    chartob->SendEvent(CvoChartEvent, &cev);
                                    return;
                                }
                            }
                        }
						if (draw[j])
							pos--;
                    }
                }
            }
        }
    }
}
