//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%	%I%	%G% %U%"); }
static void RCSID() { void("$Id: Matrix.cc,v 1.4 1994/09/21 18:18:28 prb Exp $"); }
//
// This code was initially contributed by Paul Algren 11-92
//
#include <string.h>
#include <Cvo/Matrix.h++>

inline int max(int x, int y) { return(x > y ? x : y); }
inline int max(int x, CARD32 y) { return(x > int(y) ? x : int(y)); }
inline int max(int x, unsigned short y) { return(x > int(y) ? x : int(y)); }

static Cvo_Default defs[] = {
    "*CvoMatrix.BorderWidth: 0",
};

//
//  Matrix Cell Implementation
//

// XXX -- Need to remove all the aborts

Cvo_MatrixCell::Cvo_MatrixCell(Cvo_Object *obj, Cvo_MatrixSlice *s1,
						Cvo_MatrixSlice *s2)
{
    if (s1->matrix != s2->matrix) {
	Cvo_Failure(CvoE_FATAL, CvoE_BADPARENT,
          "MatrixCell constructor: Row and Column from different matricies\n");
    }

    if (s1->Horizontal() == True && s2->Horizontal() == False) {
	row = s2;
	col = s1;
    } else if (s2->Horizontal() == True && s1->Horizontal() == False) {
	row = s1;
	col = s2;
    } else {
	Cvo_Failure(CvoE_FATAL, CvoE_BADPARENT,
          "MatrixCell constructor: Both Slices are of the same type\n");
    }

    if (obj->Parent() != row->matrix) {
	Cvo_Failure(CvoE_FATAL, CvoE_BADPARENT,
         "MatrixCell constructor: Managing an object not parented by matrix\n");
    }

    theObject = obj;
    
    //
    // Column insertion
    //
    // First in Column ?
    //
    if (col->cells == NULL) {
	north = this;
	south = NULL;
	col->cells = this;
    } else {
	Cvo_MatrixCell* row_ptr = col->cells;

	while (row_ptr->south != NULL &&
	       row_ptr->row->index < row->index) {
	    row_ptr=row_ptr->south;
	}
	
	if (row_ptr->row->index == row->index) {
	    Cvo_Failure(CvoE_WARN, CvoE_OVERWITE,
		"MatrixCell: Old cell at %d,%d deleted to make room for new\n",
		   row->index, col->index);

	    Cvo_MatrixCell* savptr = 0;
	    if (row_ptr->south) {
		savptr = row_ptr->south;
	    } else if (row_ptr->north->south) {
		savptr = row_ptr->north;
	    } 

	    delete row_ptr;

	    //
	    // if this is the only cell in the column
	    // 
	    if (savptr == 0) {
		north = this;
		south = NULL;
		col->cells = this;
	    } 

	    row_ptr = savptr;
	}
	if (row_ptr) {
	    if (row_ptr->row->index > row->index) { 	// Insert before
		if (row_ptr == col->cells) { 		// before first
		    col->cells = this;
		} else {
		    row_ptr->north->south = this;
		}
		north = row_ptr->north;
		row_ptr->north = this;
		south = row_ptr;
	    } else {		// Insert after
		if (row_ptr->south == NULL) {	// After last
		    col->cells->north = this;
		} else {
		    south->north = this;
		}
		south = NULL;
		row_ptr->south = this;
		north = row_ptr;
	    }
	} 
    }

    //
    // Row insertion
    //
    // First in Row ?
    //
    if (row->cells == NULL) {
	west = this;
	east = NULL;
	row->cells = this;
    } else {
	Cvo_MatrixCell* col_ptr = row->cells;

	while (col_ptr->east != NULL && col_ptr->col->index < col->index) {
	    col_ptr=col_ptr->east;
	}
	
	if (col_ptr->col->index > col->index) { 	// Insert before
	    if (col_ptr == row->cells) { 		// before first
		row->cells = this;
	    } else {
		col_ptr->west->east = this;
	    }
	    west = col_ptr->west;
	    col_ptr->west = this;
	    east = col_ptr;
	} else {		// Insert after
	    if (col_ptr->east == NULL) {	// After last
		row->cells->west = this;
	    } else {
		east->west = this;
	    }
	    east = NULL;
	    col_ptr->east = this;
	    west = col_ptr;
	}
    }
}

Cvo_MatrixCell::~Cvo_MatrixCell()
{
    delete theObject;
    theObject = 0;

    if (!col || !row || !north || !west) {
	return;
    } 

    if (south) {
	south->north = north;
    } else {
	col->cells->north = north;
    }

    if (north && north->south) {
	north->south = south;
    } else {
	col->cells = south;
    }
	    
    if (east) {
	east->west = west;
    } else {
	row->cells->west = west;
    }

    if (west && west->east) {
	west->east = east;
    } else {
	row->cells = east;
    }
}

//
//  Matrix Slice Implementation  -  A slice is either a row or a column
//

Cvo_MatrixSlice::Cvo_MatrixSlice(Cvo_Matrix *M, BOOL orient, int Index)
{
    if (M == NULL) {
	abort();		// Slice must be inserted into a known matrix
    }
    label = NULL;
    cells = NULL;
    matrix = M;
    index = Index;
    horizontal = orient;

    Cvo_MatrixSlice* ptr;
    Cvo_MatrixSlice** head;

    min = des = max = 0;
    origin = -1;

    if (horizontal == True) {
	head = (Cvo_MatrixSlice**)&(matrix->cols);
	ptr = matrix->cols;
    } else {
	head = (Cvo_MatrixSlice**)&(matrix->rows);
	ptr = matrix->rows;
    }

    if (ptr == NULL) {		// First in list
	*head = this;
	prev = this;
	next = NULL;
    } else {

	while (ptr->next != NULL &&
	       ptr->index < index) {
	    ptr=ptr->next;
	}

	if (ptr->index >= index) {	// insert before
	    prev = ptr->prev;
	    if (prev->next)
		prev->next = this;
	    else
		*head = this;
	    next = ptr;
	    ptr->prev = this;

	    if (ptr->index == index) {	// inc indexes above by one if needed
		ptr->index++;
		while (ptr->next) {
		    if (ptr->next->index == ptr->index) {
			ptr->next->index++;
		    }
		    ptr = ptr->next;
		}
	    }
	} else if (ptr->next == NULL) { 	// insert after
	    prev = ptr;
	    next = NULL;
	    (*head)->prev = this;
	    ptr->next = this;
	}
    }
}

Cvo_MatrixSlice::~Cvo_MatrixSlice()
{
    delete label;
    label = NULL;
    
    while (cells) {
	if (horizontal == False) {
	    if (cells->West() != cells) {
		delete cells->West();
	    } else {
		delete cells;
	    }
	} else {
	    if (cells->North() != cells) {
		delete cells->North();
	    } else {
		delete cells;
	    }
	}
    }

    if (next) {
	next->prev = prev;
    } else {
	if (matrix->rows && matrix->rows->prev == this) {// this Slice is a row
	    matrix->rows->prev = prev;
	} else {                      // this Slice is a col
	    matrix->cols->prev = prev;      
	}
    }

    if (prev->next) {
	prev->next = next;
    } else {
	if (matrix->rows == this) {   // this Slice is a row
	    matrix->rows = next;
	} else {                      // this Slice is a col
	    matrix->cols = next;      
	}
    }
}

//
//  Matrix Implementation
//

CONSTRUCTORS(Cvo_Matrix, Cvo_Window, "CvoMatrix")

void
Cvo_Matrix::_Init()
{
    horz_sep = GetResourceInt(_QhorizontalSpace, _QSpace, 3);
    vert_sep = GetResourceInt(_QverticalSpace, _QSpace, 3);

    rows = NULL;
    cols = NULL;
}

Cvo_Matrix::~Cvo_Matrix()
{
    while (rows) {
	if (rows->Prev() != rows) {
	    delete rows->Prev();
	} else {
	    delete rows;
	    rows = NULL;
	}
    }

    while (cols) {
	if (cols->Prev() != cols) {
	    delete cols->Prev();
	} else {
	    delete cols;
	    cols = NULL;
	}
    }
}

Cvo_MatrixSlice*
Cvo_Matrix::Search(Cvo_MatrixSlice* sl, int index) const
{
    Cvo_MatrixSlice* ptr = sl;

    while (ptr && ptr->Index() < index) {
	ptr = ptr->Next();
    }
    if (ptr && ptr->Index() == index) {
	return ptr;
    } else {
	return NULL;
    }
}

Cvo_MatrixSlice*
Cvo_Matrix::Row(int index) const
{
    if (index == -1) {
	return rows;
    } else {
	return (Search(rows, index));
    }
}

Cvo_MatrixSlice*
Cvo_Matrix::Col(int index) const
{
    if (index == -1) {
	return cols;
    } else {
	return (Search(cols, index));
    }
}

Cvo_MatrixSlice*
Cvo_Matrix::AddRow(int i)
{
    return(new Cvo_MatrixSlice(this, False, i));
}

Cvo_MatrixSlice*
Cvo_Matrix::AddCol(int i)
{
    return(new Cvo_MatrixSlice(this, True, i));
}

Cvo_MatrixCell*
Cvo_Matrix::AddCell(Cvo_Object* obj, int r, int c)
{
    Cvo_MatrixSlice *sr = Row(r);
    Cvo_MatrixSlice *sc = Col(c);

    if (!sr)
	sr = AddRow(r);
    if (!sc)
	sc = AddCol(c);

    return((sr && sc) ? AddCell(obj, sr, sc) : 0);
}

Cvo_MatrixCell*
Cvo_Matrix::AddCell(Cvo_Object* obj, Cvo_MatrixSlice* s1, Cvo_MatrixSlice* s2)
{
    if (s1->matrix != s2->matrix) {
	return(0);  		//  Row and column from different matricies
    }

    if (s1->Horizontal() == s2->Horizontal()) {
	return(0);		// Both Slices are of the same type
    }

    if (obj->Parent() != s1->matrix) {
	return(0);		//  Managing an object not parented by matrix
    }

    return (new Cvo_MatrixCell(obj, s1, s2));
}

Cvo_MatrixCell*
Cvo_Matrix::GetCell(Cvo_MatrixSlice* s1, Cvo_MatrixSlice* s2)
{
    if (!s1 || !s2) {
	return (0);
    } 
	
    if (s1->matrix != s2->matrix) {
	return(0);  		//  Row and column from different matricies
    }

    Cvo_MatrixSlice* row;
    Cvo_MatrixSlice* col;
    if (s1->Horizontal() == True && s2->Horizontal() == False) {
	row = s2;
	col = s1;
    } else if (s2->Horizontal() == True && s1->Horizontal() == False) {
	row = s1;
	col = s2;
    } else {
	return(0);		// Both Slices are of the same type
    }

    if (col->Cells() == 0 || row->Cells() == 0) {
	return (0);
    } 

    Cvo_MatrixCell* row_ptr = col->Cells();

    while (row_ptr->South() != NULL &&
	   row_ptr->Row()->Index() < row->Index()) {
	row_ptr=row_ptr->South();
    }
    if (row_ptr->Row()->Index() != row->Index()) {
	return (0);
    }

    return (row_ptr);
}

void
Cvo_Matrix::LayoutSlice(Cvo_MatrixSlice *s, int base, int min, int des, int pad)
{
    Cvo_MatrixSlice *ptr;

    //
    // Try to used desired sizes first.  If we don't have enough room, go back
    // to minimum size
    //
    int extra = base - des;

    if (extra < 0) {
	if ((extra = base - min) < 0)
	    extra = 0;
	for (ptr = s; ptr; ptr = ptr->Next())
	    ptr->des = ptr->min;
    }

    BOOL adjusted;

    //
    // Now loop until either we can't do anything more or we have used
    // up all the extra space, setting up where and how wide/high each
    // part of the slice will be.
    //
    do {
	int origin = pad;
	int expand = 0;

	for (ptr = s; ptr; ptr = ptr->Next()) {
	    if (!ptr->max || ptr->max > ptr->des)
		expand += ptr->expand;
	}

	adjusted = False;

	for (ptr = s; ptr; ptr = ptr->Next()) {
	    if ((!ptr->max || ptr->max > ptr->des) && ptr->expand
						   && expand
						   && extra > 0) {
		int tra = (extra * ptr->expand + expand - 1) / expand;

		if (ptr->max && ptr->des + tra > ptr->max)
		    tra = ptr->max - ptr->des;
		ptr->des += tra;
		expand -= ptr->expand;
		extra -= tra;

		adjusted = True;
	    }
	    ptr->origin = origin; 
	    origin += ptr->des + pad;
	}
    } while (adjusted && extra > 0);
}


void
Cvo_Matrix::StandardLayoutChildren(int force, int olay)
{
    if (olay)
	return;

    LayoutSlice(cols, width, layout.minimum.width, deswidth, horz_sep);
    LayoutSlice(rows, height, layout.minimum.height, desheight, vert_sep);

    Cvo_MatrixSlice* rowptr = rows;
    while (rowptr) {
	Cvo_MatrixCell* cellptr;
	if ((cellptr = rowptr->Cells()) != NULL) {
	    while (cellptr) {
		Cvo_LayoutWindow* obj = cellptr->CellObj()->ToLayoutWindow();
		if (obj) {

		    obj->Layout()->x = cellptr->Org_x();
		    obj->Layout()->y = cellptr->Org_y();
		    obj->LayoutFunction(force|1, 1, 
					cellptr->Ext_x(), cellptr->Ext_y());
		}
		cellptr = cellptr->East();
	    }
	}
	rowptr = rowptr->Next();
    }
}

int
Cvo_Matrix::StandardComputeLayoutSize(int force)
{
    Cvo_FrameSize mymin, mydes;

    ComputeUserSizes(mymin, mydes);

    desheight = 0;
    deswidth = 0;

    layout.minimum.height = 0;
    layout.minimum.width = 0;

    Cvo_Object *obj = children;
    while (obj) {
        Cvo_LayoutWindow *win = obj->ToLayoutWindow();
	win->ComputeLayoutSize(force);
	obj = obj->Sibling();
    }
    
    int separator = 1;
    
    for (Cvo_MatrixSlice* rowptr = rows; rowptr; rowptr = rowptr->Next()) {
	separator = horz_sep;
	rowptr->min = 0;
	rowptr->des = 0;
	rowptr->max = 1;
	rowptr->origin = -1;
	rowptr->expand = 0;
	if (rowptr->Label()) {
	    rowptr->Label()->ComputeLayoutSize(force);
	    Cvo_LayoutWindow* obj = rowptr->Label()->ToLayoutWindow();
	    if (obj) {
		rowptr->min = obj->Layout()->minimum.height;
		rowptr->des = obj->Layout()->desired.height;
		rowptr->max = obj->Layout()->maximum.height;
	    }
	}
	if (rowptr->Cells()) {
	    for (Cvo_MatrixCell* cellptr = rowptr->Cells(); cellptr;
		 cellptr = cellptr->East()) {
		Cvo_LayoutWindow* obj = cellptr->CellObj()->ToLayoutWindow();
		if (obj) {
		    rowptr->expand = max(rowptr->expand, obj->Layout()->expand);
		    rowptr->min =
			max(rowptr->min, obj->Layout()->minimum.height);
		    rowptr->des =
			max(rowptr->des, obj->Layout()->desired.height);
		    if (rowptr->max && obj->Layout()->maximum.height)
			rowptr->max =
			    max(rowptr->max, obj->Layout()->maximum.height);
		    else
			rowptr->max = 0;
		}
	    }
	}
	if (rowptr->max == 1)
		rowptr->max = 0;
	if (rowptr->min || rowptr->des || rowptr->max) {
	    layout.minimum.height += rowptr->min + separator;
	    desheight += rowptr->des + separator;
	    if (rowptr->max && rowptr->max <= rowptr->min) {
		rowptr->max = rowptr->min;
	    }
	}
    }

    layout.minimum.height += separator;
    desheight += separator;

    for (Cvo_MatrixSlice* colptr = cols; colptr; colptr = colptr->Next()) {
	colptr->min = 0;
	colptr->des = 0;
	colptr->max = 1;
	colptr->origin = -1;
	separator = vert_sep;
	colptr->expand = 0;
	if (colptr->Label()) {
	    colptr->Label()->ComputeLayoutSize(force);
	    Cvo_LayoutWindow* obj = colptr->Label()->ToLayoutWindow();
	    if (obj) {
		colptr->min = obj->Layout()->minimum.width;
		colptr->des = obj->Layout()->desired.width;
		colptr->max = obj->Layout()->maximum.width;
	    }
	}
	if (colptr->Cells()) {
	    for (Cvo_MatrixCell* cellptr = colptr->Cells(); cellptr;
		 cellptr = cellptr->South()) {
		Cvo_LayoutWindow* obj = cellptr->CellObj()->ToLayoutWindow();
		if (obj) {
		    colptr->expand = max(colptr->expand, obj->Layout()->expand);
		    colptr->min =
			max(colptr->min, obj->Layout()->minimum.width);
		    colptr->des =
			max(colptr->des, obj->Layout()->desired.width);
		    colptr->max =
			max(colptr->max, obj->Layout()->maximum.width);
		    if (colptr->max && obj->Layout()->maximum.width)
			colptr->max =
			    max(colptr->max, obj->Layout()->maximum.width);
		    else
			colptr->max = 0;
		}
	    }
	}
	if (colptr->max == 1)
		colptr->max = 0;
	if (colptr->min || colptr->des || colptr->max) {
	    layout.minimum.width += colptr->min + separator;
	    deswidth += colptr->des + separator;
	    if (colptr->max && colptr->max <= colptr->min) {
		colptr->max = colptr->min;
	    }
	}
    }

    layout.minimum.width += separator;
    deswidth += separator;

    int rs = layout.ignoreshadow ? 0 : raise;

    layout.minimum.height += (borderWidth + verticalPad) * 2 + rs;
    layout.minimum.width += (borderWidth + horizontalPad) * 2 + rs;

    if (layout.minimum.width < mymin.width)
	layout.minimum.width = mymin.width;

    if (layout.minimum.height < mymin.height)
	layout.minimum.height = mymin.height;

    if (deswidth < layout.minimum.width)
	deswidth = layout.minimum.width;

    if (desheight < layout.minimum.height)
	desheight = layout.minimum.height;

    if (mydes.width < layout.minimum.width)
	mydes.width = layout.minimum.width;

    if (mydes.height < layout.minimum.height)
	mydes.height = layout.minimum.height;

    if (deswidth < mydes.width)
    	layout.desired.width = mydes.width;
    else
	layout.desired.width = deswidth;

    if (desheight < mydes.height)
	layout.desired.height = mydes.height;
    else
	layout.desired.height = desheight;

    if (layout.maximum.width && layout.maximum.width < layout.minimum.width)
	layout.maximum.width = layout.minimum.width;
    if (layout.maximum.height && layout.maximum.height < layout.minimum.height)
	layout.maximum.height = layout.minimum.height;

    return (CheckLayout());
}
