/* $Id: Property.c,v 1.1.1.1 1997/03/19 13:32:22 glgay Exp $ */
/*
 Copyright (C) 1993, 1994 Jettero Heller
 Copyright (C) 1994, 1995 Peter Williams
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License 
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave, 
 Cambridge, MA 02139, USA.
*/


#include <Xmext/PropertyP.h>
#include <Xmext/PageP.h>
#include <Xmext/TabP.h>
#include <Xm/DialogS.h>
#include <Xm/PushBP.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <stdio.h>

/* PropertySheet widget */
#define Unused(x) (void)(x)
#define Offset(field) XtOffsetOf(XmPropertySheetRec, property_sheet.field)

/* class names for private widgets */
#define PS_TAB_CN "XmPropertySheetTab"
#define PS_OK_CN "OK"
#define PS_CANCEL_CN "Cancel"
#define PS_APPLY_CN "Apply"
#define PS_HELP_CN "Help"
#define PS_TAB_HOLDER_CN "Tab_Holder"
#define PS_SEPARATOR_CN "Separator"

/* local declarations */
static void activate_cb(Widget w, XtPointer udata, XtPointer cdata); 
static void change_managed(Widget w);
static void class_initialize();
static void class_part_initialize(WidgetClass class);
static void destroy(Widget w);
static XtGeometryResult geometry_manager(Widget w, 
                                         XtWidgetGeometry *request, 
                                         XtWidgetGeometry *reply);
static void initialize(Widget request, 
                       Widget c_new,
                       ArgList args,
                       Cardinal *num_args);
static void kill_active_page(Widget sheet, Boolean notify);
static void layout(Widget sheet);
static void layout_active_page(Widget sheet);
static void layout_buttons(Widget sheet);
static void layout_tabs(Widget sheet);
static void layout_separator(Widget sheet);
static void lower_tab(Widget tab);
static int page_position(Widget sheet, Widget page);
static XtGeometryResult query_geometry(Widget w, 
                                      XtWidgetGeometry *proposed, 
                                      XtWidgetGeometry *answer);
static void raise_tab(Widget tab);
/* realize sets the active page & tab if not null */
static void realize(Widget w,
                    XtValueMask *value_mask,
                    XSetWindowAttributes *attributes);
static void resize(Widget w);
static void set_active_page(Widget sheet, Widget page, Boolean notify);
static void set_active_pos(Widget sheet, int pos, Boolean notify);
static Boolean set_values(Widget current, 
                          Widget request, 
                          Widget c_new, 
                          ArgList args,
                          Cardinal *num_args);
static void tab_activate_cb(Widget w, XtPointer udata, XtPointer cdata);

#if 0
static void XmPropertySheetSetActivePage(Widget sheet, 
                                         Widget page, 
                                         Boolean notify);
#endif
static void XmPropertySheetSetActivePos(Widget sheet, 
                                        int position, 
                                        Boolean notify);

/* locale functions */
static void _XmSapplyLabelString(Widget wid, int offset, XrmValue *value);
static void _XmScancelLabelString(Widget wid, int offset, XrmValue *value);
static void _XmShelpLabelString(Widget wid, int offset, XrmValue *value);
static void _XmSokLabelString(Widget wid, int offset, XrmValue *value);

/* ========================================================================== */
/* Resources for the Property Sheet class */
static XtResource resources[] = {

   {
     XmNactivePos,
     XmCActivePos, 
     XmRInt,
     sizeof(int),
     Offset(active_pos),
     XmRImmediate,
     0
   },

   { 
     XmNapplyLabelString, /* resource_name: specify using XtN symbol */
     XmCApplyLabelString, /* resource_class: specify using XtC symbol */
     XmRXmString,          /* resource_type: actual data type of variable */
     sizeof(XmString),     /* resource_size: specify using sizeof() */
     Offset(apply_label_string), 
                           /* resource_offset: specify using XtOffsetOf() */ 
     XtRCallProc,          /* default_type: will be converted to resrc_type */
     (XtPointer) _XmSapplyLabelString 
                           /* default_address: address of default value */
    },

   {
     XmNapplyCallback,
     XmCCallback, 
     XtRCallback,
     sizeof(XtCallbackList),
     Offset(apply_callback),
     XmRImmediate,
     NULL
   },

   {
     XmNcancelLabelString, 
     XmCCancelLabelString, 
     XmRXmString, 
     sizeof(XmString),
     Offset(cancel_label_string), 
     XtRCallProc, 
     (XtPointer) _XmScancelLabelString
    },

   {
     XmNcancelCallback,
     XmCCallback, 
     XtRCallback,
     sizeof(XtCallbackList),
     Offset(cancel_callback),
     XmRImmediate,
     NULL
   },

   {
     XmNhelpLabelString, 
     XmCHelpLabelString, 
     XmRXmString, 
     sizeof(XmString),
     Offset(help_label_string), 
     XtRCallProc, 
     (XtPointer) _XmShelpLabelString
   }, 

   {
     XmNokLabelString, 
     XmCOkLabelString, 
     XmRXmString, 
     sizeof(XmString),
     Offset(ok_label_string), 
     XtRCallProc, 
     (XtPointer) _XmSokLabelString
   }, 

   {
     XmNokCallback,
     XmCCallback, 
     XtRCallback,
     sizeof(XtCallbackList),
     Offset(ok_callback),
     XmRImmediate,
     NULL
   },

   {
     XmNpageCount,
     XmCPageCount,
     XmRInt,
     sizeof(int),
     Offset(page_count),
     XmRImmediate,
     (XtPointer) 0
   },

   {
     XmNpages,
     XmCPages, 
     XmRWidgetList,
     sizeof(WidgetList),
     Offset(pages),
     XmRImmediate,
     NULL
   },

   {
     XmNtabs,
     XmCTabs, 
     XmRWidgetList,
     sizeof(WidgetList),
     Offset(tabs),
     XmRImmediate,
     NULL
   }

};

/* ========================================================================== */
/* Class record */

XmPropertySheetClassRec xmPropertySheetClassRec = {
   /* Core class part */
   {
      (WidgetClass) &xmBulletinBoardClassRec, /* superclass */
      "XmPropertySheet",                      /* class_name */
      sizeof(XmPropertySheetRec),             /* widget_size */
      class_initialize,                       /* class_initialize */
      class_part_initialize,                  /* class_part_initialize */
      FALSE,                                  /* class_inited */
      initialize,                             /* initialize */
      NULL,                                   /* initialize_hook */
      realize,                                /* realize */
      NULL,                                   /* actions */
      0,                                      /* num_actions */
      resources,                              /* resources */
      XtNumber(resources),                    /* num_resources */
      NULLQUARK,                              /* xrm_class */
      TRUE,                                   /* compress_motion */
      XtExposeCompressMultiple,               /* compress_exposure */
      TRUE,                                   /* compress_enterleave */
      FALSE,                                  /* visible_interest */
      destroy,                                /* destroy */
      resize, /* XtInheritResize,  */                      /* resize */
      XtInheritExpose,                        /* expose */
      set_values,                             /* set_values */
      NULL,                                   /* set_values_hook */
      XtInheritSetValuesAlmost,               /* set_values_almost */
      NULL,                                   /* get_values_hook */
      NULL,                                   /* accept_focus */
      XtVersion,                              /* version */
      NULL,                                   /* callback offsets */
      XtInheritTranslations,                  /* tm_table */
      query_geometry,                         /* query_geometry */
      NULL,                                   /* display_accelerator */
      NULL                                    /* extension */ 
   },
   /* Composite class part */
   {
      geometry_manager,       /* geometry manager */
      change_managed,         /* change_managed */
      XtInheritInsertChild,   /* insert_child */ 
      XtInheritInsertChild,   /* delete_child */ 
      NULL                    /* extension */ 
   },
   /* Constraint class part */
   {
      NULL,      /* FIX ME */ /* subresources */
      0,         /* FIX ME */ /* subresource_count */
      0,         /* FIX ME */ /* constraint_size */
      NULL,      /* FIX ME */ /* initialize */
      NULL,      /* FIX ME */ /* destroy */
      NULL,      /* FIX ME */ /* set_values */
      NULL                    /* extension */ 
   },
   /* XmManager class part */
   {
      NULL,                   /* translations */ 
      NULL,                   /* syn_resources */ 
      0,                      /* num_syn_resources */
      NULL,                   /* syn_constraint_resources */ 
      0,                      /* num_syn_constraint_resources */
      XmInheritParentProcess, /* parent process */
      NULL                    /* extension */ 
   },
   /* XmBulletin class part */
   {
      (Dimension) NULL                    /* extension */ 
   },
   /* XmPropertySheet class part */
   {
      NULL                    /* extension */ 
   }
};

WidgetClass xmPropertySheetWidgetClass = (WidgetClass) &xmPropertySheetClassRec;

/* ========================================================================== */
/* locale functions */

static void
_XmSapplyLabelString(Widget wid, int offset, XrmValue *value)
{
   /* here is the default for the C locale */
   value->addr = (XtPointer) "Apply";
   /* 
    * Read value from array table or file if not in C locale 
    * FIX ME 
    */
}

static void
_XmScancelLabelString(Widget wid, int offset, XrmValue *value)
{
   /* here is the default for the C locale */
   value->addr = (XtPointer) "Cancel";
   /* 
    * Read value from array table or file if not in C locale 
    * FIX ME 
    */
}

static void
_XmShelpLabelString(Widget wid, int offset, XrmValue *value)
{
   /* here is the default for the C locale */
   value->addr = (XtPointer) "Help";
   /* 
    * Read value from array table or file if not in C locale 
    * FIX ME 
    */
}

static void
_XmSokLabelString(Widget wid, int offset, XrmValue *value)
{
   /* here is the default for the C locale */
   value->addr = (XtPointer) "OK";
   /* 
    * Read value from array table or file if not in C locale 
    * FIX ME 
    */
}

/* ========================================================================== */
/* functions */

/*
 * activate_cb
 *
 * This function is used to dispatch activate callbacks for 
 * the various button activate callbacks resources.
 */
static void
activate_cb(Widget w, XtPointer udata, XtPointer cdata)
{
   XmPropertySheetWidget ps = (XmPropertySheetWidget) XtParent(w);
   int i, page_count;
   WidgetList pages;
   Widget child;

   pages = PS_Pages(ps);
   page_count = PS_PageCount(ps);

   if(w == PS_OkButton(ps))
   {
      for(i=0;i<page_count;i++)
      {
         child=pages[i];
         XtCallCallbackList((Widget)child,PP_OkCallback(child), cdata); 
      }
      XtCallCallbackList((Widget) ps,
                         PS_OkCallback(ps),
                         cdata);
   }
   else if(w == PS_ApplyButton(ps))
   {
      for(i=0;i<page_count;i++)
      {
         child=pages[i];
         XtCallCallbackList((Widget)child,PP_ApplyCallback(child), cdata); 
      }
      XtCallCallbackList((Widget) ps, 
                         PS_ApplyCallback(ps),
                         cdata);
   }
   else if(w == PS_CancelButton(ps))
   {
      for(i=0;i<page_count;i++)
      {
         child=pages[i];
         XtCallCallbackList((Widget)child,PP_CancelCallback(child), cdata); 
      }
      XtCallCallbackList((Widget) ps,
                         PS_CancelCallback(ps),
                         cdata);
   }
   else if(w == PS_HelpButton(ps))
   {
      XtCallCallbackList((Widget) ps,
                         PS_HelpCallback(ps),
                         cdata);
   }
}

static void 
change_managed(Widget w)
{
   #if defined(DEBUG)
   fprintf(stderr, "change_managed()\n");
   #endif
   /*
    * By default the first page is the first active page  
    * if active_pos is equal to zero.  If not zero then
    * the programmer set the active page before managing the
    * widget.
    */
   if (PS_PageCount(w) > 0)
   {
      if (PS_ActivePos(w) <= 0)
      {
         XmPropertySheetSetActivePos(w, 1, TRUE);
      }
   }

   if(XtIsRealized(w))
   {
      layout_active_page(w);
   }
}

static void
class_initialize()
{
}

static void
class_part_initialize(WidgetClass widget_class)
{
   Unused(widget_class);
}

static void
destroy(Widget w)
{
   Unused(w);
}

static void
display_page(Widget sheet, Widget new_page, Widget old_page)
{

   #if defined(DEBUG)
   fprintf(stderr,"display_page()\n");
   #endif

   /*
    * Move separator under active tab
    */
   XtManageChild(new_page);
   if(old_page != NULL)
   {
      XtUnmanageChild(old_page);
   }
}

static XtGeometryResult 
geometry_manager(Widget w, XtWidgetGeometry *request, XtWidgetGeometry *reply)
{
   #if defined(DEBUG)
   fprintf(stderr,"geometry_manager()\n");
   #endif

   /* FIX ME */
   Unused(w);
   Unused(request);
   Unused(reply);
   return (XtGeometryResult) NULL;
}

static void
autoSpaceWidget(Widget w, int size)
{
   int border;
   border = XtWidth(w);

   if(border < size)
   {
      int offset = (size-border)%2;
      border = (size-border)/2;
      XtVaSetValues(w, 
                    XmNmarginLeft, border,
                    XmNmarginRight, ((Dimension) border+offset), 
                    NULL);
   }

   XtVaSetValues(w, 
                 XmNhighlightThickness, 1,
                 XmNborderWidth, 0, 
                 NULL);
}

static void
initialize(Widget request,
           Widget c_new,
           ArgList args,
           Cardinal *num_args)
{

   PS_OkButton(c_new) = XmCreatePushButton(c_new, PS_OK_CN, NULL, 0);
   PS_DefaultButton(c_new) = PS_OkButton(c_new);
   PS_CancelButton(c_new) = XmCreatePushButton(c_new, PS_CANCEL_CN, NULL, 0);
   PS_ApplyButton(c_new) = XmCreatePushButton(c_new, PS_APPLY_CN, NULL, 0);
   PS_HelpButton(c_new) = XmCreatePushButton(c_new, PS_HELP_CN, NULL, 0);
   PS_Separator(c_new) = XmCreateSeparator(c_new, PS_SEPARATOR_CN, NULL, 0);
   PS_TabHolder(c_new) = NULL; /* not implemented */

   /* make the buttons look nice */
   autoSpaceWidget(PS_OkButton(c_new), 78);
   autoSpaceWidget(PS_CancelButton(c_new), 100);
   autoSpaceWidget(PS_ApplyButton(c_new), 100);
   autoSpaceWidget(PS_HelpButton(c_new), 100);

   XtVaSetValues(PS_OkButton(c_new), 
                 XmNshowAsDefault, (Dimension) 1,  
                 NULL);

   #if 1
   XtManageChild(PS_OkButton(c_new));
   XtManageChild(PS_CancelButton(c_new));
   XtManageChild(PS_ApplyButton(c_new));
   #if 0
   XtManageChild(PS_HelpButton(c_new));
   #endif
   #endif
   XtManageChild(PS_Separator(c_new));

   /* get rid of auto_unmanage for the apply and help widgets */
   XtRemoveAllCallbacks(PS_ApplyButton(c_new), XmNactivateCallback);
   #if 0
   XtRemoveAllCallbacks(PS_HelpButton(c_new), XmNactivateCallback);
   #endif

   /* add the activate callback handler */
   XtAddCallback(PS_OkButton(c_new), XmNactivateCallback, activate_cb, NULL);
   XtAddCallback(PS_CancelButton(c_new), XmNactivateCallback, activate_cb,NULL);
   XtAddCallback(PS_ApplyButton(c_new), XmNactivateCallback, activate_cb, NULL);
   XtAddCallback(PS_HelpButton(c_new), XmNactivateCallback, activate_cb, NULL);
}

/*
 * kill_active_page
 *
 * This function unmanages the active page
 * and sets the active page resource in the property sheet to null.
 */
static void
kill_active_page(Widget sheet, Boolean notify)
{
   /* if active_pos is set to zero then there is no active page currently. */
   if(PS_ActivePos(sheet) > 0)
   {
      if(notify == TRUE)
      {
         XtCallCallbackList(PS_ActivePage(sheet), 
                         PP_KillActiveCallback(PS_ActivePage(sheet)),
                         NULL);
      }

      lower_tab(PS_ActiveTab(sheet));

	      /* zero means there is no active page */
      PS_ActivePos(sheet) = 0; 
   }
}

#define VALID(w)	(w && XtIsManaged(w))
#if !defined(BB_MarginWidth)
#define BB_MarginWidth(w) \
        (((XmBulletinBoardWidget)(w))->bulletin_board.margin_width)
#endif
#if !defined(BB_MarginHeight)
#define BB_MarginHeight(w) \
        (((XmBulletinBoardWidget)(w))->bulletin_board.margin_height)
#endif

/*
* layout_tabs
*
* This function positions the tab buttons accordingly to the
* sheet's geometry.  For now it just lays them out in one row.
*/
static void
layout_tabs(Widget w)
{
   Position x, y, offset_x, offset_y;
   Dimension shadow_thickness, offset_width; 
   int i;

   /* cycle thru all tabs and lay them out or use the holder */
   if (PS_PageCount(w) > 0)
   {

      shadow_thickness = MGR_ShadowThickness(PS_ActivePage(w));
      x = BB_MarginWidth(w) + (2*shadow_thickness);
      y = BB_MarginHeight(w); 

      for(i=0; i<PS_PageCount(w); i++)
      {
         /* offset for active tab */
         if(i == (PS_ActivePos(w) - 1))
         {
            shadow_thickness = Prim_ShadowThickness(PS_Tabs(w)[i]);
            offset_x = 2*shadow_thickness;
            offset_y = shadow_thickness;
            offset_width = 4*shadow_thickness; 
         }
         else
         {
            offset_x = 0;
            offset_y = 0;
            offset_width = 0;
         }
         XtMoveWidget(PS_Tabs(w)[i],x-offset_x,y-offset_y);
         x += XtWidth(PS_Tabs(w)[i])-offset_width;
      }

   }
}

/*
* lower_tab
*
* This function gives the impression that the tab is being lowered.
*/
static void
lower_tab(Widget w)
{
   /* this will lower the tab */
   XtVaSetValues(w, XmNshowAsActive, False, 0);

   /* 
    * Lets reconfigure it too make it look smaller. 
    * This adds a nice visual effect.
    */
   {
      Position x, y;
      Dimension height, width, border_width, shadow_thickness; 

      shadow_thickness = Prim_ShadowThickness(w);
      width = XtWidth(w) - 4*shadow_thickness; 
      height = XtHeight(w) - shadow_thickness; 
      border_width = XtBorderWidth(w);
      x = XtX(w) + 2*shadow_thickness;
      y = XtY(w) + shadow_thickness;
      XtConfigureWidget(w,x,y,width,height,border_width);
   }
}

/*
* raise_tab
*
* This function gives the impression that the tab is being raised.
*/
static void
raise_tab(Widget w)
{
   /* this will raise the tab */
   XtVaSetValues(w, XmNshowAsActive, True, 0);
 
   /* 
    * Lets reconfigure it too make it look bigger. 
    * This adds a nice visual effect.
    */
   {
      Position x, y;
      Dimension height, width, border_width, shadow_thickness; 

      shadow_thickness = Prim_ShadowThickness(w);
      width = XtWidth(w) + 4*shadow_thickness; 
      height = XtHeight(w) + shadow_thickness; 
      border_width = XtBorderWidth(w);
      x = XtX(w) - 2*shadow_thickness;
      y = XtY(w) - shadow_thickness;
      XtConfigureWidget(w,x,y,width,height,border_width);
   }
}

/*
* layout_separator
*
* This function positions and resizes the separator between the active tab
* and the active page to cover the active pages shadow.
*/
static void
layout_separator(Widget w)
{
   if (PS_PageCount(w) > 0)
   {
      Dimension shadow_thickness;
      Position x, y;
      Pixel color;

      /* position separator below active tab but on top of active page */
      shadow_thickness = Prim_ShadowThickness(PS_ActiveTab(w));
      y = XtY(PS_ActivePage(w));
      x = XtX(PS_ActiveTab(w)) + shadow_thickness;
      /* width should be account for the tabs shadow thickness */
      XtWidth(PS_Separator(w)) = XtWidth(PS_ActiveTab(w)) - (2*shadow_thickness);

      XtResizeWindow(PS_Separator(w));
      XtMoveWidget(PS_Separator(w), x, y);
     
      /* While we are at it lets change the colors to match the current page */

      Prim_ShadowThickness(PS_Separator(w)) = MGR_ShadowThickness(PS_ActivePage(w));
      XtVaGetValues(PS_ActivePage(w),
                    XmNbackground, &color,
                    NULL);

      XtVaSetValues(PS_Separator(w),
                 XmNbackground, color,
                 XmNborderColor, color,
                 XmNbottomShadowColor, color,
                 XmNtopShadowColor, color,
                 XmNforeground, color,
                 NULL);
      if(XtWindow(PS_Separator(w)))
      {
         XRaiseWindow(XtDisplay(PS_Separator(w)), XtWindow(PS_Separator(w)));
      }
   }
}

/*
* layout_buttons
*
*/
static void
layout_buttons(Widget w)
{
   Position vp, hp, y;

   /* if button is managed then include it in the calculation */
   /* do calculations rigth to left */

   vp = XtHeight(w) - BB_MarginHeight(w);
   hp = XtWidth(w) - BB_MarginWidth(w);
   if(VALID(PS_HelpButton(w)))
   {
      y = vp - XtHeight(PS_HelpButton(w));
      hp -= XtWidth(PS_HelpButton(w));
      XtMoveWidget(PS_HelpButton(w), hp, y);
      #if 0
      hp -= BB_MarginWidth(w);
      #endif
   }
   
   if(VALID(PS_ApplyButton(w)))
   {
      y = vp - XtHeight(PS_ApplyButton(w));
      hp -= XtWidth(PS_ApplyButton(w));
      XtMoveWidget(PS_ApplyButton(w), hp, y);
      #if 0
      hp -= BB_MarginWidth(w);
      #endif
   }
   
   if(VALID(PS_CancelButton(w)))
   {
      y = vp - XtHeight(PS_CancelButton(w));
      hp -= XtWidth(PS_CancelButton(w));
      XtMoveWidget(PS_CancelButton(w), hp, y);
      #if 0
      hp -= BB_MarginWidth(w);
      #endif
   }

   if(VALID(PS_OkButton(w)))
   {
      y = vp - XtHeight(PS_OkButton(w));
      hp -= XtWidth(PS_OkButton(w));
      XtMoveWidget(PS_OkButton(w), hp, y);
   }
}

/*
* layout_active_page
*
* This function positions and resizes the active page accordingly to the
* sheet's geometry. 
*/
static void
layout_active_page(Widget w)
{
   if (PS_PageCount(w) > 0)
   {
      Position x, y;
      Dimension width, height, border_width;
      Dimension adjust_for_button = 0, adjust_for_shadow;

      /* if all buttons are invisible then use the whole window */
      if(VALID(PS_OkButton(w)))
      {
         adjust_for_button = XtHeight(PS_OkButton(w));
      }

      if(VALID(PS_CancelButton(w))
         && (XtHeight(PS_CancelButton(w))>adjust_for_button))
      {
         adjust_for_button = XtHeight(PS_CancelButton(w));
      }

      if(VALID(PS_ApplyButton(w))
         && (XtHeight(PS_ApplyButton(w))>adjust_for_button))
      {
         adjust_for_button = XtHeight(PS_ApplyButton(w));
      }

      if(VALID(PS_HelpButton(w)) 
         && (XtHeight(PS_HelpButton(w))>adjust_for_button))
      {
         adjust_for_button = XtHeight(PS_HelpButton(w));
      }

      /* 
      * Doesn't matter which tab we pick.  They should all have the same
      * shadow thickness.
      */
      adjust_for_shadow = Prim_ShadowThickness(PS_ActiveTab(w));
      y = XtY(PS_ActiveTab(w)) + XtHeight(PS_ActiveTab(w)) - adjust_for_shadow;
      x = BB_MarginWidth(w);
      width = XtWidth(w) - (2*BB_MarginWidth(w));
      height = XtHeight(w) - ((2*BB_MarginHeight(w)) + adjust_for_button + y);
      border_width = XtBorderWidth(PS_ActivePage(w));

      XtConfigureWidget(PS_ActivePage(w), x, y, width, height, border_width);

      if(XtWindow(PS_ActivePage(w)))
      {
         XRaiseWindow(XtDisplay(PS_ActivePage(w)), XtWindow(PS_ActivePage(w)));
      }
   }
   /* 
   * This will cover the shadow on the active page
   * between the page and the tab.
   */
   layout_separator(w); 
}

/*
* layout
*
* This function repositions child windows according to its own
* geometry.
*
* For now we only handle one row of tabs.
* Here's the plan:
*    If the sheet is resized then reposition the OK, CANCEL, APPLY buttons
*    accordingly.  Next, resize the page accordingly.
* 
*    If the OK, CANCEL and APPLY buttons do not exist then let the page
*    stretch (resize) all the way to the bottom of the window.
*    
*    Hmm, maybe we should have a separate widget with the OK, CANCEL, APPLY
*    buttons that contains a property sheet.  This maybe more flexible.
*/
static void
layout(Widget w)
{
   #if defined(DEBUG)
   fprintf(stderr,"layout()\n");
   #endif

   layout_tabs(w);
   layout_buttons(w);
   layout_active_page(w);
}

/*
 * page_position 
 *
 * This function locates the position of the page
 * widget in the widget list.  A 0 is returned on failure
 *
 */
static int 
page_position(Widget sheet, Widget page)
{
   XmPropertySheetWidget ps = (XmPropertySheetWidget)sheet;
   WidgetList pages;
   int page_cnt, i;

   pages = ps->property_sheet.pages;
   page_cnt = ps->property_sheet.page_count;

   for (i=0; i<page_cnt; i++)
   {
      if(page == pages[i])
      {
         return i+1;   
      }
   }
   return 0;
}

static XtGeometryResult 
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
   /* FIX ME */
   Unused(w);
   Unused(proposed);
   Unused(answer);
   return (XtGeometryResult) NULL;
}

static void realize(Widget w,
                    XtValueMask *value_mask,
                    XSetWindowAttributes *attributes)
{
   #if defined(DEBUG)
   fprintf(stderr, "realize()\n");
   #endif

   #define superclass (&xmBulletinBoardClassRec)
   (*superclass->core_class.realize)(w, value_mask, attributes);
   #undef superclass

   layout(w);

   /* FIX ME */
   Unused(w);
   Unused(value_mask);
   Unused(attributes);
}

static void resize(Widget w)
{
   #if defined(DEBUG)
   fprintf(stderr, "resize()\n");
   #endif

   #define superclass (&xmBulletinBoardClassRec)

   if(XtIsRealized(w))
   {  /* widgets must be resized */
      layout(w);
      /* redraw ActivePage because it's probably dirty */
      (*superclass->core_class.resize)(PS_ActivePage(w));
   }

   (*superclass->core_class.resize)(w);
   #undef superclass
}


/*
 * remove_page
 *
 * This function removes the page from the property sheet window.
 * 
 */
static void
remove_page(Widget page)
{
   XtUnmanageChild(page);
}

/*
 * remove_tab
 *
 * This function removes the tab from the property sheet window
 * and destroys the tab widget.
 */
static void
remove_tab(Widget tab)
{
   /* unmanage the tab */
   XtUnmanageChild(tab);
   /* destroy the tab widget */
   XtDestroyWidget(tab);
}
   
/*
 * set_active_page
 *
 * This function manages the new page widget and displays it in the 
 * center of the sheet.  Also, the active page resource in the property
 * sheet is set to the new page widget.
 */
static void
set_active_page(Widget sheet, Widget page, Boolean notify)
{
   set_active_pos(sheet, page_position(sheet, page), notify);
}

static void
set_active_pos(Widget sheet, int pos, Boolean notify)
{
   /* kill active pos before resetting it */
   if (!(PS_ActivePos(sheet) <= 0 || PS_ActivePage(sheet) == NULL))
   {
      kill_active_page(sheet, notify);
   }

   if(pos <= 0 || pos > PS_PageCount(sheet))
   {
      XtAppError(XtWidgetToApplicationContext(sheet),
                 "Attempted to display a widget that is not"
                 "in my list.");
      return;
   }

   PS_ActivePos(sheet) = pos;
   if(notify == TRUE)
   {
      XtCallCallbackList(PS_ActivePage(sheet), 
                         PP_SetActiveCallback(PS_ActivePage(sheet)),
                         NULL);
   }
   raise_tab(PS_ActiveTab(sheet));
}

static Boolean 
set_values(Widget old, 
           Widget request, 
           Widget c_new, 
           ArgList args,
           Cardinal *num_args)
{
   XmPropertySheetWidget ow = (XmPropertySheetWidget) old;
   XmPropertySheetWidget nw = (XmPropertySheetWidget) c_new;
   Boolean need_refresh = False;

   #if defined(DEBUG)
   fprintf(stderr, "%s:%d XmPropertySheet %s set_values\n", 
           __FILE__, __LINE__, XtName(c_new)); 
   #endif

   if (PS_ActivePos(ow) != PS_ActivePos(nw))
   {
      PS_ActivePos(nw) = PS_ActivePos(ow);
      XtAppWarning(XtWidgetToApplicationContext(old),
                   "XmPropertySheet: active_pos (XmNactivePos) cannot be "
                   "set by XtSetValues.\n");
   }

   if (PS_PageCount(ow) != PS_PageCount(nw))
   {
      PS_PageCount(nw) = PS_PageCount(ow);
      XtAppWarning(XtWidgetToApplicationContext(old),
                   "XmPropertySheet: page_count (XmNpageCount) cannot be "
                   "set by XtSetValues.\n");
   }

   if (PS_Pages(ow) != PS_Pages(nw))
   {
      PS_Pages(nw) = PS_Pages(ow);
      XtAppWarning(XtWidgetToApplicationContext(old),
                   "XmPropertySheet: pages (XmNpages) cannot be "
                   "set by XtSetValues.\n");
   }

   if (PS_Tabs(ow) != PS_Tabs(nw))
   {
      PS_Tabs(nw) = PS_Tabs(ow);
      XtAppWarning(XtWidgetToApplicationContext(old),
                   "XmPropertySheet: tabs (XmNtabs) cannot be "
                   "set by XtSetValues.\n");
   }

   Unused(request);
   Unused(args);
   Unused(num_args);

   return need_refresh;
}

/*
 * tab_activate_cb
 *
 * This function is used to dispatch the XmNsetActive callbacks and
 * the XmNkillActive callbacks to the page widget.
 */
static void 
tab_activate_cb(Widget w, XtPointer udata, XtPointer cdata)
{
   Widget sheet, new_page, old_page;

   sheet = XtParent(w);

   /* the udata should be the page widget */
   new_page = (Widget) udata;
   old_page = PS_ActivePage(sheet);

   /* if a different page was selected then */
   if(old_page != new_page)
   {
      set_active_page(sheet, new_page, TRUE);
      display_page(sheet, new_page, old_page);
   }
}

/*
 * XmCreatePropertySheet
 *
 * Creates an instance of a particular widget class or compound object.
 */
Widget 
XmCreatePropertySheet(Widget parent, char *name, ArgList arglist,
                      Cardinal cnt)
{
   return XtCreateWidget(name, 
                         xmPropertySheetWidgetClass,
                         parent, 
                         arglist, 
                         cnt);
}

/*
 * XmCreatePropertySheetDialog
 *
 * Creates an instance of a particular widget class or compound object.
 * The child widget of the dialog shell is returned.
 */
Widget 
XmCreatePropertySheetDialog(Widget parent, 
                            char *name, 
                            ArgList arglist,
                            Cardinal cnt)
{
   Widget shell;
   Arg args[10];
   int nargs;
   char *shell_name;

   nargs = 0;
   XtSetArg(args[nargs], XmNallowShellResize, True); nargs++;
 
   shell_name = XtMalloc(strlen(name) + strlen(XmDIALOG_SUFFIX) + 1);
   strcpy(shell_name, name);
   strcat(shell_name, XmDIALOG_SUFFIX);
                           
   shell = XmCreateDialogShell(parent, shell_name, args, nargs);

   XtManageChild(shell);
   XtFree(shell_name);

   return XmCreatePropertySheet(shell, name, arglist,  cnt);
}

/*
 * XmPropertySheetGetChild
 *
 * This functions will return the specified child of the PropertySheet widget.
 */
Widget
XmPropertySheetGetChild(Widget parent, unsigned char child)
{
   XmPropertySheetWidget ps = (XmPropertySheetWidget)parent;
   switch(child)
   {
      case XmDIALOG_CANCEL_BUTTON:
         return PS_CancelButton(ps);
         break;
      case XmDIALOG_OK_BUTTON:
         return PS_OkButton(ps);
         break;
      case XmDIALOG_APPLY_BUTTON:
         return PS_ApplyButton(ps);
         break;
      case XmDIALOG_HELP_BUTTON:
         return PS_HelpButton(ps);
         break;
      case XmDIALOG_DEFAULT_BUTTON:
         /*
          * There does not seem to be a difference between the values of
          * default_button and dynamic_default_button
          * so we will use the value of default_button
          */
         return PS_DefaultButton(ps);
         break;
      default:
         /* bad value for child */
         return NULL;
   }
}

/*
 * XmPropertySheetGetPage
 * 
 * This function returns the page widget at the specified location.
 * If the specified position is not in the sheet then a NULL is returned.
 */
Widget
XmPropertySheetGetPage(Widget sheet, int position)
{

   if(position > PS_PageCount(sheet) || position <= 0)
   {
      return (Widget) NULL;
   }

   return (Widget) (PS_Pages(sheet)[position - 1]);
}

/*
 * XmPropertySheetAddPage
 * 
 * This function adds a page widget to the specified location.
 * A position of 0 indicates that the page should be added to the
 * end.
 */
void
XmPropertySheetAddPage(Widget sheet, Widget page, int position)
{
   WidgetList pages;
   WidgetList tabs;
   int page_count;
   XmString tab_string;
   Widget tab;

   pages = PS_Pages(sheet);
   tabs = PS_Tabs(sheet);
   page_count = PS_PageCount(sheet);

   if(XmIsPropertyPage(page) == FALSE)
   {
      XtAppError(XtWidgetToApplicationContext(sheet),
                 "You can only add Widgets that are a"
                 " subclass of PropertyPage.");
      return;
   }
   else if(position < 0 || position > (page_count + 1))
   {
      position = 0;
   }

   /* 
    * Get the dialog title resource from the page widget 
    * and use this as the label text for the push button.
    */

   XtVaGetValues(page, XmNdialogTitle, &tab_string, NULL);
   
   /* 
    * create a tab.
    * 
    * Use the title resource to get the text for the push button
    *
    * BTW, the creation of the tab button has to be done at this
    * time because we want the user to be able to add pages on the
    * fly.  Why this would be done I don't know but we will make 
    * an attempt at implementing this.
    */

   #if 0
   tab = XmCreatePushButton(sheet, PS_TAB_CN, NULL, 0);
   XtVaSetValues(tab, 
                 XmNlabelString, tab_string, 
                 XmNdefaultButtonShadowThickness, 0,
                 XmNhighlightThickness, 0,
                 XmNborderWidth, 0, 
                 XmNnavigationType, XmTAB_GROUP,
                 XmNfillOnArm, False,
                 NULL);
   #else
   tab = XmCreateTab(sheet, PS_TAB_CN, NULL, 0);
   XtVaSetValues(tab, 
                 XmNlabelString, tab_string, 
                 XmNhighlightThickness, 1,
                 #if 0
                 XmNnavigationType, XmTAB_GROUP,
                 #endif
                 NULL);
   #endif
   XmStringFree(tab_string);

   /* FIX ME: we need to position the tab.
    * Should we have a row column to handle positioning?
    * I think we need a resource to tell us how many tabs per row.
    * 
    * For the time being we will only support a single row.
    */

   /*  
    * Add the tab_activate_cb function for handling the activate callback. 
    * Also, the page widget must be sent as the user data for the call.
    */ 
   XtAddCallback(tab, XmNarmCallback, tab_activate_cb, page);

   if(page_count <= 0)
   {
      pages = (WidgetList) XtMalloc(sizeof(Widget));
      tabs = (WidgetList) XtMalloc(sizeof(Widget));
      pages[0] = page;
      tabs[0] = tab;
      page_count++;
   }
   /* if position is zero then add page to the end of the list. */
   else if(position == 0)
   {
      pages = (WidgetList) XtRealloc((char *)pages, 
                                     sizeof(Widget)*(page_count+1));
      tabs = (WidgetList) XtRealloc((char *)tabs, 
                                     sizeof(Widget)*(page_count+1));
      pages[page_count] = page;
      tabs[page_count] = tab;
      page_count++;
   }
   /* insert page widget into list */
   else
   {
      /* use memmove because memory will overlap */
      pages = (WidgetList) XtRealloc((char *) pages, 
                                     sizeof(Widget)*(page_count+1));
      memmove(/* dest */ &pages[position],
              /* src */ &pages[position-1], 
              (sizeof(Widget)*((page_count+1)-position)));

      tabs = (WidgetList) XtRealloc((char *) tabs, 
                                     sizeof(Widget)*(page_count+1));
      memmove(/* dest */ &tabs[position],
              /* src */ &tabs[position-1], 
              (sizeof(Widget)*((page_count+1)-position)));

      pages[position-1] = page;
      tabs[position-1] = tab;
      page_count++;
   }

   /*
    * Reset lists
    */
   PS_Tabs(sheet) = tabs;
   PS_Pages(sheet) = pages;
   PS_PageCount(sheet) = page_count;

   /*
    * Make tab visible.
    */
   XtManageChild(tab);
}

/*
 * XmPropertySheetRemovePos
 * 
 * This function removes the page widget from the sheet widget.
 * This function does not delete the page widget.
 */
void
XmPropertySheetRemovePos(Widget sheet, int position)
{
   XmPropertySheetWidget ps = (XmPropertySheetWidget)sheet;
   Widget page, tab;
   WidgetList pages, tabs;
   int page_count;

   pages = ps->property_sheet.pages;
   tabs = ps->property_sheet.tabs;
   page_count = ps->property_sheet.page_count;

   if(position > page_count
      || position <= 0)
   {
      return; /* not in list */
   }

   page = pages[position - 1];
   tab = tabs[position - 1];

   /* adjust list by shifting everything down one in the list */
   memmove(/* dest */ &tabs[position-1],
           /* src */ &tabs[position], 
           (sizeof(Widget)*(page_count-position)));

   page_count--;

   ps->property_sheet.tabs = tabs;
   ps->property_sheet.pages = pages;
   ps->property_sheet.page_count = page_count;

   remove_tab(tab);
   remove_page(page);

   /* was it the active page */
   if (position == ps->property_sheet.active_pos)
   {
      /* 
       * Even though its the active page we are not going to initiate
       * the kill_activate callback  because the user did not select a
       * new tab.  The programmer actually killed it.
       */
      /*
       * At this point it should be up to the programmer to set the new active 
       * page.
       */
      ps->property_sheet.active_pos = 0;
   }
}

/*
 * XmPropertySheetRemovePage
 * 
 * This function actually searches the pages for the specified widget 
 * to determine the pos.  After the position is determined 
 * the XmPropertySheetRemovePos function is called.
 */
void
XmPropertySheetRemovePage(Widget sheet, Widget page)
{
   int pos;
   pos = page_position(sheet, page);
   XmPropertySheetRemovePos(sheet, pos);
}

#if 0
void 
XmPropertySheetSetActivePage(Widget sheet, Widget page, Boolean notify)
{
   int pos;
   pos = page_position(sheet, page);
   XmPropertySheetSetActivePos(sheet, pos, notify);
}
#endif

void 
XmPropertySheetSetActivePos(Widget sheet, int position, Boolean notify)
{
   Widget new_page, old_page;

   if(position > PS_PageCount(sheet) || position <= 0)
   {
      return;
   }
   
   old_page = PS_ActivePage(sheet);
   set_active_pos(sheet, position, notify);
   new_page = PS_ActivePage(sheet);

   display_page(sheet, new_page, old_page);
}

