/* $Id: xrun.C,v 1.3 1998/04/18 06:16:07 glgay Exp $ */
/*
 Copyright (C) 1995, 1996 Peter Williams
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License 
 version 2 as published by the Free Software Foundation.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave, 
 Cambridge, MA 02139, USA.
*/

/* Include(s) */
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <Xarm/Label.h>
#include <Xarm/PushB.h>
#include <Xarm/MessageB.h>
#include "xrun.h"

/* Declaration(s) */

#if !defined(VERSION)
#define VERSION 1
#endif
#if !defined(PATCHLEVEL)
#define PATCHLEVEL 0
#endif
#if !defined(SUBLEVEL)
#define SUBLEVEL 1
#endif

#if !defined(XRUN_VERSION)
#define makestring(str) #str
#define xmakestring(str) makestring(str)
#define XRUN_VERSION xmakestring(VERSION) "." xmakestring(PATCHLEVEL) "." xmakestring(SUBLEVEL)
#endif
#if !defined(XRUN_MODULE_NAME)
#define XRUN_MODULE_NAME "xrun"
#endif
#if !defined(XRUN_COPYRIGHT)
#define XRUN_COPYRIGHT "Copyright (c) 1995, 1996 by Peter G. Williams "
#endif

/*
 * Default locations of various system programs.
 * Change these as needed.
 */
#if !defined(DFL_SHELL_PATH)
#define DFL_SHELL_PATH "/bin/sh"
#endif

/* 
 * This is probable over kill but will keep me from 
 * making typos late at night..
 */
#define BUTTON_OK_CN "OK"
#define BUTTON_CANCEL_CN "Cancel"
#define BUTTON_BROWSE_CN "Browse"
#define LABEL_COMMAND_CN "Command"

/* Important Note:
 *
 *    For languages other then English modify all the labelString
 * resources here or in the resource file in the main app-defaults 
 * directory.  We can support foreign languages that easily.
 */

/* fallback resources, 
 *
 * The initial app-defaults file should be similar to this.
 *
 */
_XtString fallbacks[] = 
{
   "! " XRUN_MODULE_NAME " resources",
   "! Version " XRUN_VERSION,
   "! " XRUN_COPYRIGHT, 
   "! This resource file was generated by " XRUN_MODULE_NAME ".",
   "! ",
   "*Background: grey",
   "*Foreground: black",
   "*backgroundPixmap: a036.xpm",
   "*HighlightColor: black",
   "*fontList: -*-helvetica-medium-r-*--12-120-*",
   "*XmTextField.Background: white",
   "*" ".dialogTitle: Run Program",
   "*" BUTTON_OK_CN ".labelString: OK",
   "*" BUTTON_CANCEL_CN ".labelString: Cancel",
   "*" BUTTON_BROWSE_CN ".labelString: Browse...",
   "*" LABEL_COMMAND_CN ".labelString: Command Line:",
   "*programShell: " DFL_SHELL_PATH,
   NULL
};

// extra command line options
static XrmOptionDescRec options [] =
{
   { "-g", "*generateResourceFile",  XrmoptionSepArg, NULL },
   { "--generate-resource=", "*generateResourceFile",  XrmoptionSepArg, NULL },
   { "--help",  "*showUsage",  XrmoptionNoArg, "True" }, 
   { "-h",     "*showUsage",  XrmoptionNoArg, "True" },
   { "--program-shell=",     "*programShell",  XrmoptionSepArg, NULL },
   { "-s",     "*programShell",  XrmoptionSepArg, NULL },
   { "--version",     "*showVersion",  XrmoptionNoArg, "True" },
   { "-v",     "*showVersion",  XrmoptionNoArg, "True" },
};

#define XtNgenerateResourceFile "generateResourceFile"
#define XtCgenerateResourceFile "GenerateResourceFile"
#define XtNshowUsage "showUsage"
#define XtCshowUsage "ShowUsage"
#define XtNshowVersion "showVersion"
#define XtCshowVersion "ShowVersion"
#define XtNprogramShell "programShell"
#define XtCprogramShell "ProgramShell"

static XtResource resources[] = 
{
  {
     XtNgenerateResourceFile, 
     XtCgenerateResourceFile,
     XtRString, 
     sizeof(char *), 
     XtOffsetOf(Application, resource_file),
     XtRString, 
     (XtPointer) NULL
  },

  {
     XtNshowUsage, 
     XtCshowUsage,
     XtRBoolean, 
     sizeof(Boolean), 
     XtOffsetOf(Application, show_usage),
     XtRImmediate, 
     (XtPointer) False
  },

  {
     XtNshowVersion, 
     XtCshowVersion,
     XtRBoolean, 
     sizeof(Boolean), 
     XtOffsetOf(Application, show_version),
     XtRImmediate, 
     (XtPointer) False
  },

  {
     XtNprogramShell, 
     XtCprogramShell,
     XtRString, 
     sizeof(char *), 
     XtOffsetOf(Application, program_shell),
     XtRString, 
     (XtPointer) DFL_SHELL_PATH
  },

};

// =========================================================================
// functions
int
main (int argc, char **argv)
{
   // initialize application 
   Application app(Application::basename(argv[0]),argc,argv);

   // realize application
   app.realize();

   // start the main loop 
   app.mainLoop();

   // exit code just in case we get here.
   exit(EXIT_SUCCESS);
   return EXIT_SUCCESS;  /* SOME compilers like a return statement */
}

// ======================================================================= 
// Application member functions 

Application::Application(char *app_class, int &argc_in_out, char **argv_in_out):
   AppContext(app_class, options, XtNumber(options), argc_in_out, argv_in_out, 
              fallbacks)
{
   // check to see if any args are left over
   if(argc_in_out > 1)
   {
      syntax(argc_in_out, argv_in_out);
   }

   // process application resources
   process_resources();

   // set x connection to close on exec
   if(fcntl(ConnectionNumber(display()), F_SETFD, 1) == -1) 
   {
      this->error("unable to mark display connection as close on exec");   
      exit(EXIT_FAILURE);
   }

   // install window manager message handler(s)
   Atom proto = addWMProtocol(XA_WM_DELETE_WINDOW);
   addWMProtocolCallback(widget(),
                         proto,
                         (p_msg)&Application::Quit); 

   // set title
   title(app_class);

   // create 
   p_rundlg = new RunDialog(this->widget());
   p_rundlg->shell(shell());

   addCallback(p_rundlg->cancelButton(),
               XmNactivateCallback,
               (p_msg)&Application::Quit); 
   addCallback(p_rundlg->defaultButton(),
               XmNactivateCallback,
               (p_msg)&Application::Quit); 

   p_rundlg->manage();
}

void
Application::version()
{
   /* 
    * I have a working version of iostreams now but I'm too lazy to use it.
    * Hmm, maybe a xmostreams class is needed.  Sure sounds ugly or maybe not.
    */
   printf("%s version %s\n", (argv())[0], XRUN_VERSION);
   quit();
}

void
Application::usage()
{
   printf("Usage: %s [OPTION]\n", (argv())[0]);
   printf("  -g FILE  --generate-resource=FILE  Create resource file as FILE.\n");
   printf("  -h  --help  Print this message.\n");
   printf("  -s  --program-shell=PROGRAM  Name of shell program [%s].\n", program_shell );
   printf("  -v  --version  Output version info.\n");
   quit();
}

void
Application::process_resources()
{
   get_resources();

   // checked to see if the user wants the version info
   if(show_version == TRUE)
   {
      version();
   }
   // checked to see if the user wants a usage message
   if(show_usage == TRUE)
   {
      usage();
   }
   // checked to see if the user gave us a file name
   if(resource_file != NULL)
   {
      generate_resource_file(resource_file);
   }
}

void
Application::get_resources()
{
   XtGetApplicationResources(widget(), 
                             this, 
                             resources,
                             XtNumber(resources),
                             NULL,
                             0);
}

void
Application::generate_resource_file(char *fname)
{
   // I should use iostreams instead
   FILE *fp = fopen(fname,"w");
   if(fp != NULL)
   {
      // dump contents of fallbacks to the resource file
      for (int i=0; fallbacks[i] != NULL; i++)
      {
         fprintf(fp, "%s\n",fallbacks[i]);
      } 
      fclose(fp);
   }
   quit();
}

void
Application::syntax(int opt_no, char **cmd_opts)
{
   for (int i=1; i< opt_no; i++) 
   {
      fprintf(stderr,"Invalid option: %s\n", cmd_opts[i]);
   }
   quit();
}

char *
Application::basename(char *name)
{
   char *s;

   if((s=strrchr(name,'/'))!=NULL)
   {
      s++;
   }
   else
   {
      s = name;
   }	
   return s;
}

void 
Application::Quit(Widget w, XtPointer udata, XtPointer cdata)
{
   w = w;
   cdata = cdata;
   udata = udata;

   quit();
}

// ======================================================================= 
// RunDialog member functions 

RunDialog::RunDialog(_XtString name, Widget parent):
           BulletinBoard(name, parent) { }

RunDialog::RunDialog(Widget parent, ArgList arglist, Cardinal cnt):
                     BulletinBoard(parent, arglist, cnt)
{
   Label *p_label;
   
   // create child widgets
   p_label = new Label(LABEL_COMMAND_CN, this->widget());
   p_entry = new TextField(this->widget());
   p_entry->setMaxLength(256);
   p_entry->columns(40);
   p_entry->x(p_label->x());
   p_entry->y(p_label->height() + 10);

   p_ok = new PushButton(BUTTON_OK_CN, this->widget());
   p_ok->autoSpace(85);
   p_ok->x(p_entry->x());
   p_ok->y(p_entry->y() + p_entry->height() + 10);
   addCallback(p_ok->widget(), XmNactivateCallback, (p_msg) &RunDialog::Ok);

   p_cancel = new PushButton(BUTTON_CANCEL_CN, this->widget());
   p_cancel->autoSpace(85);
   p_cancel->x(p_ok->x() + p_ok->width() + 25);
   p_cancel->y(p_ok->y());
   addCallback(p_cancel->widget(), XmNactivateCallback, (p_msg) &RunDialog::Cancel);

   p_browse = new PushButton(BUTTON_BROWSE_CN, this->widget());
   p_browse->autoSpace(85);
   p_browse->x(p_cancel->x() + p_cancel->width() + 15);
   p_browse->y(p_cancel->y());
   p_browse->removeAllCallbacks(XmNactivateCallback);
   addCallback(p_browse->widget(), XmNactivateCallback, (p_msg) &RunDialog::Browse);

   // dialogTitle("Run Program");
   cancelButton(p_cancel->widget());
   defaultButton(p_ok->widget());
   p_ok->showAsDefault();

   // setup the file selection dialog for the browse button 
   p_fsdlg = new FileSelectionDialog(this->widget());
   p_fsdlg->okCallback((XtCallbackProc) XtUnmanageChild);
   p_fsdlg->cancelCallback((XtCallbackProc) XtUnmanageChild);
   addCallback(p_fsdlg->widget(), XmNokCallback, (p_msg) &RunDialog::SetEntry);
}

void
RunDialog::SetEntry(Widget w, XtPointer udata, XtPointer cdata)
{
   w = w;
   cdata = cdata;
   udata = udata; 

   char *str = p_fsdlg->textString();
   p_entry->value(str);
   XtFree(str);
}

void
RunDialog::Browse(Widget w, XtPointer udata, XtPointer cdata)
{
   w = w;
   cdata = cdata;
   udata = udata; 

   // display the file selection dialog
   p_fsdlg->manage();
}

void
RunDialog::Cancel(Widget w, XtPointer udata, XtPointer cdata)
{
   w = w;
   cdata = cdata;
   udata = udata; 
   // do nothing
}

void
RunDialog::Ok(Widget w, XtPointer udata, XtPointer cdata)
{
   w = w;
   cdata = cdata;
   udata = udata; 

   // grab text field value
   _XtString val = p_entry->value();
   if (val != NULL && *val != '\0')
   {
      execute(val);
   }
}

/*
 * execute()
 *
 * This function executes the command line val in a shell.
 * The shell is spawn like a daemon so zombies don't crop up.
 * This way the shell does all the regular expression eval.
 *
 * RETURNS: 0 if everything went ok or -1 on error.
 */
int
RunDialog::execute(char *val)
{
   char *path = shell();
   char *args[4];
   args[0] = shell();
   args[1] = "-c";
   args[2] = val;
   args[3] = NULL;

   return execute(path, args);
}

int
RunDialog::execute_and_wait(char *path, char **args)
{
   pid_t pid;
   int status = 0;

   pid = fork();
   switch(pid)
   {
      case -1:
         // well this is interesting...maybe the system is going to 
         // crash soon. 
         status = -1;
         break;
      case 0:
         execvp(path, args);
         exit(EXIT_FAILURE);
      default:
         // parent process waits for return status
         while(waitpid(pid, &status, 0) < 0)
         {
            if(errno != EINTR)
            {
               status = -1;
               break;
            }
         }
            
         if(WIFEXITED(status))
         {
            status = WEXITSTATUS(status);
         }
         else if(WIFSIGNALED(status))
         {
            status = WTERMSIG(status);
         }
         else
         {
            status = -1;
         }
         break;
   }
   return status;
}

int
RunDialog::execute(char *path, char **args)
{
   pid_t pid;
   int status = 0;

   /*
    * fork and then execute command string
    * we should make child a group leader because
    * we don't care about the return code.
    */

   pid = fork();
   switch(pid)
   {
      case -1:
         // well this is interesting...maybe the system is going to 
         // crash soon. 
         status = -1;
         break;
      case 0:
         // child process

         // now lets do a typical daemon init
         pid = fork();
         if(pid == -1)
         {
            exit(EXIT_FAILURE);
         }
         else if(pid == 0)
         {
            // make process leader again.
            setsid();
            execvp(path, args);
            exit(EXIT_FAILURE);
         }
         // parent terminates
         exit(EXIT_SUCCESS);
      default:
         // parent process
         while(waitpid(pid, &status, 0) < 0)
         {
            if(errno != EINTR)
            {
               status = -1;
               break;
            }
         }
         break;
   }
   return status;
}

