#include <stdio.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/XmP.h>

#include <Xmext/Gauge.h>
#include <Xmext/GaugeP.h>

#if XmVersion > 1001
#include <Xm/DrawP.h>
#endif

#define SWAP(a, b) { int z; z = (a); (a) = (b); (b) = z; }
#ifndef MyMax
#define MyMax(a,b) ((a) > (b) ? (a) : (b))
#endif

/*  Static routine definitions  */

static void    Initialize(XmGaugeWidget request, XmGaugeWidget new);
static void    Realize(XmGaugeWidget gaugew, Mask *window_mask, XSetWindowAttributes *window_attributes);
static void    ClassPartInitialize(WidgetClass wc);
static void    Redisplay(XmGaugeWidget gaugew, XEvent *event, Region region);
static void    Resize(XmGaugeWidget gaugew);
static void    Destroy(XmGaugeWidget gaugew);
static Boolean SetValues(XmGaugeWidget current, XmGaugeWidget request, XmGaugeWidget new);
static void    GetGaugeGC(XmGaugeWidget gaugew);
static void    GetSliderPixmap(XmGaugeWidget gaugew);
static void    LoadFontMetrics(XmGaugeWidget gaugew);
static void    CalcSliderRect(XmGaugeWidget gaugew, short int *slider_x, short int *slider_y, short int *slider_width, short int *slider_height);
static void    DisplayValue(XmGaugeWidget gaugew);
static void    DrawGauge(XmGaugeWidget gaugew);
static void    RedrawSlider(XmGaugeWidget gaugew);
static void    ForegroundPixelDefault(XmGaugeWidget widget, int offset, XrmValue *value);
static void    TroughPixelDefault(XmGaugeWidget widget, int offset, XrmValue *value);
static void    SliderPixelDefault(XmGaugeWidget widget, int offset, XrmValue *value);
static void    ProcessingDirectionDefault(XmGaugeWidget widget, int offset, XrmValue *value);

/*  Resource list for Gauge  */

static XtResource resources[] = 
{
    { XmNforeground, XmCForeground, XmRPixel, sizeof(Pixel),
      XtOffset(XmGaugeWidget, primitive.foreground),
      XmRCallProc, (caddr_t) ForegroundPixelDefault
    },

    { XmNtroughColor, XmCTroughColor, XmRPixel, sizeof(Pixel),
      XtOffset(XmGaugeWidget, gauge.trough_color),
      XmRCallProc, (caddr_t) TroughPixelDefault
    },

    { XmNsliderColor, XmCSliderColor, XmRPixel, sizeof(Pixel),
      XtOffset(XmGaugeWidget, gauge.slider_color),
      XmRCallProc, (caddr_t) SliderPixelDefault
    },

   {
      XmNfontList, XmCFontList, XmRFontList, sizeof(XmFontList),
      XtOffset(XmGaugeWidget, gauge.fontlist),
      XmRString, (caddr_t) "fixed"
    },

   {
     XmNvalue, XmCValue, XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.value),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNsliderShadowThickness, XmCSliderShadowThickness,
     XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.slider_shadow_thickness),
     XmRImmediate, (caddr_t) 1
   },

   {
     XmNtroughShadowThickness, XmCTroughShadowThickness,
     XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.trough_shadow_thickness),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNminimum, XmCMinimum, XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.minimum),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNmaximum, XmCMaximum, XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.maximum),
     XmRImmediate, (caddr_t) 100 
   },

   {
     XmNorientation, XmCOrientation, XmROrientation, sizeof(unsigned char),
     XtOffset (XmGaugeWidget, gauge.orientation),
     XmRImmediate, (caddr_t) XmVERTICAL 
   },

   {
     XmNprocessingDirection, XmCProcessingDirection, 
     XmRProcessingDirection, sizeof(unsigned char),
     XtOffset (XmGaugeWidget, gauge.processing_direction),
     XmRCallProc, (caddr_t) ProcessingDirectionDefault 
   },

   {
     XmNtickIncrement, XmCTickIncrement, XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.tick_increment),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNshowValue, XmCShowValue, XmRBoolean, sizeof(Boolean),
     XtOffset (XmGaugeWidget, gauge.show_value),
     XmRImmediate, (caddr_t) False
    },

   {
     XmNshowMinMax, XmCShowMinMax, XmRBoolean, sizeof(Boolean),
     XtOffset (XmGaugeWidget, gauge.show_min_max),
     XmRImmediate, (caddr_t) False
    },

   {
     XmNshowPercentage, XmCShowPercentage, XmRBoolean, sizeof(Boolean),
     XtOffset (XmGaugeWidget, gauge.show_percentage),
     XmRImmediate, (caddr_t) False
    },

   {
     XmNgaugeMargin, XmCGaugeMargin, XmRInt, sizeof(int),
     XtOffset (XmGaugeWidget, gauge.gauge_margin),
     XmRImmediate, (caddr_t) 7
   },

   {
     XmNfill, XmCFill, XmRBoolean, sizeof(Boolean),
     XtOffset (XmGaugeWidget, gauge.fill),
     XmRImmediate, (caddr_t) True
   },

   {
     XmNsliderPixmap, XmCSliderPixmap, XmRPixmap, sizeof(Pixmap),
     XtOffset (XmGaugeWidget, gauge.slider_pixmap),
     XmRImmediate, (caddr_t) 0
   },

};


/*  The Gauge class record definition  */

externaldef (xmgaugeclassrec) XmGaugeClassRec xmGaugeClassRec=
{
   {
      (WidgetClass) &xmPrimitiveClassRec,      /* superclass            */    
      "XmGauge",                               /* class_name            */    
      sizeof(XmGaugeRec),                      /* widget_size           */    
      NULL,                                    /* class_initialize      */    
      ClassPartInitialize,                     /* class_part_initialize */
      FALSE,                                   /* class_inited          */    
      (XtInitProc) Initialize,                 /* initialize            */    
      NULL,                                    /* initialize_hook       */
      (XtRealizeProc) Realize,                 /* realize               */    
      NULL,                                    /* actions               */    
      0,                                       /* num_actions           */    
      resources,                               /* resources             */    
      XtNumber(resources),                     /* num_resources         */    
      NULLQUARK,                               /* xrm_class             */    
      TRUE,                                    /* compress_motion       */    
      XtExposeCompressMaximal,                 /* compress_exposure     */    
      TRUE,                                    /* compress_enterleave   */
      FALSE,                                   /* visible_interest      */    
      (XtWidgetProc) Destroy,                  /* destroy               */    
      (XtWidgetProc) Resize,                   /* resize                */
      (XtExposeProc) Redisplay,                /* expose                */    
      (XtSetValuesFunc) SetValues,             /* set_values            */    
      NULL,                                    /* set_values_hook       */
      (XtAlmostProc) XtInheritSetValuesAlmost, /* set_values_almost     */
      NULL,                                    /* get_values_hook       */
      NULL,                                    /* accept_focus          */    
      XtVersion,                               /* version               */
      NULL,                                    /* callback private      */
      NULL,                                    /* tm_table              */
      NULL,                                    /* query_geometry        */
      NULL,                                    /* display_accelerator   */
      NULL,                                    /* extension             */
   },

   {
      (XtWidgetProc)_XtInherit,               /* Primitive border_highlight   */
      (XtWidgetProc)_XtInherit,               /* Primitive border_unhighlight */
      NULL,                                   /* translations                 */
      NULL,                                   /* arm_and_activate             */
      NULL,                                   /* get resources                */
      0,                                      /* num get_resources            */
      NULL,                                   /* extension                    */
   }

};

externaldef(xmgaugewidgetclass) WidgetClass xmGaugeWidgetClass =
              (WidgetClass) &xmGaugeClassRec;


/************************************************************************
 *
 *  ClassPartInitialize
 *     Set up the fast subclassing for the widget
 *
 ************************************************************************/
static void ClassPartInitialize (WidgetClass wc)
{

}

/*********************************************************************
 *
 * ForegroundPixelDefault
 *    This procedure provides the dynamic default behavior for
 *    the foreground color.
 *
 *********************************************************************/
static void ForegroundPixelDefault (XmGaugeWidget widget, int offset, XrmValue *value)
{
    static Pixel foreground;

    value->addr = (caddr_t) &foreground;
    foreground = widget->core.background_pixel;
}


/*********************************************************************
 *
 * TroughPixelDefault
 *    This procedure provides the dynamic default behavior for
 *    the trough color.
 *
 *********************************************************************/
static void TroughPixelDefault (XmGaugeWidget widget, int offset, XrmValue *value)
{
    static Pixel trough;
    XmColorData *pixel_data;

    value->addr = (caddr_t) &trough;

    pixel_data = _XmGetColors(XtScreen((Widget)widget),
        widget->core.colormap, widget->core.background_pixel);

    trough = _XmAccessColorData(pixel_data, XmSELECT);
}

/*********************************************************************
 *
 * SliderPixelDefault
 *    This procedure provides the dynamic default behavior for
 *    the slider color.
 *
 *********************************************************************/
static void SliderPixelDefault (XmGaugeWidget widget, int offset, XrmValue *value)
{
    static Pixel slider;

    value->addr = (caddr_t) &slider;
    XtVaGetValues((Widget)widget, XmNforeground, &slider, NULL);
}

/*********************************************************************
 *
 * ProcessingDirectionDefault
 *    This procedure provides the dynamic default behavior for
 *    the processing direction resource dependent on the orientation.
 *
 *********************************************************************/
static void ProcessingDirectionDefault (XmGaugeWidget widget, int offset, XrmValue *value)
{
    static unsigned char direction;

    value->addr = (caddr_t) &direction;

    if (widget->gauge.orientation == XmHORIZONTAL)
        direction = XmMAX_ON_RIGHT;
    else /* XmVERTICAL  -- range checking done during widget
                           initialization */
        direction = XmMAX_ON_TOP;
}


/************************************************************************
 *
 *  Initialize
 *     The main widget instance initialization routine.
 *
 ************************************************************************/
static void Initialize (XmGaugeWidget request, XmGaugeWidget new)
{

   /*
    *  Check the data put into the new widget from .Xdefaults
    *  or through the arg list.
    */

    if (new->gauge.minimum >= new->gauge.maximum)
    {
          _XmWarning ((Widget)new, "XmNminimum is greater than XmNmaximum");
        new->gauge.minimum = 0;
    }

    if (new->gauge.minimum >= new->gauge.maximum)
    {
        new->gauge.maximum = 100;
    }

    if (new->gauge.value < new->gauge.minimum)
    {
          _XmWarning ((Widget)new, "XmNvalue is less than XmNminimum");
        new->gauge.value = new->gauge.minimum;
    }

    if (new->gauge.value > new->gauge.maximum)
    {
        _XmWarning ((Widget)new, "XmNvalue is greater than XmNmaximum");
        new->gauge.value = new->gauge.maximum;
    }

    if (new->gauge.tick_increment >  new->gauge.maximum - new->gauge.minimum)
    {
        _XmWarning ((Widget)new, "XmNtickIncremnt is greater than Gauge range");    
        new->gauge.tick_increment = 0;
    }

    if (new->gauge.slider_shadow_thickness < 1) {
        new->gauge.slider_shadow_thickness = 1;
    }

    if ((new->gauge.orientation != XmHORIZONTAL) &&
        (new->gauge.orientation != XmVERTICAL))
    {
        new->gauge.orientation = XmVERTICAL;
        _XmWarning((Widget)new, "Invalid orientation.");
    }

    if (new->gauge.orientation == XmHORIZONTAL)
    {
        if ((new->gauge.processing_direction != XmMAX_ON_RIGHT) &&
            (new->gauge.processing_direction != XmMAX_ON_LEFT))

        {
            new->gauge.processing_direction = XmMAX_ON_RIGHT;
            _XmWarning ((Widget)new, "Invalid Processing Direction");
        }
    }
    else
    {
        if ((new->gauge.processing_direction != XmMAX_ON_TOP) &&
            (new->gauge.processing_direction != XmMAX_ON_BOTTOM))
        {
            new->gauge.processing_direction = XmMAX_ON_TOP;
            _XmWarning ((Widget)new, "Invalid Processing Direction");
        }
    }

    /*  Set up a geometry for the widget if it is currently 0.  */

    if (request->core.width == 0)
    {
        if (new->gauge.orientation == XmHORIZONTAL)
             new->core.width += 200;
        else
             new->core.width += 90;

    }
    if (request->core.height == 0)
    {
        if (new->gauge.orientation == XmHORIZONTAL)
             new->core.height += 90;
        else
             new->core.height += 200;
    }

    new->gauge.pixmap = 0;
    new->gauge.color_thresholds = NULL;
    new->gauge.fontlist = XmFontListCopy(new->gauge.fontlist);
    LoadFontMetrics(new);
    new->gauge.value_x = 0;
    new->gauge.value_y = 0;
    new->gauge.prev_value = 0;
    new->gauge.prev_percentage = 0;
    new->gauge.percentage = 0;

    /*  Get the drawing graphics contexts.  */

    GetGaugeGC(new);

    /*  Set the internal gauge variables and call resize to generate  */
    /*  the gauge drawing variable set.                               */

    Resize(new);
}


/************************************************************************
 *
 *  GetGaugeGC
 *     Get the graphics context used for drawing the gauge.
 *
 ************************************************************************/
static void GetGaugeGC (XmGaugeWidget gaugew)
{
    XGCValues values;
    XtGCMask  valueMask = GCForeground | GCBackground | GCFillStyle | GCFont;

    values.foreground = gaugew->primitive.foreground;
    values.background = gaugew->core.background_pixel;
    values.fill_style = FillSolid;
    values.font = gaugew->gauge.font->fid;
    gaugew->gauge.background_GC = XtGetGC((Widget) gaugew, valueMask, &values);

    values.foreground = gaugew->gauge.slider_color;
    values.background = gaugew->primitive.foreground;
    gaugew->gauge.foreground_GC = XtGetGC((Widget) gaugew, valueMask, &values);

    values.foreground = gaugew->primitive.foreground;
    values.background = gaugew->gauge.trough_color;
    gaugew->gauge.text_GC = XtGetGC((Widget) gaugew, valueMask, &values);

    values.foreground = gaugew->gauge.trough_color;
    values.background = gaugew->primitive.foreground;
    gaugew->gauge.text_erase_GC = XtGetGC((Widget) gaugew, valueMask, &values);

    if (gaugew->gauge.slider_pixmap != 0) {
        values.fill_style = FillTiled; 
    }
    values.foreground = gaugew->gauge.slider_color;
    values.background = gaugew->primitive.foreground;
    gaugew->gauge.slider_GC =  XtGetGC((Widget) gaugew, valueMask, &values);
    if (gaugew->gauge.slider_pixmap != 0) {
        XSetTile(XtDisplay((Widget)gaugew), gaugew->gauge.slider_GC,
                 gaugew->gauge.slider_pixmap);
    }
    
}


/************************************************************************
 *
 *  Redisplay
 *     General redisplay function called on exposure events.
 *
 ************************************************************************/
/* ARGSUSED */
static void Redisplay (XmGaugeWidget gaugew, XEvent *event, Region region)
{

    /*
    ** Create the slider pixmap, if we have not already.
    */
    if (gaugew->gauge.pixmap == 0)
        GetSliderPixmap(gaugew);

    /*
    ** Draw widget shadow if needed.
    */
    if (gaugew->primitive.shadow_thickness)
#if XmVersion == 1001
         _XmDrawShadow(XtDisplay((Widget) gaugew), XtWindow((Widget) gaugew), 
                gaugew->primitive.bottom_shadow_GC,
                gaugew->primitive.top_shadow_GC,
                gaugew->primitive.shadow_thickness,
                gaugew->primitive.highlight_thickness,
                gaugew->primitive.highlight_thickness,
                gaugew->core.width-2 * gaugew->primitive.highlight_thickness,
                gaugew->core.height-2 * gaugew->primitive.highlight_thickness);
#else
         _XmDrawShadows(XtDisplay((Widget) gaugew), XtWindow((Widget) gaugew), 
                gaugew->primitive.top_shadow_GC,
                gaugew->primitive.bottom_shadow_GC,
                gaugew->primitive.highlight_thickness,
                gaugew->primitive.highlight_thickness,
                gaugew->core.width-2 * gaugew->primitive.highlight_thickness,
                gaugew->core.height-2 * gaugew->primitive.highlight_thickness,
                gaugew->primitive.shadow_thickness,
                XmSHADOW_IN);
#endif

    /*
    ** Draw the Gauge widget.
    */
    DrawGauge(gaugew);

    /*
    ** Copy in the slider pixmap.
    */
    if (gaugew->gauge.pixmap)
        if (gaugew->gauge.slider_pixmap) 
            XCopyArea(XtDisplay((Widget) gaugew),
                 gaugew->gauge.pixmap, XtWindow((Widget) gaugew),
                 gaugew->gauge.slider_GC,
                 0, 0, gaugew->gauge.slider_width,
                 gaugew->gauge.slider_height,
                 gaugew->gauge.slider_x, gaugew->gauge.slider_y);
        else
            XCopyArea(XtDisplay((Widget) gaugew),
                 gaugew->gauge.pixmap, XtWindow((Widget) gaugew),
                 gaugew->gauge.foreground_GC,
                 0, 0, gaugew->gauge.slider_width,
                 gaugew->gauge.slider_height,
                 gaugew->gauge.slider_x, gaugew->gauge.slider_y);

    (*(((XmPrimitiveWidgetClass) XtClass(gaugew))
        ->primitive_class.border_unhighlight)) ((Widget)gaugew);
}

/************************************************************************
 *
 *  Resize
 *    Calculate the drawing rectangles and draw rectangles.
 *
 ************************************************************************/
static void Resize (XmGaugeWidget gaugew)
{
    int text_height;
    int    text_width;
    int    min_text_width;
    int    max_text_width;

    char min_string[256];
    char max_string[256];

    Dimension width  = gaugew->core.width;
    Dimension height = gaugew->core.height;

    int margin = gaugew->gauge.gauge_margin +
                 gaugew->primitive.highlight_thickness +
                 gaugew->primitive.shadow_thickness  +
                 gaugew->gauge.trough_shadow_thickness;

     text_height = gaugew->gauge.show_min_max ? gaugew->gauge.font->ascent + 
                  gaugew->gauge.font->descent : 0;

    sprintf(min_string, "%d", gaugew->gauge.minimum);
    sprintf(max_string, "%d", gaugew->gauge.maximum);

    min_text_width = XTextWidth(gaugew->gauge.font, min_string,
                            strlen(min_string));

    max_text_width = XTextWidth(gaugew->gauge.font, max_string,
                            strlen(max_string));

    text_width = MyMax(min_text_width, max_text_width);

    if (gaugew->gauge.show_percentage)
        text_width++;

    gaugew->gauge.slider_area_x = margin;
       gaugew->gauge.slider_area_y = margin ;
    gaugew->gauge.slider_area_width  = gaugew->core.width  - (2*margin);
    gaugew->gauge.slider_area_height = gaugew->core.height - (2*(margin-1));

    if (gaugew->gauge.orientation == XmHORIZONTAL)
    {
        gaugew->gauge.slider_area_height -= text_height + 2;
    
        if (gaugew->gauge.processing_direction == XmMAX_ON_RIGHT)
        {    
            gaugew->gauge.minimum_x = gaugew->gauge.slider_area_x;
            gaugew->gauge.maximum_x = width - max_text_width - margin;
        } 
        else
        {
            gaugew->gauge.minimum_x = width - min_text_width - margin;
            gaugew->gauge.maximum_x = gaugew->gauge.slider_area_x;
        }

        gaugew->gauge.minimum_y = height - margin;
        gaugew->gauge.maximum_y = height - margin;
        gaugew->gauge.value_y = gaugew->gauge.show_min_max ? height - margin :
            height / 2 + text_height / 2 + gaugew->gauge.font->descent;

    }
    else    
    {
        gaugew->gauge.slider_area_width -= text_width + 2;
        gaugew->gauge.minimum_x = margin + gaugew->gauge.slider_area_width + 2;
        gaugew->gauge.minimum_y = height - margin;
        gaugew->gauge.maximum_x = margin + gaugew->gauge.slider_area_width + 2;
        gaugew->gauge.maximum_y = margin + text_height;
        gaugew->gauge.value_y = height / 2 + text_height / 2 ;

        if (gaugew->gauge.processing_direction == XmMAX_ON_BOTTOM)
            SWAP(gaugew->gauge.minimum_y, gaugew->gauge.maximum_y);
    }

     if (XtIsRealized(gaugew))
    {
       if (gaugew->gauge.pixmap)
       {
          XFreePixmap(XtDisplay(gaugew), gaugew->gauge.pixmap);
          gaugew->gauge.pixmap = 0;
       }
       GetSliderPixmap(gaugew);
    }
    else
    {
        CalcSliderRect(gaugew, &(gaugew->gauge.slider_x),
            &(gaugew->gauge.slider_y), &(gaugew->gauge.slider_width),
            &(gaugew->gauge.slider_height));
        GetSliderPixmap(gaugew);
    }

    if ((gaugew->gauge.orientation == XmVERTICAL) &&
            (gaugew->gauge.slider_area_width <= 0))
        gaugew->gauge.slider_area_width = gaugew->gauge.slider_width;

    if ((gaugew->gauge.orientation == XmHORIZONTAL) &&
            (gaugew->gauge.slider_area_height <= 0))
        gaugew->gauge.slider_area_height = gaugew->gauge.slider_height;
}

/*********************************************************************
 *
 * Realize
 *
 ********************************************************************/
static void Realize (XmGaugeWidget gaugew, Mask *window_mask, XSetWindowAttributes *window_attributes)
{
    Pixel           color;
    ColorThreshold  p;
    XrmValue        from;
    XrmValue        to;

    (*window_mask) |= (CWBackPixel | CWBitGravity);
    window_attributes->background_pixel = gaugew->gauge.trough_color;
    window_attributes->bit_gravity = ForgetGravity;

    XtCreateWindow((Widget) gaugew, InputOutput, CopyFromParent, *window_mask,
        window_attributes);

    if (gaugew->gauge.color_thresholds)
        for (p = gaugew->gauge.color_thresholds; p; p = p->next)
        {
            from.addr = p->color_name;
            from.size = strlen(p->color_name);
            to.addr = (void *)&color;
            to.size = sizeof(color);
            if (!XtConvertAndStore((Widget) gaugew, XtRString, &from, XtRPixel, &to))
                XtAppError(XtWidgetToApplicationContext((Widget) gaugew),
                   "XmSetGaugeColorThreshold: String to Pixel conversion failed");
            else
                p->color = (*(Pixel *) to.addr);
        }            
}

/************************************************************************
 *
 *  Destroy
 *    Clean up allocated resources when the widget is destroyed.
 *
 ************************************************************************/
static void Destroy (XmGaugeWidget gaugew)
{
    ColorThreshold    p;
    ColorThreshold save;

    XmFontListFree(gaugew->gauge.fontlist);
    XtReleaseGC((Widget) gaugew, gaugew->gauge.foreground_GC);
    XtReleaseGC((Widget) gaugew, gaugew->gauge.background_GC);
    XtReleaseGC((Widget) gaugew, gaugew->gauge.text_GC);
    XtReleaseGC((Widget) gaugew, gaugew->gauge.text_erase_GC);
    XtReleaseGC((Widget) gaugew, gaugew->gauge.slider_GC);
    if (gaugew->gauge.pixmap != 0)
        XFreePixmap(XtDisplay((Widget) gaugew), gaugew->gauge.pixmap);

    for (p = gaugew->gauge.color_thresholds; p;)
    {
        save = p->next;
        XtFree(p->color_name);
        XtFree((char *) p);
        p = save;
    }
}


/************************************************************************
 *
 *  SetValues
 *
 ************************************************************************/
/* ARGSUSED */
static Boolean SetValues (XmGaugeWidget current, XmGaugeWidget request, XmGaugeWidget new)
{
    Boolean        redrawFlag = False;
    Boolean        new_threshold = False;
    Pixel          new_threshold_color;
    XGCValues      values;
    XtGCMask       valueMask = GCForeground | GCBackground | GCFillStyle | GCFont;
    ColorThreshold p;

    /*  Check the data put into the new widget.  */

    if (new->gauge.minimum >= new->gauge.maximum)
    {
          _XmWarning ((Widget)new, "XmNminimum is greater than XmNmaximum");
        new->gauge.minimum = 0;
    }

    if (new->gauge.minimum >= new->gauge.maximum)
    {
        new->gauge.maximum = 100;
    }

    if (new->gauge.value < new->gauge.minimum)
    {
          _XmWarning ((Widget)new, "XmNvalue is less than XmNminimum");
        new->gauge.value = new->gauge.minimum;
        new->gauge.percentage = 0;
    }

    if (new->gauge.value > new->gauge.maximum)
    {
          _XmWarning ((Widget)new, "XmNvalue is greater than XmNmaximum");
        new->gauge.value = new->gauge.maximum;
        new->gauge.percentage = 100;
    }

    if (new->gauge.tick_increment >  new->gauge.maximum - new->gauge.minimum)
    {
        _XmWarning ((Widget)new, "XmNtickIncremnt is greater than Gauge range");    
        new->gauge.tick_increment = 0;
    }
    if (new->gauge.slider_shadow_thickness < 1)
    {
        new->gauge.slider_shadow_thickness = 1;
    }
    if ((new->gauge.orientation != XmHORIZONTAL) &&
        (new->gauge.orientation != XmVERTICAL))
    {
        new->gauge.orientation = XmVERTICAL;
        _XmWarning((Widget)new, "Invalid orientation.");
    }

    if (new->gauge.orientation == XmHORIZONTAL)
    {
        if ((new->gauge.processing_direction != XmMAX_ON_RIGHT) &&
            (new->gauge.processing_direction != XmMAX_ON_LEFT))

        {
            new->gauge.processing_direction = XmMAX_ON_RIGHT;
            _XmWarning ((Widget)new, "Invalid Processing Direction");
        }
    }
    else
    {
        if ((new->gauge.processing_direction != XmMAX_ON_TOP) &&
            (new->gauge.processing_direction != XmMAX_ON_BOTTOM))
        {
            new->gauge.processing_direction = XmMAX_ON_TOP;
            _XmWarning ((Widget)new, "Invalid Processing Direction");
        }
    }

    if (new->gauge.value != current->gauge.value && 
        new->gauge.color_thresholds)
    {
        for (p = new->gauge.color_thresholds; p; p = p->next)
            if (new->gauge.value >= p->value)
            {
                new_threshold = True;
                new_threshold_color = p->color;
            }
    }

    /*  See if the GC's need to be regenerated and widget redrawn.  */

    if (new->core.background_pixel != current->core.background_pixel ||
        new->primitive.foreground  != current->primitive.foreground  ||
        new->gauge.fontlist        != current->gauge.fontlist        ||
        new->gauge.trough_color    != current->gauge.trough_color    ||
        new->gauge.slider_color    != current->gauge.slider_color    ||
        new->gauge.slider_pixmap   != current->gauge.slider_pixmap    )
    {
        if (new_threshold)
            new->core.background_pixel = new_threshold_color;

        if (new->gauge.fontlist != current->gauge.fontlist)
        {
            new->gauge.fontlist = XmFontListCopy(new->gauge.fontlist);
               LoadFontMetrics(new);
        }

          XtReleaseGC((Widget)new, new->gauge.foreground_GC);
          XtReleaseGC((Widget)new, new->gauge.background_GC);
          XtReleaseGC((Widget)new, new->gauge.text_GC);
          XtReleaseGC((Widget)new, new->gauge.text_erase_GC);
          XtReleaseGC((Widget)new, new->gauge.slider_GC);
          GetGaugeGC(new);
          redrawFlag = True;
    }

    if (new_threshold && !redrawFlag)
    {
        new->core.background_pixel = new_threshold_color;
         XtReleaseGC((Widget)new, new->gauge.foreground_GC);
        values.foreground = new->core.background_pixel;
           values.background = new->primitive.foreground;
        values.fill_style = FillSolid;
           values.font = new->gauge.font->fid;
           new->gauge.foreground_GC = 
                        XtGetGC((Widget)new, valueMask, &values);
    }

    if (new->gauge.slider_pixmap != current->gauge.slider_pixmap) 
        redrawFlag = True; 

    if (new->gauge.minimum                 != current->gauge.minimum              ||
        new->gauge.maximum                 != current->gauge.maximum              ||
        new->gauge.orientation             != current->gauge.orientation          ||
        new->gauge.processing_direction    != current->gauge.processing_direction ||
        new->gauge.tick_increment          != current->gauge.tick_increment       ||
        new->gauge.fill                    != current->gauge.fill                 ||
        new->gauge.show_value              != current->gauge.show_value           ||
        new->gauge.show_min_max            != current->gauge.show_min_max         ||
        new->gauge.gauge_margin            != current->gauge.gauge_margin         ||
        new->primitive.highlight_thickness != 
                           current->primitive.highlight_thickness                 ||
        new->primitive.shadow_thickness    !=  
                        current->primitive.shadow_thickness                       ||
        new->gauge.trough_shadow_thickness != 
                        current->gauge.trough_shadow_thickness                    ||
        new->gauge.slider_shadow_thickness != 
                        current->gauge.slider_shadow_thickness)
    {
        redrawFlag = True;
    }
       else if (new->gauge.value != current->gauge.value && !redrawFlag)
    {
          new->gauge.prev_value = current->gauge.value;
          new->gauge.prev_percentage = current->gauge.percentage;
        if (new->gauge.maximum && !new->gauge.minimum)
              new->gauge.percentage = (new->gauge.value * 100.0) / 
                (new->gauge.maximum + new->gauge.minimum);
        else
              new->gauge.percentage = (new->gauge.value * 100.0) / 
                (new->gauge.maximum - new->gauge.minimum);
        RedrawSlider(new);
    }
    return redrawFlag;
}


/************************************************************************
 *
 *  ClearSlider
 *  Calculate and clear the old slider.
 *
 ************************************************************************/
static void ClearSlider (XmGaugeWidget gaugew, int prev_slider_x, int prev_slider_y, int prev_slider_width, int prev_slider_height)
{
    int clear_x;
    int clear_y;
    int clear_width;
    int clear_height;

    if (gaugew->gauge.orientation == XmHORIZONTAL)
    {
        clear_y = prev_slider_y;
        clear_height = prev_slider_height;
        clear_width = prev_slider_width - gaugew->gauge.slider_width;
        if (gaugew->gauge.processing_direction == XmMAX_ON_LEFT)
            clear_x = prev_slider_x;
        else
            clear_x = gaugew->gauge.slider_x + (prev_slider_width - clear_width);
    }
    else
    {
        clear_x = prev_slider_x;
        clear_width = prev_slider_width;
        clear_height = prev_slider_height - gaugew->gauge.slider_height;
        if (gaugew->gauge.processing_direction == XmMAX_ON_TOP)
            clear_y = prev_slider_y;
        else
            clear_y = gaugew->gauge.slider_y + (prev_slider_height-clear_height);
    }

    XClearArea(XtDisplay((Widget) gaugew), XtWindow((Widget) gaugew),
               clear_x, clear_y, clear_width, clear_height, False);
}

/************************************************************************
 *
 *  RedrawSlider
 *  Clear, calculate, and redraw the slider.
 *
 ************************************************************************/
static void RedrawSlider (XmGaugeWidget gaugew)
{
    char      prev_value_string[256];
    int       prev_slider_x = gaugew->gauge.slider_x;
    int       prev_slider_y = gaugew->gauge.slider_y;
    int       prev_slider_width = gaugew->gauge.slider_width;
    int       prev_slider_height = gaugew->gauge.slider_height;
    Display  *display    = XtDisplay((Widget) gaugew);
    Window    window     = XtWindow((Widget) gaugew);


    CalcSliderRect(gaugew, &(gaugew->gauge.slider_x),
        &(gaugew->gauge.slider_y), &(gaugew->gauge.slider_width),
        &(gaugew->gauge.slider_height));

    if (XtIsRealized(gaugew))
    {
        if (!gaugew->gauge.show_percentage ||
            (gaugew->gauge.show_percentage && 
             (gaugew->gauge.prev_percentage != gaugew->gauge.percentage)))
        {
            if (gaugew->gauge.prev_value > gaugew->gauge.value)
            {
                ClearSlider(gaugew, prev_slider_x, prev_slider_y, 
                            prev_slider_width, prev_slider_height);
            }

            if (gaugew->gauge.pixmap != 0)
            {
                XFreePixmap(XtDisplay(gaugew), gaugew->gauge.pixmap);
                gaugew->gauge.pixmap = 0;
            }
            GetSliderPixmap(gaugew);

            /*
            ** Clear old text value.
            */
            if (gaugew->gauge.show_value)
            {
                if (gaugew->gauge.show_percentage)
                    sprintf(prev_value_string, "%d%%", 
                        gaugew->gauge.prev_percentage);
                else
                    sprintf(prev_value_string, "%d", gaugew->gauge.prev_value);

                XDrawString(display,
                    window,
                    gaugew->gauge.text_erase_GC,
                    gaugew->gauge.value_x,
                    gaugew->gauge.value_y,
                    prev_value_string,
                    strlen(prev_value_string));
            }

            if (gaugew->gauge.pixmap != 0)
                if (gaugew->gauge.slider_pixmap) 
                    XCopyArea(display, gaugew->gauge.pixmap, window,
                        gaugew->gauge.slider_GC,
                        0, 0, 
                        gaugew->gauge.slider_width, gaugew->gauge.slider_height,
                        gaugew->gauge.slider_x, gaugew->gauge.slider_y);
                else 
                    XCopyArea(display, gaugew->gauge.pixmap, window,
                        gaugew->gauge.foreground_GC,
                        0, 0, 
                        gaugew->gauge.slider_width, gaugew->gauge.slider_height,
                        gaugew->gauge.slider_x, gaugew->gauge.slider_y);
            DisplayValue(gaugew);
        }
    }
}

/************************************************************************
 *
 *  DisplayValue
 *  Display the value string.
 *
 ************************************************************************/
static void DisplayValue (XmGaugeWidget gaugew)
{
    Dimension width = gaugew->core.width;
    char      value_string[256];
    int       text_width;
    Display  *display    = XtDisplay((Widget) gaugew);
    Window    window     = XtWindow((Widget) gaugew);
    double    total;
    int       percent;

    if (gaugew->gauge.show_value)
    {
        if (gaugew->gauge.show_percentage)
        {
            if (gaugew->gauge.maximum && !gaugew->gauge.minimum)
            {
                total = gaugew->gauge.maximum + gaugew->gauge.minimum;
                percent = (gaugew->gauge.value / total) * 100.0;
            }
            else
            {
                total = gaugew->gauge.maximum - gaugew->gauge.minimum;
                percent = (gaugew->gauge.value / total) * 100.0;
            }
            sprintf(value_string, "%d%%", gaugew->gauge.percentage);
        }
        else
               sprintf(value_string, "%d", gaugew->gauge.value);

           text_width = XTextWidth(gaugew->gauge.font,
                               value_string, strlen(value_string));

        if (gaugew->gauge.orientation == XmHORIZONTAL)
            gaugew->gauge.value_x = width / 2 - text_width / 2;
        else if (gaugew->gauge.show_min_max)
            gaugew->gauge.value_x = gaugew->gauge.minimum_x;
        else
            gaugew->gauge.value_x = gaugew->gauge.slider_area_width / 2 +
                gaugew->gauge.slider_area_x - text_width / 2;

        if (gaugew->gauge.show_min_max)                        
               XDrawImageString(display,
                        window,
                           gaugew->gauge.text_GC,
                           gaugew->gauge.value_x,
                        gaugew->gauge.value_y,
                           value_string,
                           strlen(value_string));
        else
            XDrawString(display,
                        window,
                        gaugew->gauge.text_GC,
                        gaugew->gauge.value_x,
                        gaugew->gauge.value_y,
                        value_string,
                        strlen(value_string));
    }
}

/************************************************************************
 *
 *  XmCreateGauge
 *    Create an instance of an gauge and return the widget id.
 *
 ************************************************************************/
Widget XmCreateGauge (Widget parent, char *name, ArgList arglist, Cardinal argcount)
{
    return (XtCreateWidget(name, xmGaugeWidgetClass, parent, arglist, argcount));
}


/************************************************************************
 *
 *  LoadFontMetrics
 *    Load the font structure in the widget.    
 *
 ************************************************************************/
static void LoadFontMetrics (XmGaugeWidget gaugew)
{
    XmFontContext    context;
    XmStringCharSet  charset;
    XFontStruct     *font;

    if (!XmFontListInitFontContext(&context, gaugew->gauge.fontlist))
        _XmWarning((Widget) gaugew, "XmFontListInitFontContext Failed.");

    if (!XmFontListGetNextFont(context, &charset, &font))
        _XmWarning((Widget) gaugew, "XmFontListGetNextFont Failed.");

    gaugew->gauge.font = font;
    XtFree(charset);
    XmFontListFreeFontContext(context);
}

/************************************************************************
 *
 *  GetSliderPixmap
 *     Create the pixmap for the gauge and render the gauge.
 *
 ************************************************************************/
static void GetSliderPixmap (XmGaugeWidget gaugew)
{
    short slider_x;
    short slider_y;
    short slider_width;
    short slider_height;

    if (gaugew->gauge.value != gaugew->gauge.minimum)
    {
        CalcSliderRect(gaugew, &slider_x, &slider_y, 
                        &slider_width, &slider_height);

        gaugew->gauge.pixmap = XCreatePixmap(XtDisplay(gaugew), 
                RootWindowOfScreen(XtScreen(gaugew)),
                slider_width, slider_height, gaugew->core.depth);

        gaugew->gauge.slider_x = slider_x;
        gaugew->gauge.slider_y = slider_y;
        gaugew->gauge.slider_width = slider_width;
        gaugew->gauge.slider_height = slider_height;

        /*
        **   Draw the slider.
        */
        if (gaugew->gauge.slider_pixmap == 0)
            XFillRectangle(XtDisplay(gaugew), gaugew->gauge.pixmap,
                          gaugew->gauge.foreground_GC,
                          0, 0, gaugew->gauge.slider_width, 
                          gaugew->gauge.slider_height);
        else
            XFillRectangle(XtDisplay(gaugew), gaugew->gauge.pixmap,
                          gaugew->gauge.slider_GC,
                          0, 0, gaugew->gauge.slider_width, 
                          gaugew->gauge.slider_height);
    
#if XmVersion == 1001
        _XmDrawShadow(XtDisplay((Widget)gaugew), gaugew->gauge.pixmap,
                      gaugew->primitive.top_shadow_GC,
                      gaugew->primitive.bottom_shadow_GC,
                      gaugew->gauge.slider_shadow_thickness,
                      0, 0,
                      gaugew->gauge.slider_width, gaugew->gauge.slider_height);
#else
       _XmDrawShadows(XtDisplay((Widget)gaugew), gaugew->gauge.pixmap,
                      gaugew->primitive.top_shadow_GC,
                      gaugew->primitive.bottom_shadow_GC,
                      0, 0,
                      gaugew->gauge.slider_width, gaugew->gauge.slider_height,
                      gaugew->gauge.slider_shadow_thickness,
                      XmSHADOW_OUT);
#endif
    }
}


/************************************************************************
 *
 *  CalcSliderRect
 *     Calculate the slider location and size in pixels so that
 *     it can be drawn.
 *
 ************************************************************************/
static void CalcSliderRect (XmGaugeWidget gaugew, short int *slider_x, short int *slider_y, short int *slider_width, short int *slider_height)
{
    float userSize;
    float trueSize;
    float factor;
    float sliderSize;
    int   minSliderWidth;
    int   minSliderHeight;
    int   margin = gaugew->primitive.shadow_thickness;
    int   text_height;

    userSize = gaugew->gauge.maximum - gaugew->gauge.minimum;

    trueSize = gaugew->gauge.orientation == XmHORIZONTAL ? 
        gaugew->gauge.slider_area_width : gaugew->gauge.slider_area_height;

    factor = trueSize / userSize;

    sliderSize = (float) (gaugew->gauge.value - gaugew->gauge.minimum) * factor;

    text_height = gaugew->gauge.font->ascent + gaugew->gauge.font->descent;

    if (gaugew->gauge.orientation == XmHORIZONTAL)
    {
           *slider_width = (int) (sliderSize + 0.5) - margin * 2;
           *slider_height = gaugew->gauge.slider_area_height - margin * 2;

        if (gaugew->gauge.processing_direction == XmMAX_ON_RIGHT)
            *slider_x = gaugew->gauge.slider_area_x + margin;
        else
            *slider_x =  gaugew->gauge.slider_area_x +
                         gaugew->gauge.slider_area_width - *slider_width - margin;

           *slider_y = gaugew->gauge.slider_area_y + margin;
           minSliderWidth = MIN_SLIDER_LENGTH;
           minSliderHeight = MIN_SLIDER_THICKNESS;
    }
    else
    {
          *slider_width = gaugew->gauge.slider_area_width - margin * 2;
          *slider_height = (int) (sliderSize + 0.5) - margin * 2;
          *slider_x = gaugew->gauge.slider_area_x + margin;

        if (gaugew->gauge.processing_direction == XmMAX_ON_TOP)
               *slider_y = gaugew->gauge.slider_area_y + 
                    gaugew->gauge.slider_area_height - *slider_height - margin;
        else
            *slider_y = gaugew->gauge.slider_area_y + margin;

          minSliderWidth = MIN_SLIDER_THICKNESS;
          minSliderHeight = MIN_SLIDER_LENGTH;
    }

    if (*slider_width < minSliderWidth)
        if (gaugew->gauge.value != gaugew->gauge.minimum)
            *slider_width = minSliderWidth;
        else
            *slider_width = 0;

            
    if (*slider_height < minSliderHeight)
    {
        if (gaugew->gauge.value != gaugew->gauge.minimum)
            *slider_height = minSliderHeight;
        else
            *slider_height = 0;
        if (gaugew->gauge.orientation == XmVERTICAL)
            if (gaugew->gauge.processing_direction == XmMAX_ON_TOP)
                *slider_y = gaugew->gauge.slider_area_y +
                    gaugew->gauge.slider_area_height - *slider_height - margin;
            else
                *slider_y = gaugew->gauge.slider_area_y + margin;

    }
}



/************************************************************************
 *
 *  DrawGauge
 *    Draw the Gauge widget. 
 *   
 *
 ************************************************************************/
static void DrawGauge (XmGaugeWidget gaugew)
{
    char      min_string[256];
    char      max_string[256];
    int       gauge_margin;
    Display  *display = XtDisplay((Widget) gaugew);
    Window    window = XtWindow((Widget) gaugew);

    gauge_margin = gaugew->gauge.gauge_margin +
                  gaugew->primitive.highlight_thickness +
                  gaugew->primitive.shadow_thickness;



    /*
    ** Draw Slider Area.
    */
    if (gaugew->gauge.trough_shadow_thickness)
#if XmVersion == 1001
        _XmDrawShadow(XtDisplay((Widget) gaugew), XtWindow((Widget) gaugew),
                    gaugew->primitive.bottom_shadow_GC,
                    gaugew->primitive.top_shadow_GC,
                    gaugew->gauge.trough_shadow_thickness,
                    gaugew->gauge.slider_area_x,
                    gaugew->gauge.slider_area_y,
                    gaugew->gauge.slider_area_width,
                    gaugew->gauge.slider_area_height);
#else
        _XmDrawShadows(XtDisplay((Widget) gaugew), XtWindow((Widget) gaugew),
                    gaugew->primitive.top_shadow_GC,
                    gaugew->primitive.bottom_shadow_GC,
                    gaugew->gauge.slider_area_x,
                    gaugew->gauge.slider_area_y,
                    gaugew->gauge.slider_area_width,
                    gaugew->gauge.slider_area_height,
                    gaugew->gauge.trough_shadow_thickness,
                    XmSHADOW_IN);
#endif

    /*
    ** Display Minimum and Maximum values if requested.
    */
    if (gaugew->gauge.show_min_max)
    {
        sprintf(min_string, "%d", gaugew->gauge.minimum);
        sprintf(max_string, "%d", gaugew->gauge.maximum);

        XDrawString(display, 
                window, 
                gaugew->gauge.text_GC,
                gaugew->gauge.minimum_x, gaugew->gauge.minimum_y,
                min_string,
                strlen(min_string));

        XDrawString(display, 
                window, 
                gaugew->gauge.text_GC,
                gaugew->gauge.maximum_x, gaugew->gauge.maximum_y,
                max_string,
                strlen(max_string));
    }

    /*
    ** Display Value string.
    */
    DisplayValue(gaugew);
} 

void XmSetGaugeColorThreshold (XmGaugeWidget gaugew, int value, String color_name)
{
    Pixel            color;
    ColorThreshold  color_threshold;
    ColorThreshold    p, prev;
    XrmValue        from;
    XrmValue        to;

    if (XtIsRealized(gaugew))
    {
        from.addr = p->color_name;
        from.size = strlen(p->color_name);
        to.addr = (void *)&color;
        to.size = sizeof(color);
        if (XtConvertAndStore((Widget) gaugew, XtRString, &from, XtRPixel, &to) == False)
        {
            XtAppError(XtWidgetToApplicationContext((Widget) gaugew),
                "XmSetGaugeColorThreshold: String to Pixel conversion failed");
            return;
        }
        color = (*(Pixel *) to.addr);
    }

    color_threshold =
                (ColorThreshold) XtMalloc(sizeof(*color_threshold));
    color_threshold->color = color;
    color_threshold->value = value;
    color_threshold->color_name =
                        strcpy(XtMalloc(strlen(color_name) + 1), color_name);
    color_threshold->next = NULL;

    if (gaugew->gauge.color_thresholds)
    {
        for (prev = NULL, p = gaugew->gauge.color_thresholds;
             p;
             prev = p, p = p->next)
            if (value <= p->value)
                break;

        if (value == prev->value)
        {
            p->color = color;        
            XtFree(p->color_name);
             p->color_name = color_threshold->color_name;    
            XtFree((char *) color_threshold);
        }
        else 
        {
            if (prev)
            {
                color_threshold->next = prev->next;
                prev->next = color_threshold;
            }
            else
            {
                color_threshold->next = p;
                gaugew->gauge.color_thresholds = color_threshold;
            }
        }    
    }
    else
        gaugew->gauge.color_thresholds = color_threshold;
}
