/*
 Copyright (C) 2000 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

#if !defined(xarm_xarmtreelist_h)
#define xarm_xarmtreelist_h

#include <Xarm/XarmMatrix.h>
#include <Xarm/Callback.h>

#if defined(XARM_HAS_XBAE)

class FakeTLTarget { };

//
// The following are specific handlers for TreeList objects.
// Since TreeLists aren't real widgets, they don't have real
// Motif callbacks. Xarm handles them internally.
//

enum XaTLCallbackType {
    XaNselectCallback,
    XaNdefaultActionCallback,
    XaNexpandCallback,
    XaNrightClickCallback,
    XaNprocessDragCallback
};

template <class T>
struct TLCallbackInfo {
   typedef void (T::*p_msg)(Widget, XtPointer, XtPointer);

   p_msg _callback;
   XtPointer _closure;
   T *_target;
};

typedef TLCallbackInfo<FakeTLTarget> FakeTLInfo;

struct XaTLCallbackList {
    FakeTLInfo *ci;
    XaTLCallbackList *next;
};    

struct XarmTreeItem
{
    XarmTreeItem *prev;
    XarmTreeItem *next;
    XarmTreeItem *parent;
    XarmTreeItem *children;
    XtPointer     userData;
    bool          isExpanded;
    bool          isSelected;
    int           indent;
    Pixmap        closedIcon;
    Pixmap        openIcon;
    char          text[1];
};

struct XarmTreeDragCallbackStruct {
    XarmTreeItem *item;
    XEvent       *event;
};

class XarmTreeList : public MatrixClass
{

    XarmTreeItem    *rootNode;
    int              maxIndent;
    int              maxTextLen;
    int              reqWidth;
    int              reqHeight;
    int              scrWidth;
    int              hsbWidth;
    int              fontCharHeight;
    int              fontCharWidth;
    int              fontAscent;
    int              numVisible;
    int              lastRowClicked;
    int              numSelected;
    int              sbWidth;
    int              activeRow;
    int              disabled;
    XEvent           lastBPEvent;
    Font             fid;
    Pixel            boxPixel;
    static const int indentIncr;

    XaTLCallbackList *selectCBList;
    XaTLCallbackList *defActionCBList;
    XaTLCallbackList *expandCBList;
    XaTLCallbackList *rcCBList;
    XaTLCallbackList *dragCBList;

    void treeInit();
    void fontInit();
    void drawCell(Widget, XtPointer, XtPointer);
    void selectCell(Widget, XtPointer, XtPointer);
    void resize(Widget, XtPointer, XtPointer);
    void defaultAction(Widget, XtPointer, XtPointer);
    void mouseMotion(Widget, XtPointer, XEvent *, Boolean *);
    void showTree(XarmTreeItem *rn);
    XarmTreeItem *getItemAt(XarmTreeItem *curNode, int gRow, int &cRow);
    XarmTreeItem *getLastVisible();
    int getNumVisible(XarmTreeItem *);
    void deselectAll(XarmTreeItem *);
    void deleteAllItems(XarmTreeItem *);
    void callCB(XaTLCallbackList *, XarmTreeItem *, XEvent *ev = NULL);
    void queryTextLen(const XarmTreeItem *);

protected:

    virtual _XtString className() { return (_XtString) "XarmTreeList"; }

public:

    XarmTreeList(Widget w_parent, ArgList arglist = NULL, Cardinal cnt = 0,
                  _XtString name = NULL);
    XarmTreeList(_XtString name, Widget w_parent);
    XarmTreeList(Widget w_parent, XarmArg &args, _XtString name);

    XarmTreeItem *addItemFirst(XarmTreeItem *parent,
			      const char    *text,
			      Pixmap         openIcon   = XmUNSPECIFIED_PIXMAP,
			      Pixmap         closedIcon = XmUNSPECIFIED_PIXMAP,
			      XtPointer      closure    = NULL);
    XarmTreeItem *addItemFirst(XarmTreeItem *parent, XarmTreeItem *item);

    XarmTreeItem *addItemLast(XarmTreeItem *parent,
			      const char   *text,
			      Pixmap        openIcon   = XmUNSPECIFIED_PIXMAP,
			      Pixmap        closedIcon = XmUNSPECIFIED_PIXMAP,
			      XtPointer     closure    = NULL);
    XarmTreeItem *addItemLast(XarmTreeItem *parent, XarmTreeItem *item);

    XarmTreeItem *addItemBefore(XarmTreeItem *other,
                                const char   *text,
                                Pixmap        openIcon   = XmUNSPECIFIED_PIXMAP,
                                Pixmap        closedIcon = XmUNSPECIFIED_PIXMAP,
                                XtPointer     closure    = NULL);
    XarmTreeItem *addItemBefore(XarmTreeItem *parent, XarmTreeItem *item);

    XarmTreeItem *addItemAfter(XarmTreeItem *other,
			      const char    *text,
			      Pixmap         openIcon   = XmUNSPECIFIED_PIXMAP,
			      Pixmap         closedIcon = XmUNSPECIFIED_PIXMAP,
			      XtPointer      closure    = NULL);
    XarmTreeItem *addItemAfter(XarmTreeItem *parent, XarmTreeItem *item);

    void refreshTree();

    void removeItem(XarmTreeItem *); // remove but don't destroy
    void deleteItem(XarmTreeItem *); // remove and destroy

    void enableRedisplay() { --disabled; }
    void disableRedisplay() { ++disabled; }

    void expandTree(XarmTreeItem *branch);
    void showTree() { showTree(rootNode); }
    XarmTreeItem *getItemAt(int row);
    void deselectAll() { deselectAll(rootNode); }
    int getNumVisible() { return getNumVisible(rootNode); }
    void selectRange(int first, int last);

    void deleteAllItems() { deleteAllItems(rootNode); rootNode = NULL; }

    void addCB(XaTLCallbackType ct, FakeTLInfo *ci);

    virtual ~XarmTreeList();
};

template<class T>
void addCallback(T *obj,
                 void (T::*func) (Widget, XtPointer, XtPointer),
                 XarmTreeList *tl,
                 XaTLCallbackType type,
                 XtPointer closure=NULL)
{
    TLCallbackInfo<T> *ci = new TLCallbackInfo<T>;

    if (ci == NULL) return;

    ci->_target   = obj;
    ci->_callback = func;
    ci->_closure  = closure;

    tl->addCB(type, (FakeTLInfo *)ci);
}


#endif

#endif
