/*
 * Copyright (C) 1994 Communications and Systems Specialists, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the 
 * "Software"), to deal in the Software without restriction, including 
 * without limitation the rights to use, copy, modify, merge, publish, 
 * distribute, sublicense, and/or sell copies of the Software, and to 
 * permit  persons to whom the Software is furnished to do so, subject to 
 * the following conditions:
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
 * IN NO EVENT SHALL COMMUNICATIONS AND SYSTEMS SPECIALISTS, INC. BE 
 * LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Communications and 
 * Systems Specialists, Inc. shall not be used in advertising or otherwise
 * to promote the sale, use or other dealings in this Software without 
 * prior written authorization from Communications and Systems 
 * Specialists, Inc.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <pwd.h>
#include <sys/utsname.h>
#include <X11/Xos.h>
#include "Xmsg.h"

/*--------------------
-- Private constants
--------------------*/
#define XmsgMESSAGE_MAX_SIZE	2000
#define XmsgPARAMS_MAX		10
#define XmsgTIME_MAX_SIZE       100

/*----------------------------------------------------------------------
-- Offsets into quark list in message structure for the resource spec
----------------------------------------------------------------------*/
#define XmsgQAPP	0   /* Application name */
#define XmsgQSEV	1   /* Message severity */
#define XmsgQNAME	2   /* Message name */
#define XmsgQTYPE	3   /* Message type */
#define XmsgQRES	4   /* Resource name, changed for each lookup */
#define XmsgQNULL	5   /* NULL terminator */

#define XmsgQTABLE_MAX	6   /* Number of elements in quark list */

/*-------------------------------------
-- Multi-threading supported under R6
-------------------------------------*/
#if XtSpecificationRelease < 6
#  define XmsgLOCK_PROCESS	
#  define XmsgUNLOCK_PROCESS	
#else
#  if WATCH_THREADS
#     define XmsgLOCK_PROCESS( str )   printf( "Lock --- %s\n", (str) ); \
                                       XtProcessLock()
#     define XmsgUNLOCK_PROCESS( str ) printf( "Unlock - %s\n", (str));	\
                                       XtProcessUnlock()
#  else
#     define XmsgLOCK_PROCESS( str )   XtProcessLock()
#     define XmsgUNLOCK_PROCESS( str ) XtProcessUnlock()
#  endif
#endif


/*--------------------------------------------
-- Information describing a message instance
--------------------------------------------*/
struct _XmsgMessageStruct
{
   short int              version_major;
   short int              version_minor;
   _XmsgConst _XmsgString category_str;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString source_str;
   unsigned int           source_line;
   int                    error_number;
   XtPointer              client_data;
   XmsgPosition           position;
   String                 text_str;
   XrmQuark               name_quark_table [ XmsgQTABLE_MAX ];
   XrmQuark               class_quark_table[ XmsgQTABLE_MAX ];
   Widget                 widget;
   XtAppContext           app_context;
   XrmDatabase*           database_ptr;
   _XmsgConst _XmsgString local_time_str;
   Boolean                must_dispatch;
   Boolean                no_x_protocol;
   Boolean                free_memory;
   unsigned long          instance_counter;
   Display*               true_display_ptr;
};


/*-----------------------------------------
-- Linked list node of handler functions
-----------------------------------------*/
typedef struct _XmsgDispatchStruct
{
   XrmQuark                    name_quark;
   XrmQuark                    class_quark;
   XmsgShouldDispatchProc      should_dispatch_proc;
   XmsgHandlerProc             handler_proc;
   unsigned int                control;
   struct _XmsgDispatchStruct* next;
}
XmsgDispatchStruct;

typedef struct _XmsgServerExtensionStruct
{
   String   name_str;
   int      major_opcode;
   int      first_event;
   int      first_error;
}
XmsgServerExtensionStruct;

typedef struct _XmsgDisplayListStruct
{
   Display*                       display_ptr;
   XmsgServerExtensionStruct*     extension_ptr;
   int                            extension_total;
   struct _XmsgDisplayListStruct* next_ptr;
}
XmsgDisplayListStruct;


/*---------------------------------------
-- Early declaration of private methods
---------------------------------------*/
static void XmsgHandlerSimple(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr
#endif
);

static void XmsgDoExit( 
#if NeedFunctionPrototypes
   void 
#endif
);


/*-------------------------------------------------------
-- Memory used by Xmsg across all functions and threads
-------------------------------------------------------*/
typedef struct _XmsgPackageDataStruct
{
   Boolean       installed;                   /* Has package been installed? */
   Boolean       dispatch_all;                /* All msgs sent at least once?*/
   String        time_format_str;             /* ANCI C date/time format ----*/
   Widget        default_widget;              /* Default widget available ---*/
   String        app_name_str;                /* Application name -----------*/
   String        app_class_str;               /* Application class ----------*/
   XrmQuark      app_name_quark;              /* Quarkified appl name -------*/
   XrmQuark      app_class_quark;             /* Quarkified appl class ------*/
   XrmQuark      append_name_quark;           /* Quarkified "xmsgAppend" ----*/
   XrmQuark      append_class_quark;          /* Quarkified "XmsgAppend" ----*/
   XrmQuark      prepend_name_quark;          /* Quarkified "xmsgPrepend" ---*/
   XrmQuark      prepend_class_quark;         /* Quarkified "XmsgPrepend" ---*/
   XrmQuark      text_name_quark;             /* Quarkified "xmsgText" ------*/
   XrmQuark      text_class_quark;            /* Quarkified "XmsgText" ------*/
   XrmQuark      stderr_prepend_name_quark;   /* Quarkified "stderrPrepend" -*/
   XrmQuark      stderr_prepend_class_quark;  /* Quarkified "StderrPrepend" -*/
   String        host_name_str;               /* Name of the client host ----*/
   String        sys_name_str;                /* Name of client O/S ---------*/
   String        sys_release_str;             /* O/S release number ---------*/
   String        sys_version_str;             /* O/S version number ---------*/
   String        machine_str;                 /* Client machine type --------*/
   String        login_name_str;              /* User's login name ----------*/
   String        login_gecos_str;             /* User's real name (usually) -*/
   unsigned long instance_counter;            /* Unique one up message no. --*/

   XmsgDispatchStruct* dispatch_ptr;          /* Root of dispatcher list ----*/
   XmsgExitProc        exit_proc;             /* Called after fatal error ---*/
   XmsgHandlerProc     handler_proc;          /* Common handler method ------*/

   XmsgDisplayListStruct*  display_list_ptr;  /* Displays registered --------*/
}
XmsgPackageDataStruct;

static XmsgPackageDataStruct package_data =
{
   False,		/* installed                   */
   False,               /* dispatch_all                */
   "%c",		/* time_format_str             */
   NULL,		/* default_widget              */
   "xmsg",		/* app_name_str                */
   "Xmsg",		/* app_class_str               */ 
   NULLQUARK,		/* app_name_quark              */
   NULLQUARK,		/* app_class_quark             */
   NULLQUARK,		/* append_name_quark           */
   NULLQUARK,		/* append_class_quark          */
   NULLQUARK,		/* prepend_name_quark          */
   NULLQUARK,		/* prepend_class_quark         */
   NULLQUARK,		/* text_name_quark             */
   NULLQUARK,		/* text_class_quark            */
   NULLQUARK,		/* stderr_prepend_name_quark   */
   NULLQUARK,		/* stderr_prepend_class_quark  */
   "(host)",		/* host_name_str               */ 
   "(O/S)",             /* sys_name_str                */
   "(release)",         /* sys_release_str             */
   "(version)",         /* sys_version_str             */
   "(machine)",         /* machine_str                 */
   "(account)",		/* login_name_str              */
   "(gecos)",		/* login_gecos_str             */
   0,                   /* instance_counter            */
   NULL,		/* dispatch_ptr                */ 
   XmsgDoExit,		/* exit_proc                   */
   XmsgHandlerSimple,   /* handler_proc                */
   NULL                 /* extension_ptr               */
};


/*------------------------------
-- Private function prototypes
------------------------------*/
static void XmsgGetErrorDatabaseText( 
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  default_str,
   unsigned int       buffer_size 
#endif
);

static void XmsgAugmentText(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString buffer_in_str 
#endif
);
 
static void XmsgDispatch( 
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 
#endif
);

static _XmsgConst XmsgServerExtensionStruct* XmsgGetServerExtension(
#if NeedFunctionPrototypes
   Display* display_ptr,
   int      major_opcode 
#endif
);

static void XmsgSetupMessage(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  category_str,
   Widget             w,
   _XmsgConst _XmsgString  name_str,
   _XmsgConst _XmsgString  type_str,
   _XmsgConst _XmsgString  class_str,
   _XmsgConst _XmsgString  source_str,
   unsigned int       source_line,
   int                error_number,
   XtPointer          client_data,
   XmsgPosition       position 
#endif
);

static int XmsgXErrorHandler(
#if NeedFunctionPrototypes
   Display*     display_ptr,
   XErrorEvent* error_event_ptr 
#endif
);

static int XmsgXIOErrorHandler(
#if NeedFunctionPrototypes
   Display* display_ptr 
#endif
);

static void XmsgXtErrorHandler(
#if NeedFunctionPrototypes
   String message_str 
#endif
);

static void XmsgXtErrorMsgHandler(
#if NeedFunctionPrototypes
   String    name_str,
   String    type_str,
   String    class_str,
   String    default_str,
   String*   params_str_table,
   Cardinal* params_str_count 
#endif
);

static void XmsgXtWarningHandler(
#if NeedFunctionPrototypes
   String message_str 
#endif
);

static void XmsgXtWarningMsgHandler(
#if NeedFunctionPrototypes
   String    name_str,
   String    type_str,
   String    class_str,
   String    default_str,
   String*   params_str_table,
   Cardinal* params_str_count 
#endif
);


/*-------------------------------
-- Private function definitions
-------------------------------*/
#if NeedFunctionPrototypes
static void XmsgGetErrorDatabaseText( 
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  default_str,
   unsigned int       buffer_size )
#else
static void XmsgGetErrorDatabaseText( message_ptr, default_str, buffer_size )
   XmsgMessageStruct* message_ptr;
   _XmsgConst _XmsgString  default_str;
   unsigned int       buffer_size;
#endif
{
   _XmsgConst _XmsgString text_str;

   text_str = XmsgGetMsgDatabaseString(
      message_ptr,
      package_data.text_name_quark, 
      package_data.text_class_quark );

   if (text_str)
   {
      strcpy( message_ptr->text_str, text_str );
   }
   else
   {
      XtAppGetErrorDatabaseText( message_ptr->app_context, 
                                 message_ptr->name_str, 
                                 message_ptr->type_str, 
                                 message_ptr->class_str,
                                 default_str, 
                                 message_ptr->text_str, 
                                 buffer_size, NULL );
   }
}


#if NeedFunctionPrototypes
static void XmsgAugmentText(
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  buffer_in_str )
#else
static void XmsgAugmentText( message_ptr, buffer_in_str )
   XmsgMessageStruct* message_ptr;
   _XmsgConst _XmsgString  buffer_in_str;
#endif
{
   _XmsgConst _XmsgString text_str;
   
   message_ptr->text_str[ 0 ] = '\0';

   text_str = XmsgGetMsgDatabaseString(
      message_ptr,
      package_data.prepend_name_quark, 
      package_data.prepend_class_quark );

   if (text_str)
   {
      strcpy( message_ptr->text_str, text_str );
   }

   strcat( message_ptr->text_str, buffer_in_str );

   text_str = XmsgGetMsgDatabaseString(
      message_ptr,
      package_data.append_name_quark, 
      package_data.append_class_quark );

   if (text_str)
   {
      strcat( message_ptr->text_str, text_str );
   }
}
 

#if NeedFunctionPrototypes
static void XmsgDispatch( 
   XmsgMessageStruct* message_ptr )
#else
static void XmsgDispatch( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   _XmsgConst _XmsgString text_str;
   XmsgDispatchStruct* dispatch_ptr;
   Boolean             dispatch_message;
   Boolean             dispatched = False;

   for (dispatch_ptr = package_data.dispatch_ptr; 
        dispatch_ptr != NULL;
        dispatch_ptr = dispatch_ptr->next)
   {
      dispatch_message = False;

      if (dispatch_ptr->should_dispatch_proc)
      {
         dispatch_message = dispatch_ptr->should_dispatch_proc( message_ptr );
         if (dispatch_message)
         {
            dispatch_ptr->handler_proc( message_ptr );
            dispatched = True;
         }
      }

      if (!dispatch_message)
      {
         text_str = XmsgGetMsgDatabaseString(
            message_ptr,
            dispatch_ptr->name_quark,
            dispatch_ptr->class_quark );

         if (text_str)
         {
            if (XmsgCvtStringToBoolean( text_str, False ))
            {
               dispatch_ptr->handler_proc( message_ptr );
               dispatched = True;
            }
         }
         else if (dispatch_ptr->control & XmsgCONTROL_CALL_AS_DEFAULT)
         {
            dispatch_ptr->handler_proc( message_ptr );
            dispatched = True;
         }
      } 
   }

   if (!dispatched)
   {
      if (message_ptr->must_dispatch)
      {
         package_data.handler_proc( message_ptr );
      }
   }
}


#if NeedFunctionPrototypes
static void XmsgHandlerSimple(
   XmsgMessageStruct* message_ptr )
#else
static void XmsgHandlerSimple( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   _XmsgConst _XmsgString msg_str;
   _XmsgConst _XmsgString category_str;

   msg_str      = XmsgGetMsgText    ( message_ptr );
   category_str = XmsgGetMsgCategory( message_ptr );

   fprintf( stderr, "%s : %s\n", category_str, msg_str );
}


#if NeedFunctionPrototypes
static void XmsgDoExit( void )
#else
static void XmsgDoExit(  )
#endif
{
   exit( -1 );
}


#if NeedFunctionPrototypes
static _XmsgConst XmsgServerExtensionStruct* XmsgGetServerExtension(
   Display* display_ptr,
   int      major_opcode )
#else
static _XmsgConst XmsgServerExtensionStruct* 
XmsgGetServerExtension( display_ptr, major_opcode )
   Display* display_ptr;
   int      major_opcode;
#endif
{
   int offset;
   XmsgDisplayListStruct* display_list_ptr;
   XmsgServerExtensionStruct* extension_ptr;

   static XmsgServerExtensionStruct unknown_extension = {"(unknown)", 0, 0, 0};

   extension_ptr = &unknown_extension;

   for (display_list_ptr = package_data.display_list_ptr;
        display_list_ptr;
        display_list_ptr = display_list_ptr->next_ptr)
   {
      if (display_list_ptr->display_ptr == display_ptr)
      {
         for (offset = 0; 
              offset < display_list_ptr->extension_total;
              offset++)
         {
            if (display_list_ptr->extension_ptr[ offset ].major_opcode ==
                major_opcode)
            {
               extension_ptr = &display_list_ptr->extension_ptr[ offset ];
            }
         }
      }
   }
   return extension_ptr;
}


#if NeedFunctionPrototypes
static void XmsgSetupMessage(
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  category_str,
   Widget             w,
   _XmsgConst _XmsgString  name_str,
   _XmsgConst _XmsgString  type_str,
   _XmsgConst _XmsgString  class_str,
   _XmsgConst _XmsgString  source_str,
   unsigned int       source_line,
   int                error_number,
   XtPointer          client_data,
   XmsgPosition       position )
#else
static void XmsgSetupMessage( message_ptr, category_str, w, name_str, type_str,
                              class_str, source_str, source_line, error_number,
                              client_data, position )
   XmsgMessageStruct* message_ptr;
   _XmsgConst _XmsgString  category_str;
   Widget             w;
   _XmsgConst _XmsgString  name_str;
   _XmsgConst _XmsgString  type_str;
   _XmsgConst _XmsgString  class_str;
   _XmsgConst _XmsgString  source_str;
   unsigned int       source_line;
   int                error_number;
   XtPointer          client_data;
   XmsgPosition       position;
#endif
{
   time_t     current_time;
   struct tm* local_time_ptr;
   String     str;
   XrmQuark   class_quark;

   if (time( &current_time ) == (time_t) -1)
   {
      strcpy( (String) message_ptr->local_time_str, "(time not known)" );
   }
   else
   {
      XmsgLOCK_PROCESS( "localtime and strftime are not thread safe" );

      local_time_ptr = localtime( &current_time );

      strftime( (String) message_ptr->local_time_str, XmsgTIME_MAX_SIZE,
                package_data.time_format_str, local_time_ptr );

      XmsgUNLOCK_PROCESS( "localtime and strftime are not tread safe" );
   }

   if (w)
   {
      message_ptr->widget = w;
   }
   else
   {
      message_ptr->widget = package_data.default_widget;
   }
   message_ptr->app_context = 
      XtWidgetToApplicationContext( message_ptr->widget );

   message_ptr->database_ptr = 
      XtAppGetErrorDatabase( message_ptr->app_context );

   message_ptr->category_str     = category_str;
   message_ptr->name_str         = name_str;
   message_ptr->type_str         = type_str;
   message_ptr->class_str        = class_str;
   message_ptr->source_str       = source_str;
   message_ptr->source_line      = source_line;
   message_ptr->error_number     = error_number;
   message_ptr->client_data      = client_data;
   message_ptr->position         = position;
   message_ptr->text_str         = NULL;
   message_ptr->must_dispatch    = package_data.dispatch_all;
   message_ptr->no_x_protocol    = False;
   message_ptr->free_memory      = False;
   message_ptr->true_display_ptr = XtDisplay( message_ptr->widget );

   /*--------------------------------------------
   -- toupper and tolower may not be thread safe
   ---------------------------------------------*/
   XmsgLOCK_PROCESS( "Increment instance counter" );
   message_ptr->instance_counter = package_data.instance_counter;
   package_data.instance_counter++;
   XmsgUNLOCK_PROCESS( "Increment instance counter" );

   str = XtNewString( category_str );   

   XmsgLOCK_PROCESS( "Using tolower");
   str[ 0 ] = tolower( str[ 0 ] );
   XmsgUNLOCK_PROCESS( "Using tolower");

   message_ptr->name_quark_table[ XmsgQAPP  ] = package_data.app_name_quark;
   message_ptr->name_quark_table[ XmsgQSEV  ] = XrmStringToQuark( str      );
   message_ptr->name_quark_table[ XmsgQNAME ] = XrmStringToQuark( name_str );
   message_ptr->name_quark_table[ XmsgQTYPE ] = XrmStringToQuark( type_str );
   message_ptr->name_quark_table[ XmsgQRES  ] = NULLQUARK;
   message_ptr->name_quark_table[ XmsgQNULL ] = NULLQUARK;

   XmsgLOCK_PROCESS( "Using toupper");
   str[ 0 ] = toupper( str[ 0 ] );
   XmsgUNLOCK_PROCESS( "Using toupper");

   class_quark = XrmStringToQuark( class_str );
   message_ptr->class_quark_table[ XmsgQAPP  ] = package_data.app_class_quark;
   message_ptr->class_quark_table[ XmsgQSEV  ] = XrmStringToQuark( str );
   message_ptr->class_quark_table[ XmsgQNAME ] = class_quark;
   message_ptr->class_quark_table[ XmsgQTYPE ] = class_quark;
   message_ptr->class_quark_table[ XmsgQRES  ] = NULLQUARK;
   message_ptr->class_quark_table[ XmsgQNULL ] = NULLQUARK;
   XtFree( str );
}


#define XmsgXERROR_SEGMENT_SIZE 100
#define XmsgXERROR_BUFFER_SIZE	2000

#if NeedFunctionPrototypes
static int XmsgXErrorHandler(
   Display*     display_ptr,
   XErrorEvent* error_event_ptr )
#else
static int XmsgXErrorHandler( display_ptr, error_event_ptr )
   Display*     display_ptr;
   XErrorEvent* error_event_ptr;
#endif
{
   char  local_time_str[ XmsgTIME_MAX_SIZE ];
   char  message_str   [ XmsgXERROR_BUFFER_SIZE ]; 
   char  segment_str   [ XmsgXERROR_SEGMENT_SIZE ];
   char  number_str    [ 20 ];
   char* message_ptr;
   int   message_offset;

   XmsgMessageStruct message;

   /*------------------------
   -- Timestamp the message
   ------------------------*/
   message.local_time_str = local_time_str;

   /*----------------------
   -- Package the message
   ----------------------*/
   XmsgSetupMessage( &message, XmsgCATEGORY_ERROR, NULL, 
                     "xlib", "error", "XError",
                     NULL, 0, 0, NULL, XmsgPositionDATABASE );
                    
   message.text_str         = message_str;
   message.must_dispatch    = True;
   message.no_x_protocol    = True;
   message.true_display_ptr = display_ptr;

   message_str[0] = '\0';
   message_ptr = message_str;
   message_offset = 0;

   XGetErrorDatabaseText( display_ptr, "XlibMessage", "XError",
                          "X Error", message_ptr, 
                          XmsgXERROR_BUFFER_SIZE - message_offset );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   strcat( message_ptr, ":  " );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   XGetErrorText( display_ptr, error_event_ptr->error_code, message_ptr,
                  XmsgXERROR_BUFFER_SIZE - message_offset );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   strcat( message_ptr, "\n  " );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   XGetErrorDatabaseText( display_ptr, "XlibMessage", "MajorCode",
                          "Request Major code %d", segment_str,
                          XmsgXERROR_SEGMENT_SIZE );
   sprintf( message_ptr, segment_str, error_event_ptr->request_code );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   if (error_event_ptr->request_code < 128)
   {
      sprintf( number_str, "%d", error_event_ptr->request_code );
      XGetErrorDatabaseText( display_ptr, "XRequest", number_str, "unknown",
                             segment_str, XmsgXERROR_SEGMENT_SIZE );
      sprintf( message_ptr, " (%s)\n  ", segment_str );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset;

      switch (error_event_ptr->error_code)
      {
      case BadValue:
         XGetErrorDatabaseText( display_ptr, "XlibMessage", "Value", 
                                "Value 0x%x", segment_str, 
                                XmsgXERROR_SEGMENT_SIZE );
         break;

      case BadAtom:
         XGetErrorDatabaseText( display_ptr, "XlibMessage", "AtomID", 
                                "AtomID 0x%x", segment_str, 
                                XmsgXERROR_SEGMENT_SIZE );
         break;

      case BadWindow:
      case BadPixmap:
      case BadCursor:
      case BadFont:
      case BadDrawable:
      case BadColor:
      case BadGC:
      case BadIDChoice:
         XGetErrorDatabaseText( display_ptr, "XlibMessage", "ResourceID", 
                                "ResourceID 0x%x", segment_str, 
                                XmsgXERROR_SEGMENT_SIZE );
         break;

      default:
         segment_str[ 0 ] = '\0';
         break;
      }

      if (*segment_str)
      {
         sprintf( message_ptr, segment_str, error_event_ptr->resourceid );
         message_offset += strlen( message_ptr );
         message_ptr    = message_str + message_offset;

         strcat( message_ptr, "\n  " );
         message_offset += strlen( message_ptr );
         message_ptr    = message_str + message_offset;
      }
   }
   else
   {
      _XmsgConst XmsgServerExtensionStruct* extension_ptr;
      char                             name_str[ 100 ];

      extension_ptr = 
         XmsgGetServerExtension( display_ptr, error_event_ptr->request_code );

      sprintf( message_ptr, " (%s)\n  ", extension_ptr->name_str );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset;

      XGetErrorDatabaseText( display_ptr, "XlibMessage", "MinorCode",
                             "Request Minor code %d", segment_str,
                             XmsgXERROR_SEGMENT_SIZE );
      sprintf( message_ptr, segment_str, error_event_ptr->minor_code );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset;

      sprintf( name_str, "%s.%d", extension_ptr->name_str, 
               error_event_ptr->minor_code );
      
      XGetErrorDatabaseText( display_ptr, "XRequest", name_str, "",
                             segment_str, XmsgXERROR_SEGMENT_SIZE );
      sprintf( message_ptr, " (%s)\n  ", segment_str );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset;

      XGetErrorDatabaseText( display_ptr, "XlibMessage", "ResourceID", 
                             "ResourceID 0x%x", segment_str, 
                             XmsgXERROR_SEGMENT_SIZE );
      sprintf( message_ptr, segment_str, error_event_ptr->resourceid );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset;

      strcat( message_ptr, "\n  " );
      message_offset += strlen( message_ptr );
      message_ptr    = message_str + message_offset; 
   }

   XGetErrorDatabaseText( display_ptr, "XlibMessage", "ErrorSerial",
                          "Error Serial #%d", segment_str,
                          XmsgXERROR_SEGMENT_SIZE );
   sprintf( message_ptr, segment_str, error_event_ptr->serial );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   strcat( message_ptr, "\n  " );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   XGetErrorDatabaseText( display_ptr, "XlibMessage", "CurrentSerial",
                          "Current Serial #%d", segment_str,
                          XmsgXERROR_SEGMENT_SIZE );
   sprintf( message_ptr, segment_str, NextRequest( display_ptr ) - 1 );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   strcat( message_ptr, "\n  " );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;
   
   XGetErrorDatabaseText( display_ptr, "XlibMessage", "DisplayName",
                          "Display server name: %s", segment_str,
                          XmsgXERROR_SEGMENT_SIZE );
   sprintf( message_ptr, segment_str, DisplayString( display_ptr ) );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   strcat( message_ptr, "\n" );
   message_offset += strlen( message_ptr );
   message_ptr    = message_str + message_offset;

   XmsgDispatch( &message );
   return 0;
}


#if NeedFunctionPrototypes
static int XmsgXIOErrorHandler(
   Display* display_ptr )
#else
static int XmsgXIOErrorHandler( display_ptr )
   Display* display_ptr;
#endif
{
   char  local_time_str[ XmsgTIME_MAX_SIZE ];
   char  message_str   [ XmsgXERROR_BUFFER_SIZE ]; 

   XmsgMessageStruct message;

   /*------------------------
   -- Timestamp the message
   ------------------------*/
   message.local_time_str = local_time_str;

   /*----------------------
   -- Package the message
   ----------------------*/
   XmsgSetupMessage( &message, XmsgCATEGORY_FATAL, NULL, 
                     "xlib", "io", "XIOError",
                     NULL, 0, 0, NULL, XmsgPositionDATABASE );
                    
   message.text_str         = message_str;
   message.must_dispatch    = True;
   message.no_x_protocol    = True;
   message.true_display_ptr = display_ptr;

   message_str[0] = '\0';

   
   if (errno == EPIPE)
   {
      sprintf( 
         message_str, 
         "X connection to %s broken (explicit kill or server shutdown).\r\n",
         DisplayString( display_ptr ) );
   }
   else
   {
      sprintf(
         message_str,
         "XIO:  fatal IO error %d (%s) on X server \"%s\"\r\n"
         "      after %lu requests (%lu known processed) "
         "with %d events remaining.\r\n",
         errno, strerror( errno ), 
         DisplayString( display_ptr ),
         NextRequest( display_ptr ) - 1, 
         LastKnownRequestProcessed( display_ptr ), 
         QLength( display_ptr ) );
   }

   XmsgDispatch( &message );

   XmsgExit();
   exit( 1 );
   return 0;
}


#if NeedFunctionPrototypes
static void XmsgXtErrorHandler(
   String message_str )
#else
static void XmsgXtErrorHandler( message_str )
   String message_str;
#endif
{
   char       local_time_str[ XmsgTIME_MAX_SIZE ];

   XmsgMessageStruct message;

   /*------------------------
   -- Timestamp the message
   ------------------------*/
   message.local_time_str = local_time_str;

   /*----------------------
   -- Package the message
   ----------------------*/
   XmsgSetupMessage( &message, XmsgCATEGORY_FATAL, NULL, 
                     "xtError", "lowLevel", "XtLowLevelError",
                     NULL, 0, 0, NULL, XmsgPositionDATABASE );
                    
   message.text_str = message_str;
   message.must_dispatch = True;

   XmsgDispatch( &message );

   XmsgExit();
   exit( 1 );
}


#if NeedFunctionPrototypes
static void XmsgXtErrorMsgHandler(
   String    name_str,
   String    type_str,
   String    class_str,
   String    default_str,
   String*   params_str_table,
   Cardinal* params_str_count )
#else
static void XmsgXtErrorMsgHandler( name_str, type_str, class_str, default_str,
                                   params_str_table, params_str_count )
   String    name_str;
   String    type_str;
   String    class_str;
   String    default_str;
   String*   params_str_table;
   Cardinal* params_str_count;
#endif
{
   if (params_str_count)
   {
      Xmsg_Generic(
         XmsgCATEGORY_FATAL, NULL, name_str, type_str, class_str, default_str, 
         params_str_table, *params_str_count, "(unknown source module)", 0,
         0, NULL, XmsgPositionDATABASE );
   }
   else
   {
      Xmsg_Generic(
         XmsgCATEGORY_FATAL, NULL, name_str, type_str, class_str, default_str, 
         params_str_table, 0, "(unknown source module)", 0,
         0, NULL, XmsgPositionDATABASE );
   }
   XmsgExit();
   exit( 1 );
}


#if NeedFunctionPrototypes
static void XmsgXtWarningHandler(
   String message_str )
#else
static void XmsgXtWarningHandler( message_str )
   String message_str;
#endif
{
   char       local_time_str[ XmsgTIME_MAX_SIZE ];

   XmsgMessageStruct message;

   /*------------------------
   -- Timestamp the message
   ------------------------*/
   message.local_time_str = local_time_str;

   /*----------------------
   -- Package the message
   ----------------------*/
   XmsgSetupMessage( &message, XmsgCATEGORY_WARNING, NULL, 
                     "xtWarning", "lowLevel", "XtLowLevelWarning",
                     NULL, 0, 0, NULL, XmsgPositionDATABASE );
                    
   message.text_str = message_str;
   message.must_dispatch = True;

   XmsgDispatch( &message );
}


#if NeedFunctionPrototypes
static void XmsgXtWarningMsgHandler(
   String    name_str,
   String    type_str,
   String    class_str,
   String    default_str,
   String*   params_str_table,
   Cardinal* params_str_count )
#else
static void XmsgXtWarningMsgHandler( name_str, type_str, class_str, default_str,
                                     params_str_table, params_str_count )
   String    name_str;
   String    type_str;
   String    class_str;
   String    default_str;
   String*   params_str_table;
   Cardinal* params_str_count;
#endif
{
   if (params_str_count)
   {
      Xmsg_Generic(
         XmsgCATEGORY_WARNING,NULL, name_str, type_str, class_str, default_str, 
         params_str_table, *params_str_count, "(unknown source module)", 0,
         0, NULL, XmsgPositionDATABASE );
   }
   else
   {
      Xmsg_Generic(
         XmsgCATEGORY_WARNING,NULL, name_str, type_str, class_str, default_str, 
         params_str_table, 0, "(unknown source module)", 0,
         0, NULL, XmsgPositionDATABASE );
   }
}


/*------------------------------
-- Public Function definitions
------------------------------*/
#if NeedFunctionPrototypes
XmsgHandlerProc XmsgAddHandler(
   _XmsgConst _XmsgString      name_str,
   XmsgHandlerProc        handler_proc,
   unsigned int           control,
   XmsgShouldDispatchProc should_dispatch_proc )
#else
XmsgHandlerProc XmsgAddHandler( name_str, handler_proc, control, 
                                should_dispatch_proc )
   _XmsgConst _XmsgString      name_str;
   XmsgHandlerProc        handler_proc;
   unsigned int           control;
   XmsgShouldDispatchProc should_dispatch_proc;
#endif
{
   String str;
   XrmQuark name_quark;
   XmsgDispatchStruct* dispatch_ptr;
   XmsgHandlerProc handler_proc_old;

   str = XtNewString( name_str );
   str[ 0 ] = tolower( str[ 0 ] );
   name_quark = XrmStringToQuark( str );

   /*---------------------------------------------------------------
   -- Lock process here to prevent two handlers with the same name
   -- getting defined
   ---------------------------------------------------------------*/
   XmsgLOCK_PROCESS( "Add new handler, toupper" );

   handler_proc_old = XmsgQRemoveHandler( name_quark );

   dispatch_ptr = XtNew( XmsgDispatchStruct );
   dispatch_ptr->name_quark = name_quark;
      
   str[ 0 ] = toupper( str[ 0 ] );
   dispatch_ptr->class_quark = XrmStringToQuark( str );

   dispatch_ptr->handler_proc         = handler_proc;
   dispatch_ptr->control              = control;
   dispatch_ptr->should_dispatch_proc = should_dispatch_proc;

   dispatch_ptr->next = package_data.dispatch_ptr;
   package_data.dispatch_ptr = dispatch_ptr;

   XmsgUNLOCK_PROCESS( "Add new handler, toupper" );

   XtFree( str );

   return handler_proc_old;
}


#if NeedFunctionPrototypes
void XmsgAddDisplay( 
   Display* display_ptr )
#else
void XmsgAddDisplay( display_ptr )
   Display* display_ptr;
#endif
{
   XmsgDisplayListStruct* display_list_ptr;
   char**                 name_str_table;
   Bool                   found;
   int                    offset;

   /*---------------------------------------------------------------
   -- Lock process here to prevent two of the same display entries 
   -- from being added
   ---------------------------------------------------------------*/
   XmsgLOCK_PROCESS( "Add display" );

   for (display_list_ptr = package_data.display_list_ptr;
        display_list_ptr;
        display_list_ptr = display_list_ptr->next_ptr)
   {
      if (display_list_ptr->display_ptr == display_ptr)
      {
         break;
      }
   }   

   if (!display_list_ptr)
   {
      display_list_ptr = XtNew( XmsgDisplayListStruct );

      display_list_ptr->next_ptr = NULL;
      display_list_ptr->display_ptr = display_ptr;

      name_str_table = 
         XListExtensions( display_ptr, &display_list_ptr->extension_total );

      if (display_list_ptr->extension_total)
      {
         display_list_ptr->extension_ptr = 
            (XmsgServerExtensionStruct*) XtMalloc( 
               sizeof( XmsgServerExtensionStruct ) * 
               display_list_ptr->extension_total );

         for (offset = 0; offset < display_list_ptr->extension_total; offset++)
         {
            display_list_ptr->extension_ptr[ offset ].name_str =
               XtNewString( name_str_table[ offset ] );

            found = XQueryExtension( 
               display_ptr, name_str_table[ offset ],
               &display_list_ptr->extension_ptr[ offset ].major_opcode,
               &display_list_ptr->extension_ptr[ offset ].first_event,
               &display_list_ptr->extension_ptr[ offset ].first_error );

            if (!found)
            {
               display_list_ptr->extension_ptr[ offset ].major_opcode = 0;
            }
         }
         XFreeExtensionList( name_str_table );
      }
      else
      {
         display_list_ptr->extension_ptr = NULL;
      }
      
      display_list_ptr->next_ptr = package_data.display_list_ptr;
      package_data.display_list_ptr = display_list_ptr;
   }

   XmsgUNLOCK_PROCESS( "Add display" );
}


#if NeedFunctionPrototypes
void XmsgAppDefaultXtHandlers(
   XtAppContext app_context,
   unsigned int control )
#else
void XmsgAppDefaultXtHandlers( app_context, control )
   XtAppContext app_context;
   unsigned int control;
#endif
{
   if (control & XmsgCONTROL_INSTALL_XTERRORMSG)
   {
      XtAppSetErrorMsgHandler( app_context, XmsgXtErrorMsgHandler );
   }
   if (control & XmsgCONTROL_INSTALL_XTERRORLL)
   {
      XtAppSetErrorHandler( app_context, XmsgXtErrorHandler );
   }
   if (control & XmsgCONTROL_INSTALL_XTWARNINGMSG)
   {
      XtAppSetWarningMsgHandler( app_context, XmsgXtWarningMsgHandler );
   }
   if (control & XmsgCONTROL_INSTALL_XTWARNINGLL)
   {
      XtAppSetWarningHandler( app_context, XmsgXtWarningHandler );
   }
   if (control & XmsgCONTROL_INSTALL_XERROR)
   {
      XSetErrorHandler( XmsgXErrorHandler );
   }
   if (control & XmsgCONTROL_INSTALL_XIO)
   {
      XSetIOErrorHandler( XmsgXIOErrorHandler );
   }
}


#if NeedFunctionPrototypes
Boolean XmsgAppDirectOverrideErrorDB(
   XtAppContext	     app_context,
   _XmsgConst _XmsgString value_str )
#else
Boolean XmsgAppDirectOverrideErrorDB( app_context, value_str )
   XtAppContext	     app_context;
   _XmsgConst _XmsgString value_str;
#endif
{
   XrmDatabase* error_db_dest_ptr;
   XrmDatabase  error_db_src;
   Boolean      result = True;

   error_db_dest_ptr = XtAppGetErrorDatabase( app_context );
   if (error_db_dest_ptr)
   {
      error_db_src = XrmGetStringDatabase( value_str );
      if (error_db_src)
      {
         XrmMergeDatabases( error_db_src, error_db_dest_ptr );
      }
      else
      {
         result = False;
      }
   }
   else
   {
      result = False;
   }
   return result;
}


#if NeedFunctionPrototypes
Boolean XmsgAppMergeErrorDB(
   XtAppContext      app_context,
   _XmsgConst _XmsgString database_name_str )
#else
Boolean XmsgAppMergeErrorDB( app_context, database_name_str )
   XtAppContext      app_context;
   _XmsgConst _XmsgString database_name_str;
#endif
{
   XrmDatabase* error_db_dest_ptr;
   XrmDatabase  error_db_src;
   Boolean      result = True;

   error_db_dest_ptr = XtAppGetErrorDatabase( app_context );
   if (error_db_dest_ptr)
   {
      error_db_src = XrmGetFileDatabase( database_name_str );
      if (error_db_src)
      {
         XrmMergeDatabases( error_db_src, error_db_dest_ptr );
      }
      else
      {
         result = False;
      }
   }
   else
   {
      result = False;
   }
   return result;
}


#if NeedFunctionPrototypes
XmsgMessageStruct* XmsgCopyMessage(
   XmsgMessageStruct* message_ptr )
#else
XmsgMessageStruct* XmsgCopyMessage( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   XmsgMessageStruct* new_message_ptr;
   int                i;

   new_message_ptr = XtNew( XmsgMessageStruct );

   new_message_ptr->category_str   = XtNewString( message_ptr->category_str   );
   new_message_ptr->name_str       = XtNewString( message_ptr->name_str       );
   new_message_ptr->type_str       = XtNewString( message_ptr->type_str       );
   new_message_ptr->class_str      = XtNewString( message_ptr->class_str      );
   new_message_ptr->source_str     = XtNewString( message_ptr->source_str     );
   new_message_ptr->source_line    =              message_ptr->source_line     ;
   new_message_ptr->error_number   =              message_ptr->error_number    ;
   new_message_ptr->client_data    =              NULL                         ;
   new_message_ptr->position       =              message_ptr->position        ;
   new_message_ptr->text_str       = XtNewString( message_ptr->text_str       );
   new_message_ptr->widget         =              message_ptr->widget          ;
   new_message_ptr->app_context    =              message_ptr->app_context     ;
   new_message_ptr->database_ptr   =              message_ptr->database_ptr    ;
   new_message_ptr->local_time_str = XtNewString( message_ptr->local_time_str );
   new_message_ptr->must_dispatch  =              message_ptr->must_dispatch   ;
   new_message_ptr->no_x_protocol  =              message_ptr->no_x_protocol   ;
   new_message_ptr->true_display_ptr =            message_ptr->true_display_ptr;
   new_message_ptr->instance_counter =            message_ptr->instance_counter;

   for (i = 0; i < XmsgQTABLE_MAX; i++)
   {
      new_message_ptr->name_quark_table [i] = message_ptr->name_quark_table [i];
      new_message_ptr->class_quark_table[i] = message_ptr->class_quark_table[i];
   }
   new_message_ptr->free_memory = True;

   return new_message_ptr;
}


#if NeedFunctionPrototypes
Boolean XmsgCvtStringToBoolean(
   _XmsgConst _XmsgString str,
#  if NeedWidePrototypes
      int               default_value
#  else
      Boolean           default_value 
#  endif
)
#else
Boolean XmsgCvtStringToBoolean( str, default_value )
   _XmsgConst _XmsgString str;
   Boolean           default_value;
#endif
{
   String tmp_str;
   Boolean result;
   int   str_len;
   int i;

   if (str)
   {
      tmp_str = XtNewString( str );
      str_len = strlen( str );

      /*---------------------------------------------------
      -- tolower is not thread safe if setlocal() is used
      ---------------------------------------------------*/
      XmsgLOCK_PROCESS( "tolower may not be thread safe" );

      for (i = 0; i < str_len; i++)
      {
         tmp_str[ i ] = (char) tolower( (int) tmp_str[ i ] );
      }
      XmsgUNLOCK_PROCESS( "tolower may not be thread safe" );

      if ((strcmp( tmp_str, "true" ) == 0) ||
          (strcmp( tmp_str, "yes"  ) == 0) ||
          (strcmp( tmp_str, "on"   ) == 0) ||
          (strcmp( tmp_str, "1"    ) == 0))
      {
         result = True;
      }
      else if ((strcmp( tmp_str, "false" ) == 0) ||
               (strcmp( tmp_str, "no"    ) == 0) ||
               (strcmp( tmp_str, "off"   ) == 0) ||
               (strcmp( tmp_str, "0"     ) == 0))
      {
         result = False;
      }
      else
      {
         result = default_value;
         fprintf( stderr, "Xmsg: Invalid Boolean \"%s\" found in error "
                  "database; %s assumed\n", str, 
                  default_value ? "True" : "False" );
      }
      XtFree( tmp_str );
   }
   else
   {
      result = default_value;
   }
   return result;
}


#if NeedFunctionPrototypes
void XmsgHandlerStderr(
   XmsgMessageStruct* message_ptr )
#else
void XmsgHandlerStderr( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   _XmsgConst _XmsgString msg_str;
   _XmsgConst _XmsgString prepend_str;
   char              buffer[ 2000 ];

   msg_str = XmsgGetMsgText( message_ptr );

   prepend_str = XmsgGetMsgDatabaseString( 
      message_ptr,
      package_data.stderr_prepend_name_quark,
      package_data.stderr_prepend_class_quark );

   if (prepend_str)
   {
      XmsgTranslate( message_ptr, prepend_str, buffer );
      fprintf( stderr, "%s%s\n", buffer, msg_str );
   }
   else
   {
      fprintf( stderr, "%s\n", msg_str );
   }
}


#if NeedFunctionPrototypes
void XmsgExit( void )
#else
void XmsgExit(  )
#endif
{
   package_data.exit_proc();
}


#if NeedFunctionPrototypes
void XmsgFreeMessage(
   XmsgMessageStruct* message_ptr )
#else
void XmsgFreeMessage( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   if (message_ptr && message_ptr->free_memory)
   {
      XtFree( (char*) message_ptr->category_str   );
      XtFree( (char*) message_ptr->name_str       );
      XtFree( (char*) message_ptr->type_str       );
      XtFree( (char*) message_ptr->class_str      );
      XtFree( (char*) message_ptr->source_str     );
      XtFree( (char*) message_ptr->text_str       );
      XtFree( (char*) message_ptr->local_time_str );

      XtFree( (char*) message_ptr );
   }
}


#if NeedFunctionPrototypes
XtAppContext XmsgGetMsgAppContext(
   XmsgMessageStruct* message_ptr )
#else
XtAppContext XmsgGetMsgAppContext( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->app_context;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgApplClass(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgApplClass( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return package_data.app_class_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgApplName(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgApplName( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return package_data.app_name_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgDatabaseString(
   XmsgMessageStruct* message_ptr,
   XrmQuark           name_quark,
   XrmQuark           class_quark )
#else
_XmsgConst _XmsgString XmsgGetMsgDatabaseString( message_ptr, name_quark, 
                                           class_quark )
   XmsgMessageStruct* message_ptr;
   XrmQuark           name_quark;
   XrmQuark           class_quark;
#endif
{
   Boolean             found_resource;
   XrmRepresentation   rep_type;
   XrmValue            value;
   _XmsgConst _XmsgString   result_str;

   message_ptr->name_quark_table [XmsgQRES] = name_quark;
   message_ptr->class_quark_table[XmsgQRES] = class_quark;

   found_resource = XrmQGetResource( *message_ptr->database_ptr,
                                     message_ptr->name_quark_table, 
                                     message_ptr->class_quark_table,
                                     &rep_type, &value );
   if (found_resource)
   {
      result_str = (char*) value.addr;
   }
   else
   {
      result_str = NULL;
   }
   return result_str;
}


#if NeedFunctionPrototypes
int XmsgGetMsgErrorNumber(
   XmsgMessageStruct* message_ptr )
#else
int XmsgGetMsgErrorNumber( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->error_number;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgClass(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgClass( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->class_str;
}


#if NeedFunctionPrototypes
_XmsgConst XtPointer XmsgGetMsgClientData(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst XtPointer XmsgGetMsgClientData( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->client_data;
}


#if NeedFunctionPrototypes
Boolean XmsgGetMsgDelayProtocol(
   XmsgMessageStruct* message_ptr )
#else
Boolean XmsgGetMsgDelayProtocol( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->no_x_protocol;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgHost(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgHost( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return package_data.host_name_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgLocalTime(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgLocalTime( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->local_time_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgName(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgName( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->name_str;
}


#if NeedFunctionPrototypes
XmsgPosition XmsgGetMsgPosition(
   XmsgMessageStruct* message_ptr )
#else
XmsgPosition XmsgGetMsgPosition( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->position;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgCategory(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgCategory( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->category_str;
}


#if NeedFunctionPrototypes
unsigned int XmsgGetMsgSourceLine(
   XmsgMessageStruct* message_ptr )
#else
unsigned int XmsgGetMsgSourceLine( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->source_line;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgSourceName(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgSourceName( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->source_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgText(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgText( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->text_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgType(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgType( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->type_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgUserLogin(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgUserLogin( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return package_data.login_name_str;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgUserName(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgUserName( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return package_data.login_gecos_str;
}


#if NeedFunctionPrototypes
Widget XmsgGetMsgWidget(
   XmsgMessageStruct* message_ptr )
#else
Widget XmsgGetMsgWidget( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return message_ptr->widget;
}


#if NeedFunctionPrototypes
_XmsgConst _XmsgString XmsgGetMsgXDisplayServer(
   XmsgMessageStruct* message_ptr )
#else
_XmsgConst _XmsgString XmsgGetMsgXDisplayServer( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   return DisplayString( message_ptr->true_display_ptr );
}


#if NeedFunctionPrototypes
void XmsgInitialize(
   Widget       w )
#else
void XmsgInitialize( w )
   Widget       w;
#endif
{
   XtAppContext   app_context;
   struct passwd* user_info_ptr;

   XmsgLOCK_PROCESS( "Initializing" );

   XmsgAddDisplay( XtDisplay( w ) );

   /*------------------------------------------------------------------
   -- Convert the widget to the toplevel shell to minimize the impact
   -- of widgets being destroyed.
   ------------------------------------------------------------------*/
   while (XtParent( w ))
   {
      w = XtParent( w );
   }
   package_data.default_widget = w;

   if (!package_data.installed)
   {
      XtGetApplicationNameAndClass( XtDisplay( w ),
                                    &package_data.app_name_str,
                                    &package_data.app_class_str );

      package_data.app_name_quark = 
         XrmStringToQuark( package_data.app_name_str );

      package_data.app_class_quark =
         XrmStringToQuark( package_data.app_class_str );

      package_data.append_name_quark   = XrmPermStringToQuark( "xmsgAppend"  );
      package_data.append_class_quark  = XrmPermStringToQuark( "XmsgAppend"  );
      package_data.prepend_name_quark  = XrmPermStringToQuark( "xmsgPrepend" );
      package_data.prepend_class_quark = XrmPermStringToQuark( "XmsgPrepend" );
      package_data.text_name_quark     = XrmPermStringToQuark( "xmsgText"    );
      package_data.text_class_quark    = XrmPermStringToQuark( "XmsgText"    );

      package_data.stderr_prepend_name_quark =
         XrmPermStringToQuark( "stderrPrepend" );

      package_data.stderr_prepend_class_quark =
         XrmPermStringToQuark( "StderrPrepend" );

      /*------------------------------------------------
      -- Get the name and gecos from the password file
      ------------------------------------------------*/
      user_info_ptr = getpwuid( getuid() );
      if (user_info_ptr)
      {
         package_data.login_name_str  = XtNewString( user_info_ptr->pw_name  );
         package_data.login_gecos_str = XtNewString( user_info_ptr->pw_gecos );
      }
      else
      {
         /*--------------------------------------------------------------------
         -- Make copies in case functions are added later to free this memory
         --------------------------------------------------------------------*/
         package_data.login_name_str  = XtNewString( "unknown user" );
         package_data.login_gecos_str = XtNewString( "unknown user" );
      }

      /*----------------------------------------------------------------
      -- Get the host name.  This also provides the O/S name, however,
      -- that information is currently not been retrieved.
      ----------------------------------------------------------------*/
      {
         struct utsname sys_info;
         int            status;

         status = uname( &sys_info );
         if (status != -1)
         {
            package_data.host_name_str   = XtNewString( sys_info.nodename );
            package_data.sys_name_str    = XtNewString( sys_info.sysname );
            package_data.sys_release_str = XtNewString( sys_info.release );
            package_data.sys_version_str = XtNewString( sys_info.version );
            package_data.machine_str     = XtNewString( sys_info.machine );
         }
      }

      package_data.installed = True;
   }
   XmsgUNLOCK_PROCESS( "Initializing" );
}


#if NeedFunctionPrototypes
void XmsgInstall( 
   Widget       w,
   unsigned int control )
#else
void XmsgInstall( w, control )
   Widget       w;
   unsigned int control;
#endif
{
   package_data.dispatch_all = ((control & XmsgCONTROL_DISPATCH_ALL) != False);

   XmsgInitialize( w );

   XmsgAppDefaultXtHandlers( XtWidgetToApplicationContext( w ), control );

   if (control & XmsgCONTROL_INSTALL_STDERR)
   {
      XmsgAddHandler( "stderr", 
                      XmsgHandlerStderr, 
                      XmsgCONTROL_CALL_AS_DEFAULT,
                      NULL );
   }
}


#if NeedFunctionPrototypes
Boolean XmsgIsHandlerDefined(
   _XmsgConst _XmsgString name_str )
#else
Boolean XmsgIsHandlerDefined( name_str )
   _XmsgConst _XmsgString name_str;
#endif
{
   Boolean  defined;
   XrmQuark quark;

   quark   = XrmStringToQuark( name_str );
   defined = XmsgQIsHandlerDefined( quark );
   return defined;
}


#if NeedFunctionPrototypes
Boolean XmsgQIsHandlerDefined(
   XrmQuark name_quark )
#else
Boolean XmsgQIsHandlerDefined( name_quark )
   XrmQuark name_quark;
#endif
{
   XmsgDispatchStruct* dispatch_ptr;

   for (dispatch_ptr = package_data.dispatch_ptr; 
        dispatch_ptr != NULL;
        dispatch_ptr = dispatch_ptr->next)
   {
      if ((dispatch_ptr->name_quark  == name_quark) ||
          (dispatch_ptr->class_quark == name_quark))
      {
         return True;
      }
   }
   return False;
}


#if NeedFunctionPrototypes
XmsgHandlerProc XmsgQRemoveHandler(
   XrmQuark name_quark )
#else
XmsgHandlerProc XmsgQRemoveHandler( name_quark )
   XrmQuark name_quark;
#endif
{
   XmsgDispatchStruct* dispatch_ptr;
   XmsgDispatchStruct* prev_dispatch_ptr;
   XmsgHandlerProc     handler_proc = NULL;

   XmsgLOCK_PROCESS( "Removing a handler" );

   if (package_data.dispatch_ptr)
   {
      dispatch_ptr = package_data.dispatch_ptr; 

      if ((dispatch_ptr->name_quark  == name_quark) ||
          (dispatch_ptr->class_quark == name_quark))
      {
         handler_proc = dispatch_ptr->handler_proc;
         package_data.dispatch_ptr = dispatch_ptr->next;
         XtFree( (char*) dispatch_ptr );
      }
      else
      {
         prev_dispatch_ptr = dispatch_ptr;
         dispatch_ptr = dispatch_ptr->next; 

         while (dispatch_ptr != NULL)
         {
            if ((dispatch_ptr->name_quark  == name_quark) ||
                (dispatch_ptr->class_quark == name_quark))
            {
               handler_proc = dispatch_ptr->handler_proc;
               prev_dispatch_ptr->next = dispatch_ptr->next;
               XtFree( (char*) dispatch_ptr );
               break;
            }
            prev_dispatch_ptr = dispatch_ptr;
            dispatch_ptr = dispatch_ptr->next; 
         }
      }
   }

   XmsgUNLOCK_PROCESS( "Removing a handler" );

   return handler_proc;
}


#if NeedFunctionPrototypes
XmsgHandlerProc XmsgRemoveHandler(
   _XmsgConst _XmsgString name_str )
#else
XmsgHandlerProc XmsgRemoveHandler( name_str )
   _XmsgConst _XmsgString name_str;
#endif
{
   Boolean         defined;
   XrmQuark        quark;
   XmsgHandlerProc handler_proc = NULL;

   quark   = XrmStringToQuark( name_str );
   defined = XmsgQIsHandlerDefined( quark );

   if (defined)
   {
      handler_proc = XmsgQRemoveHandler( quark );
   }
   return handler_proc;
}


#if NeedFunctionPrototypes
XmsgExitProc XmsgSetExitHandler(
   XmsgExitProc exit_proc )
#else
XmsgExitProc XmsgSetExitHandler( exit_proc )
   XmsgExitProc exit_proc;
#endif
{
   XmsgExitProc old_exit_proc;

   if (!exit_proc)
   {
      exit_proc = XmsgDoExit;
   }

   XmsgLOCK_PROCESS( "Set exit handler" );

   old_exit_proc = package_data.exit_proc;
   package_data.exit_proc = exit_proc;

   XmsgUNLOCK_PROCESS( "Set exit handler" );

   return old_exit_proc;
}


#if NeedFunctionPrototypes
Boolean XmsgShouldPopup(
   XmsgMessageStruct* message_ptr )
#else
Boolean XmsgShouldPopup( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   if (strcmp( message_ptr->category_str, XmsgCATEGORY_FATAL ) == 0)
   {
      return True;
   }
   return False;
}


#if NeedFunctionPrototypes
void XmsgTranslate(
   XmsgMessageStruct* message_ptr,
   _XmsgConst _XmsgString  buffer_in_str,
   String             buffer_out_str )
#else
void XmsgTranslate( message_ptr, buffer_in_str, buffer_out_str )
   XmsgMessageStruct* message_ptr;
   _XmsgConst _XmsgString  buffer_in_str;
   String             buffer_out_str;
#endif
{
   String            dest_ptr = buffer_out_str;
   _XmsgConst _XmsgString src_ptr  = buffer_in_str;

   for (; *src_ptr; src_ptr++)
   {
      if (*src_ptr == '@') 
      {
         src_ptr++;
         switch( *src_ptr )
         {
         case '@':
            /*--------------------
            -- The '@' character
            --------------------*/
            dest_ptr[0] = '@';
            dest_ptr[1] = '\0';
            break;

         case 'A':
            /*--------------------
            -- Application class
            --------------------*/
            strcpy( dest_ptr, package_data.app_class_str );
            break;

         case 'a':
            /*-------------------
            -- Application name
            -------------------*/
            strcpy( dest_ptr, package_data.app_name_str );
            break;

         case 'C':
            /*-------------------
            -- Source code name
            -------------------*/
            strcpy( dest_ptr, message_ptr->source_str );
            break;

         case 'c':
            /*----------------
            -- Message class
            ----------------*/
            strcpy( dest_ptr, message_ptr->class_str );
            break;

         case 'D':
            /*------------------------
            -- X Server Display name
            ------------------------*/
            strcpy( dest_ptr, DisplayString( message_ptr->true_display_ptr ) );
            break;

         case 'E':
            /*-------------
            -- Error text
            -------------*/
            if (message_ptr->error_number)
            {
               _XmsgConst _XmsgString error_text_str;

               error_text_str = strerror( message_ptr->error_number );
               if (error_text_str)
               {
                  strcpy( dest_ptr, error_text_str );
               }
               else
               {
                  strcpy( dest_ptr, "(The error number is out of range)" );
               }
            }
            else
            {
               strcpy( dest_ptr, "No error" );
            }
            break;

         case 'e':
            /*---------------
            -- Error number
            ---------------*/
            sprintf( dest_ptr, "%d", message_ptr->error_number );
            break;

         case 'G':
         case 'u':
            strcpy( dest_ptr, package_data.login_gecos_str );
            break;

         case 'H':
            /*------------
            -- Host name
            ------------*/
            strcpy( dest_ptr, package_data.host_name_str );
            break;

         case 'L':
            /*-------------------
            -- Source code line
            -------------------*/
            sprintf( dest_ptr, "%u", message_ptr->source_line );
            break;

         case 'M':
            /*---------------
            -- Machine type
            ---------------*/
            strcpy( dest_ptr, package_data.machine_str );
            break;

         case 'm':
            /*-------------------
            -- Message category
            -------------------*/
            strcpy( dest_ptr, message_ptr->category_str );
            break;

         case 'n':
            /*---------------
            -- Message name
            ---------------*/
            strcpy( dest_ptr, message_ptr->name_str );
            break;

         case 'P':
            /*---------------------
            -- Process Identifier
            ---------------------*/
            sprintf( dest_ptr, "%ld", getpid() );
            break;

         case 'R':
            /*---------------------------
            -- Operating system release
            ---------------------------*/
            strcpy( dest_ptr, package_data.sys_release_str );
            break;

         case 'S':
            /*-------------------
            -- Operating system
            -------------------*/
            strcpy( dest_ptr, package_data.sys_name_str );
            break;

         case 'T':
            /*---------------
            -- Current time
            ---------------*/
            strcpy( dest_ptr, message_ptr->local_time_str );
            break;
                      
         case 't':
            /*---------------
            -- Message type
            ---------------*/
            strcpy( dest_ptr, message_ptr->type_str );
            break;

         case 'U':
            /*-------------
            -- Login name
            -------------*/
            strcpy( dest_ptr, package_data.login_name_str );
            break;

         case 'V':
            /*---------------------------
            -- Operating system version
            ---------------------------*/
            strcpy( dest_ptr, package_data.sys_version_str );
            break;

         case '#':
            /*--------------------------
            -- Message instance number
            --------------------------*/
            sprintf( dest_ptr, "%lu", message_ptr->instance_counter );
            break;

         default:
            dest_ptr[0] = '@';
            dest_ptr[1] = *src_ptr;
            dest_ptr[2] = '\0';
         }
         dest_ptr += strlen( dest_ptr );
      }
      else
      {
         *dest_ptr = *src_ptr;
         dest_ptr++;
      }
   }
   *dest_ptr = '\0';
}


#if NeedFunctionPrototypes
void Xmsg_Debug(
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Debug( w, name_str, type_str, class_str, default_str,
                 params_str_table, params_str_count, source_str,
                 source_line, error_number, client_data, position )
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   Xmsg_Generic( XmsgCATEGORY_DEBUG, w, name_str, type_str, class_str, 
                 default_str, params_str_table, params_str_count, source_str, 
                 source_line, error_number, client_data, position );
}

#if NeedFunctionPrototypes
void Xmsg_Fatal(
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Fatal( w, name_str, type_str, class_str, default_str,
                 params_str_table, params_str_count, source_str,
                 source_line, error_number, client_data, position )
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   Xmsg_Generic( XmsgCATEGORY_FATAL, w, name_str, type_str, class_str, 
                 default_str, params_str_table, params_str_count, source_str, 
                 source_line, error_number, client_data, position );
   XmsgExit();
   exit( -2 );
}


#if NeedFunctionPrototypes
void Xmsg_Error(
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Error( w, name_str, type_str, class_str, default_str,
                 params_str_table, params_str_count, source_str,
                 source_line, error_number, client_data, position )
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   Xmsg_Generic( XmsgCATEGORY_ERROR, w, name_str, type_str, class_str, 
                 default_str, params_str_table, params_str_count, source_str, 
                 source_line, error_number, client_data, position );
}


#if NeedFunctionPrototypes
void Xmsg_Generic(
   _XmsgConst _XmsgString category_str,
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Generic( category_str, w, name_str, type_str, class_str, default_str,
                   params_str_table, params_str_count, source_str,
                   source_line, error_number, client_data, position )
   _XmsgConst _XmsgString category_str;
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   char   buffer1_str[ XmsgMESSAGE_MAX_SIZE ];
   char   buffer2_str[ XmsgMESSAGE_MAX_SIZE ];
   String str_table  [ XmsgPARAMS_MAX       ];
   int    counter; 
   char   local_time_str[ XmsgTIME_MAX_SIZE ];

   XmsgMessageStruct message;

   /*---------------------------------------------
   -- Assign the timestamp buffer in the message
   ---------------------------------------------*/
   message.local_time_str = local_time_str;

   /*--------------------------------------------------------------------
   -- Call the install function if the package does not "appear" to be
   -- installed.  Another thread may be installing currently, but that
   -- will be checked later.  If the package is not installed, w 
   -- will not be the NULL sent by the X/Xt handlers since they cannot
   -- possibly be installed yet
   --------------------------------------------------------------------*/
   if (!package_data.installed)
   {
      XmsgInitialize( w );
   }

   /*----------------------
   -- Package the message
   ----------------------*/
   XmsgSetupMessage( &message, category_str, w, name_str, type_str,
                     class_str, source_str, source_line, error_number,
                     client_data, position );

   /*-----------------------------------------------------
   -- Get the basic message text from the error database
   -----------------------------------------------------*/
   message.text_str = buffer1_str;
   XmsgGetErrorDatabaseText( &message, default_str, XmsgMESSAGE_MAX_SIZE );

   /*---------------------------------------------------------------
   -- Add in the printf style substitutions -- this implementation
   -- is questionable, but is at least as good as what is offered
   -- by the X11R6 equivalent
   ---------------------------------------------------------------*/
   if (params_str_table && params_str_count)
   {
      if (params_str_count > XmsgPARAMS_MAX)
      {
         String params[2];

         params[0] = (String) params_str_count;
         params[1] = (String) XmsgPARAMS_MAX;
         XmsgWarning( w, "xmsg", "tooManyParams", "Xmsg",
                      "Message being processed has %d parameters (max %d); "
                      "this may cause a serious problem.", params, 2, NULL );
         params_str_count = XmsgPARAMS_MAX;
      }

      /*----------------------------------------------------------------
      -- Copy the specified values; the remainder are holes which need
      -- to be plugged
      ----------------------------------------------------------------*/
      memcpy( (void*) str_table, (void*) params_str_table, 
               params_str_count * sizeof( String ) );

      /*----------------------------------------------------------------------
      -- Loop to plug the holes instead of using memset() since the internal
      -- representation of a NULL pointer and a 0 are not necessarily
      -- the same
      ----------------------------------------------------------------------*/
      for (counter = params_str_count; counter < XmsgPARAMS_MAX; counter++)
      {
         str_table[ counter ] = NULL;
      }

      sprintf( buffer2_str , buffer1_str , str_table[0], str_table[1], 
               str_table[2], str_table[3], str_table[4], str_table[5], 
               str_table[6], str_table[7], str_table[8], str_table[9] );
   }
   else
   {
      strcpy( buffer2_str, buffer1_str );
   }

   XmsgAugmentText( &message, buffer2_str );

   message.text_str = buffer2_str;

   XmsgTranslate( &message, buffer1_str, buffer2_str ); 

   XmsgDispatch( &message );
}


#if NeedFunctionPrototypes
void Xmsg_Info(
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Info( w, name_str, type_str, class_str, default_str,
                params_str_table, params_str_count, source_str,
                source_line, error_number, client_data, position )
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   Xmsg_Generic( XmsgCATEGORY_INFO, w, name_str, type_str, class_str, 
                 default_str, params_str_table, params_str_count, source_str, 
                 source_line, error_number, client_data, position );
}

#if NeedFunctionPrototypes
void Xmsg_Warning(
   Widget            w,
   _XmsgConst _XmsgString name_str,
   _XmsgConst _XmsgString type_str,
   _XmsgConst _XmsgString class_str,
   _XmsgConst _XmsgString default_str,
   String*           params_str_table,
   Cardinal          params_str_count,
   _XmsgConst _XmsgString source_str,
   unsigned int      source_line,
   int               error_number,
   XtPointer         client_data,
   XmsgPosition      position )
#else
void Xmsg_Warning( w, name_str, type_str, class_str, default_str,
                   params_str_table, params_str_count, source_str,
                   source_line, error_number, client_data, position )
   Widget            w;
   _XmsgConst _XmsgString name_str;
   _XmsgConst _XmsgString type_str;
   _XmsgConst _XmsgString class_str;
   _XmsgConst _XmsgString default_str;
   String*           params_str_table;
   Cardinal          params_str_count;
   _XmsgConst _XmsgString source_str;
   unsigned int      source_line;
   int               error_number;
   XtPointer         client_data;
   XmsgPosition      position;
#endif
{
   Xmsg_Generic( XmsgCATEGORY_WARNING, w, name_str, type_str, class_str, 
                 default_str, params_str_table, params_str_count, source_str, 
                 source_line, error_number, client_data, position );
}

