/*----------------------------------------------------------------------------
//
//  Module:           xmubCustUtil
//
//  Project:          <>
//  System:           <>
//    Subsystem:      <>
//    Function block: <>
//
//  Description:
//    <>
//
//  Contents:
//    <>
//
//  Filename:         xmubCustUtil.c
//
//  Author:           Ulrika Bornetun
//  Creation date:    13.10.1994 
//
//  Modifications:
//
//  Version / When / Who
//  What
//  --------------------------------------------
//  %StartLog%
//  %EndLog%
//
//
//  Copyright 1994 by Union Bank of Switzerland.
//
//  Permission to use, copy, modify, and distribute this software and its
//  documentation for any purpose and without fee is hereby granted,
//  provided that the above copyright notice appear in all copies.
//  Union Bank of Switzerland makes no representations about the usability
//  of this software for any purpose. It is provided "as is" without express
//  or implied warranty.
//--------------------------------------------------------------------------*/

/* SCCS module identifier. */
#ifndef lint
static char SCCSID[] = "%Z% Module: %M%, Ver: %I%, Date: %E% %U%";
#endif

/*----------------------------------------------------------------------------
//  Include files
----------------------------------------------------------------------------*/

#ifdef VMS
#include ctype
#include stdlib
#include stdio
#include string
#include "xmubCustUtil.h"

#else
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <xmub/xmubCustUtil.h>

#endif


#include <X11/Shell.h>

/*----------------------------------------------------------------------------
//  Macro definitions
----------------------------------------------------------------------------*/

/* Definitions used for the mapping file. */
#define RESMAP_FILE_NAME       "xresource.map"
#define MAP_FILE_NAME          "xresource"
#define MAP_FILE_SUFFIX        ".map"
#define MAP_FILE_DIRTYPE       "app-defaults"
#define XRESOURCE_MAP_FILE     "XRESOURCE_MAP_FILE"

/* Defaults. */
#define DEFAULT_DELIMITERS     " "

/* "Functions" */
#define Max( a, b )  ( ( a ) > ( b ) ? ( a ) : ( b ) )

#ifdef VMS
#define unlink( filename )  delete( filename )
#endif

#define BUFFER_LENGTH   200

#if XlibSpecificationRelease < 5
#define XrmEnumAllLevels 0
#define XrmEnumOneLevel  1
#endif

/*----------------------------------------------------------------------------
//  Type declarations
----------------------------------------------------------------------------*/

typedef struct 
{
  xmubResourceMappingInfoRef  info;
  XrmQuark                    keyword_quark;
  XrmQuark                    *sets;
  int                         num_sets;
  int                         set_list_length;
  
} ListClassesInfo;

#if XlibSpecificationRelease < 5
typedef void* XPointer;
#endif

typedef Bool (*LoopDatabaseProc)
( XrmDatabase*, XrmBindingList, XrmQuarkList, XrmRepresentation*,
  XrmValue*, XPointer );

/*
// The parameters for the LoopDatabaseProc function are the same as for
// the function called by XrmEnumerateDatabase in X11R5.
*/
  

/*----------------------------------------------------------------------------
//  Global definitions
----------------------------------------------------------------------------*/

XrmOptionDescRec  xmub_options[] = {
  { "-palette", "*palette", XrmoptionSepArg, NULL },
  { "-fmap",    "*fmap",    XrmoptionSepArg, NULL },
};

Cardinal  xmub_num_options = XtNumber( xmub_options );

xmubAppResDesc  xmub_resource_sets[] = {
  {  "palette", "Palette" },
  {  "fmap",    "Fmap" } };

Cardinal  xmub_num_resource_sets = XtNumber( xmub_resource_sets );

/*----------------------------------------------------------------------------
//  Function prototypes
----------------------------------------------------------------------------*/


static Bool
  ExtractKeywordClass( XrmDatabase        *db,
                       XrmBindingList     bindings,
                       XrmQuarkList       quarks,
                       XrmRepresentation  *type,
                       XrmValue           *value,
                       XPointer           closure );

static Boolean
  FileReadAccess( char*  file_name );

static void
  GetResourceValues( xmubResourceMappingInfoRef  info,
                     xmubAppResDesc              resource_sets[],
                     Cardinal                    num_resource_sets,
                     char**                      values );

static Bool
  InsertGeneralMappingResources( XrmDatabase        *db,
                                 XrmBindingList     bindings,
                                 XrmQuarkList       quarks,
                                 XrmRepresentation  *type,
                                 XrmValue           *value,
                                 XPointer           closure );

static Bool
  InsertSpecificMappingResources( XrmDatabase        *db,
                                  XrmBindingList     bindings,
                                  XrmQuarkList       quarks,
                                  XrmRepresentation  *type,
                                  XrmValue           *value,
                                  XPointer           closure );

static Bool
  LoopThroughDatabase( XrmDatabase       db,
                       XrmNameList       name_prefix,
                       XrmClassList      class_prefix,
                       int               unsupported,
                       LoopDatabaseProc  proc,
                       XPointer          client_data );
  /*
  // LoopThroughDatabase provides a simple XrmEnumerateDatabase
  // functionality for X11R4.
  // name_list and class_list only support lists of length 1.
  */

static Bool
  ModifyKeywordResource( XrmDatabase        *db,
                         XrmBindingList     bindings,
                         XrmQuarkList       quarks,
                         XrmRepresentation  *type,
                         XrmValue           *value,
                         XPointer           closure );

static Bool
  ModifySubstringResource( XrmDatabase        *db,
                           XrmBindingList     bindings,
                           XrmQuarkList       quarks,
                           XrmRepresentation  *type,
                           XrmValue           *value,
                           XPointer           closure );

static int
  StringComp( const void  *s1,
              const void  *s2 );



/*----------------------------------------------------------------------------
//  Functions
----------------------------------------------------------------------------*/

static Bool
  ExtractKeywordClass( XrmDatabase        *db,
                       XrmBindingList     bindings,
                       XrmQuarkList       quarks,
                       XrmRepresentation  *type,
                       XrmValue           *value,
                       XPointer           closure )
{

  int              index;
  int              length;
  ListClassesInfo  *listinfo;
  
  /* Code. */

  listinfo = (ListClassesInfo *) closure;

  /* To be valid, we must have 2 quarks, and the 2nd must match our keyword.
     In that case, the 1st is our candidate. */
  for( length = 0; quarks[ length ] != NULLQUARK; length ++ );
  if( length != 2 )
    return( False );

  if( quarks[ 1 ] == listinfo -> keyword_quark ){
    /* Insert the first one in the list. */

    /* Check for duplicates. */
    for( index = 0; index < listinfo -> num_sets; index ++ ){
      if( quarks[ 0 ] == listinfo -> sets[ index ] )
        return( False );
    }

    /* We can insert it. */
    if( listinfo -> num_sets == listinfo -> set_list_length ){
      ( listinfo -> set_list_length ) += 5;

      if( listinfo -> sets == NULL )
        listinfo -> sets = (XrmQuark *) XtMalloc(
          sizeof( XrmQuark ) * listinfo -> set_list_length );

      else
        listinfo -> sets = (XrmQuark *) XtRealloc(
          (char *) listinfo -> sets,
          sizeof( XrmQuark ) * listinfo -> set_list_length );
    }

    listinfo -> sets[ listinfo -> num_sets ] = quarks[ 0 ];
    ( listinfo -> num_sets ) ++;

  }
  
  
  return( False );
  
}


/*----------------------------------------------------------------------*/

static Boolean
  FileReadAccess( char*  file_name )
{
  /* This function is a replacement for the POSIX function access. */
  FILE*  file;
  
  file = fopen( file_name, "r" );
  
  if( file == NULL )
    return False;

  fclose( file );
  
  return True;
  
}


/*----------------------------------------------------------------------*/

static void
  GetResourceValues( xmubResourceMappingInfoRef  info,
                     xmubAppResDesc              resource_sets[],
                     Cardinal                    num_resource_sets,
                     char**                      values )
{
  int       index;
  int       length;
  int       max_length;
  Bool      ok;
  char*     res_class;
  char*     res_name;
  char*     str_type_return;
  XrmValue  value_return;
  
  /* Code. */

  /* Check max length of strings. */
  for( index = 0, max_length = 0; index < num_resource_sets; index ++ ){

    /* Name. */
    if( resource_sets[ index ].res_name != NULL )
      length = strlen( resource_sets[ index ].res_name );
    else
      length = 0;

    max_length = Max( max_length, length );
   
    if( resource_sets[ index ].res_class != NULL )
      length = strlen( resource_sets[ index ].res_class );
    else
      length = 0;

    max_length = Max( max_length, length );

  }

  res_name  = XtMalloc( max_length + strlen( info -> app_name ) + 5 );
  res_class = XtMalloc( max_length + strlen( info -> app_class ) + 5 );
  

  for( index = 0; index < num_resource_sets; index ++ ){
    sprintf( res_name, "%s.%s", info -> app_name,
             resource_sets[ index ].res_name );
    sprintf( res_class, "%s.%s", info -> app_class,
             resource_sets[ index ].res_class );
    
    ok = XrmGetResource( XtDatabase( info -> display ), res_name, res_class, 
                         &str_type_return, &value_return );

   if( ok )
     values[ index ] = XtNewString( value_return.addr );
   else
     values[ index ] = NULL;
    
  }
  
  XtFree( res_name );
  XtFree( res_class );
  
}


/*----------------------------------------------------------------------*/

static Bool
  InsertGeneralMappingResources( XrmDatabase        *db,
                                 XrmBindingList     bindings,
                                 XrmQuarkList       quarks,
                                 XrmRepresentation  *type,
                                 XrmValue           *value,
                                 XPointer           closure )
{

  xmubResourceMappingInfoRef  info;
  
  /* Code. */

  info = (xmubResourceMappingInfoRef) closure;

  /* The resource should just be inserted in the translation database. */
  XrmQPutResource( &( info -> translation_db ), bindings,
                   quarks, *type, value );
  

  info -> selected = True;
  

  return( False );

}


/*----------------------------------------------------------------------*/

static Bool
  InsertSpecificMappingResources( XrmDatabase        *db,
                                  XrmBindingList     bindings,
                                  XrmQuarkList       quarks,
                                  XrmRepresentation  *type,
                                  XrmValue           *value,
                                  XPointer           closure )
{

  xmubResourceMappingInfoRef  info;
  int                         length;
  
  /* Code. */

  info = (xmubResourceMappingInfoRef) closure;

  /* Here we will get both general and specific resources. Select just
     the ones that are specific. */
  /* For specific resources, the quark list is longer than one item. */
  for( length = 0; quarks[ length ] != NULLQUARK; length ++ );

  if( length > 1 ){

    char  *qs;
    char  *res_spec;

    /* For the specific ones, we have to construct a string name. */
    qs = XrmQuarkToString( quarks[ 1 ] );

    res_spec = (char *) XtMalloc( strlen( qs ) + 5 );
    sprintf( res_spec, "*%s", qs );

    XrmPutResource( &( info -> translation_db ), res_spec,
                    XrmRepresentationToString( *type ), value );
    XtFree( res_spec );

    info -> selected = True;

  }

  
  return( False );

}

  
/*----------------------------------------------------------------------*/

static Bool
  LoopThroughDatabase( XrmDatabase       db,
                       XrmNameList       name_prefix,
                       XrmClassList      class_prefix,
                       int               unsupported,
                       LoopDatabaseProc  proc,
                       XPointer          client_data )
{

  XrmBindingList  binding_list;
  char            buffer[ BUFFER_LENGTH ];
  FILE            *db_file;
  char            db_filename[ L_tmpnam + 1 ];
  int             index;
  int             length;
  int             num_bindings;
  XrmQuarkList    quark_list;
  Boolean         selected;
  int             start_index;
  XrmQuark        string_rep;
  Bool            terminate = False;
  XrmValue        value;
  XrmQuark        wildcard_quark;
  
  /* Code. */

  /* Dump the database to a file. */
  (void) tmpnam( db_filename );
  XrmPutFileDatabase( db, db_filename );

  db_file = fopen( db_filename, "r" );
  
  string_rep     = XrmStringToRepresentation( "String" );
  wildcard_quark = XrmStringToQuark( "*" );
  
    
  /* Read the resource specifications. */
  while( fgets( buffer, BUFFER_LENGTH, db_file ) != NULL ){

    /* Split resource definition from value. */
    length = strcspn( buffer, ":" );

    if( length == 0 )
      continue;
    
    buffer[ length ] = '\0';

    for( num_bindings = 0, index = 0; buffer[ index ] != '\0'; index ++ ){
      if( ( buffer[ index ] == '*' ) || ( buffer[ index ] == '.' ) )
        num_bindings ++;
    }
    

    binding_list =
      (XrmBindingList) XtMalloc( ( num_bindings + 3 ) * sizeof( XrmBinding ) );
    quark_list =
      (XrmQuarkList) XtMalloc( ( num_bindings + 3 ) * sizeof( XrmQuark ) );

    XrmStringToBindingQuarkList( buffer, binding_list, quark_list );

    /* Find the start of the value. */
    for( start_index = length + 1;
         ( buffer[ start_index ] != '\0' ) && isspace( buffer[ start_index ] );
         start_index ++ );

    value.addr = (XPointer) &( buffer[ start_index ] );
    value.size = strlen( &( buffer[ start_index ] ) );

    /* Mask away trailing newline and spaces. */
    while( ( value.size > 0 ) &&
           isspace( *( (char *) value.addr + value.size - 1 ) ) )
      value.size --;
    *( (char *) value.addr + value.size ) = '\0';
    value.size ++;
    
    /* Check if selected. */
    selected = False;

    if( ( name_prefix == NULL ) || ( class_prefix == NULL ) )
      selected = True;
    else if( ( name_prefix[ 0 ] == wildcard_quark ) ||
             ( class_prefix[ 0 ] == wildcard_quark ) ){
      /* Select if specification starts with "*". */
      if( buffer[ 0 ] == '*' )
        selected = True;

    } else {
      /* The first quark must match name or class quark, or we must start
         with '*'. */
      if( buffer[ 0 ] == '*' )
        selected = True;

      else {
        if( ( name_prefix[ 0 ] == quark_list[ 0 ] ) ||
            ( class_prefix[ 0 ] == quark_list[ 0 ] ) )
          selected = True;
      }
    }
    
    if( selected )
      terminate = (* ( proc ) ) ( &db, binding_list, quark_list, &string_rep,
                                  &value, client_data );


    XtFree( (char *) binding_list );
    XtFree( (char *) quark_list );

    if( terminate )
      break;
    
  }

  fclose( db_file );
  unlink( db_filename );

  
  return terminate;

}


/*----------------------------------------------------------------------*/

static Bool
  ModifyKeywordResource( XrmDatabase        *db,
                         XrmBindingList     bindings,
                         XrmQuarkList       quarks,
                         XrmRepresentation  *type,
                         XrmValue           *value,
                         XPointer           closure )
{

  Bool                        bok;
  Boolean                     found;
  int                         index;
  xmubResourceMappingInfoRef  info;
  int                         length;
  Bool                        ok;
  String                      res_class;
  String                      res_name;
  char                        *str_type_return;
  XrmValue                    value_return;

  /* Code. */

  info = (xmubResourceMappingInfoRef) closure;


  /* If the value is a token that has a mapping, it should be replaced
     with the mapping. */

  /* Only for string resources. */
  if( *type == info -> string_quark ){

    /* Check number of quarks. */
    for( length = 0; quarks[ length ] != NULLQUARK; length ++ );

    if( info -> specific && ( length == 1 ) )
      /* This is a general specification. */
      return( False );

    /* If a resource selection has been defined, pick the ones we need. */
    if( info -> num_selections > 0 ){

      found = False;
      
      for( index = 0; index < info -> num_selections; index ++ ){
        if( info -> selection[ index ] == quarks[ length - 1 ] ){
          found = True;
          break;
        }
      }
      
      if( ( found && !info -> inclusive_selection ) ||
          ( !found && info -> inclusive_selection ) )
        return( False );
    }
    


    /* Allocate storage for keyword search. */
    res_name  = (char *) XtMalloc( strlen( info -> app_name ) + 
                                   value -> size + 5 );
    res_class = (char *) XtMalloc( strlen( info -> app_class ) + 
                                   value -> size + 5 );

    /* Check if the token has a mapping. */
    sprintf( res_name, "%s.", info -> app_name );
    strncat( res_name, (char *) value -> addr, value -> size );
    res_name[ strlen( info -> app_name ) + value -> size + 1 ] = '\0';

    sprintf( res_class, "%s.", info -> app_class );
    strncat( res_class, (char *) value -> addr, value -> size );
    res_class[ strlen( info -> app_class ) + value -> size + 1 ] = '\0';

    ok = XrmGetResource( info -> translation_db, res_name, res_class, 
                         &str_type_return, &value_return );

    if( ok ){
      /* We have a translation. */

      XrmQPutResource( &( info -> merge_db ), bindings, quarks, 
                       XrmStringToRepresentation( str_type_return ),
                       &value_return );

      /* Save original string if required. */
      if( info -> save_in_orig )
        XrmQPutResource( &( info -> orig_db ), bindings, quarks,
                         *type, value );

    } else if( info -> replace_substrings ){
      /* No match for whole keyword, but check if substring matches. */

      XrmQuark  class_list[ 2 ];
      XrmQuark  name_list[ 2 ];
      
      name_list[ 0 ] = XrmStringToQuark( info -> app_name );
      name_list[ 1 ] = NULLQUARK;

      class_list[ 0 ] = XrmStringToQuark( info -> app_class );
      class_list[ 1 ] = NULLQUARK;

      info -> replaced = False;
      info -> value    = XtMalloc( value -> size + 1 );
      strncpy( info -> value, (char *) value -> addr, value -> size );
      info -> value[ value -> size ] = '\0';
      
#if XlibSpecificationRelease > 4
      bok = XrmEnumerateDatabase(
#else
      bok = LoopThroughDatabase(
#endif
        info -> translation_db, name_list, class_list,
        XrmEnumAllLevels, ModifySubstringResource, (XPointer) info );
      
      if( info -> replaced ){
        XrmQPutStringResource( &( info -> merge_db ), bindings, quarks, 
                               info -> value );
        
        /* Save original string if required. */
        if( info -> save_in_orig )
          XrmQPutResource( &( info -> orig_db ), bindings, quarks,
                           *type, value );
      }

      XtFree( info -> value );
      
    }
    
    XtFree( res_name );
    XtFree( res_class );
  }


  return( False );

} /* ModifyKeywordResource */

  
/*----------------------------------------------------------------------*/

static Bool
  ModifySubstringResource( XrmDatabase        *db,
                           XrmBindingList     bindings,
                           XrmQuarkList       quarks,
                           XrmRepresentation  *type,
                           XrmValue           *value,
                           XPointer           closure )
{

  xmubResourceMappingInfoRef  info;
  int                         key_length;
  char*                       keyword;
  int                         length;
  char*                       start_ptr;
  char*                       tmp_string;
  int                         tok_index;
  char*                       tok_start;
  
  /* Code. */

  info = (xmubResourceMappingInfoRef) closure;


  /* Check if the substring is present in the value. */
  /* The keyword is in the last quark. */
  for( length = 0; quarks[ length ] != NULLQUARK; length ++ );

  keyword    = XrmQuarkToString( quarks[ length - 1 ] );
  key_length = strlen( keyword );
  
  
  start_ptr = info -> value;

  while( start_ptr < ( info -> value + value -> size ) ){

    if( ( tok_start = strstr( start_ptr, keyword ) ) != NULL ){

      tok_index = (int) ( tok_start - start_ptr );

      /* Check if the substring is delimited by the correct characters. */
      if( ( ( tok_index == 0 ) ||
            ( strchr( info -> substring_delimiters,
                      info -> value[ tok_index - 1 ] ) != NULL ) ) &&
          ( ( strlen( tok_start ) == key_length ) ||
            ( strchr( info -> substring_delimiters,
                      info -> value[ tok_index + key_length ] ) != NULL ) ) ){

        /* Ok. Replace the substring. */
        /* The translation is in value. */

        if( value -> size - 1 == key_length )
          strncpy( tok_start, (char *) value -> addr, value -> size - 1 );

        else if( value -> size - 1 < key_length ){
          strncpy( tok_start, (char *) value -> addr, value -> size - 1 );
          memmove( (void *) &( info -> value[ tok_index +
                                              value -> size - 1 ] ),
                   (void *) &( info -> value[ tok_index + key_length ] ),
                   strlen( &( info -> value[ tok_index + key_length ] ) ) +
                   1 );
          
        } else {
          /* The replacement is longer, so we need to extend the string. */
          info -> value = XtRealloc( info -> value,
                                     strlen( info -> value ) + value -> size );
          
          memmove( (void *) &( info -> value[ tok_index +
                                              value -> size - 1 ] ),
                   (void *) &( info -> value[ tok_index + key_length ] ),
                   strlen( &( info -> value[ tok_index + key_length ] ) ) +
                   1 );
          strncpy( &( info -> value[ tok_index ] ), (char *) value -> addr,
                   value -> size - 1 );
        }
        

        info -> replaced = True;
      }

      start_ptr = tok_start + key_length;
      

    } else
      /* The substring did not occur. */
      break;
    
  }
  

  return( False );

} /* ModifySubstringResource */

  
/*----------------------------------------------------------------------*/

static int
  StringComp( const void  *s1,
              const void  *s2 )
{

  String  *str1;
  String  *str2;

  /* Code. */

  str1 = (String *) s1;
  str2 = (String *) s2;


#ifdef VMS
  return( strcmp( *str1, *str2 ) );
#else
  return( strcoll( *str1, *str2 ) ); 
#endif

} /* StringComp */


/*----------------------------------------------------------------------*/

Boolean
xmubResAddMapFile( xmubResourceMappingInfoRef  info,
                   char*                       file_name,
                   Boolean                     add_defaults,
                   xmubAppResDesc              resource_sets[],
                   Cardinal                    num_resource_sets )
{

  Boolean      allocated_file_name = False;
  XrmDatabase  file_db;
  Boolean      ok;
  
  /* Code. */

  if( file_name == NULL ){
    file_name = xmubResGetMappingFileName( info -> display );
    allocated_file_name = True;
  }
  
  if( file_name == NULL )
    return False;
  
  /* Read the contents of the file into a temporary database. */
  /* The function also tests if the file is accessible. */
  if( ( file_db = XrmGetFileDatabase( file_name ) ) == NULL )
    return( False );

  ok = xmubResAddMappings( info, file_db, add_defaults,
                           resource_sets, num_resource_sets );
  

  if( allocated_file_name )
    XtFree( file_name );


  /* Save the mapping base in the mapping database.
     file_db is automatically destroyed. */
  XrmMergeDatabases( file_db, &( info -> mapping_db ) );
  

  return( ok );
  
}


/*----------------------------------------------------------------------*/

Boolean
xmubResAddMappings( xmubResourceMappingInfoRef  info,
                    XrmDatabase                 db,
                    Boolean                     add_defaults,
                    xmubAppResDesc              resource_sets[],
                    Cardinal                    num_resource_sets )
{

  Bool      bok;
  int       index;
  XrmQuark  name_list[ 2 ];

  /* Code. */

  info -> selected = False;
  
  /* Add the general resource specifications. */  
  if( add_defaults ){

    name_list[ 0 ] = XrmStringToQuark( "*" );
    name_list[ 1 ] = NULLQUARK;

    info -> specific = False;

#if XlibSpecificationRelease > 4
    bok = XrmEnumerateDatabase(
#else
    bok = LoopThroughDatabase(
#endif
      db, name_list, name_list, XrmEnumAllLevels,
      InsertGeneralMappingResources, (XPointer) info );
  }
  

  /* Specific resources from sets. */
  if( num_resource_sets > 0 ){

    char**  values;
    
    /* Extract the values for the defined resource sets. */
    values = (char **) XtMalloc( sizeof( char *) * num_resource_sets );
    GetResourceValues( info, resource_sets, num_resource_sets, values );

    info -> specific = True;

    /* For some reason, we must call for the different options separately. */
    for( index = 0; index < num_resource_sets; index ++ ){

      if( values[ index ] != NULL ){
        name_list[ 0 ] = XrmStringToQuark( values[ index ] );
        name_list[ 1 ] = NULLQUARK;

#if XlibSpecificationRelease > 4
        bok = XrmEnumerateDatabase(
#else
        bok = LoopThroughDatabase(
#endif
          db, name_list, name_list, XrmEnumAllLevels,
          InsertSpecificMappingResources, (XPointer) info );
      }
      
    }

    for( index = 0; index < num_resource_sets; index ++ )
      if( values[ index ] != NULL )
        XtFree( values[ index ] );
    
    XtFree( (char  *) values );
    
  }

  return info -> selected;
    
}


/*----------------------------------------------------------------------*/

void
xmubResAddSetChangedCallback( xmubResourceMappingInfoRef  info,
                              xmubResSetChangeCallback    callback,
                              XtPointer                   client_data )
{
  /* Code. */

  info -> num_set_change_callbacks ++;
  
  /* Extend the list. */
  if( info -> num_set_change_callbacks == 1 )
    info -> set_change_callbacks = (xmubResSetChangeCallbackList)
      XtMalloc( sizeof( xmubResSetChangeCallbackRec ) );

  else
    info -> set_change_callbacks = (xmubResSetChangeCallbackList)
      XtRealloc( (char *) info -> set_change_callbacks,
                 sizeof( xmubResSetChangeCallbackRec ) *
                 info -> num_set_change_callbacks );

  /* Insert the information. */
  info -> set_change_callbacks[
    info -> num_set_change_callbacks - 1 ].callback = callback;
  info -> set_change_callbacks[
    info -> num_set_change_callbacks - 1 ].client_data = client_data;
  
}


/*----------------------------------------------------------------------*/

void
xmubResAddStringMapping( xmubResourceMappingInfoRef  info,
                         char*                       keyword,
                         char*                       translation )
{
  char*  specifier;

  /* Code. */

  specifier = XtMalloc( strlen( keyword ) + 2 );
  sprintf( specifier, "*%s", keyword );

  XrmPutStringResource( &( info -> mapping_db ), specifier, translation );
  XrmPutStringResource( &( info -> translation_db ), specifier, translation );

  XtFree( specifier );
  
}


/*----------------------------------------------------------------------*/

Widget
  xmubAppInitialize( XtAppContext                *app_context_return,
                     String                      application_class,
                     XrmOptionDescList           options,
                     Cardinal                    num_options,
                     Cardinal                    *argc_in_out,
                     String                      *argv_in_out,
                     String                      *fallback_resources,
                     ArgList                     args, 
                     Cardinal                    num_args,
                     xmubAppResDesc              *resource_sets,
                     Cardinal                    num_resource_sets,
                     xmubResourceMappingInfoRef  *ref_return )
{

  XtAppContext                app;
  Display                     *display;
  Arg                         iargs[ 5 ];
  int                         index;
  xmubResourceMappingInfoRef  info;
  Cardinal                    n;
  Cardinal                    orig_argc;
  String                      *orig_argv = NULL;
  Widget                      toplevel;

  /* Code. */

  /* Make a copy of the original arguments. */
  orig_argc = *argc_in_out;

  if( orig_argc > 0 ){
    orig_argv = (char**) XtMalloc( sizeof( char* ) * orig_argc );
  
    for( index = 0; index < orig_argc; index++ )
      orig_argv[ index ] = XtNewString( argv_in_out[ index ] );
  }
  
  /* Initialize the application. */
  XtToolkitInitialize();
  app = XtCreateApplicationContext();

  /* Fallback resources. */
  if( fallback_resources != NULL )
    XtAppSetFallbackResources( app, fallback_resources );
  
  /* Open the display. */
  display = XtOpenDisplay( app, NULL, NULL, application_class,
                           options, num_options, 
                           (int *) argc_in_out, argv_in_out );

  xmubInitResourceMapping( display, resource_sets, num_resource_sets, &info );

  /* Create the shell. */
  n = 0;
  XtSetArg( iargs[ n ], XtNargc, orig_argc ); n++;
  XtSetArg( iargs[ n ], XtNargv, orig_argv ); n++;
  toplevel = XtAppCreateShell( NULL, application_class, 
                               applicationShellWidgetClass,
                               display, iargs, n );

  /* We don't have to free argv. */

  /* Set the arguments on the shell. */
  XtSetValues( toplevel, args, num_args );


  /* Set returned values. */
  if( app_context_return != NULL )
    *app_context_return = app;

  if( ref_return != NULL )
    *ref_return = info;
  else
    xmubResFreeInfo( info );


  return( toplevel );

}


/*----------------------------------------------------------------------*/

void
xmubInitResourceMapping( Display*                    display,
                         xmubAppResDesc              resource_sets[],
                         Cardinal                    num_resource_sets,
                         xmubResourceMappingInfoRef  *res_info )
{

  xmubResourceMappingInfoRef  info;
  Boolean                     ok;
  
  /* Code. */

  /* Initialize out parameter. */
  if( res_info != NULL )
    *res_info = NULL;
  
  /* Initialize. */
  info = xmubResInitialize( display );
  if( info == NULL )
    return;
  
  /* Add mappings from the default file. */
  ok = xmubResAddMapFile( info, NULL, True, resource_sets, num_resource_sets );
  if( ! ok ){
    xmubResFreeInfo( info );
    return;
  }
  
  /* Replace the keywords. */
  xmubResReplace( info, NULL );

  /* Merge the translations into the display database. */
  xmubResMergeMappedResources( info, NULL );

  if( res_info != NULL )
    *res_info = info;
  else
    xmubResFreeInfo( info );

}


/*----------------------------------------------------------------------*/

void
xmubResClearInternalDatabases(
  xmubResourceMappingInfoRef  info,
  Boolean                     clear_orig_db,
  Boolean                     clear_mapping_db,
  Boolean                     clear_translation_db,
  Boolean                     clear_merge_db )
{

  /* Code. */

  if( clear_orig_db ){
    XrmDestroyDatabase( info -> orig_db );
    info -> orig_db = NULL;
  }
  
  if( clear_mapping_db ){
    XrmDestroyDatabase( info -> mapping_db );
    info -> mapping_db = NULL;
  }
  
  if( clear_translation_db ){
    XrmDestroyDatabase( info -> translation_db );
    info -> translation_db = NULL;
  }
  
  if( clear_merge_db ){
    XrmDestroyDatabase( info -> merge_db );
    info -> merge_db = NULL;
  }
}


/*----------------------------------------------------------------------*/

void
xmubResFreeInfo( xmubResourceMappingInfoRef  info )
{

  /* Code. */

  /* Destroy the databases. */
  xmubResClearInternalDatabases( info, True, True, True, True );

  /* Free allocated strings. */
  XtFree( info -> app_name );
  XtFree( info -> app_class );

  /* Temporary fields must be freed by the allocator. */

  /* Free the record itself. */
  XtFree( (char *) info );
  
}


/*----------------------------------------------------------------------*/

#ifdef VMS
char*
xmubResGetMappingFileName( Display  *display )
{

  /* VMS version. */

  String   app_class;
  Boolean  class_file;
  String   filename = NULL;
  String   fname;
  String   env;
  String   name;

  /* Code. */

  /* Start by trying the environment variable. */
  env = getenv( XRESOURCE_MAP_FILE );
  if( ( env != NULL ) && FileReadAccess( env ) ){
    filename = XtNewString( env );
    return( filename );
  }

  /* File not accessible via environment variable. */
  /* Look for file with the name "app-class".map in all paths. 
     If not found, look for default file "xresource.map" in the same paths. */

  XtGetApplicationNameAndClass( display, &name, &app_class );


  class_file = True;

  for( ;; ){

    if( class_file )
      fname = app_class;
    else
      fname = MAP_FILE_NAME;

    /* Try user defaults. */
    filename = (String) XtMalloc( strlen( fname ) + 30 );
    sprintf( filename, "DECW$USER_DEFAULTS:%s%s", fname, MAP_FILE_SUFFIX );

    if( FileReadAccess( filename ) )
      return( filename );

    /* Try system defaults. */
    sprintf( filename, "DECW$SYSTEM_DEFAULTS:%s%s", fname, MAP_FILE_SUFFIX );

    if( FileReadAccess( filename ) )
      return( filename );
    else
      XtFree( filename );

    /* Try local directory. */
    sprintf( filename, "%s%s", fname, MAP_FILE_SUFFIX );

    if( FileReadAccess( filename ) )
      return( filename );
    else
      XtFree( filename );

    /* Second round checks for general file. */
    if( class_file )
      class_file = False;
    else
      break;

  }


  return( NULL );

}

#else


/*----------------------------------------------------------------------*/

char*
xmubResGetMappingFileName( Display  *display )
{

  /* Unix version. */

  String   app_class;
  Boolean  class_file;
  String   filename = NULL;
  String   fname;
  String   env;
  String   homedir = NULL;
  String   name;
  String   path = NULL;
  
  /* Code. */

  /* Start by trying the environment variable. */
  env = getenv( XRESOURCE_MAP_FILE );
  if( ( env != NULL ) && FileReadAccess( env ) ){
    filename = XtNewString( env );
    return( filename );
  }

  /* File not accessible via environment variable. */
  /* Look for file with the name "app-class".map in all paths. 
     If not found, look for default file "xresource.map" in the same paths. */

  XtGetApplicationNameAndClass( display, &name, &app_class );


  class_file = True;

  for( ;; ){

    if( class_file )
      fname = app_class;
    else
      fname = MAP_FILE_NAME;

    /* Try user file search path. */
    env = getenv( "XUSERFILESEARCHPATH" );
    if( env != NULL ){
      /* Copy path so it is not overwritten. */
      path = XtNewString( env );

      filename = XtResolvePathname( display, MAP_FILE_DIRTYPE, fname, 
                                    MAP_FILE_SUFFIX, path, 
                                    (Substitution) NULL, 0, 
                                    (XtFilePredicate) NULL );

      XtFree( path );
      path = NULL;
      
      if( ( filename != NULL ) && FileReadAccess( filename ) )
        return( filename );
      else if( filename != NULL )
        XtFree( filename );
    }

    /* Try application resource directory or home directory. */
    homedir = getenv( "HOME" );
    
    env = getenv( "XAPPLRESDIR" );
    if( env != NULL ){

      /* Construct path containing appl res dir. */
      path = (String) XtMalloc( strlen( env ) * 3 +
                                ( homedir != NULL ? strlen( homedir ) : 0 ) +
                                40 );
      if( homedir != NULL )
        sprintf( path, "%s%s:%s%s:%s%s:%s%s:.", 
                       env, "/%L/%N/%S", env, "/%l/%N%S", env, "/%N%S",
                       homedir, "/%N%S" );
      else
        sprintf( path, "%s%s:%s%s:%s%s:.", 
                       env, "/%L/%N/%S", env, "/%l/%N%S", env, "/%N%S" );

    } else if( homedir != NULL ){
      /* XAPPLRESDIR was not defined. Use path with home directory. */
      path = (String) XtMalloc( strlen( homedir ) * 3 + 25 );
      sprintf( path, "%s%s:%s%s:%s%s:.",
                     homedir, "/%L/%N%S", homedir, "/%l%N%S", 
                     homedir, "/%N%S" );
    }

    if( path != NULL ){
      filename = XtResolvePathname( display, MAP_FILE_DIRTYPE, fname, 
                                    MAP_FILE_SUFFIX, path, 
                                    (Substitution) NULL, 0, 
                                    (XtFilePredicate) NULL );
      XtFree( path );
    }
    
    if( ( filename != NULL ) && FileReadAccess( filename ) )
      return( filename );
    else if( filename != NULL )
      XtFree( filename );


    /* Last chance is the system default path. */
    filename = XtResolvePathname( display, MAP_FILE_DIRTYPE, fname, 
                                  MAP_FILE_SUFFIX, NULL,
                                  (Substitution) NULL, 0, 
                                  (XtFilePredicate) NULL );

    if( ( filename != NULL ) && FileReadAccess( filename ) )
      return( filename );
    else if( filename != NULL )
      XtFree( filename );


    /* Second round checks for general file. */
    if( class_file )
      class_file = False;
    else
      break;
  } 


  /* If we had found a file we would have returned by now. */

  return( NULL );

} 
#endif

/*----------------------------------------------------------------------*/

xmubResourceMappingInfoRef
xmubResInitialize( Display*  display )
{

  char*          app_class;
  char*          app_name;
  xmubResourceMappingInfoRef  info;
  
  /* Code. */

  info = (xmubResourceMappingInfoRef) XtMalloc(
    sizeof( xmubResourceMappingInfo ) );

  XtGetApplicationNameAndClass( display, &app_name, &app_class );
  
  /* Copy the name and class of the application. */
  info -> app_name  = XtNewString( app_name );
  info -> app_class = XtNewString( app_class );

  /* Set the database entries to NULL. */
  info -> orig_db         = NULL;
  info -> mapping_db      = NULL;
  info -> translation_db  = NULL;
  info -> merge_db        = NULL;
  
  info -> display = display;
  
  info -> string_quark = XrmStringToQuark( "String" );

  info -> set_change_callbacks     = NULL;
  info -> num_set_change_callbacks = 0;


  return( info );
  
}


/*----------------------------------------------------------------------*/

char**
xmubResListClasses( xmubResourceMappingInfoRef  info,
                    char*                       keyword,
                    Boolean                     sorted,
                    XrmDatabase                 db )
{

  Bool             bok;
  char**           class_table;
  char*            filename;
  int              index;
  ListClassesInfo  listinfo;
  
  /* Code. */

  if( db == NULL )
    db = info -> mapping_db;
  
  if( db == NULL )
    return NULL;

  /* Loop through all resources. */
  listinfo.info            = info;
  listinfo.keyword_quark   = XrmStringToQuark( keyword );
  listinfo.sets            = NULL;
  listinfo.num_sets        = 0;
  listinfo.set_list_length = 0;
  
  /* XrmEnumerateDatabase cannot handle NULL parameters for name_list
     and class_list, but LoopThroughDatabase can!! */
  bok = LoopThroughDatabase(
    db, NULL, NULL, XrmEnumAllLevels, ExtractKeywordClass,
    (XPointer) &listinfo );

  if( listinfo.num_sets > 0 ){
    class_table = (char **) XtMalloc( sizeof( char* ) *
                                      ( listinfo.num_sets + 1 ) );

    for( index = 0; index < listinfo.num_sets; index ++ )
      class_table[ index ] =
        XtNewString( XrmQuarkToString( listinfo.sets[ index ] ) );

    class_table[ listinfo.num_sets ] = NULL;

  } else
    class_table = NULL;
  
  if( listinfo.sets != NULL )
    XtFree( (char *) listinfo.sets );


  /* Sort the table. */
  if( sorted && ( class_table != NULL ) )
    qsort( (void *) class_table, (size_t) listinfo.num_sets, 
           (size_t) sizeof( char* ), StringComp );
  
  
  return( class_table );
  
}


/*----------------------------------------------------------------------*/

void
xmubResMergeMappedResources( xmubResourceMappingInfoRef  info,
                             XrmDatabase                 target_db )
{
  /* Code. */

  if( target_db == NULL )
    target_db = XtDatabase( info -> display );

  /* This call destroys the merge database. Replace it later. */
  XrmMergeDatabases( info -> merge_db, &( target_db ) );

  /* The database was destroyed. */
  info -> merge_db = NULL;
  
}


/*----------------------------------------------------------------------*/

void
xmubResRemoveSetChangedCallback( xmubResourceMappingInfoRef  info,
                                 xmubResSetChangeCallback    callback,
                                 XtPointer                   client_data )
{

  Boolean  found = False;
  int      pos;
  
  /* Code. */

  /* Find the callback info position. */
  for( pos = 0; pos < info -> num_set_change_callbacks; pos ++ ){
    if( ( info -> set_change_callbacks[ pos ].callback == callback ) &&
        ( info -> set_change_callbacks[ pos ].client_data == client_data ) ){
      found = True;
      break;
    }
  }

  if( ! found )
    return;

  /* The callback was present. */
  if( info -> num_set_change_callbacks == 1 ){
    info -> num_set_change_callbacks = 0;
    XtFree( (char *) info -> set_change_callbacks );
    info -> set_change_callbacks = NULL;

    return;
  }

  /* Just overwrite the old one. Don't care about wasted memory. */
  memmove( &( info -> set_change_callbacks[ pos ] ),
           &( info -> set_change_callbacks[ pos + 1 ] ),
           info -> num_set_change_callbacks - pos - 1 );

  info -> num_set_change_callbacks --;
  
}


/*----------------------------------------------------------------------*/

void
xmubResReplace( xmubResourceMappingInfoRef  info,
                XrmDatabase                 start_db )
{

  Boolean   allocated_delimiters = False;
  char*     delimiters;
  int       index;
  Bool      ok;
  char*     res_class;
  char*     res_name;
  char*     search_string;
  char**    str_list;
  int       str_list_index;
  int       str_list_size;
  char*     str_type_return;
  char*     token;
  XrmValue  value_return;

  /* Code. */

  /* First pass, replace without substring support. */
  xmubResReplaceKeywords( info, start_db, False, NULL, NULL, 0, False );

  /* Get the resource definitions. */
  /* We cannot use XtGetApplication resources, since it requires a widget. */
  res_name  = XtMalloc( strlen( info -> app_name ) + 30 );
  res_class = XtMalloc( strlen( info -> app_class ) + 30 );
  
  sprintf( res_name,  "%s.%s", info -> app_name, XmUbNsubstringResources );
  sprintf( res_class, "%s.%s", info -> app_class, XmUbCSubstringResources );
  
  /* Get the resource first from the source database, if not specified
     there, try display database. */
  if( start_db != NULL )
    ok = XrmGetResource( start_db, res_name, res_class, 
                         &str_type_return, &value_return );
  else
    ok = (Bool) False;
  
  if( ! ok )
    ok = XrmGetResource( XtDatabase( info -> display ), res_name, res_class, 
                         &str_type_return, &value_return );

  if( ok ){
    /* The resource was specified. */
    /* Construct string list from extracted string. */
    str_list_size = 5;
    str_list_index = 0;
    str_list = (char **) XtMalloc( sizeof( char *) * str_list_size );
    search_string = (char *) value_return.addr;
    

    while( ( token = strtok( search_string, " " ) ) != NULL ){

      if( str_list_index == str_list_size ){
        str_list_size += 5;
        str_list = (char **) XtRealloc( (char *) str_list,
                                        sizeof( char *) * str_list_size );
      }

      str_list[ str_list_index ] = XtNewString( token );
      str_list_index ++;
      
      /* Nullify for next call. */
      search_string = NULL;
    }
    
    /* The string list has been constructed. */
    /* Get the delimiter list. */
    sprintf( res_name,  "%s.%s", info -> app_name, XmUbNsubstringDelimiters );
    sprintf( res_class, "%s.%s", info -> app_class, XmUbCSubstringDelimiters );

    if( start_db != NULL )
      ok = XrmGetResource( start_db, res_name, res_class, 
                           &str_type_return, &value_return );
    else
      ok = (Bool) False;

    if( ! ok )
      ok = XrmGetResource( XtDatabase( info -> display ), res_name, res_class, 
                           &str_type_return, &value_return );

    if( ok ){
      delimiters = XtMalloc( sizeof( char ) * ( value_return.size + 1 ) );
      strncpy( delimiters, (char *) value_return.addr, value_return.size );
      delimiters[ value_return.size ] = '\0';
      allocated_delimiters = True;
      
    } else
      delimiters = DEFAULT_DELIMITERS;

    /* Replace the keywords. */
    xmubResReplaceKeywords( info, start_db, True, delimiters,
                            str_list, str_list_index, True );
    

    /* Free up. */
    for( index = 0; index < str_list_index; index ++ )
      XtFree( str_list[ index ] );

    XtFree( (char *) str_list );
    if( allocated_delimiters )
      XtFree( delimiters );
    
  }
  
  XtFree( res_name );
  XtFree( res_class );
  
}


/*----------------------------------------------------------------------*/

void
xmubResReplaceKeywords( xmubResourceMappingInfoRef  info,
                        XrmDatabase                 start_db,
                        Boolean                     replace_substrings,
                        char*                       substring_delimiters,
                        char**                      resource_selection,
                        Cardinal                    num_resource_selection,
                        Boolean                     inclusive_selection )
{

  Bool         bok;
  XrmQuark     class_list[ 2 ];
  int          index;
  XrmQuark     name_list[ 2 ];

  /* Code. */

  /* If no input database is defined, take the display database. */
  if( start_db == NULL )
    start_db = XtDatabase( info -> display );
  
  /* If a resource selection was defined, set it up as quarks, to allow for
     faster comparison. */
  info -> num_selections = num_resource_selection;

  if( num_resource_selection > 0 ){
    info -> selection =
      (XrmQuark *) XtMalloc( num_resource_selection * sizeof( XrmQuark ) );
    info -> inclusive_selection = inclusive_selection;
    
    for( index = 0; index < num_resource_selection; index ++ )
      info -> selection[ index ] =
        XrmStringToQuark( resource_selection[ index ] );

  } else
    info -> selection = NULL;

  
  /* We need two passes since we want to insert the general resources first,
     and override with the specific ones later. */

  /* General resources. */
  name_list[ 0 ] = XrmStringToQuark( "*" );
  name_list[ 1 ] = NULLQUARK;

  class_list[ 0 ] = XrmStringToQuark( "*" );
  class_list[ 1 ] = NULLQUARK;

  info -> specific     = False;

  info -> replace_substrings   = replace_substrings; 
  info -> substring_delimiters = substring_delimiters;

  if( start_db == info -> orig_db )
    info -> save_in_orig = False;
  else
    info -> save_in_orig = True;
  
#if XlibSpecificationRelease > 4
  bok = XrmEnumerateDatabase(
#else
  bok = LoopThroughDatabase(
#endif
    start_db, name_list, class_list, XrmEnumAllLevels, ModifyKeywordResource,
    (XPointer) info );

  /* Application specific resources. */
  name_list[ 0 ] = XrmStringToQuark( info -> app_name );
  name_list[ 1 ] = NULLQUARK;

  class_list[ 0 ] = XrmStringToQuark( info -> app_class );
  class_list[ 1 ] = NULLQUARK;

  info -> specific = True;

#if XlibSpecificationRelease > 4
  bok = XrmEnumerateDatabase(
#else
  bok = LoopThroughDatabase(
#endif
    start_db, name_list, class_list, XrmEnumAllLevels, ModifyKeywordResource,
    (XPointer) info );


  /* Clean up. */
  if( info -> selection != NULL )
    XtFree( (char *) info -> selection );
  
}


/*----------------------------------------------------------------------*/

Boolean
xmubResUpdateAllResources( xmubResourceMappingInfoRef  info,
                           xmubAppResDesc              resource_sets[],
                           Cardinal                    num_resource_sets,
                           xmubResSetChangeCallback    apply_callback,
                           XtPointer                   client_data )
{

  Bool                            bok;
  xmubResSetChangeCallbackStruct  cb;
  int                             index;
  Boolean                         ok;
  char*                           res_class;
  char*                           res_name;
  int                             res_number;
  char*                           str_type_return;
  XrmValue                        value_return;
  
  /* Code. */

  /* Clear the translation and merge databases. */
  xmubResClearInternalDatabases( info, False, False, True, True );

  ok = xmubResAddMappings( info, info -> mapping_db, True,
                           resource_sets, num_resource_sets );
  
  if( ok ){
    xmubResReplace( info, info -> orig_db );
    xmubResMergeMappedResources( info, NULL );
  }

  /* Call callbacks for each resource. */

  /* Set up the callback structure. */
  cb.reason          = XMUB_RES_SET_CHANGED;
  cb.res_name        = NULL;
  cb.res_class       = NULL;
  cb.new_value       = NULL;
  cb.mapping_changed = ok;

  /* Call the apply callback. */
  if( apply_callback != NULL )
    (* ( apply_callback ) ) ( info, client_data, &cb );

  /* Call the resource change callbacks. */
  for( res_number = 0; res_number < num_resource_sets; res_number ++ ){

    res_name = XtMalloc( strlen( resource_sets[ res_number ].res_name ) +
                         strlen( info -> app_name ) + 5 );
    sprintf( res_name, "%s.%s", info -> app_name,
                                resource_sets[ res_number ].res_name );
    
    res_class = XtMalloc( strlen( resource_sets[ res_number ].res_class ) +
                          strlen( info -> app_class ) + 5 );
    sprintf( res_class, "%s.%s", info -> app_class,
                                 resource_sets[ res_number ].res_class );
    
    bok = XrmGetResource( XtDatabase( info -> display ), res_name, res_class,
                          &str_type_return, &value_return );
    
    cb.reason          = XMUB_RES_SET_CHANGED;
    cb.res_name        = resource_sets[ res_number ].res_name;
    cb.res_class       = resource_sets[ res_number ].res_class;
    if( bok )
      cb.new_value = (char *) value_return.addr;
    else
      cb.new_value = NULL;
    cb.mapping_changed = True;
    
    for( index = 0; index < info -> num_set_change_callbacks; index ++ )
      (* ( info -> set_change_callbacks[ index ].callback ) )
        ( info, info -> set_change_callbacks[ index ].client_data, &cb );

    XtFree( res_name );
    XtFree( res_class );
    

  }
  
  return( ok );

}


/*----------------------------------------------------------------------*/

Boolean
xmubResUpdateResourceSet( xmubResourceMappingInfoRef  info,
                          char*                       res_name,
                          char*                       res_class,
                          char*                       new_value,
                          xmubResSetChangeCallback    apply_callback,
                          XtPointer                   client_data )
{

  xmubResSetChangeCallbackStruct  cb;
  XrmDatabase                     db;
  int                             index;
  Boolean                         ok;
  char*                           resource_line;
  xmubAppResDesc                  res_sets;
  
  /* Code. */

  /* Set the new resource in the display database. */
  resource_line = XtMalloc( strlen( res_name ) + strlen( new_value ) + 5 );
  sprintf( resource_line, "*%s: %s", res_name, new_value );
  db = XtDatabase( info -> display );

  XrmPutLineResource( &db, resource_line );

  XtFree( resource_line );

  /* Clear the translation and merge databases. */
  xmubResClearInternalDatabases( info, False, False, True, True );

  /* Select the mappings from the mapping database. */
  res_sets.res_name  = res_name;
  res_sets.res_class = res_class;
  ok = xmubResAddMappings( info, info -> mapping_db, False, &res_sets, 1 );
  
  if( ok ){
    xmubResReplace( info, info -> orig_db );
    xmubResMergeMappedResources( info, NULL );
  }

  /* Set up the callback structure. */
  cb.reason          = XMUB_RES_SET_CHANGED;
  cb.res_name        = res_name;
  cb.res_class       = res_class;
  cb.new_value       = new_value;
  cb.mapping_changed = ok;

  /* Call the apply callback. */
  if( apply_callback != NULL )
    (* ( apply_callback ) ) ( info, client_data, &cb );

  /* Call the resource change callbacks. */
  for( index = 0; index < info -> num_set_change_callbacks; index ++ )
    (* ( info -> set_change_callbacks[ index ].callback ) )
      ( info, info -> set_change_callbacks[ index ].client_data, &cb );


  return( ok );
  
}
