/*----------------------------------------------------------------------------
--
--  Module:           xtmAccBase
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Base access control functions.
--
--  Filename:         xtmAccBase.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1995-02-03
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmAccBase.c, Version: 1.1, Date: 95/02/18 15:51:49";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xitError.h"
#include "xtmAccBase.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Access control file. */
#define ACCESS_FILE      "dbAccess"
#define AFS_ACCESS_FILE  "dbAfsAccess"

/* Keys to use in the access file. */
#define KEY_ACCESS_CONTROL  "Access"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmAccessBase";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_AB_STATUS
  xtmAbGetAccessInfo( char           *directory,
                      LST_DESC_TYPE  *access_info )
{

  /* Variables. */
  char                buffer[ 200 ];
  char                flags[ 20 ];
  char                group[ 30 ];
  char                key[ 50 ];
  char                user[ 30 ];
  char                *char_ref;
  FILE                *file_ref;
  LST_STATUS          lst_status;
  XTM_AB_ACCESS_INFO  access;


  /* Code. */

  /* Create an new access list. */
  *access_info = LstLinkNew( sizeof( XTM_AB_ACCESS_INFO ), NULL );


  /* Build the filename. */
  sprintf( buffer, "%s/%s", directory, ACCESS_FILE );

  /* Open the access control file. */
  file_ref = fopen( buffer, "r" );
  if( file_ref == NULL )
    return( XTM_AB_ERROR );

  
  /* Start reading from the file, skip all blank lines and comments. */
  do {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL ) {
      fclose( file_ref );
      return( XTM_AB_OK );
    } 

    /* Skip leading blanks. */
    while( isspace( *char_ref ) )
      char_ref++;


    /* What kind of line is it? */

    /* Blank line? */
    if( *char_ref == '\0' ) {
      ;

    /* Comment? */
    } else if( *char_ref == '#' ) {
      ;

    /* Access information? */
    } else if( strncmp( char_ref, KEY_ACCESS_CONTROL, 
               strlen( KEY_ACCESS_CONTROL) ) == 0 ) {

      user[  0 ] ='\0';
      group[ 0 ] ='\0';
      flags[ 0 ] ='\0';

      sscanf( char_ref, "%[^:]: %[^ ] %[^ ] %[^ \n]",
              key, 
              user, group, flags );

      /* Convert user name to uid. */
      if( strcmp( user, "*" ) == 0 )
        access.uid = -1;
      else
        access.uid = atoi( user );

      if( strcmp( group, "*" ) == 0 )
        access.gid = -1;
      else
        access.gid = atoi( group );


      /* Check the flags. */
      access.flags = 0;

      if( strchr( flags, 'r' ) != NULL )
        flagSet( access.flags, XTM_AB_FLAG_READ );

      if( strchr( flags, 'w' ) != NULL )
        flagSet( access.flags, XTM_AB_FLAG_WRITE );

      if( strchr( flags, 'm' ) != NULL )
        flagSet( access.flags, XTM_AB_FLAG_MSG );

      if( strchr( flags, 'p' ) != NULL )
        flagSet( access.flags, XTM_AB_FLAG_PRIV );

      lst_status = LstLinkInsertLast( *access_info, &access );

    /* Unknown key. */
    } else {
      sprintf( buffer, "Unknown key %s", char_ref );

      fprintf( stderr, "XDiary, %s %s: %s\n",
               module_name, "xtmAbGetAccessInfo", buffer );
    } /* if */

  } while( True );


} /* xtmAbGetAccessInfo */


/*----------------------------------------------------------------------*/

XTM_AB_STATUS
  xtmAbGetAfsAccessInfo( char           *directory,
                         LST_DESC_TYPE  *access_info )
{

  /* Variables. */
  char                 buffer[ 600 ];
  char                 flags[ 20 ];
  char                 key[ 50 ];
  char                 *char_ref;
  FILE                 *file_ref;
  LST_STATUS           lst_status;
  XTM_AB_AFS_ACL_INFO  acl_id;


  /* Code. */

  /* Create an new access list. */
  *access_info = LstLinkNew( sizeof( XTM_AB_AFS_ACL_INFO ), NULL );

  /* Build the filename. */
  sprintf( buffer, "%s/%s", directory, AFS_ACCESS_FILE );

  /* Open the access control file. */
  file_ref = fopen( buffer, "r" );
  if( file_ref == NULL )
    return( XTM_AB_ERROR );

  
  /* Start reading from the file, skip all blank lines and comments. */
  do {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL ) {
      fclose( file_ref );
      return( XTM_AB_OK );
    } 

    /* Skip leading blanks. */
    while( isspace( *char_ref ) )
      char_ref++;


    /* What kind of line is it? */

    /* Blank line? */
    if( *char_ref == '\0' ) {
      ;

    /* Comment? */
    } else if( *char_ref == '#' ) {
      ;

    /* Access information? */
    } else if( strncmp( char_ref, KEY_ACCESS_CONTROL, 
               strlen( KEY_ACCESS_CONTROL) ) == 0 ) {

      flags[ 0 ] ='\0';

      sscanf( char_ref, "%[^:]: %[^ ] %[^ ]\n]", key, acl_id.id, flags );

      /* Check the flags. */
      acl_id.flags = 0;

      if( strchr( flags, 'r' ) != NULL )
        flagSet( acl_id.flags, XTM_AB_FLAG_READ );

      if( strchr( flags, 'w' ) != NULL )
        flagSet( acl_id.flags, XTM_AB_FLAG_WRITE );

      if( strchr( flags, 'm' ) != NULL )
        flagSet( acl_id.flags, XTM_AB_FLAG_MSG );

      lst_status = LstLinkInsertLast( *access_info, &acl_id );

    /* Unknown key. */
    } else {
      sprintf( buffer, "Unknown key %s", char_ref );

      fprintf( stderr, "XDiary, %s %s: %s\n",
               module_name, "xtmAbGetAfsAccessInfo", buffer );
    } /* if */

  } while( True );


} /* xtmAbGetAfsAccessInfo */


/*----------------------------------------------------------------------*/

XTM_AB_STATUS
  xtmAbSaveAccessInfo( char           *directory,
                       LST_DESC_TYPE  access_info )
{

  /* Variables. */
  char                buffer[ 200 ];
  char                group[ 30 ];
  char                user[ 30 ];
  FILE                *file_ref;
  LST_STATUS          lst_status;
  XTM_AB_ACCESS_INFO  access;


  /* Code. */

  /* Write the database locations. */
  if( LstLinkElements( access_info ) == 0 )
    return( XTM_AB_OK );


  /* Build the filename. */
  sprintf( buffer, "%s/%s", directory, ACCESS_FILE );

  /* Open the access control file. */
  file_ref = fopen( buffer, "w" );
  if( file_ref == NULL )
    return( XTM_AB_ERROR );


  /* Save all access entries. */  
  lst_status = LstLinkCurrentFirst( access_info );

  while( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( access_info, &access );

    /* Take care of user and group id. */
    if( access.uid == -1 )
      sprintf( user, "*" );
    else
      sprintf( user, "%d", access.uid );

    if( access.gid == -1 )
      sprintf( group, "*" );
    else
      sprintf( group, "%d", access.gid );

    fprintf( file_ref, "%s: %-20.20s %-20.20s ", 
             KEY_ACCESS_CONTROL, 
             user, group );

    /* Write the flags. */
    if( flagIsSet( access.flags, XTM_AB_FLAG_READ ) )
      fprintf( file_ref, "r" );

    if( flagIsSet( access.flags, XTM_AB_FLAG_WRITE ) )
      fprintf( file_ref, "w" );

    if( flagIsSet( access.flags, XTM_AB_FLAG_MSG ) )
      fprintf( file_ref, "m" );

    fprintf( file_ref, "\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( access_info );

  } /* while */

  fclose( file_ref );

  return( XTM_AB_OK );

} /* xtmAbSaveAccessInfo */


/*----------------------------------------------------------------------*/

XTM_AB_STATUS
  xtmAbSaveAfsAccessInfo( char           *directory,
                          LST_DESC_TYPE  access_info )
{

  /* Variables. */
  char                 buffer[ 600 ];
  FILE                 *file_ref;
  LST_STATUS           lst_status;
  XTM_AB_AFS_ACL_INFO  acl_id;


  /* Code. */

  /* Write the database locations. */
  if( LstLinkElements( access_info ) == 0 )
    return( XTM_AB_OK );


  /* Build the filename. */
  sprintf( buffer, "%s/%s", directory, AFS_ACCESS_FILE );

  /* Open the access control file. */
  file_ref = fopen( buffer, "w" );
  if( file_ref == NULL )
    return( XTM_AB_ERROR );


  /* Save all access entries. */  
  lst_status = LstLinkCurrentFirst( access_info );

  while( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( access_info, &acl_id );

    fprintf( file_ref, "%s: %s ", KEY_ACCESS_CONTROL, acl_id.id );

    /* Write the flags. */
    if( flagIsSet( acl_id.flags, XTM_AB_FLAG_READ ) )
      fprintf( file_ref, "r" );

    if( flagIsSet( acl_id.flags, XTM_AB_FLAG_WRITE ) )
      fprintf( file_ref, "w" );

    if( flagIsSet( acl_id.flags, XTM_AB_FLAG_MSG ) )
      fprintf( file_ref, "m" );

    fprintf( file_ref, "\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( access_info );

  } /* while */

  fclose( file_ref );

  return( XTM_AB_OK );

} /* xtmAbSaveAfsAccessInfo */


/*----------------------------------------------------------------------*/

XTM_AB_STATUS
  xtmAbGetUserOperations( char    *directory,
                          int     user_id,
                          int     group_id,
                          UINT32  *operations )
{

  /* Variables. */
  XTM_AB_ACCESS_INFO  access;
  XTM_AB_STATUS       ac_status;
  LST_DESC_TYPE       access_info;
  LST_STATUS          lst_status;


  /* Code. */

  *operations = 0;

  /* Fetch the access control list for this database. */
  ac_status = xtmAbGetAccessInfo( directory, &access_info );
  if( ac_status != XTM_AB_OK )
    return( XTM_AB_OK );

  /* Check the access information. */
  lst_status = LstLinkCurrentFirst( access_info );

  while( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( access_info, &access );

    /* Do we have a match? */
    if( (access.uid == -1       && access.gid == -1 ) ||
        (access.uid == -1       && access.gid == group_id) ||
        (access.uid == user_id  && access.gid == -1 ) ||
        (access.uid == user_id  && access.gid == group_id) ) {

      /* Check the flags. */
      if( flagIsSet( access.flags, XTM_AB_FLAG_READ ) ) {
        flagSet( *operations, XTM_DB_FLAG_MODE_READ );
      }

      if( flagIsSet( access.flags, XTM_AB_FLAG_WRITE ) ) {
        flagSet( *operations, XTM_DB_FLAG_MODE_READ );
        flagSet( *operations, XTM_DB_FLAG_MODE_WRITE );
        flagSet( *operations, XTM_DB_FLAG_MODE_MSG );
      }

      if( flagIsSet( access.flags, XTM_AB_FLAG_MSG ) ) {
        flagSet( *operations, XTM_DB_FLAG_MODE_MSG );
      }
 
      if( flagIsSet( access.flags, XTM_AB_FLAG_PRIV ) ) {
        flagSet( *operations, XTM_DB_FLAG_MODE_READ );
        flagSet( *operations, XTM_DB_FLAG_MODE_WRITE );
        flagSet( *operations, XTM_DB_FLAG_MODE_MSG );
        flagSet( *operations, XTM_DB_FLAG_MODE_PRIV );
      }

      if( *operations != 0 )
        flagSet( *operations, XTM_DB_FLAG_MODE_SETID );

      LstLinkClear( access_info );

      return( XTM_AB_OK );

    } /* if */

    /* Next entry. */
    lst_status = LstLinkCurrentNext( access_info );

  } /* while */

  LstLinkClear( access_info );


  return( XTM_AB_OK );

} /* xtmAbGetUserOperations */
