/*----------------------------------------------------------------------------
--
--  Module:           xtmCustBase
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Module to handle the base customization operations. You can save and
--    fetch customization data.
--
--  Filename:         xtmCustBase.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustBase.c, Version: 1.1, Date: 95/02/18 15:52:01";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "Scramble.h"

#include "xtmGlobal.h"
#include "xtmArchive.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmGroupDb.h"
#include "xtmTools.h"
#include "xtmCustBase.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Version of the customize file. */
#define FILE_VER_NO                      18

/* Customize file for a user. */
#define DEFAULT_CUSTOM_FILE              "XDiary.custom"

/* Default language. */
#define DEFAULT_LANGUAGE                 "En_US"

/* What to fetch from the custom file. */
#define GET_ALL                    (1<<0)
#define GET_ARCHIVES               (1<<1)
#define GET_CALENDARS              (1<<2)
#define GET_COLORS                 (1<<3)
#define GET_FLAGS                  (1<<4)
#define GET_GROUPS                 (1<<5)
#define GET_PARAMETERS             (1<<6)

/* Keys to use in the custom file. */
#define KEY_VER_KEY                      "VersionDef"
#define KEY_VER_NO                       "No"

#define KEY_CAL_ALARM_TAGS               "AlarmTags"
#define KEY_CAL_LOCATION                 "Location"
#define KEY_CAL_EXT_HOST                 "ExternHost"
#define KEY_CAL_EXT_PWD                  "ExternPassword"
#define KEY_CAL_EXT_UID                  "ExternUid"
#define KEY_CAL_EXT_TYPE                 "ExternType"
#define KEY_CAL_FLAGS                    "Flags"
#define KEY_CAL_INCL_TAGS                "Include"
#define KEY_CAL_INCL_BG_COLOR            "IncludeBgColor"
#define KEY_CAL_KEY                      "CalDef"
#define KEY_CAL_MAIL                     "Mail"
#define KEY_CAL_NAME                     "Name"
#define KEY_CAL_STACK_ORDER              "StackOrder"
#define KEY_CAL_VIEW_TAGS                "ViewTags"

#define KEY_GROUP_KEY                    "GroupDef"
#define KEY_GROUP_NAME                   "Name"
#define KEY_GROUP_MEMBERS                "Members"

#define KEY_ARCH_KEY                     "ArchiveDef"
#define KEY_ARCH_NAME                    "Name"
#define KEY_ARCH_FILE                    "File"

#define KEY_INCL_KEY                     "IncludeInfo"
#define KEY_INCL_FILE                    "File"

#define KEY_PAR_KEY                      "Parameters"
#define KEY_PAR_ALARM_LINES              "AlarmLines"
#define KEY_PAR_APP_PANE_HEIGHT          "AppPaneHeight"
#define KEY_PAR_DATE_FORMAT              "DateFormat"
#define KEY_PAR_DAY_WIDTH                "DayWidth"
#define KEY_PAR_DV_DAY_SWITCH            "DayViewDaySwitch"
#define KEY_PAR_ENTRY_LINES              "EntryLines"
#define KEY_PAR_ENTRY_PANE_WIDTH         "EntryPaneWidth"
#define KEY_PAR_MSG_LANGUAGE             "MessageLanguage"
#define KEY_PAR_NOTE_PANE_HEIGHT         "NotePaneHeight"
#define KEY_PAR_PAPER_SIZE               "PaperSize"
#define KEY_PAR_PRINT_DEF_LINES          "PrintDefLines"
#define KEY_PAR_REFRESH_RATE             "RefreshRate"
#define KEY_PAR_SHEETS_PER_PAGE          "SheetsPerPage"
#define KEY_PAR_START_DV_ON              "StartDayViewOn"
#define KEY_PAR_SUMMARY_DEF_LINES        "SummaryDefLines"
#define KEY_PAR_TIME_FORMAT              "TimeFormat"
#define KEY_PAR_TUNE                     "DefaultTune"
#define KEY_PAR_WORKDAYS                 "Workdays"
#define KEY_PAR_VIEW_DAYS                "DayViewDays"
#define KEY_PAR_VIEW_DELTA               "DayViewDelta"
#define KEY_PAR_VIEW_START               "DayViewStarts"
#define KEY_PAR_VIEW_STOP                "DayViewStops"

#define KEY_FLAG_ALARM                   "GiveAlarm"
#define KEY_FLAG_ALARM_NEW_APP           "AlarmsForNewEntries"
#define KEY_FLAG_COMPRESSED              "Compressed"
#define KEY_FLAG_CONF                    "ConfAction"
#define KEY_FLAG_DO_REFRESH              "DoRefresh"
#define KEY_FLAG_ENTRY_FLAGS             "EntryFlags"
#define KEY_FLAG_ENTRY_INTER             "EntriesInteractive"
#define KEY_FLAG_KEY                     "FlagsDef"
#define KEY_FLAG_LIST_LAY                "LayoutTime"
#define KEY_FLAG_MONTH_EXTEND            "CalMonthExtend"
#define KEY_FLAG_NAV_CAL                 "NavigateCal"
#define KEY_FLAG_PROMPT_REPEAT           "PromptRepeat"
#define KEY_FLAG_PR_WORKDAYS             "PrintOnlyWorkdays"
#define KEY_FLAG_PR_IN_3D                "PrintIn3D"
#define KEY_FLAG_SHOW_STAND              "ShowStanding"
#define KEY_FLAG_SHOW_TO_DO              "ShowToDo"
#define KEY_FLAG_TRUE_COLOR_INC          "TrueColorInc"
#define KEY_FLAG_WEEK_NO                 "WeekNumbers"

/* Old definition (will be removed later). */
#define OLD_KEY_CAL_KEY                  "DayViewDef"
#define OLD_KEY_CAL_LOCATION             "Directory"
#define OLD_KEY_FLAG_MON_DAY1            "MondayDay1"
#define OLD_KEY_FLAG_US_DATE             "UsDate"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* States when reading the customize file. */
typedef enum {
  ARCH_DEF,
  CAL_DEF,
  END_DEF,
  FLAG_DEF,
  GROUP_DEF,
  INCL_DEF,
  PAR_DEF,
  UNKNOWN_DEF,
  VER_DEF
} READ_STATE;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustBase";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  clearArchCB( void  *element );

static void
  clearInclCB( void  *element );

static LST_COMPARE
  compareInclCB( XTM_GL_INCLUDE_FILE  *incl_ref,
                 char                 *incl_file );

static int
  fetchFileVersion( FILE  *file_ref );

static void
  getDataFromFile( UINT32                  do_flags,
                   FILE                    *file_ref,
                   XTM_GL_CUSTOM_DATA_REF  custom_data );

static Boolean
  readArchiveInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                   FILE                    *file_ref,
                   char                    *key,
                   char                    *line,
                   XTM_AR_ARCHIVE_FILE     *archive_ref );

static Boolean
  readCalendarInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref,
                    char                    *key,
                    char                    *line,
                    XTM_CD_CAL_INFO         *db_info_ref,
                    XTM_CD_INCL_CALS        *db_incl_ref );

static Boolean
  readFlagInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                FILE                    *file_ref,
                char                    *key,
                char                    *line );

static Boolean
  readGroupInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                 FILE                    *file_ref,
                 char                    *key,
                 char                    *line,
                 XTM_GD_ENTRY_REF        group_ref );

static Boolean
  readInclInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                UINT32                  do_flags,
                FILE                    *file_ref,
                char                    *key,
                char                    *line );

static Boolean
  readParameterInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                     FILE                    *file_ref,
                     char                    *key,
                     char                    *line );

static void
  setDefaultValues( XTM_GL_CUSTOM_DATA_REF  custom_data );

static Boolean
  writeArchiveInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref );

static Boolean
  writeCalendarInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                     FILE                    *file_ref );

static Boolean
  writeFlagInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                 FILE                    *file_ref );

static Boolean
  writeGroupInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                  FILE                    *file_ref );

static Boolean
  writeIncludeInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref );

static Boolean
  writeParameterInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                      FILE                    *file_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CB_STATUS
  xtmCbGetDataFromFile( XTM_GL_CUSTOM_DATA_REF  custom_data )
{

  /* Variables. */
  int   size;
  int   version;
  char  *char_ref;
  FILE  *file_ref;


  /* Code. */

  /* Free any allocated lists. */
  xtmCdFree( custom_data -> cal_db_handle );
  xtmGdFree( custom_data -> group_db_handle );

  LstLinkClearDataAndList( custom_data -> archive_files, clearArchCB );
  LstLinkClearDataAndList( custom_data -> include_files, clearInclCB );


  /* Calendar locations. */
  custom_data -> cal_db_handle = xtmCdInitialize();

  /* Groups. */
  custom_data -> group_db_handle = xtmGdInitialize();

  /* Archive files. */
  custom_data -> archive_files = LstLinkNew( sizeof( XTM_AR_ARCHIVE_FILE ), 
                                             NULL );
  /* Included files. */
  custom_data -> include_files = LstLinkNew( sizeof( XTM_GL_INCLUDE_FILE ), 
                                             NULL );

  /* Initialize. */
  custom_data -> incl_depth = 0;
  setDefaultValues( custom_data );


  /* If no file name is given, build the file name. */
  if( custom_data -> custom_file != NULL && 
      strlen( custom_data -> custom_file ) == 0 ) {

    SysFree( custom_data -> custom_file );

    custom_data -> custom_file = NULL;
  }

  if( custom_data -> custom_file == NULL ) {
    char_ref = getenv( "HOME" );

    if( char_ref == NULL ) {
      custom_data -> custom_file = 
        (char *) SysMalloc( strlen( DEFAULT_CUSTOM_FILE ) + 10 );

      sprintf( custom_data -> custom_file, "%s",
               DEFAULT_CUSTOM_FILE );
    } else {
      custom_data -> custom_file = 
        (char *) SysMalloc( strlen( char_ref ) + 
                            strlen( DEFAULT_CUSTOM_FILE ) + 10 );

      sprintf( custom_data -> custom_file, "%s/%s", 
               char_ref, DEFAULT_CUSTOM_FILE );
    }

  } /* if */


  /* Open the customize file. */
  file_ref = fopen( custom_data -> custom_file, "r" );
  if( file_ref == NULL )
    return( XTM_CB_ERROR );


  /* Fetch the file version. */
  version = fetchFileVersion( file_ref );


  /* Make sure we can understand the custom file. */
  if( version > FILE_VER_NO )
    return( XTM_CB_WRONG_VERSION );

  /* Fetch the customized data. */
  if( version > 0 )
    getDataFromFile( GET_ALL, file_ref, custom_data );


  fclose( file_ref );


  /* If the file is not the current version, make a copy. */
  if( version != FILE_VER_NO ) {

    size     = 2 * strlen( custom_data -> custom_file ) + 100;
    char_ref = (char *) SysMalloc( size );

    sprintf( char_ref, "cp %s %s_ver%d",
             custom_data -> custom_file,
             custom_data -> custom_file, version );

    system( char_ref );
    SysFree( char_ref );

    /* Save the current file. */
    (void) xtmCbPutDataToFile( custom_data );

    custom_data -> new_release = True;

  } /* if */



  return( XTM_CB_OK );  

} /* xtmCbGetDataFromFile */


/*----------------------------------------------------------------------*/

XTM_CB_STATUS
  xtmCbPutDataToFile( XTM_GL_CUSTOM_DATA_REF  custom_data )
{

  /* Variables. */
  char  *char_ref;
  FILE  *file_ref;


  /* Code. */


  /* If no file name is given, build the file name. */
  if( custom_data -> custom_file != NULL &&
      strlen( custom_data -> custom_file ) == 0 ) {
    SysFree( custom_data -> custom_file );
    custom_data -> custom_file = NULL;
  }

  if( custom_data -> custom_file == NULL ) {
    char_ref = getenv( "HOME" );

    custom_data -> custom_file = SysMalloc( strlen( char_ref ) + 20 );

    if( char_ref == NULL )
      sprintf( custom_data -> custom_file, "%s",
               DEFAULT_CUSTOM_FILE );
    else
      sprintf( custom_data -> custom_file, "%s/%s", 
               char_ref, DEFAULT_CUSTOM_FILE );
  } /* if */

  /* Open the customize file. */
  file_ref = fopen( custom_data -> custom_file, "w" );
  if( file_ref == NULL )
    return( XTM_CB_ERROR );

  /* Version of customize file. */
  fprintf( file_ref, "%s:\n", KEY_VER_KEY );
  fprintf( file_ref, "\t%s: %d\n\n", KEY_VER_NO, FILE_VER_NO );

 
  /* Save included files. */
  (void) writeIncludeInfo( custom_data, file_ref );

  /* Save the customer calendars. */
  (void) writeCalendarInfo( custom_data, file_ref );

  /* Save the groups. */
  (void) writeGroupInfo( custom_data, file_ref );

  /* Save the archive files. */
  (void) writeArchiveInfo( custom_data, file_ref );

  /* Save the parameters. */
  (void) writeParameterInfo( custom_data, file_ref );

  /* Write the flags. */
  (void) writeFlagInfo( custom_data, file_ref );

  fclose( file_ref );


  return( XTM_CB_OK );

} /* xtmCbPutDataToFile */


/*----------------------------------------------------------------------*/

XTM_CB_STATUS
  xtmCbRereadParamFromFile( XTM_GL_CUSTOM_DATA_REF  custom_data )
{

  /* Variables. */
  int   version;
  FILE  *file_ref;


  /* Code. */

  /* Initialize. */
  custom_data -> incl_depth = 0;
  setDefaultValues( custom_data );


  /* Open the customize file. */
  file_ref = fopen( custom_data -> custom_file, "r" );
  if( file_ref == NULL )
    return( XTM_CB_ERROR );


  /* Fetch the file version. */
  version = fetchFileVersion( file_ref );

  /* Fetch the customized data. */
  if( version > 0 )
    getDataFromFile( (GET_FLAGS|GET_PARAMETERS), file_ref, custom_data );

  fclose( file_ref );


  return( XTM_CB_OK );  

} /* xtmCbRereadParamFromFile */


/*----------------------------------------------------------------------*/

int
  xtmCbWhatVersion()
{

  /* Code. */

  return( FILE_VER_NO );

} /* xtmCbWhatVersion */


/*----------------------------------------------------------------------*/

static int
  fetchFileVersion( FILE  *file_ref )
{

  /* Variables. */
  int   version;
  char  buffer[ 500 ];
  char  key[ 50 ];
  char  *char_ref;


  /* Code. */

  version = 0;

  /* Serach the version key. */
  do {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      break;

    /* Skip leading blanks. */
    while( isspace( *char_ref ) )
      char_ref++;

    /* The key (if there is one). */
    key[ 0 ] ='\0';
    sscanf( char_ref, "%[^:]", key );

   if( strcmp( key, KEY_VER_NO ) == 0 ) {
     sscanf( char_ref, "%[^:]: %d", key, &version );
     break;
   }

  } while( True );


  rewind( file_ref );


  return( version );

} /* fetchFileVersion */


/*----------------------------------------------------------------------*/

static void
  getDataFromFile( UINT32                  do_flags,
                   FILE                    *file_ref,
                   XTM_GL_CUSTOM_DATA_REF  custom_data )
{

  /* Variables. */
  Boolean               new_state;
  Boolean               ok;
  char                  buffer[ 500 ];
  char                  key[ 50 ];
  char                  *char_ref;
  LST_STATUS            lst_status;
  READ_STATE            old_state;
  READ_STATE            state;
  XTM_GD_ENTRY          group;
  XTM_AR_ARCHIVE_FILE   archive_file;
  XTM_CD_CAL_INFO       db_info;
  XTM_CD_INCL_CALS      db_incl;


  /* Code. */

  state     = UNKNOWN_DEF;
  old_state = state;

  /* Start reading from the file, skip all blank lines and comments. */
  do {

    new_state = False;

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );

    /* Read until EOF. */
    if( char_ref == NULL ) {

      state = END_DEF;
      new_state = True;

    } else {

      /* Skip leading blanks. */
      while( isspace( *char_ref ) )
	char_ref++;

      /* The key (if there is one). */
      key[ 0 ] ='\0';
      sscanf( char_ref, "%[^:]", key );

      /* Blank line? */
      if( *char_ref == '\0' ) {
	;

      /* Comment? */
      } else if( *char_ref == '#' || *char_ref == '!' ) {
	;

      } else if( strcmp( key, KEY_VER_KEY ) == 0 ) {
	new_state = True;
	state     = VER_DEF;

      } else if( strcmp( key, KEY_CAL_KEY     ) == 0 ||
		 strcmp( key, OLD_KEY_CAL_KEY ) == 0 ) {
	new_state = True;
	state     = CAL_DEF;

      } else if( strcmp( key, KEY_GROUP_KEY ) == 0 ) {
	new_state = True;
	state     = GROUP_DEF;

      } else if( strcmp( key, KEY_ARCH_KEY ) == 0 ) {
	new_state = True;
	state     = ARCH_DEF;

      } else if( strcmp( key, KEY_INCL_KEY ) == 0 ) {
	new_state = True;
	state     = INCL_DEF;

      } else if( strcmp( key, KEY_PAR_KEY ) == 0 ) {
	new_state = True;
	state     = PAR_DEF;

      } else if( strcmp( key, KEY_FLAG_KEY ) == 0 ) {
	new_state = True;
	state     = FLAG_DEF;


      /* Version definition. */
      } else if( state == VER_DEF ) {
	if( strcmp( key, KEY_VER_NO ) == 0 )
	  ;


      /* Calendar definition. */
      } else if( state == CAL_DEF ) {
	if( flagIsSet( do_flags, (GET_ALL|GET_CALENDARS) ) )
	  (void) readCalendarInfo( custom_data, file_ref, key, char_ref,
				   &db_info, &db_incl );

      /* Group definitions. */
      } else if( state == GROUP_DEF ) {
	if( flagIsSet( do_flags, (GET_ALL|GET_GROUPS) ) )
	  (void) readGroupInfo( custom_data, file_ref, key, char_ref, &group );

      /* Archive definitions. */
      } else if( state == ARCH_DEF ) {
	if( flagIsSet( do_flags, (GET_ALL|GET_ARCHIVES) ) )
	  (void) readArchiveInfo( custom_data, file_ref, key, char_ref,
				  &archive_file );

      /* Include file definitions. */
      } else if( state == INCL_DEF ) {
	(void) readInclInfo( custom_data, do_flags, file_ref, key, char_ref );

      /* Parameter definitions. */
      } else if( state == PAR_DEF ) {
	if( flagIsSet( do_flags, (GET_ALL|GET_PARAMETERS) ) )
	  (void) readParameterInfo( custom_data, file_ref, key, char_ref );

      /* Flag definitions. */
      } else if( state == FLAG_DEF ) {
	if( flagIsSet( do_flags, (GET_ALL|GET_FLAGS) ) )
	  (void) readFlagInfo( custom_data, file_ref, key, char_ref );

      } /* if */

    } /* if */


    /* Actions when we change state. */
    if( new_state ) {

      switch( old_state ) {

        case CAL_DEF:
          if( flagIsSet( do_flags, (GET_ALL|GET_CALENDARS) ) )
            ok = xtmCdAddEntry( custom_data -> cal_db_handle,
                                &db_info, &db_incl );
          break;

        case GROUP_DEF:
          if( flagIsSet( do_flags, (GET_ALL|GET_GROUPS) ) )
            (void) xtmGdAddEntry( custom_data -> group_db_handle,
                                  group.name, &group );

          SysFree( group.members );
          break;

        case ARCH_DEF:
          if( flagIsSet( do_flags, (GET_ALL|GET_ARCHIVES) ) )
            lst_status = LstLinkInsertLast( custom_data -> archive_files, 
                                            &archive_file );
          break;

        case FLAG_DEF:
        case INCL_DEF:
        case PAR_DEF:
        case VER_DEF:
          break;

        default:
          break;

      } /* switch */

      old_state = state;

    } /* if */

  } while( state != END_DEF );


} /* getDataFromFile */


/*----------------------------------------------------------------------*/

static Boolean
  readArchiveInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                   FILE                    *file_ref,
                   char                    *key,
                   char                    *line,
                   XTM_AR_ARCHIVE_FILE     *archive_ref )
{

  /* Variables. */
  char  file_name[ PATH_MAX + 1 ];


  /* Code. */

  if( strcmp( key, KEY_ARCH_NAME ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, archive_ref -> short_name );

    archive_ref  -> flags     = 0;
    archive_ref  -> file_name = (char *) SysMalloc( 1 );
    *archive_ref -> file_name = '\0';

  } else if( strcmp( key, KEY_ARCH_FILE ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, file_name );

    archive_ref -> file_name = SysExpandFilename( file_name );

  } else {
    fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );

  } /* if */


  return( True );

} /* readArchiveInfo */


/*----------------------------------------------------------------------*/

static Boolean
  readCalendarInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref,
                    char                    *key,
                    char                    *line,
                    XTM_CD_CAL_INFO         *db_info_ref,
                    XTM_CD_INCL_CALS        *db_incl_ref )
{

  /* Variables. */
  int          display;
  int          index;
  int          status;
  char         buffer[ 128 ];
  char         cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char         flags_str[ 20 ];
  struct stat  file_status;


  /* Code. */

  /* New calendar? */
  if( strcmp( key, KEY_CAL_NAME ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, db_info_ref -> short_name );

    db_info_ref -> has_extern_pwd    = False;
    db_info_ref -> flags             = 0;
    db_info_ref -> alarm_tags[ 0 ]   = '\0';
    db_info_ref -> extern_host[ 0 ]  = '\0';
    db_info_ref -> extern_pwd[ 0 ]   = '\0';
    db_info_ref -> extern_uid[ 0 ]   = '\0';
    db_info_ref -> extern_type[ 0 ]  = '\0';
    db_info_ref -> mail_address[ 0 ] = '\0';
    db_info_ref -> view_tags[ 0 ]    = '\0';
    db_info_ref -> include           = NULL;
    db_info_ref -> incl_bg_index     = 0;
    db_info_ref -> win_stack_order   = 0;

    db_incl_ref -> no = 0;

    if( custom_data -> incl_depth > 0 )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_IS_INCLUDED );


  /* Calendar location. */
  } else if( strcmp( key, KEY_CAL_LOCATION     ) == 0 ||
	     strcmp( key, OLD_KEY_CAL_LOCATION ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, db_info_ref -> directory );

    /* Database operations possible. */
    xtmDbCheckDbOperations( db_info_ref -> directory, True, 
			    &db_info_ref -> operations );

    /* Owner? */
    db_info_ref -> owner_uid = -1;

    status = stat( db_info_ref -> directory, &file_status );
    if( status == 0 )
      db_info_ref -> owner_uid = file_status.st_uid;


  /* External host? */
  } else if( strcmp( key, KEY_CAL_EXT_HOST ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> extern_host );


  /* External password? */
  } else if( strcmp( key, KEY_CAL_EXT_PWD ) == 0 ) {
    db_info_ref -> has_extern_pwd = True;

    sscanf( line, "%[^:]: %[^\n]", key, buffer );
    ScrDecryptFromNumString( NULL, buffer, db_info_ref -> extern_pwd );


  /* External user ID? */
  } else if( strcmp( key, KEY_CAL_EXT_UID ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> extern_uid );


  /* External user ID? */
  } else if( strcmp( key, KEY_CAL_EXT_TYPE ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> extern_type );


  /* Mail address? */
  } else if( strcmp( key, KEY_CAL_MAIL ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> mail_address );


  /* Background when included. */
  } else if( strcmp( key, KEY_CAL_INCL_BG_COLOR ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &db_info_ref -> incl_bg_index );


  /* Calendar stacking order. */
  } else if( strcmp( key, KEY_CAL_STACK_ORDER ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &db_info_ref -> win_stack_order );


  /* Flags. */
  } else if( strcmp( key, KEY_CAL_FLAGS ) == 0 ) {
    flags_str[ 0 ] = '\0';

    sscanf( line, "%[^:]: %[^ \n]", key, flags_str );

    /* Check the flags. */
    if( strchr( flags_str, 'A' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_ALARM );

    if( strchr( flags_str, 'D' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_DEFAULT_DB );

    if( strchr( flags_str, 'E' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_IS_EXTERNAL );

    if( strchr( flags_str, 'H' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_HIDE_IN_CAL );

    if( strchr( flags_str, 'N' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_NOTIFY );

    if( strchr( flags_str, 'S' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_DEFAULT_START_ICON );

    if( strchr( flags_str, 'W' ) != NULL )
      flagSet( db_info_ref -> flags, XTM_CD_FLAG_DEFAULT_START_WIN );


  /* View tags. */
  } else if( strcmp( key, KEY_CAL_VIEW_TAGS ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> view_tags );
    strcat( db_info_ref -> view_tags, " " );


  /* Alarm tags. */
  } else if( strcmp( key, KEY_CAL_ALARM_TAGS ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, db_info_ref -> alarm_tags );
    strcat( db_info_ref -> alarm_tags, " " );


  /* Included calendars. */
  } else if( strcmp( key, KEY_CAL_INCL_TAGS ) == 0 ) {
    index = db_incl_ref -> no;

    if( index < XTM_CD_MAX_CAL_INCL ) {
      display = 1;

      sscanf( line, "%[^:]: %s %d", key, cal_name, &display );

      if( display == 1 ) {
        strcpy( db_incl_ref -> db[ index ].name, cal_name );

        db_incl_ref -> no++;
      }
    }


  } else {
    fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );

  } /* if */


  return( True );

} /* readCalendarInfo */


/*----------------------------------------------------------------------*/

static Boolean
  readFlagInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                FILE                    *file_ref,
                char                    *key,
                char                    *line )
{

  /* Variables. */
  int  flag;


  /* Code. */

  if( strcmp( key, OLD_KEY_FLAG_US_DATE ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    if( flag == 1 ) {
      strcpy( custom_data -> date_format, "MDY/0100" );
      strcpy( custom_data -> time_format, "12ampm:0" );
      custom_data -> default_paper_size = 3;
    }

  } else if( strcmp( key, KEY_FLAG_CONF ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> confirm_actions = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_ALARM ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> give_alarm = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_ALARM_NEW_APP ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> alarm_for_new_app = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_MONTH_EXTEND ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> cal_month_extend = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_LIST_LAY ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> list_layout = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_NAV_CAL ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> display_nav_cal = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_PROMPT_REPEAT ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> prompt_repeat_win = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_PR_WORKDAYS ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> pr_only_workdays = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_PR_IN_3D ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> pr_in_3d = (Boolean) flag;

  } else if( strcmp( key, OLD_KEY_FLAG_MON_DAY1 ) == 0 ) {
    ;

  } else if( strcmp( key, KEY_FLAG_SHOW_STAND ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> show_stand_in_cal = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_SHOW_TO_DO ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> show_day_to_do = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_TRUE_COLOR_INC ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> true_color_included = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_WEEK_NO ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> show_week_numbers = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_DO_REFRESH ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> do_refresh = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_ENTRY_FLAGS ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> display_entry_flags = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_ENTRY_INTER ) == 0 ) {
    sscanf( line, "%[^:]: %d", key, &flag );
    custom_data -> entries_interactive = (Boolean) flag;

  } else if( strcmp( key, KEY_FLAG_COMPRESSED ) == 0 ) {
    ;

  } else {
    fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );

  } /* if */


  return( True );

} /* readFlagInfo */


/*----------------------------------------------------------------------*/

static Boolean
  readGroupInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                 FILE                    *file_ref,
                 char                    *key,
                 char                    *line,
                 XTM_GD_ENTRY_REF        group_ref )
{

  /* Variables. */
  char  members[ 5 * 200 + 1 ];
  char  *char_ref;


  /* Code. */

  if( strcmp( key, KEY_GROUP_NAME ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, group_ref -> name );

    group_ref -> flags    = 0;
    group_ref -> members  = (char *) SysMalloc( 1 );
    *group_ref -> members = '\0';

    if( custom_data -> incl_depth > 0 )
      flagSet( group_ref -> flags, XTM_GD_FLAG_IS_INCLUDED );

  } else if( strcmp( key, KEY_GROUP_MEMBERS ) == 0 ) {

    sscanf( line, "%[^:]: %[^\n]", key, members );

    /* Remove all , in the string. */
    char_ref = members;

    while( *char_ref != '\0' ) {
      if( *char_ref == ',' )
        *char_ref = ' ';
      char_ref++;
    }

    group_ref -> members = 
      (char *) SysRealloc( group_ref -> members,
                           strlen( group_ref -> members ) +
                           strlen( members ) + 2 );

    strcat( group_ref -> members, members );
    strcat( group_ref -> members, " " );


  } else {
    fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );

  } /* if */


  return( True );

} /* readGroupInfo */


/*----------------------------------------------------------------------*/

static Boolean
  readInclInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                UINT32                  do_flags,
                FILE                    *file_ref,
                char                    *key,
                char                    *line )
{

  /* Variables. */
  char                 file_name[ PATH_MAX + 1 ];
  FILE                 *new_file_ref;
  XTM_GL_INCLUDE_FILE  incl;


  /* Code. */

  if( strcmp( key, KEY_INCL_FILE ) == 0 ) {
    sscanf( line, "%[^:]: %[^ \n]", key, file_name );

    /* Save the include file? */
    if( custom_data -> incl_depth == 0 &&
        LstLinkSearchFirst( 
          custom_data -> include_files, (void *) file_name,
          (EQUALS_FUNC_TYPE) compareInclCB ) == LST_NOT_FOUND ) {

      incl.orig_file_name = SysNewString( file_name );
      incl.file_name      = SysExpandFilename( file_name );

      (void) LstLinkInsertLast( custom_data -> include_files, &incl );

    } /* if */

    /* Too deply nested? */
    if( custom_data -> incl_depth >= 5 ) {
      fprintf( stderr, "%s: Include files todeeply nested!\n", module_name );

      return( True );
    }

    /* Open the new customize file and read the data. */
    new_file_ref = fopen( file_name, "r" );
    if( new_file_ref != NULL ) {

      custom_data -> incl_depth++;
      getDataFromFile( do_flags, new_file_ref, custom_data );
      custom_data -> incl_depth--;

      fclose( new_file_ref );
    }

    return( True );

  } /* if */


  /* Unknown line. */
  fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );


  return( True );

} /* readInclInfo */


/*----------------------------------------------------------------------*/

static Boolean
  readParameterInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                     FILE                    *file_ref,
                     char                    *key,
                     char                    *line )
{

  /* Variables. */
  int   index;
  char  buffer[ 500 ];
  char  *char_ref;


  /* Code. */

  if( strcmp( key, KEY_PAR_ALARM_LINES ) == 0 ) {
    sscanf( line, "%[^:]: %d", 
	    key, &custom_data -> alarm_show_lines );

  } else if( strcmp( key, KEY_PAR_VIEW_DAYS ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> day_view_days );

  } else if( strcmp( key, KEY_PAR_VIEW_START ) == 0 ) {
    sscanf( line, "%[^:]: %d", 
	    key, &custom_data -> start_hour );

  } else if( strcmp( key, KEY_PAR_VIEW_DELTA ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> default_entry_delta );

  } else if( strcmp( key, KEY_PAR_ENTRY_LINES ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> entry_show_lines );

  } else if( strcmp( key, KEY_PAR_VIEW_STOP ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> stop_hour );

  } else if( strcmp( key, KEY_PAR_DAY_WIDTH ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> def_day_width );

  } else if( strcmp( key, KEY_PAR_DV_DAY_SWITCH ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> day_view_day_switch );

  } else if( strcmp( key, KEY_PAR_TUNE ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> default_tune );

  } else if( strcmp( key, KEY_PAR_REFRESH_RATE ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> refresh_rate );

  } else if( strcmp( key, KEY_PAR_START_DV_ON ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> start_day_view_on );

  } else if( strcmp( key, KEY_PAR_SUMMARY_DEF_LINES ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> summary_def_lines );

  } else if( strcmp( key, KEY_PAR_PAPER_SIZE ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> default_paper_size );

  } else if( strcmp( key, KEY_PAR_SHEETS_PER_PAGE ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> print_def_sheets );

  } else if( strcmp( key, KEY_PAR_PRINT_DEF_LINES ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> print_def_lines );

  } else if( strcmp( key, KEY_PAR_MSG_LANGUAGE ) == 0 ) {
    sscanf( line, "%[^:]: %s",
	    key, custom_data -> msg_language );

  } else if( strcmp( key, KEY_PAR_NOTE_PANE_HEIGHT ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> note_pane_height );

  } else if( strcmp( key, KEY_PAR_APP_PANE_HEIGHT ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> app_pane_height );

  } else if( strcmp( key, KEY_PAR_ENTRY_PANE_WIDTH ) == 0 ) {
    sscanf( line, "%[^:]: %d",
	    key, &custom_data -> entry_pane_width );

  } else if( strcmp( key, KEY_PAR_DATE_FORMAT ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, buffer );
    strcpy( custom_data -> date_format, buffer );

  } else if( strcmp( key, KEY_PAR_TIME_FORMAT ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, buffer );
    strcpy( custom_data -> time_format, buffer );

  } else if( strcmp( key, KEY_PAR_WORKDAYS ) == 0 ) {
    sscanf( line, "%[^:]: %[^\n]", key, buffer );

    char_ref = buffer;
    index    = 0;

    while( index < 7 && *char_ref != '\0' ) {
      if( *char_ref == '0' )
        custom_data -> workdays[ index ] = False;
      else
        custom_data -> workdays[ index ] = True;

      index++;
      char_ref++;
    }

  } else {
    fprintf( stderr, "%s: Unknown line:\n  '%s'", module_name, line );

  } /* if */


  return( True );

} /* readParameterInfo */


/*----------------------------------------------------------------------*/

static void
  setDefaultValues( XTM_GL_CUSTOM_DATA_REF  custom_data )
{

  /* Code. */

  strcpy( custom_data -> date_format,  custom_data -> default_date_format );
  strcpy( custom_data -> msg_language, DEFAULT_LANGUAGE );
  strcpy( custom_data -> time_format,  custom_data -> default_time_format );

  custom_data -> workdays[ 0 ] = True;
  custom_data -> workdays[ 1 ] = True;
  custom_data -> workdays[ 2 ] = True;
  custom_data -> workdays[ 3 ] = True;
  custom_data -> workdays[ 4 ] = True;
  custom_data -> workdays[ 5 ] = False;
  custom_data -> workdays[ 6 ] = False;


  return;

} /* setDefaultValues */


/*----------------------------------------------------------------------*/

static Boolean
  writeArchiveInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref )
{

  /* Variables. */
  LST_STATUS            lst_status;
  XTM_AR_ARCHIVE_FILE   archive_file;


  /* Code. */

  /* Save all the calendars. */
  lst_status = LstLinkCurrentFirst( custom_data -> archive_files );

  while( lst_status == LST_OK ) {

    lst_status = LstLinkGetCurrent( custom_data -> archive_files, 
				    &archive_file );

    fprintf( file_ref, "%s:\n", KEY_ARCH_KEY );

    fprintf( file_ref, "\t%s: %s\n", 
	     KEY_ARCH_NAME, archive_file.short_name );

    fprintf( file_ref, "\t%s: %s\n", 
	     KEY_ARCH_FILE, archive_file.file_name );

    fprintf( file_ref, "\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( custom_data -> archive_files );

  } /* while */


  return( True );

} /* writeArchiveInfo */


/*----------------------------------------------------------------------*/

static Boolean
  writeCalendarInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                     FILE                    *file_ref )
{

  /* Variables. */
  Boolean           ok;
  int               char_read;
  int               index;
  char              buffer[ 128 ];
  char              cal_name[ XTM_GL_MAX_CAL_NAME ];
  char              flags_str[ 20 ];
  char              *cal_names;
  char              *cal_ref;
  XTM_CD_HANDLE     cal_db_handle;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  cal_db_handle = custom_data -> cal_db_handle;


  /* Write the user defined calendars. */
  ok = xtmCdFetchDbNames( cal_db_handle, &cal_names );
  if( ! ok )
    return( False );

  cal_ref = cal_names;

  do {

    while( isspace( *cal_ref ) )
      cal_ref++;

    if( *cal_ref == '\0' )
      break;

    char_read = strlen( cal_ref );
    sscanf( cal_ref, "%s%n", cal_name, &char_read );
    cal_ref = cal_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( cal_db_handle, cal_name, &db_info, &db_incl );

    /* Do not save included calendars. */
    if( flagIsSet( db_info.flags, XTM_CD_FLAG_IS_INCLUDED ) )
      continue;

    /* The diary flags. */
    flags_str[ 0 ] = '\0';

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_ALARM ) )
      strcat( flags_str, "A" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_DEFAULT_DB ) )
      strcat( flags_str, "D" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_IS_EXTERNAL ) )
      strcat( flags_str, "E" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_HIDE_IN_CAL ) )
      strcat( flags_str, "H" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_NOTIFY ) )
      strcat( flags_str, "N" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_DEFAULT_START_ICON ) )
      strcat( flags_str, "S" );

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_DEFAULT_START_WIN ) )
      strcat( flags_str, "W" );


    /* Save the database information. */
    fprintf( file_ref, "%s:\n", KEY_CAL_KEY );

    fprintf( file_ref, "\t%s: %s\n", 
             KEY_CAL_NAME, db_info.short_name );

    fprintf( file_ref, "\t%s: %s\n", 
             KEY_CAL_LOCATION, db_info.directory );

    if( strlen( flags_str ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
                 KEY_CAL_FLAGS, flags_str );

    if( strlen( db_info.extern_host ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
               KEY_CAL_EXT_HOST, db_info.extern_host );

    if( db_info.has_extern_pwd && strlen( db_info.extern_pwd ) > 0 ) {
      ScrEncryptToNumString( NULL, db_info.extern_pwd, buffer );

      fprintf( file_ref, "\t%s: %s\n", KEY_CAL_EXT_PWD, buffer );
    }

    if( strlen( db_info.extern_uid ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
               KEY_CAL_EXT_UID, db_info.extern_uid );

    if( strlen( db_info.extern_type ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
               KEY_CAL_EXT_TYPE, db_info.extern_type );

    if( strlen( db_info.mail_address ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
               KEY_CAL_MAIL, db_info.mail_address );

    if( strlen( db_info.view_tags ) > 0 )
      fprintf( file_ref, "\t%s: %s\n",
               KEY_CAL_VIEW_TAGS, db_info.view_tags );

    if( strlen( db_info.alarm_tags ) > 0 )
      fprintf( file_ref, "\t%s: %s\n", 
               KEY_CAL_ALARM_TAGS, db_info.alarm_tags );

    if( db_info.incl_bg_index > 0 )
      fprintf( file_ref, "\t%s: %d\n", 
               KEY_CAL_INCL_BG_COLOR, db_info.incl_bg_index );

    if( db_info.win_stack_order > 0 )
      fprintf( file_ref, "\t%s: %d\n", 
               KEY_CAL_STACK_ORDER, db_info.win_stack_order );


    /* Include diaries. */
    for( index = 0; index < db_incl.no; index++ ) {

      fprintf( file_ref, "\t%s: %s 1\n",
               KEY_CAL_INCL_TAGS, db_incl.db[ index ].name );

    } /* loop */

    fprintf( file_ref, "\n" );

  } while( True );

  SysFree( cal_names );


  return( True );

} /* writeCalendarInfo */


/*----------------------------------------------------------------------*/

static Boolean
  writeFlagInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                 FILE                    *file_ref )
{

  /* Code. */

  fprintf( file_ref, "%s:\n", KEY_FLAG_KEY );

  if( custom_data -> confirm_actions )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_CONF );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_CONF );

  if( custom_data -> give_alarm )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_ALARM );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_ALARM );

  if( custom_data -> alarm_for_new_app )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_ALARM_NEW_APP );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_ALARM_NEW_APP );

  if( custom_data -> cal_month_extend )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_MONTH_EXTEND );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_MONTH_EXTEND );

  if( custom_data -> show_stand_in_cal )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_SHOW_STAND );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_SHOW_STAND );

  if( custom_data -> show_day_to_do )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_SHOW_TO_DO );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_SHOW_TO_DO );

  if( custom_data -> true_color_included )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_TRUE_COLOR_INC );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_TRUE_COLOR_INC );

  if( custom_data -> list_layout )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_LIST_LAY );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_LIST_LAY );

  if( custom_data -> display_nav_cal )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_NAV_CAL );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_NAV_CAL );

  if( custom_data -> prompt_repeat_win )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_PROMPT_REPEAT );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_PROMPT_REPEAT );

  if( custom_data -> pr_only_workdays )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_PR_WORKDAYS );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_PR_WORKDAYS );

  if( custom_data -> pr_in_3d )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_PR_IN_3D );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_PR_IN_3D );

  if( custom_data -> show_week_numbers )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_WEEK_NO );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_WEEK_NO );

  if( custom_data -> do_refresh )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_DO_REFRESH );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_DO_REFRESH );

  if( custom_data -> display_entry_flags )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_ENTRY_FLAGS );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_ENTRY_FLAGS );

  if( custom_data -> entries_interactive )
    fprintf( file_ref, "\t%s: 1\n", KEY_FLAG_ENTRY_INTER );
  else
    fprintf( file_ref, "\t%s: 0\n", KEY_FLAG_ENTRY_INTER );


  return(True );

} /* writeFlagInfo */


/*----------------------------------------------------------------------*/

static Boolean
  writeGroupInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                  FILE                    *file_ref )
{

  /* Variables. */
  Boolean       ok;
  int           char_read;
  char          group_name[ XTM_GD_MAX_GROUP_NAME + 1 ];
  char          *group_name_ref;
  char          *group_names;
  char          *members_ref;
  XTM_GD_HANDLE group_db_handle;
  XTM_GD_ENTRY  group_info;


  /* Code. */

  group_db_handle = custom_data -> group_db_handle;


  /* What groups do we have? */
  ok = xtmGdFetchGroupNames( group_db_handle, &group_names );
  if( ! ok )
    return( False );

  group_name_ref = group_names;

  do {

    while( isspace( *group_name_ref ) )
      group_name_ref++;

    if( *group_name_ref == '\0' )
      break;

    char_read = strlen( group_name_ref );
    sscanf( group_name_ref, "%s%n", group_name, &char_read );
    group_name_ref = group_name_ref + char_read;


    /* Fetch information about the group. */
    (void) xtmGdFetchNamedGroup( group_db_handle, group_name, &group_info );


    /* Do not save included groups. */
    if( flagIsSet( group_info.flags, XTM_GD_FLAG_IS_INCLUDED ) )
      continue;


    /* Save the group information. */
    fprintf( file_ref, "%s:\n", KEY_GROUP_KEY );

    fprintf( file_ref, "\t%s: %s\n", KEY_GROUP_NAME, group_info.name );


    /* Save the members of the group (in clusters of 5). */
    members_ref = group_info.members;

    if( members_ref != NULL ) {

      Boolean  header_written  = False;
      int      members_written = 0;
      char     member[ 250 ];

      do {

        while( isspace( *members_ref ) )
          members_ref++;

        if( *members_ref == '\0' )
          break;

        char_read = strlen( members_ref );
        sscanf( members_ref, "%s%n", member, &char_read );
        members_ref = members_ref + char_read;

        if( members_written >= 5 ) {
          fprintf( file_ref, "\n" );
          header_written  = False;
          members_written = 0;
        }

        if( ! header_written ) {
          fprintf( file_ref, "\t%s: ", KEY_GROUP_MEMBERS );
          header_written = True;
        }

        fprintf( file_ref, "%s ", member );
        members_written++;

      } while( True );

      if( header_written )
        fprintf( file_ref, "\n" );

    } /* if */

    SysFree( group_info.members );

    fprintf( file_ref, "\n" );

  } while( True );

  SysFree( group_names );


  return( True );

} /* writeGroupInfo */


/*----------------------------------------------------------------------*/

static Boolean
  writeIncludeInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                    FILE                    *file_ref )
{

  /* Variables. */
  Boolean               first = True;
  LST_STATUS            lst_status;
  XTM_GL_INCLUDE_FILE   incl_file;


  /* Code. */

  /* Save include file names. */
  lst_status = LstLinkCurrentFirst( custom_data -> include_files );

  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( custom_data -> include_files, &incl_file );

    if( first ) {
      fprintf( file_ref, "%s:\n", KEY_INCL_KEY );
      first = False;
    }

    fprintf( file_ref, "\t%s: %s\n", 
	     KEY_INCL_FILE, incl_file.orig_file_name );

    fprintf( file_ref, "\n" );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( custom_data -> include_files );

  } /* while */


  return( True );

} /* writeIncludeInfo */


/*----------------------------------------------------------------------*/

static Boolean
  writeParameterInfo( XTM_GL_CUSTOM_DATA_REF  custom_data,
                      FILE                    *file_ref )
{

  /* Variables. */
  int  index;


  /* Code. */

  fprintf( file_ref, "%s:\n", KEY_PAR_KEY );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_ALARM_LINES, 
           custom_data -> alarm_show_lines );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_VIEW_DAYS, 
           custom_data -> day_view_days );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_VIEW_START, 
           custom_data -> start_hour );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_VIEW_STOP, 
           custom_data -> stop_hour );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_DAY_WIDTH, 
           custom_data -> def_day_width );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_DV_DAY_SWITCH, 
           custom_data -> day_view_day_switch );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_VIEW_DELTA, 
           custom_data -> default_entry_delta );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_ENTRY_LINES, 
           custom_data -> entry_show_lines );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_TUNE, 
           custom_data -> default_tune );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_REFRESH_RATE, 
           custom_data -> refresh_rate );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_START_DV_ON, 
           custom_data -> start_day_view_on );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_PAPER_SIZE, 
           custom_data -> default_paper_size );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_SHEETS_PER_PAGE, 
           custom_data -> print_def_sheets );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_PRINT_DEF_LINES, 
           custom_data -> print_def_lines );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_SUMMARY_DEF_LINES, 
           custom_data -> summary_def_lines );

  fprintf( file_ref, "\t%s: %s\n", KEY_PAR_MSG_LANGUAGE, 
           custom_data -> msg_language );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_NOTE_PANE_HEIGHT, 
           custom_data -> note_pane_height );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_APP_PANE_HEIGHT, 
           custom_data -> app_pane_height );

  fprintf( file_ref, "\t%s: %d\n", KEY_PAR_ENTRY_PANE_WIDTH, 
           custom_data -> entry_pane_width );

  fprintf( file_ref, "\t%s: %s\n", KEY_PAR_DATE_FORMAT, 
           custom_data -> date_format );

  fprintf( file_ref, "\t%s: %s\n", KEY_PAR_TIME_FORMAT, 
           custom_data -> time_format );

  fprintf( file_ref, "\t%s: ", KEY_PAR_WORKDAYS ); 
  for( index = 0; index < 7; index++ ) {
    if( custom_data -> workdays[ index ] )
      fprintf( file_ref, "1" );
    else
      fprintf( file_ref, "0" );
  }
  fprintf( file_ref, "\n" );

  fprintf( file_ref, "\n" );


  return( True );

} /* writeParameterInfo */


/*----------------------------------------------------------------------*/

static void
  clearArchCB( void  *element )
{

  /* Variables. */
  XTM_AR_ARCHIVE_FILE  *entry_ref;


  /* Code. */

  entry_ref = (XTM_AR_ARCHIVE_FILE *) element;

  if( entry_ref -> file_name != NULL )
    SysFree( entry_ref -> file_name );


  return;

} /* clearArchCB */


/*----------------------------------------------------------------------*/

static void
  clearInclCB( void  *element )
{

  /* Variables. */
  XTM_GL_INCLUDE_FILE  *entry_ref;


  /* Code. */

  entry_ref = (XTM_GL_INCLUDE_FILE *) element;

  if( entry_ref -> file_name != NULL )
    SysFree( entry_ref -> file_name );

  if( entry_ref -> orig_file_name != NULL )
    SysFree( entry_ref -> orig_file_name );


  return;

} /* clearInclCB */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  compareInclCB( XTM_GL_INCLUDE_FILE  *incl_ref,
                 char                 *incl_file )
{

  /* Code. */

  if( strcmp( incl_ref -> orig_file_name, incl_file ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* compareInclCB */
