/*----------------------------------------------------------------------------
--
--  Module:           xtmDbInclude
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Selection window with databases to include.
--
--  Filename:         xtmDbInclude.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-07
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDbInclude.c, Version: 1.1, Date: 95/02/18 15:52:08";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/ScrolledW.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmDbInclude.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the include window. */
#define incLa          dataLocalW[ 0 ]
#define incSw          dataLocalW[ 1 ]
#define incRc          dataLocalW[ 2 ]
#define setClearRc     dataLocalW[ 3 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Include record. */
typedef struct {

  /* The current calendar. */
  char  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The include window. */
  Widget  includeW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_DI_ACTION_CB  actionCB;

} INCLUDE_REC, *INCLUDE_REC_REF;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmDbInclude";

/* IDs for the help windows. */
static char  *include_window_id = "CalInclude";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  applyCB( Widget           widget,
           INCLUDE_REC_REF  include_ref,
           XtPointer        call_data );

static void
  cancelCB( Widget           widget,
            INCLUDE_REC_REF  include_ref,
            XtPointer        call_data );

static void
  clearAllCB( Widget           widget,
              INCLUDE_REC_REF  include_ref,
              XtPointer        call_data );

static int
  compareStringsCB( const void  *string1,
                    const void  *string2 );

static Widget
  createIncludeWindow( INCLUDE_REC_REF  include_ref,
                       Widget           parent,
                       Boolean          app_modal );

static void 
  destroyCB( Widget           widget,
             INCLUDE_REC_REF  include_ref,
             XtPointer        call_data );

static void
  fetchWindowData( INCLUDE_REC_REF       include_ref,
                   XTM_CD_INCL_CALS_REF  db_incl );

static void 
  helpCB( Widget           widget,
          INCLUDE_REC_REF  include_ref,
          XtPointer        call_data );

static void
  okCB( Widget           widget,
        INCLUDE_REC_REF  include_ref,
        XtPointer        call_data );

static void
  setAllCB( Widget           widget,
            INCLUDE_REC_REF  include_ref,
            XtPointer        call_data );

static void
  setClearAll( INCLUDE_REC_REF  include_ref,
               Boolean          state );

static void
  setIncludeValues( INCLUDE_REC_REF  include_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_DI_HANDLE
  xtmDiInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   Boolean               app_modal,
                   XTM_DI_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  INCLUDE_REC_REF  include_ref;


  /* Code. */

  /* Create and initialize our private data. */
  include_ref = SysNew( INCLUDE_REC );
  if( include_ref == NULL )
    return( NULL );

  include_ref -> appl_data_ref = appl_data_ref;
  include_ref -> db_name[ 0 ]  = '\0';
  include_ref -> actionCB      = actionCB;
  include_ref -> user_data     = user_data;


  /* Create the include window. */
  include_ref -> includeW = createIncludeWindow( include_ref, parent,
                                                 app_modal );
  if( include_ref -> includeW == NULL ) {
    SysFree( include_ref );

    return( NULL );
  }


  return( (XTM_DI_HANDLE) include_ref );

} /* xtmDiInitialize */


/*----------------------------------------------------------------------*/

void
  xtmDiDestroy( XTM_DI_HANDLE  include_handle )
{

  /* Variables. */
  INCLUDE_REC_REF  include_ref;


  /* Code. */

  if( include_handle == NULL )
    return;

  /* Our private data. */
  include_ref = (INCLUDE_REC_REF) include_handle;


  /* Destroy the window. */
  XtDestroyWidget( include_ref -> includeW );


  return;

} /* xtmDiDestroy */


/*----------------------------------------------------------------------*/

void
  xtmDiDisplayIncludeWindow( XTM_DI_HANDLE  include_handle,
                             char           *cal_name )
{

  /* Variables. */
  INCLUDE_REC_REF  include_ref;


  /* Code. */

  if( include_handle == NULL )
    return;

  /* Our private data. */
  include_ref = (INCLUDE_REC_REF) include_handle;


  /* Display the calendars. */
  if( strlen( cal_name ) > 0 ) {
    strcpy( include_ref -> db_name, cal_name );

    setIncludeValues( include_ref );
  }


  /* Make sure the window is displayed. */
  XtManageChild( include_ref -> includeW );


  return;

} /* xtmDiDisplayIncludeWindow */


/*----------------------------------------------------------------------*/

static Widget
  createIncludeWindow( INCLUDE_REC_REF  include_ref,
                       Widget           parent,
                       Boolean          app_modal )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 4 ];
  Widget    incFd;
  Widget    setClearBu[ 2 ];
  Widget    workFo;

  static XIT_PUSH_STRUCT set_clear_def[] = {
    { "SetAllPb",   "", "", True, NULL },
    { "ClearAllPb", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", applyCB,  NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = include_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 1 ].data  = include_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = include_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = include_ref;

  set_clear_def[ 0 ].title = msgGetText( MXDI_CLEAR_ALL_LABEL );
  set_clear_def[ 1 ].title = msgGetText( MXDI_SELECT_ALL_LABEL );


  /* Create a form dialog with buttons. */
  incFd = xitCreateFormDialog( parent, "IncFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  XtAddCallback( incFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) include_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_INCLUDE_DB_TITLE ) ); n++;
  XtSetValues( XtParent( incFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( incFd, args, n );

  if( app_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( incFd, args, n );
  }


  /* Create a container for the contents of the window. */
  workFo = XtNameToWidget( incFd, "IncFdFo" );

  incLa = xitCreateLabel( workFo, "IncLa", 
                          msgGetText( MXDI_INCLUDE_DB_LABEL ), -1 );

  /* Scrolled window which conatins the databases you include. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmAS_NEEDED ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy,        XmAUTOMATIC ); n++;
  incSw = XmCreateScrolledWindow( workFo, "IncSw", args, n );


  /* The databases you include. */
  n = 0;
  incRc = XmCreateRowColumn( incSw, "IncRc", args, n );


  /* Set/clear all buttons. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  setClearRc = XmCreateRowColumn( workFo, "SetClearRc", args, n );

  for( index = 0; index < XtNumber( set_clear_def ); index++ )
    setClearBu[ index ] = xitCreatePushButton( setClearRc,
                                               &set_clear_def[ index ] );

  XtAddCallback( setClearBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) clearAllCB, (XtPointer) include_ref );
  XtAddCallback( setClearBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) setAllCB, (XtPointer) include_ref );


  /* Put the Parts together. */
  xitAttachWidget( incLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( incSw,
                   XmATTACH_WIDGET, incLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( setClearRc,
                   XmATTACH_WIDGET, incSw, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( incLa,      args, n );
  XtSetValues( incSw,      args, n );
  XtSetValues( setClearRc, args, n );

  /* Manage the children. */
  xitManageChildren( setClearBu, XtNumber( setClearBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the initial sizes. */
  xitSetSizeFormDialog( incFd, True );


  /* Make the final attachments. */
  xitAttachWidget( setClearRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( incSw,
                   XmATTACH_WIDGET, incLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_WIDGET, setClearRc );


  return( incFd );

} /* createIncludeWindow */


/*----------------------------------------------------------------------*/

static void
  fetchWindowData( INCLUDE_REC_REF       include_ref,
                   XTM_CD_INCL_CALS_REF  db_incl )
{

  /* Variables. */
  int       cal_count;
  int       index;
  char      buffer[ 50 ];
  char      *char_ref;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( include_ref -> includeW, 
                          "IncFdFo.IncSw.ScrolledWindowClipWindow.IncRc" );

  /* Fetch the data. */
  cal_count     = 0;
  index         = 0;
  db_incl -> no = 0;

  do {

    /* Do we have an entry here? */
    sprintf( buffer, "IncDb%dTb", index + 1 );
    tempW = XtNameToWidget( mainW, buffer );
    if( tempW == NULL )
      break;

    /* Fetch name of DB to include. */
    char_ref = xitStringGetLabel( tempW );

    /* Only calendars marked for display. */
    if( XmToggleButtonGetState( tempW ) ) {
      sscanf( char_ref, "%s", db_incl -> db[ cal_count ].name );

      cal_count++;
    }
 
    SysFree( char_ref );

    if( cal_count >= XTM_CD_MAX_CAL_INCL )
      break;

    index++;

  } while( True );

  db_incl -> no = cal_count;


  return;

} /* fetchWindowData */


/*----------------------------------------------------------------------*/

static void
  setClearAll( INCLUDE_REC_REF  include_ref,
               Boolean          state )
{

  /* Variables. */
  int     index;
  char    buffer[ 50 ];
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( include_ref -> includeW, 
                          "IncFdFo.IncSw.ScrolledWindowClipWindow.IncRc" );

  /* Set or clear all include indicators. */
  index = 0;

  do {

    /* Do we have an entry here? */
    sprintf( buffer, "IncDb%dTb", index + 1 );
    tempW = XtNameToWidget( mainW, buffer );
    if( tempW == NULL )
      break;

    XmToggleButtonSetState( tempW, state, False );

    index++;

  } while( True );


  return;

} /* setClearAll */


/*----------------------------------------------------------------------*/

static void
  setIncludeValues( INCLUDE_REC_REF  include_ref )
{

  /* Type definitions. */
  typedef  char  *str_ref;


  /* Variables. */
  Boolean                 ok;
  int                     cal_count;
  int                     index;
  char                    buffer[ 100 ];
  char                    *calendars[ 200 ];
  char                    *char_ref;
  char                    *db_names;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         curr_db_info;
  XTM_CD_INCL_CALS        curr_db_incl;


  /* Code. */

  custom_data_ref = include_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( include_ref -> includeW, "IncFdFo" );


  /* Fetch the calendar database. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          include_ref -> db_name,
                          &curr_db_info, &curr_db_incl );
  if( ! ok )
    return;


  /* Remove all old calendars. */
  baseW = XtNameToWidget( mainW, "IncSw.ScrolledWindowClipWindow.IncRc" );
  index = 1;

  do {
    sprintf( buffer, "IncDb%dTb", index );

    tempW = XtNameToWidget( baseW, buffer );
    if( tempW == NULL )
      break;

    XtDestroyWidget( tempW );
    index++;
  } while( True );


  /* Display all calendars we know of. */
  ok = xtmCdFetchDbNames( custom_data_ref -> cal_db_handle, &db_names );
  char_ref  = db_names;
  cal_count = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the calendar. */
    (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                              &db_info, NULL );


    /* Not the current calendar. */
    if( strcmp( db_info.short_name, curr_db_info.short_name ) == 0 )
      continue;

    /* Only calendars we can read. */
    if( flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      continue;

    /* Calendar already included? */
    if( xtmCdSearchIncludeDb( &curr_db_incl, db_info.short_name ) )
      sprintf( buffer, "%s 1", db_info.short_name );
    else
      sprintf( buffer, "%s 0", db_info.short_name );


    /* Save the calandar names to be sorted. */
    calendars[ cal_count ] = SysNewString( buffer );

    cal_count++;
    if( cal_count >= 200 )
      break;

  } while( True );

  SysFree( db_names );


  /* Sort the calendars. */
  qsort( (void *) calendars,
         (size_t) cal_count, sizeof( str_ref ),
         compareStringsCB );


  /* Create the list of calendars. */
  baseW = XtNameToWidget( mainW, "IncSw.ScrolledWindowClipWindow.IncRc" );

  for( index = 0; index < cal_count; index++ ) {

    int   included;
    char  widget_name[ 50 ];

    sprintf( widget_name, "IncDb%dTb", index + 1 );

    sscanf( calendars[ index ], "%s %d", buffer, &included );
    if( included == 1 )
      tempW = xitCreateToggleButton( baseW, widget_name, 
                                     buffer, True );
    else
      tempW = xitCreateToggleButton( baseW, widget_name, 
                                     buffer, False );

    XtManageChild( tempW );

    SysFree( calendars[ index ] );

  } /* loop */


  return;

} /* setIncludeValues */


/*----------------------------------------------------------------------*/

static void
  applyCB( Widget           widget,
           INCLUDE_REC_REF  include_ref,
           XtPointer        call_data )
{

  /* Variables. */
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  /* Fetch the window data. */
  fetchWindowData( include_ref, &db_incl );


  /* Call callback function? */
  if( include_ref -> actionCB != NULL )
    (* include_ref -> actionCB)( XTM_DI_REASON_APPLY,
                                 &db_incl, include_ref -> user_data );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget           widget,
            INCLUDE_REC_REF  include_ref,
            XtPointer        call_data )
{

  /* Code. */

  /* Call callback function? */
  if( include_ref -> actionCB != NULL )
    (* include_ref -> actionCB)( XTM_DI_REASON_CANCEL,
                                 NULL, include_ref -> user_data );

  XtUnmanageChild( include_ref -> includeW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void
  clearAllCB( Widget           widget,
              INCLUDE_REC_REF  include_ref,
              XtPointer        call_data )
{

  /* Code. */

  setClearAll( include_ref, False );


  return;

} /* clearAllCB */


/*----------------------------------------------------------------------*/

static int
  compareStringsCB( const void  *string1,
                    const void  *string2 )
{

  /* Code. */

  return( strcmp( (char *) string1, (char *) string2 ) );

} /* compareStringsCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget           widget,
             INCLUDE_REC_REF  include_ref,
             XtPointer        call_data )
{

  /* Code. */

  SysFree( include_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget           widget,
          INCLUDE_REC_REF  include_ref,
          XtPointer        call_data )
{

  /* Code. */

  xtmHlDisplayHelp( include_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    include_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget           widget,
        INCLUDE_REC_REF  include_ref,
        XtPointer        call_data )
{

  /* Variables. */
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  /* Fetch the window data. */
  fetchWindowData( include_ref, &db_incl );


  /* Call callback function? */
  if( include_ref -> actionCB != NULL )
    (* include_ref -> actionCB)( XTM_DI_REASON_OK,
                                 &db_incl, include_ref -> user_data );


  /* Unmap the window. */
  XtUnmanageChild( include_ref -> includeW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void
  setAllCB( Widget           widget,
            INCLUDE_REC_REF  include_ref,
            XtPointer        call_data )
{

  /* Code. */

  setClearAll( include_ref, True );


  return;

} /* setAllCB */


