/*----------------------------------------------------------------------------
--
--  Module:           xtmEditSkip
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Edit the preferences for an XDiary entry.
--
--  Filename:         xtmEditSkip.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEditSkip.c, Version: 1.1, Date: 95/02/18 15:52:17";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmEditSkip.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the window. */
#define resetPb           dataLocalW[  0 ]
#define skipLa            dataLocalW[  1 ]
#define skipSw            dataLocalW[  2 ]
#define skipThisWeekTb    dataLocalW[  3 ]
#define skipWeekRc        dataLocalW[  4 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the window. */
typedef struct {

  /* Skip window. */
  Widget  skipW;

  /* Default date. */
  TIM_TIME_REF  default_date;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_EK_ACTION_CB  actionCB;
  void              *user_data;

} SKIP_REC, *SKIP_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditSkip";

/* IDs for the help windows. */
static char  *skip_window_id = "EntrySkip";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            SKIP_REC_REF  skip_ref,
            XtPointer     call_data );

static Widget
  createSkipWindow( Widget        parent,
                    SKIP_REC_REF  skip_ref );

static void 
  destroyCB( Widget        widget,
             SKIP_REC_REF  skip_ref,
             XtPointer     call_data );

static Boolean
  fetchSkipValues( SKIP_REC_REF          skip_ref,
                   XTM_EK_GET_SKIP_DATA  *values_ref );

static void 
  helpCB( Widget        widget,
          SKIP_REC_REF  skip_ref,
          XtPointer     call_data );

static void 
  okCB( Widget        widget,
        SKIP_REC_REF  skip_ref,
        XtPointer     call_data );

static void 
  resetCB( Widget        widget,
           SKIP_REC_REF  skip_ref,
           XtPointer     call_data );

static void
  setSkipValues( SKIP_REC_REF          skip_ref,
                 XTM_EK_SET_SKIP_DATA  *values_ref );

static void 
  skipThisWeekCB( Widget                        widget,
                  SKIP_REC_REF                  skip_ref,
                  XmToggleButtonCallbackStruct  *call_data );

static void 
  toggleWeekCB( Widget                        widget,
                SKIP_REC_REF                  skip_ref,
                XmToggleButtonCallbackStruct  *call_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_EK_HANDLE
  xtmEkInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_EK_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  SKIP_REC_REF  skip_ref;


  /* Code. */

  /* Create and initialize our private data. */
  skip_ref = SysNew( SKIP_REC );
  if( skip_ref == NULL )
    return( NULL );

  skip_ref -> appl_data_ref = appl_data_ref;
  skip_ref -> actionCB      = actionCB;
  skip_ref -> user_data     = user_data;
  skip_ref -> skipW         = NULL;


  /* Create the base window. */
  skip_ref -> skipW = createSkipWindow( parent, skip_ref );
  if( skip_ref -> skipW == NULL ) {
    SysFree( skip_ref );

    return( NULL );
  }


  return( (XTM_EK_HANDLE) skip_ref );

} /* xtmEkInitialize */


/*----------------------------------------------------------------------*/

void
  xtmEkDestroy( XTM_EK_HANDLE  skip_handle )
{

  /* Variables. */
  SKIP_REC_REF  skip_ref;


  /* Code. */

  if( skip_handle == NULL )
    return;

  /* Our private data. */
  skip_ref = (SKIP_REC_REF) skip_handle;

  /* Get rid of all windows. */
  if( skip_ref -> skipW != NULL )
    XtDestroyWidget( skip_ref -> skipW );


  return;

} /* xtmEkDestroy */


/*----------------------------------------------------------------------*/

void
  xtmEkEditSkip( XTM_EK_HANDLE         skip_handle,
                 XTM_EK_SET_SKIP_DATA  *values_ref )
{

  /* Variables. */
  SKIP_REC_REF  skip_ref;


  /* Code. */

  /* Our private data. */
  skip_ref = (SKIP_REC_REF) skip_handle;


  /* Set the window data. */
  setSkipValues( skip_ref, values_ref );


  /* Make sure the window is visible. */
  XtManageChild( skip_ref -> skipW );


  return;

} /* xtmEkEditSkip */


/*----------------------------------------------------------------------*/

static Widget
  createSkipWindow( Widget        parent,
                    SKIP_REC_REF  skip_ref )
{

  /* Variables. */
  int       index;
  char      buffer[ 80 ];
  char      label[ 20 ];
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 5 ];
  Widget    skipWeekTb[ 53 ];
  Widget    workFo;
  Widget    skipFd;


  static XIT_PUSH_STRUCT reset_button_def[] = {
    { "ResetPb", "", "", True, NULL }
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = skip_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = skip_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = skip_ref;

  reset_button_def[ 0 ].title = msgGetText( MXDI_SKIP_RESET );


  /* Create a form dialog with buttons. */
  skipFd = xitCreateFormDialog( parent, "SkipFd",
                                1, 0,
                                action_buttons, 
                                XtNumber( action_buttons ) );

  XtAddCallback( skipFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) skip_ref );

  /* We want to know when this window is closed. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( skipFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( skipFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( skipFd ), wm_delete_window, 
                             (XtCallbackProc) cancelCB, (XtPointer) skip_ref );
  } /* block */

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( skipFd, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_SKIP_TITLE ) ); n++;
  XtSetValues( XtParent( skipFd ), args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( skipFd, "SkipFdFo" );


  /* Create the instruction label. */
  skipLa = xitCreateLabel( workFo, "SkipLa", 
                           msgGetText( MXDI_SKIP_INSTR_LABEL ), -1 );

  /* Scrolled window which conatins the weeks of the year. */
  n = 0;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmAS_NEEDED ); n++;
  XtSetArg( args[ n ], XmNscrollingPolicy,        XmAUTOMATIC ); n++;
  skipSw = XmCreateScrolledWindow( workFo, "SkipSw", args, n );


  /* The week to skip during the year. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns,  14 );  n++;
  skipWeekRc = XmCreateRowColumn( skipSw, "SkipWeekRc", args, n );

  /* Add the week numbers (1-53). */
  for( index = 0; index < 53; index++ ) {
    sprintf( buffer, "SkipWeek%dTb", index + 1 );
    sprintf( label,  "%2d", index + 1 );

    skipWeekTb[ index ] = xitCreateToggleButton( skipWeekRc, buffer,
                                                 label, False );

    n = 0;
    XtSetArg( args[ n ], XmNuserData, index + 1 ); n++;
    XtSetValues( skipWeekTb[ index ], args, n );

    XtAddCallback( skipWeekTb[ index ], XmNvalueChangedCallback,
                   (XtCallbackProc) toggleWeekCB, (XtPointer) skip_ref );
  } /* loop */


  /* Reset all 'skip weeks'. */
  resetPb = xitCreatePushButton( workFo, &reset_button_def[ 0 ] );

  XtAddCallback( resetPb, XmNactivateCallback, 
                 (XtCallbackProc) resetCB, (XtPointer) skip_ref );


  /* Skip this week? */
  skipThisWeekTb = xitCreateToggleButton( workFo, "SkipThisWeekTb",
                                          " ", False );

  XtAddCallback( skipThisWeekTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) skipThisWeekCB, (XtPointer) skip_ref );


  /* Put the elements together. */
  xitAttachWidget( skipLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( skipSw,
                   XmATTACH_WIDGET, skipLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( resetPb,
                   XmATTACH_WIDGET, skipSw, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( skipThisWeekTb,
                   XmATTACH_WIDGET, resetPb, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );

  /* Make sure there are some distance between the widget groups. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( skipLa,         args, n );
  XtSetValues( skipSw,         args, n );
  XtSetValues( resetPb,        args, n );
  XtSetValues( skipThisWeekTb, args, n );


  /* Manage the widgets. */
  XtManageChildren( skipWeekTb,  XtNumber( skipWeekTb ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( skipFd, True );


  /* Make the final attachments. */
  xitAttachWidget( skipThisWeekTb,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( resetPb,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, skipThisWeekTb );
  xitAttachWidget( skipSw,
                   XmATTACH_WIDGET, skipLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_WIDGET, resetPb );


  return( skipFd );

} /* createSkipWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchSkipValues( SKIP_REC_REF          skip_ref,
                   XTM_EK_GET_SKIP_DATA  *values_ref )
{

  /* Variables. */
  int     index;
  UINT32  *flags;
  UINT32  skip_flag;
  char    buffer[ 100 ];
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  /* We need a waindow. */
  if( skip_ref -> skipW == NULL )
    return( False );


  mainW = XtNameToWidget( skip_ref -> skipW, "SkipFdFo" );


  /* Get the weeks we skip this entry. */
  baseW = XtNameToWidget( mainW, 
                          "SkipSw.ScrolledWindowClipWindow.SkipWeekRc" );

  for( index = 1; index <= 53; index++ ) {

    skip_flag = (1 << (index % 30));
    if( index  > 30 )
      flags = &values_ref -> skip_week[ 0 ];
    else
      flags = &values_ref -> skip_week[ 1 ];

    sprintf( buffer, "SkipWeek%dTb", index );
    tempW = XtNameToWidget( baseW, buffer );

    if( XmToggleButtonGetState( tempW ) )  
      flagSet( *flags, skip_flag );
    else
      flagClear( *flags, skip_flag );

  } /* loop */


  return( True );

} /* fetchSkipValues */


/*----------------------------------------------------------------------*/

static void
  setSkipValues( SKIP_REC_REF          skip_ref,
                 XTM_EK_SET_SKIP_DATA  *values_ref )
{

  /* Variables. */
  int     index;
  int     week_no;
  UINT32  flags;
  UINT32  skip_flag;
  char    buffer[ 80 ];
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  /* We need a waindow. */
  if( skip_ref -> skipW == NULL )
    return;


  mainW = XtNameToWidget( skip_ref -> skipW, "SkipFdFo" );

  skip_ref -> default_date = values_ref -> default_date;


  /* Set the weeks we skip this entry. */
  baseW = XtNameToWidget( mainW, 
                          "SkipSw.ScrolledWindowClipWindow.SkipWeekRc" );

  for( index = 1; index <= 53; index++ ) {

    skip_flag = (1 << (index % 30));
    if( index > 30 )
      flags = values_ref -> skip_week[ 0 ];
    else
      flags = values_ref -> skip_week[ 1 ];

    sprintf( buffer, "SkipWeek%dTb", index );
    tempW = XtNameToWidget( baseW, buffer );

    if( flagIsSet( flags, skip_flag ) )
      XmToggleButtonSetState( tempW, True, False );
    else
      XmToggleButtonSetState( tempW, False, False );

  } /* lopp */


  /* Do we skip the entry this week? */
  week_no = TimIndexOfIsoWeek( values_ref -> default_date );

  skip_flag = (1 << (week_no % 30));
  if( week_no > 30 )
    flags = values_ref -> skip_week[ 0 ];
  else
    flags = values_ref -> skip_week[ 1 ];

  tempW = XtNameToWidget( mainW, "SkipThisWeekTb" );
  sprintf( buffer, msgGetText( MXDI_SKIP_THIS_WEEK ), week_no );

  xitStringSetLabel( tempW, buffer );

  if( flagIsSet( flags, skip_flag ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* setSkipValues */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            SKIP_REC_REF  skip_ref,
            XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( skip_ref -> actionCB != NULL )
    (* skip_ref -> actionCB)( XTM_EK_REASON_CANCEL,
                              skip_ref -> user_data, NULL );

  XtUnmanageChild( skip_ref -> skipW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             SKIP_REC_REF  skip_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( skip_ref -> actionCB != NULL )
    (* skip_ref -> actionCB)( XTM_EK_REASON_DESTROY,
                              skip_ref -> user_data, NULL );

  SysFree( skip_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget        widget,
          SKIP_REC_REF  skip_ref,
          XtPointer     call_data )
{

  /* Code. */

  xtmHlDisplayHelp( skip_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    skip_window_id, "" );


  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget        widget,
        SKIP_REC_REF  skip_ref,
        XtPointer     call_data )
{

  /* Variables. */
  Boolean               ok;
  XTM_EK_GET_SKIP_DATA  values;


  /* Code. */

  ok = fetchSkipValues( skip_ref, &values );
  if( ! ok )
    return;

  /* Do we have a user action callback registered? */
  if( skip_ref -> actionCB != NULL )
    (* skip_ref -> actionCB)( XTM_EK_REASON_OK,
                              skip_ref -> user_data,
                              &values );

  XtUnmanageChild( skip_ref -> skipW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  resetCB( Widget        widget,
           SKIP_REC_REF  skip_ref,
           XtPointer     call_data )
{

  /* Variables. */
  int     index;
  char    buffer[ 50 ];
  Widget  mainW;
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( skip_ref -> skipW, "SkipFdFo" );
  baseW = XtNameToWidget( mainW, 
                          "SkipSw.ScrolledWindowClipWindow.SkipWeekRc" );

  /* Reset all skip buttons. */
  for( index = 0; index < 53; index++ ) {

    sprintf( buffer, "SkipWeek%dTb", index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

     XmToggleButtonSetState( tempW, False, False );

  }


  /* Reset 'this week' button. */
  tempW = XtNameToWidget( mainW, "SkipThisWeekTb" );

  XmToggleButtonSetState( tempW, False, False );


  return;

} /* resetCB */


/*----------------------------------------------------------------------*/

static void 
  skipThisWeekCB( Widget                        widget,
                  SKIP_REC_REF                  skip_ref,
                  XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  int     week_no;
  char    buffer[ 50 ];
  Widget  mainW;
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( skip_ref -> skipW, "SkipFdFo" );
  baseW = XtNameToWidget( mainW, 
                          "SkipSw.ScrolledWindowClipWindow.SkipWeekRc" );

  week_no = TimIndexOfIsoWeek( skip_ref -> default_date );

  sprintf( buffer, "SkipWeek%dTb", week_no );
  tempW = XtNameToWidget( baseW, buffer );

  if( call_data -> set )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* skipThisWeekCB */


/*----------------------------------------------------------------------*/

static void 
  toggleWeekCB( Widget                        widget,
                SKIP_REC_REF                  skip_ref,
                XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  int       selected_week_no;
  int       week_no;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  if( call_data -> reason != XmCR_ACTIVATE )
    return;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &selected_week_no ); n++;
  XtGetValues( widget, args, n );

  week_no = TimIndexOfIsoWeek( skip_ref -> default_date );

  /* Is this the same week no as the button? */
  if( week_no != selected_week_no )
    return;

  /* Set or reset? */
  mainW = XtNameToWidget( skip_ref -> skipW, "SkipFdFo" );
  tempW = XtNameToWidget( mainW, "SkipThisWeekTb" );

  if( call_data -> set )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* toggleWeekCB */
