/*----------------------------------------------------------------------------
--
--  Module:           xtmSchedTool
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--   Various tools for the schedule.
--
--  Filename:         xtmSchedTool.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-05-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSchedTool.c, Version: 1.1, Date: 95/02/18 15:52:46";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>

#include "Message.h"
#include "System.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"

#include "xtmSchedPriv.h"
#include "xtmSchedTool.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSchedTools";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmStClearEntryTime( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TrackLa" );
  xitStringSetLabel( tempW, "                            " );

  {
    Arg       args[ 10 ];
    Cardinal  n;
    Pixmap    use_pixmap;

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TrackPx" );

    use_pixmap = sched_ref -> tick_pixmap[ 0 ];

    n = 0;
    XtSetArg( args[ n ], XmNlabelPixmap, use_pixmap ); n++;
    XtSetValues( tempW, args, n );

    sched_ref -> tick_pixmap_index = 0;
  }


  return;

} /* xtmStClearEntryTime */


/*----------------------------------------------------------------------*/

ENTRY_INFO
  *xtmStFetchSelectedInfo( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Arg         args[ 10 ];
  Cardinal    n;
  ENTRY_INFO  *entry_info_ref;


  /* Code. */

  if( sched_ref -> selected_widget == NULL )
    return( NULL );

  /* Selected entry. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &entry_info_ref ); n++;
  XtGetValues( sched_ref -> selected_widget, args, n );

  if( entry_info_ref == NULL )
    return( NULL );


  return( entry_info_ref );

} /* xtmStFetchSelectedInfo */


/*----------------------------------------------------------------------*/

Dimension
  xtmStHeightEntry( SCHED_REC_REF  sched_ref,
                    int            duration )
{

  /* Variables. */
  float      pixel_per_min;
  Dimension  dimension;


  /* Code. */

  pixel_per_min = (2.0 * sched_ref -> time_line_height) / 
                   sched_ref -> entry_delta;

  dimension = (Dimension) ( duration * pixel_per_min + 0.1 );
  if( dimension < 5 )
    dimension = 5;


  return( dimension );

} /* xtmStHeightEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmStPickDate( SCHED_REC_REF     sched_ref,
                 XTM_ST_PICK_WHAT  pick_what,
                 TIM_TIME_REF      *picked_date )
{

  /* Variables. */
  int           dummy;
  char          *msg_ref;
  Cursor        cursor;
  Widget        eventW;
  Widget        mainW;
  Widget        tempW;
  Widget        wrapW = NULL;
  Window        confine_to_win = None;
  XEvent        event;
  XtAppContext  context;


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  msg_ref = msgGetText( MXDI_PICK_A_DAY_LABEL );

  eventW = XtNameToWidget( mainW,  "SchedPa.Pane2Fo.EntrySw" );
  eventW = XtNameToWidget( eventW, "ScrolledWindowClipWindow.EntryBb" );

  switch( pick_what ) {
    case XTM_ST_PICK_NOTE:
      wrapW = XtNameToWidget( mainW, "SchedPa.Pane1Fo.NoteSw" );
      wrapW = XtNameToWidget( wrapW, "ScrolledWindowClipWindow.NoteBb" );
      break;

    case XTM_ST_PICK_APP:
    default:
      wrapW = eventW;
      break;
  };

  context = XtWidgetToApplicationContext( eventW );


  /* Tell the user what we are up to. */
  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TrackLa" );
  xitStringSetLabel( tempW, msg_ref );


  /* We should have a nice cursor. */
  cursor = XCreateFontCursor( XtDisplay( eventW ), XC_draped_box );

  /* Grab the pointer to get all events we want. */
  if( ! sched_ref -> appl_data_ref -> custom_data -> wrap_bug )
    confine_to_win = XtWindow( wrapW );

  XGrabPointer( XtDisplay( eventW ), XtWindow( eventW ),
                False,
                (ButtonPressMask | ButtonMotionMask | ButtonReleaseMask),
                GrabModeAsync, GrabModeAsync,
                confine_to_win, cursor, CurrentTime );


  /* This is our local event handler. */
  do {

    XtAppNextEvent( context, &event );

    /* Exit on button press. */
    if( event.xany.type == ButtonRelease )
      break;

    /* Do not report any button events. */
    if( event.xany.type != ButtonPress )
      XtDispatchEvent( &event );

  } while( True );


  /* Clean up. */
  XUngrabPointer( XtDisplay( eventW ), CurrentTime );
  xtmStClearEntryTime( sched_ref );


  /* Button 1 is the select button, anything else is abort. */
  if( event.xbutton.button != 1 )
    return( False );


  /* Convert the position to a date. */
  xtmStPosXToDate( sched_ref, (int) event.xbutton.x, picked_date, &dummy );


  return( True );

} /* xtmStPickDate */


/*----------------------------------------------------------------------*/

Position
  xtmStPositionEntryX( SCHED_REC_REF  sched_ref,
                       int            day_index )
{

  /* Variables. */
  Position  position;


  /* Code. */

  position = (Position) (day_index * sched_ref -> day_width + 6);


  return( position );

} /* xtmStPositionEntryX */


/*----------------------------------------------------------------------*/

Position
  xtmStPositionEntryY( SCHED_REC_REF  sched_ref,
                       TIM_TIME_REF   entry_start )
{

  /* Variables. */
  float           pixel_per_min;
  Position        position;
  TIM_DELTA_TYPE  delta;


  /* Code. */

  pixel_per_min = (2.0 * sched_ref -> time_line_height) / 
                   sched_ref -> entry_delta;

  (void) TimDelta( entry_start, sched_ref -> entry_start, &delta );

  position = (Position) (delta.minutes * pixel_per_min +
                         sched_ref -> time_line_height / 2) + 4;


  return( position );

} /* xtmStPositionEntryY */


/*----------------------------------------------------------------------*/

Position
  xtmStPositionNoteX( SCHED_REC_REF  sched_ref,
                      int            day_index )
{

  /* Variables. */
  Position  position;


  /* Code. */

  position = (Position) (day_index * sched_ref -> day_width + 6);


  return( position );

} /* xtmStPositionNoteX */


/*----------------------------------------------------------------------*/

void
  xtmStPosXToDate( SCHED_REC_REF  sched_ref,
                   int            pos_x,
                   TIM_TIME_REF   *new_start_date,
                   int            *new_pos_x )
{

  /* Variables. */
  int  day_number;


  /* Code. */

  *new_start_date = sched_ref -> schedule_start;
  day_number      = pos_x / (int) sched_ref -> day_width;

  TimAddDays( new_start_date, day_number );
  *new_pos_x = day_number * sched_ref -> day_width + 6;


  return;

} /* xtmStPosXToDate */


/*----------------------------------------------------------------------*/

void
  xtmStPosYToDuration( SCHED_REC_REF  sched_ref,
                       int            pos_y_start,
                       int            pos_y_end,
                       float          grid_minutes,
                       int            *minutes,
                       int            *new_pos_y_end )
{

  /* Variables. */
  int    delta_y;
  float  pixel_per_min;


  /* Code. */

  pixel_per_min = (2.0 * sched_ref -> time_line_height) / 
                   sched_ref -> entry_delta;

  delta_y = abs( pos_y_end - pos_y_start + 1 );

  *minutes       = (int) (delta_y / pixel_per_min);
  *minutes       = (int) ((int) (*minutes / grid_minutes) * grid_minutes );
  *new_pos_y_end = (int) (pixel_per_min * *minutes);


  return;

} /* xtmStPosYToDuration */


/*----------------------------------------------------------------------*/

void
  xtmStPosYToTime( SCHED_REC_REF  sched_ref,
                   int            pos_y,
                   float          grid_minutes,
                   TIM_TIME_REF   *new_start_time,
                   int            *new_pos_y )
{

  /* Variables. */
  int           minutes;
  int           offset;
  float         pixel_per_min;


  /* Code. */

  *new_start_time = sched_ref -> entry_start;
  offset          = sched_ref -> time_line_height / 2 + 4;

  pixel_per_min   = (2.0 * sched_ref -> time_line_height) / 
                     sched_ref -> entry_delta;

  pos_y = pos_y - offset;

  minutes = (int) (pos_y / pixel_per_min);
  minutes = (int) ((int) (minutes / grid_minutes) * grid_minutes );

  TimAddMinutes( new_start_time, minutes );
  *new_pos_y = (int) (pixel_per_min * minutes) + offset;


  return;

} /* xtmStPosYToTime */


/*----------------------------------------------------------------------*/

void
  xtmStPresentEntryTime( SCHED_REC_REF         sched_ref,
                         Boolean               only_start_time,
                         Boolean               display_pixmap,
                         XTM_ST_ANI_DIRECTION  ani_direction,
                         TIM_TIME_REF          start_time,
                         int                   duration )
{

  /* Variables. */
  char                    buffer[ 50 ];
  char                    time_buffer[ 50 ];
  Widget                  mainW;
  Widget                  tempW;
  TIM_TIME_REF            temp_time;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* We always display the start time. */
  xtmFoFormatTime( start_time, buffer, sizeof( buffer ) );

  strcpy( time_buffer, " " );
  strcat( time_buffer, buffer );


  /* Include the end time? */
  if( ! only_start_time && duration > 0 ) {

    temp_time = start_time;
    TimAddMinutes( &temp_time, duration );

    xtmFoFormatTime( temp_time, buffer, sizeof( buffer ) );

    strcat( time_buffer, " - " );
    strcat( time_buffer, buffer );

    sprintf( buffer, "%d:%02d", duration / 60, duration % 60 );

    strcat( time_buffer, " (" );
    strcat( time_buffer, buffer );
    strcat( time_buffer, ")" );

  } /* if */


  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TrackLa" );
  xitStringSetLabel( tempW, time_buffer );

  /* Time pixmap. */
  if( display_pixmap ) {

    Arg       args[ 10 ];
    Cardinal  n;
    Pixmap    use_pixmap;

    if( sched_ref -> tick_pixmap_index < 1 )
      sched_ref -> tick_pixmap_index = 1;

    /* Move in what direction? */
    if( ani_direction == XTM_ST_ANI_FORWARD )
      sched_ref -> tick_pixmap_index++;
    else if( ani_direction == XTM_ST_ANI_BACKWARD )
      sched_ref -> tick_pixmap_index--;

    if( sched_ref -> tick_pixmap_index >= 7 )
      sched_ref -> tick_pixmap_index = 1;
    else if( sched_ref -> tick_pixmap_index < 1 )
      sched_ref -> tick_pixmap_index = 6;

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TrackPx" );

    use_pixmap = sched_ref -> tick_pixmap[ sched_ref -> tick_pixmap_index ];

    n = 0;
    XtSetArg( args[ n ], XmNlabelPixmap, use_pixmap ); n++;
    XtSetValues( tempW, args, n );

  } /* if */


  return;

} /* xtmStPresentEntryTime */


/*----------------------------------------------------------------------*/

void
  xtmStStartOnDay( SCHED_REC_REF  sched_ref,
                   TIM_TIME_REF   *start_time )
{

  /* Code. */

  /* Start the day view on a specific day? */
  if( sched_ref -> appl_data_ref -> custom_data -> start_day_view_on == 0 )
    return;

  while( True ) {
    if( TimIndexOfDayInIsoWeek( *start_time ) ==
        sched_ref -> appl_data_ref -> custom_data -> start_day_view_on )
      break;

    TimAddDays( start_time, -1 );
  }


  return;

} /* xtmStStartOnDay */


/*----------------------------------------------------------------------*/

void
  xtmStUpdateMinuteMarker( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Arg           args[ 10 ];
  Cardinal      n;
  Dimension     height;
  Position      position;
  Widget        mainW;
  Widget        tempW;
  TIM_TIME_REF  now;


  /* Code. */
  now = TimLocalTime( TimMakeTimeNow() );
  now = TimMakeTime( 1970, 1, 1, TimHour( now ), TimMinute( now ), 0 );

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TimeDispSw" );
  tempW = XtNameToWidget( tempW, "ScrolledWindowClipWindow" );
  tempW = XtNameToWidget( tempW, "TimeDispBb.TimeIndicatorAr" );

  if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) &&
      now > sched_ref -> entry_start &&
      now < sched_ref -> entry_stop ) {

    n = 0;
    XtSetArg( args[ n ], XmNheight, &height ); n++;
    XtGetValues( tempW, args, n );

    position = xtmStPositionEntryY( sched_ref, now );
    position = position - (height / 2);

    n = 0;
    XtSetArg( args[ n ], XmNy, position ); n++;
    XtSetArg( args[ n ], XmNx, 0 ); n++;
    XtSetValues( tempW, args, n );

    XtManageChild( tempW );
    XRaiseWindow( XtDisplay( tempW ), XtWindow( tempW ) );

  } else {

    XtUnmanageChild( tempW );

  } /* if */


  return;

} /* xtmStUpdateMinuteMarker */


/*----------------------------------------------------------------------*/

Dimension
  xtmStWidthEntry( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Dimension  dimension;


  /* Code. */

  if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) &&
      flagIsSet(   sched_ref -> flags, XTM_SM_ENTRY_HANDLES ) )
    dimension = (Dimension) (sched_ref -> day_width - 30);
  else
    dimension = (Dimension) (sched_ref -> day_width - 10);

  if( dimension < 5 )
    dimension = 5;


  return( dimension );

} /* xtmStWidthEntry */


/*----------------------------------------------------------------------*/

Dimension
  xtmStWidthNote( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Dimension  dimension;


  /* Code. */

  dimension = (Dimension) (sched_ref -> day_width - 10);

  if( dimension < 5 )
    dimension = 5;


  return( dimension );

} /* xtmStWidthNote */


