/*----------------------------------------------------------------------------
--
--  Module:           xtmSchedWin
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--   Window manipulation routines in the schedule.
--
--  Filename:         xtmSchedWin.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSchedWin.c, Version: 1.1, Date: 95/02/18 15:52:46";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/Scale.h>
#include <Xm/ScrolledW.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "xitRwPixB.h"

#include "LstLinked.h"
#include "Message.h"
#include "System.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbMisc.h"
#include "xtmFormat.h"
#include "xtmSchedAct.h"
#include "xtmSchedTool.h"
#include "xtmTags.h"
#include "xitError.h"
#include "xitTools.h"
#include "XmUbTimeB.h"

#include "xtmSchedPriv.h"
#include "xtmSchedWin.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Max number of windows in the stack table. */
#define MAX_WIN_STACK_REC      30
#define MAX_WIN_STACK_COLUMNS  4

/* Max number of notes and appointment widgets. */
#define MAX_APP_WIDGETS        400
#define MAX_NOTE_WIDGETS       400


/* Local widgets in the 'Display days in day view' window. */
#define scaleValLa        dataLocalW[ 0 ]
#define scaleValSc        dataLocalW[ 1 ]



/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Table to keep track of window stacking. */
typedef struct {
  int  start;
  int  end;
  int  column;
} WIN_STACK_REC;

typedef struct {
  int            entries;
  WIN_STACK_REC  entry[ MAX_WIN_STACK_REC ];
} WIN_STACK_TABLE, *WIN_STACK_TABLE_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSchdWin";

/* Trace double clicks on entries. */
static Time  day_time_last   = 0;
static Time  entry_time_last = 0;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  dayWidthApplyCB( Widget         widget, 
                   SCHED_REC_REF  sched_ref,
                   XtPointer      call_data );

static void 
  dayWidthOkCB( Widget         widget, 
                SCHED_REC_REF  sched_ref,
                XtPointer      call_data );

static void 
  dispDaysApplyCB( Widget         widget, 
                   SCHED_REC_REF  sched_ref,
                   XtPointer      call_data );

static void 
  dispDaysOkCB( Widget         widget, 
                SCHED_REC_REF  sched_ref,
                XtPointer      call_data );

static void 
  doDayWidthCB( Widget         widget, 
                SCHED_REC_REF  sched_ref );

static void 
  doDispDaysCB( Widget         widget, 
                SCHED_REC_REF  sched_ref );

static void
  findEntriesRange( LST_DESC_TYPE  entries,
                    TIM_TIME_REF   default_start,
                    TIM_TIME_REF   default_stop,
                    TIM_TIME_REF   *entry_start,
                    TIM_TIME_REF   *entry_stop );

static int
  findMarkColumn( WIN_STACK_TABLE_REF  stack_table,
                  int                  fit_start,
                  int                  fit_end );

static Boolean
  isCacheEntry( XTM_DB_ALL_ENTRY_REF  entry_ref,
                ENTRY_INFO_REF        cache_entry_ref );

static void 
  scrollWindow( SCHED_REC_REF  sched_ref,
                Widget         scrollBar,
                int            value );

static void
  selectEntryMarkCB( Widget         widget,
                     SCHED_REC_REF  sched_ref,
                     XtPointer      call_data );

static void
  updateSchedDayNames( SCHED_REC_REF  sched_ref,
                       TIM_TIME_REF   start_date,
                       TIM_TIME_REF   stop_date );

static void
  updateSchedEntries( SCHED_REC_REF  sched_ref,
                      TIM_TIME_REF   schedule_start,
                      TIM_TIME_REF   schedule_stop,
                      TIM_TIME_REF   entry_start,
                      TIM_TIME_REF   entry_stop,
                      LST_DESC_TYPE  entries );

static void
  updateSchedEntriesBg( SCHED_REC_REF  sched_ref,
                        TIM_TIME_REF   schedule_start,
                        TIM_TIME_REF   schedule_stop,
                        Dimension      note_height,
                        Dimension      entry_height );

static void
  updateSchedTimes( SCHED_REC_REF  sched_ref,
                    TIM_TIME_REF   start_time,
                    TIM_TIME_REF   stop_time );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmSwSelectScaleValue( SCHED_REC_REF     sched_ref,
                         SCHED_SCALE_TYPE  scale_type )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 2 ];
  Widget    scaleValFd;
  Widget    workFo;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   NULL, NULL },
    { NULL, NULL, NULL },
    { "",   NULL, NULL },
    { NULL, NULL, NULL },
    { "",   NULL, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = sched_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 2 ].data  = sched_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 4 ].data  = sched_ref;

  if( scale_type == SCHED_DISP_DAYS ) {
    action_buttons[ 0 ].callback = dispDaysOkCB;
    action_buttons[ 2 ].callback = dispDaysApplyCB;
  } else if( scale_type == SCHED_DAY_WIDTH ) {
    action_buttons[ 0 ].callback = dayWidthOkCB;
    action_buttons[ 2 ].callback = dayWidthApplyCB;
  }


  /* Create a form dialog with buttons. */
  scaleValFd = xitCreateFormDialog( sched_ref -> scheduleW, "ScaleValFd",
                                    1, 0,
                                    action_buttons,
                                    XtNumber( action_buttons ) );

  n = 0;
  if( scale_type == SCHED_DISP_DAYS ) {
    XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_DISP_DAYS_TITLE ) ); n++;
    XtSetValues( XtParent( scaleValFd ), args, n );

  } else if( scale_type == SCHED_DAY_WIDTH ) {
    XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_DAY_WIDTH_TITLE ) ); n++;
    XtSetValues( XtParent( scaleValFd ), args, n );
  }

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( scaleValFd, args, n );


  /* Container for the contents of the snooze window. */
  workFo = XtNameToWidget( scaleValFd, "ScaleValFdFo" );


  if( scale_type == SCHED_DISP_DAYS )
    scaleValLa = xitCreateLabel( workFo, "ScaleValLa", 
                                 msgGetText( MXDI_DISP_DAYS_LABEL ), -1 );

  else if( scale_type == SCHED_DAY_WIDTH )
    scaleValLa = xitCreateLabel( workFo, "ScaleValLa", 
                                 msgGetText( MXDI_DAY_WIDTH_LABEL ), -1 );

  /* Create a slide bar with the time in minutes to wait. */
  n = 0;

  if( scale_type == SCHED_DISP_DAYS ) {
    XtSetArg( args[ n ], XmNvalue, 1 ); n++;
    XtSetArg( args[ n ], XmNminimum, 1 ); n++;
    XtSetArg( args[ n ], XmNmaximum, 31 ); n++;
  } else if( scale_type == SCHED_DAY_WIDTH ) {
    XtSetArg( args[ n ], XmNvalue, 130 ); n++;
    XtSetArg( args[ n ], XmNminimum, 40 ); n++;
    XtSetArg( args[ n ], XmNmaximum, 500 ); n++;
  }
  XtSetArg( args[ n ], XmNprocessingDirection, XmMAX_ON_RIGHT ); n++;
  XtSetArg( args[ n ], XmNshowValue, True ); n++;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  scaleValSc = XmCreateScale( workFo, "ScaleValSc", args, n );


  /* Put the elements together. */
  xitAttachWidget( scaleValLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( scaleValSc,
                   XmATTACH_WIDGET, scaleValLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,       XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( scaleValLa, args, n );
  XtSetValues( scaleValSc, args, n );


  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( scaleValFd, True );


  /* Default value minutes. */
  if( scale_type == SCHED_DISP_DAYS )
    XmScaleSetValue( scaleValSc, sched_ref -> show_no_days );

  else if( scale_type == SCHED_DAY_WIDTH )
    XmScaleSetValue( scaleValSc, (int) sched_ref -> day_width );


  XtManageChild( scaleValFd );


  return;

} /* xtmSwSelectScaleValue */


/*----------------------------------------------------------------------*/

void 
  xtmSwEntryScrolledHorizCB( Widget                     widget,
                             SCHED_REC_REF              sched_ref,
                             XmScrollBarCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    scrollBar;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Scroll the day names window. */
  tempW = XtNameToWidget( mainW, "DayDispSw" );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &scrollBar ); n++;
  XtGetValues( tempW, args, n );

  scrollWindow( sched_ref, scrollBar, call_data -> value );


  /* Scroll the notes window. */
  tempW = XtNameToWidget( mainW, "SchedPa.Pane1Fo.NoteSw" );

  n = 0;
  XtSetArg( args[ n ], XmNhorizontalScrollBar, &scrollBar ); n++;
  XtGetValues( tempW, args, n );

  scrollWindow( sched_ref, scrollBar, call_data -> value );


  return;

} /* xtmSwEntryScrolledHorizCB */


/*----------------------------------------------------------------------*/

void 
  xtmSwEntryScrolledVertCB( Widget                     widget,
                            SCHED_REC_REF              sched_ref,
                            XmScrollBarCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    scrollBar;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Scroll the times window. */
  tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TimeDispSw" );

  n = 0;
  XtSetArg( args[ n ], XmNverticalScrollBar, &scrollBar ); n++;
  XtGetValues( tempW, args, n );

  scrollWindow( sched_ref, scrollBar, call_data -> value );


  return;

} /* xtmSwEntryScrolledVertCB */


/*----------------------------------------------------------------------*/

void
  xtmSwSelectEntryCB( Widget               widget,
                      SCHED_REC_REF        sched_ref,
                      XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean                 double_click = False;
  Arg                     args[ 5 ];
  Cardinal                n;
  Time                    time_now;
  Widget                  mainW;
  ENTRY_INFO              *entry_info_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  if( call_data -> event -> xany.type != ButtonRelease )
    return;


  /* Re-make window stacking. */
  XRaiseWindow( XtDisplay( widget ), XtWindow( widget ) );


  /* Double click? */
  time_now = call_data -> event -> xbutton.time;

  if( abs( time_now - entry_time_last ) < 500 && entry_time_last != 0 )
    double_click = True;

  entry_time_last = time_now;


  /* Double click on the same entry -> view it. */
  if( double_click ) {
    sched_ref -> selected_widget = widget;
    entry_info_ref = xtmStFetchSelectedInfo( sched_ref );

    xtmSaAppointmentEditorShow( sched_ref );

    return;
  }


  /* If selected same as previous, toggle the button. */
  if( widget == sched_ref -> selected_widget ) {
    xtmSwUnselectEntry( sched_ref );

    return;
  }


  /* Unselect any previous entry? */
  if( sched_ref -> selected_widget != NULL )
    xtmSwUnselectEntry( sched_ref );


  /* Fetch current background. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, &sched_ref -> entry_saved_bg ); n++;
  XtGetValues( widget, args, n );


  /* Selected background. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, 
            custom_data_ref -> entry_select_bg ); n++;
  XtSetValues( widget, args, n );

  sched_ref -> selected_widget = widget;


  /* Display time and duration for the entry? */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );

  if( entry_info_ref -> type == XTM_DB_DAY_ENTRY )
    xtmStPresentEntryTime( sched_ref, False, True, XTM_ST_ANI_STILL,
                           entry_info_ref -> time_stamp,
                           entry_info_ref -> duration );


  return;

} /* xtmSwSelectEntryCB */


/*----------------------------------------------------------------------*/

void
  xtmSwSetSchedule( SCHED_REC_REF  sched_ref,
                    Widget         widget )
{

  /* Variables. */
  UINT32                  flags;
  char                    buffer[ 100 ];
  char                    *char_ref;
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  LST_DESC_TYPE           entries;
  TIM_TIME_REF            default_start;
  TIM_TIME_REF            default_stop;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  if( ! custom_data_ref -> widget_caching )
    sched_ref -> force_update = True;

  /* Current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name,
                            &db_info, NULL );

  /* Get the entries within the date range, (with filtering). */
  flags = XTM_DB_FETCH_ALL_TEXT;

  db_status = xtmDmFetchFilteredEntriesRange( 
                appl_data_ref,
                sched_ref      -> db_name,
                appl_data_ref  -> context, 
                sched_ref      -> scheduleW,
                sched_ref      -> schedule_start,
                sched_ref      -> schedule_stop,
                flags,
                &sched_ref -> filter_rec,
                &entries );

  if( db_status != XTM_DB_OK ) {
    xtmDmDeleteEntriesList( entries );

    return;
  }

  /* Find the limits for the entries (earliest and latest). */
  default_start = TimMakeTime( 1970, 1, 1, custom_data_ref -> start_hour,
                               0, 0 );
  default_stop = TimMakeTime( 1970, 1, 1, custom_data_ref -> stop_hour,
                              0, 0 );

  findEntriesRange( entries, 
                    default_start, default_stop,
                    &sched_ref -> entry_start, &sched_ref -> entry_stop );


  /* Clear any animate traces. */
  xtmStClearEntryTime( sched_ref );


  /* Update the day names. */
  updateSchedDayNames( sched_ref,
                       sched_ref -> schedule_start,
                       sched_ref -> schedule_stop );

  /* Update the entry time scale. */
  updateSchedTimes( sched_ref,
                    sched_ref -> entry_start, sched_ref -> entry_stop );

  /* Update the entries and the notes. */
  updateSchedEntries( sched_ref,
                      sched_ref -> schedule_start, sched_ref -> schedule_stop,
                      sched_ref -> entry_start,    sched_ref -> entry_stop,
                      entries );

  xtmDmDeleteEntriesList( entries );


  /* Window and icon title. */
  char_ref = msgGetText( MXDI_SCHED_TITLE );

  sprintf( buffer, "%s %s", char_ref, db_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( sched_ref -> scheduleW, args, n );

  sprintf( buffer, "%s", db_info.short_name );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, buffer ); n++;
  XtSetValues( sched_ref -> scheduleW, args, n );


  /* Show what is valid for list/button mode. */
  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) ) {
    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TimeDispSw" );
    XtUnmapWidget( tempW );

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.ZoomFs" );
    XtUnmapWidget( tempW );

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.MakeAppPb" );
    XtMapWidget( tempW );

  } else {
    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.TimeDispSw" );
    XtMapWidget( tempW );

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.ZoomFs" );
    XtMapWidget( tempW );

    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.MakeAppPb" );
    XtUnmapWidget( tempW );

  } /* if */


  /* Read only? */
  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) ) {
    tempW = XtNameToWidget( mainW, "SchedPa.Pane2Fo.MakeAppPb" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "SchedPa.Pane1Fo.MakeNotePb" );
    XtSetSensitive( tempW, False );
  }

  sched_ref -> force_update = False;


  return;

} /* xtmSwSetSchedule */


/*----------------------------------------------------------------------*/

void
  xtmSwSetScheduleDates( SCHED_REC_REF  sched_ref,
                         Widget         widget,
                         Boolean        fetch )
{

  /* Variables. */
  Boolean               ok;
  char                  buffer[ 50 ];
  char                  *char_ref;
  Widget                mainW;
  Widget                tempW;
  TIM_TIME_REF          start_date;
  TIM_TIME_REF          stop_date;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = sched_ref -> appl_data_ref;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Fetch and check start date. */
  if( fetch ) {
    tempW = XtNameToWidget( mainW, "DateRc.FromTb" );

    ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &start_date );
    if( ! ok )
      return;

    sched_ref -> schedule_start = start_date;

    /* Start the day view on a specific day? */
    xtmStStartOnDay( sched_ref, &sched_ref -> schedule_start );
  }

  tempW = XtNameToWidget( mainW, "DateRc.FromTb" );
  XmUbTimeBoxSetStartDate( tempW, sched_ref -> schedule_start );


  /* Some more information. */
  {
    int       start_week;
    int       stop_week;
    char      info_buffer[ 100 ];

    info_buffer[ 0 ] = '\0';
    strcat( info_buffer, " " );

    /* Stop date. */
    stop_date = sched_ref -> schedule_start;
    TimAddDays( &stop_date, sched_ref -> show_no_days - 1 );

    sched_ref -> schedule_stop = stop_date;

    xtmFoFormatDate( stop_date, buffer, sizeof( buffer ) );

    strcat( info_buffer, buffer );
    strcat( info_buffer, "  " );
    strcat( info_buffer, msgGetText( MXDI_WEEK_LABEL ) );

    start_week = TimIndexOfWeek( sched_ref -> schedule_start );
    stop_week  = TimIndexOfWeek( sched_ref -> schedule_stop );

    sprintf( buffer, " %d", start_week );
    strcat( info_buffer, buffer );

    if( start_week != stop_week ) {
      sprintf( buffer, "-%d", stop_week );
      strcat( info_buffer, buffer );
    }

    tempW = XtNameToWidget( mainW, "DateRc.DateInfoLa" );

    xitStringSetLabel( tempW, info_buffer );
  }


  return;

} /* xtmSwSetScheduleDates */


/*----------------------------------------------------------------------*/

void
  xtmSwUnselectEntry( SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Pixel     background;
  Widget    mainW;


  /* Code. */

  if( sched_ref -> selected_widget == NULL )
    return;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  background = sched_ref -> entry_saved_bg;

  n = 0;
  XtSetArg( args[ n ], XmNbackground, background ); n++;
  XtSetValues( sched_ref -> selected_widget, args, n );

  sched_ref -> selected_widget = NULL;


  /* Clear the appointment date/time. */    
  xtmStClearEntryTime( sched_ref );


  return;

} /* xtmSwUnselectEntry */


/*----------------------------------------------------------------------*/

static void
  findEntriesRange( LST_DESC_TYPE  entries,
                    TIM_TIME_REF   default_start,
                    TIM_TIME_REF   default_stop,
                    TIM_TIME_REF   *entry_start,
                    TIM_TIME_REF   *entry_stop )
{

  /* Variables. */
  LST_STATUS            lst_status;
  TIM_TIME_REF          max_time;
  TIM_TIME_REF          min_time;
  TIM_TIME_REF          tmp_time;
  XTM_DB_ALL_ENTRY_DEF  entry_record;


  /* Code. */

  *entry_start = default_start;
  *entry_stop  = default_stop;

  min_time = 0;
  max_time = TimMakeTime( 1970, 1, 1, 24, 0, 0 );


  /* Loop through all entries and look for max and min times. */
  lst_status = LstLinkCurrentFirst( entries );

  while( lst_status == LST_OK ) {

    lst_status = LstLinkGetCurrent( entries, &entry_record );

    /* Skip this entry? */
    if( entry_record.entry.entry_type != XTM_DB_DAY_ENTRY ||
        flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW ) ||
        entry_record.entry.day_list_lines <= 0 ) {
      lst_status = LstLinkCurrentNext( entries );
      continue;
    }

    /* Start time? */
    if( entry_record.entry.time_stamp < *entry_start )
      *entry_start = entry_record.entry.time_stamp;

    /* Stop time? */
    tmp_time = entry_record.entry.time_stamp;
    TimAddMinutes( &tmp_time, entry_record.entry.duration );

    if( tmp_time > max_time )
      tmp_time = max_time;

    if( tmp_time > *entry_stop )
      *entry_stop = tmp_time;

    /* Next entry. */
    lst_status = LstLinkCurrentNext( entries );

  } /* while */


  /* Round start time down to closest hour. */
  if( *entry_start > min_time )
    *entry_start = TimMakeTime( 1970, 1, 1, TimHour( *entry_start ), 0, 0 );

  /* Round stop time up to closest hour. */
  if( *entry_stop < max_time )
    *entry_stop = TimMakeTime( 1970, 1, 1, TimHour( *entry_stop ) + 1, 0, 0 );


  return;

} /* findEntriesRange */


/*----------------------------------------------------------------------*/

static Boolean
  isCacheEntry( XTM_DB_ALL_ENTRY_REF  entry_ref,
                ENTRY_INFO_REF        cache_entry_ref )
{

  /* Code. */

  if( entry_ref -> entry.id != cache_entry_ref -> id )
    return( False );

  if( strcmp( entry_ref -> db_name, cache_entry_ref -> db_name ) == 0 )
    return( True );


  return( False );

} /* isCacheEntry */


/*----------------------------------------------------------------------*/

static void 
  scrollWindow( SCHED_REC_REF  sched_ref,
                Widget         scrollBar,
                int            value )
{

  /* Variables. */
  int       sb_increment;
  int       sb_maximum;
  int       sb_minimum;
  int       sb_page_increment;
  int       sb_slider_size;
  int       sb_value;
  Arg       args[ 10 ];
  Cardinal  n;


  /* Code. */

  /* Information about scroll bar. */
  n = 0;
  XtSetArg( args[ n ], XmNminimum, &sb_minimum ); n++;
  XtSetArg( args[ n ], XmNmaximum, &sb_maximum ); n++;
  XtSetArg( args[ n ], XmNsliderSize, &sb_slider_size ); n++;
  XtSetArg( args[ n ], XmNincrement, &sb_increment ); n++;
  XtSetArg( args[ n ], XmNpageIncrement, &sb_page_increment ); n++;
  XtSetArg( args[ n ], XmNvalue, &sb_value ); n++;
  XtGetValues( scrollBar, args, n );

  if( value < sb_minimum )
    value = sb_minimum;
  if( value > (sb_maximum - sb_slider_size) )
    value = (sb_maximum - sb_slider_size);

  /* Set position for scroll bar. */
  XmScrollBarSetValues( scrollBar, value, sb_slider_size,
                        sb_increment, sb_page_increment, True );

  /* The scrolled window does not react on valueChanged callback. */
  {
    XmScrollBarCallbackStruct call_data;

    /* Create the call data, I hope it is OK to skip event and pixel. */
    call_data.value  = value;
    call_data.reason = XmCR_DRAG;

    /* The drag callback makes the scrolled window to react. */
    XtCallCallbacks( scrollBar, XmNdragCallback, (XtPointer) &call_data );

  } /* block */

  
  return;

} /* scrollWindow */


/*----------------------------------------------------------------------*/

static void
  updateSchedDayNames( SCHED_REC_REF  sched_ref,
                       TIM_TIME_REF   start_date,
                       TIM_TIME_REF   stop_date )
{

  /* Variables. */
  Boolean                 update_labels = True;
  int                     day_index;
  int                     year = 0;
  char                    buffer[ 50 ];
  char                    buffer1[ 50 ];
  char                    day_buffer[ 50 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Dimension               days_height = 0;
  Dimension               day_width;
  Dimension               width;
  Widget                  dayDispBbW;
  Widget                  dayNameLaW;
  Widget                  mainW;
  TIM_TIME_REF            curr_date;
  TIM_TIME_REF            now;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_PUSH_STRUCT button_data[] = {
    "", " ", "", True, NULL
  };


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  /* Change in dates? */
  if( sched_ref -> schedule_start == sched_ref -> old_schedule_start &&
      sched_ref -> schedule_stop  == sched_ref -> old_schedule_stop &&
      ! sched_ref -> force_update )
    update_labels = False;

  sched_ref -> old_schedule_start = sched_ref -> schedule_start;
  sched_ref -> old_schedule_stop  = sched_ref -> schedule_stop;


  /* Widgets we will need. */
  dayDispBbW = XtNameToWidget( 
                 mainW, 
                 "DayDispSw.ScrolledWindowClipWindow.DayDispBb" );

  now = TimLocalTime( TimMakeTimeNow() );


  /* Width of each day. */
  day_width = sched_ref -> day_width;
  if( day_width < 40 )
    day_width = 40;


  /* BB for day names. */
  width = (Dimension) sched_ref -> show_no_days * day_width + 5;

  n = 0;
  XtSetArg( args[ n ], XmNwidth,  width + 5 ); n++;
  XtSetValues( dayDispBbW, args, n );


  /* Loop through all days. */
  curr_date = start_date;
  day_index = 0;


  while( curr_date <= stop_date ) {

    int    day_in_week;
    Pixel  bg_color;
    Pixel  fg_color;

    /* Does the day label exists? */
    sprintf( buffer, "DayName%dLa", day_index + 1 );

    dayNameLaW = XtNameToWidget( dayDispBbW, buffer );

    if( dayNameLaW == NULL )
      dayNameLaW = xitCreateLabelWidget( dayDispBbW, buffer, " ", -1 );

    /* Default colors. */
    n = 0;
    XtSetArg( args[ n ], XmNbackground, &bg_color ); n++;
    XtSetArg( args[ n ], XmNforeground, &fg_color ); n++;
    XtGetValues( dayDispBbW, args, n );

    /* Name of the day. */
    if( update_labels ) {
      TimFormatStrTime( curr_date, "%a, %b ", buffer,  sizeof( buffer ) );
      TimFormatStrTime( curr_date, "%d",      buffer1, sizeof( buffer1 ) );

      if( buffer1[ 0 ] == '0' )
        buffer1[ 0 ] = ' ';
      strcat( buffer, buffer1 );

      sprintf( day_buffer, "%-25.25s", buffer );
      xitStringSetLabel( dayNameLaW, day_buffer );
    }


    /* Size of day. */
    n = 0;
    XtSetArg( args[ n ], XmNheight, &days_height ); n++;
    XtGetValues( dayNameLaW, args, n );

    n = 0;
    XtSetArg( args[ n ], XmNx, (Position) day_width * day_index ); n++;
    XtSetArg( args[ n ], XmNwidth, day_width ); n++;
    XtSetValues( dayNameLaW, args, n );

    /* Color mark the day? */
    day_in_week = TimIndexOfDayInIsoWeek( curr_date );

    if( day_in_week == 1 )
      fg_color = custom_data_ref -> monday_fg;
    else if( day_in_week == 2 )
      fg_color = custom_data_ref -> tuesday_fg;
    else if( day_in_week == 3 )
      fg_color = custom_data_ref -> wednesday_fg;
    else if( day_in_week == 4 )
      fg_color = custom_data_ref -> thursday_fg;
    else if( day_in_week == 5 )
      fg_color = custom_data_ref -> friday_fg;
    else if( day_in_week == 6 )
      fg_color = custom_data_ref -> saturday_fg;
    else if( day_in_week == 7 )
      fg_color = custom_data_ref -> sunday_fg;

    if( xtmHoIsHoliday( curr_date ) )
      fg_color = appl_data_ref -> custom_data -> holiday_fg;

    if( TimIsSameDate( now, curr_date ) == TIM_YES )
      bg_color = appl_data_ref -> custom_data -> today_bg;

    n = 0;
    XtSetArg( args[ n ], XmNforeground, fg_color ); n++;
    XtSetArg( args[ n ], XmNbackground, bg_color ); n++;
    XtSetValues( dayNameLaW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( dayNameLaW ) )
      XtManageChild( dayNameLaW );
    else
      XtMapWidget( dayNameLaW );

    /* Next day. */
    day_index++;    
    TimAddDays( &curr_date, 1 );

  } /* while */


  /* Unmap days not used. */
  do {
    sprintf( buffer, "DayName%dLa", day_index + 1 );

    dayNameLaW = XtNameToWidget( dayDispBbW, buffer );
    if( dayNameLaW == NULL )
      break;

    XtUnmapWidget( dayNameLaW );

    day_index++;
  } while( True );

  n = 0;
  XtSetArg( args[ n ], XmNheight, days_height ); n++;
  XtSetValues( dayDispBbW, args, n );


  return;

} /* updateSchedDayNames */


/*----------------------------------------------------------------------*/

static void
  updateSchedEntries( SCHED_REC_REF  sched_ref,
                      TIM_TIME_REF   schedule_start,
                      TIM_TIME_REF   schedule_stop,
                      TIM_TIME_REF   entry_start,
                      TIM_TIME_REF   entry_stop,
                      LST_DESC_TYPE  entries )
{
  /* Types. */
  typedef struct {
    int     stacking;
    Widget  widget;
  } WIN_STACK_INFO;

  /* Variables. */
  int                     apps_win = 0;
  int                     day_index = 0;
  int                     db_bg_index = 0;
  int                     index;
  char                    buffer[ 100 ];
  char                    entry_buffer[ 100 ];
  char                    *char_ref;
  char                    *entry_text;
  Arg                     args[ 20 ];
  Cardinal                n;
  Dimension               height;
  Dimension               width;
  Display                 *display;
  Position                entry_anchor_x = 3;
  Position                entry_anchor_y = 0;
  Position                notes_anchor_x = 3;
  Position                notes_anchor_y = 0;
  Position                entry_max_y    = 0;
  Position                notes_max_y    = 0;
  Widget                  entryBbW;
  Widget                  entry_w;
  Widget                  mainW;
  Widget                  mark_w;
  Widget                  noteBbW;
  XmString                xstr;
  XmString                xstr_text;
  XtTranslations          entry_leave_trans;
  XtTranslations          note_leave_trans;
  ENTRY_CACHE_REF         cache_ref;
  ENTRY_INFO              *entry_info;
  LST_STATUS              lst_status;
  TIM_TIME_REF            last_date = 0;
  WIN_STACK_INFO          win_stack_info[ MAX_APP_WIDGETS ];
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;

  static XIT_PUSH_STRUCT button_data[] = {
    " ", " ", " ", True, NULL
  };


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;
  cache_ref       = sched_ref -> entry_cache_ref;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  noteBbW = XtNameToWidget( 
              mainW, 
              "SchedPa.Pane1Fo.NoteSw.ScrolledWindowClipWindow.NoteBb" );

  entryBbW = XtNameToWidget( 
               mainW, 
               "SchedPa.Pane2Fo.EntrySw.ScrolledWindowClipWindow.EntryBb" );

  /* Make sure the BB's are big enough (correct size will be set later). */
  n = 0;
  XtSetArg( args[ n ], XmNheight, 10000 ); n++;
  XtSetArg( args[ n ], XmNwidth, 10000 ); n++;
  XtSetValues( noteBbW, args, n );
  XtSetValues( entryBbW, args, n );


  db_info.short_name[ 0 ] = '\0';

  display = XtDisplay( mainW );


  /* Mark the entries in cache we are going to use. */
  lst_status = LstLinkCurrentFirst( entries );

  while( lst_status == LST_OK ) {

    /* Fetch the record. */
    lst_status = LstLinkGetCurrent( entries, &entry_record );

    /* Is this entry already in the cache? */
    for( index = 0; index < cache_ref -> max_used; index++ ) {
      if( cache_ref -> used[ index ] == NULL )
        continue;

      if( cache_ref -> used[ index ] -> cache_state != IS_USED )
        continue;

      if( isCacheEntry( &entry_record, cache_ref -> used[ index ] ) &&
          ! sched_ref -> force_update ) {
        cache_ref -> used[ index ] -> cache_state = IS_CACHED;
        break;
      }
    } /* loop */

    lst_status = LstLinkCurrentNext( entries );

  } /* while */


  /* Hide the entries/widgets we are not going to use. */
  for( index = 0; index < cache_ref -> max_used; index++ ) {
    if( cache_ref -> used[ index ] == NULL )
      continue;

    if( cache_ref -> used[ index ] -> cache_state == IS_USED ) {

      if( cache_ref -> used[ index ] -> noteW != NULL )
        XtUnmapWidget( cache_ref -> used[ index ] -> noteW );

      if( cache_ref -> used[ index ] -> appW != NULL )
        XtUnmapWidget( cache_ref -> used[ index ] -> appW );

      if( cache_ref -> used[ index ] -> appMarkW != NULL )
        XtUnmapWidget( cache_ref -> used[ index ] -> appMarkW );

      cache_ref -> used[ index ] -> cache_state = IS_EMPTY;
    }
  } /* loop */


  /* No entries selected. */
  sched_ref -> selected_widget = NULL;


  /* Translations needed for the animation stuff. */
#ifdef XD_HAS_NO_PRINTF_PTR
  sprintf( buffer, "<LeaveWindow>: Leave() LeaveHook(Note,%d) Disarm()",
           (int) sched_ref );
  note_leave_trans = XtParseTranslationTable( buffer );

  sprintf( buffer, "<LeaveWindow>: Leave() LeaveHook(Entry,%d) Disarm()",
           (int) sched_ref );
  entry_leave_trans = XtParseTranslationTable( buffer );
#else
  sprintf( buffer, "<LeaveWindow>: Leave() LeaveHook(Note,%p) Disarm()",
           (void *) sched_ref );
  note_leave_trans = XtParseTranslationTable( buffer );

  sprintf( buffer, "<LeaveWindow>: Leave() LeaveHook(Entry,%p) Disarm()",
           (void *) sched_ref );
  entry_leave_trans = XtParseTranslationTable( buffer );
#endif


  /* Loop through all entries and create the widgets for the entries. */
  lst_status = LstLinkCurrentFirst( entries );

  while( lst_status == LST_OK ) {

    int              entry_bg_index;
    int              entry_fg_index;
    Boolean          has_tag_color;
    Pixel            background;
    Pixel            foreground;
    Pixel            tag_bg;
    Pixel            tag_fg;
    TIM_DELTA_TYPE   delta;
    WIN_STACK_TABLE  win_stack_table;

    /* Fetch the record. */
    lst_status = LstLinkGetCurrent( entries, &entry_record );


    /* If we change day, create what is missing and set parameters. */
    if( last_date != entry_record.entry.date_stamp ) {

      (void) TimDelta( entry_record.entry.date_stamp, schedule_start,
                       &delta );

      day_index = delta.days;

      /* Horizontal anchor the current day. */
      notes_anchor_x = xtmStPositionNoteX( sched_ref, day_index );
      notes_anchor_y = 3;

      entry_anchor_x = xtmStPositionEntryX( sched_ref, day_index );
      entry_anchor_y = 0;

      /* Start all over again with the window stacking. */
      win_stack_table.entries = 0;

      last_date = entry_record.entry.date_stamp;

    } /* if */


    /* Any lines to display? */
    if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW ) ||
        entry_record.entry.day_list_lines <= 0 ) {
      lst_status = LstLinkCurrentNext( entries );
      continue;
    }


    /* Information about the calendar. */
    if( strcmp( entry_record.db_name, db_info.short_name ) != 0 )
      (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                                entry_record.db_name, 
                                &db_info, NULL );


    /* Color for included calendars? */
    if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) ) {
      db_bg_index = db_info.incl_bg_index;

      if( db_bg_index <  0 ||
          db_bg_index >= custom_data_ref -> db_colors_bg.no_colors )
        db_bg_index = 0;
    } /* if */


    /* Check limits of colors to use. */
    entry_bg_index = (int) entry_record.entry.bg_color_index;
    if( entry_bg_index <  0 ||
        entry_bg_index >= custom_data_ref -> colorsDB_bg.no_colors )
      entry_bg_index = 0;

    entry_fg_index = (int) entry_record.entry.fg_color_index;
    if( entry_fg_index <  0 ||
        entry_fg_index >= custom_data_ref -> colorsDB_fg.no_colors )
      entry_fg_index = 0;


    /* Tags with colors? */
    if( entry_record.entry.tag[ 0 ] != '\0' )
      has_tag_color = xtmTgFetchTagColor( custom_data_ref -> tags_db,
                                          entry_record.entry.tag,
                                          &tag_fg, &tag_bg );
    else
      has_tag_color = False;


    /* Foreground and background colors. */
    if( entry_bg_index > 0 &&
        (flagIsClear( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) ||
         flagIsSet( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC )) )
      background = custom_data_ref -> colorsDB_bg.color[ entry_bg_index ];

    else if( has_tag_color )
      background = tag_bg;

    else if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) &&
             db_bg_index > 0 )
      background = custom_data_ref -> db_colors_bg.color[ db_bg_index ];

    else if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) )
      background = custom_data_ref -> entry_include_bg;

    else
      background = custom_data_ref -> entry_unselect_bg;


    if( entry_fg_index > 0 &&
        (flagIsClear( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) ||
         flagIsSet( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC )) )
      foreground = custom_data_ref -> colorsDB_fg.color[ entry_fg_index ];

    else if( has_tag_color )
      foreground = tag_fg;

    else if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_IMPORTANT ) )
      foreground = custom_data_ref -> entry_important_fg;

    else if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
      foreground = custom_data_ref -> entry_note_fg;

    else
      foreground = custom_data_ref -> entry_normal_fg;


    /* Start to build the entry. */
    entry_buffer[ 0 ] = '\0';

    /* If list layout, display the time. */
    if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) ) {
      if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY ) {
        xtmFoFormatEntryTimes( &entry_record, buffer, sizeof( buffer ) );

        strcat( entry_buffer, buffer );
        strcat( entry_buffer, " " );
      }
    }

    /* Flags for the entry. */
    xtmFoFormatEntryFlags( custom_data_ref, &entry_record,
                           buffer, sizeof( buffer ) );
    if( strlen( buffer ) > 0 && 
        flagIsSet( sched_ref -> flags, XTM_SM_ENTRY_FLAGS ) ) {
      strcat( entry_buffer, buffer );
      strcat( entry_buffer, " " );
    }

    /* Included database? */
    if( flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_INCLUDE ) &&
        flagIsSet( sched_ref -> flags, XTM_SM_ENTRY_FLAGS ) ) {
      sprintf( buffer, "(%s)\n", entry_record.db_name );

      strcat( entry_buffer, buffer );
    }

    /* Fetch text for the entry. */
    if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) &&
        entry_record.entry.entry_type == XTM_DB_DAY_ENTRY )
      entry_text = xtmFoFormatText( entry_record.entry.text, 
                                    2, 
                                    entry_record.entry.day_list_lines,
                                    80 );
    else
      entry_text = xtmFoFormatText( entry_record.entry.text, 
                                    0, 
                                    entry_record.entry.day_list_lines,
                                    80 );


    /* Fetch a reference to the entry record. */
    entry_info = NULL;

    /* Already in cache? */
    for( index = 0; index < cache_ref -> max_used; index++ ) {
      if( cache_ref -> used[ index ] == NULL )
        continue;

      if( cache_ref -> used[ index ] -> cache_state != IS_CACHED )
        continue;

      if( isCacheEntry( &entry_record, cache_ref -> used[ index ] ) ) {
        entry_info = cache_ref -> used[ index ];
        break;
      }
    }

    /* Search an empty slot? */
    if( entry_info == NULL ) {
      for( index = 0; index < cache_ref -> max_used; index++ ) {
        if( cache_ref -> used[ index ] == NULL )
          continue;

        if( cache_ref -> used[ index ] -> cache_state == IS_EMPTY ) {
          entry_info = cache_ref -> used[ index ];

          if( entry_info -> entry_text != NULL )
            SysFree( entry_info -> entry_text );
          entry_info -> entry_text  = NULL;
          entry_info -> cache_state = IS_NEW;

          break;
        }
      }
    } /* if */

    /* A complete new entry. */
    if( entry_info == NULL ) {
      entry_info = SysNew( ENTRY_INFO );
      entry_info -> entry_text  = NULL;
      entry_info -> cache_state = IS_NEW;

      entry_info -> noteW    = NULL;
      entry_info -> appW     = NULL;
      entry_info -> appMarkW = NULL;

      for( index = 0; index < cache_ref -> max_used; index++ ) {
        if( cache_ref -> used[ index ] == NULL ) {
          cache_ref -> used[ index ] = entry_info;
          break;
        }
      }
    } /* if */


    /* Save information about the entry. */
    entry_info -> id         = entry_record.entry.id;
    entry_info -> date_stamp = last_date;
    entry_info -> time_stamp = entry_record.entry.time_stamp;
    entry_info -> duration   = entry_record.entry.duration;
    entry_info -> type       = entry_record.entry.entry_type;
    entry_info -> category   = entry_record.entry.entry_category;
    entry_info -> flags      = entry_record.entry.flags;
    entry_info -> uid        = entry_record.entry.owner;

    entry_info -> db_name[ 0 ] = '\0';
    strcpy( entry_info -> db_name, entry_record.db_name );


    /* Save the text for the entry. */
    if( entry_info -> entry_text == NULL ) {

      if( entry_record.all_text != NULL )
        char_ref = entry_record.all_text;
      else
        char_ref = entry_record.entry.text;

      if( strlen( char_ref ) <= 1000 ) {
        entry_info -> entry_text = SysNewString( char_ref );
      } else {
        entry_info -> entry_text = (char *) SysMalloc( 1000 + 1 );
        strncpy( entry_info -> entry_text, char_ref, 1000 );
        *(entry_info -> entry_text + 1000) = '\0';
      }
    } /* if */


    /* Is this a note? */
    if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE ) {

      if( strlen( entry_buffer ) == 0 && entry_text == NULL ) {
        xstr = XmStringCreate( " ", CS );

      } else if( strlen( entry_buffer ) != 0 && entry_text == NULL ) {
        xstr = XmStringCreateLtoR( entry_buffer, CS );

      } else if( strlen( entry_buffer ) == 0 && entry_text != NULL ) {
        xstr = XmStringCreateLtoR( entry_text, CS );

      } else {
        xstr      = XmStringCreateLtoR( entry_buffer, CS );
        xstr_text = XmStringCreateLtoR( entry_text, CS );
        xstr      = xitStringConcatAndFree( xstr, xstr_text );
      }

      width = xtmStWidthNote( sched_ref );

      /* Widget to use. */
      entry_w = entry_info -> noteW;

      if( entry_w == NULL ) {

        n = 0;
        XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
        XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
        XtSetArg( args[ n ], XmNspacing, 0 ); n++;
        entry_w = XtCreateWidget( "NotePb", xitRwPixButtonWidgetClass,
                                  noteBbW, args, n );

        /* Callbacks and event handlers. */
        XtAddEventHandler( entry_w, ButtonPressMask, False,
                           (XtEventHandler) xtmSaActionNoteHook,
                           (XtPointer) sched_ref );

        if( custom_data_ref -> entries_interactive &&
            flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) )
          XtOverrideTranslations( entry_w, note_leave_trans );

        XtAddCallback( entry_w, XmNactivateCallback, 
                       (XtCallbackProc) xtmSwSelectEntryCB,
                       (XtPointer) sched_ref );

        entry_info -> noteW = entry_w;

      } /* if */

      n = 0;
      XtSetArg( args[ n ], XmNuserData, (void *) entry_info ); n++;
      XtSetArg( args[ n ], XmNx, notes_anchor_x ); n++;
      XtSetArg( args[ n ], XmNy, notes_anchor_y ); n++;
      XtSetArg( args[ n ], XmNwidth, width ); n++;
      XtSetArg( args[ n ], XmNbackground, background ); n++;
      XtSetArg( args[ n ], XmNforeground, foreground ); n++;

      if( entry_info -> cache_state != IS_CACHED ) {
        XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
      }

      XtSetValues( entry_w, args, n );

      XmStringFree( xstr );

      n = 0;
      XtSetArg( args[ n ], XmNheight, &height ); n++;
      XtGetValues( entry_w, args, n );

      notes_anchor_y = notes_anchor_y + height + 3;
      if( notes_anchor_y > notes_max_y )
        notes_max_y = notes_anchor_y;

      if( entry_info -> appW != NULL )
        XtUnmapWidget( entry_info -> appW );

      if( entry_info -> appMarkW != NULL )
        XtUnmapWidget( entry_info -> appMarkW );

      if( ! XtIsManaged( entry_w ) )
        XtManageChild( entry_w );

      XtMapWidget( entry_w );

    } /* if */


    /* Is this an appointment? */
    if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY ) {

      int       marker_col;
      Position  y;
      Position  x;

      if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
        strcat( entry_buffer, "\n" );

      if( strlen( entry_buffer ) == 0 && entry_text == NULL ) {
        xstr = XmStringCreate( " ", CS );

      } else if( strlen( entry_buffer ) > 0 && entry_text == NULL ) {
        xstr = XmStringCreateLtoR( entry_buffer, CS );

      } else if( strlen( entry_buffer ) == 0 && entry_text != NULL ) {
        xstr = XmStringCreateLtoR( entry_text, CS );

      } else {
        xstr      = XmStringCreateLtoR( entry_buffer, CS );
        xstr_text = XmStringCreateLtoR( entry_text, CS );
        xstr      = xitStringConcatAndFree( xstr, xstr_text );
      }

      if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
        y = entry_anchor_y;
      else
        y = xtmStPositionEntryY( sched_ref, entry_record.entry.time_stamp );

      width  = xtmStWidthEntry( sched_ref );
      height = xtmStHeightEntry( sched_ref, 
                                 (int) entry_record.entry.duration );

      /* Widget to use. */
      entry_w = entry_info -> appW;

      if( entry_w == NULL ) {

        n = 0;
        XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_BEGINNING ); n++;
        XtSetArg( args[ n ], XitRwNoppositeAlignment, XmALIGNMENT_BEGINNING ); n++;
        XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
        XtSetArg( args[ n ], XmNspacing, 0 ); n++;
        entry_w = XtCreateWidget( "AppPb", xitRwPixButtonWidgetClass,
                                  entryBbW, args, n );

        /* Callbacks and event handlers. */
        XtAddEventHandler( entry_w, ButtonPressMask, False,
                           (XtEventHandler) xtmSaActionAppHook,
                           (XtPointer) sched_ref );

        if( custom_data_ref -> entries_interactive &&
            flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) )
          XtOverrideTranslations( entry_w, entry_leave_trans );

        XtAddCallback( entry_w, XmNactivateCallback, 
                       (XtCallbackProc) xtmSwSelectEntryCB,
                       (XtPointer) sched_ref );

        entry_info -> appW = entry_w;

      } /* if */

      n = 0;
      XtSetArg( args[ n ], XmNuserData, (void *) entry_info ); n++;
      XtSetArg( args[ n ], XmNx, entry_anchor_x ); n++;
      XtSetArg( args[ n ], XmNy, y ); n++;
      XtSetArg( args[ n ], XmNwidth, width ); n++;
      XtSetArg( args[ n ], XmNbackground, background ); n++;
      XtSetArg( args[ n ], XmNforeground, foreground ); n++;

      if( entry_info -> cache_state != IS_CACHED ) {
        XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
      }

      if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) ) {
        XtSetArg( args[ n ], XmNheight, 0 ); n++;
      } else {
        if( entry_record.entry.duration == 0 ) {
          XtSetArg( args[ n ], XmNheight, 0 ); n++;
        } else {
          XtSetArg( args[ n ], XmNheight, height ); n++;
        }
      }

      if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) ) {
        XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
      } else {
        XtSetArg( args[ n ], XmNshadowThickness, 2 ); n++;
      }

      XtSetValues( entry_w, args, n );
      XmStringFree( xstr );


      /* Make sure the widget is displayed. */
      if(entry_info -> noteW != NULL )
        XtUnmapWidget( entry_info ->  noteW );

      if( ! XtIsManaged( entry_w ) )
        XtManageChild( entry_w );

      XtMapWidget( entry_w );


      /* Fetch the height of the entry button. */
      n = 0;
      XtSetArg( args[ n ], XmNheight, &height ); n++;
      XtGetValues( entry_w, args, n );

      
      if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
        entry_anchor_y = entry_anchor_y + height + 3;

      if( entry_anchor_y > entry_max_y )
        entry_max_y = entry_anchor_y;

      /* Do we have a special appointment stacking? */
      if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) ) {

        int  stack_pos = 0;

        if( db_info.win_stack_order == 2 || 
            flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_KEEP_IN_BG ) )
          stack_pos = 2;

        if( db_info.win_stack_order == 1 ||
            flagIsSet( entry_record.entry.flags, XTM_DB_FLAG_KEEP_IN_FG ) )
          stack_pos = 1;

        if( stack_pos != 0 ) {
          win_stack_info[ apps_win ].stacking = stack_pos;
          win_stack_info[ apps_win ].widget   = entry_w;
          apps_win++;
        }

      } /* if */


      /* For each appointment button, we make a selection button. */
      if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) &&
          flagIsSet(   sched_ref -> flags, XTM_SM_ENTRY_HANDLES ) ) {

        /* Find the column for the marker. */
        marker_col = findMarkColumn( &win_stack_table, 
                                     (int) y, (int) (y + height - 1) );

        /* Create a button? */
        mark_w = entry_info -> appMarkW;

        if( mark_w == NULL && marker_col != -1 ) {

          button_data[ 0 ].name = "AppMarkPb";

          mark_w = xitCreatePushButton( entryBbW, &button_data[ 0 ] );

          n = 0;
          XtSetArg( args[ n ], XmNwidth, 4 ); n++;
          XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
          XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
          XtSetValues( mark_w, args, n );

          XtAddCallback( mark_w, XmNactivateCallback, 
                         (XtCallbackProc) selectEntryMarkCB,
                         (XtPointer) sched_ref );

          /* Event handler for popup menu. */
          XtAddEventHandler( mark_w, ButtonPressMask, False,
                             (XtEventHandler) xtmSaActionAppHook,
                             (XtPointer) sched_ref );

          entry_info -> appMarkW = mark_w;

        } /* if */

        if( marker_col != -1 ) {

          n = 0;
          XtSetArg( args[ n ], XmNforeground, &background ); n++;
          XtGetValues( entry_w, args, n );

          x = (Position) (xtmStWidthEntry( sched_ref ) + 2 + 
                        marker_col * 6 + entry_anchor_x);
          n = 0;
          XtSetArg( args[ n ], XmNx, x ); n++;
          XtSetArg( args[ n ], XmNy, y + 2 ); n++;
          XtSetArg( args[ n ], XmNheight, height - 4 ); n++;
          XtSetArg( args[ n ], XmNbackground, background ); n++;
          XtSetArg( args[ n ], XmNuserData, (void *) entry_w ); n++;
          XtSetValues( mark_w, args, n );

          /* Make sure the widget is displayed. */
          if( ! XtIsManaged( mark_w ) )
            XtManageChild( mark_w );

          XtMapWidget( mark_w );

        } /* if */

      } else {
        if( cache_ref -> used[ index ] -> appMarkW != NULL )
          XtUnmapWidget( cache_ref -> used[ index ] -> appMarkW );

      } /* if */

    } /* if */


    /* Mark the entry as used in the cache. */
    entry_info -> cache_state = IS_USED;


    /* Free allocated data. */
    SysFree( entry_text );

    /* Next entry. */
    lst_status = LstLinkCurrentNext( entries );

  } /* while */


  /* Re-stack appointments. */
  for( index = 0; index < apps_win; index++ ) {
    if( win_stack_info[ index ].stacking == 1 )
      XRaiseWindow( display, XtWindow( win_stack_info[ index ].widget ) );

    if( win_stack_info[ index ].stacking == 2 )
      XLowerWindow( display, XtWindow( win_stack_info[ index ].widget ) );
  }


  /* Paint the entries backgrount. */
  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    height = (Dimension) entry_max_y + 50;
  else
    height = sched_ref -> time_label_height;

  updateSchedEntriesBg( sched_ref,
                        schedule_start, schedule_stop,
                        (Dimension) notes_max_y + 50, height );


  return;

} /* updateSchedEntries */


/*----------------------------------------------------------------------*/

static void
  updateSchedEntriesBg( SCHED_REC_REF  sched_ref,
                        TIM_TIME_REF   schedule_start,
                        TIM_TIME_REF   schedule_stop,
                        Dimension      note_height,
                        Dimension      entry_height )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 50 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Dimension               width;
  Dimension               day_width;
  Display                 *display = XtDisplay( sched_ref -> scheduleW );
  Position                day_pos;
  Position                pos;
  Position                stop_pos;
  Widget                  sepW;
  Widget                  entryBbW;
  Widget                  mainW;
  Widget                  noteBbW;
  TIM_TIME_REF            curr_date;
  TIM_TIME_REF            curr_time;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Widgets we will need. */
  noteBbW = XtNameToWidget( 
              mainW, 
              "SchedPa.Pane1Fo.NoteSw.ScrolledWindowClipWindow.NoteBb" );

  entryBbW = XtNameToWidget( 
               mainW, 
               "SchedPa.Pane2Fo.EntrySw.ScrolledWindowClipWindow.EntryBb" );



  /* Width of each day. */
  day_width = sched_ref -> day_width;
  if( day_width < 40 )
    day_width = 40;


  /* Set the correct size of the bulletin boards. */
  width = (Dimension) sched_ref -> show_no_days * day_width + 5;

  /* Set the width and height. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, width ); n++;
  XtSetValues( entryBbW, args, n );
  XtSetValues( noteBbW,  args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, note_height + 5 ); n++;
  XtSetValues( noteBbW,  args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, entry_height ); n++;
  XtSetValues( entryBbW, args, n );


  /* Make the day separators. */
  curr_date = schedule_start;
  index     = 1;
  day_pos   = (Position) day_width;

  while( curr_date <= schedule_stop ) {

    /* Does the note separator exists? */
    sprintf( buffer, "NoteDaySep%dSp", index );

    sepW = XtNameToWidget( noteBbW, buffer );

    if( sepW == NULL ) {

      n = 0;
      XtSetArg( args[ n ], XmNy, 0 ); n++;
      XtSetArg( args[ n ], XmNorientation, XmVERTICAL );  n++;
      sepW = XmCreateSeparator( noteBbW, buffer, args, n );

    } /* if */

    n = 0;
    XtSetArg( args[ n ], XmNx, day_pos ); n++;
    XtSetArg( args[ n ], XmNheight, note_height ); n++;
    XtSetValues( sepW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( sepW ) )
      XtManageChild( sepW );
    else
    XtMapWidget( sepW );


    /* Does the entry separator exists? */
    sprintf( buffer, "EntryDaySep%dSp", index );

    sepW = XtNameToWidget( entryBbW, buffer );

    if( sepW == NULL ) {

      n = 0;
      XtSetArg( args[ n ], XmNy, 0 ); n++;
      XtSetArg( args[ n ], XmNorientation, XmVERTICAL );  n++;
      sepW = XmCreateSeparator( entryBbW, buffer, args, n );

    } /* if */

    n = 0;
    XtSetArg( args[ n ], XmNx, day_pos ); n++;
    XtSetArg( args[ n ], XmNheight, entry_height ); n++;
    XtSetValues( sepW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( sepW ) )
      XtManageChild( sepW );
    else
    XtMapWidget( sepW );


    /* Next day. */
    day_pos = day_pos + (Position) day_width;

    index++;    
    TimAddDays( &curr_date, 1 );

  } /* while */


  /* Unmap separators not used. */
  do {
    sprintf( buffer, "NoteDaySep%dSp", index );

    sepW = XtNameToWidget( noteBbW, buffer );
    if( sepW == NULL )
      break;

    XtUnmapWidget( sepW );

    sprintf( buffer, "EntryDaySep%dSp", index );

    sepW = XtNameToWidget( entryBbW, buffer );
    if( sepW == NULL )
      break;

    XtUnmapWidget( sepW );

    index++;
  } while( True );


  /* Make the grid lines. */
  index     = 1;
  pos       = (Position) (sched_ref -> time_line_height / 2) + 2;
  stop_pos  = (Position) entry_height;
  curr_time = sched_ref -> entry_start;

  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    stop_pos = pos;

  while( pos < stop_pos ) {

    /* Does the separator exists? */
    sprintf( buffer, "Grid%dSp", index );

    sepW = XtNameToWidget( entryBbW, buffer );

    if( sepW == NULL ) {

      n = 0;
      XtSetArg( args[ n ], XmNx, 0 ); n++;
      XtSetArg( args[ n ], XmNheight, 1 ); n++;
      XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
      sepW = XmCreateSeparator( entryBbW, buffer, args, n );

    } /* if */

    width = (Dimension) sched_ref -> show_no_days * day_width + 5;

    n = 0;
    XtSetArg( args[ n ], XmNy, pos ); n++;
    XtSetArg( args[ n ], XmNwidth, width ); n++;
    XtSetValues( sepW, args, n );

    n = 0;
    if( TimMinute( curr_time ) == 0 ) {
      XtSetArg( args[ n ], XmNseparatorType, XmSINGLE_LINE ); n++;
    } else {
      XtSetArg( args[ n ], XmNseparatorType, XmSINGLE_DASHED_LINE ); n++;
    }
    XtSetValues( sepW, args, n );

    /* Make sure it is displayed. */
    if( ! XtIsManaged( sepW ) )
      XtManageChild( sepW );
    else
    XtMapWidget( sepW );

    XLowerWindow( display, XtWindow( sepW ) );

    /* Next grid. */
    pos = pos + (Position) sched_ref -> time_line_height * 2;
    TimAddMinutes( &curr_time, sched_ref -> entry_delta );

    index++;
  } /* while */

  /* Unmap grid not used. */
  do {
    sprintf( buffer, "Grid%dSp", index );

    sepW = XtNameToWidget( entryBbW, buffer );
    if( sepW == NULL )
      break;

    XtUnmapWidget( sepW );

    index++;
  } while( True );


  /* Horizontal note separator. */
  width = (Dimension) sched_ref -> show_no_days * day_width + 5;

  sprintf( buffer, "NoteBottomSp" );
  sepW = XtNameToWidget( noteBbW, buffer );

  n = 0;
  XtSetArg( args[ n ], XmNx, 0 ); n++;
  XtSetArg( args[ n ], XmNy, (Position) note_height ); n++;
  XtSetArg( args[ n ], XmNwidth, width ); n++;
  XtSetValues( sepW, args, n );


  /* Horizontal entry separator. */
  width = (Dimension) sched_ref -> show_no_days * day_width + 5;

  sprintf( buffer, "EntryBottomSp" );
  sepW = XtNameToWidget( entryBbW, buffer );

  n = 0;
  XtSetArg( args[ n ], XmNx, 0 ); n++;
  XtSetArg( args[ n ], XmNy, (Position) entry_height - 2 ); n++;
  XtSetArg( args[ n ], XmNwidth, width ); n++;
  XtSetValues( sepW, args, n );


  return;

} /* updateSchedEntriesBg */


/*----------------------------------------------------------------------*/

static void
  updateSchedTimes( SCHED_REC_REF  sched_ref,
                    TIM_TIME_REF   start_time,
                    TIM_TIME_REF   stop_time )
{

  /* Variables. */
  Boolean               first_time;
  int                   label_lines;
  char                  buffer[ 50 ];
  char                  time_buffer[ 5000 ];
  Arg                   args[ 5 ];
  Cardinal              n;
  Widget                mainW;
  Widget                timeDispW;
  Widget                timeLaW;
  TIM_TIME_REF          current_time;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = sched_ref -> appl_data_ref;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Get the widgets we need. */
  timeDispW = XtNameToWidget( mainW, 
                              "SchedPa.Pane2Fo.TimeDispSw" );
  timeLaW = XtNameToWidget( timeDispW, 
                            "ScrolledWindowClipWindow.TimeDispBb.TimeDispLa" );


  /* Create a label containing the times. */
  label_lines      = 0;
  current_time     = start_time;
  first_time       = True;
  time_buffer[ 0 ] = '\0';

  while( current_time <= stop_time ) {
    xtmFoFormatTime( current_time, buffer, sizeof( buffer ) );

    if( ! first_time ) {
      strcat( time_buffer, "\n-\n" );
      label_lines++;
    } else {
      first_time = False;
    }

    strcat( time_buffer, buffer );
    label_lines++;

    TimAddMinutes( &current_time, sched_ref -> entry_delta );
  } /* while */


  /* Set the time label? */
  if( sched_ref -> entry_start != sched_ref -> old_entry_start ||
      sched_ref -> entry_stop  != sched_ref -> old_entry_stop ||
      sched_ref -> entry_delta != sched_ref -> old_entry_delta ||
      sched_ref -> force_update )
    xitStringSetLabel( timeLaW, time_buffer );


  /* The height of a single line in the time label. */
  {
    Dimension   height;
    Dimension   width;
    XmString    xstr;
    XmFontList  font_list;

    n = 0;
    XtSetArg( args[ n ], XmNfontList, &font_list ); n++;
    XtGetValues( timeLaW, args, n );

    xstr = XmStringCreate( " ", CS );
    XmStringExtent( font_list, xstr, &width, &height );
    XmStringFree( xstr );

    sched_ref -> time_line_height = height;
  }

  n = 0;
  XtSetArg( args[ n ], XmNheight, &sched_ref -> time_label_height ); n++;
  XtGetValues( timeLaW, args, n );


  /* Set the minute marker. */
  xtmStUpdateMinuteMarker( sched_ref );

  sched_ref -> old_entry_delta = sched_ref -> entry_delta;
  sched_ref -> old_entry_start = sched_ref -> entry_start;
  sched_ref -> old_entry_stop  = sched_ref -> entry_stop;


  return;

} /* updateSchedTimes */


/*----------------------------------------------------------------------*/

static void 
  dayWidthApplyCB( Widget         widget, 
                   SCHED_REC_REF  sched_ref,
                   XtPointer      call_data )
{

  /* Code. */

  doDayWidthCB( widget, sched_ref );


  return;

} /* dayWidthApplyCB */


/*----------------------------------------------------------------------*/

static void 
  dayWidthOkCB( Widget         widget, 
                SCHED_REC_REF  sched_ref,
                XtPointer      call_data )
{

  /* Variables. */
  Widget  baseW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ScaleValFd" );
  if( baseW == NULL )
    return;

  doDayWidthCB( widget, sched_ref );

  XtDestroyWidget( baseW );


  return;

} /* dayWidthOkCB */


/*----------------------------------------------------------------------*/

static void 
  dispDaysApplyCB( Widget         widget, 
                   SCHED_REC_REF  sched_ref,
                   XtPointer      call_data )
{

  /* Code. */

  doDispDaysCB( widget, sched_ref );


  return;

} /* dispDaysApplyCB */


/*----------------------------------------------------------------------*/

static void 
  dispDaysOkCB( Widget         widget, 
                SCHED_REC_REF  sched_ref,
                XtPointer      call_data )
{

  /* Variables. */
  Widget  baseW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ScaleValFd" );
  if( baseW == NULL )
    return;

  doDispDaysCB( widget, sched_ref );

  XtDestroyWidget( baseW );


  return;

} /* dispDaysOkCB */


/*----------------------------------------------------------------------*/

static void 
  doDayWidthCB( Widget         widget, 
                SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  int     value;
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ScaleValFd" );
  if( baseW == NULL )
    return;


  /* Fetch the slider value. */
  tempW = XtNameToWidget( baseW, "ScaleValFdFo.ScaleValSc" );

  XmScaleGetValue( tempW, &value );

  sched_ref -> day_width = (Dimension) value;


  /* Update the start and stop dates. */
  xtmSwSetScheduleDates( sched_ref, widget, True );

  /* Display the new schedule. */
  sched_ref -> force_update = True;

  xtmSwSetSchedule( sched_ref, widget );


  return;

} /* doDayWidthCB */


/*----------------------------------------------------------------------*/

static void 
  doDispDaysCB( Widget         widget, 
                SCHED_REC_REF  sched_ref )
{

  /* Variables. */
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  baseW = xitGetParentWidget( widget, "ScaleValFd" );
  if( baseW == NULL )
    return;


  /* Fetch the slider value. */
  tempW = XtNameToWidget( baseW, "ScaleValFdFo.ScaleValSc" );

  XmScaleGetValue( tempW, &sched_ref -> show_no_days );


  /* Update the start and stop dates. */
  xtmSwSetScheduleDates( sched_ref, widget, True );

  /* Display the new schedule. */
  sched_ref -> force_update = True;

  xtmSwSetSchedule( sched_ref, widget );


  return;

} /* doDispDaysCB */


/*----------------------------------------------------------------------*/

static int
  findMarkColumn( WIN_STACK_TABLE_REF  stack_table,
                  int                  fit_start,
                  int                  fit_end )
{

  /* Variables. */
  Boolean  overlapping;
  int      column;
  int      fit_column = -1;
  int      item;


  /* Code. */

  /* Search all columns. */
  for( column = 0; column < MAX_WIN_STACK_COLUMNS; column++ ) {

    overlapping = False;

    for( item = 0; item < stack_table -> entries; item++ ) {

      if( stack_table -> entry[ item ].column == column ) {

        if( ! (fit_end   < stack_table -> entry[ item ].start ||
               fit_start > stack_table -> entry[ item ].end ) )
          overlapping = True;

      } /* if */

    } /* loop */

    /* If no overlapping entries, this is our column. */
    if( ! overlapping ) {
      fit_column = column;
      break;
    }

  } /* loop */


  /* Save the entry in the table? */
  if( fit_column >= 0 ) {
    if( stack_table -> entries < MAX_WIN_STACK_REC ) {
      stack_table -> entry[ stack_table -> entries ].start  = fit_start;
      stack_table -> entry[ stack_table -> entries ].end    = fit_end;
      stack_table -> entry[ stack_table -> entries ].column = fit_column;

      stack_table -> entries++;
    }
  }


  return( fit_column );

} /* findMarkColumn */


/*----------------------------------------------------------------------*/

static void
  selectEntryMarkCB( Widget         widget,
                     SCHED_REC_REF  sched_ref,
                     XtPointer      call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    entryPbW;


  /* Code. */

  /* Fetch the appointmnet widget. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &entryPbW ); n++;
  XtGetValues( widget, args, n );

  XRaiseWindow( XtDisplay( entryPbW ), XtWindow(  entryPbW ) );


  return;

} /* selectEntryMarkCB */
