/*----------------------------------------------------------------------------
--
--  Module:           xtmUpdate
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Centralize update management. Various sources of update requests
--    can be managed.
--
--  Filename:         xtmUpdate.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmUpdate.c, Version: 1.1, Date: 95/02/18 15:52:54";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "TimDate.h"

#include "xtmUpdate.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define  MAX_UPDATE_REG  200


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  /* Id of the update function (unique). If 0, slot is free. */
  UINT32  id;

  /* Call the update function when one of the following happens. */
  UINT32  action_flags;

  /* Function to call. */
  XTM_UP_UPDATE_CB  updateCB;

  /* User data. */
  void  *user_data;

} UPDATE_REC, *UPDATE_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmUpdate";

/* Unique ID counter. */
static  int  unique_id = 0;

/* Delta time in munites for refresh calls. */
static  int  refresh_min;

/* The last time checked. */
static  TIM_TIME_REF  last_check = 0;

/* Our update DB. */
static  UPDATE_REC  updateDb[ MAX_UPDATE_REG ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  minuteTimer( XtAppContext  context );

static void
  refreshTimer( XtAppContext  context );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmUpDoUpdate( UINT32  flags,
                 void    *update_user_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Do we have a registred callback? */
  for( index = 0; index < MAX_UPDATE_REG; index++ ) {

    /* Correct pattern. */
    if( updateDb[ index ].id != 0 &&
        flagIsSet( updateDb[ index ].action_flags, flags ) ) {

      /* Call the callback. */
      (* updateDb[ index ].updateCB)( flags, 
                                      updateDb[ index ].user_data,
                                      update_user_data );

    } /* if */

  } /* loop */


  return;

} /* xtmUpDoUpdate */


/*----------------------------------------------------------------------*/

void
  xtmUpInitialize( XtAppContext  context,
                   int           refresh_delta )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* No callbacks registered. */
  for( index = 0; index < MAX_UPDATE_REG; index++ )
    updateDb[ index ].id = 0;


  /* Start the minute timer. */
  minuteTimer( context );

  /* Start the refresh timer. */
  refresh_min = refresh_delta;

  if( refresh_min > 0 )
    refreshTimer( context );


  return;

} /* xtmUpInitialize */


/*----------------------------------------------------------------------*/

UINT32
  xtmUpRegister( UINT32            flags,
                 XTM_UP_UPDATE_CB  updateCB,
                 void              *user_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Any free slots? */
  for( index = 0; index < MAX_UPDATE_REG; index++ ) {
    if( updateDb[ index ].id == 0 )
      break;
  }

  if( index == MAX_UPDATE_REG )
    return( 0 );


  /* Save the update data. */
  unique_id++;

  updateDb[ index ].id           = unique_id;
  updateDb[ index ].action_flags = flags;
  updateDb[ index ].updateCB     = updateCB;
  updateDb[ index ].user_data    = user_data;


  return( unique_id );

} /* xtmUpRegister */


/*----------------------------------------------------------------------*/

void
  xtmUpRemove( UINT32  id )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Find the registered id. */
  for( index = 0; index < MAX_UPDATE_REG; index++ ) {

    if( updateDb[ index ].id == id ) {
      updateDb[ index ].id = 0;

      return;
    }

  } /* loop */


  return;

} /* xtmUpRemove */


/*----------------------------------------------------------------------*/

static void
  minuteTimer( XtAppContext  context )
{

  /* Variables. */
  int           flags = 0;
  int           next_minute;
  TIM_TIME_REF  now;


  /* Code. */

  now = TimLocalTime( TimMakeTimeNow() );

  /* Do we have a new day? */
  if( last_check != 0 ) {
    if( TimIsSameDate( now, last_check ) != TIM_YES )
      flagSet( flags, XTM_UP_NEW_DAY );
  }


  /* Call the update function. */
  flagSet( flags, XTM_UP_MINUTE_TICK );

  xtmUpDoUpdate( flags, NULL );


  last_check = now;

  /* Schedule the next wakeup call. */
  next_minute = (60 - now % 60) * 1000;

  XtAppAddTimeOut( context, next_minute,
                   (XtTimerCallbackProc) minuteTimer, (XtPointer) context );


  return;

} /* minuteTimer */


/*----------------------------------------------------------------------*/

static void
  refreshTimer( XtAppContext  context )
{

  /* Variables. */
  int           next_call;
  TIM_TIME_REF  now;


  /* Code. */

  now = TimLocalTime( TimMakeTimeNow() );


  /* Call the update function. */
  xtmUpDoUpdate( XTM_UP_REFRESH_TICK, NULL );


  /* Schedule the next wakeup call. */
  next_call = (refresh_min - 1) * 60 * 1000 + (60 - now % 60) * 1000;

  XtAppAddTimeOut( context, next_call,
                   (XtTimerCallbackProc) refreshTimer, (XtPointer) context );


  return;

} /* refreshTimer */


