/*----------------------------------------------------------------------------
--
--  Module:           XmUbMonthDisplay
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation of the widget.
--
--  Filename:         XmUbMonthD.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-26
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: XmUbMonthD.c, Version: 1.2, Date: 95/06/21 20:47:47";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <Xm/XmP.h>

#if XmVersion > 1001
#include <Xm/AtomMgr.h>
#include <Xm/DragDrop.h>
#endif

#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>

/* Rogge's tools. */
#include "TimDate.h"

/* Private widget header file. */
#include "XmUbMonthDP.h"

/* Other widgets. */
#include "XmUbArrLab.h"

/* Drag and drop. */
#if XmVersion > 1001
#include "XmUbDragDrop.h"
#include "DateDragIcon.h"
#include "DateDragMask.h"
#include "DateDragInvIcon.h"
#include "DateDragInvIconMask.h"
#endif

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#ifdef MAX
#undef MAX
#endif

#ifdef MIN
#undef MIN
#endif

#define MAX( x, y )  ( ( x ) > ( y ) ? ( x ) : ( y ) )
#define MIN( x, y )  ( ( x ) < ( y ) ? ( x ) : ( y ) )

#define XmUbMD_CHILD_ERROR  -1

#define LABEL_BUFFER_LENGTH  30
#define TEXT_BUFFER_LENGTH   20

#define PREVIOUS_MONTH_OFFSET  100
#define NEXT_MONTH_OFFSET      200

#define TITLE_ID_USER_DATA     1

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  Dimension  width;
  Dimension  height;
  Position   x;
  Position   y;

} KidDimensionRec;

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Internal Motif functions (how do we know about this....?) */
extern void
  _XmBackgroundColorDefault();
extern void
  _XmForegroundColorDefault();

static char  default_month_string[] = "Month";
static char  default_year_string[]  = "Year";

/* The month display resource list. */
static XtResource resources[] = {
  {
    XmUbNmdiDateSelectedCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbMonthDisplayWidget, md.date_selected_callback ),
    XmRCallback,
    NULL
  },
  {
    XmUbNmdiNewMonthCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbMonthDisplayWidget, md.new_month_callback ),
    XmRCallback,
    NULL
  },
  {
    XmUbNmdiTitleCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbMonthDisplayWidget, md.title_callback ),
    XmRImmediate,
    (XtPointer) NULL
  },
  {
    XmNmarginHeight,
    XmCMarginHeight,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbMonthDisplayWidget, md.margin_height ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmNmarginWidth,
    XmCMarginWidth,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbMonthDisplayWidget, md.margin_width ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNmdiOtherMonthForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.other_month_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiArrowSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbMonthDisplayWidget, md.arrow_spacing ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNmdiCurrentMonth,
    XmUbCMdiCurrentMonth,
    XtRInt, sizeof( int ),
    XtOffset( XmUbMonthDisplayWidget, md.current_month ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNmdiCurrentYear,
    XmUbCMdiCurrentYear,
    XtRInt, sizeof( int ),
    XtOffset( XmUbMonthDisplayWidget, md.current_year ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNmdiDayChars,
    XmUbCMdiDayChars,
    XtRInt, sizeof( int ),
    XtOffset( XmUbMonthDisplayWidget, md.day_name_characters ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNmdiEnableSeparator,
    XmUbCMdiEnableSeparator,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.enable_separator ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiFillOutWeek,
    XmUbCMdiFillOutWeek,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.fill_out_week ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiLabelSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbMonthDisplayWidget, md.label_spacing ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNmdiMonthLabel,
    XmUbCMdiMonthLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbMonthDisplayWidget, md.month_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNmdiMonthArrows,
    XmUbCMdiMonthArrows,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.month_arrows ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiRecomputeHeight,
    XmUbCMdiRecomputeHeight,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.recompute_height ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiRecomputeWidth,
    XmUbCMdiRecomputeWidth,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.recompute_width ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiSaturdayForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.sat_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiSundayForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.sun_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiTitleAlignment,
    XmCAlignment,
    XmRAlignment, sizeof( unsigned char ),
    XtOffset( XmUbMonthDisplayWidget, md.title_alignment ),
    XtRImmediate,
    (XtPointer) XmALIGNMENT_CENTER
  },
  {
    XmUbNmdiTitleShadows,
    XmUbCMdiTitleShadows,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.title_shadows ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiTodayForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.today_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiUpdateDayCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbMonthDisplayWidget, md.update_day_callback ),
    XmRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNmdiUseDefaultTitleCallback,
    XmUbCMdiUseDefaultTitleCallback,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.use_default_title_callback ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiWeekLabel,
    XmUbCMdiWeekLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbMonthDisplayWidget, md.week_number_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNmdiWeekNumberForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.week_number_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiWeekNumbers,
    XmUbCMdiWeekNumbers,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.week_numbers ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiWeekdayForeground,
    XmCForeground,
    XtRPixel, sizeof( Pixel ),
    XtOffset( XmUbMonthDisplayWidget, md.weekday_foreground ),
    XmRCallProc,
    (XtPointer) _XmForegroundColorDefault
  },
  {
    XmUbNmdiYearArrows,
    XmUbCMdiYearArrows,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbMonthDisplayWidget, md.year_arrows ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNmdiYearLabel,
    XmUbCMdiYearLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbMonthDisplayWidget, md.year_label ),
    XtRImmediate,
    (XtPointer) NULL
  },

};  /* resources */


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Class methods. */
static void
  ChangeManaged( Widget  widget );

static void              
  ClassInitialize();

static void
  DeleteChild( Widget  widget );

static void 
  Destroy( Widget   widget );

static void
  ExposeMonthDisplay( Widget  widget,
                      XEvent  *event,
                      Region  region );

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply );

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args );

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args );

static void
  InsertChild( Widget  widget );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget    widget );

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args );


/* Internal functions. */

/* Gets the date that is set for a button. */
static void
  ButtonDate( XmUbMonthDisplayWidget  md,
              Widget                  button,
              int                     *year,
              int                     *month,
              int                     *day );

static void
  CallNewMonthCallbacks( XmUbMonthDisplayWidget  md );

/* If not in the child list, returns XmUbMD_CHILD_ERROR. */
static int
  ChildIndex( XmUbMonthDisplayWidget  md,
              Widget                  child );

static void
  CreateMonthRowColumns( XmUbMonthDisplayWidget  md,
                         String                  name );

static void
  CreateInternalWidgets( XmUbMonthDisplayWidget  md );

static Widget
  DateButton( XmUbMonthDisplayWidget  md,
              int                     day_number );

static int
  DateButtonIndex( XmUbMonthDisplayWidget  md,
                   int                     day_number );

#if XmVersion > 1001
static Boolean
  DateDragConvertProc( Widget         w,
                       Atom           *selection,
                       Atom           *target,
                       Atom           *typeRtn,
                       XtPointer      *valueRtn,
                       unsigned long  *lengthRtn,
                       int            *formatRtn,
                       unsigned long  *max_lengthRtn,
                       XtPointer      client_data,
                       XtRequestId    *request_id );

static void
  DateEnterDropSiteCB( Widget                         drag_context,
                       XmUbMonthDisplayWidget         md,
                       XmDropSiteEnterCallbackStruct  *call_data );

static void
  DateLeaveDropSiteCB( Widget                         drag_context,
                       XmUbMonthDisplayWidget         md,
                       XmDropSiteLeaveCallbackStruct  *call_data );
#endif

static void
  DayNumberButtonActivatedCB( Widget                      pb,
                              XmUbMonthDisplayWidget      md,
                              XmPushButtonCallbackStruct  *call_data );


/* The sizes array  must have been processed by GetChildPrefSizes 
   before DoLayout is called. */
static void
  DoLayout( XmUbMonthDisplayWidget  md,
            Widget                  initiator,
            XtWidgetGeometry        *request,
            KidDimensionRec         sizes[] );

/* Fills in the width and height fields in the kids dimension recs. */
static void
  GetChildPrefSizes( XmUbMonthDisplayWidget  md,
                     Widget                  initiator,
                     XtWidgetGeometry        *request,
                     KidDimensionRec         sizes[] );


/* The sizes array must have been prepared by GetChildPrefSizes before this
   function is called. */
static void
  GetOwnPreferredSize( XmUbMonthDisplayWidget  md,
                       KidDimensionRec         sizes[],
                       Dimension               *pref_width,
                       Dimension               *pref_height );

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired );

static void
  MonthScrolledCB( Widget                        pb,
                   XmUbMonthDisplayWidget        md,
                   XmUbArrowLabelCallbackStruct  *call_data );

/* ResizeIfNeeded fills in the sizes array. Does not have to be initialized. */
static Boolean
  ResizeIfNeeded( XmUbMonthDisplayWidget  md,
                  KidDimensionRec         sizes[] );

static void
  SetDayNames( XmUbMonthDisplayWidget  md );

#if XmVersion > 1001
static void
  StartDateDrag( Widget    w,
                 XEvent    *event,
                 String    *params,
                 Cardinal  num_params );
#endif

static void
  TitleBarPressedCB( Widget                      pb,
                     XmUbMonthDisplayWidget      md,
                     XmPushButtonCallbackStruct  *call_data );

static void
  UpdateDayButton( XmUbMonthDisplayWidget  md,
                   int                     index,
                   int                     date,
                   XmUbWhichMonth          which_month );

static void
  UpdateNewMonth( XmUbMonthDisplayWidget  md );

static void
  UpdateWeekButton( XmUbMonthDisplayWidget  md,
                    int                     index,
                    TIM_TIME_REF            time_of_first_day,
                    Boolean                 mapped );

static void
  WarningNoResourceChange( XmUbMonthDisplayWidget  md,
                           String                  resource );

static void
  WeekNumberButtonActivatedCB( Widget                      pb,
                               XmUbMonthDisplayWidget      md,
                               XmPushButtonCallbackStruct  *call_data );

static void
  YearScrolledCB( Widget                        pb,
                  XmUbMonthDisplayWidget        md,
                  XmUbArrowLabelCallbackStruct  *call_data );

/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

#if XmVersion > 1001

/* Date label translation list. */
static char dateTranslationTable[] =
  "<Btn2Down>: start-date-drag()";

static XtTranslations  dateTranslations;

/* Action list. */
static XtActionsRec actionsList[] =
{
  { "start-date-drag", (XtActionProc) StartDateDrag },
};

#endif

/* This initialization has to be done after the methods have been declared. */
XmUbMonthDisplayClassRec xmUbMonthDisplayClassRec = {

  { /* Core class fields. */
    /* superclass */                    (WidgetClass) &xmManagerClassRec,
    /* class_name */                    "XmUbMonthDisplay",
    /* widget_size */                   sizeof( XmUbMonthDisplayRec ),
    /* class_initialize */              ClassInitialize,
    /* class_part_initialize */         NULL,
    /* class_inited */                  False,
    /* initialize */                    Initialize,
    /* initialize_hook */               NULL,
    /* realize */                       XtInheritRealize, 
#if XmVersion > 1001
    /* actions */                       actionsList,
    /* num_actions */                   XtNumber( actionsList ),
#else
    /* actions */                       NULL,
    /* num_actions */                   0,
#endif
    /* resources */                     resources,
    /* num_resources */                 XtNumber( resources ),
    /* xrm_class */                     NULLQUARK,
    /* compress_motion */               True,
    /* compress_exposure */             True,
    /* compress_enterleave */           True,
    /* visible_interest */              False,
    /* destroy */                       Destroy,
    /* resize */                        Resize,
    /* expose */                        ExposeMonthDisplay,
    /* set_values */                    SetValues,
    /* set_values_hook */               NULL,
    /* set_values_almost */             XtInheritSetValuesAlmost,
    /* get_values_hook */               GetValuesHook,
    /* accept_focus */                  NULL,
    /* version */                       XtVersion,
    /* callback_private */              NULL,
    /* tm_table */                      NULL,
    /* query_geometry */                QueryGeometry,
    /* display_accelerator */           XtInheritDisplayAccelerator,
    /* extension */                     NULL
  },
  { /* Composite class part. */
    /* geometry_manager */              GeometryManager,
    /* change_managed */                ChangeManaged,
    /* insert_child */                  InsertChild,
    /* delete_child */                  DeleteChild,
    /* extension */                     NULL
  },
  { /* Constraint class fields. */
    /* subresources */                  NULL,
    /* subresource_count */             0,
    /* constraint_size */               sizeof( 
                                          XmUbMonthDisplayConstraintsRec ),
    /* initialize */                    NULL,
    /* destroy */                       NULL,
    /* set_values */                    NULL,
    /* extension */                     NULL
  },
  { /* XmManager class part. */
    /* translations */                  NULL,
    /* get_resources */                 NULL,
    /* num_get_resources */             0,
    /* get_constraint_resources */      NULL,
    /* num_get_constraint_resources */  0,
    /* extension */                     NULL
  },
  { /* Month display class part. */
    /* extension */                     NULL
  },

}; 


/* Class record pointer. */
WidgetClass 
  xmUbMonthDisplayWidgetClass = (WidgetClass) &xmUbMonthDisplayClassRec;



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/


static void
  ButtonDate( XmUbMonthDisplayWidget  md,
              Widget                  button,
              int                     *year,
              int                     *month,
              int                     *day )
{
  /* Variables. */
  Arg       args[ 1 ];
  Cardinal  n;
  int       user_data;

  /* Code. */

  /* The date information is in the user data resource of the button. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &user_data ); n++;
  XtGetValues( button, args, n );

  *day = user_data % 100;

  if( user_data < PREVIOUS_MONTH_OFFSET ){
    /* Current month. */
    *month = md -> md.current_month;
    *year  = md -> md.current_year;

  } else if( user_data < NEXT_MONTH_OFFSET ){
    /* Previous month. */
    *month = md -> md.current_month - 1;
    *year  = md -> md.current_year;

    if( *month < 1 ){
      *month = 12;
      (*year) --;
    }

  } else {
    /* Next month. */
    *month = md -> md.current_month + 1;
    *year  = md -> md.current_year;

    if( *month > 12 ){
      *month = 1;
      (*year) ++;
    }
  }


  return;

} /* ButtonDate */


/*----------------------------------------------------------------------*/

static void
  CallNewMonthCallbacks( XmUbMonthDisplayWidget  md )
{
  /* Variables. */
  XmUbMonthDisplayCallbackStruct  cb;

  /* Code. */

  if( md -> md.new_month_callback == NULL )
    return;

  /* Callbacks are defined. */
  /* Set up the callback structure. */
  cb.reason         = XmUbCR_NEW_MONTH;
  cb.event          = NULL;
  cb.selected_month = md -> md.current_month;
  cb.selected_year  = md -> md.current_year;

  /* Call the user callbacks. */
  XtCallCallbackList( (Widget) md, md -> md.new_month_callback,
                      (XtPointer) &cb );


  return;

} /* CallNewMonthCallbacks */


/*----------------------------------------------------------------------*/

static void
  ChangeManaged( Widget  widget )
{
  /* Variables. */
  Boolean                 layout_done;
  KidDimensionRec         kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbMonthDisplayWidget  md;


  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* ResizeIfNeeded checks the size of all children and resizes the widget. */
  layout_done = ResizeIfNeeded( md, kids_sizes );

  /* Do the layout. */
  if( !layout_done )
    DoLayout( md, NULL, NULL, kids_sizes );


  return;

} /* ChangeManaged */


/*----------------------------------------------------------------------*/

static int
  ChildIndex( XmUbMonthDisplayWidget  md,
              Widget                  child )
{
  /* Variables. */
  int  index;

  /* Code. */

  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){

    if( md -> md.internal_children[ index ] == child )
      return index;

  } /* for */

 
  /* Specified child not found. */
  return XmUbMD_CHILD_ERROR;

} /* ChildIndex */


/*----------------------------------------------------------------------*/

static void              
  ClassInitialize()
{
  /* Code. */

#if XmVersion > 1001
  /* Parse translations tables. */
  dateTranslations = XtParseTranslationTable( (String) dateTranslationTable );
#endif

  return;

} /* ClassInitialize */


/*----------------------------------------------------------------------*/

static void
  CreateInternalWidgets( XmUbMonthDisplayWidget  md )
{
  /* Variables. */
  Arg           args[ 5 ];
  Cardinal      n;
  char          *name_buffer;
  String        wname;
  XmString      xm;

  /* Code. */

  /* Get the name of the "parent" widget. */
  wname = XtName( md );
  name_buffer = XtMalloc( strlen( wname ) + 4 );

  /* Title bar. */

  sprintf( name_buffer, "%sTi", wname );

  n = 0;
  if( md -> md.title_shadows ){
    XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
  } else {
    XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  }

  XtSetArg( args[ n ], XmNuserData, TITLE_ID_USER_DATA ); n++;
  md -> md.internal_children[ XmUbMD_CHILD_TITLE ] = 
    XmCreatePushButton( (Widget) md, name_buffer, args, n );
  XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_TITLE ] );

  XtAddCallback( md -> md.internal_children[ XmUbMD_CHILD_TITLE ],
                 XmNactivateCallback,
                 (XtCallbackProc) TitleBarPressedCB, (XtPointer) md );

  /* Provide some user feedback -- no pressing if no action. */
  if( ( md -> md.title_callback == NULL ) && 
      ! md -> md.use_default_title_callback ){

    n = 0;
    XtSetArg( args[ n ], XmNfillOnArm, False ); n++;
    XtSetValues( md -> md.internal_children[ XmUbMD_CHILD_TITLE ], args, n );
  }

#if XmVersion > 1001
  /* We have our own processing of drag. */
  XtOverrideTranslations( md -> md.internal_children[ XmUbMD_CHILD_TITLE ],
                          dateTranslations ); 
#endif

  /* Month display. */

  sprintf( name_buffer, "%sRc", wname );
  CreateMonthRowColumns( md, name_buffer );

  XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_ROW_COLUMN ] );
  if( md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ] != NULL )
    XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ] );

  /* Month arrows. */
  if( md -> md.month_arrows ){
    sprintf( name_buffer, "%sMa", wname );

    n = 0;

    if( md -> md.month_label == NULL )
      xm = XmStringCreateLtoR( default_month_string,
                               XmSTRING_DEFAULT_CHARSET );
    else
      xm = md -> md.month_label;

    XtSetArg( args[ n ], XmNlabelString, xm ); n++;

    md -> md.internal_children[ XmUbMD_CHILD_MONTH_ARROW ] = 
      XmUbCreateArrowLabel( (Widget) md, name_buffer, args, n );

    if( md -> md.month_label == NULL )
      XmStringFree( xm );

    XtAddCallback( md -> md.internal_children[ XmUbMD_CHILD_MONTH_ARROW ],
                   XmNactivateCallback,
                   (XtCallbackProc) MonthScrolledCB, (XtPointer) md );

    XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_MONTH_ARROW ] );
  }

  /* Year arrows. */
  if( md -> md.year_arrows ){
    sprintf( name_buffer, "%sYa", wname );

    n = 0;

    if( md -> md.year_label == NULL )
      xm = XmStringCreateLtoR( default_year_string,
                               XmSTRING_DEFAULT_CHARSET );
    else
      xm = md -> md.year_label;

    XtSetArg( args[ n ], XmNlabelString, xm ); n++;

    md -> md.internal_children[ XmUbMD_CHILD_YEAR_ARROW ] = 
      XmUbCreateArrowLabel( (Widget) md, name_buffer, args, n );

    if( md -> md.year_label == NULL )
      XmStringFree( xm );

    XtAddCallback( md -> md.internal_children[ XmUbMD_CHILD_YEAR_ARROW ],
                   XmNactivateCallback,
                   (XtCallbackProc) YearScrolledCB, (XtPointer) md );

    XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_YEAR_ARROW ] );
  }

  /* Create separator if there are any arrow buttons. */
  if( ( md -> md.year_arrows || md -> md.month_arrows ) && 
      md -> md.enable_separator ){
    sprintf( name_buffer, "%sSp", wname );

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;

    md -> md.internal_children[ XmUbMD_CHILD_SEPARATOR ] = 
      XmCreateSeparator( (Widget) md, name_buffer, args, n );

    XtManageChild( md -> md.internal_children[ XmUbMD_CHILD_SEPARATOR ] );
  }


  XtFree( name_buffer );


  return;

} /* CreateInternalWidgets */


/*----------------------------------------------------------------------*/

static void
  CreateMonthRowColumns( XmUbMonthDisplayWidget  md,
                         String                  name )
{
  /* Variables. */
  Arg        args[ 6 ];
  XmString   empty_xm;
  Dimension  height, max_height;
  int        index;
  Cardinal   n;
  char       *name_buffer;
  Widget     rc;
  Widget     tmp_widget;
  Dimension  width, max_width;
  XmString   xm;

  /* Code. */

  name_buffer = XtMalloc( strlen( name ) + 5 );

  sprintf( name_buffer, "%sRc", name );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 7 ); n++;
  XtSetArg( args[ n ], XmNspacing, 0 ); n++;
  rc = XmCreateRowColumn( (Widget) md, name_buffer, args, n );

  /* We have to start by setting the label empty, since we look for the 
     biggest one and don't want ghosts. */
  empty_xm = XmStringCreateLtoR( "", XmSTRING_DEFAULT_CHARSET );

  /* Create the day name buttons. */
  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, False ); n++;
  XtSetArg( args[ n ], XmNlabelString, empty_xm ); n++;

  for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ ){
    sprintf( name_buffer, "%sDn%d", name, index );
    md -> md.day_name_buttons[ index ] = 
      XmCreateLabel( rc, name_buffer, args, n );
  }

  /* Create the day number buttons. */
  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
  XtSetArg( args[ n ], XmNlabelString, empty_xm ); n++;
  XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_END ); n++;

  for( index = 0; index < NO_DAY_BUTTONS; index ++ ){
    sprintf( name_buffer, "%sDb%d", name, index );
    md -> md.day_number_buttons[ index ] = 
      XmCreatePushButton( rc, name_buffer, args, n );

    XtAddCallback( md -> md.day_number_buttons[ index ], XmNactivateCallback,
                   (XtCallbackProc) DayNumberButtonActivatedCB, 
                   (XtPointer) md );

#if XmVersion > 1001
    /* We have our own processing of drag. */
    XtOverrideTranslations( md -> md.day_number_buttons[ index ], 
                            dateTranslations ); 
#endif
  }

  /* Save the default shadow colors. */
  n = 0;
  XtSetArg( args[ n ], XmNtopShadowColor, 
                       & md -> md.pushb_default_topshadow ); n++;
  XtSetArg( args[ n ], XmNbottomShadowColor, 
                       & md -> md.pushb_default_bottomshadow ); n++;
  XtGetValues( md -> md.day_number_buttons[ 0 ], args, n );


  /* Create the week number row column if desired. */
  if( md -> md.week_numbers ){

    sprintf( name_buffer, "%sWrc", name );

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
    XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
    XtSetArg( args[ n ], XmNnumColumns, 7 ); n++;
    XtSetArg( args[ n ], XmNspacing, 0 ); n++;
    md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ] = 
      XmCreateRowColumn( (Widget) md, name_buffer, args, n );

    for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ ){
      sprintf( name_buffer, "%sWb%d", name, index );

      n = 0;
      XtSetArg( args[ n ], XmNlabelString, empty_xm ); n++;
      XtSetArg( args[ n ], XmNuserData, 0 ); n++;
      if( index == 0 ){
        XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
      } else {
        XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
      }
      XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_END ); n++;

      if( index == 0 )
        md -> md.week_number_buttons[ index ] = 
          XmCreateLabel( md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ],
                         name_buffer, args, n );
      else {
        md -> md.week_number_buttons[ index ] = XmCreatePushButton( 
          md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ],
          name_buffer, args, n );

        XtAddCallback( md -> md.week_number_buttons[ index ], 
                       XmNactivateCallback,
                       (XtCallbackProc) WeekNumberButtonActivatedCB, 
                       (XtPointer) md );

#if XmVersion > 1001
        /* We have our own processing of drag. */
        XtOverrideTranslations( md -> md.week_number_buttons[ index ], 
                                dateTranslations ); 
#endif
      }
    }

    /* Set the week number "title". */
    if( md -> md.week_number_label == NULL )
      xm = XmStringCreateLtoR( "W", XmSTRING_DEFAULT_CHARSET );
    else
      xm = md -> md.week_number_label;

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xm ); n++;
    XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
    XtSetValues( md -> md.week_number_buttons[ 0 ], args, n );

    if( md -> md.week_number_label == NULL )
      XmStringFree( xm );
  }

  /* Set the texts of the buttons and the title. */
  UpdateNewMonth( md );

  /* Manage the created internal buttons. */
  XtManageChildren( md -> md.day_name_buttons, NO_DAY_NAME_BUTTONS );
  XtManageChildren( md -> md.day_number_buttons, NO_DAY_BUTTONS );
  if( md -> md.week_numbers )
    XtManageChildren( md -> md.week_number_buttons, NO_WEEK_NUMBER_BUTTONS );

  /* Find out biggest button and set all to that size. */

  max_height = 0;
  max_width  = 0;

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  
  for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ ){
    XtGetValues( md -> md.day_name_buttons[ index ], args, n );

    if( height > max_height )
      max_height = height;
    if( width > max_width )
      max_width = width;

  } /* for */

  for( index = 0; index < NO_DAY_BUTTONS; index ++ ){
    XtGetValues( md -> md.day_number_buttons[ index ], args, n );

    if( height > max_height )
      max_height = height;
    if( width > max_width )
      max_width = width;

  } /* for */

  if( md -> md.week_numbers ){
    for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ ){
      XtGetValues( md -> md.week_number_buttons[ index ], args, n );

      if( height > max_height )
        max_height = height;
      if( width > max_width )
        max_width = width;

    } /* for */
  }   

  /* Set all to max size and hint not to resize again. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, max_width ); n++;
  XtSetArg( args[ n ], XmNheight, max_height ); n++;
  XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
  XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_END ); n++;

  for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ )
    XtSetValues( md -> md.day_name_buttons[ index ], args, n );
  for( index = 0; index < NO_DAY_BUTTONS; index ++ )
    XtSetValues( md -> md.day_number_buttons[ index ], args, n );

  if( md -> md.week_numbers ){
    for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ )
      XtSetValues( md -> md.week_number_buttons[ index ], args, n );
  }

  /* Set row column geometry explicitly. */
#if XmVersion < 1002
  {
    Dimension  margin_height;
    Dimension  margin_width;
    Widget     wrc;

    /* Code. */

    n = 0;
    XtSetArg( args[ n ], XmNmarginWidth, &margin_width ); n++;
    XtSetArg( args[ n ], XmNmarginHeight, &margin_height ); n++;
    XtGetValues( rc, args, n );

    n = 0;  
    XtSetArg( args[ n ], XmNwidth, max_width * 7 + 2 * margin_width ); n++;
    XtSetArg( args[ n ], XmNheight, max_height * 7 + 2 * margin_height ); n++;
    XtSetArg( args[ n ], XmNnumColumns, 7 ); n++;
    XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
    XtSetArg( args[ n ], XmNresizeHeight, False ); n++;
    XtSetArg( args[ n ], XmNresizeWidth, False ); n++;
    XtSetValues( rc, args, n );

    if( md -> md.week_numbers ){
      wrc = md -> md.internal_children[ XmUbMD_CHILD_WEEK_RC ];
      n = 0;
      XtSetArg( args[ n ], XmNmarginWidth, &margin_width ); n++;
      XtSetArg( args[ n ], XmNmarginHeight, &margin_height ); n++;
      XtGetValues( wrc, args, n );

      n = 0;  
      XtSetArg( args[ n ], XmNwidth, max_width + 2 * margin_width ); n++;
      XtSetArg( args[ n ], XmNheight, 
                           max_height * 7 + 2 * margin_height ); n++;
      XtSetArg( args[ n ], XmNnumColumns, 7 ); n++;
      XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
      XtSetArg( args[ n ], XmNresizeHeight, False ); n++;
      XtSetArg( args[ n ], XmNresizeWidth, False ); n++;
      XtSetValues( wrc, args, n );
    }
  }
#endif

  /* Cleanup. */
  XtFree( name_buffer );
  XmStringFree( empty_xm );


  md -> md.internal_children[ XmUbMD_CHILD_ROW_COLUMN ] = rc;


  return;

} /* CreateMonthRowColumns */


/*----------------------------------------------------------------------*/

static Widget
  DateButton( XmUbMonthDisplayWidget  md,
              int                     day_number )
{
  /* Variables. */
  int  day_index;
  
  /* Code. */

  day_index = DateButtonIndex( md, day_number );

  if( day_index > ( NO_DAY_BUTTONS - 1 ) )
    return( NULL );

  else
    return( md -> md.day_number_buttons[ day_index ] );


} /* DateButton */


/*----------------------------------------------------------------------*/

static int
  DateButtonIndex( XmUbMonthDisplayWidget  md,
                   int                     day_number )
{
  /* Code. */

  return( md -> md.first_date_index + day_number - 1 );

} /* DateButtonIndex */


/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static Boolean
  DateDragConvertProc( Widget         w,
                       Atom           *selection,
                       Atom           *target,
                       Atom           *typeRtn,
                       XtPointer      *valueRtn,
                       unsigned long  *lengthRtn,
                       int            *formatRtn,
                       unsigned long  *max_lengthRtn,
                       XtPointer      client_data,
                       XtRequestId    *request_id )
{
  /* Variables. */
  Arg                     args[ 5 ];
  char                    buffer[ 20 ];
  Widget                  button;
  Atom                    compound_text_atom;
  char                    *ctext;
  Atom                    date_transfer_atom;
  int                     day;
  int                     index;
  XmUbMonthDisplayWidget  md;
  int                     month;
  Cardinal                n;
  char                    *passtext;
  TIM_TIME_REF            time;
  Widget                  tmp_widget;
  XmUbDateTransfer        *transfer_data;
  Boolean                 week;
  int                     week_index;
  XmString                xm;
  int                     year;

  /* Code. */

  compound_text_atom = XmInternAtom( XtDisplay( w ), "COMPOUND_TEXT", False );
  date_transfer_atom = XmInternAtom( XtDisplay( w ), XmUbDATE_TRANSFER, 
                                     False );

  /* Refuse incompatible transfer. */
  if( ! ( ( *target == compound_text_atom ) || 
          ( *target == date_transfer_atom ) ) )
    return( False );

  /* The id of the button widget is set in the clientData resource of the
     DragContext widget. */
  n = 0;
  XtSetArg( args[ n ], XmNclientData, &button ); n++;
  XtGetValues( w, args, n );

  /* Find month display widget. */
  tmp_widget = XtParent( button );
  while( ! XtIsSubclass( tmp_widget, xmUbMonthDisplayWidgetClass ) )
    tmp_widget = XtParent( tmp_widget );
  
  md = (XmUbMonthDisplayWidget) tmp_widget;

  if( *target == date_transfer_atom ){

    /* Transfer in special date format. */
    transfer_data = 
      (XmUbDateTransfer*) XtMalloc( sizeof( XmUbDateTransfer ) );

    /* The type of transfer depends on which type of button was pressed. */
    if( button == md -> md.internal_children[ XmUbMD_CHILD_TITLE ] ){
      /* Dragging a month. */
      transfer_data -> type = XmUbTRANSFER_RANGE;

      transfer_data -> year  = md -> md.current_year;
      transfer_data -> month = md -> md.current_month;
      transfer_data -> day   = 1;

      time = TimMakeTime( md -> md.current_year, md -> md.current_month, 1, 
                          0, 0, 0 );

      transfer_data -> end_year  = md -> md.current_year;
      transfer_data -> end_month = md -> md.current_month;
      transfer_data -> end_day   = TimDaysInMonth( time );
      
    } else {

      /* This could be a week number or a date button. Faster to loop through
         the week numbers. */
      week = False;
      if( md -> md.week_numbers ){
        for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ ){
          if( button == md -> md.week_number_buttons[ index ] ){
            week = True;
            week_index = index;
            break;
          }
        }
      }

      if( week ){
        /* Find the first date button for the week. */
        index = ( week_index - 1 ) * 7;

        ButtonDate( md, md -> md.day_number_buttons[ index ], 
                    &year, &month, &day );

        transfer_data -> type  = XmUbTRANSFER_RANGE;
        transfer_data -> year  = year;
        transfer_data -> month = month;
        transfer_data -> day   = day;

        /* End date. */
        index += 6;

        ButtonDate( md, md -> md.day_number_buttons[ index ], 
                    &year, &month, &day );

        transfer_data -> end_year  = year;
        transfer_data -> end_month = month;
        transfer_data -> end_day   = day;

      } else {
        /* Date transfer. */

        /* Get current month and year and construct string with date. */
        ButtonDate( md, button, &year, &month, &day );

        transfer_data -> type  = XmUbTRANSFER_DATE;
        transfer_data -> year  = year;
        transfer_data -> month = month;
        transfer_data -> day   = day;
      }
    }
 
    /* Set up the transfer values. */
    *typeRtn   = date_transfer_atom;
    *valueRtn  = (XtPointer) transfer_data;
    *lengthRtn = sizeof( XmUbDateTransfer );
    *formatRtn = 8;

  } else {
    /* We should just transfer the label string. */

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, &xm ); n++;
    XtGetValues( button, args, n );

    ctext = XmCvtXmStringToCT( xm );

    /* We must copy the string for transfer. */
    passtext = XtMalloc( strlen( ctext ) + 1 );
    memcpy( passtext, ctext, strlen( ctext ) + 1 );

    /* Free what has been allocated here. passtext will be freed by Xt. */
    XmStringFree( xm );
    XtFree( ctext );

    /* Set up the transfer values. */
    *typeRtn   = compound_text_atom;
    *valueRtn  = (XtPointer) passtext;
    *lengthRtn = strlen( passtext );
    *formatRtn = 8;

  }

  return( True );

} /* DateDragConvertProc */

#endif

/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static void
  DateEnterDropSiteCB( Widget                         drag_context,
                       XmUbMonthDisplayWidget         md,
                       XmDropSiteEnterCallbackStruct  *call_data )
{
  /* Variables. */
  Arg       args[ 2 ];
  Cardinal  n;

  /* Code. */

  n = 0;

  if( call_data -> dropSiteStatus == XmVALID_DROP_SITE ){
    XtSetArg( args[ n ], XmNsourcePixmapIcon, md -> md.date_icon ); n++;
    XtSetArg( args[ n ], XmNsourceCursorIcon, md -> md.date_icon ); n++;
    
  } else {
    XtSetArg( args[ n ], XmNsourcePixmapIcon, md -> md.inv_date_icon ); n++;
    XtSetArg( args[ n ], XmNsourceCursorIcon, md -> md.inv_date_icon ); n++;
  }

  XtSetValues( drag_context, args, n );


  return;

} /* DateEnterDropSiteCB */


#endif

/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static void
  DateLeaveDropSiteCB( Widget                         drag_context,
                       XmUbMonthDisplayWidget         md,
                       XmDropSiteLeaveCallbackStruct  *call_data )
{
  /* Variables. */
  Arg       args[ 2 ];
  Cardinal  n;

  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNsourcePixmapIcon, md -> md.inv_date_icon ); n++;
  XtSetArg( args[ n ], XmNsourceCursorIcon, md -> md.inv_date_icon ); n++;
  XtSetValues( drag_context, args, n );


  return;

} /* DateLeaveDropSiteCB */

#endif

/*----------------------------------------------------------------------*/

static void
  DayNumberButtonActivatedCB( Widget                      pb,
                              XmUbMonthDisplayWidget      md,
                              XmPushButtonCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbMonthDisplayCallbackStruct  cb;
  int                             day_number;
  int                             month_number;
  int                             year_number;

  /* Code. */

  /* Get the day number for the button. */
  ButtonDate( md, pb, &year_number, &month_number, &day_number );

  cb.selected_year  = year_number;
  cb.selected_month = month_number;
  cb.selected_day   = day_number;

  /* Invoke the callbacks. */

  cb.reason = XmUbCR_DATE_SELECTED;
  cb.event  = call_data -> event;
  cb.child  = pb;
  cb.range  = False;

  XtCallCallbackList( (Widget) md, md -> md.date_selected_callback,
                      (XtPointer) &cb );


  return;

} /* DayNumberButtonActivatedCB */


/*----------------------------------------------------------------------*/

static void
  DeleteChild( Widget  widget )
{

  /* Variables. */
  int                     index;
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) XtParent( widget );

  /* Clear the internal reference. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( md -> md.internal_children[ index ] == widget ){
      md -> md.internal_children[ index ] = NULL;
      break;
    }
  }

  /* Perform the actual operation */
  (* ( (CompositeWidgetClass) (xmUbMonthDisplayWidgetClass ->
     core_class.superclass) ) -> composite_class.delete_child ) ( widget );


  return;

} /* DeleteChild */


/*----------------------------------------------------------------------*/

static void 
  Destroy( Widget   widget )
{
  /* Variables. */
  XmUbMonthDisplayWidget  md;

  /* Code. */
  md = (XmUbMonthDisplayWidget) widget;

  /* Free XmStrings. */
  if( md -> md.week_number_label != NULL )
    XmStringFree( md -> md.week_number_label );
  if( md -> md.month_label != NULL )
    XmStringFree( md -> md.month_label );
  if( md -> md.year_label != NULL )
    XmStringFree( md -> md.year_label );

  /* Remove callbacks. */
  XtRemoveAllCallbacks( widget, XmUbNmdiDateSelectedCallback );
  XtRemoveAllCallbacks( widget, XmUbNmdiNewMonthCallback );
  XtRemoveAllCallbacks( widget, XmUbNmdiTitleCallback );
  XtRemoveAllCallbacks( widget, XmUbNmdiUpdateDayCallback );

  /* Remove pixmaps if created. */
  if( md -> md.date_icon_bm != None )
    XFreePixmap( XtDisplay( widget ), md -> md.date_icon_bm );
  if( md -> md.date_icon_mask != None )
    XFreePixmap( XtDisplay( widget ), md -> md.date_icon_mask );
  if( md -> md.inv_date_icon_bm != None )
    XFreePixmap( XtDisplay( widget ), md -> md.inv_date_icon_bm );
  if( md -> md.inv_date_icon_mask != None )
    XFreePixmap( XtDisplay( widget ), md -> md.inv_date_icon_mask );


  return;

} /* Destroy */


/*----------------------------------------------------------------------*/

static void
  DoLayout( XmUbMonthDisplayWidget  md,
            Widget                  initiator,
            XtWidgetGeometry        *request,
            KidDimensionRec         sizes[] )
{
  /* Variables. */
  int                arrow_index;
  Position           arrow_y_pos;
  int                index;
  Widget             kid;
  Dimension          label_offset;
  Position           rc_lowest_pos;
  Dimension          rc_offset;
  Dimension          rc_width;

  /* Code. */

  /* Let the children keep their sizes. They will be clipped if window is too
     small. */
  /* Let the week number rc be part of the calculations. Its width and
     height will be zero anyway if it is not enabled. */

  /* Y positions. */
  sizes[ XmUbMD_CHILD_TITLE ].y       = (Position) md -> md.margin_height;
  sizes[ XmUbMD_CHILD_ROW_COLUMN ].y  = sizes[ XmUbMD_CHILD_TITLE ].y +
    (Position) ( sizes[ XmUbMD_CHILD_TITLE ].height + md -> md.margin_height );
  sizes[ XmUbMD_CHILD_WEEK_RC ].y     = sizes[ XmUbMD_CHILD_ROW_COLUMN ].y;

  rc_lowest_pos = MAX( sizes[ XmUbMD_CHILD_ROW_COLUMN ].y +
                       (Position) sizes[ XmUbMD_CHILD_ROW_COLUMN ].height,
                       sizes[ XmUbMD_CHILD_WEEK_RC ].y +
                       (Position) sizes[ XmUbMD_CHILD_WEEK_RC ].height );

  sizes[ XmUbMD_CHILD_SEPARATOR ].y = rc_lowest_pos;

  arrow_y_pos = sizes[ XmUbMD_CHILD_SEPARATOR ].y +
    (Position) sizes[ XmUbMD_CHILD_SEPARATOR ].height +
    (Position) md -> md.arrow_spacing;

  if( md -> md.month_arrows )
    sizes[ XmUbMD_CHILD_MONTH_ARROW ].y = arrow_y_pos;

  if( md -> md.year_arrows )
    sizes[ XmUbMD_CHILD_YEAR_ARROW ].y = arrow_y_pos;

  /* X positions. */

  /* The separator should extend to the edge of the widget. */
  sizes[ XmUbMD_CHILD_SEPARATOR ].x     = 0;
  sizes[ XmUbMD_CHILD_SEPARATOR ].width = md -> core.width;

  rc_width = sizes[ XmUbMD_CHILD_ROW_COLUMN ].width +
             sizes[ XmUbMD_CHILD_WEEK_RC ].width;

  if( md -> md.title_alignment == XmALIGNMENT_CENTER ){

    /* The label and row column should be centered. */
    if( sizes[ XmUbMD_CHILD_TITLE ].width > rc_width ){
      label_offset = 0;
      rc_offset    = ( sizes[ XmUbMD_CHILD_TITLE ].width - rc_width ) / 2;
    } else {
      label_offset = ( rc_width - sizes[ XmUbMD_CHILD_TITLE ].width ) / 2;
      rc_offset    = 0;
    }

    sizes[ XmUbMD_CHILD_TITLE ].x = (Position) ( md -> md.margin_width +
      label_offset );
    sizes[ XmUbMD_CHILD_WEEK_RC ].x = (Position) ( md -> md.margin_width +
      rc_offset );
    sizes[ XmUbMD_CHILD_ROW_COLUMN ].x = sizes[ XmUbMD_CHILD_WEEK_RC ].x +
      (Position) sizes[ XmUbMD_CHILD_WEEK_RC ].width;

  } else {

    /* The row column should be centered in the widget and the title should
       be placed according to its alignment. */
  
    if( rc_width < md -> core. width )
      sizes[ XmUbMD_CHILD_WEEK_RC ].x = (Position) ( md -> core.width -
        rc_width ) / 2;
    else
      sizes[ XmUbMD_CHILD_WEEK_RC ].x = 0;

    sizes[ XmUbMD_CHILD_ROW_COLUMN ].x = sizes[ XmUbMD_CHILD_WEEK_RC ].x +
      (Position) sizes[ XmUbMD_CHILD_WEEK_RC ].width;

    /* Place the label according to its defined alignment. */
    if( md -> md.title_alignment == XmALIGNMENT_BEGINNING )
      sizes[ XmUbMD_CHILD_TITLE ].x = (Position) md -> md.margin_width;
    else {
      if( sizes[ XmUbMD_CHILD_TITLE ].width < 
          ( md -> core.width - md -> md.margin_width ) )
        sizes[ XmUbMD_CHILD_TITLE ].x = (Position) ( md -> core.width -
          md -> md.margin_width - sizes[ XmUbMD_CHILD_TITLE ].width );
      else
        sizes[ XmUbMD_CHILD_TITLE ].x = 0;
    }
  }

  /* If both arrow buttons are enabled, they should be glued to the left 
     and right of the bottom side of the row column. 
     Month to the left, year to the right. */
  /* The Y positions have already been set. */

  if( md -> md.month_arrows && md -> md.year_arrows ){

    sizes[ XmUbMD_CHILD_MONTH_ARROW ].x = sizes[ XmUbMD_CHILD_WEEK_RC ].x;

    if( rc_width > sizes[ XmUbMD_CHILD_YEAR_ARROW ].width )
      sizes[ XmUbMD_CHILD_YEAR_ARROW ].x = sizes[ XmUbMD_CHILD_ROW_COLUMN ].x +
        (Position) sizes[ XmUbMD_CHILD_ROW_COLUMN ].width -
        (Position) sizes[ XmUbMD_CHILD_YEAR_ARROW ].width;
    else
      sizes[ XmUbMD_CHILD_YEAR_ARROW ].x = sizes[ XmUbMD_CHILD_WEEK_RC ].x;

  } else if ( md -> md.month_arrows || md -> md.year_arrows ){
    /* One of the arrows is enabled. Center it. */
    if( md -> md.month_arrows )
      arrow_index = XmUbMD_CHILD_MONTH_ARROW;
    else
      arrow_index = XmUbMD_CHILD_YEAR_ARROW;

    if( sizes[ arrow_index ].width > rc_width + 2 * md -> md.margin_width )
      sizes[ arrow_index ].x = 0;
    else
      sizes[ arrow_index ].x = (Position) ( rc_width + 
        2 * md -> md.margin_width - sizes[ arrow_index ].width ) / 2;
  }
  
  /* Configure the children. */
  /* All positions and dimensions are now in the sizes array. */
  for( index = XmUbMD_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = md -> md.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) )
      XtConfigureWidget( kid, sizes[ index ].x, sizes[ index ].y,
        sizes[ index ].width, sizes[ index ].height, 
        kid -> core.border_width );
  }


  return;

} /* DoLayout */


/*----------------------------------------------------------------------*/

static void
  ExposeMonthDisplay( Widget  widget,
                      XEvent  *event,
                      Region  region )
{
  /* Variables. */
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* We don't need to draw anything, just call the month update callback the
     first time the widget is exposed. */

  if( !md -> md.exposed ){
    CallNewMonthCallbacks( md );
    md -> md.exposed = True;
  }


  return;

} /* ExposeMonthDisplay */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply )
{

  XmUbMonthDisplayWidget  md;
  XtWidgetGeometry        own_request;
  Dimension               old_width, old_height;
  Dimension               pref_height;
  Dimension               pref_width;
  KidDimensionRec         kids_sizes[ NO_INTERNAL_CHILDREN ];
  XtGeometryResult        result;

  /* Code. */

  md = (XmUbMonthDisplayWidget) XtParent( widget );

  /* Find out how big the widget would be if the resize were allowed. */
  GetChildPrefSizes( md, NULL, request, kids_sizes );
  GetOwnPreferredSize( md, kids_sizes, &pref_width, &pref_height );

  /* If no change in dimensions, allow the request. */
  if( ( pref_width == md -> core.width ) && 
      ( pref_height == md -> core.height )){
    DoLayout( md, widget, request, kids_sizes );
    return XtGeometryYes;
  }

  /* We must ask our parent to resize us. */
  own_request.request_mode = CWWidth | CWHeight;
  own_request.width  = pref_width;
  own_request.height = pref_height;

  /* Save dimensions. */
  old_width  = md -> core.width;
  old_height = md -> core.height;

  md -> md.resize_called = False;

  /* We are not interested in any compromise geometry. */
  result = XtMakeGeometryRequest( (Widget) md, &own_request, NULL );

  /* Reset to old dimensions if request not granted. */
  if( result != XtGeometryYes ){
    md -> core.width  = old_width;
    md -> core.height = old_height;

  } else {
    if( !md -> md.resize_called )
      Resize( (Widget) md );
  }

/* !!!!!!!!!!!!!!!!!!!!!! */
  /* Always grant child's request. */
  return XtGeometryYes;

} /* GeometryManager */


/*----------------------------------------------------------------------*/

static void
  GetChildPrefSizes( XmUbMonthDisplayWidget  md,
                     Widget                  initiator,
                     XtWidgetGeometry        *request,
                     KidDimensionRec         sizes[] )
{
  /* Variables. */
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;

  /* Code. */

  /* Initialize. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    sizes[ index ].width  = 0;
    sizes[ index ].height = 0;
    sizes[ index ].x      = 0;
    sizes[ index ].y      = 0;
  }

  /* Get the preferred sizes for the children. */
  for( index = XmUbMD_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = md -> md.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){

      KidsPreferredGeometry( kid, initiator, request, &desired );

      sizes[ index ].width  = desired.width;
      sizes[ index ].height = desired.height;

    }

  } /* for */

  
  return;

} /* GetChildPrefSizes */


/*----------------------------------------------------------------------*/

static void
  GetOwnPreferredSize( XmUbMonthDisplayWidget  md,
                       KidDimensionRec         sizes[],
                       Dimension               *pref_width,
                       Dimension               *pref_height )
{
  /* Code. */

  *pref_height = sizes[ XmUbMD_CHILD_TITLE ].height +
                 MAX( sizes[ XmUbMD_CHILD_ROW_COLUMN ].height,
                      sizes[ XmUbMD_CHILD_WEEK_RC ].height ) +
                 MAX( sizes[ XmUbMD_CHILD_YEAR_ARROW ].height,
                      sizes[ XmUbMD_CHILD_MONTH_ARROW ].height ) +
                 2 * md -> md.margin_height + md -> md.label_spacing +
                 ( ( md -> md.month_arrows || md -> md.year_arrows ) ?
                   ( sizes[ XmUbMD_CHILD_SEPARATOR ].height +
                     md -> md.arrow_spacing ) : 0 );

  *pref_width = MAX( MAX( sizes[ XmUbMD_CHILD_TITLE ].width,
                          sizes[ XmUbMD_CHILD_ROW_COLUMN ].width + 
                          sizes[ XmUbMD_CHILD_WEEK_RC ].width ),
                     MAX( sizes[ XmUbMD_CHILD_YEAR_ARROW ].width,
                          sizes[ XmUbMD_CHILD_MONTH_ARROW ].width ) ) +
                2 * md -> md.margin_width;


  return;

} /* GetOwnPreferredSize */


/*----------------------------------------------------------------------*/

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args )
{
  /* Variables. */
  int                     index;
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) w;

  /* Copy the XmStrings. */
  for( index = 0; index < *num_args; index ++ ){

    if( strcmp( args[ index ].name, XmUbNmdiMonthLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( md -> md.month_label );

    } else if( strcmp( args[ index ].name, XmUbNmdiWeekLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( md -> md.week_number_label );

    } else if( strcmp( args[ index ].name, XmUbNmdiYearLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( md -> md.year_label );
    } 

  }


  return;

} /* GetValuesHook */


/*----------------------------------------------------------------------*/

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args )
{
  /* Variables. */
  int                     index;
  KidDimensionRec         kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbMonthDisplayWidget  new;
  Dimension               pref_height;
  Dimension               pref_width;
  XmString                xm;

  /* Code. */

  new = (XmUbMonthDisplayWidget) tnew;

  /* Initialize private fields. */
  /* Nullify all widget ids. */
  for( index = XmUbMD_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ )
    new -> md.internal_children[ index ] = NULL;
  for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ )
    new -> md.day_name_buttons[ index ] = NULL;
  for( index = 0; index < NO_DAY_BUTTONS; index ++ )
    new -> md.day_number_buttons[ index ] = NULL;
  for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ )
    new -> md.week_number_buttons[ index ] = NULL;
  
  new -> md.date_icon          = NULL;
  new -> md.date_icon_bm       = None;
  new -> md.date_icon_mask     = None;
  new -> md.inv_date_icon      = NULL;
  new -> md.inv_date_icon_bm   = None;
  new -> md.inv_date_icon_mask = None;

  new -> md.exposed = False;

  /* Start size if none supplied. */
  if( new -> core.width == 0 )
    new -> core.width = 300;
  else
    new -> md.recompute_width = False;

  if( new -> core.height == 0 )
    new -> core.height = 100;
  else
    new -> md.recompute_height = False;

  /* Copy XmStrings. */
  if( new -> md.month_label != NULL ){
    xm = XmStringCopy( new -> md.month_label );
    new -> md.month_label = xm;
  }
  if( new -> md.year_label != NULL ){
    xm = XmStringCopy( new -> md.year_label );
    new -> md.year_label = xm;
  }
  if( new -> md.week_number_label != NULL ){
    xm = XmStringCopy( new -> md.week_number_label );
    new -> md.week_number_label = xm;
  }


  /* Set defaults for current year and month. */
  if( new -> md.current_month == 0 )
    new -> md.current_month = 
      TimIndexOfMonth( TimLocalTime( TimMakeTimeNow() ) );

  if( new -> md.current_year == 0 )
    new -> md.current_year = 
      TimIndexOfYear( TimLocalTime( TimMakeTimeNow() ) );

  new -> md.first_date_index = 0;


  /* Create the internal widgets. */
  new -> md.internal_widgets_created = False;

  CreateInternalWidgets( new );

  new -> md.internal_widgets_created = True;

  /* Get the preferred dimensions of the widget. */
  GetChildPrefSizes( new, NULL, NULL, kids_sizes );
  GetOwnPreferredSize( new, kids_sizes, &pref_width, &pref_height );

  /* Set the desired size in the height and width fields. */
  new -> core.width  = pref_width;
  new -> core.height = pref_height;


  return;

} /* Initialize */


/*----------------------------------------------------------------------*/

static void
  InsertChild( Widget  widget )
{

  /* Variables. */
  Cardinal                num_params;
  String                  params[ 1 ];
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) XtParent( widget );

  /* We do not allow the application to create children. */
  if( md -> md.internal_widgets_created ){

    params[ 0 ] = XtClass( (Widget) md ) -> core_class.class_name;
    num_params  = 1;
    XtAppErrorMsg( XtWidgetToApplicationContext( widget ),
                   "childError", "number", "WidgetError",
                   "Applications cannot add children to %s widgets.",
                   params, &num_params );
  }

  /* This is an internal child. Adding it is OK. */
  (* ( (CompositeWidgetClass) (xmUbMonthDisplayWidgetClass ->
     core_class.superclass) ) -> composite_class.insert_child ) ( widget );


  return;

} /* InsertChild */


/*----------------------------------------------------------------------*/

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired )
{

  /* Code. */

  if( ( kid == initiator ) && ( request != NULL ) ){
    /* The initiator should not be queried. */
    if( request -> request_mode & CWWidth )
      desired -> width = request -> width;
    else
      desired -> width = (Dimension) kid -> core.width;

    if( request -> request_mode & CWHeight )
      desired -> height = request -> height;
    else
      desired -> height = (Dimension) kid -> core.height;

  } else
    (void) XtQueryGeometry( kid, NULL, desired );


  return;

} /* KidsPreferredGeometry */


/*----------------------------------------------------------------------*/

static void
  MonthScrolledCB( Widget                        pb,
                   XmUbMonthDisplayWidget        md,
                   XmUbArrowLabelCallbackStruct  *call_data )
{
  /* Variables. */
  TIM_TIME_REF  time;

  /* Code. */

  switch( call_data -> reason ){

    case XmUbCR_BACK_ARROW_ACTIVATED:
      md -> md.current_month --;
      break;

    case XmUbCR_FORWARD_ARROW_ACTIVATED:
      md -> md.current_month ++;
      break;

    default:
      return;
  }

  if( md -> md.current_month > 12 ){
    md -> md.current_month = 1;
    md -> md.current_year ++;
  
  } else if(  md -> md.current_month < 1 ){
    md -> md.current_month = 12;
    md -> md.current_year --;

    if( md -> md.current_year < 1970 ){
      md -> md.current_year  = 1970;
      md -> md.current_month = 1;
    }
  }


  UpdateNewMonth( md );
  CallNewMonthCallbacks( md );


  return;

} /* MonthScrolledCB */


/*----------------------------------------------------------------------*/

static void
  Resize( Widget    widget )
{
  /* Variables. */
  KidDimensionRec         kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  md -> md.resize_called = True;

  /* We have to get the preferred size of the children before we organize
     the layout. */
  GetChildPrefSizes( md, NULL, NULL, kids_sizes );
  DoLayout( md, NULL, NULL, kids_sizes );


  return;

} /* Resize */


/*----------------------------------------------------------------------*/

static Boolean
  ResizeIfNeeded( XmUbMonthDisplayWidget  md,
                  KidDimensionRec         sizes[] )
{

  /* Variables. */
  Boolean           layout_done;
  Dimension         pref_height;
  Dimension         pref_width;
  XtWidgetGeometry  request;
  XtGeometryResult  result;

  /* Code. */

  /* Initialize. */
  layout_done = False;

  /* Get the preferred dimensions of the widget. */
  GetChildPrefSizes( md, NULL, NULL, sizes );
  GetOwnPreferredSize( md, sizes, &pref_width, &pref_height );

  /* If we want the same dimensions, no resizing is needed. */
  if(( pref_width  == md -> core.width ) &&
     ( pref_height == md -> core.height ))
    return False;

  /* Dimensions are different. Try to resize. */
  request.request_mode = CWWidth | CWHeight;

  request.width  = pref_width;
  request.height = pref_height;

  md -> md.resize_called = False;

  do {

    result = XtMakeGeometryRequest( (Widget) md, &request, &request );

  } while( result == XtGeometryAlmost );

  if( result == XtGeometryNo )
    return False;

  /* Resize done. Core fields have already been updated. */
  return( md -> md.resize_called );

} /* ResizeIfNeeded */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  KidDimensionRec         kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbMonthDisplayWidget  md;
  Dimension               pref_height;
  Dimension               pref_width;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* Get dimensions that we *really* want. */
  GetChildPrefSizes( md, NULL, NULL, kids_sizes );
  GetOwnPreferredSize( md, kids_sizes, &pref_width, &pref_height );

  answer -> request_mode = CWWidth | CWHeight;
  answer -> width  = pref_width;
  answer -> height = pref_height;


  if( proposed == NULL ){
    /* This is a query for the requested geometry. */

    if(( answer -> height == (int) md -> core.height ) &&
       ( answer -> width  == (int) md -> core.width ))
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
  }

  /* The parent supplied a geometry suggestion. */
  if(( answer -> height == proposed -> height ) &&
     ( answer -> width  == proposed -> width ))
    return XtGeometryYes;

  if( ( proposed -> height <= 1 ) ||
      ( proposed -> width  <= 1 ) )
    /* That's too small ! */
    return XtGeometryNo;


  /* Only a compromise left. */
  return XtGeometryAlmost;

} /* QueryGeometry */


/*----------------------------------------------------------------------*/

static void
  SetDayNames( XmUbMonthDisplayWidget  md )
{
  /* Variables. */
  Arg           args[ 2 ];
  char          day_name_buffer[ LABEL_BUFFER_LENGTH ];
  int           first_day_index;
  int           index;
  char          label_buffer[ LABEL_BUFFER_LENGTH ];
  Cardinal      n;
  XmString      old_xm;
  TIM_TIME_REF  time;
  XmString      xm;

  /* Code. */

  /* The week title. */
  if( md -> md.week_numbers ){

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, &old_xm ); n++;
    XtGetValues( md -> md.week_number_buttons[ 0 ], args, n );
    
    if( ! XmStringCompare( old_xm, md -> md.week_number_label ) ){
      n = 0;
      XtSetArg( args[ n ], XmNlabelString, md -> md.week_number_label ); n++;
      XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
      XtSetValues( md -> md.week_number_buttons[ 0 ], args, n );
    }
  }


  /* The day names. */

  first_day_index = TimIndexOfFirstDayInWeek();

  /* We need day names from the locale, so create a real time we can 
     play with. */

  time = TimMakeTimeNow();

  /* first_day_index     : 0 = Sunday, 1 = Monday. */
  /* TimIndexOfDayInWeek : first day in week = 1. */

  /* Step forwards until we are on the first day of the week. */
  while( TimIndexOfDayInWeek( time ) != 1 )
    TimAddDays( &time, 1 );

  /* We are on the first day of a week. */
  for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ ){
    
    TimFormatStrTime( time, "%a", label_buffer, LABEL_BUFFER_LENGTH - 1 );

    if( md -> md.day_name_characters > 0 ){
      strncpy( day_name_buffer, label_buffer, 
               MIN( md -> md.day_name_characters, LABEL_BUFFER_LENGTH ) );
      day_name_buffer[ 
        MIN( md -> md.day_name_characters, LABEL_BUFFER_LENGTH ) ] = '\0';
    } else
      strcpy( day_name_buffer, label_buffer );

    xm = XmStringCreateLtoR( day_name_buffer, XmSTRING_DEFAULT_CHARSET );

    /* Get old string and compare to see if we need to change. */
    n = 0;
    XtSetArg( args[ n ], XmNlabelString, &old_xm ); n++;
    XtGetValues( md -> md.day_name_buttons[ index ], args, n );

    if( ! XmStringCompare( old_xm, xm ) ){
      n = 0;
      XtSetArg( args[ n ], XmNlabelString, xm ); n++;
      XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
      XtSetValues( md -> md.day_name_buttons[ index ], args, n );
    }

    XmStringFree( xm );
    
    /* Next day. */
    TimAddDays( &time, 1 );

  }


  return;

} /* SetDayNames */


/*----------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args )
{
#ifdef Differs
#undef Differs
#endif

#define Differs( field )  ( curW -> field != newW -> field )

  /* Variables. */
  XmUbMonthDisplayWidget  curW;
  Boolean                 do_layout = False;
  Boolean                 fill;
  int                     index;
  Arg                     local_args[ 3 ];
  Cardinal                n;
  Boolean                 new_fill;
  XmUbMonthDisplayWidget  newW;
  Boolean                 redisplay = False;
  Cardinal                saved_n;
  Boolean                 update_month = False;
  Boolean                 visual_changed = False;
  XmString                xm;

  /* Code. */

  curW = (XmUbMonthDisplayWidget) current;
  newW = (XmUbMonthDisplayWidget) new;

  /* Width and height. */
  /* Resizing is handled higher up. */

  if( Differs( md.recompute_width ) && newW -> md.recompute_width )
    visual_changed = True;
  else if( Differs( core.width ) )
    newW -> md.recompute_width = False;

  if( Differs( md.recompute_height ) && newW -> md.recompute_height )
    visual_changed = True;
  else if( Differs( core.height ) )
    newW -> md.recompute_height = False;

  /* Margins and spacings. */
  if( Differs( md.margin_width )  ||
      Differs( md.margin_height ) ||
      Differs( md.label_spacing ) ||
      Differs( md.arrow_spacing ) )
    visual_changed = True;

  /* These resources may only be set at creation. */
  if( Differs( md.enable_separator ) ){
    WarningNoResourceChange( newW, "XmUbNmdiEnableSeparator" );
    newW -> md.enable_separator = curW -> md.enable_separator;
  }
  if( Differs( md.month_arrows ) ){
    WarningNoResourceChange( newW, "XmUbNmdiMonthArrows" );
    newW -> md.month_arrows = curW -> md.month_arrows;
  }
  if( Differs( md.year_arrows ) ){
    WarningNoResourceChange( newW, "XmUbNmdiYearArrows" );
    newW -> md.year_arrows = curW -> md.year_arrows;
  }
  if( Differs( md.week_numbers ) ){
    WarningNoResourceChange( newW, "XmUbNmdiWeekNumbers" );
    newW -> md.week_numbers = curW -> md.week_numbers;
  }
  if( Differs( md.day_name_characters ) ){
    WarningNoResourceChange( newW, "XmUbNmdiDayChars" );
    newW -> md.day_name_characters = curW -> md.day_name_characters;
  }
  if( Differs( md.week_number_label ) ){
    WarningNoResourceChange( newW, "XmUbNmdiWeekLabel" );
    newW -> md.week_number_label = curW -> md.week_number_label;
  }

  /* These resources affect child widgets. */
  if( Differs( md.use_default_title_callback ) ||
      Differs( md.title_callback ) ){

    /* We may have to change the resources for the title bar. */
    n = 0;
    XtSetArg( local_args[ n ], XmNfillOnArm, &fill ); n++;
    XtGetValues( newW -> md.internal_children[ XmUbMD_CHILD_TITLE ], 
                 local_args, n );

    new_fill = newW -> md.use_default_title_callback || 
               ( newW -> md.title_callback != NULL );

    if( fill != new_fill ){
      n = 0;
      XtSetArg( local_args[ n ], XmNfillOnArm, new_fill ); n++;
      XtSetValues( newW -> md.internal_children[ XmUbMD_CHILD_TITLE ], 
                   local_args, n );
    }
  }

  if( Differs( md.week_number_label ) ){

    if( curW -> md.week_number_label != NULL )
      XmStringFree( curW -> md.week_number_label );

    if( newW -> md.week_number_label != NULL ){
      xm = XmStringCopy( newW -> md.week_number_label );
      newW -> md.week_number_label = xm;
    }

    update_month = True;
  }

  if( Differs( md.month_label )     ||
      Differs( md.year_label )      ||
      Differs( manager.foreground ) ||
      Differs( core.background_pixel ) ){

    n = 0;

    if( Differs( manager.foreground ) ){
      XtSetArg( local_args[ n ], XmNforeground, 
                                 newW -> manager.foreground ); n++;
    }
    if( Differs( core.background_pixel ) ){
      XtSetArg( local_args[ n ], XmNbackground, 
                                 newW -> core.background_pixel ); n++;

      /* The background was changed. We must recalculate shadow colors. */
      XmGetColors( XtScreen( (Widget) newW ), newW -> core.colormap,
                   newW -> core.background_pixel, NULL,
                   & newW -> md.pushb_default_topshadow,
                   & newW -> md.pushb_default_bottomshadow, NULL );
    }

    /* The pushbuttons are only affected by the change in colors. */
    if( n != 0 ){
      XtSetValues( newW -> md.internal_children[ XmUbMD_CHILD_TITLE ],
                   local_args, n );
      if( newW -> md.week_numbers )
        XtSetValues( newW -> md.week_number_buttons[ 0 ], local_args, n );

      for( index = 0; index < NO_DAY_NAME_BUTTONS; index ++ )
        XtSetValues( newW -> md.day_name_buttons[ index ], local_args, n );

      update_month = True;
    }

    saved_n = n;

    if( Differs( md.month_label ) ){
      if( curW -> md.month_label != NULL )
        XmStringFree( curW -> md.month_label );

      if( newW -> md.month_label != NULL ){
        xm = XmStringCopy( newW -> md.month_label );
        newW -> md.month_label = xm;
      }

      XtSetArg( local_args[ n ], XmNlabelString, newW -> md.month_label ); n++;
    }
    if( n != 0 )
      XtSetValues( newW -> md.internal_children[ XmUbMD_CHILD_MONTH_ARROW ],
                   local_args, n );

    n = saved_n;
    if( Differs( md.year_label ) ){

      if( curW -> md.year_label != NULL )
        XmStringFree( curW -> md.year_label );

      if( newW -> md.year_label != NULL ){
        xm = XmStringCopy( newW -> md.year_label );
        newW -> md.year_label = xm;
      }

      XtSetArg( local_args[ n ], XmNlabelString, newW -> md.year_label ); n++;
    }
    if( n != 0 )
      XtSetValues( newW -> md.internal_children[ XmUbMD_CHILD_YEAR_ARROW ],
                   local_args, n );

  }


  if( Differs( md.current_year )           ||
      Differs( md.current_month )          ||
      Differs( md.fill_out_week )          || 
      Differs( md.weekday_foreground )     ||
      Differs( md.sun_foreground )         ||
      Differs( md.sat_foreground )         ||
      Differs( md.today_foreground )       ||
      Differs( md.other_month_foreground ) ||
      Differs( md.week_number_foreground ) )
    update_month = True;

  if( Differs( md.title_alignment ) )
    do_layout = True;


  if( update_month ){
    UpdateNewMonth( newW );
    CallNewMonthCallbacks( newW );
  }

  if( visual_changed || do_layout ){

    KidDimensionRec  kids_sizes[ NO_INTERNAL_CHILDREN ];
    Boolean          resized;

    /* Code. */

    if( visual_changed )
      resized = ResizeIfNeeded( newW, kids_sizes );
    else
      resized = False;

    /* If the widget was resized, the layout has already been done. */
    if( !resized )
      DoLayout( newW, NULL, NULL, kids_sizes );

  }

  return( redisplay || visual_changed || do_layout || update_month );

#undef Differs

} /* SetValues */


/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static void
  StartDateDrag( Widget    w,
                 XEvent    *event,
                 String    *params,
                 Cardinal  num_params )
{
  /* Variables. */
  Arg                     args[ 8 ];
  Atom                    compound_text_atom;
  Atom                    date_transfer_atom;
  Widget                  drag_context;
  Atom                    exportList[ 2 ];
  XmUbMonthDisplayWidget  md;
  Cardinal                n;
  Widget                  tmp_widget;

  /* Code. */

  compound_text_atom = XmInternAtom( XtDisplay( w ), "COMPOUND_TEXT", False );
  date_transfer_atom = XmInternAtom( XtDisplay( w ), XmUbDATE_TRANSFER, 
                                     False );
  
  /* Establish the list of valid target types. */
  exportList[ 0 ] = date_transfer_atom;
  exportList[ 1 ] = compound_text_atom;

  tmp_widget = XtParent( w );
  while( ! XtIsSubclass( tmp_widget, xmUbMonthDisplayWidgetClass ) )
    tmp_widget = XtParent( tmp_widget );
  md = (XmUbMonthDisplayWidget) tmp_widget;

  /* Create the drag icon if it doesn't exist. */
  md -> md.internal_widgets_created = False;

  if( md -> md.date_icon == NULL ){

    if( md -> md.date_icon_bm == None )
      md -> md.date_icon_bm = XCreateBitmapFromData( 
        XtDisplay( w ), XtWindow( w ), 
        (char*) DateDragIcon_bits, DateDragIcon_width, DateDragIcon_height );

    if( md -> md.date_icon_mask == None )
      md -> md.date_icon_mask = XCreateBitmapFromData( 
        XtDisplay( w ), XtWindow( w ), 
        (char*) DateDragMask_bits, DateDragMask_width, DateDragMask_height );

    n = 0;
    XtSetArg( args[ n ], XmNmask, md -> md.date_icon_mask ); n++;
    XtSetArg( args[ n ], XmNpixmap, md -> md.date_icon_bm ); n++;
    XtSetArg( args[ n ], XmNdepth, 1 ); n++;
    XtSetArg( args[ n ], XmNwidth, DateDragIcon_width ); n++;
    XtSetArg( args[ n ], XmNheight, DateDragIcon_height ); n++;
    md -> md.date_icon = XmCreateDragIcon( (Widget) md, "dragIcon", args, n );
  }

  if( md -> md.inv_date_icon == NULL ){

    if( md -> md.inv_date_icon_bm == None )
      md -> md.inv_date_icon_bm = XCreateBitmapFromData( 
        XtDisplay( w ), XtWindow( w ), 
        (char*) DateDragInvIcon_bits, DateDragInvIcon_width,
        DateDragInvIcon_height );

    if( md -> md.inv_date_icon_mask == None )
      md -> md.inv_date_icon_mask = XCreateBitmapFromData( 
        XtDisplay( w ), XtWindow( w ), 
        (char*) DateDragInvIconMask_bits, DateDragInvIconMask_width, 
        DateDragInvIconMask_height );

    n = 0;
    XtSetArg( args[ n ], XmNmask, md -> md.inv_date_icon_mask ); n++;
    XtSetArg( args[ n ], XmNpixmap, md -> md.inv_date_icon_bm ); n++;
    XtSetArg( args[ n ], XmNdepth, 1 ); n++;
    XtSetArg( args[ n ], XmNwidth, DateDragInvIcon_width ); n++;
    XtSetArg( args[ n ], XmNheight, DateDragInvIcon_height ); n++;
    md -> md.inv_date_icon = XmCreateDragIcon( (Widget) md, "invDragIcon", 
                                               args, n );
  }

  md -> md.internal_widgets_created = True;

  n = 0;
  XtSetArg( args[ n ], XmNexportTargets, exportList ); n++;
  XtSetArg( args[ n ], XmNnumExportTargets, 2 ); n++;
  XtSetArg( args[ n ], XmNdragOperations, XmDROP_COPY ); n++;
  XtSetArg( args[ n ], XmNconvertProc, DateDragConvertProc ); n++;
  XtSetArg( args[ n ], XmNclientData, (XtPointer) w ); n++;
  XtSetArg( args[ n ], XmNsourcePixmapIcon, md -> md.inv_date_icon ); n++;
  XtSetArg( args[ n ], XmNsourceCursorIcon, md -> md.inv_date_icon ); n++;

  drag_context = XmDragStart( w, event, args, n );

  /* Handle custom visual effects. */
  XtAddCallback( drag_context, XmNdropSiteEnterCallback,
                 (XtCallbackProc) DateEnterDropSiteCB, (XtPointer) md );
  XtAddCallback( drag_context, XmNdropSiteLeaveCallback,
                 (XtCallbackProc) DateLeaveDropSiteCB, (XtPointer) md );


  return;

} /* StartDateDrag */

#endif

/*----------------------------------------------------------------------*/

static void
  TitleBarPressedCB( Widget                      pb,
                     XmUbMonthDisplayWidget      md,
                     XmPushButtonCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbMonthDisplayCallbackStruct  cb;
  int                             old_month;
  int                             old_year;
  TIM_TIME_REF                    time;

  /* Code. */

  /* Save current year and month for later. */
  old_month = md -> md.current_month;
  old_year  = md -> md.current_year;

  /* Default action ? */
  if( md -> md.use_default_title_callback ){

    /* Set current year and month. */

    time = TimLocalTime( TimMakeTimeNow() );

    /* Set the current year and month. */
    md -> md.current_month = TimIndexOfMonth( time );
    md -> md.current_year  = TimIndexOfYear( time );
  }

  /* Call user's callbacks. */
  if( md -> md.title_callback != NULL ){

    /* Callbacks are defined. */
    /* Set up the callback structure. */
    cb.reason         = XmUbCR_TITLE_PRESSED;
    cb.event          = call_data -> event;
    cb.selected_month = md -> md.current_month;
    cb.selected_year  = md -> md.current_year;

    /* Call the user callbacks. */
    XtCallCallbackList( (Widget) md, md -> md.title_callback,
                        (XtPointer) &cb );
  }

  /* If the month or year was changed, we should update the display. */
  if( ( old_month != md -> md.current_month ) ||
      ( old_year  != md -> md.current_year ) ){
    UpdateNewMonth( md );
    CallNewMonthCallbacks( md );
  }


  return;

} /* TitleBarPressedCB */


/*----------------------------------------------------------------------*/

static void
  UpdateDayButton( XmUbMonthDisplayWidget  md,
                   int                     index,
                   int                     date,
                   XmUbWhichMonth          which_month )
{
  /* Variables. */
  Arg                             args[ 10 ];
  XmUbMdiUpdateDayCallbackStruct  cb;
  char                            label_buffer[ 20 ];
  Cardinal                        n;
  TIM_TIME_REF                    now;
  int                             old_date;
  Pixel                           old_fg, old_bg, old_top, old_bottom;
  Boolean                         old_mapped;
  TIM_TIME_REF                    time;
  int                             user_data;
  Widget                          w;
  XmString                        xm = NULL;

  /* Code. */

  w = md -> md.day_number_buttons[ index ];

  /* Get the current values. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &old_date ); n++;
  XtSetArg( args[ n ], XmNforeground, &old_fg ); n++;
  XtSetArg( args[ n ], XmNbackground, &old_bg ); n++;
  XtSetArg( args[ n ], XmNtopShadowColor, &old_top ); n++;
  XtSetArg( args[ n ], XmNbottomShadowColor, &old_bottom ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, &old_mapped ); n++;
  XtGetValues( w, args, n );

  /* Fill in the callback structure. Can be used also if no callback. */
  cb.reason     = XmUbCR_UPDATE_DAY;
  cb.event      = NULL;
  cb.day        = date;
  cb.loop_index = index;
  
  switch( which_month ){

    case XmUbCURRENT_MONTH:
      cb.month = md -> md.current_month;
      cb.year  = md -> md.current_year;
      break;

    case XmUbPREVIOUS_MONTH:
      cb.month = md -> md.current_month - 1;
      cb.year  = md -> md.current_year;

      if( cb.month < 1 ){
        cb.month = 12;
        cb.year --;
      }

      break;

    case XmUbNEXT_MONTH:
      cb.month = md -> md.current_month + 1;
      cb.year  = md -> md.current_year;

      if( cb.month > 12 ){
        cb.month = 1;
        cb.year ++;
      }

      break;
  }

  time = TimMakeTime( cb.year, cb.month, cb.day, 0, 0, 0 );

  cb.weekday = TimIndexOfDayInWeek( time );
  if( TimIndexOfFirstDayInWeek() == 0 )
    cb.weekday --;
  if( cb.weekday == 0 )
    cb.weekday = 7;

  cb.day_num_in_year = TimIndexOfDayInYear( time );
  cb.week_number     = TimIndexOfWeek( time );
  cb.foreground      = old_fg;
  cb.background      = old_bg;
  cb.frame           = old_top;
  cb.mapped          = old_mapped;
  cb.use_shadows     = False;

  /* Check if the day displayed is "today". */
  now = TimLocalTime( TimMakeTimeNow() );
  if( ( TimIndexOfDay( time )   == TimIndexOfDay( now ) ) &&
      ( TimIndexOfMonth( time ) == TimIndexOfMonth( now ) ) &&
      ( TimIndexOfYear( time )  == TimIndexOfYear( now ) ) )
    cb.today = True;
  else
    cb.today = False;
  

  /* For dates outside the current month, other_month resource overrides. */
  if( which_month != XmUbCURRENT_MONTH ){

    cb.default_foreground = md -> md.other_month_foreground;
    cb.default_background = md -> core.background_pixel;
    cb.default_frame      = md -> core.background_pixel;
    cb.default_mapped     = md -> md.fill_out_week;

  } else {
    /* This is a date in the current month. */

    if( cb.today )
      cb.default_foreground = md -> md.today_foreground;

    else if( cb.weekday == 7 )
      cb.default_foreground = md -> md.sun_foreground;

    else if( cb.weekday == 6 )
      cb.default_foreground = md -> md.sat_foreground;

    else
      cb.default_foreground = md -> md.weekday_foreground;

    cb.default_mapped = True;
  }

  cb.default_background = md -> core.background_pixel;
  cb.default_frame      = md -> core.background_pixel;
  cb.which_month        = which_month;

  if( md -> md.update_day_callback ){
    /* Let the user determine what should be displayed. */

    XtCallCallbackList( (Widget) md, md -> md.update_day_callback,
                        (XtPointer) &cb );

  } else {
    /* We should display the defaults. */
    cb.foreground = cb.default_foreground;
    cb.background = cb.default_background;
    cb.frame      = cb.default_frame;
    cb.mapped     = cb.default_mapped;
  }

  /* Set the values. */
  n = 0;

  if( cb.foreground != old_fg ){
    XtSetArg( args[ n ], XmNforeground, cb.foreground ); n++;
  }
  if( cb.background != old_bg ){
    XtSetArg( args[ n ], XmNbackground, cb.background ); n++;
  }

  if( ! cb.use_shadows ){
    if( cb.frame != old_top ){
      XtSetArg( args[ n ], XmNtopShadowColor, cb.frame ); n++;
      XtSetArg( args[ n ], XmNbottomShadowColor, cb.frame ); n++;
    }

  } else {
    /* Pushbutton shadows on button. */

    if( old_top != md -> md.pushb_default_topshadow ){
      XtSetArg( args[ n ], XmNtopShadowColor, 
                           md -> md.pushb_default_topshadow ); n++;
    }
    if( old_bottom != md -> md.pushb_default_bottomshadow ){
      XtSetArg( args[ n ], XmNbottomShadowColor, 
                           md -> md.pushb_default_bottomshadow ); n++;
    }
  }

  if( cb.mapped != old_mapped ){
    XtSetArg( args[ n ], XmNmappedWhenManaged, cb.mapped ); n++;
  }

  /* If we are lucky, we don't need to change the text... */
  if( ( old_date % 100 ) != date ){
    sprintf( label_buffer, "%d", date );
    xm = XmStringCreateLtoR( label_buffer, XmSTRING_DEFAULT_CHARSET );

    XtSetArg( args[ n ], XmNlabelString, xm ); n++;

    switch( which_month ){

      case XmUbCURRENT_MONTH:
        user_data = date;
        break;

      case XmUbPREVIOUS_MONTH:
        user_data = date + PREVIOUS_MONTH_OFFSET;
        break;

      case XmUbNEXT_MONTH:
        user_data = date + NEXT_MONTH_OFFSET;
        break;
    }
    
    XtSetArg( args[ n ], XmNuserData, user_data ); n++;
  }

  /* Update the button. */
  if( n > 0 )
    XtSetValues( w, args, n );

  /* Cleanup. */
  if( xm != NULL )
    XmStringFree( xm );


  return;

} /* UpdateDayButton */


/*----------------------------------------------------------------------*/

static void
  UpdateNewMonth( XmUbMonthDisplayWidget  md )
{
  /* Variables. */
  Arg           args[ 8 ];
  int           day_number;
  int           days_in_month;
  int           first_date_index;
  int           index;
  char          label_buffer[ LABEL_BUFFER_LENGTH ];
  Boolean       mapped;
  int           month_number;
  Cardinal      n;
  TIM_TIME_REF  time;
  TIM_TIME_REF  time_of_first_date;
  int           user_data;
  int           year_number;
  XmString      xm;

  /* Code. */

  /* Title text. */

  time_of_first_date = TimMakeTime( md -> md.current_year, 
                                    md -> md.current_month, 1, 0, 0, 0 );

  TimFormatStrTime( time_of_first_date, "%B %Y", label_buffer, 
                    LABEL_BUFFER_LENGTH - 1 );

  xm = XmStringCreateLtoR( label_buffer, XmSTRING_DEFAULT_CHARSET );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xm ); n++;
  XtSetValues( md -> md.internal_children[ XmUbMD_CHILD_TITLE ], args, n );
  XmStringFree( xm );

  /* The day names or the week title may have to be changed. */
  SetDayNames( md );

  /* Set the day numbers. */

  /* Find out on which index we should start. */
  /* first_day_index     : 0 = Sunday, 1 = Monday. */
  /* TimIndexOfDayInWeek : 1 for the first day of the week. */

  first_date_index = TimIndexOfDayInWeek( time_of_first_date ) - 1;

  /* If no days from the previous month are displayed, we should step back a 
     week and display the last week of the previous month. */
  if( ( md -> md.fill_out_week ) && ( first_date_index == 0 ) )
    first_date_index += 7;

  md -> md.first_date_index = first_date_index;

  /* Set the days in the preceding month. */
  time = time_of_first_date;
  TimAddDays( &time, -1 );
  days_in_month = TimDaysInMonth( time );

  for( index = 0, day_number = days_in_month - first_date_index + 1;
       index < first_date_index;
       index ++, day_number ++ )
    UpdateDayButton( md, index, day_number, XmUbPREVIOUS_MONTH );


  /* The current month. */
  time = time_of_first_date;
  days_in_month = TimDaysInMonth( time );

  for( index = first_date_index, day_number = 1;
       day_number <= days_in_month;
       index ++, day_number ++ )
    UpdateDayButton( md, index, day_number, XmUbCURRENT_MONTH );

  /* Fill out with the following month. */
  for( day_number = 1; index < NO_DAY_BUTTONS; index ++, day_number ++ )
    UpdateDayButton( md, index, day_number, XmUbNEXT_MONTH );

  /* Update the week numbers. */
  if( md -> md.week_numbers ){

    /* For every week, display the week number of the first day. */
    /* Converting to the time handles all problems when switching year. */
    for( index = 1; index < NO_WEEK_NUMBER_BUTTONS; index ++ ){

      ButtonDate( md, md -> md.day_number_buttons[ ( index - 1 ) * 7 ], 
                  &year_number, &month_number, &day_number );

      time = TimMakeTime( year_number, month_number, day_number, 0, 0, 0 );

      /* We don't want to display week numbers for empty weeks. */
      if( ! md -> md.fill_out_week && 
          ( month_number != md -> md.current_month ) &&
          ( year_number >= md -> md.current_year ) &&
          ( time > time_of_first_date ) )
        mapped = False;
      else
        mapped = True;

      UpdateWeekButton( md, index, time, mapped );
    }
  }


  return;

} /* UpdateNewMonth */


/*----------------------------------------------------------------------*/

static void
  UpdateWeekButton( XmUbMonthDisplayWidget  md,
                    int                     index,
                    TIM_TIME_REF            time_of_first_day,
                    Boolean                 mapped )
{
  /* Variables. */
  Arg                             args[ 10 ];
  XmUbMdiUpdateDayCallbackStruct  cb;
  char                            label_buffer[ 20 ];
  Cardinal                        n;
  int                             old_date;
  Pixel                           old_fg, old_bg, old_top, old_bottom;
  Boolean                         old_mapped;
  int                             user_data;
  Widget                          w;
  XmString                        xm = NULL;

  /* Code. */

  w = md -> md.week_number_buttons[ index ];

  /* Get the current values. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &user_data ); n++;
  XtSetArg( args[ n ], XmNforeground, &old_fg ); n++;
  XtSetArg( args[ n ], XmNbackground, &old_bg ); n++;
  XtSetArg( args[ n ], XmNtopShadowColor, &old_top ); n++;
  XtSetArg( args[ n ], XmNbottomShadowColor, &old_bottom ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, &old_mapped ); n++;
  XtGetValues( w, args, n );

  /* Fill in the callback structure. Can be used also if no callback. */
  cb.reason      = XmUbCR_UPDATE_WEEK;
  cb.event       = NULL;
  cb.day         = TimIndexOfDay( time_of_first_day );
  cb.month       = TimIndexOfMonth( time_of_first_day );
  cb.year        = TimIndexOfYear( time_of_first_day );
  cb.week_number = TimIndexOfWeek( time_of_first_day );
  cb.foreground  = old_fg;
  cb.background  = old_bg;
  cb.frame       = old_top;
  cb.use_shadows = False;
  cb.mapped      = old_mapped;

  cb.default_foreground = md -> md.week_number_foreground;
  cb.default_background = md -> core.background_pixel;
  cb.default_frame      = md -> core.background_pixel;
  cb.default_mapped     = mapped;
  cb.loop_index         = index - 1;
  
  if( md -> md.update_day_callback ){
    /* Let the user determine what should be displayed. */

    XtCallCallbackList( (Widget) md, md -> md.update_day_callback,
                        (XtPointer) &cb );

  } else {
    /* We should display the defaults. */
    cb.foreground = cb.default_foreground;
    cb.background = cb.default_background;
    cb.frame      = cb.default_frame;
    cb.mapped     = cb.default_mapped;
  }

  /* Set the values. */
  n = 0;

  if( cb.foreground != old_fg ){
    XtSetArg( args[ n ], XmNforeground, cb.foreground ); n++;
  }
  if( cb.background != old_bg ){
    XtSetArg( args[ n ], XmNbackground, cb.background ); n++;
  }

  if( ! cb.use_shadows ){
    if( cb.frame != old_top ){
      XtSetArg( args[ n ], XmNtopShadowColor, cb.frame ); n++;
      XtSetArg( args[ n ], XmNbottomShadowColor, cb.frame ); n++;
    }

  } else {
    /* Pushbutton shadows on button. */

    if( old_top != md -> md.pushb_default_topshadow ){
      XtSetArg( args[ n ], XmNtopShadowColor, 
                           md -> md.pushb_default_topshadow ); n++;
    }
    if( old_bottom != md -> md.pushb_default_bottomshadow ){
      XtSetArg( args[ n ], XmNbottomShadowColor, 
                           md -> md.pushb_default_bottomshadow ); n++;
    }
  }

  /* If we are lucky, we don't need to change the text... */
  if( user_data != cb.week_number ){
    sprintf( label_buffer, "%d", cb.week_number );
    xm = XmStringCreateLtoR( label_buffer, XmSTRING_DEFAULT_CHARSET );

    XtSetArg( args[ n ], XmNlabelString, xm ); n++;
    XtSetArg( args[ n ], XmNuserData, cb.week_number ); n++;
  }

  /* Mapped or not? */
  if( old_mapped != cb.mapped ){
    XtSetArg( args[ n ], XmNmappedWhenManaged, cb.mapped ); n++;
  }

  /* Update the button. */
  if( n > 0 )
    XtSetValues( w, args, n );

  /* Cleanup. */
  if( xm != NULL )
    XmStringFree( xm );


  return;

} /* UpdateWeekButton */


/*----------------------------------------------------------------------*/

static void
  WarningNoResourceChange( XmUbMonthDisplayWidget  md,
                           String                  resource )
{
  /* Variables. */
  Cardinal  num_params;
  String    params[ 2 ];

  /* Code. */

  params[ 0 ] = resource;
  params[ 1 ] = XtClass( md ) -> core_class.class_name;
  num_params  = 2;
  XtAppWarningMsg( XtWidgetToApplicationContext( (Widget) md ),
		   "resourceError", "setValues", "WidgetError",
		   "Resource %s may not be changed in %s widgets.",
		   params, &num_params );

  return;

} /* WarningNoResourceChange */


/*----------------------------------------------------------------------*/

static void
  WeekNumberButtonActivatedCB( Widget                      pb,
                               XmUbMonthDisplayWidget      md,
                               XmPushButtonCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbMonthDisplayCallbackStruct  cb;
  int                             date_index;
  int                             day_number;
  int                             index;
  int                             month_number;
  int                             year_number;

  /* Code. */

  /* Get the index of the week button. */
  for( index = 0; index < NO_WEEK_NUMBER_BUTTONS; index ++ ){
    if( pb == md -> md.week_number_buttons[ index ] )
      break;
  }

  /* The first day of that week. */
  date_index = ( index - 1 ) * 7;

  /* Get the day number for the week start button. */
  ButtonDate( md, md -> md.day_number_buttons[ date_index ], 
              &year_number, &month_number, &day_number );

  cb.selected_year  = year_number;
  cb.selected_month = month_number;
  cb.selected_day   = day_number;

  /* End date. */
  date_index += 6;

  ButtonDate( md, md -> md.day_number_buttons[ date_index ], 
              &year_number, &month_number, &day_number );

  cb.end_year  = year_number;
  cb.end_month = month_number;
  cb.end_day   = day_number;

  /* Invoke the callbacks. */

  cb.reason = XmUbCR_DATE_SELECTED;
  cb.event  = call_data -> event;
  cb.child  = pb;
  cb.range  = True;

  XtCallCallbackList( (Widget) md, md -> md.date_selected_callback,
                      (XtPointer) &cb );


  return;

} /* WeekNumberButtonActivatedCB */


/*----------------------------------------------------------------------*/

Widget  
  XmUbCreateMonthDisplay( Widget    parent,
                          String    name,
                          ArgList   arglist,
                          Cardinal  argcount )
{

  /* Code. */

  return XtCreateWidget( name, xmUbMonthDisplayWidgetClass, 
                         parent, arglist, argcount );

} /* XmUbCreateMonthDisplay */


/*----------------------------------------------------------------------*/

Widget 
  XmUbMonthDisplayGetChild( Widget  widget,
                            int     child )
{
  /* Variables. */
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;


  return( md -> md.internal_children[ child ] );

} /* XmUbMonthDisplayGetChild */


/*----------------------------------------------------------------------*/

void
  XmUbMonthDisplaySetFocusToDay( Widget  widget,
                                 int     day_number )
{
  /* Variables. */
  Widget                  button;
  XmUbMonthDisplayWidget  md;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* Find the widget id of the button. */
  button = DateButton( md, day_number );

  if( button != NULL )
    XmProcessTraversal( button, XmTRAVERSE_CURRENT );


  return;

} /* XmUbMonthDisplaySetFocusToDay */


/*----------------------------------------------------------------------*/

void
  XmUbMonthDisplaySetDateBackground( Widget  widget,
                                     int     day_number,
                                     Pixel   bg )
{
  /* Variables. */
  Arg                     args[ 2 ];
  Widget                  button;
  XmUbMonthDisplayWidget  md;
  Cardinal                n;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* Find the widget id of the button. */
  button = DateButton( md, day_number );

  if( button != NULL ){
    n = 0;
    XtSetArg( args[ n ], XmNbackground, bg ); n++;
    XtSetValues( button, args, n );
  }


  return;

} /* XmUbMonthDisplaySetDateBackground */


/*----------------------------------------------------------------------*/

void
  XmUbMonthDisplaySetDateForeground( Widget  widget,
                                     int     day_number,
                                     Pixel   fg )
{
  /* Variables. */
  Arg                     args[ 2 ];
  Widget                  button;
  XmUbMonthDisplayWidget  md;
  Cardinal                n;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* Find the widget id of the button. */
  button = DateButton( md, day_number );

  if( button != NULL ){
    n = 0;
    XtSetArg( args[ n ], XmNforeground, fg ); n++;
    XtSetValues( button, args, n );
  }


  return;

} /* XmUbMonthDisplaySetDateForeground */


/*----------------------------------------------------------------------*/

void
  XmUbMonthDisplaySetDateFrame( Widget  widget,
                                int     day_number,
                                Pixel   color )
{
  /* Variables. */
  Arg                     args[ 2 ];
  Widget                  button;
  XmUbMonthDisplayWidget  md;
  Cardinal                n;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  /* Find the widget id of the button. */
  button = DateButton( md, day_number );

  if( button != NULL ){
    n = 0;
    XtSetArg( args[ n ], XmNtopShadowColor, color ); n++;
    XtSetArg( args[ n ], XmNbottomShadowColor, color ); n++;
    XtSetValues( button, args, n );
  }


  return;

} /* XmUbMonthDisplaySetDateFrame */


/*----------------------------------------------------------------------*/

void
  XmUbMonthDisplaySetMonth( Widget  widget,
                            int     year,
                            int     month )
{
  /* Variables. */
  Arg                     args[ 2 ];
  XmUbMonthDisplayWidget  md;
  Cardinal                n;

  /* Code. */

  md = (XmUbMonthDisplayWidget) widget;

  if( year != 0 )
    md -> md.current_year  = year;
  if( month != 0 )
    md -> md.current_month = month;


  UpdateNewMonth( md );


  return;

} /* XmUbMonthDisplaySetMonth */


/*----------------------------------------------------------------------*/

static void
  YearScrolledCB( Widget                        pb,
                  XmUbMonthDisplayWidget        md,
                  XmUbArrowLabelCallbackStruct  *call_data )
{
  /* Variables. */
  TIM_TIME_REF  time;

  /* Code. */

  switch( call_data -> reason ){

    case XmUbCR_BACK_ARROW_ACTIVATED:
      if( md -> md.current_year > 1970 )
        md -> md.current_year --;
      else
        md -> md.current_month = 1;
      break;

    case XmUbCR_FORWARD_ARROW_ACTIVATED:
      md -> md.current_year ++;
      break;

    default:
      return;
  }

  UpdateNewMonth( md );
  CallNewMonthCallbacks( md );

  return;

} /* YearScrolledCB */
