/*----------------------------------------------------------------------------
--
--  Module:           xitCreFoDia
--
--  Project:          xit  - X Internal Toolkit
--  System:           xit  - X Internal Toolkit
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manages a form dialog with n action buttons. Enter and Return
--    are used as activate keys (if a default button is defined).
--
--  Filename:         xitCreFoDia.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-20
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitCreFoDia.c, Version: 1.1, Date: 95/02/18 15:10:23";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>

#include "System.h"
#include "xitTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define DEFAULT_TIGHTNESS  5


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Structure to hold the min. sizes of the shell. */
typedef struct {
  Dimension  min_width;
  Dimension  min_height;
} MIN_SIZE, *MIN_SIZE_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  destroySizeDataCB( Widget               widget,
                     MIN_SIZE_REF         size_ref,
                     XmAnyCallbackStruct  *call_data );

static void
  removeCB( Widget     widget,
            Widget     toplevel_widget,
            XtPointer  call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateFormDialog( Widget                parent,
                       char                  *name,
                       int                   default_button_no,
                       int                   tightness,
                       XIT_ACTION_AREA_ITEM  *actions,
                       int                   num_actions )
{

  /* Variables. */
  int         index;
  int         extra_offset = 0;
  char        buffer[ 50 ];
  Arg         args[ 20 ];
  Cardinal    n;
  Dimension   height;
  Dimension   width;
  Dimension   max_height = 0;
  Dimension   max_width  = 0;
  Widget      dialogFo;
  Widget      ho1Sp = NULL;
  Widget      widget;
  Widget      workFo;

  static XIT_PUSH_STRUCT action_bu_def[] = {
    { "", "", "", True, NULL },
  };  


  /* Code. */

  if( tightness < 1 )
    tightness = DEFAULT_TIGHTNESS;


  /* Create a form dialog, our base widget. */
  n = 0;
  XtSetArg( args[ n ], XmNautoUnmanage,   False     ); n++;
  XtSetArg( args[ n ], XmNdeleteResponse, XmDESTROY ); n++;
  XtSetArg( args[ n ], XmNskipAdjust,     True );  n++;

  if( num_actions > 1 ) {
    XtSetArg( args[ n ], XmNfractionBase, tightness * num_actions - 1 ); n++;
  }
  dialogFo = XmCreateFormDialog( parent, name, args, n );


  /* Work area form. */
  sprintf( buffer, "%sFo", name );

  n = 0;
  workFo = XmCreateForm( dialogFo, buffer, args, n );

  XtManageChild( workFo );


  /* Separator. */
  if( num_actions > 0 ) {
    n = 0;
    sprintf( buffer, "%sSp", name );
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
    ho1Sp = XmCreateSeparator( dialogFo, buffer, args, n );

    XtManageChild( ho1Sp );
  }

  /* Attachments. */
  if( num_actions > 0 ) {
    xitAttachWidget( workFo,
                     XmATTACH_FORM, NULL, XmATTACH_FORM,   NULL,
                     XmATTACH_FORM, NULL, XmATTACH_WIDGET, ho1Sp );
    xitAttachWidget( ho1Sp,
                     XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );
  } else {
    xitAttachWidget( workFo,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );
  }

  /* Create the action buttons. */
  for( index = 0; index < num_actions; index++ ) {

    /* Create the action button and set resources. */
    sprintf( buffer, "Bu%d", index + 1 );

    action_bu_def[ 0 ].title = actions[ index ].label;
    action_bu_def[ 0 ].name  = buffer;
    widget = xitCreatePushButton( dialogFo, &action_bu_def[ 0 ] );
    
    n = 0;

    if( index == 0 ) {
      XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
      XtSetArg( args[ n ], XmNleftOffset,     5             ); n++;
    } else {
      XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION ); n++;
    }

    XtSetArg( args[ n ], XmNleftPosition,     tightness * index ); n++;

    XtSetArg( args[ n ], XmNtopAttachment,    XmATTACH_WIDGET   ); n++;
    XtSetArg( args[ n ], XmNtopWidget,        ho1Sp             ); n++;
    XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM     ); n++;

    if( index != num_actions - 1 ) {
      XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION );  n++;
    } else {
      XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
      XtSetArg( args[ n ], XmNrightOffset,     5             ); n++;
    }

    XtSetArg( args[ n ], XmNrightPosition, 
              tightness * index + (tightness - 1) );  n++;

    if( default_button_no - 1 == index ) {
      XtSetArg( args[ n ], XmNshowAsDefault, True );  n++;
      XtSetArg( args[ n ], XmNtopOffset,     5 ); n++;
      XtSetArg( args[ n ], XmNbottomOffset,  5 ); n++;
    } else {
      XtSetArg( args[ n ], XmNtopOffset,     5 + extra_offset ); n++;
      XtSetArg( args[ n ], XmNbottomOffset,  5 + extra_offset ); n++;
    }


    /* Fake button? */
    if( actions[ index ].label == NULL ) {
      XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
    }

    XtSetValues( widget, args, n );

    XtManageChild( widget );


    /* Callback and callback data. */
    if( actions[ index ].callback != NULL )
      XtAddCallback( widget, XmNactivateCallback,
                     actions[ index ].callback, actions[ index ].data );
    else
      XtAddCallback( widget, XmNactivateCallback, 
                     (XtCallbackProc) removeCB, (XtPointer) dialogFo );


    /* Search the max height of the widgets in the action area. */
    n = 0;
    XtSetArg( args[ n ], XmNheight, &height ); n++;
    XtSetArg( args[ n ], XmNwidth,  &width  ); n++;
    XtGetValues( widget, args, n );

    if( height > max_height )
      max_height = height;

    if( width > max_width )
      max_width = width;


    /* Activate the default button widget? */
    if( default_button_no - 1 == index ) {
      n = 0;
      XtSetArg( args[ n ], XmNdefaultButton, widget ); n++;
      XtSetValues( dialogFo, args, n );
    }

  } /* loop */


  /* The separator should be above the action buttons. */
  if( num_actions > 0 ) {
    height = max_height + 2 * 5;

    n = 0;
    XtSetArg( args[ n ], XmNbottomOffset, height ); n++;
    XtSetValues( ho1Sp, args, n );
  }


  /* Save the min width and height. */
  {
    MIN_SIZE_REF  size_ref;

    size_ref = SysNew( MIN_SIZE );

    if( num_actions > 0 ) {
      size_ref -> min_width  = max_width * num_actions + 
                               (num_actions * 3 + 1) * 5;
      size_ref -> min_height = max_height + 2 * 5 + 5;
    } else {
      size_ref -> min_width  = 1;
      size_ref -> min_height = 1;
    }

    /* Save as userdata to the baseFo. */
    n = 0;
    XtSetArg( args[ n ], XmNuserData, (void *) size_ref ); n++;
    XtSetValues( workFo, args, n );

    /* Make sure we can release the userdata. */
    XtAddCallback( workFo, XmNdestroyCallback, 
                   (XtCallbackProc) destroySizeDataCB, (XtPointer) size_ref );
    
  } /* block */


  return( dialogFo );

} /* xitCreateFormDialog */


/*----------------------------------------------------------------------*/

void
  xitSetSizeFormDialog( Widget   toplevel,
                        Boolean  tight )
{

  /* Variables. */
  Boolean       ok;
  int           index;
  char          buffer[ 100 ];
  char          *name;
  void          *user_data;
  Arg           args[ 10 ];
  Cardinal      n;
  Cardinal      num_children;
  Dimension     height;
  Dimension     user_height = 0;
  Dimension     user_width  = 0;
  Dimension     width;
  Position      x;
  Position      y;
  WidgetList    children;
  Widget        tempW;
  MIN_SIZE_REF  min_size_ref;


  /* Code. */

  /* Get the name of the toplevel widget. */
  name = xitGetWidgetName( toplevel );
  if( name == NULL )
    return;

  /* When we realize the toplevel form, all geometries are calculated. */
  XtRealizeWidget( toplevel );

  /* Find the max height and width of the form. */
  sprintf( buffer, "%sFo", name, name );

  tempW = XtNameToWidget( toplevel, buffer );
  if( tempW == NULL )
    return;

  ok = xitGetChildren( tempW, &num_children, &children );
  if( ! ok )
    return;

  for( index = 0; index < num_children; index++ ) {
    n = 0;
    XtSetArg( args[ n ], XmNx,      &x ); n++;
    XtSetArg( args[ n ], XmNy,      &y ); n++;
    XtSetArg( args[ n ], XmNwidth,  &width ); n++;
    XtSetArg( args[ n ], XmNheight, &height ); n++;
    XtGetValues( *(children + index ), args, n );

    if( x + width > user_width )
      user_width = x + width;

    if( y + height > user_height )
      user_height = y + height;
  }

  if( ! tight ) {
    user_height = user_height + 10;
    user_width  = user_width  + 10;
  }

  /* Min sizes saved as user data in widget nameBase. */
  sprintf( buffer, "%sFo", name );

  tempW = XtNameToWidget( toplevel, buffer );
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &user_data ); n++;
  XtGetValues( tempW, args, n );

  min_size_ref = (MIN_SIZE_REF) user_data;


  /* Set the min sizes of the shell. */
  if( user_width > min_size_ref -> min_width )
    width = user_width;
  else
    width = min_size_ref -> min_width;

  height = min_size_ref -> min_height + user_height;

  n = 0;
  XtSetArg( args[ n ], XmNwidth,  width ); n++;
  XtSetArg( args[ n ], XmNheight, height ); n++;
  XtSetValues( toplevel, args, n );


  SysFree( name );


  return;

} /* xitSetSizeFormDialog */


/*----------------------------------------------------------------------*/

static void
  destroySizeDataCB( Widget               widget,
                     MIN_SIZE_REF         size_ref,
                     XmAnyCallbackStruct  *call_data )
{

  /* Code. */

  SysFree( size_ref );


  return;

} /* destroySizeDataCB */


/*----------------------------------------------------------------------*/

static void
  removeCB( Widget     widget,
            Widget     toplevel_widget,
            XtPointer  call_data )
{

  /* Code. */

  XtDestroyWidget( toplevel_widget );

  return;

} /* removeCB */


