/*----------------------------------------------------------------------------
--
--  Module:           xitFieldSelect
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    The field select widget contains a single line Text widget. 
--    Various values can be selected from a pull-down menu to the right
--    of the text field.
--
--  Filename:         xitFieldSel.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-03
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitFieldSel.c, Version: 1.1, Date: 95/02/18 15:10:29";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawnB.h>
#include <Xm/DrawingA.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"

#include "xitTools.h"
#include "xitFieldSel.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef Pixel  *COLOR_TABLE;

/* Record associated with the widget. */
typedef struct {
  Boolean         append;
  Boolean         display_text;
  Boolean         editable;
  Boolean         is_visible;
  XtPointer       client_data;
  int             elements;
  int             max_length;
  int             columns;
  Dimension       button_width;
  Dimension       text_width;
  COLOR_TABLE     bg_list;
  COLOR_TABLE     fg_list;
  Widget          field;
  Widget          menu;
  Widget          menuBr;
  Widget          select_button;
  XmString        select_string;
  XmStringTable   list;
  XtCallbackProc  call_proc;
} SelectRec, *SelectRecPtr;


/* Definitions to prefetch fontlist info before widget is created. */
typedef struct {
  XmFontList  fontlist;
} ResStruct, *ResPtr;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Resource list for items. */
static XtResource  item_resources[] = {
  { XmNbgItems, XmCBgItems, XmRPixel, sizeof( COLOR_TABLE ),
    XtOffset( SelectRecPtr, bg_list ), XmRImmediate, NULL },

  { XmNfgItems, XmCFgItems, XmRPixel, sizeof( COLOR_TABLE ),
    XtOffset( SelectRecPtr, fg_list ), XmRImmediate, NULL },

  { XmNlabelString, XmCLabelString, XmRXmString, sizeof( XmString ),
    XtOffset( SelectRecPtr, select_string ), XmRImmediate, NULL },

  { XmNcolumns, XmCColumns, XmRInt, sizeof( int ),
    XtOffset( SelectRecPtr, columns ), XmRString, "10" },

  { XmNeditable, XmCEditable, XmRBoolean, sizeof( Boolean ),
    XtOffset( SelectRecPtr, editable ), XmRString, "True" },

  { XmNitems, XmCItems, XmRXmStringTable, sizeof( XmStringTable ),
    XtOffset( SelectRecPtr, list ), XmRImmediate, NULL },

  { XmNitemCount, XmCItemCount, XmRInt, sizeof( int ),
    XtOffset( SelectRecPtr, elements ), XmRString, "0" },

  { XmNisVisible, XmCIsVisible, XmRBoolean, sizeof( Boolean ),
    XtOffset( SelectRecPtr, is_visible ), XmRString, "True" },

  { XmNmaxLength, XmCMaxLength, XmRInt, sizeof( int ),
    XtOffset( SelectRecPtr, max_length ), XmRString, "50" },

  { XmNmodeAppend, XmCModeAppend, XmRBoolean, sizeof( Boolean ),
    XtOffset( SelectRecPtr, append ), XmRString, "False" },
};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  callCallback( SelectRecPtr  rec,
                XEvent        *event );

static void
  destroyCB( Widget        widget,
             SelectRecPtr  rec,
             XtPointer     call_data );

static void
  menuSelectCB( Widget                     widget,
                SelectRecPtr               rec,
                XmRowColumnCallbackStruct  *call_data );

static void
  resizeCB( Widget               widget,
            SelectRecPtr         rec,
            XmAnyCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateFieldSelect( Widget          parent,
                        String          name,
                        ArgList         add_args,
                        Cardinal        num_args,
                        XtCallbackProc  proc,
                        XtPointer       client_data )
{

  /* Variables. */
  int            index;
  char           buffer[ 50 ];
  Arg            args[ 10 ];
  ArgList        merge_args;
  Cardinal       n;
  Dimension      select_height;
  Dimension      text_height;
  Dimension      text_width;
  SelectRecPtr   rec;
  Widget         manager;
  Widget         menuPb;
  XmString       xstr;

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "",  NULL, 1, True },
  };


  /* Code. */

  /* Initialize widget record, used in callbacks. */
  rec = SysNew( SelectRec );

  rec -> call_proc    = proc;
  rec -> client_data  = client_data;
  rec -> display_text = True;

  /* Drawing area as manager. */
  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[ n ], XmNuserData, rec ); n++;
  manager = XmCreateDrawingArea( parent, name, args, n );

  XtAddCallback( manager, XmNresizeCallback,
                 (XtCallbackProc) resizeCB, (XtPointer) rec );
  XtAddCallback( manager, XmNdestroyCallback,
                 (XtCallbackProc) destroyCB, (XtPointer) rec );


  /* Initialize the rest of the callback record. */
  XtGetSubresources( manager, (XtPointer) rec, name, FIELD_SELECT_CLASS,
                     item_resources, XtNumber( item_resources ),
                     add_args, num_args );


  /* Text display. */
  sprintf( buffer, "%sTx", name );

  text_buffer[ 0 ].name = buffer;
  rec -> field = xitCreateText( manager, &text_buffer[ 0 ] );

  if( ! rec -> is_visible ) {
    rec -> columns = 1;
    rec -> display_text = False;
  }

  n = 0;
  XtSetArg( args[ n ], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
  XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
  XtSetArg( args[ n ], XmNeditable, rec -> editable ); n++;
  XtSetArg( args[ n ], XmNmaxLength, rec -> max_length ); n++;
  XtSetArg( args[ n ], XmNcolumns, (short) rec -> columns ); n++;

  merge_args = XtMergeArgLists( add_args, num_args, args, n );

  XtSetValues( rec -> field, merge_args, num_args + n );

  if( rec -> display_text )
    XtManageChild( rec -> field );

  SysFree( merge_args );


  /* Create the menubar and menu cascades. */
  sprintf( buffer, "%sBr", name );

  n = 0;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  rec -> menuBr = XmCreateMenuBar( manager, buffer, args, n );

  XtManageChild( rec -> menuBr );


  /* Create the pull-down menu to use. */
  sprintf( buffer, "%sPd", name );

  n = 0;
  rec -> menu = XmCreatePulldownMenu( rec -> menuBr, buffer, args, n );

  XtAddCallback( rec -> menu, XmNentryCallback,
                 (XtCallbackProc) menuSelectCB, (XtPointer) rec );


  /* The cascade button. */
  sprintf( buffer, "%sCa", name );

  if( rec -> select_string == NULL )
    xstr = XmStringCreate( "-", CS );
  else
    xstr = rec -> select_string;

  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNsubMenuId, rec -> menu ); n++;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  rec -> select_button = XmCreateCascadeButton( rec -> menuBr, buffer, 
                                                args, n );
  if( rec -> select_string == NULL )
    XmStringFree( xstr );

  XtManageChild( rec -> select_button );


  /* Assign all the labels in the menu. */
  for( index = 0; index < rec -> elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, rec -> list[ index ] ); n++;

    if( rec -> bg_list != NULL &&
        rec -> fg_list != NULL &&
        rec -> bg_list[ index ] != rec -> fg_list[ index ] ) {
      XtSetArg( args[ n ], XmNbackground, rec -> bg_list[ index ] ); n++;
      XtSetArg( args[ n ], XmNforeground, rec -> fg_list[ index ] ); n++;
    }

    menuPb = XmCreatePushButton( rec -> menu, buffer, args, n );

    XtManageChild( menuPb );
  } /* loop */


  /* Align text and select button. */
  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtGetValues( rec -> field, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &text_width ); n++;
  XtGetValues( rec -> field, args, n );

  if( ! rec -> display_text )
    text_width = 1;

  n = 0;
  XtSetArg( args[ n ], XmNheight, &select_height ); n++;
  XtGetValues( rec -> select_button, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &rec -> button_width ); n++;
  XtGetValues( rec -> menuBr, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNx, text_width + 2 ); n++;
  XtSetArg( args[ n ], XmNy, (text_height - select_height) / 2 - 2 ); n++;
  XtSetValues( rec -> menuBr, args, n );


  return( manager );

} /* xitCreateFieldSelect */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectEditable( Widget   widget,
                          Boolean  editable )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  XmTextSetEditable( rec -> field, editable );


  return;

} /* xitFieldSelectEditable */


/*----------------------------------------------------------------------*/

Widget
  xitFieldSelectGetChild( Widget                  widget,
                          xitFieldSelectChildren  child )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( NULL );

  switch( child ) {
    case xitFIELD_SELECT_TEXT_FIELD:
      return( rec -> field );

    case xitFIELD_SELECT_BUTTON:
      return( rec -> select_button );

    default:
      break;
  } /* switch */


  return( NULL );

} /* xitFieldSelectGetChild */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectGetCurrent( Widget  widget,
                            char    **value_return )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  *value_return = NULL;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  *value_return = xitStringGetText( rec -> field );


  return;

} /* xitFieldSelectGetCurrent */


/*----------------------------------------------------------------------*/

Boolean
  xitFieldSelectGetIndex( Widget  widget,
                          int     *selected_index )
{

  /* Variables. */
  int           index;
  char          buffer[ 50 ];
  char          *curr_str;
  char          *label_str;
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;
  Widget        tempW;


  /* Code. */

  *selected_index = 0;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( False );

  curr_str = xitStringGetText( rec -> field );


  /* Search the color in the range of colors. */
  for( index = 0; index < rec -> elements; index++ ) {

    sprintf( buffer, "%dPb", index + 1 );

    tempW = XtNameToWidget( rec -> menu, buffer );

    label_str = xitStringGetLabel( tempW );

    if( strcmp( curr_str, label_str ) == 0 ) {
      SysFree( label_str );
      SysFree( curr_str );

      *selected_index = index;

      return( True );
    }

    SysFree( label_str );

  } /* loop */

  SysFree( curr_str );


  return( False );

} /* xitFieldSelectGetIndex */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetAppend( Widget   widget,
                           Boolean  mode )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  rec -> append = mode;


  return;

} /* xitFieldSelectSetAppend */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetCurrent( Widget   widget,
                            char     *new_value,
                            Boolean  call )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  if( new_value == NULL )
    return;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  XmTextSetString( rec -> field, new_value );

  if( call )
    callCallback( rec, NULL );


  return;

} /* xitFieldSelectSetCurrent */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetIndex( Widget   widget,
                          int      set_index,
                          Boolean  call )
{

  /* Variables. */
  char          buffer[ 50 ];
  char          *label_str;
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;
  Widget        tempW;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  if( set_index < 0 || set_index > rec -> elements - 1 )
    return;

  sprintf( buffer, "%dPb", set_index + 1 );
  tempW = XtNameToWidget( rec -> menu, buffer );

  label_str = xitStringGetLabel( tempW );


  /* Assign the text. */
  XmTextSetString( rec -> field, label_str );
  SysFree( label_str );


  if( call )
    callCallback( rec, NULL );


  return;

} /* xitFieldSelectSetIndex */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetEditable( Widget   widget,
                             Boolean  editable )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  n = 0;
  XtSetArg( args[ n ], XmNeditable, editable ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, editable ); n++;
  XtSetValues( rec -> field, args, n );

  rec -> editable = editable;


  return;

} /* xitFieldSelectSetEditable */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetMenu( Widget  widget,
                         char    *new_menu[],
                         int     elements )
{

  /* Variables. */
  int           index;
  char          buffer[ 50 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Widget        menuPb;
  Widget        tempW;
  SelectRecPtr  rec;
  XmString      xstr;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  /* New menu buttons. */
  for( index = 0; index < elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    xstr = XmStringCreate( new_menu[ index ], CS );

    menuPb = XtNameToWidget( rec -> menu, buffer );
    if( menuPb == NULL )
      menuPb = XmCreatePushButton( rec -> menu, buffer, args, 0 );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
    XtSetValues( menuPb, args, n );

    XtManageChild( menuPb );

    XmStringFree( xstr );
  } /* loop */

  for( index = elements; index < rec -> elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    tempW = XtNameToWidget( rec -> menu, buffer );
    if( tempW != NULL )
      XtUnmanageChild( tempW );
  }

  XmTextSetString( rec -> field, "" );

  rec -> elements = elements;


  return;

} /* xitFiledSelectSetMenu */


/*----------------------------------------------------------------------*/

void
  xitFieldSelectSetLength( Widget  widget,
                           int     max_length )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, max_length ); n++;
  XtSetValues( rec -> field, args, n );

  rec -> max_length = max_length;


  return;

} /* xitFieldSelectSetLength */


/*----------------------------------------------------------------------*/

static void
  callCallback( SelectRecPtr  rec,
                XEvent        *event )
{

  /* Variables. */
  char                          *value;
  xitFieldSelectCallbackStruct  cb;


  /* Code. */

  if( rec -> call_proc != NULL ) {

    value = xitStringGetText( rec -> field );

    cb.reason   = XmCR_VALUE_CHANGED;
    cb.event    = event;
    cb.value    = value;

    (* rec -> call_proc) ( rec -> field,
                           (XtPointer) rec -> client_data, (XtPointer) &cb );

    SysFree( value );

  } /* if */


  return;

} /* callCallback */


/*----------------------------------------------------------------------*/

static void
  destroyCB( Widget        widget,
             SelectRecPtr  rec,
             XtPointer     call_data )
{

  /* Code. */

  /* Free the allocated record. */
  SysFree( rec );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  menuSelectCB( Widget                     widget,
                SelectRecPtr               rec,
                XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  int       new_length;
  int       old_length;
  char      *buffer_ref;
  char      *char_ref;
  char      *new_string;
  Arg       args[ 3 ];
  Cardinal  n;
  XmString  xstr;


  /* Code. */

  if( call_data -> reason != XmCR_ACTIVATE )
    return;

  /* Fetch the selected tag with colors. */
  n = 0;
  XtSetArg( args[ n ], XmNlabelString, &xstr ); n++;
  XtGetValues( call_data -> widget, args, n );

  /* Parse the string, get the selected tag. */
  buffer_ref = xitStringGetString( xstr, CS );

  char_ref   = xitStringGetText( rec -> field );
  new_length = strlen( buffer_ref );
  old_length = strlen( char_ref );

  /* Set the field with the selected value. */
  if( (rec -> append) && 
      (old_length > 0) &&
      (old_length + new_length + 1 < rec -> max_length) ) {

    new_string = SysMalloc( old_length + new_length + 2 );
    sprintf( new_string, "%s %s", char_ref, buffer_ref );

    XmTextSetString( rec -> field, new_string );

    SysFree( new_string );

  } else {
    XmTextSetString( rec -> field, buffer_ref );

  } /* if */

  SysFree( char_ref );

  callCallback( rec, call_data -> event );

  SysFree( buffer_ref );


  return;

} /* menuSelectCB */


/*----------------------------------------------------------------------*/

static void
  resizeCB( Widget               widget,
            SelectRecPtr         rec,
            XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg        args[ 5 ];
  Cardinal   n;
  Dimension  height;
  Dimension  select_height;
  Dimension  text_height;
  Dimension  text_width;
  Dimension  width;
  Position   new_y;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( widget, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtSetArg( args[ n ], XmNwidth,  &text_width ); n++;
  XtGetValues( rec -> field, args, n );

  if( ! rec -> display_text )
    text_width = 1;

  if( height > text_height )
    new_y = (height - text_height) / 2;
  else
    new_y = 0;

  /* Stop resizing the drawing area now. */
  n = 0;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  XtSetValues( widget, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &select_height ); n++;
  XtGetValues( rec -> select_button, args, n );

  /* Resize the children. */
  n = 0;
  XtSetArg( args[ n ], XmNy, new_y ); n++;
  XtSetValues( rec -> field, args, n );

  new_y = new_y + (text_height - select_height) / 2 - 2;

  n = 0;
  XtSetArg( args[ n ], XmNx, text_width + 2 ); n++;
  XtSetArg( args[ n ], XmNy, new_y ); n++;
  XtSetValues( rec -> menuBr, args, n );


  return;

} /* resizeCB */
