/*----------------------------------------------------------------------------
--
--  Module:           xitFieldStep
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    A step list widget consists of four widgets, a drawArea, a Label and
--    two arrows.
--
--  Contents:
--    xitCreateFieldStep
--    xitFieldStepGetChild
--    xitFieldStepGetCurrent
--    xitFieldStepSetCurrent
--
--  Filename:         xitFieldStep.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-03
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitFieldStep.c, Version: 1.1, Date: 95/02/18 15:10:30";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/DrawnB.h>
#include <Xm/DrawingA.h>

#include "System.h"

#include "xitFieldStep.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record associated with the widget. */
typedef struct {
  XmStringTable   list;
  int             current;
  int             elements;
  Widget          arrowDown;
  Widget          arrowUp;
  Widget          field;
  Dimension       arrow_width;
  Dimension       text_width;
  XtCallbackProc  call_proc;
  XtPointer       client_data;
} StepRec, *StepRecPtr;


/* Definitions to prefetch fontlist info before widget is created. */
typedef struct {
  XmFontList  fontlist;
} ResStruct, *ResPtr;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Resource list for items. */
static XtResource  item_resources[] = {
  { XmNitems, XmCItems, XmRXmStringTable, sizeof( XmStringTable ),
    XtOffset( StepRecPtr, list ), XmRImmediate, NULL },

  { XmNitemCount, XmCItemCount, XmRInt, sizeof( int ),
    XtOffset( StepRecPtr, elements ), XmRString, "0" },

  { XmNarrowWidth, XmCArrowWidth, XmRDimension, sizeof( Dimension ),
    XtOffset( StepRecPtr, arrow_width ), XmRString, "16" },
};

/* Resource for fontlist prefetch. */
static XtResource  fontlist_resource[] = {
  { XmNfontList, XmCFontList, XmRFontList, sizeof( XmFontList ),
    XtOffset( ResPtr, fontlist ), XmRString, "fixed" },
};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  callCallback( StepRecPtr  rec,
                XEvent      *event,
                int         previous );

static void
  destroyCB( Widget      widget,
             StepRecPtr  rec,
             XtPointer   call_data );

static XmFontList
  getFontList( Widget  parent,
               char    *name,
               char    *class );

static XmString
  longest( XmString    *xmlist,
           int         no_elements,
           XmFontList  font_list );

static void
  resizeCB( Widget               widget,
            StepRecPtr           rec,
            XmAnyCallbackStruct  *call_data );

static void
  stepDownCB( Widget               widget,
              StepRecPtr           rec,
              XmAnyCallbackStruct  *call_data );

static void
  stepUpCB( Widget               widget,
            StepRecPtr           rec,
            XmAnyCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateFieldStep( Widget          parent,
                      String          name,
                      ArgList         add_args,
                      Cardinal        num_args,
                      XtCallbackProc  proc,
                      XtPointer       client_data )
{

  /* Variables. */
  int            index;
  char           buffer[ 50 ];
  Arg            args[ 6 ];
  ArgList        merge_args;
  Cardinal       n;
  Dimension      text_height;
  StepRecPtr     rec;
  Widget         arrowDown;
  Widget         arrowUp;
  Widget         manager;
  Widget         text;
  XmFontList     text_font;
  XmStringTable  new_list;


  /* Code. */

  /* Initialize widget record, used in callbacks. */
  rec = SysNew( StepRec );

  rec -> current     = 1;
  rec -> call_proc   = proc;
  rec -> client_data = client_data;

  /* Drawing area as manager. */
  sprintf( buffer, "%sDa", name );

  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  XtSetArg( args[ n ], XmNuserData,     rec ); n++;
  manager = XtCreateManagedWidget( buffer, xmDrawingAreaWidgetClass,
                                   parent,
                                   args, n );

  XtAddCallback( manager, XmNresizeCallback,
                 (XtCallbackProc) resizeCB, (XtPointer) rec );
  XtAddCallback( manager, XmNdestroyCallback,
                 (XtCallbackProc) destroyCB, (XtPointer) rec );


  /* Initialize the rest of the callback record. */
  XtGetSubresources( manager, (XtPointer) rec, name, FIELD_STEP_CLASS,
                     item_resources, XtNumber( item_resources ),
                     add_args, num_args );

  /* Copy the XmStrings given. */
  if( rec -> elements > 0 ) {
    new_list = (XmStringTable) SysMalloc( sizeof( XmString ) * 
                                          rec -> elements );

    for( index = 0; index < rec -> elements; index++ )
      new_list[ index ] = XmStringCopy( rec -> list[ index ] );

    rec -> list = new_list;
  } /* if */


  /* Get the fontlist which will be used in the text display. */
  text_font = getFontList( manager, name, "XmDrawnButton" );


  /* Text display (with the longest item). */
  sprintf( buffer, "%sLa", name );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, 
            longest( rec -> list, rec -> elements, text_font ) ); n++;
  XtSetArg( args[ n ], XmNrecomputeSize,     False ); n++;
  XtSetArg( args[ n ], XmNpushButtonEnabled, False ); n++;
  XtSetArg( args[ n ], XmNalignment,         XmALIGNMENT_BEGINNING ); n++;

  merge_args = XtMergeArgLists( add_args, num_args, args, n );

  text = XtCreateManagedWidget( buffer, xmDrawnButtonWidgetClass,
                                manager,
                                merge_args, num_args + n );
  rec -> field = text;

  SysFree( merge_args );


  /* Step up button. */
  sprintf( buffer, "%sUpAr", name );

  n = 0;
  XtSetArg( args[ n ], XmNarrowDirection, XmARROW_UP ); n++;
  XtSetArg( args[ n ], XmNwidth,          rec -> arrow_width ); n++;
  arrowUp = XtCreateManagedWidget( buffer, xmArrowButtonWidgetClass,
                                   manager,
                                   args, n );
  rec -> arrowUp = arrowUp;

  XtSetSensitive( arrowUp, False );

  XtAddCallback( arrowUp, XmNactivateCallback,
                 (XtCallbackProc) stepUpCB, (XtPointer) rec );


  /* Step down button. */
  sprintf( buffer, "%sDownAr", name );

  n = 0;
  XtSetArg( args[ n ], XmNarrowDirection, XmARROW_DOWN ); n++;
  XtSetArg( args[ n ], XmNwidth,          rec -> arrow_width ); n++;
  arrowDown = XtCreateManagedWidget( buffer, xmArrowButtonWidgetClass,
                                     manager,
                                     args, n );
  rec -> arrowDown = arrowDown;

  XtSetSensitive( arrowDown, rec -> elements > 1 );

  XtAddCallback( arrowDown, XmNactivateCallback,
                 (XtCallbackProc) stepDownCB, (XtPointer) rec );


  /* Align text and arrows. */
  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtGetValues( text, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, text_height ); n++;
  XtSetValues( arrowUp,   args, n );
  XtSetValues( arrowDown, args, n );

  /* Set the text to the first label. */
  if( rec -> elements > 1 ) {

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, rec -> list[ 0 ] ); n++;
    XtSetValues( text, args, n );

  } /* if */

  return( manager );

} /* xitCreateFieldStep */


/*----------------------------------------------------------------------*/

Widget
  xitFieldStepGetChild( Widget                widget,
                        xitFieldStepChildren  child )
{

  /* Variables. */
  Arg         args[ 2 ];
  Cardinal    n;
  StepRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( NULL );

  switch( child ) {
    case xitFIELD_STEP_TEXT_FIELD:
      return( rec -> field );

    case xitFIELD_STEP_ARROW_UP:
      return( rec -> arrowUp );

    case xitFIELD_STEP_ARROW_DOWN:
      return( rec -> arrowDown );

    default:
      break;
  } /* switch */

  return( NULL );

} /* xitFieldStepGetChild */


/*----------------------------------------------------------------------*/

void
  xitFieldStepGetCurrent( Widget    widget,
                          int       *current_return,
                          XmString  *label_return )
{

  /* Variables. */
  Arg         args[ 2 ];
  Cardinal    n;
  StepRecPtr  rec;


  /* Code. */

  *current_return = 0;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  if( rec -> current < 1 || rec -> current > rec -> elements )
    return;

  *current_return = rec -> current;
  *label_return   = XmStringCopy( rec -> list[ rec -> current - 1 ] );

  return;

} /* xitFieldStepGetCurrent */


/*----------------------------------------------------------------------*/

void
  xitFieldStepSetCurrent( Widget   widget,
                          int      new,
                          Boolean  call )
{

  /* Variables. */
  int         previous;
  Arg         args[ 2 ];
  Cardinal    n;
  StepRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  if( new < 1 || new > rec -> elements )
    return;

  previous = rec -> current;
  rec -> current = new;

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, rec -> list[ rec -> current - 1 ] ); 
  n++;
  XtSetValues( rec -> field, args, n );

  XtSetSensitive( rec -> arrowUp,   rec -> current > 1 );
  XtSetSensitive( rec -> arrowDown, rec -> current < rec -> elements );

  if( call )
    callCallback( rec, NULL, previous );

  return;

} /* xitFieldStepSetCurrent */


/*----------------------------------------------------------------------*/

static void
  callCallback( StepRecPtr  rec,
                XEvent      *event,
                int         previous )
{

  /* Variables. */
  xitFieldStepCallbackStruct  cb;


  /* Code. */

  if( rec -> call_proc != NULL ) {
    cb.reason   = XmCR_VALUE_CHANGED;
    cb.event    = event;
    cb.list     = rec -> list;
    cb.elements = rec -> elements;
    cb.current  = rec -> current;
    cb.previous = previous;

    (* rec -> call_proc) ( rec -> field,
                           (XtPointer) rec -> client_data, (XtPointer) &cb );
  } /* if */

  return;

} /* callCallback */


/*----------------------------------------------------------------------*/
static void
  destroyCB( Widget      widget,
             StepRecPtr  rec,
             XtPointer   call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Free any strings in the selection list. */
  if( rec -> elements > 0 ) {
    for( index = 0; index < rec -> elements; index++ )
      XmStringFree( rec -> list[ index ] );
  }

  /* Free the allocated record. */
  SysFree( rec );

  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static XmFontList
  getFontList( Widget  parent,
               char    *name,
               char    *class )
{

  /* Variables. */
  ResStruct  res;


  /* Code. */

  XtGetSubresources( parent, (XtPointer) &res, name, class,
                     fontlist_resource, XtNumber( fontlist_resource ),
                     NULL, 0 );

  return( res.fontlist );

} /* getFontList */


/*----------------------------------------------------------------------*/

static XmString
  longest( XmString    *xmlist,
           int         no_elements,
           XmFontList  font_list )
{

  /* Variables. */
  int  index;
  int  max = 0;
  int  maxno = -1;
  int  new;

  /* Code. */

  for( index = 0; index < no_elements; index++ ) {
    new = XmStringWidth( font_list, xmlist[ index ] );

    if( max < new ) {
      max = new;
      maxno = index;
    }
  }

  if( maxno >= 0 )
    return( xmlist[ maxno ] );
  else
    return( NULL );

} /* longest */


/*----------------------------------------------------------------------*/

static void
  resizeCB( Widget               widget,
            StepRecPtr           rec,
            XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg        args[ 5 ];
  Cardinal   n;
  Dimension  arrow_width;
  Dimension  height;
  Dimension  text_height;
  Dimension  text_width;
  Dimension  width;
  Position   new_y;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( widget, args, n );

  if( width < 3 * rec -> arrow_width )
    arrow_width = width / 3;
  else
    arrow_width = rec -> arrow_width;

  text_width = width - 2 * arrow_width;

  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtGetValues( rec -> field, args, n );

  if( height > text_height )
    new_y = (height - text_height) / 2;
  else
    new_y = 0;

  /* Stop resizing the drawing area now. */
  n = 0;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  XtSetValues( widget, args, n );

  /* Resize the children. */
  n = 0;
  XtSetArg( args[ n ], XmNy,     new_y ); n++;
  XtSetArg( args[ n ], XmNwidth, text_width ); n++;
  XtSetValues( rec -> field, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNx,     text_width ); n++;
  XtSetArg( args[ n ], XmNy,     new_y ); n++;
  XtSetArg( args[ n ], XmNwidth, arrow_width ); n++;
  XtSetValues( rec -> arrowUp, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNx,     text_width + arrow_width ); n++;
  XtSetArg( args[ n ], XmNy,     new_y ); n++;
  XtSetArg( args[ n ], XmNwidth, arrow_width ); n++;
  XtSetValues( rec -> arrowDown, args, n );

  return;

} /* resizeCB */


/*----------------------------------------------------------------------*/

static void
  stepDownCB( Widget               widget,
              StepRecPtr           rec,
              XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  int       previous;
  Arg       args[ 5 ];
  Cardinal  n;


  /* Code. */

  if( rec -> current < rec -> elements ) {
    previous = rec -> current;
    rec -> current++;

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, rec -> list[ rec -> current - 1 ] );
    n++;
    XtSetValues( rec -> field, args, n );

    XtSetSensitive( rec -> arrowUp, True );
    XtSetSensitive( rec -> arrowDown, rec -> current < rec -> elements );

    callCallback( rec, call_data -> event, previous );
  } /* if */

  return;

} /* stepDownCB */


/*----------------------------------------------------------------------*/

static void
  stepUpCB( Widget               widget,
            StepRecPtr           rec,
            XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  int       previous;
  Arg       args[ 5 ];
  Cardinal  n;


  /* Code. */

  if( rec -> current > 1 ) {
    previous = rec -> current;
    rec -> current--;

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, rec -> list[ rec -> current - 1 ] ); 
    n++;
    XtSetValues( rec -> field, args, n );

    XtSetSensitive( rec -> arrowUp, rec -> current > 1 );
    XtSetSensitive( rec -> arrowDown, True );

    callCallback( rec, call_data -> event, previous );
  } /* if */

  return;

} /* stepUpCB */
