/*----------------------------------------------------------------------------
--
--  Module:           xitFileSel
--
--  Project:          Xtools
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Display a file selection dialog and let the user select a file.
--
--  Filename:         xitFileSel.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-05-18
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitFileSel.c, Version: 1.1, Date: 95/02/18 15:10:31";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/FileSB.h>

#include "System.h"

#include "xitError.h"
#include "xitTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Internal data to use. */
typedef struct {

  /* Flags. */
  UINT32  flags;

  /* Messages to use. */
  char  *no_file_sel_msg;
  char  *cannot_read_file_msg;

  /* The file selection window. */
  Widget  mainW;

  /* User data and use callback. */
  XIT_FILE_SEL_CB  actionCB;
  void             *user_data;

} SELECT_INFO, *SELECT_INFO_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitFileSel";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  cancelCB( Widget           widget,
            SELECT_INFO_REF  sel_ref,
            XtPointer        call_data );

static void
  destroyCB( Widget           widget,
             SELECT_INFO_REF  sel_ref,
             XtPointer        call_data );

static void
  okCB( Widget                            widget,
        SELECT_INFO_REF                   sel_ref,
        XmFileSelectionBoxCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateFileSelect( Widget               parent,
                       char                 *name,
                       UINT32               flags,
                       char                 *source_dir,
                       XIT_FILE_SEL_LABELS  *labels,
                       XIT_FILE_SEL_CB      actionCB,
                       void                 *user_data )
{

  /* Variables. */
  Arg              args[ 10 ];
  Cardinal         n;
  Widget           fileSelW;
  Widget           tempW;
  XmString         xstr;
  SELECT_INFO_REF  sel_ref;


  /* Code. */

  /* Only directories? */
  if( flagIsSet( flags, XIT_FILE_SELECT_ONLY_DIRS ) )
    flagSet( flags, XIT_FILE_SELECT_DIRS_OK );


  n = 0;

  /* This window is always deleted. */
  XtSetArg( args[ n ], XmNdeleteResponse, XmDESTROY ); n++;

  /* Create and initialize our private data. */
  sel_ref = SysNew( SELECT_INFO );
  if( sel_ref == NULL )
    return( NULL );

  sel_ref -> flags                = flags;
  sel_ref -> no_file_sel_msg      = labels -> no_file_sel_msg;
  sel_ref -> cannot_read_file_msg = labels -> cannot_read_file_msg;

  sel_ref -> actionCB  = actionCB;
  sel_ref -> user_data = user_data;


  /* Create the file selection dialog. */
  fileSelW = XmCreateFileSelectionDialog( parent, name, args, n );

  sel_ref -> mainW = fileSelW;

  XtAddCallback( fileSelW, XmNokCallback,
                 (XtCallbackProc) okCB, (XtPointer) sel_ref );
  XtAddCallback( fileSelW, XmNcancelCallback,
                 (XtCallbackProc) cancelCB, (XtPointer) sel_ref );
  XtAddCallback( fileSelW, XmNdestroyCallback,
                 (XtCallbackProc) destroyCB, (XtPointer) sel_ref );


  /* The labels. */
  if( strlen( labels -> filter_label ) > 0 ) {
    xstr = XmStringCreate( labels -> filter_label, CS );

    XtSetArg( args[ 0 ], XmNfilterLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  if( strlen( labels -> dir_list_label ) > 0 ) {
    xstr = XmStringCreate( labels -> dir_list_label, CS );

    XtSetArg( args[ 0 ], XmNdirListLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  if( strlen( labels -> file_list_label ) > 0 ) {
    xstr = XmStringCreate( labels -> file_list_label, CS );

    XtSetArg( args[ 0 ], XmNfileListLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  if( strlen( labels -> selection_label ) > 0 ) {
    xstr = XmStringCreate( labels -> selection_label, CS );

    XtSetArg( args[ 0 ], XmNselectionLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  /* The action buttons. */
  if( strlen( labels -> ok_button ) > 0 ) {
    xstr = XmStringCreate( labels -> ok_button, CS );

    XtSetArg( args[ 0 ], XmNokLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  if( strlen( labels -> filter_button ) > 0 ) {
    xstr = XmStringCreate( labels -> filter_button, CS );

    XtSetArg( args[ 0 ], XmNapplyLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }

  if( strlen( labels -> cancel_button ) > 0 ) {
    xstr = XmStringCreate( labels -> cancel_button, CS );

    XtSetArg( args[ 0 ], XmNcancelLabelString, xstr );
    XtSetValues( fileSelW, args, 1 );

    XmStringFree( xstr );
  }


  /* We don't use the Help button. */
  tempW = XmFileSelectionBoxGetChild( fileSelW, XmDIALOG_HELP_BUTTON );
  XtUnmanageChild( tempW );


  /* No title. */
  n = 0;
  XtSetArg( args[ n ], XmNtitle, " " ); n++;
  XtSetValues( XtParent( fileSelW ), args, n );


  return( fileSelW );

} /* xitCreateFileSelect */


/*----------------------------------------------------------------------*/

void
  xitFileSelectDisplay( Widget  widget )
{

  /* Code. */

  if( widget == NULL )
    return;


  /* Make sure the file selection window is visable. */
  XtManageChild( widget );

  XtPopup( XtParent( widget ), XtGrabNone );

  XRaiseWindow( XtDisplay( widget ), XtWindow( widget ) );
  XtMapWidget(  widget );


  return;

} /* xitFileSelectDisplay */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget           widget,
            SELECT_INFO_REF  sel_ref,
            XtPointer        call_data )
{

  /* Code. */

  /* User callback? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XIT_FILE_SELECT_CANCEL, NULL,
                             sel_ref -> user_data );

  /* Remove the window. */
  XtUnmanageChild( sel_ref -> mainW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void
  destroyCB( Widget           widget,
             SELECT_INFO_REF  sel_ref,
             XtPointer        call_data )
{

  /* Code. */

  /* Free the internal data. */
  if( sel_ref != NULL )
    SysFree( sel_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget                            widget,
        SELECT_INFO_REF                   sel_ref,
        XmFileSelectionBoxCallbackStruct  *call_data )
{

  /* Variables. */
  int          status;
  char         *file_name;
  struct stat  file_info;


  /* Code. */

  /* The selected file name. */
  file_name = xitStringGetString( call_data -> value, CS );


  /* Only directory? */
  if( flagIsSet( sel_ref -> flags, XIT_FILE_SELECT_ONLY_DIRS ) ) {
    SysFree( file_name );

    file_name = xitStringGetString( call_data -> dir, CS );
  }


  /* No file selected? */
  if( file_name == NULL || *file_name == '\0' ) {
    xitErMessage( sel_ref -> mainW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  sel_ref -> no_file_sel_msg );

    SysFree( file_name );
    return;
  }

  /* Is the file a normal file and can it be read? */
  status = stat( file_name, &file_info );

  if( status != 0 ) {
    xitErMessage( sel_ref -> mainW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  sel_ref -> cannot_read_file_msg );

    SysFree( file_name );
    return;
  }

  if( S_ISDIR( file_info.st_mode ) &&
      flagIsClear( sel_ref -> flags, XIT_FILE_SELECT_DIRS_OK ) ) {
    xitErMessage( sel_ref -> mainW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  sel_ref -> cannot_read_file_msg );

    SysFree( file_name );
    return;
  }


  status = access( file_name, (F_OK | R_OK) );

  if( status != 0 ) {
    xitErMessage( sel_ref -> mainW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  sel_ref -> cannot_read_file_msg );

    SysFree( file_name );
    return;
  }


  /* User callback? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XIT_FILE_SELECT_OK, file_name,
                             sel_ref -> user_data );


  /* Remove the window. */
  XtUnmanageChild( sel_ref -> mainW );

  SysFree( file_name );


  return;

} /* okCB */
