/*----------------------------------------------------------------------------
--
--  Module:           xitInfoSrch
--
--  Project:          xit - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Search a string in the information database.
--
--  Filename:         xitInfoSrch.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfoSrch.c, Version: 1.1, Date: 95/02/18 15:10:38";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgTopic.h"
#include "xitInfoFile.h"
#include "xitTools.h"
#include "xitInfoSrch.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the print window. */
#define gotoRc                dataLocalW[  0 ]
#define resultLa              dataLocalW[  1 ]
#define resultLi              dataLocalW[  2 ]
#define sep1                  dataLocalW[  3 ]
#define sep2                  dataLocalW[  4 ]
#define srchCaseTb            dataLocalW[  5 ]
#define srchForLa             dataLocalW[  6 ]
#define srchForTx             dataLocalW[  7 ]
#define srchHeaderTb          dataLocalW[  8 ]
#define srchInBooksTb         dataLocalW[  9 ]
#define srchInFr              dataLocalW[ 10 ]
#define srchInRc              dataLocalW[ 11 ]
#define srchTextTb            dataLocalW[ 12 ]

/* Max number of matches we can handle. */
#define MAX_MATCHES      500


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record while searching entries. */
typedef struct {

  /* The selected section. */
  int  selected_item;

  /* The name of the current book. */
  char  *curr_book_name;

  /* The search window. */
  Widget  searchW;

  /* The books we are using. */
  LST_DESC_TYPE  *books;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XIT_IS_ACTION_CB  actionCB;

} SEARCH_REC, *SEARCH_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfoSrch";

/* Name of text domain. */
static char  *td = "Topic";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget          widget, 
            SEARCH_REC_REF  search_ref,
            XtPointer       call_data );

static Widget
  createSearchWindow( SEARCH_REC_REF  search_ref,
                      Widget          parent );

static void 
  destroyCB( Widget          widget,
             SEARCH_REC_REF  search_ref,
             XtPointer       call_data );

static void 
  displaySelectedSection( SEARCH_REC_REF  search_ref );

static void 
  gotoCB( Widget                     widget, 
          SEARCH_REC_REF             search_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  resultLiCB( Widget                widget, 
              SEARCH_REC_REF        search_ref,
              XmListCallbackStruct  *call_data );

static void 
  searchCB( Widget          widget, 
            SEARCH_REC_REF  search_ref,
            XtPointer       call_data );

static void
  setGotoButtons( SEARCH_REC_REF  search_ref,
                  Boolean         sensitive );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_IS_HANDLE
  xitIsInitialize( Widget            parent,
                   LST_DESC_TYPE     *use_books,
                   XIT_IS_ACTION_CB  actionCB,
                   void              *user_data )
{

  /* Variables. */
  SEARCH_REC_REF  search_ref;


  /* Code. */

  /* Create and initialize our private data. */
  search_ref = SysNew( SEARCH_REC );
  if( search_ref == NULL )
    return( NULL );

  search_ref -> curr_book_name = NULL;
  search_ref -> books          = use_books;
  search_ref -> actionCB       = actionCB;
  search_ref -> user_data      = user_data;


  /* Create the search window. */
  search_ref -> searchW = createSearchWindow( search_ref, parent );

  if( search_ref -> searchW == NULL ) {
    SysFree( search_ref );

    return( NULL );
  }


  return( (XIT_IS_HANDLE) search_ref );

} /* xitIsInitialize */


/*----------------------------------------------------------------------*/

void
  xitIsDestroy( XIT_IS_HANDLE  search_handle )
{

  /* Variables. */
  SEARCH_REC_REF  search_ref;


  /* Code. */

  if( search_handle == NULL )
    return;

  /* Our private data. */
  search_ref = (SEARCH_REC_REF) search_handle;


  /* Destroy the window. */
  XtDestroyWidget( search_ref -> searchW );


  return;

} /* xitIsDestroy */


/*----------------------------------------------------------------------*/

void 
  xitIsDisplaySearch( XIT_IS_HANDLE  search_handle,
                      char           *curr_book_name )
{

  /* Variables. */
  Widget          mainW;
  Widget          tempW;
  SEARCH_REC_REF  search_ref;


  /* Code. */

  search_ref = (SEARCH_REC_REF) search_handle;
  if( search_ref == NULL )
    return;

  mainW = XtNameToWidget( search_ref -> searchW, "SearchFdFo" );

  if( search_ref -> curr_book_name != NULL )
    SysFree( search_ref -> curr_book_name );
  search_ref -> curr_book_name = SysNewString( curr_book_name );


  /* The Goto buttons are not valid. */
  setGotoButtons( search_ref, False );


  /* Make sure the window is visible. */
  XtManageChild( search_ref -> searchW );


  return;

} /* xitIsDisplaySearch */


/*----------------------------------------------------------------------*/

static Widget
  createSearchWindow( SEARCH_REC_REF  search_ref,
                      Widget          parent )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 13 ];
  Widget    gotoBu[ 2 ];
  Widget    searchFd;
  Widget    workFo;

  static XIT_PUSH_STRUCT goto_def[] = {
    { "GotoNextBu", "", "", True, NULL },
    { "GotoPrevBu", "", "", True, NULL },
  };  

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "SrchForTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   searchCB, NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgDGetText( td, MINF_DO_SEARCH_BUTTON );
  action_buttons[ 0 ].data  = search_ref;
  action_buttons[ 3 ].label = msgDGetText( td, MINF_CLOSE_BUTTON );
  action_buttons[ 3 ].data  = search_ref;

  goto_def[ 0 ].title = msgDGetText( td, MINF_GOTO_NEXT_BUTTON );
  goto_def[ 1 ].title = msgDGetText( td, MINF_GOTO_PREV_BUTTON );


  /* Create a form dialog with buttons. */
  searchFd = xitCreateFormDialog( parent, "SearchFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  XtAddCallback( searchFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) search_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgDGetText( td, MINF_SEARCH_TITLE ) ); n++;
  XtSetValues( XtParent( searchFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( searchFd, args, n );

  /* Container for the contents of the window. */
  workFo = XtNameToWidget( searchFd, "SearchFdFo" );


  /* Create a search expression. */
  srchForLa = xitCreateLabel( workFo, "SrchForLa",
                              msgDGetText( td, MINF_SEARCH_FOR_LABEL ), -1 );
 
  srchForTx = xitCreateText( workFo, &text_buffer[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 30 );  n++;
  XtSetArg( args[ n ], XmNcolumns,   30 );  n++;
  XtSetValues( srchForTx, args, n );  


  /* Create a 'search where' selection. */
  srchInFr = XmCreateFrame(     workFo,   "SrchInFr", args, 0 );
  srchInRc = XmCreateRowColumn( srchInFr, "SrchInRc", args, 0 );

  srchCaseTb    = xitCreateToggleButton( 
                    srchInRc, "SrchCaseTb", 
                    msgDGetText( td, MINF_SEARCH_CASE_SENS ),
                    False );

  srchInBooksTb = xitCreateToggleButton( 
                    srchInRc, "SrchInBooksTb", 
                    msgDGetText( td, MINF_SEARCH_IN_BOOKS_LABEL ),
                    True );

  srchHeaderTb  = xitCreateToggleButton( 
                    srchInRc, "SrchHeaderTb", 
                    msgDGetText( td, MINF_SEARCH_HEADERS_LABEL ),
                    True );

  srchTextTb    = xitCreateToggleButton( 
                    srchInRc, "SrchTextTb", 
                    msgDGetText( td, MINF_SEARCH_TEXT_LABEL ), 
                    True );


  /* Create result list. */
  resultLa = xitCreateLabel( workFo, "ResultLa",
                             msgDGetText( td, MINF_RESULT_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  resultLi = XmCreateScrolledList( workFo, "ResultLi", args, n );

  XtAddCallback( resultLi, XmNsingleSelectionCallback, 
                 (XtCallbackProc) resultLiCB, (XtPointer) search_ref );
  XtAddCallback( resultLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) resultLiCB, (XtPointer) search_ref );


  /* Create the action form. */
  n = 0;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  gotoRc = XmCreateRowColumn( workFo, "GotoRc", args, n );

  XtAddCallback( gotoRc, XmNentryCallback, 
                 (XtCallbackProc) gotoCB, (XtPointer) search_ref );

  for( index = 0; index < XtNumber( gotoBu ); index++ ) {
    gotoBu[ index ] = xitCreatePushButton( gotoRc, &goto_def[ index ] );

    XtAddCallback( gotoBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) gotoCB, (XtPointer) index );
  }


  /* Create separators. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  sep1 = XmCreateSeparatorGadget( workFo, "Sep1", args, n );
  sep2 = XmCreateSeparatorGadget( workFo, "Sep2", args, n );


  /* Place the various elements together. */
  xitAttachWidget( srchForLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( srchForTx,
                   XmATTACH_WIDGET, srchForLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( srchInFr,
                   XmATTACH_WIDGET, srchForTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( sep1,
                   XmATTACH_WIDGET, srchInFr, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( resultLa,
                   XmATTACH_WIDGET, sep1, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( resultLi ),
                   XmATTACH_WIDGET, resultLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( sep2,
                   XmATTACH_WIDGET, XtParent( resultLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( gotoRc,
                   XmATTACH_WIDGET, sep2, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( gotoRc, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( srchForLa,            args, n );
  XtSetValues( srchForTx,            args, n );
  XtSetValues( srchInFr,             args, n );
  XtSetValues( resultLa,             args, n );
  XtSetValues( XtParent( resultLi ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( sep1, args, n );
  XtSetValues( sep2, args, n );


  /* Manage all the children except the search form. */
  xitManageChildren( gotoBu, XtNumber( gotoBu  ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW  ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( searchFd, True );


  /* Make the final attachments. */
  xitAttachWidget( gotoRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( sep2,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, gotoRc );
  xitAttachWidget( XtParent( resultLi ),
                   XmATTACH_WIDGET, resultLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_WIDGET, sep2 );


  return( searchFd );

} /* createSearchWindow */


/*----------------------------------------------------------------------*/

static void 
  displaySelectedSection( SEARCH_REC_REF  search_ref )
{

  /* Variables. */
  int       item_count;
  int       level;
  char      book[ 50 ];
  char      chapter_id[ 50 ];
  char      title[ 100 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    resultLiW;
  XmString  *items;


  /* Code. */

  mainW     = XtNameToWidget( search_ref -> searchW, "SearchFdFo" );
  resultLiW = XtNameToWidget( mainW, "ResultLiSW.ResultLi" );


  /* Fetch the selected item. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &item_count ); n++;
  XtGetValues( resultLiW, args, n );


  /* There can only be ONE selection. */
  if( item_count != 1 )
    return;


  /* Get the selected string. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItems, &items ); n++;
  XtGetValues( resultLiW, args, n );

  char_ref = xitStringGetString( *items, CS );


  /* Extract header, chapter id and header level. */
  sscanf( char_ref, "%[^#]#%[^#]#%d#%s", title, chapter_id, &level, book );

  SysFree( char_ref );


  /* Tell our parent what we have found. */
  if( search_ref -> actionCB != NULL )
    (* search_ref -> actionCB)( XIT_IS_REASON_NEW_SECTION, 
                                search_ref -> user_data,
                                book, chapter_id, title );

  return;

} /* displaySelectedSection */


/*----------------------------------------------------------------------*/

static void
  setGotoButtons( SEARCH_REC_REF  search_ref,
                  Boolean         sensitive )
{

  /* Variables. */
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  mainW = XtNameToWidget( search_ref -> searchW, "SearchFdFo" );
  tempW = XtNameToWidget( mainW, "GotoRc" );

  XtSetSensitive( tempW, sensitive );


  return;
    
} /* setGotoButtons */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget          widget, 
            SEARCH_REC_REF  search_ref,
            XtPointer       call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( search_ref -> actionCB != NULL )
    (* search_ref -> actionCB)( XIT_IS_REASON_POPDOWN, 
                                search_ref -> user_data, NULL, NULL, NULL );


  XtUnmanageChild( search_ref -> searchW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             SEARCH_REC_REF  search_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( search_ref -> actionCB != NULL )
    (* search_ref -> actionCB)( XIT_IS_REASON_DESTROY, 
                                search_ref -> user_data, NULL, NULL, NULL );

  /* Free allocated data. */
  if( search_ref -> curr_book_name != NULL )
    SysFree( search_ref -> curr_book_name );

  SysFree( search_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  gotoCB( Widget                     widget, 
          SEARCH_REC_REF             search_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  int       item_count;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    mainW;
  Widget    resultLiW;


  /* Code. */

  mainW = XtNameToWidget( search_ref -> searchW, "SearchFdFo" );

  /* Elements in the list. */
  resultLiW = XtNameToWidget( mainW, "ResultLiSW.ResultLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtGetValues( resultLiW, args, n );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Next? */
    case 0:
      if( search_ref -> selected_item >= item_count )
        return;

      /* Make the selected position the top position. */
      XmListSetPos( resultLiW, search_ref -> selected_item + 1 );

      /* Select the next position in the list. */
      XmListSelectPos( resultLiW, 
                       search_ref -> selected_item + 1, True );
      break;


    /* Previous? */
    case 1:
      if( search_ref -> selected_item <= 1 )
        return;

      /* Make the selected position the top position. */
      XmListSetPos( resultLiW, search_ref -> selected_item - 1 );

      /* Select the next position in the list. */
      XmListSelectPos( resultLiW, 
                       search_ref -> selected_item - 1, True );
      break;

  } /* switch */


  return;

} /* gotoCB */


/*----------------------------------------------------------------------*/

static void 
  resultLiCB( Widget                widget, 
              SEARCH_REC_REF        search_ref,
              XmListCallbackStruct  *call_data )
{

  /* Code. */

  /* Save the select section. */
  search_ref -> selected_item = call_data -> item_position;

  /* We can use the goto buttons. */
  setGotoButtons( search_ref, True );


  /* Display the info text for this section. */
  displaySelectedSection( search_ref );


  return;

} /* resultLiCB */


/*----------------------------------------------------------------------*/

static void 
  searchCB( Widget          widget, 
            SEARCH_REC_REF  search_ref,
            XtPointer       call_data )
{

  /* Variables. */
  Boolean                in_all_books;
  int                    book_index;
  int                    index;
  int                    list_items_no = 0;
  int                    matches_no;
  char                   buffer[ 200 ];
  char                   *char_ref;
  char                   *search_string_ref;
  Arg                    args[ 10 ];
  Cardinal               n;
  Cursor                 wait_cursor;
  Display                *display = XtDisplay( search_ref -> searchW );
  Widget                 baseW;
  Widget                 mainW;
  Widget                 tempW;
  Window                 window = XtWindow(  search_ref -> searchW );
  XmString               list_items[ MAX_MATCHES ];
  LST_STATUS             lst_status;
  XIT_IF_BOOK_REC        book;
  XIT_IF_HANDLE          info_handle;
  XIT_IF_SECTION_REF     matches[ MAX_MATCHES ];
  XIT_IF_SEARCH_PROFILE  search_rec;
  XIT_IF_STATUS          status;


  /* Code. */

  mainW = XtNameToWidget( search_ref -> searchW, "SearchFdFo" );


  /* Fetch the search string. */
  tempW = XtNameToWidget( mainW, "SrchForTx" );

  search_string_ref = xitStringGetText( tempW );

  if( strlen( search_string_ref ) == 0 ) {
    SysFree( search_string_ref );

    return;
  }

  search_rec.string = search_string_ref;


  /* Create the search profile. */
  baseW = XtNameToWidget( mainW, "SrchInFr.SrchInRc" );

  tempW = XtNameToWidget( baseW, "SrchCaseTb" );
  search_rec.case_sensitive = XmToggleButtonGetState( tempW );

  tempW = XtNameToWidget( baseW, "SrchHeaderTb" );
  search_rec.headers = XmToggleButtonGetState( tempW );

  tempW = XtNameToWidget( baseW, "SrchTextTb" );
  search_rec.text = XmToggleButtonGetState( tempW );

  tempW = XtNameToWidget( baseW, "SrchInBooksTb" );
  in_all_books = XmToggleButtonGetState( tempW );


  /* The Goto buttons are not valid. */
  setGotoButtons( search_ref, False );


  /* This might take time. */
  wait_cursor = XCreateFontCursor( display, XC_watch );
  XDefineCursor( display, window, wait_cursor );

  XFlush( display );


  /* Search the books. */
  for( book_index = 0; book_index < 2; book_index++ ) {

    lst_status = LstLinkCurrentFirst( search_ref -> books[ book_index ] );
    while( lst_status == LST_OK ) {

      lst_status = LstLinkGetCurrent( search_ref -> books[ book_index ],
                                      &book );

      /* Is this a place to search? */
      if( ! in_all_books && 
          strcmp( book.name, search_ref -> curr_book_name ) != 0 ) {
        lst_status = LstLinkCurrentNext( search_ref -> books[ book_index ] );
        continue;
      }

      /* Initalize the info file. */
      info_handle = xitIfInitialize( book.file );

      if( info_handle == NULL ) {
        lst_status = LstLinkCurrentNext( search_ref -> books[ book_index ] );
        continue;
      }


      /* Search. */
      status = xitIfSearchString( info_handle, 
                                  &search_rec,
                                  XtNumber( matches ), &matches[ 0 ], 
                                  &matches_no );

      /* Clean up. */
      xitIfDestroy( info_handle );

      /* Assign the found sections to the result list. */
      for( index = 0; index < matches_no; index++ ) {

        if( list_items_no >= MAX_MATCHES )
          break;

        /* Strip blanks. */
        char_ref = matches[ index ] -> title;
        while( isspace( *char_ref ) )
          char_ref++;

        sprintf( buffer, "%-80.80s#%s#%d#%s", 
                 char_ref, matches[ index ] -> chapter_id, 
                 matches[ index ] -> toc_level, book.name );

        list_items[ list_items_no ] = XmStringCreate( buffer, CS );
        list_items_no++;

      } /* loop */

      xitIfFreeMatchInfo( &matches[ 0 ], matches_no );

      /* Read the next book. */
      lst_status = LstLinkCurrentNext( search_ref -> books[ book_index ] );

    } /* while */

  } /* loop */


  /* We do not need the search string anymore. */
  SysFree( search_string_ref );

  if( list_items_no == 0 )
    tempW = xitCreateInformationDialog( 
              mainW, "InformationDialog",
              msgDGetText( td, MINF_INFORMATION_LABEL ),
              msgDGetText( td, MINF_SEARCH_NO_MATCH ),
              NULL, NULL );

  XUndefineCursor( display, window );

  tempW = XtNameToWidget( mainW, "ResultLiSW.ResultLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, list_items_no ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free the match records. */
  for( index = 0; index < list_items_no; index++ )
    XmStringFree( list_items[ index ] );


  return;

} /* searchCB */
