/*----------------------------------------------------------------------------
--
--  Module:           XitRwPixButton
--
--  Project:          XitRw - Rogge's Motif widgets
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation file for the widget.
--    A Pix button can display both a text and a pixmap.
--
--  Filename:         xitRwPixB.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitRwPixB.c, Version: 1.1, Date: 95/02/18 15:10:46";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <Xm/XmP.h>

#if XmVersion > 1001
#  include <Xm/PrimitiveP.h>
#endif

#include "xitRwPixBP.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define max( a, b )  ((a) < (b) ? (b) : (a) )

#define DEFAULT_MARGIN       2
#define DEFAULT_MARGIN_NULL  0
#define INSENS_WIDTH         10
#define INSENS_HEIGHT        10

#define XmRPixmapPosition    "PixmapPosition"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef int pixmapPosition;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

static void
  Activate( Widget  w,
            XEvent  *event );

static void
  Arm( Widget  w,
       XEvent  *event );

static void
  ArmAndActivate( Widget   w,
		  XEvent   *event );

static void
  Disarm( Widget   w,
	  XEvent   *event,
	  String   *params,
	  Cardinal num_params );

static void
  Enter( Widget   w,
	 XEvent   *event,
	 String   *params,
	 Cardinal num_params );

static void
  PixButFocusIn( Widget   w,
                 XEvent   *event,
                 String   *params,
                 Cardinal num_params );

static void
  PixButFocusOut( Widget   w,
                  XEvent   *event,
                  String   *params,
                  Cardinal num_params );

static void
  Leave( Widget   w,
	 XEvent   *event,
	 String   *params,
	 Cardinal num_params );

static void
  TraverseNextTabGroup( Widget   w,
                        XEvent   *event,
	                String   *params,
	                Cardinal num_params );

static void
  TraversePrevTabGroup( Widget   w,
                        XEvent   *event,
	                String   *params,
	                Cardinal num_params );

static void
  TraverseDown( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params );
static void
  TraverseHome( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params );
static void
  TraverseLeft( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params );
static void
  TraverseRight( Widget   w,
                 XEvent   *event,
	         String   *params,
	         Cardinal num_params );
static void
  TraverseUp( Widget   w,
              XEvent   *event,
	      String   *params,
	      Cardinal num_params );

static void
  Unmap( Widget   w,
         XEvent   *event,
         String   *params,
         Cardinal num_params );


/* Resource converters. */
static void
  ConvertToPixmapPosition( XrmValue   *args,
                           Cardinal   *num_args,
                           XrmValue   *from,
                           XrmValue   *to );

/* Resource list. */
static XtResource resources[] = {
  {
    XmNactivateCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XitRwPixButtonWidget, pixBut.activate_callback ),
    XmRPointer,
    (XtPointer) NULL
  },
  {
    XmNalignment,
    XmCAlignment,
    XmRAlignment, sizeof( unsigned char ),
    XtOffset( XitRwPixButtonWidget, pixBut.alignment ),
    XmRImmediate,
    (XtPointer) XmALIGNMENT_CENTER
  },
  {
    XmNarmCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XitRwPixButtonWidget, pixBut.arm_callback ),
    XmRPointer,
    (XtPointer) NULL
  },
  {
    XmNdisarmCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XitRwPixButtonWidget, pixBut.disarm_callback ),
    XmRPointer,
    (XtPointer) NULL
  },
  {
    XmNfontList,
    XmCFontList,
    XmRFontList, sizeof( XmFontList ),
    XtOffset( XitRwPixButtonWidget, pixBut.font_list ),
    XmRString,
    (XtPointer) "Fixed"
  },
  {
    XmNlabelPixmap,
    XmCLabelPixmap,
    XmRInt, sizeof( Pixmap ),
    XtOffset( XitRwPixButtonWidget, pixBut.pixmap ),
    XtRImmediate,
    (XtPointer) XmUNSPECIFIED_PIXMAP
  },
  {
    XmNlabelString,
    XmCLabelString,
    XmRString, sizeof( XmString ),
    XtOffset( XitRwPixButtonWidget, pixBut.label ),
    XtRImmediate,
    NULL
  },
  {
    XmNmarginHeight,
    XmCMarginHeight,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.margin_height ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN
  },
  {
    XmNmarginWidth,
    XmCMarginWidth,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.margin_width ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN
  },
  {
    XmNmarginTop,
    XmCMarginTop,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.top_margin ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN_NULL
  },
  {
    XmNmarginLeft,
    XmCMarginLeft,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.left_margin ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN_NULL
  },
  {
    XmNmarginRight,
    XmCMarginRight,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.right_margin ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN_NULL
  },
  {
    XmNmarginBottom,
    XmCMarginBottom,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.bottom_margin ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN_NULL
  },
  {
    XitRwNoppositeAlignment,
    XmCAlignment,
    XmRAlignment, sizeof( unsigned char ),
    XtOffset( XitRwPixButtonWidget, pixBut.opposite_alignment ),
    XmRImmediate,
    (XtPointer) XmALIGNMENT_CENTER
  },
  {
    XitRwNpixmapPosition,
    XitRwCPixmapPosition,
    XmRPixmapPosition, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.pixmap_position ),
    XtRImmediate,
    (XtPointer) XitRwLEFT
  },
  {
    XmNrecomputeSize,
    XmCRecomputeSize,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XitRwPixButtonWidget, pixBut.recompute_size ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmNshadowType,
    XmCShadowType,
    XmRShadowType, sizeof( unsigned char ),
    XtOffset( XitRwPixButtonWidget, pixBut.shadow_type ),
    XmRImmediate,
    (XtPointer) XmSHADOW_OUT
  },
  {
    XmNspacing,
    XmCSpacing,
    XtRInt, sizeof( int ),
    XtOffset( XitRwPixButtonWidget, pixBut.spacing ),
    XtRImmediate,
    (XtPointer) DEFAULT_MARGIN
  },
};


/* Pixmap for insensitive pixmap. */
static unsigned char insens_bits[] = {
   0x55, 0x01, 0xaa, 0x02, 0x55, 0x01, 0xaa, 0x02, 0x55, 0x01, 0xaa, 0x02,
   0x55, 0x01, 0xaa, 0x02, 0x55, 0x01, 0xaa, 0x02};

/* Action list. */
static XtActionsRec actionsList[] =
{
  { "Activate",              (XtActionProc) Activate },
  { "Arm",                   (XtActionProc) Arm },
  { "ArmAndActivate",        (XtActionProc) ArmAndActivate },
  { "PrimitiveNextTabGroup", (XtActionProc) TraverseNextTabGroup },
  { "PrimitivePrevTabGroup", (XtActionProc) TraversePrevTabGroup },
  { "PrimitiveTabDown",      (XtActionProc) TraverseDown },
  { "PrimitiveTabHome",      (XtActionProc) TraverseHome },
  { "PrimitiveTabLeft",      (XtActionProc) TraverseLeft },
  { "PrimitiveTabRight",     (XtActionProc) TraverseRight },
  { "PrimitiveTabUp",        (XtActionProc) TraverseUp },
  { "Disarm",                (XtActionProc) Disarm },
  { "Enter",                 (XtActionProc) Enter },
  { "FocusIn",               (XtActionProc) PixButFocusIn },
  { "FocusOut",              (XtActionProc) PixButFocusOut },
  { "Leave",                 (XtActionProc) Leave },
  { "Unmap",                 (XtActionProc) Unmap },
};


/* Default translation list. */
static char defaultTranslations[] =
  "<Btn1Down>:     Arm()                \n\
   <Btn1Up>:       Activate() Disarm()  \n\
   <EnterWindow>:  Enter()              \n\
   <FocusIn>:      FocusIn()            \n\
   <FocusOut>:     FocusOut()           \n\
   <LeaveWindow>:  Leave()              \n\
   <Key>Return:    ArmAndActivate()     \n\
   <Key>space:     ArmAndActivate()     \n\
   <Key>Home:      PrimitiveTabHome()   \n\
   <Key>Down:      PrimitiveTabDown()   \n\
   <Key>Left:      PrimitiveTabLeft()   \n\
   <Key>Right:     PrimitiveTabRight()  \n\
   <Key>Up:        PrimitiveTabUp()     \n\
   <Key>osfDown:   PrimitiveTabDown()   \n\
   <Key>osfLeft:   PrimitiveTabLeft()   \n\
   <Key>osfRight:  PrimitiveTabRight()  \n\
   <Key>osfUp:     PrimitiveTabUp()     \n\
   ~Ctrl ~Alt ~Meta ~Shift<Key>Tab:  PrimitiveNextTabGroup() \n\
   ~Ctrl ~Alt ~Meta  Shift<Key>Tab:  PrimitivePrevTabGroup() \n\
   <Unmap>:        Unmap()";


/* Quarks. */
static XrmQuark   quark_bottom;
static XrmQuark   quark_left;
static XrmQuark   quark_right;
static XrmQuark   quark_top;



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Core methods. */
static void
  ClassInitialize();

static void 
  Destroy( Widget  widget );

static void
  ExposeIt( Widget  widget,
            XEvent  *event,
            Region  region );

static void
  Initialize( Widget    treq,
              Widget    tnew,
              ArgList   args,
              Cardinal  *num_args );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget  widget );

static Boolean 
  SetValues( Widget    current,
             Widget    request,
             Widget    new,
             ArgList   args,
             Cardinal  *num_args );

/* These core methods are not needed. 
--
--  static void Realize();
--
*/

/* Local functions. */
static void
  DesiredDimensions( Widget     widget,
                     Boolean    override_no_recompute,
                     Dimension  *width,
                     Dimension  *height );

static void 
  DrawButton( Widget   widget,
              Boolean  draw_as_armed );

static void 
  DrawShadow( Widget   widget,
              Boolean  armed );

static void 
  GetDrawingGC( Widget  widget );

static void
  SetComponentDimensions( Widget  widget );

static void
  SetComponentPositions( Widget  widget );


/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

/* This initialization has to be done after the methods have been declared. */
XitRwPixButtonClassRec xitRwPixButtonClassRec = {

  { /* Core class fields. */
    /* superclass */                  (WidgetClass) &xmPrimitiveClassRec,
    /* class_name */                  "XitRwPixButton",
    /* widget_size */                 sizeof( XitRwPixButtonRec ),
    /* class_initialize */            ClassInitialize,
    /* class_part_initialize */       NULL,
    /* class_inited */                False,
    /* initialize */                  Initialize,
    /* initialize_hook */             NULL,
    /* realize */                     XtInheritRealize,
    /* actions */                     actionsList,
    /* num_actions */                 XtNumber( actionsList ),
    /* resources */                   resources,
    /* num_resources */               XtNumber( resources ),
    /* xrm_class */                   NULLQUARK,
    /* compress_motion */             True,
    /* compress_exposure */           True,
    /* compress_enterleave */         True,
    /* visible_interest */            False,
    /* destroy */                     Destroy,
    /* resize */                      Resize,
    /* expose */                      ExposeIt,
    /* set_values */                  SetValues,
    /* set_values_hook */             NULL,
    /* set_values_almost */           XtInheritSetValuesAlmost,
    /* get_values_hook */             NULL,
    /* accept_focus */                NULL,
    /* version */                     XtVersion,
    /* callback_private */            NULL,
    /* tm_table */                    defaultTranslations,
    /* query_geometry */              QueryGeometry,
    /* display_accelerator */         XtInheritDisplayAccelerator,
    /* extension */                   NULL
  },
  { /* Primitive class part. */
    /* border_highlight */            (XtWidgetProc) _XtInherit,
    /* border_unhighlight */          (XtWidgetProc) _XtInherit,
    /* translations */                NULL,
#if XmVersion < 1002
    /* arm_and_activate */            (XmArmAndActivate) ArmAndActivate,
#else
    /* arm_and_activate */            (XtActionProc) ArmAndActivate,
#endif
    /* get_resources */               NULL,
    /* num_get_resources */           0,
    /* extension */                   NULL
  },
  { /* PixBut button class part. */
    /* extension */                   NULL
  },

}; /* XitRwPixButtonClassRec */


/* Class record pointer. */
WidgetClass 
  xitRwPixButtonWidgetClass = (WidgetClass) &xitRwPixButtonClassRec;


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

static void
  Activate( Widget  w,
            XEvent  *event )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;
  XmAnyCallbackStruct   callback_info;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

  /* Event outside window? */
  if( ((event->type == ButtonPress) || 
       (event->type == ButtonRelease)) && 
      ((event->xbutton.x > cb -> core.width) ||
       (event->xbutton.y > cb -> core.height )) )
    return;


  /* Callback. */
  if( cb -> pixBut.activate_callback != NULL ) {

    /* Prepare callback data. */
    callback_info.reason = XmCR_ACTIVATE;
    callback_info.event  = event;

    /* Call the activate callback. */
    XtCallCallbacks( (Widget) cb, XmNactivateCallback,
                     (XtPointer) &callback_info );

  }

  return;

} /* Activate */


/*---------------------------------------------------------------------------*/

static void
  Arm( Widget  w,
       XEvent  *event )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;
  XmAnyCallbackStruct   callback_info;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

  /* Grab traversal. */
  /* Grab focus. */
  XmProcessTraversal( w, XmTRAVERSE_CURRENT );

  cb -> pixBut.armed = True;
  DrawButton( (Widget) cb, True );

  if( cb -> pixBut.arm_callback != NULL ) {
    XFlush( XtDisplay( (Widget) cb ) );
    
    /* Prepare callback data. */
    callback_info.reason = XmCR_ARM;
    callback_info.event  = event;

    XtCallCallbacks( (Widget) cb, XmNarmCallback, (XtPointer) &callback_info );
  }


  return;

} /* Arm */


/*---------------------------------------------------------------------------*/

static void
  ArmAndActivate( Widget   w,
		  XEvent   *event )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;

  /* Code. */
  cb = (XitRwPixButtonWidget) w;

  /* Activate called by keyboard event. Arm during callback. */
  Arm( w, event );

  Activate( w, event );

  /* Disarm and draw it. */
  cb -> pixBut.armed = False;
  DrawButton( (Widget) cb, False );

  XFlush( XtDisplay( (Widget) cb ));


  return;

} /* ArmAndActivate */


/*---------------------------------------------------------------------------*/

static void
  ClassInitialize()
{

  /* Code. */

  /* Convert quarks for comparison. */
  quark_bottom = XrmStringToQuark( "BOTTOM" );
  quark_left   = XrmStringToQuark( "LEFT" );
  quark_right  = XrmStringToQuark( "RIGHT" );
  quark_top    = XrmStringToQuark( "TOP" );

  /* Register type converter for XitRwCPixmapPosition. */
  XtAddConverter( XmRString, XmRPixmapPosition, ConvertToPixmapPosition, 
                  NULL, 0 );


  return;

} /* ClassInitialize */


/*---------------------------------------------------------------------------*/

static void
  ConvertToPixmapPosition( XrmValue  *args,
                           Cardinal  *num_args,
                           XrmValue  *from,
                           XrmValue  *to )
{

  /* Variables. */
  XrmQuark    quark_resource;
  static int  pixmap_value;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToPixmapPosition", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );
 
  /* Convert argument. */
  quark_resource = XrmStringToQuark( from -> addr );

  if( quark_resource == quark_bottom )
    pixmap_value = XitRwBOTTOM;
  else if( quark_resource == quark_left )
    pixmap_value = XitRwLEFT;
  else if( quark_resource == quark_right )
    pixmap_value = XitRwRIGHT;
  else if( quark_resource == quark_top )
    pixmap_value = XitRwTOP;
  else {
    XtWarningMsg( "wrongParameters", "ConvertToPixmapPosition", 
      "XtToolkitError", "Cannot convert to type pixmapPosition",
      (String *) NULL, (Cardinal *) 0 );
    pixmap_value = XitRwLEFT;
  }

  (*to).size = sizeof( int );
  (*to).addr = (XtPointer) &pixmap_value;


  return;

} /* ConvertToPixmapPosition */


/*---------------------------------------------------------------------------*/

static void
  DesiredDimensions( Widget     widget,
                     Boolean    override_no_recompute,
                     Dimension  *width,
                     Dimension  *height )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  if( cb -> pixBut.user_set_width || 
      ( !cb -> pixBut.recompute_size && !override_no_recompute ) )
    *width = cb -> core.width;

  /* Adapt size to contents. */
  else {
    /* Basic margins are always counted. */
    *width  = (Dimension)( 2 * cb -> pixBut.margin_width +
			   cb -> pixBut.left_margin +
			   cb -> pixBut.right_margin +
			   2 * (int) cb -> primitive.shadow_thickness +
			   2 * (int) cb -> primitive.highlight_thickness );

    switch( cb -> pixBut.pixmap_position ) {

      case XitRwLEFT:
      case XitRwRIGHT:
	/* Horizontal organization. */
	*width  += (Dimension) ( cb -> pixBut.spacing +
				 cb -> pixBut.pixmap_width +
				 cb -> pixBut.string_width );
	break;

      case XitRwTOP:
      case XitRwBOTTOM:
	/* Vertical organization. */
	*width  += max( (Dimension) cb -> pixBut.string_width, 
			(Dimension) cb -> pixBut.pixmap_width );
	break;

    } /* switch */
  }

  if( cb -> pixBut.user_set_height || 
      ( !cb -> pixBut.recompute_size && !override_no_recompute ) )
    *height = cb -> core.height;

  /* Adapt size to contents. */
  else {
    *height = (Dimension)( 2 * cb -> pixBut.margin_height +
			   cb -> pixBut.top_margin + 
			   cb -> pixBut.bottom_margin +
			   2 * (int) cb -> primitive.shadow_thickness +
			   2 * (int) cb -> primitive.highlight_thickness );

    switch( cb -> pixBut.pixmap_position ) {

      case XitRwLEFT:
      case XitRwRIGHT:
	/* Horizontal organization. */
	*height += max( (Dimension) cb -> pixBut.string_height, 
			(Dimension) cb -> pixBut.pixmap_height );
	break;

      case XitRwTOP:
      case XitRwBOTTOM:
	/* Vertical organization. */
	*height += (Dimension) ( cb -> pixBut.spacing + 
				 cb -> pixBut.pixmap_height +
				 cb -> pixBut.string_height );
	break;

    } /* switch */

  }


  return;

} /* DesiredDimensions */


/*---------------------------------------------------------------------------*/

static void 
  Destroy( Widget  widget )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  /* Free gc's */
  if( cb -> pixBut.gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.gc );

  if( cb -> pixBut.clear_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.clear_gc );

  if( cb -> pixBut.select_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.select_gc );

  if( cb -> pixBut.select_clear_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.select_clear_gc );

  if( cb -> pixBut.insensitive_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.insensitive_gc );

  /* Free allocated string. */
  if( cb -> pixBut.label != NULL )
    XmStringFree( cb -> pixBut.label );

  /* Free insensitive pixmap. */
  if( cb -> pixBut.insensitive_pixmap != (XID) NULL )
    XFreePixmap( XtDisplay( (Widget) cb ), cb -> pixBut.insensitive_pixmap );


  return;

} /* Destroy */


/*---------------------------------------------------------------------------*/

static void
  Disarm( Widget    w,
	  XEvent    *event,
	  String    *params,
	  Cardinal  num_params )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;
  XmAnyCallbackStruct   callback_info;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

  if( cb -> pixBut.armed ) {
    cb -> pixBut.armed = False;
    DrawButton( (Widget) cb, False );
  }

  if( cb -> pixBut.disarm_callback != NULL ) {
    XFlush( XtDisplay( (Widget) cb ) );
    
    /* Prepare callback data. */
    callback_info.reason = XmCR_DISARM;
    callback_info.event  = event;

    XtCallCallbacks( (Widget) cb, XmNdisarmCallback,
                     (XtPointer) &callback_info );
  }


  return;

} /* Disarm */


/*---------------------------------------------------------------------------*/

static void 
  DrawButton( Widget   widget,
              Boolean  draw_as_armed )
{

  /* Variables. */
  GC                    clear_gc;
  GC                    normal_gc;
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  clear_gc  = ( draw_as_armed ? cb -> pixBut.select_clear_gc :
                                cb -> pixBut.clear_gc );
  normal_gc = ( draw_as_armed ? cb -> pixBut.select_gc :
                                cb -> pixBut.gc );


  /* Clear the button for redraw. */
  XFillRectangle( XtDisplay( widget ), cb -> core.window, 
                  clear_gc, 0, 0,
                  (unsigned int) cb -> core.width,
                  (unsigned int) cb -> core.height );

  /* Draw pixmap. */
  if( cb -> pixBut.pixmap != XmUNSPECIFIED_PIXMAP )
    XCopyArea( XtDisplay( widget ), cb -> pixBut.pixmap,
               cb -> core.window, normal_gc, 0, 0,
               cb -> pixBut.pixmap_width,
               cb -> pixBut.pixmap_height,
               cb -> pixBut.x_pixmap,
               cb -> pixBut.y_pixmap );

  /* Draw string. */
  if( cb -> pixBut.label != NULL )
    XmStringDraw( XtDisplay( widget ),
                  cb -> core.window, cb -> pixBut.font_list,
		  cb -> pixBut.label, normal_gc,
		  cb -> pixBut.x_string, cb -> pixBut.y_string,
		  (Dimension) cb -> pixBut.string_width,
                  XmALIGNMENT_BEGINNING, 
		  XmSTRING_DIRECTION_L_TO_R, NULL );

  /* Insensitive state. Stipple it. */
  if( ! cb -> core.sensitive )
    XFillRectangle( XtDisplay( widget ), cb -> core.window,
                    cb -> pixBut.insensitive_gc,
                    0, 0,
                    (unsigned int) cb -> core.width,
                    (unsigned int) cb -> core.height );

  /* Shadows. */
  DrawShadow( (Widget) cb, draw_as_armed ); 

  /* Highlight? */
  if( cb -> primitive.highlighted )
    _XmHighlightBorder( (Widget) cb );
  else
    _XmUnhighlightBorder( (Widget) cb );


} /* DrawButton */


/*---------------------------------------------------------------------------*/

static void 
  DrawShadow( Widget   widget,
              Boolean  armed )
{

  /* Variables. */
  Boolean               shadow_in;
  GC                    bottom_gc;
  GC                    top_gc;
  XitRwPixButtonWidget  cb;


  /* Code. */

  if( ! xitIsRealized( widget ) )
    return;

  cb = (XitRwPixButtonWidget) widget;

  if( cb -> primitive.shadow_thickness == 0 )
    return;


  /* Find out shadow type. */
  shadow_in = (( cb -> pixBut.shadow_type == XmSHADOW_IN )  && !armed ) ||
              (( cb -> pixBut.shadow_type == XmSHADOW_OUT ) && armed );

  if( shadow_in ) {
    top_gc    = cb -> primitive.bottom_shadow_GC;
    bottom_gc = cb -> primitive.top_shadow_GC;
  } else {
    top_gc    = cb -> primitive.top_shadow_GC;
    bottom_gc = cb -> primitive.bottom_shadow_GC;
  }

  _XmDrawShadow( XtDisplay( widget ), XtWindow( widget ),
                 top_gc, bottom_gc,
                 cb -> primitive.shadow_thickness,
                 cb -> primitive.highlight_thickness,
                 cb -> primitive.highlight_thickness,
                 (int) cb -> core.width - 
                       2 * cb -> primitive.highlight_thickness,
                 (int) cb -> core.height - 
                       2 * cb -> primitive.highlight_thickness );

  return;

} /* DrawShadow */


/*---------------------------------------------------------------------------*/

static void
  Enter( Widget    w,
	 XEvent    *event,
	 String    *params,
	 Cardinal  num_params )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

#if XmVersion > 1001
  _XmPrimitiveEnter( (Widget) cb, event, NULL, NULL );
#else
  _XmPrimitiveEnter( (XmPrimitiveWidget) cb, event );
#endif

  /* Only bother to redraw if the widget is armed. */
  if( cb -> pixBut.armed )
    DrawButton( (Widget) cb, True );


  return;

} /* Enter */


/*---------------------------------------------------------------------------*/

static void
  ExposeIt( Widget  widget,
            XEvent  *event,
            Region  region )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  DrawButton( (Widget) cb, cb -> pixBut.armed );


  return;

} /* ExposeIt */


/*---------------------------------------------------------------------------*/

static void
  PixButFocusIn( Widget   w,
                 XEvent   *event,
                 String   *params,
                 Cardinal num_params )
{

  XitRwPixButtonWidget   cb;

  /* Code. */

  cb = (XitRwPixButtonWidget) w;

#if XmVersion > 1001
  _XmPrimitiveFocusIn( (Widget) w, event, NULL, NULL );
#else
  _XmPrimitiveFocusIn( (XmPrimitiveWidget) w, event );
#endif

  DrawButton( (Widget) cb, cb -> pixBut.armed );


  return;

} /* PixButFocusIn */


/*---------------------------------------------------------------------------*/

static void
  PixButFocusOut( Widget   w,
                  XEvent   *event,
                  String   *params,
                  Cardinal num_params )
{

  XitRwPixButtonWidget   cb;

  /* Code. */

  cb = (XitRwPixButtonWidget) w;

  DrawButton( (Widget) cb, cb -> pixBut.armed );

#if XmVersion > 1001
  _XmPrimitiveFocusOut( (Widget) w, event, NULL, NULL );
#else
  _XmPrimitiveFocusOut( (XmPrimitiveWidget) w, event );
#endif


  return;

} /* PixButFocusOut */


/*---------------------------------------------------------------------------*/

static void 
  GetDrawingGC( Widget  widget )
{

  /* Variables. */
  XFontStruct           *font;
  XGCValues             gc_values;
  XitRwPixButtonWidget  cb;
  Pixel                 select_color;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  /* Get font for GC, Take first font in list. */
  {
    XmFontContext    context;
    XmStringCharSet  char_set;

    (void) XmFontListInitFontContext( &context, cb -> pixBut.font_list );
    (void) XmFontListGetNextFont( context, &char_set, &font );
    XmFontListFreeFontContext( context );
    XtFree( char_set );
  } /* block */

  /* Get the background color for armed state. */
  {
    Pixel   dummy;

    XmGetColors( XtScreen( (Widget) cb ), 
		 DefaultColormapOfScreen( XtScreen( (Widget) cb )),
		 cb -> core.background_pixel, &dummy, &dummy, &dummy,
  		 &select_color );
  }

  /* Use a shareable GC. */
  if( cb -> pixBut.gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.gc );

  gc_values.foreground = cb -> primitive.foreground;
  gc_values.background = cb -> core.background_pixel;
  gc_values.font       = font -> fid;

  cb -> pixBut.gc = XtGetGC( (Widget) cb,
                             GCForeground | GCBackground | GCFont,
                             &gc_values );


  /* The clear_gc for clearing the window. */
  if( cb -> pixBut.clear_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.clear_gc );

  gc_values.foreground = cb -> core.background_pixel;
  gc_values.background = cb -> primitive.foreground;
  gc_values.fill_style = FillSolid;

  cb -> pixBut.clear_gc = XtGetGC( (Widget) cb, 
                                   GCForeground | GCBackground | GCFillStyle,
                                   &gc_values );

  /* The select_gc for drawing the button in its armed state. */
  if( cb -> pixBut.select_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.select_gc );

  gc_values.foreground   = cb -> primitive.foreground;
  gc_values.background   = select_color;
  gc_values.font         = font -> fid;  
  cb -> pixBut.select_gc = XtGetGC( (Widget) cb, 
                                  GCForeground | GCBackground | GCFont,
                                  &gc_values );

  /* The select_clear_gc for clearing the pixmap when armed. */
  if( cb -> pixBut.select_clear_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.select_clear_gc );

  gc_values.foreground = select_color;
  gc_values.background = cb -> primitive.foreground;
  gc_values.fill_style = FillSolid;
  cb -> pixBut.select_clear_gc = XtGetGC( (Widget) cb, 
                                           GCForeground | GCBackground,
                                           &gc_values );

  /* The insensitive GC for insensitive state. */
  if( cb -> pixBut.insensitive_gc != NULL )
    XtReleaseGC( (Widget) cb, cb -> pixBut.insensitive_gc );

  gc_values.foreground = cb -> core.background_pixel;
  gc_values.stipple    = cb -> pixBut.insensitive_pixmap;
  gc_values.fill_style = FillStippled;

  cb -> pixBut.insensitive_gc =
    XtGetGC( (Widget) cb,
             GCForeground | GCStipple | GCFillStyle,
             &gc_values );

  return;

} /* GetDrawingGC */


/*---------------------------------------------------------------------------*/

static void
  Initialize( Widget    treq,
              Widget    tnew,
              ArgList   args,
              Cardinal  *num_args )
{

  /* Variables. */
  XitRwPixButtonWidget  new;
  XmString              tmp_label;


  /* Code. */

  new = (XitRwPixButtonWidget) tnew;

  /* Reset all private fields. */
  new -> pixBut.gc                = NULL;
  new -> pixBut.clear_gc          = NULL;
  new -> pixBut.select_gc         = NULL;
  new -> pixBut.select_clear_gc   = NULL;
  new -> pixBut.insensitive_gc    = NULL;
  new -> pixBut.pixmap_width      = 0;
  new -> pixBut.pixmap_height     = 0;
  new -> pixBut.string_width      = 0;
  new -> pixBut.string_height     = 0;
  new -> pixBut.x_pixmap          = 0;
  new -> pixBut.y_pixmap          = 0;
  new -> pixBut.x_string          = 0;
  new -> pixBut.y_string          = 0;
  new -> pixBut.user_set_width    = False;
  new -> pixBut.user_set_height   = False;
  new -> pixBut.armed             = False;

  /* Copy label string. */
  if( new -> pixBut.label != NULL ){

    tmp_label           = XmStringCopy( new -> pixBut.label );
    new -> pixBut.label = tmp_label;
  }

  /* Check the dimensions of the stuff to be displayed. */
  SetComponentDimensions( (Widget) new );

  /* Get our initial size. */
  DesiredDimensions( (Widget) new, True, &new -> core.width, 
                                         &new -> core.height );

  /* Avoid trying to create a widget with zero width and/or height. */
  if( new -> core.width <= 0 )
    new -> core.width = 10;
  if( new -> core.height <= 0 )
    new -> core.height = 10;

  /* Lay out components within the window. */
  SetComponentPositions( (Widget) new );

  /* Create insensitive pixmap. */
  new -> pixBut.insensitive_pixmap =
    XCreateBitmapFromData( XtDisplay( (Widget) new ),
                           RootWindowOfScreen( XtScreen( (Widget) new )),
                           (char *) insens_bits,
                           INSENS_WIDTH, INSENS_HEIGHT );

  /* Create GC for drawing. */
  GetDrawingGC( (Widget) new );


  return;

} /* Initialize */


/*---------------------------------------------------------------------------*/

static void
  Leave( Widget    w,
	 XEvent    *event,
	 String    *params,
	 Cardinal  num_params )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

#if XmVersion > 1001
  _XmPrimitiveLeave( (Widget) cb, event, NULL, NULL );
#else
  _XmPrimitiveLeave( (XmPrimitiveWidget) cb, event );
#endif

  /* Only bother to redraw if the widget is armed. */
  if( cb -> pixBut.armed )
    DrawButton( (Widget) cb, False );


  return;

} /* Leave */


/*---------------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  /* Suggest the calculated width and height. */
  answer -> request_mode = CWWidth | CWHeight;

  DesiredDimensions( (Widget) cb, False, &answer -> width, &answer -> height );
  
  /* Check if our calculated dimensions correspond to the propsed ones. */
  if( (( proposed -> request_mode & ( CWWidth | CWHeight )) == 
         ( CWWidth | CWHeight )) &&
     ( proposed -> width  == answer -> width ) &&
     ( proposed -> height == answer -> height ) )
    return( XtGeometryYes );

  else if( ( answer -> width  == cb -> core.width ) &&
           ( answer -> height == cb -> core.height ) )
    return( XtGeometryNo );

  else
    return( XtGeometryAlmost );


} /* QueryGeometry */


/*---------------------------------------------------------------------------*/

static void
  Resize( Widget  widget )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  /* When this method is called, the parameter widget structure contains the
     new values for width, height and border width. We must obey. */

  /* Calculate positions of the components. */
  SetComponentPositions( (Widget) cb );

  /* X will generate an expose event and we can draw the button in the
     expose method. */

  return;


} /* Resize */


/*---------------------------------------------------------------------------*/

static void
  SetComponentDimensions( Widget  widget )
{

  /* Variables. */
  int                   x;
  int                   y;
  unsigned int          border_width;
  unsigned int          depth;
  unsigned int         pixmap_height;
  unsigned int         pixmap_width;
  Dimension             string_height;
  Dimension             string_width;
  Window                root_window;
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  /* Get the dimensions of the stuff that shall be displayed. */

  /* Pixmap. */
  if( cb -> pixBut.pixmap != XmUNSPECIFIED_PIXMAP )
    XGetGeometry( XtDisplay( widget ), cb -> pixBut.pixmap, &root_window,
                  &x, &y, &pixmap_width, &pixmap_height, &border_width, 
                  &depth );
  else {
    pixmap_height = 0;
    pixmap_width  = 0;
  }


  /* String. */
  if( cb -> pixBut.label != NULL ){
    string_height = XmStringHeight( cb -> pixBut.font_list,
                                    cb -> pixBut.label );
    string_width  = XmStringWidth(  cb -> pixBut.font_list,
                                    cb -> pixBut.label );
  } else {
    string_height = 0;
    string_width  = 0;
  }

  cb -> pixBut.pixmap_height = (int) pixmap_height;
  cb -> pixBut.pixmap_width  = (int) pixmap_width;
  cb -> pixBut.string_height = (int) string_height;
  cb -> pixBut.string_width  = (int) string_width;


  return;

} /* SetComponentDimensions */


/*---------------------------------------------------------------------------*/

static void
  SetComponentPositions( Widget  widget )
{

  /* Variables. */
  int                   basic_bottom_offset;
  int                   basic_left_offset;
  int                   basic_right_offset;
  int                   basic_top_offset;
  int                   middle_x_pos;
  int                   middle_y_pos;
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) widget;

  basic_left_offset   = cb -> primitive.shadow_thickness + 
                        cb -> primitive.highlight_thickness +
                        cb -> pixBut.margin_width +
                        cb -> pixBut.left_margin;

  basic_top_offset    = cb -> primitive.shadow_thickness + 
                        cb -> primitive.highlight_thickness +
                        cb -> pixBut.margin_height +
                        cb -> pixBut.top_margin;

  basic_right_offset  = cb -> primitive.shadow_thickness + 
                        cb -> primitive.highlight_thickness +
                        cb -> pixBut.margin_width +
                        cb -> pixBut.right_margin;

  basic_bottom_offset = cb -> primitive.shadow_thickness + 
                        cb -> primitive.highlight_thickness +
                        cb -> pixBut.margin_height +
                        cb -> pixBut.bottom_margin;

  middle_x_pos  = (( cb -> core.width - basic_left_offset - 
                     basic_right_offset ) / 2 ) + basic_left_offset;

  middle_y_pos  = (( cb -> core.height - basic_top_offset - 
                     basic_bottom_offset ) / 2 ) + basic_top_offset;


  /* X position. */
  switch( cb -> pixBut.pixmap_position ){

    case XitRwLEFT:
      /* Horizontal layout. Pixmap to the left. */

      switch( cb -> pixBut.alignment ){
        case XmALIGNMENT_BEGINNING:
	  cb -> pixBut.x_pixmap = basic_left_offset;
	  cb -> pixBut.x_string = cb -> pixBut.x_pixmap +
                                  cb -> pixBut.pixmap_width +
			          cb -> pixBut.spacing;
          break;

        case XmALIGNMENT_END:
          cb -> pixBut.x_string = cb -> core.width - basic_right_offset - 
                                  cb -> pixBut.string_width;
          cb -> pixBut.x_pixmap = cb -> pixBut.x_string -
                                  cb -> pixBut.spacing -
                                  cb -> pixBut.pixmap_width;
          break;

        case XmALIGNMENT_CENTER:
          cb -> pixBut.x_pixmap = middle_x_pos - ( cb -> pixBut.pixmap_width + 
                                  cb -> pixBut.spacing +
                                  cb -> pixBut.string_width ) / 2;
          cb -> pixBut.x_string = cb -> pixBut.x_pixmap +
                                  cb -> pixBut.pixmap_width +
                                  cb -> pixBut.spacing;
          break;
      }

      break;

    case XitRwRIGHT:
      /* Horizontal layout. Pixmap to the right. */

      switch( cb -> pixBut.alignment ){
        case XmALIGNMENT_BEGINNING:
	  cb -> pixBut.x_string = basic_left_offset;
	  cb -> pixBut.x_pixmap = cb -> pixBut.x_string +
                                  cb -> pixBut.string_width +
			          cb -> pixBut.spacing;
          break;

        case XmALIGNMENT_END:
          cb -> pixBut.x_pixmap = cb -> core.width - basic_right_offset - 
                                  cb -> pixBut.pixmap_width;
          cb -> pixBut.x_string = cb -> pixBut.x_pixmap -
                                  cb -> pixBut.spacing -
                                  cb -> pixBut.string_width;
          break;

        case XmALIGNMENT_CENTER:
          /* Center the components. */
          cb -> pixBut.x_string = middle_x_pos - 
                                  (cb -> pixBut.pixmap_width + 
                                   cb -> pixBut.spacing + 
                                   cb -> pixBut.string_width) / 2;
          cb -> pixBut.x_pixmap = cb -> pixBut.x_string +
                                  cb -> pixBut.string_width +
                                  cb -> pixBut.spacing;
          break;
      }

      break;

    case XitRwTOP:
    case XitRwBOTTOM:
      /* Vertical organization. */

      /* Components should be centered in the horizontal dimension. */
      cb -> pixBut.x_string = middle_x_pos - (cb -> pixBut.string_width / 2);
      cb -> pixBut.x_pixmap = middle_x_pos - (cb -> pixBut.pixmap_width / 2);

      break;

  }

  /* Y position. */
  switch( cb -> pixBut.pixmap_position ){

    case XitRwTOP:
      /* Vertical layout. Pixmap over string. */

      switch( cb -> pixBut.alignment ){
        case XmALIGNMENT_BEGINNING:
	  cb -> pixBut.y_pixmap = basic_top_offset;
	  cb -> pixBut.y_string = cb -> pixBut.y_pixmap + 
                                  cb -> pixBut.pixmap_height +
			          cb -> pixBut.spacing;
          break;

        case XmALIGNMENT_END:
          /* As low as possible. */
          cb -> pixBut.y_string = cb -> core.height - basic_bottom_offset - 
                                  cb -> pixBut.string_height;
          cb -> pixBut.y_pixmap = cb -> pixBut.y_string - 
                                  cb -> pixBut.spacing -
                                  cb -> pixBut.pixmap_height;
          break;

        case XmALIGNMENT_CENTER:
          /* Center the components. */
          cb -> pixBut.y_pixmap = middle_y_pos -
                                  (cb -> pixBut.pixmap_height + 
                                   cb -> pixBut.spacing +
                                   cb -> pixBut.string_height) / 2;
          cb -> pixBut.y_string = cb -> pixBut.y_pixmap + 
                                  cb -> pixBut.pixmap_height +
                                  cb -> pixBut.spacing;
          break;
      }

      break;

    case XitRwBOTTOM:
      /* Vertical layout. Pixmap under string. */

      switch( cb -> pixBut.alignment ){
        case XmALIGNMENT_BEGINNING:
	  cb -> pixBut.y_string = basic_top_offset;
	  cb -> pixBut.y_pixmap = cb -> pixBut.y_string +
                                  cb -> pixBut.string_height +
			          cb -> pixBut.spacing;
          break;

        case XmALIGNMENT_END:
          cb -> pixBut.y_pixmap = cb -> core.height - basic_bottom_offset - 
                                  cb -> pixBut.pixmap_height;
          cb -> pixBut.y_string = cb -> pixBut.y_pixmap - 
                                  cb -> pixBut.spacing -
                                  cb -> pixBut.string_height;
          break;

        case XmALIGNMENT_CENTER:
          cb -> pixBut.y_string = middle_y_pos -
                                  (cb -> pixBut.pixmap_height + 
                                   cb -> pixBut.spacing +
                                   cb -> pixBut.string_height) / 2;
          cb -> pixBut.y_pixmap = cb -> pixBut.y_string +
                                  cb -> pixBut.string_height +
                                  cb -> pixBut.spacing;
          break;
      }

      break;

    case XitRwLEFT:
    case XitRwRIGHT:
      /* Horizontal organization. */
      if( cb -> pixBut.string_height > cb -> core.height ||
          cb -> pixBut.opposite_alignment == XmALIGNMENT_BEGINNING )
        cb -> pixBut.y_string = basic_top_offset;
      else
        cb -> pixBut.y_string = middle_y_pos - 
                                (cb -> pixBut.string_height / 2);

      cb -> pixBut.y_pixmap = middle_y_pos - (cb -> pixBut.pixmap_height / 2);

      break;

  }


  return;

} /* SetComponentPositions */


/*---------------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget    current,
             Widget    request,
             Widget    new,
             ArgList   args,
             Cardinal  *num_args )
{

  /* Variables. */
  Boolean               new_size;
  Boolean               redraw = False;
  Dimension             desired_height;
  Dimension             desired_width;
  XitRwPixButtonWidget  curW;
  XitRwPixButtonWidget  newW;
  XmString              tmp_label;


  /* Code. */

  curW = (XitRwPixButtonWidget) current;
  newW = (XitRwPixButtonWidget) new;

  /* Initialize. */
  new_size = False;

  if( newW -> core.width != curW -> core.width ) {
    if( newW -> core.width == 0 ) {
      if( newW -> pixBut.user_set_width ) {
        newW -> pixBut.user_set_width = False;
        new_size = True;
      } else {
        newW -> core.width = curW -> core.width;
      }
    } else {
      newW -> pixBut.user_set_width = True;
    }
  }

  if( newW -> core.height != curW -> core.height ) {
    if( newW -> core.height == 0 ) {
      if( newW -> pixBut.user_set_height ) {
        newW -> pixBut.user_set_height = False;
        new_size = True;
      } else {
        newW -> core.height = curW -> core.height;
      }
    } else {
      newW -> pixBut.user_set_height = True;
    }
  }

  if( newW -> pixBut.font_list != curW -> pixBut.font_list )
    new_size = True;


  if( newW -> pixBut.label != curW -> pixBut.label ) {
    if( curW -> pixBut.label != NULL )
      XmStringFree( curW -> pixBut.label );

    tmp_label            = XmStringCopy( newW -> pixBut.label );
    newW -> pixBut.label = tmp_label;

    new_size = True;
  }


  if( newW -> pixBut.pixmap != curW -> pixBut.pixmap ) 
    new_size = True;


  /* Check if any resource values were changed that may affect other fields
     or require that the picture be redrawn. */

  if( ( newW -> pixBut.top_margin    != curW -> pixBut.top_margin )    ||
      ( newW -> pixBut.left_margin   != curW -> pixBut.left_margin )   ||
      ( newW -> pixBut.right_margin  != curW -> pixBut.right_margin )  ||
      ( newW -> pixBut.bottom_margin != curW -> pixBut.bottom_margin ) ||
      ( newW -> pixBut.margin_width  != curW -> pixBut.margin_width )  ||
      ( newW -> pixBut.margin_height != curW -> pixBut.margin_height ) ||
      ( newW -> pixBut.spacing       != curW -> pixBut.spacing ) )
    new_size = True;

  if( newW -> core.sensitive != curW -> core.sensitive )
    redraw = True;

  /* Color change affects only gc and redraw. */
  if( ( newW -> primitive.foreground  != curW -> primitive.foreground )  ||
      ( newW -> core.background_pixel != curW -> core.background_pixel ) ||
      ( newW -> pixBut.font_list      != curW -> pixBut.font_list ) ) {
    GetDrawingGC( (Widget) new );
    redraw = True;
  }

  /* Check if we want to resize ourselves. */
  if( new_size ) {

    redraw = True;

    /* Size request must be based on new values. */
    SetComponentDimensions( (Widget) new );

    DesiredDimensions( (Widget) new, False, &desired_width, &desired_height );

    SetComponentPositions( (Widget) new );


    /* Only request new size if really needed. */
    if( desired_width  != curW -> core.width ||
        desired_height != curW -> core.height ||
        newW -> core.height == 0 ||
        newW -> core.width == 0 ) {

      new -> core.width  = desired_width;
      new -> core.height = desired_height;

      /* Should not change these according to the Xt documentation, but try. */
      request -> core.width  = desired_width;
      request -> core.height = desired_height;

    }

  } /* if */

  /* Redraw ourselves? */
  return( redraw );


} /* SetValues */


/*---------------------------------------------------------------------------*/

static void
  TraverseNextTabGroup( Widget   w,
                        XEvent   *event,
	                String   *params,
	                Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_NEXT_TAB_GROUP );

  
  return;

} /* TraverseNextTabGroup */

/*---------------------------------------------------------------------------*/

static void
  TraversePrevTabGroup( Widget   w,
                        XEvent   *event,
	                String   *params,
	                Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_PREV_TAB_GROUP );


  return;

} /* TraversePrevTabGroup */

/*---------------------------------------------------------------------------*/

static void
  TraverseDown( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_DOWN );


  return;

} /* TraverseDown */

/*---------------------------------------------------------------------------*/

static void
  TraverseHome( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_HOME );


  return;

} /* TraverseHome */

/*---------------------------------------------------------------------------*/

static void
  TraverseLeft( Widget   w,
                XEvent   *event,
	        String   *params,
	        Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_LEFT );


  return;

} /* TraverseLeft */

/*---------------------------------------------------------------------------*/

static void
  TraverseRight( Widget   w,
                 XEvent   *event,
	         String   *params,
	         Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_RIGHT );


  return;

} /* TraverseRight */

/*---------------------------------------------------------------------------*/

static void
  TraverseUp( Widget   w,
              XEvent   *event,
	      String   *params,
	      Cardinal num_params )
{
  /* Code. */

  XmProcessTraversal( w, XmTRAVERSE_UP );


  return;

} /* TraverseUp */

/*---------------------------------------------------------------------------*/

static void
  Unmap( Widget   w,
         XEvent   *event,
         String   *params,
         Cardinal num_params )
{

  /* Variables. */
  XitRwPixButtonWidget  cb;


  /* Code. */

  cb = (XitRwPixButtonWidget) w;

#if XmVersion > 1001
  _XmPrimitiveUnmap( (Widget) cb, event, NULL, NULL );
#else
  _XmPrimitiveUnmap( (XmPrimitiveWidget) cb, event );
#endif

  /* We cannot allow the widget to stay armed if it disappears. */
  cb -> pixBut.armed = False;


  return;

} /* Unmap */


/*---------------------------------------------------------------------------*/

Boolean
  xitRwPixButIsArmed( Widget  w )
{

  /* Code. */


  return( ((XitRwPixButtonWidget) w) -> pixBut.armed );

} /* xitRwPixButIsArmed */
