/*----------------------------------------------------------------------------
--
--  Module:           xtmCalDb
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Routines for the calendar database.
--
--  Filename:         xtmCalDb.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-07-18
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCalDb.c, Version: 1.1, Date: 95/02/18 15:51:57";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "xtmGlobal.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmCalDb.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Prefix to use for shadow entries. */
#define  SHADOW_PREFIX  "@_"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Calendar database description. */
typedef struct {

  /* ID to use for shadow entries. */
  int  shadow_id;

  /* Calendars from the customize file. */
  LST_DESC_TYPE  cal_list;

  /* Shadow calendars. */
  LST_DESC_TYPE  shadow_cal_list;

} CAL_DB, *CAL_DB_REF;


/* Shadow record. */
typedef struct {

  /* Shadow name. */
  char  shadow_name[ 10 ];

  /* Calendar record. */
  XTM_CD_CAL_INFO  db_info;

} SHADOW_REC, *SHADOW_REC_PTR;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCalDb";



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  clearCalEntry( void  *data_ref );

static void
  clearShadowEntry( void  *data_ref );

static void
  copyIncludeDb( XTM_CD_HANDLE     cal_handle,
                 Boolean           only_if_read,
                 XTM_CD_INCL_CALS  *from_incl_ref,
                 XTM_CD_INCL_CALS  **to_incl_ref );

static LST_COMPARE
  searchDbName( XTM_CD_CAL_INFO  *db_ref,
                char             *db_name );

static LST_COMPARE
  searchShadowDbName( SHADOW_REC  *db_ref,
                      char        *db_name );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmCdAddDatabase( XTM_CD_HANDLE  cal_handle,
                    char           *cal_db_name,
                    char           *location,
                    UINT32         flags )
{

  /* Variables. */
  Boolean          status;
  CAL_DB_REF       cal_db_ref;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  if( cal_handle == NULL )
    return;

  cal_db_ref = (CAL_DB_REF) cal_handle;


  strcpy( db_info.short_name, cal_db_name );
  strcpy( db_info.directory,  location );

  db_info.flags             = flags;
  db_info.incl_bg_index     = 0;
  db_info.normal_bg_index   = 0;
  db_info.win_stack_order   = 0;
  db_info.has_extern_pwd    = False;
  db_info.alarm_tags[ 0 ]   = '\0';
  db_info.extern_host[ 0 ]  = '\0';
  db_info.extern_pwd[ 0 ]   = '\0';
  db_info.extern_uid[ 0 ]   = '\0';
  db_info.extern_type[ 0 ]  = '\0';
  db_info.mail_address[ 0 ] = '\0';
  db_info.view_tags[ 0 ]    = '\0';
  db_info.include           = NULL;


  /* Database operations possible. */
  xtmDbCheckDbOperations( location, True, &db_info.operations );


  /* Owner? */
  {
    int          ret;
    struct stat  file_status;

    db_info.owner_uid = -1;

    ret = stat( db_info.directory, &file_status );
    if( ret == 0 )
      db_info.owner_uid = file_status.st_uid;
  }


  /* Add the calendar entry. */
  status = xtmCdAddEntry( cal_handle, &db_info, NULL );


  return;

} /* xtmCdAddDatabase */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdAddEntry( XTM_CD_HANDLE     cal_handle,
                 XTM_CD_CAL_INFO   *db_info_ref,
                 XTM_CD_INCL_CALS  *db_incl_ref )
{

  /* Variables. */
  CAL_DB_REF  cal_db_ref;
  LST_STATUS  lst_status;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Remove any duplicates. */
  (void) xtmCdDeleteEntry( cal_handle, db_info_ref -> short_name );


  /* Any included calendars? */
  db_info_ref -> include = NULL;

  if( db_incl_ref != NULL )
    copyIncludeDb( cal_handle, False, db_incl_ref, &db_info_ref -> include );


  /* Add the entry to the end of the list. */
  lst_status = LstLinkInsertLast( cal_db_ref -> cal_list, db_info_ref );
  if( lst_status != LST_OK )
    return( False );


  return( True );

} /* xtmCdAddEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdChangeEntry( XTM_CD_HANDLE     cal_handle,
                    char              *cal_db_name,
                    XTM_CD_CAL_INFO   *new_db_info_ref,
                    XTM_CD_INCL_CALS  *new_db_incl_ref )
{

  /* Variables. */
  CAL_DB_REF       cal_db_ref;
  LST_STATUS       lst_status;
  SHADOW_REC       *shadow_rec_ref;
  XTM_CD_CAL_INFO  *db_info_ref = NULL;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Search the entry in the main list. */
  lst_status = LstLinkSearchFirst( cal_db_ref -> cal_list,
                                   (void *) cal_db_name,
                                   (EQUALS_FUNC_TYPE) searchDbName );

  if( lst_status == LST_OK ) {
    db_info_ref = (XTM_CD_CAL_INFO *) LstLinkGetCurrentRef( 
                                        cal_db_ref -> cal_list );


  /* Search the entry in the shadow list. */
  } else {
    lst_status = LstLinkSearchFirst( cal_db_ref -> shadow_cal_list,
                                     (void *) cal_db_name,
                                     (EQUALS_FUNC_TYPE) searchShadowDbName );
    if( lst_status == LST_OK ) {
      shadow_rec_ref = (SHADOW_REC *) LstLinkGetCurrentRef( 
                                        cal_db_ref -> shadow_cal_list );

      db_info_ref = &shadow_rec_ref -> db_info;
    }

  } /* if */


  if( db_info_ref == NULL )
    return( False );


  /* Any old included calendars? */
  if( db_info_ref -> include != NULL )
    SysFree( db_info_ref -> include );


  /* Copy the new entry. */
  memcpy( (void *) db_info_ref, (void *) new_db_info_ref, 
          sizeof( XTM_CD_CAL_INFO ) );


  /* Any included calendars? */
  db_info_ref -> include = NULL;

  if( new_db_incl_ref != NULL )
    copyIncludeDb( cal_handle, False, 
                   new_db_incl_ref, &db_info_ref -> include );


  return( True );

} /* xtmCdChangeEntry */


/*----------------------------------------------------------------------*/

XTM_CD_HANDLE
  xtmCdCopy( XTM_CD_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_DB_REF       cal_db_ref;
  CAL_DB_REF       new_cal_db_ref;
  LST_STATUS       lst_status;
  XTM_CD_HANDLE    new_cal_handle;
  XTM_CD_CAL_INFO  *db_ref;
  XTM_CD_CAL_INFO  *new_db_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( NULL );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Create a new calendar database. */
  new_cal_handle = xtmCdInitialize();
  if( new_cal_handle == NULL )
    return( NULL );

  new_cal_db_ref = (XTM_CD_HANDLE) new_cal_handle;


  /* Search all calendars and make a copy. */
  lst_status = LstLinkCurrentFirst( cal_db_ref -> cal_list );

  while( lst_status == LST_OK ) {

    db_ref = (XTM_CD_CAL_INFO *) 
               LstLinkGetCurrentRef( cal_db_ref -> cal_list );


    /* Copy the calendar record. */
    new_db_ref = SysNew( XTM_CD_CAL_INFO );

    memcpy( (void *) new_db_ref, (void *) db_ref, sizeof( XTM_CD_CAL_INFO ) );


    /* Included calendars? */
    if( db_ref -> include != NULL ) {
      new_db_ref -> include = SysNew( XTM_CD_INCL_CALS );

      memcpy( (void *) new_db_ref -> include, 
              (void *) db_ref -> include,
              sizeof( XTM_CD_INCL_CALS ) );
    }

    /* Calendar record. */
    (void) LstLinkInsertLast( new_cal_db_ref -> cal_list, new_db_ref );

    lst_status = LstLinkCurrentNext( cal_db_ref -> cal_list );

  } /* while */


  return( new_cal_handle );

} /* xtmCdCopy */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdCreateShadowEntry( XTM_CD_HANDLE  cal_handle,
                          char           *cal_db_name,
                          char           *shadow_db_name )
{

  /* Variables. */
  Boolean           ok;
  CAL_DB_REF        cal_db_ref;
  LST_STATUS        lst_status;
  SHADOW_REC        db_shadow_info;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  *shadow_db_name = '\0';

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Fetch the original database. */
  ok = xtmCdFetchNamedDb( cal_handle, cal_db_name,
                          &db_info, &db_incl );
  if( ! ok )
    return( False );


  /* A new unique name for the shadow entry. */
  sprintf( db_shadow_info.shadow_name, "%s%d",
           SHADOW_PREFIX, cal_db_ref -> shadow_id );

  cal_db_ref -> shadow_id++;
  if( cal_db_ref -> shadow_id > 9999 )
    cal_db_ref -> shadow_id = 0;


  memcpy( (void *) &db_shadow_info.db_info,
          (void *) &db_info,
          sizeof( XTM_CD_CAL_INFO ) );


  /* Any included calendars? */
  db_shadow_info.db_info.include = NULL;

  if( db_incl.no > 0 )
    copyIncludeDb( cal_handle, True, 
                   &db_incl, &db_shadow_info.db_info.include );


  /* Add the entry to the end of the list. */
  lst_status = LstLinkInsertLast( cal_db_ref -> shadow_cal_list,
                                  &db_shadow_info );
  if( lst_status != LST_OK )
    return( False );


  strcpy( shadow_db_name, db_shadow_info.shadow_name );


  return( True );

} /* xtmCdCreateShadowEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdDeleteEntry( XTM_CD_HANDLE  cal_handle,
                    char           *cal_db_name )
{

  /* Variables. */
  CAL_DB_REF       cal_db_ref;
  LST_STATUS       lst_status;
  SHADOW_REC       *shadow_rec_ref;
  XTM_CD_CAL_INFO  *db_info_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Search the entry in the main list. */
  lst_status = LstLinkSearchFirst( cal_db_ref -> cal_list,
                                   (void *) cal_db_name,
                                   (EQUALS_FUNC_TYPE) searchDbName );

  if( lst_status == LST_OK ) {
    db_info_ref = (XTM_CD_CAL_INFO *) LstLinkGetCurrentRef( 
                                        cal_db_ref -> cal_list );

    if( db_info_ref -> include != NULL )
      SysFree( db_info_ref -> include );

    (void) LstLinkDeleteCurrent( cal_db_ref -> cal_list );

    return( True );
  }


  /* Search the entry in the shadow list. */
  lst_status = LstLinkSearchFirst( cal_db_ref -> shadow_cal_list,
                                   (void *) cal_db_name,
                                   (EQUALS_FUNC_TYPE) searchShadowDbName );
  if( lst_status == LST_OK ) {
    shadow_rec_ref = (SHADOW_REC *) LstLinkGetCurrentRef( 
                                      cal_db_ref -> shadow_cal_list );

    db_info_ref = &shadow_rec_ref -> db_info;

    if( db_info_ref -> include != NULL )
      SysFree( db_info_ref -> include );

    (void) LstLinkDeleteCurrent( cal_db_ref -> shadow_cal_list );

    return( True );
  }


  return( False );

} /* xtmCdDeleteEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdFetchDbNames( XTM_CD_HANDLE  cal_handle,
                     char           **db_names )
{

  /* Variables. */
  int              db_items;
  CAL_DB_REF       cal_db_ref;
  LST_STATUS       lst_status;
  XTM_CD_CAL_INFO  *db_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  db_items = LstLinkElements( cal_db_ref -> cal_list );

  if( db_items <= 0 ) {
    *db_names = SysNewString( " " );
    return( False );
  }

  /* We know the length of string to return. */
  *db_names  = (char *) SysMalloc( db_items * (XTM_GL_MAX_CAL_NAME + 1) + 5 );
  **db_names = '\0';


  /* Search all calendars. */
  lst_status = LstLinkCurrentFirst( cal_db_ref -> cal_list );

  while( lst_status == LST_OK ) {

    db_ref = (XTM_CD_CAL_INFO *) 
               LstLinkGetCurrentRef( cal_db_ref -> cal_list );

    strcat( *db_names, db_ref -> short_name );
    strcat( *db_names, " " );

    lst_status = LstLinkCurrentNext( cal_db_ref -> cal_list );

  } /* while */


  return( True );

} /* xtmCdFetchDbNames */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdFetchDefaultDb( XTM_CD_HANDLE     cal_handle,
                       XTM_CD_CAL_INFO   *db_info_ref,
                       XTM_CD_INCL_CALS  *db_incl_ref )
{

  /* Variables. */
  CAL_DB_REF       cal_db_ref;
  LST_STATUS       lst_status;
  XTM_CD_CAL_INFO  *db_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  if( LstLinkElements( cal_db_ref -> cal_list ) <= 0 )
    return( False );


  /* Search all calendars. */
  lst_status = LstLinkCurrentFirst( cal_db_ref -> cal_list );

  while( lst_status == LST_OK ) {

    db_ref = (XTM_CD_CAL_INFO *) 
               LstLinkGetCurrentRef( cal_db_ref -> cal_list );

    if( flagIsSet( db_ref -> flags, XTM_CD_FLAG_DEFAULT_DB ) ) {
      memcpy( (void *) db_info_ref,
              (void *) db_ref,
              sizeof( XTM_CD_CAL_INFO ) );

      if( db_incl_ref != NULL && db_info_ref -> include != NULL )
        memcpy( (void *) db_incl_ref, 
                (void *) db_info_ref -> include,
                sizeof( XTM_CD_INCL_CALS ) );

      else if( db_incl_ref != NULL )
        db_incl_ref -> no = 0;

      db_info_ref -> include = NULL;

      return( True );
    }

    lst_status = LstLinkCurrentNext( cal_db_ref -> cal_list );

  } /* while */


  return( False );

} /* xtmCdFetchDefaultDb */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdFetchNamedDb( XTM_CD_HANDLE     cal_handle,
                     char              *cal_db_name,
                     XTM_CD_CAL_INFO   *db_info_ref,
                     XTM_CD_INCL_CALS  *db_incl_ref )
{

  /* Variables. */
  CAL_DB_REF       cal_db_ref;
  LST_STATUS       lst_status;
  SHADOW_REC       *shadow_db_ref;
  XTM_CD_CAL_INFO  *db_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Search the entry in the main list. */
  lst_status = LstLinkSearchFirst( cal_db_ref -> cal_list,
                                   (void *) cal_db_name,
                                   (EQUALS_FUNC_TYPE) searchDbName );

  if( lst_status == LST_OK ) {
    db_ref = (XTM_CD_CAL_INFO *) 
               LstLinkGetCurrentRef( cal_db_ref -> cal_list );

    memcpy( (void *) db_info_ref,
            (void *) db_ref,
            sizeof( XTM_CD_CAL_INFO ) );

    if( db_incl_ref != NULL && db_info_ref -> include != NULL )
      memcpy( (void *) db_incl_ref, 
              (void *) db_info_ref -> include,
              sizeof( XTM_CD_INCL_CALS ) );

    else if( db_incl_ref != NULL )
      db_incl_ref -> no = 0;

    db_info_ref -> include = NULL;

    return( True );
  }


  /* Search the entry in the shadow list. */
  lst_status = LstLinkSearchFirst( cal_db_ref -> shadow_cal_list,
                                   (void *) cal_db_name,
                                   (EQUALS_FUNC_TYPE) searchShadowDbName );
  if( lst_status == LST_OK ) {
    shadow_db_ref = (SHADOW_REC *) 
                      LstLinkGetCurrentRef( cal_db_ref -> shadow_cal_list );

    db_ref = &shadow_db_ref -> db_info;

    memcpy( (void *) db_info_ref,
            (void *) db_ref,
            sizeof( XTM_CD_CAL_INFO ) );

    if( db_incl_ref != NULL && db_info_ref -> include != NULL )
      memcpy( (void *) db_incl_ref, 
              (void *) db_info_ref -> include,
              sizeof( XTM_CD_INCL_CALS ) );

    else if( db_incl_ref != NULL )
      db_incl_ref -> no = 0;

    db_info_ref -> include = NULL;

    return( True );
  }


  return( False );

} /* xtmCdFetchNamedDb */


/*----------------------------------------------------------------------*/

void
  xtmCdFree( XTM_CD_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_DB_REF  cal_db_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  cal_db_ref = (CAL_DB_REF) cal_handle;


  /* Free the shadow list. */
  if( cal_db_ref -> shadow_cal_list != NULL )
    LstLinkClearDataAndList( cal_db_ref -> shadow_cal_list, 
                             clearShadowEntry );

  /* Free the calendar list. */
  if( cal_db_ref -> cal_list != NULL )
    LstLinkClearDataAndList( cal_db_ref -> cal_list, clearCalEntry );


  /* Free the record itself. */
  SysFree( cal_db_ref );


  return;

} /* xtmCdFree */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdFreeShadowEntry( XTM_CD_HANDLE  cal_handle,
                        char           *shadow_db_name )
{

  /* Variables. */
  Boolean  status;


  /* Code. */

  if( cal_handle == NULL )
    return( False );

  /* The standard delete can do this. */
  status = xtmCdDeleteEntry( cal_handle, shadow_db_name );


  return( status );

} /* xtmCdFreeShadowEntry */


/*----------------------------------------------------------------------*/

XTM_CD_HANDLE
  xtmCdInitialize()
{

  /* Variables. */
  CAL_DB_REF  cal_db_ref;


  /* Code. */

  /* Allocate space for a new calendar database. */
  cal_db_ref = SysNew( CAL_DB );

  cal_db_ref -> cal_list = 
   LstLinkNew( sizeof( XTM_CD_CAL_INFO ), NULL );
  cal_db_ref -> shadow_cal_list = 
    LstLinkNew( sizeof( SHADOW_REC ), NULL );

  cal_db_ref -> shadow_id = 0;
  

  return( (XTM_CD_HANDLE) cal_db_ref );

} /* xtmCdInitialize */


/*----------------------------------------------------------------------*/

Boolean
  xtmCdSearchIncludeDb( XTM_CD_INCL_CALS  *db_incl_ref,
                        char              *cal_db_name )
{

  /* Variables. */
  int  index;


  /* Code. */

  if( db_incl_ref == NULL )
    return( False );

  for( index = 0; index < db_incl_ref -> no; index++ ) {
    if( strcmp( cal_db_name, db_incl_ref -> db[ index ].name ) == 0 )
      return( True );
  }


  return( False );

} /* xtmCdSearchIncludeDb */


/*----------------------------------------------------------------------*/

static void
  clearCalEntry( void  *data_ref )
{

  /* Variables. */
  XTM_CD_CAL_INFO  *db_ref;


  /* Code. */

  db_ref = (XTM_CD_CAL_INFO *) data_ref;

  if( db_ref -> include != NULL ) {
    SysFree( db_ref -> include );

    db_ref -> include = NULL;
  }


  return;

} /* clearCalEntry */


/*----------------------------------------------------------------------*/

static void
  clearShadowEntry( void  *data_ref )
{

  /* Variables. */
  SHADOW_REC  *db_shadow_ref;


  /* Code. */

  db_shadow_ref = (SHADOW_REC *) data_ref;

  if( db_shadow_ref -> db_info.include != NULL ) {
    SysFree( db_shadow_ref -> db_info.include );

    db_shadow_ref -> db_info.include = NULL;
  }


  return;

} /* clearShadowEntry */


/*----------------------------------------------------------------------*/

static void
  copyIncludeDb( XTM_CD_HANDLE     cal_handle,
                 Boolean           only_if_read,
                 XTM_CD_INCL_CALS  *from_incl_ref,
                 XTM_CD_INCL_CALS  **to_incl_ref )
{

  /* Variables. */
  Boolean          ok;
  int              index;
  int              index1 = 0;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  *to_incl_ref = NULL;

  if( from_incl_ref == NULL || from_incl_ref -> no <= 0 )
    return;

  *to_incl_ref = SysNew( XTM_CD_INCL_CALS );


  /* Copy the included databases. */
  for( index = 0; index < from_incl_ref -> no; index++ ) {

    ok = True;
    db_info.operations = 0;

    (void) xtmCdFetchNamedDb( cal_handle,
                              from_incl_ref -> db[ index ].name,
                              &db_info, NULL );

    if( only_if_read && 
        flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      ok = False;

    if( ok ) {

      strcpy( (*to_incl_ref) -> db[ index1 ].name,
              from_incl_ref  -> db[ index ].name );

      index1++;

    } /* if */

  } /* loop */


  (*to_incl_ref) -> no = index1;


  return;

} /* copyIncludeDb */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  searchDbName( XTM_CD_CAL_INFO  *db_ref,
                char             *db_name )
{

  /* Code. */

  if( strcmp( db_ref -> short_name, db_name ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* searchDbName */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  searchShadowDbName( SHADOW_REC  *db_ref,
                      char        *db_name )
{

  /* Code. */

  if( strcmp( db_ref -> shadow_name, db_name ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* searchShadowDbName */
