/*----------------------------------------------------------------------------
--
--  Module:           xtmEditAlarm
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Edit the alarms for an XDiary entry.
--
--  Filename:         xtmEditAlarm.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEditAlarm.c, Version: 1.1, Date: 95/02/18 15:52:14";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xitTune.h"
#include "xtmEditAlarm.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the alarm window. */
#define actDoRc            dataLocalW[  0 ]
#define actUseShell        dataLocalW[  1 ]
#define actUseText         dataLocalW[  2 ]
#define actionFr           dataLocalW[  3 ]
#define actionRc           dataLocalW[  4 ]
#define alarmFr            dataLocalW[  5 ]
#define alarmLa            dataLocalW[  6 ]
#define alarmRc            dataLocalW[  7 ]
#define launchWhenLa       dataLocalW[  8 ]
#define selAlarmRc         dataLocalW[  9 ]
#define testPb             dataLocalW[ 10 ]
#define testTuneFo         dataLocalW[ 11 ]
#define tuneFr             dataLocalW[ 12 ]
#define tuneLa             dataLocalW[ 13 ]
#define tuneRb             dataLocalW[ 14 ]
#define tuneRc             dataLocalW[ 15 ]
#define whichAlarmLa       dataLocalW[ 16 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the alarm window. */
typedef struct {

  /* Alarms window. */
  Widget  alarmW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_EA_ACTION_CB  actionCB;
  void              *user_data;

} ALARM_REC, *ALARM_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditAlarm";

/* IDs for the help windows. */
static char  *alarms_window_id = "EntryAlarms";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget         widget,
            ALARM_REC_REF  alarm_ref,
            XtPointer      call_data );

static Widget
  createAlarmWindow( Widget         parent,
                     ALARM_REC_REF  alarm_ref );

static void 
  destroyCB( Widget         widget,
             ALARM_REC_REF  alarm_ref,
             XtPointer      call_data );

static Boolean
  fetchAlarmValues( ALARM_REC_REF          alarm_ref,
                    XTM_EA_GET_ALARM_DATA  *values_ref );

static void 
  helpCB( Widget         widget,
          ALARM_REC_REF  alarm_ref,
          XtPointer      call_data );

static void 
  okCB( Widget         widget,
        ALARM_REC_REF  alarm_ref,
        XtPointer      call_data );

static void 
  playTuneCB( Widget         widget,
              ALARM_REC_REF  alarm_ref,
              XtPointer      call_data );

static void
  setAlarmValues( ALARM_REC_REF          alarm_ref,
                  XTM_EA_SET_ALARM_DATA  *values_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_EA_HANDLE
  xtmEaInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_EA_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  ALARM_REC_REF  alarm_ref;


  /* Code. */

  /* Create and initialize our private data. */
  alarm_ref = SysNew( ALARM_REC );
  if( alarm_ref == NULL )
    return( NULL );

  alarm_ref -> appl_data_ref = appl_data_ref;
  alarm_ref -> actionCB      = actionCB;
  alarm_ref -> user_data     = user_data;
  alarm_ref -> alarmW        = NULL;


  /* Create the base alarm window. */
  alarm_ref -> alarmW = createAlarmWindow( parent, alarm_ref );
  if( alarm_ref -> alarmW == NULL ) {
    SysFree( alarm_ref );

    return( NULL );
  }


  return( (XTM_EA_HANDLE) alarm_ref );

} /* xtmEaInitialize */


/*----------------------------------------------------------------------*/

void
  xtmEaDestroy( XTM_EA_HANDLE  alarm_handle )
{

  /* Variables. */
  ALARM_REC_REF  alarm_ref;


  /* Code. */

  if( alarm_handle == NULL )
    return;

  /* Our private data. */
  alarm_ref = (ALARM_REC_REF) alarm_handle;

  /* Get rid of all windows. */
  if( alarm_ref -> alarmW != NULL )
    XtDestroyWidget( alarm_ref -> alarmW );


  return;

} /* xtmEaDestroy */


/*----------------------------------------------------------------------*/

void
  xtmEaEditAlarms( XTM_EA_HANDLE          alarm_handle,
                   XTM_EA_SET_ALARM_DATA  *values_ref )
{

  /* Variables. */
  ALARM_REC_REF  alarm_ref;


  /* Code. */

  /* Our private data. */
  alarm_ref = (ALARM_REC_REF) alarm_handle;


  /* Set the window data. */
  setAlarmValues( alarm_ref, values_ref );


  /* Make sure the window is visible. */
  XtManageChild( alarm_ref -> alarmW );


  return;

} /* xtmEaEditAlarms */


/*----------------------------------------------------------------------*/

static Widget
  createAlarmWindow( Widget         parent,
                     ALARM_REC_REF  alarm_ref )
{

  /* Variables. */
  int         index;
  char        buffer[ 50 ];
  char        label[ 20 ];
  Arg         args[ 10 ];
  Cardinal    n;
  Widget      actAlarmTb[ XTM_EA_MAX_ALARMS ];
  Widget      alarmAfterTb[ XTM_EA_MAX_ALARMS ];
  Widget      alarmBeforeTb[ XTM_EA_MAX_ALARMS ];
  Widget      alarmFd;
  Widget      alarmNoLa[ XTM_EA_MAX_ALARMS ];
  Widget      alarmRb[ XTM_EA_MAX_ALARMS ];
  Widget      alarmTx[ XTM_EA_MAX_ALARMS ];
  Widget      alarmValueRc[ XTM_EA_MAX_ALARMS ];
  Widget      melodyTb[ XTM_EA_MAX_TUNES ];
  Widget      dataLocalW[ 17 ];
  Widget      workFo;

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "", NULL, 1, True },
  };

  static XIT_PUSH_STRUCT tune_test_button_def[] = {
    { "TestPb", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = alarm_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = alarm_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = alarm_ref;


  /* Create a form dialog with buttons. */
  alarmFd = xitCreateFormDialog( parent, "AlarmFd",
                                 1, 0,
                                 action_buttons, 
                                 XtNumber( action_buttons ) );

  XtAddCallback( alarmFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) alarm_ref );

  /* We want to know when this window is closed. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( alarmFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( alarmFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( alarmFd ), wm_delete_window, 
                             (XtCallbackProc) cancelCB,
                             (XtPointer) alarm_ref );
  } /* block */

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( alarmFd, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_ALARM_PAR_TITLE ) ); n++;
  XtSetValues( XtParent( alarmFd ), args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( alarmFd, "AlarmFdFo" );


  /* Create the form for alarms. */
  alarmFr  = XmCreateFrame(     workFo,  "AlarmFr", args, 0 );
  alarmRc  = XmCreateRowColumn( alarmFr, "AlarmRc", args, 0 );

  /* Create the alarm label. */
  alarmLa = xitCreateLabel( alarmRc, "AlarmLa", 
                            msgGetText( MXDI_ALARM_LABEL ), -1 );

  /* Create alarm time and after/before buttons */
  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {

    sprintf( buffer, "AlarmValue%dRc", index + 1 );

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
    alarmValueRc[ index ] = XmCreateRowColumn( alarmRc, buffer, args, n );

    /* Number each alarm. */
    sprintf( buffer, "%d:", index + 1 );
    alarmNoLa[ index ] = xitCreateLabel( alarmValueRc[ index ],
                                         "AlarmNoLa", buffer, -1 );

    /* Text buffer with alarm time. */
    sprintf( buffer, "Alarm%dTx", index + 1 );
    text_buffer[ 0 ].name = buffer;

    alarmTx[ index ] = xitCreateTextCols( alarmValueRc[ index ], 
                                          &text_buffer[ 0 ], 9 );

    n = 0;
    XtSetArg( args[ n ], XmNmaxLength, 9 );  n++;
    XtSetValues( alarmTx[ index ], args, n );  

    sprintf( buffer, "Alarm%dRb", index + 1 );

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
    XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
    XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
    alarmRb[ index ] = XmCreateRowColumn( alarmValueRc[ index ], 
                                          buffer, args, n );

    sprintf( buffer, "AlarmBefore%dTb", index + 1 );

    alarmBeforeTb[ index ] = xitCreateToggleButton( 
                               alarmRb[ index ], buffer,
                               msgGetText( MXDI_BEFORE_LABEL ), False );

    sprintf( buffer, "AlarmAfter%dTb", index + 1 );

    alarmAfterTb[ index ] = xitCreateToggleButton( 
                              alarmRb[ index ], buffer, 
                              msgGetText( MXDI_AFTER_LABEL ), False );

  } /* loop */


  /* Create the form for tunes. */
  tuneFr = XmCreateFrame( workFo, "TuneFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  tuneRc = XmCreateRowColumn( tuneFr, "TuneRc", args, n );

  tuneLa = xitCreateLabel( tuneRc, "TuneLa", 
                           msgGetText( MXDI_TUNE_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  tuneRb = XmCreateRowColumn( tuneRc, "TuneRb", args, n );

  for( index = 0; index < XTM_EA_MAX_TUNES; index++ ) {

    sprintf( label, "%s %d",
             msgGetText( MXDI_TUNE_ID_LABEL ), index + 1 );

    sprintf( buffer, "Melody%dTb", index + 1 );

    melodyTb[ index ] = xitCreateToggleButton( 
                          tuneRb, buffer, label, False );

  } /* loop */

  /* Test button for tunes. */
  testTuneFo = XmCreateForm( tuneRc, "TestTuneFo", args, 0 );

  tune_test_button_def[ 0 ].title = msgGetText( MXDI_TUNE_TEST );

  testPb = xitCreatePushButton( testTuneFo, &tune_test_button_def[ 0 ] );

  XtAddCallback( testPb, XmNactivateCallback,
                 (XtCallbackProc) playTuneCB, (XtPointer) alarm_ref );


  /* Form for alarm action to do. */
  actionFr = XmCreateFrame( workFo, "ActionFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  actionRc = XmCreateRowColumn( actionFr, "ActionRc", args, n );

  whichAlarmLa = xitCreateLabel( 
                   actionRc, "WhichAlarmLa", 
                   msgGetText( MXDI_WHICH_AL_ACTION_LABEL ), -1 );

  /* Launch action for this alarm. */  
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  selAlarmRc = XmCreateRowColumn( actionRc, "SelAlarmRc", args, n );

  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {

    sprintf( buffer, "ActAlarm%dTb", index + 1 );
    sprintf( label,  "%d", index + 1 );

    actAlarmTb[ index ] = xitCreateToggleButton( selAlarmRc, buffer,
                                                 label, False );
  }

  /* Launch action when. */
  launchWhenLa = xitCreateLabel( 
                   actionRc, "LaunchWhenLa", 
                   msgGetText( MXDI_LAUNCH_WHEN_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  actDoRc = XmCreateRowColumn( actionRc, "ActDoRc", args, n );

  actUseShell = xitCreateToggleButton( 
                  actDoRc, "ActUseShell", 
                  msgGetText( MXDI_ACT_USE_SHELL_LABEL ), False );

  actUseText = xitCreateToggleButton( 
                 actDoRc, "ActUseText",
                 msgGetText( MXDI_ACT_USE_TEXT_LABEL ), False );


  /* Put the elements together. */
  xitAttachWidget( alarmFr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( tuneFr,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, alarmFr,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );
  xitAttachWidget( actionFr,
                   XmATTACH_WIDGET, alarmFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( alarmFr,  args, n );
  XtSetValues( tuneFr,   args, n );
  XtSetValues( actionFr, args, n );


  /* Manage the widgets. */
  XtManageChildren( alarmValueRc,   XtNumber( alarmValueRc ) );

  xitManageChildren( actAlarmTb,    XtNumber( actAlarmTb  ) );
  xitManageChildren( alarmAfterTb,  XtNumber( alarmAfterTb ) );
  xitManageChildren( alarmBeforeTb, XtNumber( alarmBeforeTb ) );
  xitManageChildren( alarmNoLa,     XtNumber( alarmNoLa ) );
  xitManageChildren( alarmRb,       XtNumber( alarmRb ) );
  xitManageChildren( alarmTx,       XtNumber( alarmTx ) );
  xitManageChildren( melodyTb,      XtNumber( melodyTb ) );
  xitManageChildren( dataLocalW,    XtNumber( dataLocalW ) );


  /* Set the initial sizes. */
  xitSetSizeFormDialog( alarmFd, False );


  return( alarmFd );

} /* createAlarmWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchAlarmValues( ALARM_REC_REF          alarm_ref,
                    XTM_EA_GET_ALARM_DATA  *values_ref )
{

  /* Variables. */
  Boolean  ok;
  int      index;
  int      offset;
  char     buffer[ 100 ];
  char     *char_ref;
  Widget   baseW;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  /* We need a waindow. */
  if( alarm_ref -> alarmW == NULL )
    return( False );

  mainW = XtNameToWidget( alarm_ref -> alarmW, "AlarmFdFo" );

  values_ref -> alarm_action = 0;
  values_ref -> what_action  = 0;


  /* Alarms. */
  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {

    sprintf( buffer, "AlarmFr.AlarmRc.AlarmValue%dRc", index + 1 );
    baseW = XtNameToWidget( mainW, buffer );

    sprintf( buffer, "Alarm%dTx", index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    char_ref = xitStringGetText( tempW );

    /* Up to 7 days before/after. */
    ok = xtmFoFetchMinuteTime( char_ref, 
                               True, alarm_ref -> alarmW,
                               7 * 24 * 60, &offset );
    SysFree( char_ref );

    if( ! ok )
      return( False );


    /* Save the alarm. */
    if( offset >= 0 ) {
      sprintf( buffer, "Alarm%dRb.AlarmBefore%dTb", index + 1, index + 1 );
      tempW = XtNameToWidget( baseW, buffer );

      if( XmToggleButtonGetState( tempW ) )
        offset = (-1) * offset;

      values_ref -> alarm_valid[  index ] = True;
      values_ref -> alarm_offset[ index ] = offset;

    } else {
      values_ref -> alarm_valid[  index ] = False;
      values_ref -> alarm_offset[ index ] = 0;
    }

  } /* loop */


  /* Melody selected. */
  baseW = XtNameToWidget( mainW, "TuneFr.TuneRc.TuneRb" );

  for( index = 0; index < XTM_EA_MAX_TUNES; index++ ) {
    sprintf( buffer, "Melody%dTb", index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    if( XmToggleButtonGetState( tempW ) )
      values_ref -> alarm_melody = index + 1;
  }


  /* What to do when an alarm sounds? */
  sprintf( buffer, "ActionFr.ActionRc.SelAlarmRc" );
  baseW = XtNameToWidget( mainW, buffer );

  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {
    sprintf( buffer, "ActAlarm%dTb", index + 1 );

    tempW = XtNameToWidget( baseW, buffer );

    if( XmToggleButtonGetState( tempW ) ) {

      switch( index + 1 ) {
        case 1:
          flagSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM1 );
          break;
        case 2:
          flagSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM2 );
          break;
        case 3:
          flagSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM3 );
          break;
        case 4:
          flagSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM4 );
          break;
        case 5:
          flagSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM5 );
          break;
      }

    } /* if */

  } /* loop */


  /* Kind of action to do. */
  sprintf( buffer, "ActionFr.ActionRc.ActDoRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "ActUseShell" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> what_action, XTM_DB_FLAG_ACTION_SCRIPT );

  tempW = XtNameToWidget( baseW, "ActUseText" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> what_action, XTM_DB_FLAG_ACTION_TEXT );


  return( True );

} /* fetchAlarmValues */


/*----------------------------------------------------------------------*/

static void
  setAlarmValues( ALARM_REC_REF          alarm_ref,
                  XTM_EA_SET_ALARM_DATA  *values_ref )
{

  /* Variables. */
  Boolean   after;
  Boolean   before;
  int       alarm_offset;
  int       days;
  int       hours;
  int       index;
  int       minutes;
  char      buffer[ 80 ];
  Widget    baseW;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  /* We need a waindow. */
  if( alarm_ref -> alarmW == NULL )
    return;


  mainW = XtNameToWidget( alarm_ref -> alarmW, "AlarmFdFo" );


  /* Set alarm times. */
  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {

    sprintf( buffer, "AlarmFr.AlarmRc.AlarmValue%dRc", 
             index + 1 );
    baseW = XtNameToWidget( mainW, buffer );


    /* Set alarm time. */
    sprintf( buffer, "Alarm%dTx", index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    /* sprintf( buffer, "" ); */

    if( values_ref -> alarm_valid[ index ] ) {

      alarm_offset = (int) values_ref -> alarm_offset[ index ];

      if( alarm_offset < 0 )
        alarm_offset = (-1) * alarm_offset;

      days = alarm_offset / (24 * 60);
      alarm_offset = alarm_offset - (days * 24 * 60);

      hours = alarm_offset / 60;
      alarm_offset = alarm_offset - (hours * 60);

      minutes = alarm_offset;

      /* The format is dd hh:mm. */
      if( days > 0 )
        sprintf( buffer, "%d %d:%02d", days, hours, minutes );
      else
        sprintf( buffer, "%d:%02d", hours, minutes );

    } /* if */

    XmTextSetString( tempW, buffer );


    /* Set alarm before/after. */
    before = False;
    after  = False;

    if( values_ref -> alarm_offset[ index ] < 0 ) {
      before = True;
      after  = False;
    } else if( values_ref -> alarm_offset[ index ] > 0 ) {
      before = False;
      after  = True;
    }

    sprintf( buffer, "Alarm%dRb.AlarmBefore%dTb", 
             index + 1, index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    XmToggleButtonSetState( tempW, before, False );

    sprintf( buffer, "Alarm%dRb.AlarmAfter%dTb",
             index + 1, index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    XmToggleButtonSetState( tempW, after, False );

  } /* loop */


  /* Melody selected. */
  sprintf( buffer, "TuneFr.TuneRc.TuneRb" );
  baseW = XtNameToWidget( mainW, buffer );

  /* Set the correct melody. */
  for( index = 1; index < 6; index++ ) {
    sprintf( buffer, "Melody%dTb", index );
    tempW = XtNameToWidget( baseW, buffer );

    if( values_ref -> alarm_melody == index )
      XmToggleButtonSetState( tempW, True,  False );
    else
      XmToggleButtonSetState( tempW, False, False );
  }


  /* Launch action when alarm values. */
  sprintf( buffer, "ActionFr.ActionRc.SelAlarmRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "ActAlarm1Tb" );
  if( flagIsSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM1 ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );
    
  tempW = XtNameToWidget( baseW, "ActAlarm2Tb" );
  if( flagIsSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM2 ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );
    
  tempW = XtNameToWidget( baseW, "ActAlarm3Tb" );
  if( flagIsSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM3 ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );
    
  tempW = XtNameToWidget( baseW, "ActAlarm4Tb" );
  if( flagIsSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM4 ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );
    
  tempW = XtNameToWidget( baseW, "ActAlarm5Tb" );
  if( flagIsSet( values_ref -> alarm_action, XTM_DB_FLAG_ACTION_ALARM5 ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* What to do? */    
  sprintf( buffer, "ActionFr.ActionRc.ActDoRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "ActUseShell" );
  if( flagIsSet( values_ref -> what_action, XTM_DB_FLAG_ACTION_SCRIPT ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "ActUseText" );
  if( flagIsSet( values_ref -> what_action, XTM_DB_FLAG_ACTION_TEXT ) )
    XmToggleButtonSetState( tempW, True,  False );
  else
    XmToggleButtonSetState( tempW, False, False );
    

  return;

} /* setAlarmValues */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget         widget,
            ALARM_REC_REF  alarm_ref,
            XtPointer      call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( alarm_ref -> actionCB != NULL )
    (* alarm_ref -> actionCB)( XTM_EA_REASON_CANCEL,
                               alarm_ref -> user_data, NULL );

  XtUnmanageChild( alarm_ref -> alarmW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget         widget,
             ALARM_REC_REF  alarm_ref,
             XtPointer      call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( alarm_ref -> actionCB != NULL )
    (* alarm_ref -> actionCB)( XTM_EA_REASON_DESTROY,
                               alarm_ref -> user_data, NULL );

  SysFree( alarm_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget         widget,
          ALARM_REC_REF  alarm_ref,
          XtPointer      call_data )
{

  /* Code. */

  xtmHlDisplayHelp( alarm_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX, alarms_window_id, "" );


  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget         widget,
        ALARM_REC_REF  alarm_ref,
        XtPointer      call_data )
{

  /* Variables. */
  Boolean                ok;
  XTM_EA_GET_ALARM_DATA  values;


  /* Code. */

  ok = fetchAlarmValues( alarm_ref, &values );
  if( ! ok )
    return;

  /* Do we have a user action callback registered? */
  if( alarm_ref -> actionCB != NULL )
    (* alarm_ref -> actionCB)( XTM_EA_REASON_OK,
                               alarm_ref -> user_data,
                               &values );

  XtUnmanageChild( alarm_ref -> alarmW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  playTuneCB( Widget         widget,
              ALARM_REC_REF  alarm_ref,
              XtPointer      call_data )
{

  /* Variables. */
  int                   index;
  char                  buffer[ 20 ];
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = alarm_ref -> appl_data_ref;


  if( alarm_ref -> alarmW == NULL )
    return;

  mainW = XtNameToWidget( alarm_ref -> alarmW, "AlarmFdFo" );
  baseW = XtNameToWidget( mainW, "TuneFr.TuneRc.TuneRb" );

  for( index = 0; index < XTM_EA_MAX_TUNES; index++ ) {

    sprintf( buffer, "Melody%dTb", index + 1 );
    tempW = XtNameToWidget( baseW, buffer );

    if( XmToggleButtonGetState( tempW ) )
      break;

  } /* loop */

  if( index >= 5 )
    return;

  xtmToPlayMelody( widget, 
                   appl_data_ref -> custom_data -> tune_player,
                   appl_data_ref -> custom_data -> tune_player_param,
                   appl_data_ref -> custom_data -> tune_file[ index ],
                   appl_data_ref -> custom_data -> alarm_volume,
                   appl_data_ref -> custom_data -> tune_dur_delta );


  return;

} /* playTuneCB */
