/*----------------------------------------------------------------------------
--
--  Module:           xtmEditPref
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Edit the preferences for an XDiary entry.
--
--  Filename:         xtmEditPref.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEditPref.c, Version: 1.1, Date: 95/02/18 15:52:16";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitColorSel.h"
#include "xitError.h"
#include "xitTools.h"
#include "xitTune.h"
#include "xtmEditPref.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the preference window. */
#define colorBgCs          dataLocalW[  0 ]
#define colorEntryRc       dataLocalW[  1 ]
#define colorFgCs          dataLocalW[  2 ]
#define colorFr            dataLocalW[  3 ]
#define colorLa            dataLocalW[  4 ]
#define colorRc            dataLocalW[  5 ]
#define defColorPb         dataLocalW[  6 ]
#define flagsFr            dataLocalW[  7 ]
#define flagsRc            dataLocalW[  8 ]
#define hideFr             dataLocalW[  9 ]
#define hideLa             dataLocalW[ 10 ]
#define hideRc             dataLocalW[ 11 ]
#define hideWhatRc         dataLocalW[ 12 ]
#define lineAlarmLa        dataLocalW[ 13 ]
#define lineAlarmTx        dataLocalW[ 14 ]
#define lineEntryLa        dataLocalW[ 15 ]
#define lineEntryRc        dataLocalW[ 16 ]
#define lineEntryTx        dataLocalW[ 17 ]
#define linesFr            dataLocalW[ 18 ]
#define linesLa            dataLocalW[ 19 ]
#define linesRc            dataLocalW[ 20 ]
#define sampleLa           dataLocalW[ 21 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the preference window. */
typedef struct {

  /* Preference window. */
  Widget  prefW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_EP_ACTION_CB  actionCB;
  void              *user_data;

} PREF_REC, *PREF_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditPref";

/* IDs for the help windows. */
static char  *pref_window_id = "EntryPref";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            PREF_REC_REF  pref_ref,
            XtPointer     call_data );

static Widget
  createPrefWindow( Widget        parent,
                    PREF_REC_REF  pref_ref );

static void 
  defaultColorCB( Widget        widget,
                  PREF_REC_REF  pref_ref,
                  XtPointer     call_data );

static void 
  destroyCB( Widget        widget,
             PREF_REC_REF  pref_ref,
             XtPointer     call_data );

static Boolean
  fetchPrefValues( PREF_REC_REF          pref_ref,
                   XTM_EP_GET_PREF_DATA  *values_ref );

static void 
  helpCB( Widget        widget,
          PREF_REC_REF  pref_ref,
          XtPointer     call_data );

static void 
  newColorCB( Widget                        widget,
              PREF_REC_REF                  pref_ref,
              xitColorSelectCallbackStruct  *call_data );

static void 
  okCB( Widget        widget,
        PREF_REC_REF  pref_ref,
        XtPointer     call_data );

static void
  setPrefValues( PREF_REC_REF          pref_ref,
                 XTM_EP_SET_PREF_DATA  *values_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_EP_HANDLE
  xtmEpInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_EP_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  PREF_REC_REF  pref_ref;


  /* Code. */

  /* Create and initialize our private data. */
  pref_ref = SysNew( PREF_REC );
  if( pref_ref == NULL )
    return( NULL );

  pref_ref -> appl_data_ref = appl_data_ref;
  pref_ref -> actionCB      = actionCB;
  pref_ref -> user_data     = user_data;
  pref_ref -> prefW         = NULL;


  /* Create the base window. */
  pref_ref -> prefW = createPrefWindow( parent, pref_ref );
  if( pref_ref -> prefW == NULL ) {
    SysFree( pref_ref );

    return( NULL );
  }


  return( (XTM_EP_HANDLE) pref_ref );

} /* xtmEpInitialize */


/*----------------------------------------------------------------------*/

void
  xtmEpDestroy( XTM_EP_HANDLE  pref_handle )
{

  /* Variables. */
  PREF_REC_REF  pref_ref;


  /* Code. */

  if( pref_handle == NULL )
    return;

  /* Our private data. */
  pref_ref = (PREF_REC_REF) pref_handle;

  /* Get rid of all windows. */
  if( pref_ref -> prefW != NULL )
    XtDestroyWidget( pref_ref -> prefW );


  return;

} /* xtmEpDestroy */


/*----------------------------------------------------------------------*/

void
  xtmEpEditPreferences( XTM_EP_HANDLE         pref_handle,
                        XTM_EP_SET_PREF_DATA  *values_ref )
{

  /* Variables. */
  PREF_REC_REF  pref_ref;


  /* Code. */

  /* Our private data. */
  pref_ref = (PREF_REC_REF) pref_handle;


  /* Make sure the window is visible. */
  XtManageChild( pref_ref -> prefW );


  /* Set the window data. */
  setPrefValues( pref_ref, values_ref );


  return;

} /* xtmEaEditPreferences */


/*----------------------------------------------------------------------*/

static Widget
  createPrefWindow( Widget        parent,
                    PREF_REC_REF  pref_ref )
{

  /* Variables. */
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 22 ];
  Widget                  hideTb[ 6 ];
  Widget                  prefFd;
  Widget                  flagsTb[ 4 ];
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  static XIT_TEXT_STRUCT text_buffer[] = {
    { "LineEntryTx", NULL, 1, True },
    { "LineAlarmTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };

  static XIT_PUSH_STRUCT doit_buttons_def[] = {
    { "DefColorPb", "", "", True, NULL },
  };


  /* Code. */

  appl_data_ref   = pref_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = pref_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = pref_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = pref_ref;

  doit_buttons_def[ 0 ].title = msgGetText( MXDI_DEFAULT_COLORS_LABEL );


  /* Create a form dialog with buttons. */
  prefFd = xitCreateFormDialog( parent, "PrefFd",
                                1, 0,
                                action_buttons, 
                                XtNumber( action_buttons ) );

  XtAddCallback( prefFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) pref_ref );

  /* We want to know when this window is closed. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( prefFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( prefFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( prefFd ), wm_delete_window, 
                             (XtCallbackProc) cancelCB, (XtPointer) pref_ref );
  } /* block */

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( prefFd, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_PREF_TITLE ) ); n++;
  XtSetValues( XtParent( prefFd ), args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( prefFd, "PrefFdFo" );


  /* Form for the lines to show. */
  linesFr = XmCreateFrame( workFo, "LinesFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  linesRc = XmCreateRowColumn( linesFr, "LinesRc", args, n );

  linesLa = xitCreateLabel( linesRc, "LinesLa", 
                            msgGetText( MXDI_LINE_LABEL ), -1 );
  
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  lineEntryRc = XmCreateRowColumn( linesRc, "LineEntryRc", args, n );

  lineEntryLa = xitCreateLabel( 
                  lineEntryRc, "LineEntryLa", 
                  msgGetText( MXDI_LINE_ENTRY_LABEL ), -1 );

  lineEntryTx = xitCreateTextCols( lineEntryRc, &text_buffer[ 0 ], 5 );

  lineAlarmLa = xitCreateLabel( 
                  lineEntryRc, "LineAlarmLa", 
                  msgGetText( MXDI_LINE_ALARM_LABEL ), -1 );

  lineAlarmTx = xitCreateTextCols( lineEntryRc, &text_buffer[ 1 ], 5 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 3 );  n++;
  XtSetValues( lineEntryTx, args, n );  
  XtSetValues( lineAlarmTx, args, n );  


  /* Form for the foreground and background colors. */
  colorFr = XmCreateFrame( workFo, "ColorFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNadjustLast, False ); n++;
  colorRc = XmCreateRowColumn( colorFr, "ColorRc", args, n );

  colorLa = xitCreateLabel( colorRc, "ColorLa", 
                            msgGetText( MXDI_ENTRY_COLOR_LABEL ), -1 );
  
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing, 10 ); n++;
  colorEntryRc = XmCreateRowColumn( colorRc, "ColorEntryRc", args, n );

  colorFgCs = xtmFlCreateColorField( colorEntryRc, "ColorFgCs",
                                     5, 
                                     custom_data_ref -> color_fg_list,
                                     (XtCallbackProc) newColorCB,
                                     (void *) pref_ref );

  colorBgCs = xtmFlCreateColorField( colorEntryRc, "ColorBgCs",
                                     5, 
                                     custom_data_ref -> color_bg_list,
                                     (XtCallbackProc) newColorCB,
                                     (void *) pref_ref );

  sampleLa = xitCreateLabelWidget( colorRc, "SampleLa", 
                                   msgGetText( MXDI_COLOR_SAMPLE_LABEL ),
                                   -1 );

  defColorPb = xitCreatePushButton( colorRc, &doit_buttons_def[ 0 ] );
  
  n = 0;
  XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
  XtSetValues( defColorPb, args, n );

  XtAddCallback( defColorPb, XmNactivateCallback,
                 (XtCallbackProc) defaultColorCB, (XtPointer) pref_ref );


  /* Where the entry should be hidden. */
  hideFr = XmCreateFrame( workFo, "HideFr", args, 0 );
  hideRc = XmCreateRowColumn( hideFr, "HideRc", args, 0 );

  hideLa = xitCreateLabel( hideRc, "HideLa", 
                           msgGetText( MXDI_HIDE_IN_LABEL ), -1 );
  
  n = 0;
  XtSetArg( args[ n ], XmNnumColumns, 2 );  n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  hideWhatRc = XmCreateRowColumn( hideRc, "HideWhatRc", args, n );

  hideTb[ 0 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide1Tb", 
                 msgGetText( MXDI_HIDE_IN_CAL_LABEL ), False );

  hideTb[ 1 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide2Tb", 
                 msgGetText( MXDI_HIDE_IN_DAY_VIEW_LABEL ), False );

  hideTb[ 2 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide3Tb", 
                 msgGetText( MXDI_HIDE_IN_SUMMARY_LABEL ), False );

  hideTb[ 3 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide4Tb", 
                 msgGetText( MXDI_HIDE_IN_PRINT_LABEL ), False );

  hideTb[ 4 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide5Tb", 
                 msgGetText( MXDI_HIDE_IN_ALARM_LABEL ), False );

  hideTb[ 5 ] = xitCreateToggleButton(
                 hideWhatRc, "Hide6Tb", 
                 msgGetText( MXDI_HIDE_IN_DUMP_LABEL ), False );


  /* Various flags for the entry. */
  flagsFr = XmCreateFrame( workFo, "FlagsFr", args, 0 );
  flagsRc = XmCreateRowColumn( flagsFr, "FlagsRc", args, 0 );

  flagsTb[ 0 ] = xitCreateToggleButton(
                  flagsRc, "Flags1Tb", 
                  msgGetText( MXDI_PROT_NOT_OWNER_CHANGE_LABEL ), False );

  flagsTb[ 1 ] = xitCreateToggleButton(
                  flagsRc, "Flags2Tb", 
                  msgGetText( MXDI_PROT_NOT_OWNER_DELETE_LABEL ), False );

  flagsTb[ 2 ] = xitCreateToggleButton(
                  flagsRc, "Flags3Tb", 
                  msgGetText( MXDI_KEEP_ENTRY_IN_FG_LABEL ), False );

  flagsTb[ 3 ] = xitCreateToggleButton(
                  flagsRc, "Flags4Tb", 
                  msgGetText( MXDI_KEEP_ENTRY_IN_BG_LABEL ), False );


  /* Put the elements together. */
  xitAttachWidget( linesFr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );
  xitAttachWidget( colorFr,
                   XmATTACH_WIDGET, linesFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( hideFr,
                   XmATTACH_WIDGET, colorFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( flagsFr,
                   XmATTACH_WIDGET, hideFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( linesFr, args, n );
  XtSetValues( hideFr,  args, n );
  XtSetValues( flagsFr, args, n );
  XtSetValues( colorFr, args, n );


  /* Manage the widgets. */
  xitManageChildren( hideTb,  XtNumber( hideTb ) );
  xitManageChildren( flagsTb, XtNumber( flagsTb ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the initial sizes. */
  xitSetSizeFormDialog( prefFd, False );


  return( prefFd );

} /* createPrefWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchPrefValues( PREF_REC_REF          pref_ref,
                   XTM_EP_GET_PREF_DATA  *values_ref )
{

  /* Variables. */
  Boolean  ok;
  int      color_index;
  int      index;
  char     buffer[ 100 ];
  char     *char_ref;
  Widget   baseW;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  /* We need a waindow. */
  if( pref_ref -> prefW == NULL )
    return( False );


  /* Preference window. */
  mainW = XtNameToWidget( pref_ref -> prefW, "PrefFdFo" );


  /* Default values. */
  values_ref -> alarm_lines    = 0;
  values_ref -> day_list_lines = 0;
  values_ref -> flags          = 0;
  values_ref -> bg_color_index = 0;
  values_ref -> fg_color_index = 0;


  /* Lines to show. */
  sprintf( buffer, "LinesFr.LinesRc.LineEntryRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "LineEntryTx" );

  char_ref = xitStringGetText( tempW );
  index    = atoi( char_ref );
  SysFree( char_ref );

  if( index > 0 )
    values_ref -> day_list_lines = index;

  tempW = XtNameToWidget( baseW, "LineAlarmTx" );

  char_ref = xitStringGetText( tempW );
  index    = atoi( char_ref );
  SysFree( char_ref );

  if( index > 0 )
    values_ref -> alarm_lines = index;


  /* Hide in? */
  sprintf( buffer, "HideFr.HideRc.HideWhatRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "Hide1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_CALENDAR );

  tempW = XtNameToWidget( baseW, "Hide2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW );

  tempW = XtNameToWidget( baseW, "Hide3Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_SUMMARY );

  tempW = XtNameToWidget( baseW, "Hide4Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_PRINT );

  tempW = XtNameToWidget( baseW, "Hide5Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_ALARM );

  tempW = XtNameToWidget( baseW, "Hide6Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_DUMP );


  /* Flags for the entry? */
  sprintf( buffer, "FlagsFr.FlagsRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "Flags1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_ONLY_OWNER_CHANGE );

  tempW = XtNameToWidget( baseW, "Flags2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_ONLY_OWNER_DELETE );

  tempW = XtNameToWidget( baseW, "Flags3Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_KEEP_IN_FG );

  tempW = XtNameToWidget( baseW, "Flags4Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( values_ref -> flags, XTM_DB_FLAG_KEEP_IN_BG );


  /* Foreground and background color. */
  sprintf( buffer, "ColorFr.ColorRc.ColorEntryRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "ColorBgCs" );
  ok = xitColorSelectGetIndex( tempW, &color_index );

  if( ok )
    values_ref -> bg_color_index = color_index;

  tempW = XtNameToWidget( baseW, "ColorFgCs" );
  ok = xitColorSelectGetIndex( tempW, &color_index );

  if( ok )
    values_ref -> fg_color_index = color_index;


  return( True );

} /* fetchPrefValues */


/*----------------------------------------------------------------------*/

static void
  setPrefValues( PREF_REC_REF          pref_ref,
                 XTM_EP_SET_PREF_DATA  *values_ref )
{

  /* Variables. */
  int                     color_index;
  char                    buffer[ 80 ];
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = pref_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* We need a waindow. */
  if( pref_ref -> prefW == NULL )
    return;


  mainW = XtNameToWidget( pref_ref -> prefW, "PrefFdFo" );


  /* Lines to show. */
  sprintf( buffer, "LinesFr.LinesRc.LineEntryRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "LineEntryTx" );

  sprintf( buffer, "%d ", values_ref -> day_list_lines );
  XmTextSetString( tempW, buffer );

  tempW = XtNameToWidget( baseW, "LineAlarmTx" );

  sprintf( buffer, "%d ", values_ref -> alarm_lines );
  XmTextSetString( tempW, buffer );


  /* Hide the entry in? */
  sprintf( buffer, "HideFr.HideRc.HideWhatRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "Hide1Tb" );

  if( values_ref -> entry_type == XTM_DB_TYPE_APPOINTMENT ||
      values_ref -> entry_type == XTM_DB_TYPE_NOTE ) {
    XmToggleButtonSetState( tempW, False, False );
    XtSetSensitive( tempW, False );

  } else {
    if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_CALENDAR ) )
      XmToggleButtonSetState( tempW, True, False );
    else
      XmToggleButtonSetState( tempW, False, False );
    XtSetSensitive( tempW, True );
  }

  tempW = XtNameToWidget( baseW, "Hide2Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Hide3Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_SUMMARY ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Hide4Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_PRINT ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Hide5Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_ALARM ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Hide6Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_HIDE_IN_DUMP ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Flags for the entry? */
  sprintf( buffer, "FlagsFr.FlagsRc" );
  baseW = XtNameToWidget( mainW, buffer );

  tempW = XtNameToWidget( baseW, "Flags1Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_ONLY_OWNER_CHANGE ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Flags2Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_ONLY_OWNER_DELETE ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Flags3Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_KEEP_IN_FG ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "Flags4Tb" );
  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_KEEP_IN_BG ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Foreground and background color. */
  sprintf( buffer, "ColorFr.ColorRc.ColorEntryRc" );
  baseW = XtNameToWidget( mainW, buffer );

  color_index = values_ref -> bg_color_index;
  if( color_index < 0 ||
      color_index >= custom_data_ref -> colorsDB_bg.no_colors )
    color_index = 0;

  tempW = XtNameToWidget( baseW, "ColorBgCs" );
  xitColorSelectSetIndex( tempW, color_index, False );

  color_index = values_ref -> fg_color_index;
  if( color_index < 0 ||
      color_index >= custom_data_ref -> colorsDB_fg.no_colors )
    color_index = 0;

  tempW = XtNameToWidget( baseW, "ColorFgCs" );
  xitColorSelectSetIndex( tempW, color_index, True );


  return;

} /* setPrefValues */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget        widget,
            PREF_REC_REF  pref_ref,
            XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( pref_ref -> actionCB != NULL )
    (* pref_ref -> actionCB)( XTM_EP_REASON_CANCEL,
                              pref_ref -> user_data, NULL );

  XtUnmanageChild( pref_ref -> prefW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  defaultColorCB( Widget        widget,
                  PREF_REC_REF  pref_ref,
                  XtPointer     call_data )
{

  /* Variables. */
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( pref_ref -> prefW, "PrefFdFo" );

  /* Set the new colors. */
  baseW = XtNameToWidget( mainW, "ColorFr.ColorRc.ColorEntryRc" );

  tempW = XtNameToWidget( baseW, "ColorBgCs" );
  xitColorSelectSetIndex( tempW, 0, False );

  tempW = XtNameToWidget( baseW, "ColorFgCs" );
  xitColorSelectSetIndex( tempW, 0, True );


  return;

} /* defaultColorCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             PREF_REC_REF  pref_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( pref_ref -> actionCB != NULL )
    (* pref_ref -> actionCB)( XTM_EP_REASON_DESTROY,
                              pref_ref -> user_data, NULL );

  SysFree( pref_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget        widget,
          PREF_REC_REF  pref_ref,
          XtPointer     call_data )
{

  /* Code. */

  xtmHlDisplayHelp( pref_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    pref_window_id, "" );


  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  newColorCB( Widget                        widget,
              PREF_REC_REF                  pref_ref,
              xitColorSelectCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean   ok;
  Arg       args[ 10 ];
  Cardinal  n;
  Pixel     bg;
  Pixel     fg;
  Widget    baseW;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( pref_ref -> prefW, "PrefFdFo" );


  /* Fetch the selected background and foreground. */
  baseW = XtNameToWidget( mainW, "ColorFr.ColorRc.ColorEntryRc" );

  tempW = XtNameToWidget( baseW, "ColorBgCs" );
  ok    = xitColorSelectGetPixel( tempW, &bg );

  tempW = XtNameToWidget( baseW, "ColorFgCs" );
  ok    = xitColorSelectGetPixel( tempW, &fg );


  /* Set the color sample. */
  baseW = XtNameToWidget( mainW, "ColorFr.ColorRc" );
  tempW = XtNameToWidget( baseW, "SampleLa" );

  n = 0;
  XtSetArg( args[ n ], XmNbackground, bg ); n++;
  XtSetArg( args[ n ], XmNforeground, fg ); n++;
  XtSetValues( tempW, args, n );  


  return;

} /* newColorCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget        widget,
        PREF_REC_REF  pref_ref,
        XtPointer     call_data )
{

  /* Variables. */
  Boolean               ok;
  XTM_EP_GET_PREF_DATA  values;


  /* Code. */

  ok = fetchPrefValues( pref_ref, &values );
  if( ! ok )
    return;

  /* Do we have a user action callback registered? */
  if( pref_ref -> actionCB != NULL )
    (* pref_ref -> actionCB)( XTM_EP_REASON_OK,
                              pref_ref -> user_data,
                              &values );

  XtUnmanageChild( pref_ref -> prefW );


  return;

} /* okCB */
