/*----------------------------------------------------------------------------
--
--  Module:           xtmFreeze
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    A quick way to freeze a standing entry. The standing entry is
--    skipped the appropriate week and a stand-alone copy is made
--    of the entry.
--
--  Filename:         xtmFreeze.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-02-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmFreeze.c, Version: 1.1, Date: 95/02/18 15:52:22";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "xtmFreeze.h"
#include "XmUbTimeB.h"
#include "xtmDbMisc.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the duplicate window. */
#define messageLa           dataLocalW[  0 ]
#define parRc               dataLocalW[  1 ]
#define skipWeekTb          dataLocalW[  2 ]
#define toTimeTb            dataLocalW[  3 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Local information. */
typedef struct {

  /* Entry to freeze. */
  UINT32  entry_id;

  /* Calendar. */
  char  *calendar;

  /* Local window. */
  Widget  localW;

  /* Date for the entry. */
  TIM_TIME_REF  entry_date;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

} LOCAL_REC, *LOCAL_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmFreeze";

/* IDs for the help windows. */
static char  *help_window_id = "TearOff";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Widget
  createWindow( Widget                parentW,
                LOCAL_REF             local_ref,
                XTM_DB_ALL_ENTRY_REF  entry_ref );

static void 
  destroyCB( Widget     widget,
             LOCAL_REF  local_ref,
             XtPointer  call_data );

static void 
  helpCB( Widget     widget,
          LOCAL_REF  local_ref,
          XtPointer  call_data );

static void
  okCB( Widget     widget,
        LOCAL_REF  local_ref,
        XtPointer  call_data );

static void
  setValues( LOCAL_REF             local_ref,
             XTM_DB_ALL_ENTRY_REF  entry_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmFzFreezeEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Widget                parentW,
                    char                  *calendar,
                    UINT32                entry_id,
                    TIM_TIME_REF          entry_date )
{

  /* Variables. */
  Boolean                 ok;
  LOCAL_REF               local_ref;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, calendar,
                          &db_info, NULL );
  if( ! ok )
    return;

  /* We need write access. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) ) {
    xitErMessage( parentW, XIT_ER_ERROR, 
                  module_name, "xtmFzFreezeEntry",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );
    return;
  }


  /* Create and initialize our private data. */
  local_ref = SysNew( LOCAL_REC );
  if( local_ref == NULL )
    return;

  local_ref -> appl_data_ref = appl_data_ref;
  local_ref -> calendar      = SysNewString( calendar );
  local_ref -> entry_id      = entry_id;
  local_ref -> entry_date    = entry_date;


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, calendar, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    raise exception;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               &entry_record, NULL );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( parentW, XIT_ER_ERROR,
                  module_name, "xtmFzFreezeEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }


  /* Only standing entries can be frozen. */
  if( entry_record.entry.entry_category != XTM_DB_REP_ENTRY_LIST ) {
    xitErMessage( parentW, XIT_ER_ERROR,
                  module_name, "xtmFzFreezeEntry",
                  msgGetText( MXDI_FREEZE_ONLY_REPEATED ) );

    raise exception;
  }


  /* Create the freeze window. */
  local_ref -> localW = createWindow( parentW, local_ref, &entry_record );
  if( local_ref -> localW == NULL )
    raise exception;

  setValues( local_ref, &entry_record );


  /* Make the form visible. */
  XtManageChild( local_ref -> localW );


  return;


  /* Exception handler. */
  exception:
    SysFree( local_ref -> calendar );
    SysFree( local_ref );

    return;

} /* xtmFzFreezeEntry */


/*----------------------------------------------------------------------*/

static Widget
  createWindow( Widget                parentW,
                LOCAL_REF             local_ref,
                XTM_DB_ALL_ENTRY_REF  entry_ref )
{

  /* Variables. */
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 4 ];
  Widget                  freezeFd;
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,   NULL },
    { "", NULL,   NULL },
    { "", helpCB, NULL },
  };


  /* Code. */

  appl_data_ref = local_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = local_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = local_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = local_ref;


  /* Create a form dialog with buttons. */
  freezeFd = xitCreateFormDialog( parentW, "FreezeFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_FREEZE_TITLE ) ); n++;
  XtSetValues( XtParent( freezeFd ), args, n );

  XtAddCallback( freezeFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) local_ref );

  workFo = XtNameToWidget( freezeFd, "FreezeFdFo" );


  /* Information message. */
  messageLa = xitCreateLabel( workFo, "MessageLa",
                              msgGetText( MXDI_FREEZE_INFO_LABEL ), -1 );


  /* Date and time selection. */
  if( entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY )
    toTimeTb = xtmFlCreateDateTimeField( workFo, "ToTimeTb" );
  else
    toTimeTb = xtmFlCreateDateField( workFo, "ToTimeTb" );


  /* Parameters. */
  parRc = XmCreateRowColumn( workFo, "ParRc", args, 0 );

  skipWeekTb = xitCreateToggleButton( 
                 parRc, "SkipWeekTb",
                 msgGetText( MXDI_FREEZE_DO_SKIP_WEEK ), True );


  /* Put the parts together. */
  xitAttachWidget( messageLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( toTimeTb,
                   XmATTACH_WIDGET, messageLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( parRc,
                   XmATTACH_WIDGET, toTimeTb, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( messageLa, args, n );
  XtSetValues( toTimeTb,  args, n );
  XtSetValues( parRc,     args, n );


  /* Manage all the children. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( freezeFd, True );


  return( freezeFd );

} /* createWindow */


/*----------------------------------------------------------------------*/

static void
  setValues( LOCAL_REF             local_ref,
             XTM_DB_ALL_ENTRY_REF  entry_ref )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( local_ref -> localW, "FreezeFdFo" );


  /* To date and time. */
  tempW = XtNameToWidget( mainW, "ToTimeTb"  );

  (void) XmUbTimeBoxSetStartDate( tempW, local_ref -> entry_date );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY )
    (void) XmUbTimeBoxSetStartTime( tempW, entry_ref -> entry.time_stamp );


  return;

} /* setValues */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget     widget,
             LOCAL_REF  local_ref,
             XtPointer  call_data )
{

  /* Code. */

  /* Release the user data. */
  SysFree( local_ref -> calendar );
  SysFree( local_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget     widget,
          LOCAL_REF  local_ref,
          XtPointer  call_data )
{

  /* Code. */

  xtmHlDisplayHelp( local_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    help_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget     widget,
        LOCAL_REF  local_ref,
        XtPointer  call_data )
{

  /* Variables. */
  Boolean                 ok;
  UINT32                  id;
  char                    *entry_text;
  Widget                  mainW;
  Widget                  tempW;
  TIM_TIME_REF            to_date;
  TIM_TIME_REF            to_time;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_ID_REQUEST       id_request;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = local_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( local_ref -> localW, "FreezeFdFo" );


  /* Open the calendar where the entry is. */
  ok = xtmDmOpenDatabase( appl_data_ref, local_ref -> calendar,
                          XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;


  /* Fetch the entry again. */
  db_status = xtmDbFetchEntry( &database, local_ref -> entry_id, 
                               &entry_record, &entry_text );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( local_ref -> localW, XIT_ER_ERROR,
                  module_name, "okCB",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }


  /* Fetch the new date and time to use. */
  tempW = XtNameToWidget( mainW, "ToTimeTb" );

  ok = xtmFoFetchDate( local_ref -> localW, tempW, XTM_FO_START_DATE, False,
                       &to_date );
  if( ! ok )
    raise exception;

  if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY ) {
    ok = xtmFoFetchTime( local_ref -> localW, tempW, XTM_FO_START_TIME, False,
                         &to_time );
    if( ! ok )
      raise exception;
  }


  /* Skip the entry this week? */
  tempW = XtNameToWidget( mainW, "ParRc.SkipWeekTb" );

  /* Mark the week as 'skipped'. */
  if( XmToggleButtonGetState( tempW ) ) {

    int     week_no;
    UINT32  *flags;
    UINT32  skip_flag;

    week_no = TimIndexOfWeek( local_ref -> entry_date );

    skip_flag = (1 << (week_no % 30));
    if( week_no  > 30 )
      flags = &entry_record.stand_entry.skip_week[ 0 ];
    else
      flags = &entry_record.stand_entry.skip_week[ 1 ];

    flagSet( *flags, skip_flag );


    /* Save the modified entry. */
    (void) xtmDbDeleteEntry( &database, local_ref -> entry_id );

    ok = xtmDmInsertEntry( local_ref -> localW,
                           &database, local_ref -> entry_id,
                           &entry_record, entry_text );
    if( ! ok )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "okCB",
                    msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );

  } /* if */


  /* Make a normal entry out of the repeated one. */
  entry_record.entry.entry_category = XTM_DB_ENTRY_LIST;
  entry_record.entry.date_stamp     = (UINT32) to_date;

  if( entry_record.entry.entry_type == XTM_DB_DAY_ENTRY )
    entry_record.entry.time_stamp = (UINT32) to_time;


  /* Generate new entry id. */
  id_request.directory  = database.database_dir;
  id_request.operations = 0;
  id_request.lock_file  = False;

  db_status = xtmDbGenerateId( &id_request, &id );
  if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "okCB",
                    msgGetText( MXDI_ERRMSG_GENERATE_ID ) );

  entry_record.entry.id       = id;
  entry_record.stand_entry.id = id;


  /* Write the entry. */
  db_status = xtmDbInsertEntry( &database, &entry_record, entry_text );

  if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "okCB",
                    msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );


  /* Now we are done. */
  if( entry_text != NULL )
    SysFree( entry_text );

  xtmDbCloseEntryDb( &database );

  XtDestroyWidget( local_ref -> localW );


  /* Update the day list/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );

  
  return;

  /* Exception handler. */
  exception:
    xtmDbCloseEntryDb( &database );

    return;

} /* okCB */
