/*----------------------------------------------------------------------------
--
--  Module:           xtmTags
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Handles the various tags used in XDiary.
--
--  Filename:         xtmTags.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-03-05
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmTags.c, Version: 1.1, Date: 95/02/18 15:52:53";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include "System.h"

#include "xtmTags.h"
#include "xitTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define MAX_TAGS        100
#define MAX_TAG_LENGTH  20


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for a tag with data. */
typedef struct {
  Boolean  bg_color_valid;
  Boolean  fg_color_valid;
  char     tag[ MAX_TAG_LENGTH + 1 ];
  Pixel    fg_color;
  Pixel    bg_color;
} TAG_REC, *TAG_REC_REF;

/* The tags database. */
typedef struct {
  int          tags_no;
  TAG_REC_REF  tags;
} TAGS_DB, *TAGS_DB_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmTags";



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_TG_HANDLE
  xtmTgBuildTagDb( Widget  parentW,
                   char    *tags_ref )
{

  /* Variables. */
  int          char_read;
  int          items;
  int          tag_count;
  char         buffer[ 250 ];
  char         bg_color_buffer[ 250 ];
  char         fg_color_buffer[ 250 ];
  char         tag_buffer[ 250 ];
  char         *char_ref;
  TAG_REC      tags[ MAX_TAGS ];
  TAGS_DB_REF  tags_db_ref;


  /* Code. */

  char_ref  = tags_ref;
  tag_count = 0;

  do {
    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    items = sscanf( char_ref, "%s%n", buffer, &char_read );
    if( items != 1 )
      break;

    /* Any color definition? */
    tags[ tag_count ].bg_color_valid = False;
    tags[ tag_count ].fg_color_valid = False;

    items = sscanf( buffer, "%[^(](%[^,],%[^)])",
                    tag_buffer, fg_color_buffer, bg_color_buffer );

    if( items == 3 ) {

      Boolean  bg_ok;
      Boolean  fg_ok;
      Pixel    bg_color;
      Pixel    fg_color;

      fg_ok = xitAllocNamedColor( parentW, fg_color_buffer, &fg_color );
      bg_ok = xitAllocNamedColor( parentW, bg_color_buffer, &bg_color );

      tags[ tag_count ].bg_color_valid = bg_ok;
      tags[ tag_count ].bg_color       = bg_color;

      tags[ tag_count ].fg_color_valid = fg_ok;
      tags[ tag_count ].fg_color       = fg_color;

    }

    /* Save the tag. */
    strncpy( tags[ tag_count ].tag, tag_buffer, MAX_TAG_LENGTH );
    tags[ tag_count ].tag[ MAX_TAG_LENGTH ] = '\0';


    tag_count++;

    char_ref = char_ref + char_read;

  } while( tag_count < MAX_TAGS );


  /* Allocate memory to hold the tags DB. */
  tags_db_ref = (TAGS_DB_REF) SysMalloc( sizeof( TAGS_DB ) );
  tags_db_ref -> tags_no = tag_count;
  tags_db_ref -> tags    = NULL;

  if( tag_count > 0 ) {

    TAG_REC_REF  tags_ref;

    tags_ref = (TAG_REC_REF) SysMalloc( sizeof( TAG_REC ) * tag_count );

    memcpy( (void *) tags_ref,
            (void *) tags,
            sizeof( TAG_REC ) * tag_count );

    tags_db_ref -> tags = tags_ref;
  }


  return( (XTM_TG_HANDLE) tags_db_ref );

} /* xtmTgBuildTagDb */


/*----------------------------------------------------------------------*/

Boolean
  xtmTgFetchTagColor( XTM_TG_HANDLE  tag_handle,
                      char           *search_tag,
                      Pixel          *fg_color,
                      Pixel          *bg_color )
{

  /* Variables. */
  int          index;
  TAGS_DB_REF  tags_db_ref;


  /* Code. */

  if( tag_handle == NULL )
    return(False );

  tags_db_ref = (TAGS_DB_REF) tag_handle;

  if( tags_db_ref -> tags == NULL )
    return( False );

  for( index = 0; index < tags_db_ref -> tags_no; index++ ) {

    if( strcmp( tags_db_ref -> tags[ index ].tag, search_tag ) == 0 ) {
      if( ! tags_db_ref -> tags[ index ].fg_color_valid ||
          ! tags_db_ref -> tags[ index ].bg_color_valid )
        return( False );

      *fg_color = tags_db_ref -> tags[ index ].fg_color;
      *bg_color = tags_db_ref -> tags[ index ].bg_color;

      return( True );
    }

  } /* loop */


  return( False );

} /* xtmTgFreeTagDb */


/*----------------------------------------------------------------------*/

void
  xtmTgFreeTagDb( XTM_TG_HANDLE  tag_handle )
{

  /* Variables. */
  TAGS_DB_REF  tags_db_ref;


  /* Code. */

  if( tag_handle == NULL )
    return;

  tags_db_ref = (TAGS_DB_REF) tag_handle;

  if( tags_db_ref -> tags != NULL )
    SysFree( tags_db_ref -> tags );

  SysFree( tags_db_ref );


  return;

} /* xtmTgFreeTagDb */

