/*----------------------------------------------------------------------------
--
--  Module:           DirList
--
--  Project:          Tools - General C objects.
--  System:           Dir - Directory routines.
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This module contains functions to read the contents of a
--    directory.
--
--    To list the contents of a directory, you can use the following
--    functions:
--
--      DirFindFirst
--      DirFindNext
--      DirEnd
-- 
--    DirFindFirst() is given a filespec which may contain wildcard
--    characters. It searches the directory specified by filespec
--    for files which match. If at least one such file is found,
--    the first matching file is returned (files are ordered
--    alphabetically). Matching entries which are directories are
--    not returned.
--
--    DirFindNext() returns the next matching file found.
--
--    DirEnd() releases memory allocated by DirFindFirst().
--
--    DirFindFirst() and DirFindNext() return -1 if there are no matches
--    (or no remaining matches in the case of DirFindNext()) or 0 if a match
--    was returned.
--
--  Filename:         DirList.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-09
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: DirList.c, Version: 1.1, Date: 95/02/18 14:32:26";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>
 
#include "System.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* If FULL_LIST is defined, return all entries (including directories). */

/* Various constants to use. */
#define PATH_SEPARATOR  '/'


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

static int     current_entry;
static char*   file;
static size_t  file_length;
static char    local_spec[ PATH_MAX + 1 ];
static int     match_count;
static char**  name_list;
static char*   path;
static int     simple_path;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static int
  compareStringsCB( const void  *string1,
                    const void  *string2 );

static int
  matchFilename( struct dirent* entry );
 


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

int
  DirFindFirst( char *filespec, 
                char *filename )
{

  /* Variables. */
  int            dir_size;
  char           *pathend;
  DIR            *directory;
  struct dirent  *dirp;


  /* Code. */

  match_count = 0;

  if( ! filespec )
    return( -1 );
 
  strcpy( local_spec, filespec );

  pathend = strrchr( local_spec, PATH_SEPARATOR );
  if( ! pathend ) {

    file = local_spec;
    path = ".";
    simple_path = 1;

  } else {

    *pathend = '\0';
    file = pathend + 1;
    path = local_spec;
    simple_path = 0;

  } /* if */

  file_length = strlen( file );
 
  /* Verify we have a directory name. */
  directory = opendir( path );
  if( ! directory )
    return( -1 );

  dir_size = 100;
  name_list = (char**) SysMalloc( dir_size * sizeof( char* ) );

  while( (dirp = readdir( directory )) != NULL ) {
    if( matchFilename( dirp ) ) {
      if( match_count == dir_size ) {
        dir_size = dir_size + 100;
        name_list = (char**) SysRealloc( name_list,
                                         dir_size * sizeof( char* ) );
      }
      name_list[ match_count ] = SysNewString( dirp -> d_name );
      match_count++;
    }
  }

  closedir( directory );

  if( match_count == 0 ) {
    SysFree( name_list );
    return( -1 );
  } 
 
  qsort( (void*) name_list, (size_t) match_count, sizeof( char* ),
         compareStringsCB );

  current_entry = 0;
  if( !simple_path ) {
    strcpy( filename, path );
    strcat( filename, "/" );
    strcat( filename, name_list[ current_entry ] );
  } else {
    strcpy( filename, name_list[current_entry ] );
  }

  SysFree( name_list[ current_entry++ ] );
  match_count--;

 
  return( 0 );

} /* DirFindFirst */


/*----------------------------------------------------------------------*/

int
  DirFindNext( char* filename )
{

  /* Code. */

  if( ! match_count )
    return( -1 );
 
  if( ! simple_path ) {
    strcpy( filename, path );
    strcat( filename, "/" );
    strcat( filename, name_list[ current_entry ] );
  } else {
    strcpy( filename, name_list[ current_entry ] );
  }

  SysFree( name_list[ current_entry++ ] );
  if( ! (--match_count) )
    SysFree( name_list );


  return( 0 );

} /* DirFindNext */


/*----------------------------------------------------------------------*/

void
  DirEnd()
{

  /* Code. */

  if( ! match_count )
    return;

  while( match_count ) {
    SysFree( name_list[ current_entry++ ] );
    if( ! (--match_count) )
      SysFree( name_list );
  }


  return;

} /* DirEnd */


/*----------------------------------------------------------------------*/

static int
  compareStringsCB( const void  *string1,
                    const void  *string2 )
{
 
  /* Code. */
 
  return( strcmp( (char *) string1, (char *) string2 ) );
 
} /* compareStringsCB */
 
 
/*----------------------------------------------------------------------*/

static int
  matchFilename( struct dirent* entry )
{

  /* Variables. */
  DIR   *directory;
  char  filename[ PATH_MAX + 1 ];
  char  *scan_entry;
  char  *scan_match;


  /* Code. */
  if( ! entry )
    return( 0 );

  if( file_length > strlen( entry -> d_name ) )
    return( 0 );
 
  scan_match = file;
  scan_entry = entry -> d_name;

  while( *scan_match != '\0' ) {

    /* If match string has a '*', match all characters until
     * the next character in match string is encountered.    */
 
    if( '*' == *scan_match ) {
      scan_match++;

      while( *scan_entry != *scan_match ) {

        /* Matching end-of-string is OK but since we didn't match,
         * this end-of-string in the entry means we've run out so
         * no match! */
 
        if( '\0' == *scan_entry )
          return( 0 );

        scan_entry++;

      } /* while */


    /* A '?' matches any character and, of course, we continue if
     * the corresponding characters do in fact match. */
 
    } else if( '?' == *scan_match || *scan_entry == *scan_match ) {

      scan_match++;
      scan_entry++;

    /* Otherwise, no good so return 0. */
    } else {

      return( 0 );

    } /* if */

  } /* while */
 
  /* We matched all of the match string if we got here.
   * But we could have some of the entry string left. */
 
  if( *scan_entry != '\0' )
    return 0;
 
  /* Make sure this entry is not itself a directory. */
  strcpy( filename, path );
  strcat( filename, "/" );
  strcat( filename, entry -> d_name );

  directory = opendir( filename );
  if( ! directory )
    return( 1 );

  closedir( directory );


  return( 0 );

} /* matchFilename */
