/*----------------------------------------------------------------------------
--
--  Module:           Message
--
--  Project:          Tools - General C objects.
--  System:           Msg - Message catalogue
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Handles the message catalogue.
--
--  Filename:         Message.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-03-14
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: Message.c, Version: 1.1, Date: 95/02/18 14:32:29";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#include "Standard.h"
#include "System.h"

#include "Message.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Exception handler (raise exception). */
#define raise  goto


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record containing messages for one text domain. */
typedef struct {
  char  domain_name[ 40 ];
  int   no_of_messages;
  char  **msg_index;
  char  *msg_text;
} MSG_DOMAIN_MESSAGES;

/* Record with all text domains. */
typedef struct {
  int                  no_of_domains;
  MSG_DOMAIN_MESSAGES  *domains[ MSG_MAX_DOMAINS ];
} MSG_DOMAINS;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* The text domains. */
static  MSG_DOMAINS  all_domains;

/* The current text domain and index. */
static char  current_domain[ 40 ];
static char  old_current_domain[ 40 ];
static int   current_domain_index;

/* Default message if requested message not found. */
static char *default_message = "Undef";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

extern  char *getenv( char *);



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Boolean
  msgAddFileTextDomain( char  *domain_name,
                        char  *msg_file )
{

  /* Types. */
  typedef enum {LINE_START, LINE_CONT, LINE_INVALID } LINE_STATE;

  /* Variables. */
  int         chars;
  int         index;
  int         items;
  int         length;
  int         msg_id;
  int         msg_size;
  int         msg_text_len;
  int         no_of_msgs;
  char        *char_ref;
  char        *line_ref       = NULL;
  char        **msg_index_ref = NULL;
  char        *msg_text_ref   = NULL;
  char        buffer[ 250 ];
  char        language[ 20 ];
  char        version[ 20 ];
  FILE        *file_ref  = NULL;
  LINE_STATE  line_state = LINE_START;


  /* Code. */

  /* Can we store this text domain? */
  if( all_domains.no_of_domains >= MSG_MAX_DOMAINS )
    return( False );


  /* Try to open the text file for read. */
  file_ref = fopen( msg_file, "r" );
  if( file_ref == NULL )
    raise exception;


  /* Fetch the catalogue id line. */
  char_ref = fgets( buffer, sizeof( buffer ), file_ref );
  if( char_ref == NULL || strncmp( buffer, "@(#):", 5 ) != 0 ) {
    fprintf( stderr, "Message: %s is not a message catalogue!\n",
             msg_file );
    raise exception;
  }

  sscanf( buffer, "@(#):%[^:]:%[^:]", version, language );

  
  /* Fetch the size line. */
  char_ref = fgets( buffer, sizeof( buffer ), file_ref );
  if( char_ref == NULL || buffer[ 0 ] != 's' ) {
    fprintf( stderr, "Message: %s is not a message catalogue!\n",
             msg_file );
    raise exception;
  }

  sscanf( buffer, "s%d:%d", &no_of_msgs, &msg_size );


  /* Allocate the memory needed. */
  msg_index_ref = (char **) SysMalloc( no_of_msgs * sizeof( char * ) );
  if( msg_index_ref == NULL )
    raise exception;

  msg_text_len = 0;
  msg_text_ref = SysMalloc( msg_size );
  if( msg_text_ref == NULL )
    raise exception;


  /* Start to read the messages from file. */
  while( True ) {

    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      break;


    /* Start of a message line? */
    if( buffer[ 0 ] == 'i' ) {

      /* Get the message identifier. */
      chars = strlen( buffer );
      items = sscanf( buffer, "i%d:%n", &msg_id, &chars );
      if( items != 1 )
        continue;

      line_ref   = &buffer[ chars ];
      line_state = LINE_START;


    /* Continuation line? */
    } else if( buffer[ 0 ] == '>' ) {

      line_ref   = &buffer[ 1 ];
      line_state = LINE_CONT;

    } else {
      line_state = LINE_INVALID;

    } /* if */


    /* Line continuation? */
    if( line_state == LINE_CONT )
      msg_text_len--;

    /* Save the index?. */
    if( line_state == LINE_START )
      *(msg_index_ref + msg_id) = (char *)(msg_text_ref + msg_text_len);

    /* Save the message text. */
    if( line_state != LINE_INVALID ) {

      length = strlen( line_ref );
      if( *(line_ref + length - 1) == '\n' )
        *(line_ref + length - 1) = '\0';

      /* Convert message catalogue special characters. */
      char_ref = line_ref;
      while( *char_ref != '\0' ) {
        switch( *char_ref ) {
          case 0x01:
            *char_ref = '\n';
            break;
          default:
            break;
	}

        char_ref++;
      }

      strcpy( msg_text_ref + msg_text_len, line_ref );

      msg_text_len = msg_text_len + strlen( line_ref ) + 1;
    }

  } /* while. */

  fclose( file_ref );


  /* Save domain name and messages. */
  index = all_domains.no_of_domains;

  all_domains.domains[ index ] = SysNew( MSG_DOMAIN_MESSAGES );

  all_domains.domains[ index ] -> no_of_messages = no_of_msgs;
  all_domains.domains[ index ] -> msg_index      = msg_index_ref;
  all_domains.domains[ index ] -> msg_text       = msg_text_ref;

  strcpy( all_domains.domains[ index ] -> domain_name, domain_name );

  all_domains.no_of_domains++;


  return( True );


  /* Exception handler. */
  exception:
    if( file_ref != NULL )
      fclose( file_ref );

    if( msg_index_ref != NULL )
      SysFree( msg_index_ref );

    if( msg_text_ref != NULL )
      SysFree( msg_text_ref );

    return( False );

} /* msgAddFileTextDomain */


/*----------------------------------------------------------------------*/

Boolean
  msgAddTextDomain( char  *domain_name,
                    char  **message_index )
{

  /* Variables. */
  int  index;
  int  no_of_messages;


  /* Code. */

  if( all_domains.no_of_domains >= MSG_MAX_DOMAINS )
    return( False );


  /* How many messages do we have? */
  no_of_messages = 0;

  while( True ) {
    if( *(message_index + no_of_messages) == NULL )
      break;

    no_of_messages++;
  }


  /* Save domain name and messages. */
  index = all_domains.no_of_domains;

  all_domains.domains[ index ] = SysNew( MSG_DOMAIN_MESSAGES );

  all_domains.domains[ index ] -> no_of_messages = no_of_messages;
  all_domains.domains[ index ] -> msg_index      = message_index;
  all_domains.domains[ index ] -> msg_text       = NULL;

  strcpy( all_domains.domains[ index ] -> domain_name, domain_name );

  all_domains.no_of_domains++;


  return( True );

} /* msgAddTextDomain */


/*----------------------------------------------------------------------*/

char
  *msgDGetText( char  *domain_name,
                int   message_id )
{

  /* Variables. */
  int  domain_index;
  int  index;


  /* Code. */

  for( index = 0; index < all_domains.no_of_domains; index++ ) {
    if( strcmp( all_domains.domains[ index ] -> domain_name, 
                domain_name ) == 0 )
      break;
  }

  if( index == all_domains.no_of_domains )
    return( default_message );

  domain_index = index;


  /* Message ids are alwaus integers. */
  index = message_id;

  if( index < 0 || 
      index >= all_domains.domains[ domain_index ] -> no_of_messages )
    return( default_message );


  return( all_domains.domains[ domain_index ] -> msg_index[ index ] );

} /* msgDGetText */


/*----------------------------------------------------------------------*/

char
  *msgGetText( int  message_id )
{

  /* Variables. */
  int   index;


  /* Code. */

  if( current_domain_index < 0 )
    return( default_message );

  /* Message ids are always integers. */
  index = message_id;

  if( index < 0 || 
      index >= all_domains.domains[ current_domain_index ] -> no_of_messages )
    return( default_message );

  return( all_domains.domains[ current_domain_index ] -> msg_index[ index ] );

} /* msgGetText */


/*----------------------------------------------------------------------*/

void
  msgInitialize()
{

  /* Variables. */
  int  index;


  /* Code. */

  current_domain[ 0 ]  = '\0';
  current_domain_index = -1;

  all_domains.no_of_domains = 0;

  for( index = 0; index < MSG_MAX_DOMAINS; index++ )
    all_domains.domains[ index ] = NULL;


  return;

} /* msgInitialize */


/*----------------------------------------------------------------------*/

Boolean
  msgInitCatalogue( char  *text_domain,
                    char  *directory,
                    char  *language,
                    char  **default_msgs )
{

  /* Variables. */
  Boolean  ok;
  char     msg_cat[ 250 ];
  char     *path;


  /* Code. */

  /* Try the catalogue in directory? */
  if( directory != NULL ) {
    sprintf( msg_cat, "%s/%sMsg%s.cat", 
             directory, text_domain, language );

    ok = msgAddFileTextDomain( text_domain, msg_cat );
    if( ok )
      return( True );

    if( default_msgs != NULL ) {
      msgAddTextDomain( text_domain, default_msgs );

      return( True );
    }
  }


  /* Search the meesage catalogue in the XDHOME directory. */
  path = getenv( "XDHOME" );

  if( path == NULL ) {
    if( default_msgs == NULL )
      return( False );

    msgAddTextDomain( text_domain, default_msgs );

    return( True );
  }

  /* Search in the XDHOME directory. */
  sprintf( msg_cat, "%s/%sMsg%s.cat", path, text_domain, language );

  ok = msgAddFileTextDomain( text_domain, msg_cat );

  /* Default catalogue? */
  if( ! ok ) {
    if( default_msgs != NULL ) {
      msgAddTextDomain( text_domain, default_msgs );

      return( True );
    }
  }


  return( True );

} /* msgInitCatalogue */


/*----------------------------------------------------------------------*/

char
  *msgTextDomain( char  *domain_name )
{

  /* Variables. */
  int  index;


  /* Code. */

  for( index = 0; index < all_domains.no_of_domains; index++ ) {
    if( strcmp( all_domains.domains[ index ] -> domain_name, 
                domain_name ) == 0 )
      break;
  }

  if( index == all_domains.no_of_domains )
    return( NULL );

  current_domain_index = index;

  strcpy( old_current_domain, current_domain );
  strcpy( current_domain, domain_name );


  return( old_current_domain );      

} /* msgTextDomain */

