/*----------------------------------------------------------------------------
--
--  Module:           xitInfoToc
--
--  Project:          xit - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Table of contents for the information database.
--
--  Filename:         xitInfoToc.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfoToc.c, Version: 1.1, Date: 95/02/18 15:10:39";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"

#include "msgTopic.h"
#include "xitInfoFile.h"
#include "xitTools.h"
#include "xitInfoToc.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the print window. */
#define gotoRc                dataLocalW[  0 ]
#define sep1                  dataLocalW[  1 ]
#define tocLa                 dataLocalW[  2 ]
#define tocLi                 dataLocalW[  3 ]

/* Max number of matches we can handle. */
#define MAX_MATCHES      500


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record while searching entries. */
typedef struct {

  /* The selected section. */
  int  selected_item;

  /* File where the book is. */
  char  *book_file;

  /* The toc window. */
  Widget  tocW;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XIT_IT_ACTION_CB  actionCB;

} TOC_REC, *TOC_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfoToc";

/* Name of text domain. */
static char  *td = "Topic";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget       widget, 
            TOC_REC_REF  toc_ref,
            XtPointer    call_data );

static Widget
  createTocWindow( TOC_REC_REF  toc_ref,
                   Widget       parent );

static void 
  destroyCB( Widget       widget,
             TOC_REC_REF  toc_ref,
             XtPointer    call_data );

static void 
  displaySelectedSection( TOC_REC_REF  toc_ref );

static void 
  displayToc( TOC_REC_REF  toc_ref );

static void 
  gotoCB( Widget                     widget, 
          TOC_REC_REF                toc_ref,
          XmRowColumnCallbackStruct  *call_data );

static void
  setGotoButtons( TOC_REC_REF  toc_ref,
                  Boolean      sensitive );

static void 
  tocLiCB( Widget                widget, 
           TOC_REC_REF           toc_ref,
           XmListCallbackStruct  *call_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_IT_HANDLE
  xitItInitialize( Widget            parent,
                   XIT_IT_ACTION_CB  actionCB,
                   void              *user_data )
{

  /* Variables. */
  TOC_REC_REF  toc_ref;


  /* Code. */

  /* Create and initialize our private data. */
  toc_ref = SysNew( TOC_REC );
  if( toc_ref == NULL )
    return( NULL );

  toc_ref -> book_file  = NULL;
  toc_ref -> actionCB   = actionCB;
  toc_ref -> user_data  = user_data;


  /* Create the toc window. */
  toc_ref -> tocW = createTocWindow( toc_ref, parent );

  if( toc_ref -> tocW == NULL ) {
    SysFree( toc_ref );

    return( NULL );
  }


  return( (XIT_IT_HANDLE) toc_ref );

} /* xitItInitialize */


/*----------------------------------------------------------------------*/

void
  xitItDestroy( XIT_IT_HANDLE  toc_handle )
{

  /* Variables. */
  TOC_REC_REF  toc_ref;


  /* Code. */

  if( toc_handle == NULL )
    return;

  /* Our private data. */
  toc_ref = (TOC_REC_REF) toc_handle;


  /* Destroy the window. */
  XtDestroyWidget( toc_ref -> tocW );


  return;

} /* xitItDestroy */


/*----------------------------------------------------------------------*/

void 
  xitItDisplayToc( XIT_IT_HANDLE  toc_handle,
                   char           *book_file )
{

  /* Variables. */
  Arg          args[ 10 ];
  Cardinal     n;
  Widget       mainW;
  TOC_REC_REF  toc_ref;


  /* Code. */

  toc_ref = (TOC_REC_REF) toc_handle;
  if( toc_ref == NULL )
    return;

  mainW = XtNameToWidget( toc_ref -> tocW, "TocFdFo" );

  if( toc_ref -> book_file != NULL )
    SysFree( toc_ref -> book_file );
  toc_ref -> book_file = SysNewString( book_file );


  /* The Goto buttons are not valid. */
  setGotoButtons( toc_ref, False );


  /* Make sure the window is visible. */
  XtManageChild( toc_ref -> tocW );


  /* Don't let our children spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( toc_ref -> tocW, args, n );


  /* Display the TOC. */
  displayToc( toc_ref );


  return;

} /* xitItDisplayToc */


/*----------------------------------------------------------------------*/

static Widget
  createTocWindow( TOC_REC_REF  toc_ref,
                   Widget       parent )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 4 ];
  Widget    gotoBu[ 2 ];
  Widget    tocFd;
  Widget    workFo;

  static XIT_PUSH_STRUCT goto_def[] = {
    { "GotoNextBu", "", "", True, NULL },
    { "GotoPrevBu", "", "", True, NULL },
  };  

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   cancelCB, NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgDGetText( td, MINF_CLOSE_BUTTON );
  action_buttons[ 0 ].data  = toc_ref;

  goto_def[ 0 ].title = msgDGetText( td, MINF_GOTO_NEXT_BUTTON );
  goto_def[ 1 ].title = msgDGetText( td, MINF_GOTO_PREV_BUTTON );


  /* Create a form dialog with buttons. */
  tocFd = xitCreateFormDialog( parent, "TocFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  XtAddCallback( tocFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) toc_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgDGetText( td, MINF_TOC_TITLE ) ); n++;
  XtSetValues( XtParent( tocFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( tocFd, args, n );

  /* Container for the contents of the window. */
  workFo = XtNameToWidget( tocFd, "TocFdFo" );


  /* Create result list. */
  tocLa = xitCreateLabel( workFo, "TocLa",
                          msgDGetText( td, MINF_TOC_LIST_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  tocLi = XmCreateScrolledList( workFo, "TocLi", args, n );

  XtAddCallback( tocLi, XmNsingleSelectionCallback, 
                 (XtCallbackProc) tocLiCB, (XtPointer) toc_ref );
  XtAddCallback( tocLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) tocLiCB, (XtPointer) toc_ref );


  /* Create the action form. */
  n = 0;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  gotoRc = XmCreateRowColumn( workFo, "GotoRc", args, n );

  XtAddCallback( gotoRc, XmNentryCallback, 
                 (XtCallbackProc) gotoCB, (XtPointer) toc_ref );

  for( index = 0; index < XtNumber( gotoBu ); index++ ) {
    gotoBu[ index ] = xitCreatePushButton( gotoRc, &goto_def[ index ] );

    XtAddCallback( gotoBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) gotoCB, (XtPointer) index );
  }


  /* Create separators. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  sep1 = XmCreateSeparatorGadget( workFo, "Sep1", args, n );


  /* Place the various elements together. */
  xitAttachWidget( tocLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( tocLi ),
                   XmATTACH_WIDGET, tocLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( sep1,
                   XmATTACH_WIDGET, XtParent( tocLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,              XmATTACH_NONE, NULL );
  xitAttachWidget( gotoRc,
                   XmATTACH_WIDGET, sep1, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( gotoRc, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( tocLa,             args, n );
  XtSetValues( XtParent( tocLi ), args, n );
  XtSetValues( gotoRc,            args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( sep1, args, n );


  /* Manage all the children except the search form. */
  xitManageChildren( gotoBu, XtNumber( gotoBu  ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW  ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( tocFd, True );


  /* Make the final attachments. */
  xitAttachWidget( gotoRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( sep1,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, gotoRc );
  xitAttachWidget( XtParent( tocLi ),
                   XmATTACH_WIDGET, tocLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_WIDGET, sep1 );


  return( tocFd );

} /* createTocWindow */


/*----------------------------------------------------------------------*/

static void 
  displaySelectedSection( TOC_REC_REF  toc_ref )
{

  /* Variables. */
  int       item_count;
  char      chapter_id[ 50 ];
  char      title[ 100 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tocLiW;
  XmString  *items;


  /* Code. */

  mainW  = XtNameToWidget( toc_ref -> tocW, "TocFdFo" );
  tocLiW = XtNameToWidget( mainW, "TocLiSW.TocLi" );


  /* Fetch the selected item. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &item_count ); n++;
  XtGetValues( tocLiW, args, n );


  /* There can only be ONE selection. */
  if( item_count != 1 )
    return;


  /* Get the selected string. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItems, &items ); n++;
  XtGetValues( tocLiW, args, n );

  char_ref = xitStringGetString( *items, CS );


  /* Extract header and chapter id. */
  sscanf( char_ref, "%[^#]#%[^#]", title, chapter_id );

  SysFree( char_ref );


  /* Tell our parent what we have found. */
  if( toc_ref -> actionCB != NULL )
    (* toc_ref -> actionCB)( XIT_IT_REASON_NEW_SECTION, 
                             toc_ref -> user_data, chapter_id, title );

  return;

} /* displaySelectedSection */


/*----------------------------------------------------------------------*/

static void 
  displayToc( TOC_REC_REF  toc_ref )
{

  /* Variables. */
  int                    index;
  int                    matches_no;
  char                   buffer[ 200 ];
  char                   format[ 20 ];
  char                   indent[ 200 ];
  char                   *char_ref;
  Arg                    args[ 10 ];
  Cardinal               n;
  Widget                 mainW;
  Widget                 tempW;
  XmString               list_items[ MAX_MATCHES ];
  XIT_IF_HANDLE          info_handle;
  XIT_IF_SECTION_REF     matches[ MAX_MATCHES ];
  XIT_IF_STATUS          status;


  /* Code. */

  mainW = XtNameToWidget( toc_ref -> tocW, "TocFdFo" );


  /* The Goto buttons are not valid. */
  setGotoButtons( toc_ref, False );


  /* Initalize the info file. */
  info_handle = xitIfInitialize( toc_ref -> book_file );
  if( info_handle == NULL )
    return;


  /* Do the toc. */
  {

    Cursor   wait_cursor;
    Display  *display = XtDisplay( toc_ref -> tocW );
    Window   window   = XtWindow(  toc_ref -> tocW );

    /* This might take time. */
    wait_cursor = XCreateFontCursor( display, XC_watch );
    XDefineCursor( display, window, wait_cursor );

    XFlush( display );

    /* Toc. */
    status = xitIfGetToc( info_handle, 
                          XtNumber( matches ), &matches[ 0 ], 
                          &matches_no );

    /* Clean up. */
    xitIfDestroy( info_handle );

    XUndefineCursor( display, window );

  } /* block */


  /* Assign the found sections to the result list. */
  for( index = 0; index < matches_no; index++ ) {

    /* Create the indent string. */
    sprintf( format, "%%-%d.%ds", 
             (matches[ index ] -> toc_level - 1) * 2, 
             (matches[ index ] -> toc_level - 1) * 2 );

    sprintf( indent, format, "" );

    /* Strip blanks. */
    char_ref = matches[ index ] -> title;
    while( isspace( *char_ref ) )
      char_ref++;

    sprintf( buffer, "%s%-80.80s#%s#%d#", 
             indent,
             char_ref, 
             matches[ index ] -> chapter_id, 
             matches[ index ] -> toc_level );

    list_items[ index ] = XmStringCreate( buffer, CS );

  } /* loop */


  tempW = XtNameToWidget( mainW, "TocLiSW.TocLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, matches_no ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free the match records. */
  for( index = 0; index < matches_no; index++ )
    XmStringFree( list_items[ index ] );

  xitIfFreeMatchInfo( &matches[ 0 ], matches_no );


  return;

} /* displayToc */


/*----------------------------------------------------------------------*/

static void
  setGotoButtons( TOC_REC_REF  toc_ref,
                  Boolean      sensitive )
{

  /* Variables. */
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  mainW = XtNameToWidget( toc_ref -> tocW, "TocFdFo" );
  tempW = XtNameToWidget( mainW, "GotoRc" );

  XtSetSensitive( tempW, sensitive );


  return;
    
} /* setGotoButtons */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget       widget, 
            TOC_REC_REF  toc_ref,
            XtPointer    call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( toc_ref -> actionCB != NULL )
    (* toc_ref -> actionCB)( XIT_IT_REASON_POPDOWN, 
                             toc_ref -> user_data, NULL, NULL );


  XtUnmanageChild( toc_ref -> tocW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             TOC_REC_REF  toc_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( toc_ref -> actionCB != NULL )
    (* toc_ref -> actionCB)( XIT_IT_REASON_DESTROY, 
                             toc_ref -> user_data, NULL, NULL );

  /* Free allocated data. */
  if( toc_ref -> book_file != NULL )
    SysFree( toc_ref -> book_file );

  SysFree( toc_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  gotoCB( Widget                     widget, 
          TOC_REC_REF                toc_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  int       item_count;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tocLiW;


  /* Code. */

  mainW = XtNameToWidget( toc_ref -> tocW, "TocFdFo" );

  /* Elements in the list. */
  tocLiW = XtNameToWidget( mainW, "TocLiSW.TocLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtGetValues( tocLiW, args, n );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Next? */
    case 0:
      if( toc_ref -> selected_item >= item_count )
        return;

      /* Make the selected position the top position. */
      XmListSetPos( tocLiW, toc_ref -> selected_item + 1 );

      /* Select the next position in the list. */
      XmListSelectPos( tocLiW, toc_ref -> selected_item + 1, True );
      break;


    /* Previous? */
    case 1:
      if( toc_ref -> selected_item <= 1 )
        return;

      /* Make the selected position the top position. */
      XmListSetPos( tocLiW, toc_ref -> selected_item - 1 );

      /* Select the next position in the list. */
      XmListSelectPos( tocLiW, toc_ref -> selected_item - 1, True );
      break;

  } /* switch */


  return;

} /* gotoCB */


/*----------------------------------------------------------------------*/

static void 
  tocLiCB( Widget                widget, 
           TOC_REC_REF           toc_ref,
           XmListCallbackStruct  *call_data )
{

  /* Code. */

  /* Save the select section. */
  toc_ref -> selected_item = call_data -> item_position;

  /* We can use the goto buttons. */
  setGotoButtons( toc_ref, True );


  /* Display the info text for this section. */
  displaySelectedSection( toc_ref );


  return;

} /* tocLiCB */
