/*----------------------------------------------------------------------------
--
--  Module:           xitParSet
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    The parameter set window allows you to control the settings of
--    one or more parameters. The typical use is in a customize
--    window.
--
--  Filename:         xitParSet.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-05-31
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitParSet.c, Version: 1.1, Date: 95/02/18 15:10:43";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/ToggleBG.h>
#include <Xm/ToggleB.h>
#include <Xm/Text.h>

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "System.h"
#include "TimDate.h"

#include "xitError.h"
#include "xitTools.h"
#include "xitColorSel.h"
#include "xitFieldSel.h"
#include "xitParSet.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the entry edit window. */
#define colorFs             dataLocalW[  0 ]
#define descLa              dataLocalW[  1 ]
#define descTx              dataLocalW[  2 ]
#define indexFs             dataLocalW[  3 ]
#define intFo               dataLocalW[  4 ]
#define intSc               dataLocalW[  5 ]
#define intTx               dataLocalW[  6 ]
#define noTb                dataLocalW[  7 ]
#define offTb               dataLocalW[  8 ]
#define onOffRb             dataLocalW[  9 ]
#define onTb                dataLocalW[ 10 ]
#define parListLa           dataLocalW[ 11 ]
#define parListLi           dataLocalW[ 12 ]
#define selectFs            dataLocalW[ 13 ]
#define textTx              dataLocalW[ 14 ]
#define valueLa             dataLocalW[ 15 ]
#define yesNoRb             dataLocalW[ 16 ]
#define yesTb               dataLocalW[ 17 ]
#define centuryTb           dataLocalW[ 18 ]
#define dateOrderRb         dataLocalW[ 19 ]
#define dateSepRc           dataLocalW[ 20 ]
#define dayZeroTb           dataLocalW[ 21 ]
#define dmyTb               dataLocalW[ 22 ]
#define mdyTb               dataLocalW[ 23 ]
#define monthZeroTb         dataLocalW[ 24 ]
#define ymdTb               dataLocalW[ 25 ]
#define dateSepLa           dataLocalW[ 26 ]
#define dateFormatFo        dataLocalW[ 27 ]
#define dateSepTx           dataLocalW[ 28 ]
#define dateParRc           dataLocalW[ 29 ]
#define hour12Tb            dataLocalW[ 30 ]
#define hour12Tx            dataLocalW[ 31 ]
#define hour24Tb            dataLocalW[ 32 ]
#define hour24Tx            dataLocalW[ 33 ]
#define hourZeroTb          dataLocalW[ 34 ]
#define timeFormatFo        dataLocalW[ 35 ]
#define timeParRc           dataLocalW[ 36 ]
#define timeSepLa           dataLocalW[ 37 ]
#define timeSepRc           dataLocalW[ 38 ]
#define timeSepTx           dataLocalW[ 39 ]
#define timeTypeRb          dataLocalW[ 40 ]
#define week1stDayFs        dataLocalW[ 41 ]
#define week1stDayLa        dataLocalW[ 42 ]
#define week1stDayRc        dataLocalW[ 43 ]
#define hiddenLa            dataLocalW[ 44 ]
#define hiddenPb            dataLocalW[ 45 ]
#define hiddenRc            dataLocalW[ 46 ]
#define chidWhatLa          dataLocalW[ 47 ]
#define chidTx              dataLocalW[ 48 ]
#define workdaysRc          dataLocalW[ 49 ]


/* The number of parameter we can do. */
#define  MAX_PARAMETERS     100

/* Pre-defined entry tags in a menu, how many? */
#define MAX_MENU_TAGS  50


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Are we doing a reset or default action? */
typedef enum {
  DO_RESET,
  DO_DEFAULT
} ACTION;


/* Temporary placeholder for values while working. */
typedef struct {
  void  *value;
} WORK_VALUE_REC;


/* Internal data to use. */
typedef struct {

  /* Do update of field. */
  Boolean  do_update;

  /* General flags to use. */
  UINT32  flags;

  /* Value for hidden field. */
  int     change_hidden_max_columns;
  int     change_hidden_max_length;
  char    hidden_buf[ 128 ];
  Widget  changeHiddenW;

  /* The main widget. */
  Widget  parSetW;

  /* The last widget value and parameter type used. */
  int     last_value_no;
  Widget  last_valueW;

  /* The complete parameter set. */
  int          rec_no;
  XIT_PAR_REC  *par_rec_ref;

  /* Values while working. */
  WORK_VALUE_REC  work[ MAX_PARAMETERS ];

  /* Action to do. */
  ACTION  do_action;

  /* Messages. */
  char  *msg_confirm_default;
  char  *msg_confirm_reset;
  char  *msg_invalid_value;

  /* User callback and user data. */
  XIT_PS_ACTION_CB  actionCB;
  XtPointer         user_data;

} PAR_SET_REC, *PAR_SET_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget           widget,
            PAR_SET_REC_REF  par_set_ref,
            XtPointer        call_data );

static void 
  changeHiddenCB( Widget           widget,
                  PAR_SET_REC_REF  par_set_ref,
                  XtPointer        call_data );

static void
  changeHiddenCancelCB( Widget          widget,
                       PAR_SET_REC_REF  par_set_ref,
                       XtPointer        call_data );

static void
  changeHiddenOkCB( Widget           widget,
                    PAR_SET_REC_REF  par_set_ref,
                    XtPointer        call_data );

static void
  createFieldMenu( char    *menu_string,
                   String  *tags,
                   int     max_tags,
                   int     *num_tags );

static void 
  defaultsCB( Widget           widget,
              PAR_SET_REC_REF  par_set_ref,
              XtPointer        call_data );

static void 
  destroyCB( Widget           widget,
             PAR_SET_REC_REF  par_set_ref,
             XtPointer        call_data );

static void 
  doDefaultResetCB( Widget           widget,
                    PAR_SET_REC_REF  par_set_ref,
                    XtPointer        call_data );

static void 
  intScChangedCB( Widget                 widget,
                  PAR_SET_REC_REF        par_set_ref,
                  XmScaleCallbackStruct  *call_data );

static void 
  intTxChangedCB( Widget           widget,
                  PAR_SET_REC_REF  par_set_ref,
                  XtPointer        call_data );

static void 
  okCB( Widget           widget,
        PAR_SET_REC_REF  par_set_ref,
        XtPointer        call_data );

static void 
  parListCB( Widget                widget, 
             PAR_SET_REC_REF       par_set_ref,
             XmListCallbackStruct  *call_data );
static void 
  resetCB( Widget           widget,
           PAR_SET_REC_REF  par_set_ref,
           XtPointer        call_data );

static Boolean
  saveParameter( PAR_SET_REC_REF  par_set_ref );

static void
  setParameter( PAR_SET_REC_REF  par_set_ref,
                XIT_PAR_REC      *par_rec_ref,
                WORK_VALUE_REC   *work_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateParameterSet( Widget            parent,
                         UINT32            flags,
                         XIT_PS_LABELS     *labels,
                         XIT_PS_ACTION_CB  actionCB,
                         XtPointer         user_data )
{

  /* Variables. */
  int              index;
  int              tag_count;
  Arg              args[ 10 ];
  Cardinal         n;
  String           tag_array[ MAX_MENU_TAGS ];
  Widget           chidFd;
  Widget           dataLocalW[ 50 ];
  Widget           parSetFd;
  Widget           workFo;
  Widget           workdaysBu[ 10 ];
  Widget           tempW;
  PAR_SET_REC_REF  par_set_ref;

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "DescTx",   NULL, 2, False },
    { "TextTx",   NULL, 1, True },
    { "IntTx",    NULL, 1, True },
    { "SepTx",    NULL, 1, True },
    { "Hour12Tx", NULL, 1, True },
    { "Hour24Tx", NULL, 1, True },
  };

  static XIT_PUSH_STRUCT button_def[] = {
    { "HiddenPb", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,       NULL },
    { "", resetCB,    NULL },
    { "", defaultsCB, NULL },
    { "", cancelCB,   NULL },
  };


  /* Code. */

  /* Create and initialize our private data. */
  par_set_ref = SysNew( PAR_SET_REC );
  if( par_set_ref == NULL )
    return( NULL );

  par_set_ref -> do_update            = True;
  par_set_ref -> par_rec_ref          = NULL;
  par_set_ref -> flags                = flags;
  par_set_ref -> last_valueW          = NULL;
  par_set_ref -> msg_confirm_default  = labels -> msg_confirm_default;
  par_set_ref -> msg_confirm_reset    = labels -> msg_confirm_reset;
  par_set_ref -> msg_invalid_value    = labels -> msg_invalid_value;
  par_set_ref -> actionCB             = actionCB;
  par_set_ref -> user_data            = user_data;

  for( index = 0; index < MAX_PARAMETERS; index++ )
    par_set_ref -> work[ index ].value = NULL;


  /* The window buttons. */
  button_def[ 0 ].title = labels -> change_button;

  action_buttons[ 0 ].label = labels -> ok_button;
  action_buttons[ 0 ].data  = par_set_ref;
  action_buttons[ 1 ].label = labels -> reset_button;
  action_buttons[ 1 ].data  = par_set_ref;
  action_buttons[ 2 ].label = labels -> defaults_button;
  action_buttons[ 2 ].data  = par_set_ref;
  action_buttons[ 3 ].label = labels -> cancel_button;
  action_buttons[ 3 ].data  = par_set_ref;


  /* Create a form dialog with buttons. */
  parSetFd = xitCreateFormDialog( parent, "ParSetFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

  par_set_ref -> parSetW = parSetFd;

  XtAddCallback( parSetFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) par_set_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, labels -> window_title ); n++;
  XtSetValues( XtParent( parSetFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNuserData, (void *) par_set_ref ); n++;
  XtSetArg( args[ n ], XmNdeleteResponse, XmDESTROY ); n++;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( parSetFd, args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( parSetFd, "ParSetFdFo" );


  /* The labels we need. */
  parListLa = xitCreateLabel( workFo, "ParListLa", 
                              labels -> parameter_list, -1 );

  descLa = xitCreateLabel( workFo, "DescLa", 
                           labels -> description, -1 );

  valueLa = xitCreateLabel( workFo, "ValueLa", 
                            labels -> parameter_value, -1 );


  /* The parameter list. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  parListLi = XmCreateScrolledList( workFo, "ParListLi", args, n );

  XtAddCallback( parListLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) parListCB, (XtPointer) par_set_ref );
  XtAddCallback( parListLi, XmNdefaultActionCallback,
                 (XtCallbackProc) parListCB, (XtPointer) par_set_ref );


  /* The description text. */
  descTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( descTx, args, n );


  /* Yes/No radio box. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  yesNoRb = XmCreateRowColumn( workFo, "YesNoRb", args, n );

  yesTb = xitCreateToggleButton( yesNoRb, "YesTb",
                                 labels -> yes_label, False );

  noTb  = xitCreateToggleButton( yesNoRb, "NoTb",
                                 labels -> no_label, False );


  /* On/Off radio box. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  onOffRb = XmCreateRowColumn( workFo, "OnOffRb", args, n );

  onTb  = xitCreateToggleButton( onOffRb, "OnTb",
                                 labels -> on_label, False );

  offTb = xitCreateToggleButton( onOffRb, "OffTb",
                                 labels -> off_label, False );


  /* A slider bar and text field for integer values. */
  intFo = XmCreateForm( workFo, "IntFo", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNvalue, 1 ); n++;
  XtSetArg( args[ n ], XmNminimum, 1 ); n++;
  XtSetArg( args[ n ], XmNmaximum, 100 ); n++;
  XtSetArg( args[ n ], XmNprocessingDirection, XmMAX_ON_RIGHT ); n++;
  XtSetArg( args[ n ], XmNshowValue, True ); n++;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  intSc = XmCreateScale( intFo, "IntSc", args, n );

  XtAddCallback( intSc, XmNvalueChangedCallback, 
                 (XtCallbackProc) intScChangedCB, (XtPointer) par_set_ref );
  XtAddCallback( intSc, XmNdragCallback, 
                 (XtCallbackProc) intScChangedCB, (XtPointer) par_set_ref );

  intTx = xitCreateText( intFo, &text_buffer_def[ 2 ] );

  XtAddCallback( intTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) intTxChangedCB, (XtPointer) par_set_ref );


  /* Text field. */
  textTx = xitCreateText( workFo, &text_buffer_def[ 1 ] );


  /* Color field. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetArg( args[ n ], XmNcolumns, 5 ); n++;
  colorFs = xitCreateColorSelect( workFo, "ColorFs", args, n, NULL, NULL );


  /* Index select field. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetArg( args[ n ], XmNcolumns, 10 ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  indexFs = xitCreateFieldSelect( workFo, "IndexFs", args, n, NULL, NULL );


  /* Select field. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetArg( args[ n ], XmNcolumns, 10 ); n++;
  selectFs = xitCreateFieldSelect( workFo, "SelectFs", args, n, NULL, NULL );


  tempW = xitFieldSelectGetChild( indexFs, xitFIELD_SELECT_TEXT_FIELD );
  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
  XtSetValues( tempW, args, n );


  /* Hidden field. */
  {
    Pixel   bg;
    Widget  tmpWorkFo;

    static XIT_TEXT_STRUCT text_buffer[] = {
      { "ChidTx", NULL, 1, True },
    };

    static XIT_ACTION_AREA_ITEM  action_buttons[] = {
      { "",   changeHiddenOkCB,     NULL },
      { NULL, NULL,                 NULL },
      { "",   changeHiddenCancelCB, NULL },
    };

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
    hiddenRc = XmCreateRowColumn( workFo, "HiddenRc", args, n );

    hiddenLa = xitCreateLabel( hiddenRc, "HiddenLa", "               ", -1 );
    hiddenPb = xitCreatePushButton( hiddenRc, &button_def[ 0 ] );

    XtAddCallback( hiddenPb, XmNactivateCallback, 
                   (XtCallbackProc) changeHiddenCB, (XtPointer) par_set_ref );

    /* Change hidden popup. */
    action_buttons[ 0 ].label = labels -> ok_button;
    action_buttons[ 0 ].data  = par_set_ref;
    action_buttons[ 2 ].label = labels -> cancel_button;
    action_buttons[ 2 ].data  = par_set_ref;

    /* Create a form dialog with buttons. */
    chidFd = xitCreateFormDialog( par_set_ref -> parSetW, "ChidFd",
                                  1, 0,
                                  action_buttons,
                                  XtNumber( action_buttons ) );

    n = 0;
    XtSetArg( args[ n ], XmNtitle, " " ); n++;
    XtSetValues( XtParent( chidFd ), args, n );

    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( chidFd, args, n );

    tmpWorkFo = XtNameToWidget( chidFd, "ChidFdFo" );

    chidWhatLa = xitCreateLabel( tmpWorkFo, "PwdWhatLa", 
                                 labels -> change_hidden_label,
                                 -1 );

    chidTx = xitCreateTextCols( tmpWorkFo, &text_buffer[ 0 ], 20 );

    n = 0;
    XtSetArg( args[ n ], XmNbackground, &bg ); n++;
    XtGetValues( chidTx, args, n );

    n = 0;
    XtSetArg( args[ n ], XmNforeground, bg ); n++;
    XtSetValues( chidTx, args, n );

    xitAttachWidget( chidWhatLa,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                     XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
    xitAttachWidget( chidTx,
                     XmATTACH_WIDGET, chidWhatLa, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );

    n = 0;
    XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
    XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
    XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
    XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
    XtSetValues( chidWhatLa, args, n );
    XtSetValues( chidTx,     args, n );

    par_set_ref -> changeHiddenW = chidFd;
  }


  /* Workdays. */
  {
    int   char_read;
    char  buffer[ 256 ];
    char  label[ 256 ];
    char  *char_ref;

    n = 0;
    XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
    XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
    XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
    workdaysRc = XmCreateRowColumn( workFo, "WorkdaysRc", args, n );

    char_ref = labels -> weekdays_short;

    for( index = 0; index < XtNumber( workdaysBu ); index++ ) {
      if( index < 7 ) {
        char_read = strlen( char_ref );
        sscanf( char_ref, "%s%n", label, &char_read );
        char_ref = char_ref + char_read;
      } else {
        strcpy( label, " " );
      }

      sprintf( buffer, "Workdays%dBu", index + 1 );

      workdaysBu[ index ] = xitCreateToggleButton( workdaysRc, buffer,
                                                   label, False );

      if( index >= 7 ) {
        n = 0;
        XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
        XtSetValues( workdaysBu[ index ], args, n );
      }
    } /* loop */

  } /* block */


  /* Date format. */
  dateFormatFo = XmCreateForm( workFo, "DateFormatFo", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dateOrderRb = XmCreateRadioBox( dateFormatFo, "DateOrderRb", args, n );

  ymdTb  = xitCreateToggleButton( dateOrderRb, "YmdTb",
                                  labels -> ymd_label, False );
  dmyTb  = xitCreateToggleButton( dateOrderRb, "DmyTb",
                                  labels -> dmy_label, False );
  mdyTb  = xitCreateToggleButton( dateOrderRb, "MdyTb",
                                  labels -> mdy_label, False );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dateSepRc = XmCreateRowColumn( dateFormatFo, "DateSepRc", args, n );

  dateSepLa = xitCreateLabel( dateSepRc, "DateSepLa", 
                              labels -> date_sep_label, -1 );
  dateSepTx = xitCreateText( dateSepRc, &text_buffer_def[ 3 ] );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  dateParRc = XmCreateRowColumn( dateFormatFo, "DateParRc", args, n );

  dayZeroTb   = xitCreateToggleButton( dateParRc, "DayZeroTb",
                                       labels -> day_zero_label, False );
  monthZeroTb = xitCreateToggleButton( dateParRc, "MonthZeroTb",
                                       labels -> month_zero_label, False );
  centuryTb   = xitCreateToggleButton( dateParRc, "CenturyTb",
                                       labels -> century_label, False );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  week1stDayRc = XmCreateRowColumn( dateFormatFo, "Week1stDayRc", args, n );

  week1stDayLa = xitCreateLabel( week1stDayRc, "Week1stDayLa", 
                                 labels -> week_1st_day_label, -1 );

  n = 0;
  XtSetArg( args[ n ], XmNitems, NULL ); n++;
  XtSetArg( args[ n ], XmNitemCount, 0 ); n++;
  XtSetArg( args[ n ], XmNcolumns, 15 ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  week1stDayFs = xitCreateFieldSelect( week1stDayRc, "Week1stDayFs",
                                       args, n, NULL, NULL );
  xitFieldSelectEditable( week1stDayFs, False );

  tempW = xitFieldSelectGetChild( week1stDayFs, xitFIELD_SELECT_TEXT_FIELD );
  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
  XtSetValues( tempW, args, n );

  createFieldMenu( labels -> week_1st_days, tag_array, MAX_MENU_TAGS,
                   &tag_count );
  xitFieldSelectSetMenu( week1stDayFs, tag_array, tag_count );

  for( index = 0; index < tag_count; index++ )
    SysFree( tag_array[ index ] );


  xitAttachWidget( dateOrderRb,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( dateSepRc,
                   XmATTACH_WIDGET, dateOrderRb, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( dateParRc,
                   XmATTACH_WIDGET, dateSepRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( week1stDayRc,
                   XmATTACH_WIDGET, dateParRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( dateOrderRb,  args, n );
  XtSetValues( dateSepRc,    args, n );
  XtSetValues( dateParRc,    args, n );
  XtSetValues( week1stDayRc, args, n );


  /* Time format. */
  timeFormatFo = XmCreateForm( workFo, "TimeFormatFo", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  XtSetArg( args[ n ], XmNspacing, 15 ); n++;
  timeTypeRb = XmCreateRowColumn( timeFormatFo, "TimeTypeRb", args, n );

  hour12Tb  = xitCreateToggleButton( timeTypeRb, "Hour12Tb",
                                     labels -> hour12_label, False );
  hour24Tb  = xitCreateToggleButton( timeTypeRb, "Hour24Tb",
                                     labels -> hour24_label, False );

  hour12Tx = xitCreateText( timeFormatFo, &text_buffer_def[ 4 ] );
  hour24Tx = xitCreateText( timeFormatFo, &text_buffer_def[ 5 ] );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  timeSepRc = XmCreateRowColumn( timeFormatFo, "TimeSepRc", args, n );

  timeSepLa = xitCreateLabel( timeSepRc, "TimeSepLa", 
                              labels -> time_sep_label, -1 );

  timeSepTx = xitCreateText( timeSepRc, &text_buffer_def[ 3 ] );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  timeParRc = XmCreateRowColumn( timeFormatFo, "TimeParRc", args, n );

  hourZeroTb = xitCreateToggleButton( timeParRc, "HourZeroTb",
                                      labels -> hour_zero_label, False );

  xitAttachWidget( timeTypeRb,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( hour12Tx,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, timeTypeRb,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( hour24Tx,
                   XmATTACH_WIDGET, hour12Tx, XmATTACH_WIDGET, timeTypeRb,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( timeSepRc,
                   XmATTACH_WIDGET, hour24Tx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( timeParRc,
                   XmATTACH_WIDGET, timeSepRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( timeTypeRb, args, n );
  XtSetValues( timeSepRc,  args, n );
  XtSetValues( timeParRc,  args, n );
  XtSetValues( hour12Tx,   args, n );
  XtSetValues( hour24Tx,   args, n );


  /* Place the various elements together. */
  xitAttachWidget( parListLa,
                   XmATTACH_FORM, NULL, 
                   XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, 
                   XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( parListLi ),
                   XmATTACH_WIDGET, parListLa,
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( descLa,
                   XmATTACH_FORM,   NULL, 
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( XtParent( descTx ),
                   XmATTACH_WIDGET, descLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL,
                   XmATTACH_NONE, NULL );
  xitAttachWidget( valueLa,
                   XmATTACH_WIDGET, XtParent( descTx ),
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );

  xitAttachWidget( colorFs,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( indexFs,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( onOffRb,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( selectFs,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( textTx,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( intFo,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( dateFormatFo,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( hiddenRc,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( workdaysRc,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( timeFormatFo,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( intSc,
                   XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,
                   XmATTACH_WIDGET, intTx,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( intTx,
                   XmATTACH_NONE, NULL,
                   XmATTACH_NONE, NULL,
                   XmATTACH_FORM, NULL, 
                   XmATTACH_FORM, NULL );
  xitAttachWidget( yesNoRb,
                   XmATTACH_WIDGET, valueLa,
                   XmATTACH_WIDGET, XtParent( parListLi ),
                   XmATTACH_NONE,   NULL, 
                   XmATTACH_NONE,   NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( parListLa,             args, n );
  XtSetValues( descLa,                args, n );
  XtSetValues( XtParent( parListLi ), args, n );
  XtSetValues( XtParent( descTx ),    args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,     5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   15 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  15 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,  5 ); n++;
  XtSetValues( colorFs,      args, n );
  XtSetValues( indexFs,      args, n );
  XtSetValues( onOffRb,      args, n );
  XtSetValues( selectFs,     args, n );
  XtSetValues( textTx,       args, n );
  XtSetValues( valueLa,      args, n );
  XtSetValues( intSc,        args, n );
  XtSetValues( yesNoRb,      args, n );
  XtSetValues( dateFormatFo, args, n );
  XtSetValues( timeFormatFo, args, n );
  XtSetValues( hiddenRc,     args, n );
  XtSetValues( workdaysRc,   args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,   15 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  15 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,  0 ); n++;
  XtSetValues( intTx, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset, 15 ); n++;
  XtSetValues( descLa,             args, n );
  XtSetValues( XtParent( descTx ), args, n );


  /* Manage all the children. */
  xitManageChildren( workdaysBu, XtNumber( workdaysBu ) );
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( parSetFd, True );
  xitSetSizeFormDialog( chidFd, True );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( parSetFd ), args, n );


  /* Don't display any parameter widgets. */
  XtUnmanageChild( colorFs );
  XtUnmanageChild( indexFs );
  XtUnmanageChild( onOffRb );
  XtUnmanageChild( selectFs );
  XtUnmanageChild( textTx );
  XtUnmanageChild( intFo );
  XtUnmanageChild( yesNoRb );
  XtUnmanageChild( dateFormatFo );
  XtUnmanageChild( timeFormatFo );
  XtUnmanageChild( hiddenRc );
  XtUnmanageChild( workdaysRc );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( parListLi ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( descTx ), args, n );


  return( parSetFd );

} /* xitCreateParameterSet */


/*----------------------------------------------------------------------*/

void
  xitParameterSetDisplay( Widget       parSetW,
                          XIT_PAR_REC  par_rec_ref[],
                          int          rec_no )
{

  /* Variables. */
  int              index;
  size_t           size;
  Arg              args[ 5 ];
  Cardinal         n;
  XmString         list_items[ 100 ];
  Widget           mainW;
  Widget           tempW;
  PAR_SET_REC_REF  par_set_ref;


  /* Code. */

  /* Get our private data. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, (void *) &par_set_ref ); n++;
  XtGetValues( parSetW, args, n );

  mainW = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );


  /* Save the parameter data (needed later ). */
  if( par_set_ref -> par_rec_ref != NULL )
    SysFree( par_set_ref -> par_rec_ref );

  par_set_ref -> par_rec_ref = 
    (XIT_PAR_REC *) SysMalloc( sizeof( XIT_PAR_REC ) * rec_no );

  memcpy( par_set_ref -> par_rec_ref, par_rec_ref,
          rec_no * sizeof( XIT_PAR_REC ) );

  par_set_ref -> rec_no = rec_no;


  /* Free any data allocated for working parameters. */
  for( index = 0; index < MAX_PARAMETERS; index++ ) {
    if( par_set_ref -> work[ index ].value != NULL )
      SysFree( par_set_ref -> work[ index ].value );
  }


  /* Copy the parameters to be used when working. */
  for( index = 0; index < rec_no; index++ ) {
    size = par_rec_ref[ index ].size;

    par_set_ref -> work[ index ].value = (void *) SysMalloc( size );

    switch( par_rec_ref[ index ].parameter_type ) {
      case XIT_PS_TYPE_SEPARATOR:
        break;

      default:
        memcpy( par_set_ref -> work[ index ].value, 
                par_rec_ref[ index ].value,
                size );
    }
  }

  if( par_set_ref -> last_valueW != NULL ) {
    XtUnmanageChild( par_set_ref -> last_valueW );
    par_set_ref -> last_valueW   = NULL;
  }

  par_set_ref -> last_value_no = 0;


  /* Fill the parameter list. */
  for( index = 0; index < rec_no; index++ ) {
    if( par_rec_ref[ index ].parameter_type == XIT_PS_TYPE_SEPARATOR )
      list_items[ index ] = XmStringCreate( "---", CS );
    else
      list_items[ index ] = 
        XmStringCreate( par_rec_ref[ index ].parameter_label, CS );
  }


  /* Assign the parameters to the parameter list. */
  tempW = XtNameToWidget( mainW, "ParListLiSW.ParListLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, rec_no ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index = 0; index < rec_no; index++ )
    XmStringFree( list_items[ index ] );

  XmListSelectPos( tempW, 1, True );


  return;

} /* xitParameterSetDisplay */


/*----------------------------------------------------------------------*/

static void
  createFieldMenu( char    *menu_string,
                   String  *tags,
                   int     max_tags,
                   int     *num_tags )
{

  /* Variables. */
  int   char_read;
  int   items;
  int   tag_count;
  char  buffer[ 150 ];
  char  *char_ref;


  /* Code. */

  tag_count = 0;
  char_ref  = menu_string;

  do {
    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    items = sscanf( char_ref, "%s%n", buffer, &char_read );
    if( items != 1 )
      break;

    *(tags + tag_count ) = SysNewString( buffer );
    tag_count++;

    char_ref = char_ref + char_read;

  } while( tag_count < max_tags );

  *num_tags = tag_count;


  return;

} /* createFieldMenu */


/*----------------------------------------------------------------------*/

static Boolean
  saveParameter( PAR_SET_REC_REF  par_set_ref )
{

  /* Variables. */
  int              hour;
  int              index_color;
  int              int_value;
  int              items;
  int              minute;
  UINT32           flag_value;
  char             buffer1[ 50 ];
  char             buffer2[ 50 ];
  char             *char_string;
  Pixel            pixel;
  Widget           mainW;
  Widget           tempW;
  XColor           color;
  TIM_STATUS_TYPE  tstatus;
  TIM_TIME_REF     time_value;
  XIT_PAR_REC      *par_rec_ref;
  WORK_VALUE_REC   *work_ref;


  /* Code. */

  if( par_set_ref -> last_valueW == NULL )
    return( True );

  mainW       = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );
  par_rec_ref = &par_set_ref -> par_rec_ref[ par_set_ref -> last_value_no ];
  work_ref    = &par_set_ref -> work[ par_set_ref -> last_value_no ];


  /* Save the previous value. */
  switch( par_rec_ref -> parameter_type ) {

    case XIT_PS_TYPE_SEPARATOR:
      break;

    case XIT_PS_TYPE_INTEGER:
      tempW = XtNameToWidget( mainW, "IntFo.IntTx" );

      char_string = xitStringGetText( tempW );
      items = sscanf( char_string, "%d", &int_value );
      SysFree( char_string );

      if( items != 1 || 
          int_value < par_rec_ref -> min || 
          int_value > par_rec_ref -> max )
        return( False );

      *((int *) work_ref -> value) = int_value;
      break;


    case XIT_PS_TYPE_YES_NO:
    case XIT_PS_TYPE_ON_OFF:
      if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO )
        tempW = XtNameToWidget( mainW, "YesNoRb.YesTb" );
      else
        tempW = XtNameToWidget( mainW, "OnOffRb.OnTb" );

      if( XmToggleButtonGetState( tempW ) )
        *((Boolean *) work_ref -> value) = True;
      else
        *((Boolean *) work_ref -> value) = False;
      break;


    case XIT_PS_TYPE_YES_NO_FLAG:
    case XIT_PS_TYPE_ON_OFF_FLAG:
      if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO_FLAG )
        tempW = XtNameToWidget( mainW, "YesNoRb.YesTb" );
      else
        tempW = XtNameToWidget( mainW, "OnOffRb.OnTb" );

      flag_value = *((UINT32 *) work_ref -> value);

      if( XmToggleButtonGetState( tempW ) )
        flagSet( flag_value, par_rec_ref -> flag_offset );
      else
        flagClear( flag_value, par_rec_ref -> flag_offset );

      *((UINT32 *) work_ref -> value) = flag_value;
      break;


    case XIT_PS_TYPE_TEXT_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );

      char_string = xitStringGetText( tempW );
      strcpy( ((char *) work_ref -> value), char_string );
      SysFree( char_string );
      break;


    case XIT_PS_TYPE_INTEGER_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );

      char_string = xitStringGetText( tempW );

      items = sscanf( char_string, "%d", &int_value );
      SysFree( char_string );

      if( items != 1 || 
          int_value < par_rec_ref -> min || 
          int_value > par_rec_ref -> max )
        return( False );

      *((int *) work_ref -> value) = int_value;
      break;


    case XIT_PS_TYPE_DATE_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );

      char_string = xitStringGetText( tempW );

      items = sscanf( char_string, "%s", buffer1 );
      SysFree( char_string );

      if( items != 1 )
        return( False );

      tstatus = TimMakeDateFromString( &time_value, buffer2 );
      if( tstatus != TIM_OK )
        return( False );

      *((TIM_TIME_REF *) work_ref -> value) = time_value;
      break;


    case XIT_PS_TYPE_TIME_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );

      char_string = xitStringGetText( tempW );

      items = sscanf( char_string, "%s", buffer1 );
      SysFree( char_string );

      if( items != 1 )
        return( False );

      tstatus = TimMakeTimeFromString( &time_value, buffer2 );
      if( tstatus != TIM_OK )
        return( False );

      *((TIM_TIME_REF *) work_ref -> value) = time_value;
      break;


    case XIT_PS_TYPE_DURATION_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );

      char_string = xitStringGetText( tempW );

      items = sscanf( char_string, "%d:%d", &hour, &minute );
      SysFree( char_string );

      if( items == 1 && hour >= 0 )
        int_value = hour;
      else if( items == 2 && hour >= 0 && minute >= 0 && minute < 60 )
        int_value = hour * 60 + minute;
      else
        return( False );

      if( int_value < par_rec_ref -> min || 
          int_value > par_rec_ref -> max )
        return( False );

      *((int *) work_ref -> value) = int_value;
      break;


    case XIT_PS_TYPE_COLOR_FIELD:
      tempW = XtNameToWidget( mainW, "ColorFs" );

      (void) xitColorSelectGetColor( tempW, 
                                     &pixel, &color,
                                     buffer1 );

      *((Pixel *) work_ref -> value) = (Pixel) color.pixel;
      break;


    case XIT_PS_TYPE_COLOR_INDEX_FIELD:
      tempW = XtNameToWidget( mainW, "ColorFs" );

      (void) xitColorSelectGetIndex( tempW, 
                                     &index_color );

      *((int *) work_ref -> value) = index_color;
      break;


    case XIT_PS_TYPE_SELECT_FIELD:
      tempW = XtNameToWidget( mainW, "SelectFs" );

      xitFieldSelectGetCurrent( tempW, &char_string );
      strcpy( ((char *) work_ref -> value), char_string );

      SysFree( char_string );
      break;


    case XIT_PS_TYPE_SELECT_INDEX_FIELD:
      tempW = XtNameToWidget( mainW, "IndexFs" );

      xitFieldSelectGetIndex( tempW, &int_value );

      *((int *) work_ref -> value) = int_value;
      break;


    case XIT_PS_TYPE_HIDDEN_TEXT_FIELD:
      strcpy( ((char *) work_ref -> value), par_set_ref -> hidden_buf );
      break;


    case XIT_PS_TYPE_WORKDAYS_FIELD:
      {
        Boolean  valid = False;
        int      index;
        char     buffer[ 256 ];

        for( index = 0; index < 7; index++ ) {
          sprintf( buffer, "WorkdaysRc.Workdays%dBu", index + 1 );
          tempW = XtNameToWidget( mainW, buffer );

          if( XmToggleButtonGetState( tempW ) )
            valid = True;
        }

        if( ! valid )
          return( False);

        for( index = 0; index < 7; index++ ) {
          sprintf( buffer, "WorkdaysRc.Workdays%dBu", index + 1 );
          tempW = XtNameToWidget( mainW, buffer );

          if( XmToggleButtonGetState( tempW ) )
	    *(((Boolean *) work_ref -> value) + index) = True;
          else
	    *(((Boolean *) work_ref -> value) + index) = False;
        }
      }
      break;


    case XIT_PS_TYPE_DATE_FORMAT_FIELD:
      {
        char  buffer[ 50 ];
        char  date_format[ 50 ];

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.YmdTb" );
        if( XmToggleButtonGetState( tempW ) )
          strncpy( date_format, "YMD", 3 );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.MdyTb" );
        if( XmToggleButtonGetState( tempW ) )
          strncpy( date_format, "MDY", 3 );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.DmyTb" );
        if( XmToggleButtonGetState( tempW ) )
          strncpy( date_format, "DMY", 3 );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateSepRc.SepTx" );
        char_string = xitStringGetText( tempW );

        if( isdigit( *char_string ) ) {
          SysFree( char_string );
          return( False );
        }

        if( *char_string == '\0' )
          date_format[ 3 ] = '0';
        else
          date_format[ 3 ] = *char_string;
        SysFree( char_string );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.DayZeroTb" );
        if( XmToggleButtonGetState( tempW ) )
          date_format[ 4 ] = '1';
        else
          date_format[ 4 ] = '0';

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.MonthZeroTb" );
        if( XmToggleButtonGetState( tempW ) )
          date_format[ 5 ] = '1';
        else
          date_format[ 5 ] = '0';

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.CenturyTb" );
        if( XmToggleButtonGetState( tempW ) )
          date_format[ 6 ] = '1';
        else
          date_format[ 6 ] = '0';

        tempW = XtNameToWidget( mainW,
                                "DateFormatFo.Week1stDayRc.Week1stDayFs" );
        (void) xitFieldSelectGetIndex( tempW, &int_value );
        sprintf( buffer, "%d", int_value );
        date_format[ 7 ] = buffer[ 0 ];

        date_format[ 8 ] = '\0';

        strcpy( ((char *) work_ref -> value), date_format );
      }
      break;


    case XIT_PS_TYPE_TIME_FORMAT_FIELD:
      {
        Boolean  hour12_format = False;
        char     time_format[ 50 ];

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeTypeRb.Hour12Tb" );
        if( XmToggleButtonGetState( tempW ) ) {
          strncpy( time_format, "12", 2 );
          hour12_format = True;
        }

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeTypeRb.Hour24Tb" );
        if( XmToggleButtonGetState( tempW ) )
          strncpy( time_format, "24", 2 );

        strncpy( &time_format[ 2 ], "  ", 2 );

        if( hour12_format ) {
          tempW = XtNameToWidget( mainW, "TimeFormatFo.Hour12Tx" );
          char_string = xitStringGetText( tempW );

          if( *char_string != '\0' ) {
            time_format[ 2 ] = *char_string;
            if( *(char_string + 1) != '\0' )
              time_format[ 3 ] = *(char_string + 1);
          }

          SysFree( char_string );
        }

        if( hour12_format && strncmp( &time_format[ 2 ], "  ", 2 ) == 0 )
          strncpy( &time_format[ 2 ], "am", 2 );

        strncpy( &time_format[ 4 ], "  ", 2 );

        tempW = XtNameToWidget( mainW, "TimeFormatFo.Hour24Tx" );
        char_string = xitStringGetText( tempW );

        if( *char_string != '\0' ) {
          time_format[ 4 ] = *char_string;
          if( *(char_string + 1) != '\0' )
            time_format[ 5 ] = *(char_string + 1);
        }

        SysFree( char_string );

        if( hour12_format && strncmp( &time_format[ 4 ], "  ", 2 ) == 0 )
          strncpy( &time_format[ 4 ], "pm", 2 );

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeSepRc.SepTx" );
        char_string = xitStringGetText( tempW );

        if( isdigit( *char_string ) ) {
          SysFree( char_string );
          return( False );
        }

        if( *char_string == '\0' )
          time_format[ 6 ] = '0';
        else
          time_format[ 6 ] = *char_string;
        SysFree( char_string );

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeParRc.HourZeroTb" );
        if( XmToggleButtonGetState( tempW ) )
          time_format[ 7 ] = '1';
        else
          time_format[ 7 ] = '0';

        time_format[ 8 ] = '\0';

        strcpy( ((char *) work_ref -> value), time_format );
      }
      break;

  } /* switch */


  return( True );

} /* saveParameter */


/*----------------------------------------------------------------------*/

static void
  setParameter( PAR_SET_REC_REF  par_set_ref,
                XIT_PAR_REC      *par_rec_ref,
                WORK_VALUE_REC   *work_ref )
{

  /* Variables. */
  int           index;
  int           index_color;
  int           int_value;
  int           tag_count;
  UINT32        flag_value;
  char          *char_ref;
  char          buffer[ 201 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Pixel         pixel_value;
  String        tag_array[ MAX_MENU_TAGS ];
  Widget        mainW;
  Widget        tempW;
  TIM_TIME_REF  time_value;


  /* Code. */

  mainW = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );

  /* Set the value. */
  switch( par_rec_ref -> parameter_type ) {

    case XIT_PS_TYPE_SEPARATOR:
      break;

    case XIT_PS_TYPE_INTEGER:
      tempW = XtNameToWidget( mainW, "IntFo" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      int_value = *((int *) work_ref -> value);

      tempW = XtNameToWidget( mainW, "IntFo.IntSc" );
      n = 0;
      XtSetArg( args[ n ], XmNvalue,   int_value ); n++;
      XtSetArg( args[ n ], XmNminimum, par_rec_ref -> min ); n++;
      XtSetArg( args[ n ], XmNmaximum, par_rec_ref -> max ); n++;
      XtSetValues( tempW, args, n );

      tempW = XtNameToWidget( mainW, "IntFo.IntTx" );
      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      sprintf( buffer, "%d", int_value );
      if( strlen( buffer ) < 2 )
        strcat( buffer, " " );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_YES_NO:
    case XIT_PS_TYPE_ON_OFF:
      if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO )
        tempW = XtNameToWidget( mainW, "YesNoRb" );
      else
        tempW = XtNameToWidget( mainW, "OnOffRb" );

      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      if( *((Boolean *) work_ref -> value) ) {
        if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO )
          tempW = XtNameToWidget( mainW, "YesNoRb.YesTb" );
        else
          tempW = XtNameToWidget( mainW, "OnOffRb.OnTb" );
        XmToggleButtonSetState( tempW, True, True );

      } else {
        if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO )
          tempW = XtNameToWidget( mainW, "YesNoRb.NoTb" );
        else
          tempW = XtNameToWidget( mainW, "OnOffRb.OffTb" );
        XmToggleButtonSetState( tempW, True, True );
      }
      break;


    case XIT_PS_TYPE_YES_NO_FLAG:
    case XIT_PS_TYPE_ON_OFF_FLAG:
      if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO_FLAG )
        tempW = XtNameToWidget( mainW, "YesNoRb" );
      else
        tempW = XtNameToWidget( mainW, "OnOffRb" );

      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      flag_value = *((UINT32 *) work_ref -> value);

      if( flagIsSet( flag_value, par_rec_ref -> flag_offset ) ) {
        if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO_FLAG )
          tempW = XtNameToWidget( mainW, "YesNoRb.YesTb" );
        else
          tempW = XtNameToWidget( mainW, "OnOffRb.OnTb" );
        XmToggleButtonSetState( tempW, True, True );

      } else {
        if( par_rec_ref -> parameter_type == XIT_PS_TYPE_YES_NO_FLAG )
          tempW = XtNameToWidget( mainW, "YesNoRb.NoTb" );
        else
          tempW = XtNameToWidget( mainW, "OnOffRb.OffTb" );
        XmToggleButtonSetState( tempW, True, True );
      }
      break;


    case XIT_PS_TYPE_TEXT_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      strncpy( buffer, ((char *) work_ref -> value), 200 );
      buffer[ 200 ] = '\0';
      if( strlen( buffer ) < 2 )
        strcat( buffer, " " );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_INTEGER_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      sprintf( buffer, "%d", *((int *) work_ref -> value) );
      if( strlen( buffer ) < 2 )
        strcat( buffer, " " );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_DATE_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      time_value = *((TIM_TIME_REF *) work_ref -> value);
      TimFormatDate( time_value, buffer, sizeof( buffer ) );

      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_TIME_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      time_value = *((TIM_TIME_REF *) work_ref -> value);
      TimFormatTime( time_value, buffer, sizeof( buffer ) );

      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_DURATION_FIELD:
      tempW = XtNameToWidget( mainW, "TextTx" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      n = 0;
      XtSetArg( args[ n ], XmNmaxLength, par_rec_ref -> max_length ); n++;
      XtSetArg( args[ n ], XmNcolumns,   par_rec_ref -> max_columns ); n++;
      XtSetValues( tempW, args, n );

      int_value = *((int *) work_ref -> value);
      sprintf( buffer, "%d:%02d", int_value / 60, int_value % 60 );

      XmTextSetString( tempW, buffer );
      break;


    case XIT_PS_TYPE_COLOR_FIELD:
    case XIT_PS_TYPE_COLOR_INDEX_FIELD:
    case XIT_PS_TYPE_SELECT_FIELD:
    case XIT_PS_TYPE_SELECT_INDEX_FIELD:
      createFieldMenu( par_rec_ref -> select_values, tag_array, MAX_MENU_TAGS,
                       &tag_count );

      switch( par_rec_ref -> parameter_type ) {

        case XIT_PS_TYPE_COLOR_FIELD:
          tempW = XtNameToWidget( mainW, "ColorFs" );
          par_set_ref -> last_valueW = tempW;

          pixel_value = *((Pixel *) work_ref -> value);

          xitColorSelectSetMenu( tempW, tag_array, tag_count );
          xitColorSelectSetPixel( tempW, pixel_value, False );

          XtManageChild( tempW );
          break;

        case XIT_PS_TYPE_COLOR_INDEX_FIELD:
          tempW = XtNameToWidget( mainW, "ColorFs" );
          par_set_ref -> last_valueW = tempW;

          index_color = *((int *) work_ref -> value);

          xitColorSelectSetMenu( tempW, tag_array, tag_count );
          xitColorSelectSetIndex( tempW, index_color, False );

          XtManageChild( tempW );
          break;

        case XIT_PS_TYPE_SELECT_FIELD:
          tempW = XtNameToWidget( mainW, "SelectFs" );
          par_set_ref -> last_valueW = tempW;

          char_ref = ((char *) work_ref -> value);

          xitFieldSelectSetMenu( tempW, tag_array, tag_count );
          xitFieldSelectSetCurrent( tempW, char_ref, False );

          xitFieldSelectEditable( tempW, True );

          XtManageChild( tempW );
          break;

        case XIT_PS_TYPE_SELECT_INDEX_FIELD:
          tempW = XtNameToWidget( mainW, "IndexFs" );
          par_set_ref -> last_valueW = tempW;

          int_value = *((int *) work_ref -> value);

          xitFieldSelectSetMenu( tempW, tag_array, tag_count );
          xitFieldSelectSetIndex( tempW, int_value, False );

          xitFieldSelectEditable( tempW, False );

          XtManageChild( tempW );
          break;
        default:  /* clause added RXTN 040597 without having a clue about correctness */
	  break;
      }

      for( index = 0; index < tag_count; index++ )
        SysFree( tag_array[ index ] );
      break;


    case XIT_PS_TYPE_HIDDEN_TEXT_FIELD:
      tempW = XtNameToWidget( mainW, "HiddenRc" );
      par_set_ref -> last_valueW = tempW;

      XtManageChild( tempW );

      /* Length if we want to change. */
      par_set_ref -> change_hidden_max_length  = par_rec_ref -> max_length;
      par_set_ref -> change_hidden_max_columns = par_rec_ref -> max_columns;

      /* Save the hidden string. */
      strcpy( par_set_ref -> hidden_buf, ((char *) work_ref -> value) );

      /* Display the field as a number of --. */
      strcpy( buffer, par_set_ref -> hidden_buf );

      for( index = 0; index < strlen( buffer ); index++ )
        buffer[ index ] = '-';
    
      tempW = XtNameToWidget( mainW, "HiddenRc.HiddenLa" );
      xitStringSetLabel( tempW, buffer );
      break;


    case XIT_PS_TYPE_WORKDAYS_FIELD:
      {
        int   index;
        char  buffer[ 256 ];

        tempW = XtNameToWidget( mainW, "WorkdaysRc" );
        par_set_ref -> last_valueW = tempW;

        XtManageChild( tempW );

        for( index = 0; index < 7; index++ ) {
          sprintf( buffer, "WorkdaysRc.Workdays%dBu", index + 1 );
          tempW = XtNameToWidget( mainW, buffer );

          if( *(((Boolean *) work_ref -> value) + index) )
            XmToggleButtonSetState( tempW, True, False );
          else
            XmToggleButtonSetState( tempW, False, False );
        }
      }
      break;


    case XIT_PS_TYPE_DATE_FORMAT_FIELD:
      {
        char  date_format[ 50 ];

        tempW = XtNameToWidget( mainW, "DateFormatFo" );
        par_set_ref -> last_valueW = tempW;

        XtManageChild( tempW );

        strcpy( date_format, ((char *) work_ref -> value) );

        if( strncmp( date_format, "YMD", 3 ) == 0 )
          tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.YmdTb" );
        else if( strncmp( date_format, "MDY", 3 ) == 0 )
          tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.MdyTb" );
        else
          tempW = XtNameToWidget( mainW, "DateFormatFo.DateOrderRb.DmyTb" );

        XmToggleButtonSetState( tempW, True, True );

        if( date_format[ 3 ] == '0' )
          strcpy( buffer, "" );
        else {
          buffer[ 0 ] = date_format[ 3 ];
          buffer[ 1 ] = '\0';
        }

        if( strlen( buffer ) < 2 )
          strcat( buffer, " " );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateSepRc.SepTx" );
        XmTextSetString( tempW, buffer );

        n = 0;
        XtSetArg( args[ n ], XmNmaxLength, 1 ); n++;
        XtSetArg( args[ n ], XmNcolumns,   2 ); n++;
        XtSetValues( tempW, args, n );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.DayZeroTb" );
        if( date_format[ 4 ] == '1' )
          XmToggleButtonSetState( tempW, True, True );
        else
          XmToggleButtonSetState( tempW, False, True );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.MonthZeroTb" );
        if( date_format[ 5 ] == '1' )
          XmToggleButtonSetState( tempW, True, True );
        else
          XmToggleButtonSetState( tempW, False, True );

        tempW = XtNameToWidget( mainW, "DateFormatFo.DateParRc.CenturyTb" );
        if( date_format[ 6 ] == '1' )
          XmToggleButtonSetState( tempW, True, True );
        else
          XmToggleButtonSetState( tempW, False, True );

        buffer[ 0 ] = date_format[ 7 ];
        buffer[ 1 ] = '\0';

        int_value = atoi( buffer );
        if( int_value < 0 || int_value > 1 )
          int_value = 1;

        tempW = XtNameToWidget( mainW,
                                "DateFormatFo.Week1stDayRc.Week1stDayFs" );
        xitFieldSelectSetIndex( tempW, int_value, False );
      }
      break;


    case XIT_PS_TYPE_TIME_FORMAT_FIELD:
      {
        char  time_format[ 50 ];

        tempW = XtNameToWidget( mainW, "TimeFormatFo" );
        par_set_ref -> last_valueW = tempW;

        XtManageChild( tempW );

        strcpy( time_format, ((char *) work_ref -> value) );

        if( strncmp( time_format, "12", 2 ) == 0 )
          tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeTypeRb.Hour12Tb" );
        else 
          tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeTypeRb.Hour24Tb" );

        XmToggleButtonSetState( tempW, True, True );

        strncpy( buffer, &time_format[ 2 ], 2 );
        buffer[ 2 ] = '\0';

        if( strcmp( buffer, "  " ) == 0 ) {
          if( strncmp( time_format, "12", 2 ) == 0 )
            strcpy( buffer, "am" );
          else
            strcpy( buffer, "" );
        }

        tempW = XtNameToWidget( mainW, "TimeFormatFo.Hour12Tx" );
        XmTextSetString( tempW, buffer );

        n = 0;
        XtSetArg( args[ n ], XmNmaxLength, 2 ); n++;
        XtSetArg( args[ n ], XmNcolumns,   3 ); n++;
        XtSetValues( tempW, args, n );

        strncpy( buffer, &time_format[ 4 ], 2 );
        buffer[ 2 ] = '\0';

        if( strcmp( buffer, "  " ) == 0 ) {
          if( strncmp( time_format, "12", 2 ) == 0 )
            strcpy( buffer, "pm" );
          else
            strcpy( buffer, "" );
        }

        tempW = XtNameToWidget( mainW, "TimeFormatFo.Hour24Tx" );
        XmTextSetString( tempW, buffer );

        n = 0;
        XtSetArg( args[ n ], XmNmaxLength, 2 ); n++;
        XtSetArg( args[ n ], XmNcolumns,   3 ); n++;
        XtSetValues( tempW, args, n );

        if( time_format[ 6 ] == '0' )
          strcpy( buffer, "" );
        else {
          buffer[ 0 ] = time_format[ 6 ];
          buffer[ 1 ] = '\0';
        }

        if( strlen( buffer ) < 2 )
          strcat( buffer, " " );

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeSepRc.SepTx" );
        XmTextSetString( tempW, buffer );

        n = 0;
        XtSetArg( args[ n ], XmNmaxLength, 1 ); n++;
        XtSetArg( args[ n ], XmNcolumns,   2 ); n++;
        XtSetValues( tempW, args, n );

        tempW = XtNameToWidget( mainW, "TimeFormatFo.TimeParRc.HourZeroTb" );
        if( time_format[ 7 ] == '1' )
          XmToggleButtonSetState( tempW, True, True );
        else
          XmToggleButtonSetState( tempW, False, True );
      }
      break;

  } /* switch */


  return;

} /* setParameter */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget           widget,
            PAR_SET_REC_REF  par_set_ref,
            XtPointer        call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( par_set_ref -> actionCB != NULL )
    (* par_set_ref -> actionCB)( XIT_PS_REASON_CANCEL, 
                                 par_set_ref -> user_data );

  /* Pop-down the window. */
  XtUnmanageChild( par_set_ref -> parSetW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  changeHiddenCB( Widget           widget,
                  PAR_SET_REC_REF  par_set_ref,
                  XtPointer        call_data )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    tempW;


  /* Code. */

  /* Display the text (not wisible). */
  tempW = XtNameToWidget( par_set_ref -> changeHiddenW, "ChidFdFo.ChidTx" );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength,
                       par_set_ref -> change_hidden_max_length ); n++;
  XtSetArg( args[ n ], XmNcolumns,
                       par_set_ref -> change_hidden_max_columns ); n++;
  XtSetValues( tempW, args, n );

  XmTextSetString( tempW, "" );

  /* Make sure the window is visible. */
  XtManageChild( par_set_ref -> changeHiddenW );

  /* Set focus. */
  tempW = XtNameToWidget( par_set_ref -> changeHiddenW, "ChidFdFo.ChidTx" );
  xitSetFocus( par_set_ref -> changeHiddenW, tempW );


  return;

} /* changeHiddenCB */


/*----------------------------------------------------------------------*/

static void
  changeHiddenCancelCB( Widget          widget,
                       PAR_SET_REC_REF  par_set_ref,
                       XtPointer        call_data )
{

  /* Code. */

  /* Remove window. */
  XtUnmanageChild( par_set_ref -> changeHiddenW );


  return;

} /* changeHiddenCancelCB */


/*----------------------------------------------------------------------*/

static void
  changeHiddenOkCB( Widget           widget,
                    PAR_SET_REC_REF  par_set_ref,
                    XtPointer        call_data )
{

  /* Variables. */
  int     index;
  char    buffer[ 201 ];
  char    *char_ref;
  char    *end_ref;
  char    *start_ref;
  Widget  tempW;


  /* Code. */

  /* Fetch hidden text. */
  tempW = XtNameToWidget( par_set_ref -> changeHiddenW, "ChidFdFo.ChidTx" );
  char_ref = xitStringGetText( tempW );


  /* Trim spaces. */
  start_ref = char_ref;
  while( isspace( *start_ref ) )
    start_ref++;

  end_ref = start_ref + strlen( start_ref ) - 1;
  while( end_ref > start_ref && isspace( *end_ref ) ) {
    *end_ref = '\0';
    end_ref--;
  }

  strcpy( par_set_ref -> hidden_buf, start_ref );
  SysFree( char_ref );


  /* Remove window. */
  XtUnmanageChild( par_set_ref -> changeHiddenW );


  /* Set field. */
  tempW = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo.HiddenRc" );
  tempW = XtNameToWidget( tempW, "HiddenLa" );

  strcpy( buffer, par_set_ref -> hidden_buf );
  for( index = 0; index < strlen( buffer ); index++ )
    buffer[ index ] = '-';
    
  xitStringSetLabel( tempW, buffer );


  return;

} /* changeHiddenOkCB */


/*----------------------------------------------------------------------*/

static void 
  doDefaultResetCB( Widget           widget,
                    PAR_SET_REC_REF  par_set_ref,
                    XtPointer        call_data )
{

  /* Variables. */
  int     index;
  size_t  size;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );


  /* New work values. */
  for( index = 0; index < par_set_ref -> rec_no; index++ ) {

    size = par_set_ref -> par_rec_ref[ index ].size;

    switch( par_set_ref -> do_action ) {
      case DO_RESET:
        memcpy( par_set_ref -> work[ index ].value, 
                par_set_ref -> par_rec_ref[ index ].value,
                size );
        break;

      case DO_DEFAULT:
        memcpy( par_set_ref -> work[ index ].value,
                par_set_ref -> par_rec_ref[ index ].default_value,
                size );
        break;

    } /* switch */

  } /* loop */


  /* Select the curerent parameter. */
  tempW = XtNameToWidget( mainW, "ParListLiSW.ParListLi" );

  /* No previous 'history'. */
  if( par_set_ref -> last_valueW != NULL ) {
    XtUnmanageChild( par_set_ref -> last_valueW );
    par_set_ref -> last_valueW = NULL;
  }
    
  XmListSelectPos( tempW, 1, True );


  return;

} /* doDefaultResetCB */


/*----------------------------------------------------------------------*/

static void 
  defaultsCB( Widget           widget,
              PAR_SET_REC_REF  par_set_ref,
              XtPointer        call_data )
{

  /* Code. */

  par_set_ref -> do_action = DO_DEFAULT;

  /* Ask for confirmation. */
  (void ) xitCreateQuestionDialog( 
            par_set_ref -> parSetW, "QuestionDialog",
            " ",
            par_set_ref -> msg_confirm_default,
            doDefaultResetCB, par_set_ref, 
            NULL, NULL );


  return;

} /* defaultsCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget           widget,
             PAR_SET_REC_REF  par_set_ref,
             XtPointer        call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  /* Do we have a user action callback registered? */
  if( par_set_ref -> actionCB != NULL )
    (* par_set_ref -> actionCB)( XIT_PS_REASON_DESTROY, 
                                 par_set_ref -> user_data );


  /* Working values. */
  for( index = 0; index < MAX_PARAMETERS; index++ ) {
    if( par_set_ref -> work[ index ].value != NULL )
      SysFree( par_set_ref -> work[ index ].value );
  }

  /* Internal data. */
  if( par_set_ref != NULL )
    SysFree( par_set_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  intScChangedCB( Widget                 widget,
                  PAR_SET_REC_REF        par_set_ref,
                  XmScaleCallbackStruct  *call_data )
{

  /* Variables. */
  char    buffer[ 50 ];
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  if( ! par_set_ref -> do_update )
    return;

  mainW = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );

  /* Set the value to the text field. */
  tempW = XtNameToWidget( mainW, "IntFo.IntTx" );

  sprintf( buffer, "%d", call_data -> value );
  if( strlen( buffer ) < 2 )
    strcat( buffer, " " );

  par_set_ref -> do_update = False;
  XmTextSetString( tempW, buffer );
  par_set_ref -> do_update = True;


  return;

} /* intScChangedCB */


/*----------------------------------------------------------------------*/

static void 
  intTxChangedCB( Widget           widget,
                  PAR_SET_REC_REF  par_set_ref,
                  XtPointer        call_data )
{

  /* Variables. */
  int          items;
  int          int_value;
  char         *char_string;
  Widget       mainW;
  Widget       tempW;
  XIT_PAR_REC  *par_rec_ref;


  /* Code. */

  if( ! par_set_ref -> do_update )
    return;

  mainW       = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );
  par_rec_ref = &par_set_ref -> par_rec_ref[ par_set_ref -> last_value_no ];

  /* Fetch the new value. */
  tempW = XtNameToWidget( mainW, "IntFo.IntTx" );

  char_string = xitStringGetText( tempW );
  items = sscanf( char_string, "%d", &int_value );
  SysFree( char_string );

  if( items != 1 || 
      int_value < par_rec_ref -> min || 
      int_value > par_rec_ref -> max )
    return;


  /* Set the new value. */
  tempW = XtNameToWidget( mainW, "IntFo.IntSc" );

  par_set_ref -> do_update = False;
  XmScaleSetValue( tempW, int_value );
  par_set_ref -> do_update = True;


  return;

} /* intTxChangedCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget           widget,
        PAR_SET_REC_REF  par_set_ref,
        XtPointer        call_data )
{

  /* Variables. */
  int             index;
  UINT32          flag_value;
  UINT32          work_flag_value;
  size_t          size;
  Boolean         ok;
  XIT_PAR_REC     *par_rec_ref;
  WORK_VALUE_REC  *work_ref;


  /* Code. */

  /* Save the current value. */
  ok = saveParameter( par_set_ref );
  if( ! ok ) {
    xitErMessage( par_set_ref -> parSetW, XIT_ER_ERROR,
                  "", "", par_set_ref -> msg_invalid_value );
    return;
  }


  /* Make the work values 'real'. */
  for( index = 0; index < par_set_ref -> rec_no; index++ ) {

    par_rec_ref = &par_set_ref -> par_rec_ref[ index ];
    work_ref    = &par_set_ref -> work[ index ];
    size        = par_rec_ref -> size;
  
    switch( par_rec_ref -> parameter_type ) {

      case XIT_PS_TYPE_YES_NO_FLAG:
      case XIT_PS_TYPE_ON_OFF_FLAG:
        flag_value      = *((UINT32 *) par_rec_ref -> value);
        work_flag_value = *((UINT32 *) work_ref -> value);

        if( flagIsSet( work_flag_value, par_rec_ref -> flag_offset ) )
          flagSet( flag_value, par_rec_ref -> flag_offset );
        else
          flagClear( flag_value, par_rec_ref -> flag_offset );

        *((UINT32 *) par_rec_ref -> value) = flag_value;
        break;

      case XIT_PS_TYPE_SEPARATOR:
        break;

      default:
        memcpy( par_rec_ref -> value, work_ref -> value, size );
        break;

    } /* switch */

  } /* loop */


  /* Do we have a user action callback registered? */
  if( par_set_ref -> actionCB != NULL )
    (* par_set_ref -> actionCB)( XIT_PS_REASON_OK, 
                                 par_set_ref -> user_data );


  /* Pop-down the window. */
  XtUnmanageChild( par_set_ref -> parSetW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  parListCB( Widget                widget, 
             PAR_SET_REC_REF       par_set_ref,
             XmListCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean         ok;
  Widget          mainW;
  Widget          tempW;
  XIT_PAR_REC     *par_rec_ref;
  WORK_VALUE_REC  *work_ref;


  /* Code. */

  mainW       = XtNameToWidget( par_set_ref -> parSetW, "ParSetFdFo" );
  par_rec_ref = &par_set_ref -> par_rec_ref[ call_data -> item_position - 1 ];
  work_ref    = &par_set_ref -> work[ call_data -> item_position - 1 ];

  /* Separators cannot be selected. */
  if( par_rec_ref -> parameter_type == XIT_PS_TYPE_SEPARATOR ) {
    XmListSelectPos( widget, par_set_ref -> last_value_no + 1, False );

    return;
  }

  /* Save the previous value. */
  ok = saveParameter( par_set_ref );
  if( ! ok ) {
    XmListSelectPos( widget, par_set_ref -> last_value_no + 1, False );

    xitErMessage( par_set_ref -> parSetW, XIT_ER_ERROR,
                  "", "", par_set_ref -> msg_invalid_value );
    return;
  }


  /* Set the parameter help text. */
  tempW = XtNameToWidget( mainW, "DescTxSW.DescTx" );
  XmTextSetString( tempW, par_rec_ref -> parameter_help );


  /* Hid the current value widget (if any). */
  if( par_set_ref -> last_valueW != NULL ) {
    XtUnmanageChild( par_set_ref -> last_valueW );
    par_set_ref -> last_valueW = NULL;
  }


  /* Set the correct parameter value. */
  setParameter( par_set_ref, par_rec_ref, work_ref );


  /* We need these later. */
  par_set_ref -> last_value_no = call_data -> item_position - 1;


  return;

} /* parListCB */


/*----------------------------------------------------------------------*/

static void 
  resetCB( Widget           widget,
           PAR_SET_REC_REF  par_set_ref,
           XtPointer        call_data )
{

  /* Code. */

  par_set_ref -> do_action = DO_RESET;

  /* Ask for confirmation. */
  (void ) xitCreateQuestionDialog( 
            par_set_ref -> parSetW, "QuestionDialog",
            " ",
            par_set_ref -> msg_confirm_reset,
            doDefaultResetCB, par_set_ref, 
            NULL, NULL );


  return;

} /* resetCB */
