/*
 * config.c
 *
 * Eric Dumas (dumas@emi.u-bordeaux.fr)
 * Contribution : Paul avisseau (avisseau@emi.u-bordeaux.fr)
 *
 * $Id:$
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include <config.h>

#define NB_KEYWORD  5

#define NB_ROOM     1
#define NAME_ROOM   2
#define WIDTH_ROOM  3
#define HEIGHT_ROOM 4
#define MAP_ROOM    5

struct room_s
{   
    int     width, 
            height;
    char   *name_room;
    char  **name_station;
};
typedef struct room_s *room_t;

struct config_file_s
{   
    int nb_room;
    room_t room;
};

static char *key_word[] =
{
    "NB_ROOM",
    "NAME_ROOM",
    "WIDTH",
    "HEIGHT",
    "DATA"
};

static int is_keyword(char *buf);
static void msg_error(char *msg, char *arg,int line_cur);
static void read_word(FILE *fd, char *buf);

int get_nb_room(config_file_t cfg)
{
    if (cfg != NULL)
        return cfg->nb_room;

    return 0;
}

int get_width_room(config_file_t cfg, int n)
{
    if (cfg == NULL)
        return 0;

    return cfg->room[n].width;
}

int get_height_room(config_file_t cfg, int n)
{
    if (cfg == NULL)
        return 0;

    return cfg->room[n].height;
}

char *get_name_room(config_file_t cfg, int n)
{
    if (cfg == NULL)
        return NULL;

    return cfg->room[n].name_room;
}

char *get_name_station(config_file_t cfg, int n_room, int n_station)
{
    if (cfg == NULL)
        return NULL;

    return cfg->room[n_room].name_station[n_station];
}

config_file_t load_config_file(char *n_file)
{
    FILE *fd;
    char buf[1024];
    int i,j;
    int line_cur = 0;
    int c;
    config_file_t config_data;

    int nb;
    int state      = 0;
    int no_room    = 0;
    int no_station = 0;
    
    if ((fd = fopen(n_file, "r"))==NULL)
    {
        fprintf(stderr, "Can't open config file : %s \n", n_file);
        exit(-1);
    }

    i=j=0;
    config_data = malloc(sizeof(struct config_file_s));
    config_data->room = NULL;
    
    while( (c = fgetc(fd)) != EOF)
    {
#ifdef DEBUG_CONFIG
        printf("%c",c);
#endif        
        switch (c)
        {
            case ' ': /* blancs */
            case '\t':
            break;            

            case '\n':
               line_cur++;
            break;

            case '#': /* comment */
               while(((c = fgetc(fd)) != EOF) && (c != '\n'));
               if (c=='\n')
                   ungetc(c,fd);
            break;
            
            case '=':
            switch(state)
            {
                case 1:
                case 7:
                case 10:
                case 13:
                state++;
                break;
                
                default:
                  msg_error(" Bad = ","",line_cur );
            }            
            break;
            
            case ';':
            switch(state)
            {
                case 3:
                  state=4;
                break;

                case 9:
                case 12:
                  state = 6;
                break;
                
                default:
                  msg_error(" Bad ; ","",line_cur);
            }
            break;
            
            case '{':
            switch(state)
            {
                case 5:
                case 14:
                  state++;
                break;
                
                default:
                  msg_error(" Bad { ","",line_cur);
            }
            break;
            
            case '}':
            switch(state)
            {
                case 16:
                case 17:
                state = 18;
                break;

                case 18:
                state = 4;
                no_room++;
                no_station = 0;
                break;

                default:
                  msg_error(" Bad } ","",line_cur);
            }
            break;
            
            case ',':
            switch(state)
            {
                case 16:
                case 17:
                state = 15;
                break;
                
                default:
                  msg_error(" Bad , ","",line_cur);
            }
            break;

            default:
               if (isdigit(c))
               {
                   ungetc(c,fd);
                   fscanf(fd,"%d", &nb);

                   switch(state)
                   {
                       case 2:
                       if (!nb)
                           msg_error("can't be NULL : ",
                                  key_word[NB_ROOM-1],line_cur);
                         config_data->nb_room = nb;
                       state++;

                       break;

                       case 8:
                       if (!nb)
                           msg_error("can't be NULL : ",
                                  key_word[WIDTH_ROOM-1],line_cur);
                       config_data->room[no_room].width = nb;
                       state++;
                       break;

                       case 11:
                       if (!nb)
                           msg_error("can't be NULL : ",
                                  key_word[HEIGHT_ROOM-1],line_cur);
                       config_data->room[no_room].height = nb;
                       state++;
                       break;

                       case 15:
                         if (nb)
                           msg_error("must be NULL : ",
                                  key_word[MAP_ROOM-1],line_cur);
                       config_data->room[no_room].name_station[no_station++] = NULL;
                       state++;
                       break;
                       
                       default:
                         msg_error(" Bad digit location ","",line_cur);

                   }
                   break;
               }

               if (isalpha(c))
               {
                   int tmp;
                                      
                   ungetc(c,fd);
                   read_word(fd, buf);

                   if ((tmp = is_keyword(buf)))
                   {
                       switch(state)
                       {
                           case 0:
                           if (tmp != NB_ROOM)
                                 msg_error("Bad nb_room position","",line_cur);
                             state++;
                           break;

                           case 6:
                             if (tmp == WIDTH_ROOM)
                             {
                                 state++;
                                 break;
                             }

                             if (tmp == HEIGHT_ROOM)
                             {
                                 state = 10;
                                 break;
                             }

                             if (tmp == MAP_ROOM)
                             { 
                                 if ((!config_data->room[no_room].width) ||
                                     (!config_data->room[no_room].height))
                                     msg_error(" You must set width and height before data ",
                                           config_data->room[no_room].name_room,line_cur);

                                 config_data->room[no_room].name_station = malloc(
                                                       config_data->room[no_room].width *
                                                       config_data->room[no_room].height *
                                                       sizeof(char *));

                                 for (i = 0 ; i <config_data->room[no_room].width *
                                     config_data->room[no_room].height; i++ )
                                     config_data->room[no_room].name_station[i] = NULL;
                                 
                                 state = 13;
                                 break;
                             }
                             msg_error(" Bad key word location ","",line_cur);
                           break;

                           default:
                             msg_error(" Bad key word location ","",line_cur);
                           
                       }
                   }
                   else
                   switch(state)
                   {
                       case 4:
                       if (config_data->room == NULL)
                       {
                           config_data->room = malloc(config_data->nb_room*
                                                sizeof(struct room_s));
                           for (i=0; i < config_data->nb_room; i++)
                           {
                               config_data->room[i].name_room = NULL;
                               config_data->room[i].width     = 0;
                               config_data->room[i].height    = 0;
                               config_data->room[i].name_station = NULL;
                           }
                       }
                       no_station = 0;

                    /*   if (no_station > config_data->room[i].width*
                           config_data->room[i].height) 
                           msg_error(" Room section too long ",buf,line_cur); */

                       config_data->room[no_room].name_room = strdup(buf);
                       
                       state++;
                       break;
                       
                       case 15:
                       config_data->room[no_room].name_station[no_station++] = strdup(buf);
                       state=17;
                       break;
                       
                       default:
                       msg_error(" Bad  string location ",buf,line_cur);
                   }
               }
        }
    }
    

#ifdef DEBUG_CONFIG 
    printf(" Checking Config : (Nb Room = %d) \n",
           config_data->nb_room);
    
    for(i=0; i < config_data->nb_room ; i++)
    {
        printf(" Name Room : %s Width %d  Height %d \n",
               config_data->room[i].name_room, 
               config_data->room[i].width, 
               config_data->room[i].height);
        printf("---> Machines \n");
        for(j=0; j < config_data->room[i].width*config_data->room[i].height ; j++)
            if (config_data->room[i].name_station[j] == NULL)
                printf("0\n");
            else
                printf("%s\n",config_data->room[i].name_station[j]); 
    }
#endif 
    
    fclose(fd);

    return config_data;
}

/* Static functions */
static int is_keyword(char *buf)
{
    int i=-1;
    char *tmp = malloc(sizeof(char)*(strlen(buf)+1));

    while(buf[++i] != '\0')
        tmp[i] = toupper(buf[i]);

    tmp[i]='\0';
    
    for (i=0 ; i < NB_KEYWORD ; i++)
        if (!strcmp(tmp, key_word[i]))
        {
            free(tmp);
            return i+1;
        }
    
    free(tmp);
    return 0;
}

static void msg_error(char *msg, char *arg,int line_cur)
{
    fprintf(stderr, "Error in the config file line %d < %s > : %s\n", 
            line_cur, msg, arg);
    exit(0);
}

static void read_word(FILE *fd, char *buf)
{
    int c;
    int i=0;
    
    while ( (c = fgetc(fd)) != EOF)
        if ((isalpha(c)) || (c=='.') || (c == '_') || (isdigit(c)))
            buf[i++] = c;
        else
            break;
    ungetc(c,fd);

    buf[i] = '\0';
}



