/*
 * author: tianhua han, 
 *         than@lucent.com
 *         tian@aluxpo.micro.lucent.com
 */                       

#include <utmpx.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/procfs.h>

static void usage (char *option, char *program_name)
{
      fprintf(stderr, "usage: %s {-a -d} [-h displayname] [-w wtmpx_file] [-u utmpx_file]\n", program_name);
      fprintf(stderr, "                  [-p xdm_pid] [-l line_name] [-x xslot_start] [user]\n");
      fprintf(stderr, "  -h displayname  display name, default $DISPLAY\n");
      fprintf(stderr, "  -w wtmpx_file   wtmpx file name, default /var/adm/wtmpx\n");
      fprintf(stderr, "  -u utmpx_file   wtmpx file name, default /var/adm/utmpx\n");
      fprintf(stderr, "  -p xdm_pid      the process id of the xdm instance handling this x-session.\n");
      fprintf(stderr, "                  default is the grand-parent pid.\n");
      fprintf(stderr, "  -l line_name    the tty name for this xsession. e.g. /dev/xdm/12345 or xdm/12345.\n");
      fprintf(stderr, "                  default /dev/null or null.\n");
      fprintf(stderr, "  -x xslot_start  the number of the entry in the utmpx_file below where will be \n");
      fprintf(stderr, "                  the entries for xdm instances. default 300. (100 to 600 inclusive) \n");
      fprintf(stderr, "  user            the user's logname of this x session. default $USER\n");
      if(option) fprintf(stderr, "error at \"%s\".\n", option);
      exit (1);
}

/* check whether it's an integer */
static int isInt(char *string)
{
   int i;
   for(i=0; i<strlen(string)-1; i++) 
      if( ! isdigit((char)string[i]) ) return 0;
   return 1;
}
                    
void main(int argc, char **argv)
{
   int    addflag=0,
          delflag=0,
          i, uxfd, wxfd, slot, freeslot, found, 
          trylock, maxtrylock,
          xslot_start = 300,            /* default for xslot_start */
          sizeofutmpx = sizeof(struct utmpx),
          sizeoflock  = sizeof(struct flock);
   char   *user=NULL, 
          *displayname=NULL,
          *line_name="/dev/null",       /* default for line_name */
          *utmpx_file="/var/adm/utmpx", /* default for utmpx_file */
          *wtmpx_file="/var/adm/wtmpx", /* default for wtmpx_file */
          dev_name[256];
   struct utmpx *ux, entryx;
   struct flock otlock, mylock;
   pid_t  xdm_pid=0;      /* xdm_pid should be the grandparent pid of this program. */
   time_t date=time(NULL);
    
#ifdef DEBUG
   /* 
    *use this to show the bug in x11r5's xdm -- it cannot handle
    *  the startup and reset program well.
    */
   (void) fprintf(stderr, "debug:userid=%d.\n", (int)getuid());
#endif

   if(argc == 999 ) {
        /* this message is for strings(1). */
        printf("Here!!!.\n");
        printf("the source is in tian's pub/src. tian@aluxpo.micro.lucent.com, x7668. \n");
   }

   for(i=1; i<argc; i++) {
       char *arg = argv[i];
       if(arg[0] == '-') {
          switch (arg[1]) {
             case 'w':
                if(++i >= argc) usage("-w", argv[0]);
                wtmpx_file = argv[i];
                continue;
             case 'u':
                if(++i >= argc) usage("-u", argv[0]);
                utmpx_file = argv[i];
                continue;
             case 'l':
                if(++i >= argc) usage("-l", argv[0]);
                line_name = argv[i];
                continue;
             case 'h':
                if(++i >= argc) usage("-h", argv[0]);
                displayname = argv[i];
                continue;
             case 'x':
                if(++i >= argc) usage("-x", argv[0]);
                if( ! isInt(argv[i])) {
                     fprintf(stderr, 
                             "xslot_start is an integer (default 300).\n");
                     usage("-x", argv[0]);
                }
                xslot_start = atoi(argv[i]);
                if( 100>xslot_start || 600<=xslot_start) {
                     fprintf(stderr, 
                             "xslot_start out of range (100 to 600 inclusive.).\n");
                     usage("-x", argv[0]);
                }
                continue;
             case 'p':
                if(++i >= argc) usage("-p", argv[0]);
                if( ! isInt(argv[i])) {
                     fprintf(stderr, 
                             "xdm_pid is the pid of your xdm session.\n");
                     usage("-p", argv[0]);
                }
                xdm_pid = (pid_t)atol(argv[i]);
                continue;
             case 'a':
                addflag++;
                continue;
             case 'd':
                delflag++;
                continue;
             default: 
                usage((char*)NULL, argv[0]);
          }
       } else { 
            if(user) {
               fprintf(stderr, "%s: ambiguous users.\n", argv[0]);
               usage("user", argv[0]);
            }
            user=argv[i];
       }
   }

   /* { -a -d } */
   if ( ( addflag>0 && delflag>0) || ( addflag<=0 && delflag<=0) ) usage("{-a -d}", argv[0]);

   /* check display name which will be used as hostname */
   if ( !displayname ) 
       if((displayname=getenv("DISPLAY"))==NULL) {
           fprintf(stderr, "error: no display specified. \n");
           usage("-d", argv[0]);
       }

   /* check user-name */
   if ( !user) 
      if( (user=getenv("USER")) == NULL )  {
          fprintf(stderr, "cannot get user-name.\n");
          fprintf(stderr, "please specify user-name either on command-line or by USER environment variable.\n");
          usage("user", argv[0]);
      }

   /* check xdm_pid. */
   if ( (pid_t)0 == xdm_pid && !addflag )  {
       prpsinfo_t info;
       int fd;
       char proc[256];

       sprintf(proc,"/proc/%ld", (long)getppid());
       if ( (fd=open(proc,O_RDONLY)) == -1 ) {
             perror("open(proc,O_RDONLY))");
             fprintf(stderr, "%s: getting default xdm-pid failed.\n", argv[0]);
             exit(1);
       }

       if ( ioctl(fd, PIOCPSINFO, &info) == -1 ) {
             perror("ioctl(fd, PIOCPSINFO, &info)");
             fprintf(stderr, "%s: getting default xdm-pid failed.\n", argv[0]);
             exit(1);
       }
       xdm_pid=info.pr_ppid;

       close(fd);
   }

   /* line name may be  like /dev/xdm/12345 or simply xdm/12345 */
   if( strstr(line_name, "/dev/")) {
       if( access(line_name, F_OK) == -1 ) {
           perror(line_name);
           if(addflag)   exit(1);   /* invalid line-name is bad only when adding an entry*/ 
       }
       line_name = line_name + 5; /* stripe out the leading "/dev/" */
   } else {
       (void) sprintf(dev_name, "/dev/%s", line_name);
       if( access(dev_name, F_OK) == -1 ) {
           perror(dev_name);
           if(addflag)   exit(1);  /* invalid line-name is bad only when adding an entry*/
       }
   }

   /* now process solaris 2.5's utmpx and wtmpx files */
   /*
    * when operate on the utmpx and wtmpx files, we need set lock to avoid racing
    * problem. when adding an entry, we only try three times to lock the files.
    * when deleting an entry, we can wait a little longer, we try 10 times.
    */
   if ( addflag==1 ) maxtrylock = 3;
   else              maxtrylock = 10;

   if( (ux=(struct utmpx *)malloc(sizeofutmpx)) == NULL ) {
       perror("cannot allocate memory.\n");
       exit(1);
   }

   if ( addflag ) {
       (void) strncpy(ux->ut_user, user, sizeof(ux->ut_user));
       (void) strncpy(ux->ut_id, "nul", sizeof(ux->ut_id));
       (void) strncpy(ux->ut_line, line_name, sizeof(ux->ut_line));
       ux->ut_pid = xdm_pid;
       ux->ut_type = USER_PROCESS;
       (void) strncpy(ux->ut_host, displayname, sizeof(ux->ut_host)); 
   } else {
       bzero(ux->ut_user, sizeof(ux->ut_user));
       ux->ut_pid = 0;
       (void) strncpy(ux->ut_line, line_name, sizeof(ux->ut_line));
       ux->ut_type = DEAD_PROCESS;
       bzero(ux->ut_host, sizeof(ux->ut_host));
   }
   ux->ut_xtime = date;
   
   if( (uxfd=open(utmpx_file, O_RDWR)) == -1) {
       perror(utmpx_file);
       exit (1);
   } 

   /* to save cpu time, put all x entries near the bottom of account files*/
   if( lseek(uxfd, (long)((long)xslot_start*sizeofutmpx), 0) == -1 ) {
       perror("(int)lseek(uxfd, (long)((long)xslot_start*sizeofutmpx), 0)");
       exit(1);
   }
        
   /* prepare the lock */
   mylock.l_type   = F_WRLCK; /* a write lock (exclusive) */
   mylock.l_whence = SEEK_CUR; /* from current file pointer */
   mylock.l_start  = 0;        /* start right at mylock.l_whence */
   mylock.l_len    = 0;        /* lock up to the end. */ 
   mylock.l_pid    = getpid(); /* don't use xdm_pid, which is this ps' grandfa*/

   /* 
    *mylock: my lock;
    * otlock: the lock of other people.
    */
   bcopy(&mylock, &otlock, sizeoflock);

   /* try to lock the section to be written in the utmpx file */
   trylock=0;
   while( fcntl(uxfd, F_SETLK, &mylock) == -1) {
        /* if Set lock fails, find out who has locked the file. */
        if( fcntl(uxfd, F_GETLK, &otlock) != -1 && otlock.l_type != F_UNLCK ) {
             (void) fprintf(stderr, 
                     "%s is locked by process %d from for %d to %d for %s.\n",
                     utmpx_file,
                     (int)otlock.l_pid,
                     (int)(otlock.l_start / sizeofutmpx + xslot_start),
                     (int)(otlock.l_len / sizeofutmpx),   
                     (mylock.l_type == F_WRLCK ? "writing" : "reading") );
        } 
        if( trylock >= maxtrylock ) {
              (void) fprintf(stderr,
                             "%s failed due to file %s is locked.\n",
                             argv[0],
                             utmpx_file);
              exit (1);
        } 
        trylock ++;
        (void) sleep(1);
   }
         
   /* to make sure, do lseek again */
   if( lseek(uxfd, (long)((long)xslot_start*sizeofutmpx), 0) == -1 ) {
       perror("(int)lseek(uxfd, (long)((long)xslot_start*sizeofutmpx), 0)");
       exit(1);
   }

   freeslot = -1;
   slot     = xslot_start;
   found    =  0;
   while( read(uxfd, (char *) &entryx, sizeofutmpx) == sizeofutmpx ) {
       /* first, try to reclaim one if possible, this is based on the 
          uniqueness of display name at one time. and if you don't reclaim 
          it yourself, no one else could do it for you -- it just wastes.
       */
       if( strncmp(entryx.ut_host, displayname, sizeof(entryx.ut_host)) == 0 ) {
             found = 1;
             break;
       } 
       if(freeslot < 0 && *entryx.ut_user == '\0')
                freeslot = slot;
       ++slot;
   }

   if ( (!found) && freeslot >= 0)   slot = freeslot;

   if( lseek(uxfd, (long)((long)slot*sizeofutmpx), 0) == -1) {
        perror("lseek(uxfd, (long)((long)slot*sizeofutmpx), 0)");
        exit (1);
   }
   if(write(uxfd,(char*)ux,sizeofutmpx) != sizeofutmpx) { 
        perror("write utmpx-file");
        exit(1);
   }
 
   mylock.l_type   =  F_UNLCK;
   mylock.l_whence =  SEEK_SET;
   mylock.l_start  =  (long) xslot_start*sizeofutmpx;
   mylock.l_len    =  0;
   if(fcntl(uxfd, F_SETLKW, &mylock) == -1) {
        perror("when unlock, fcntl(uxfd, F_SETLKW, &mylock) == -1 failed.");
   }     

   (void) close(uxfd);

   if( (wxfd=open(wtmpx_file, O_WRONLY|O_APPEND)) == -1 ) {
       perror("open wtmpx");
       exit(1);
   }

   if(write(wxfd, (char*)ux, sizeofutmpx) != sizeofutmpx) {
       perror("write wtmpx_file");
       exit(1);
   }
    
   (void) close(wxfd); 
  
   free(ux);

   exit(0);
}
