/* ao_symscan.c - do initial symbol table scan */

/*  Copyright 1995 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_ao_symscan_c_sccsid[] = "@(#)ao_symscan.c	1.3 20 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>
#include "ao_ifdefs.h"

#ifdef AO_TARGET

#include <sys/types.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef AO_ELF
#	include "elfstab.h"
#else
#	include <a.out.h>
#	ifndef OS_RISCOS
#		include <stab.h>
#	endif
#endif

#include <local/ukcprog.h>

#include "ups.h"
#include "symtab.h"
#include "ci.h"
#include "st.h"
#include "ao_syms.h"
#include "ao_symload.h"
#include "ao_symscan.h"
#include "ao_symread.h"
#include "ao_elfsym.h"
#include "ao_symcb.h"
#include "util.h"

/* Options passed to gnu_cplus_demangle (in 2nd parameter). */

#define DMGL_NO_OPTS	0		/* For readability... */
#define DMGL_PARAMS	(1 << 0)	/* Include function args */
#define DMGL_ANSI	(1 << 1)	/* Include const, volatile, etc */

#define DMGL_AUTO	(1 << 8)
#define DMGL_GNU	(1 << 9)
#define DMGL_LUCID	(1 << 10)
#define DMGL_ARM	(1 << 11)
/* If none of these are set, use 'current_demangling_style' as the default. */
#define DMGL_STYLE_MASK (DMGL_AUTO|DMGL_GNU|DMGL_LUCID|DMGL_ARM)

/* Enumeration of possible demangling styles.

   Lucid and ARM styles are still kept logically distinct, even though
   they now both behave identically.  The resulting style is actual the
   union of both.  I.E. either style recognizes both "__pt__" and "__rf__"
   for operator "->", even though the first is lucid style and the second
   is ARM style. (FIXME?) */

extern enum demangling_styles
{
  unknown_demangling = 0,
  auto_demangling = DMGL_AUTO,
  gnu_demangling = DMGL_GNU,
  lucid_demangling = DMGL_LUCID,
  arm_demangling = DMGL_ARM
} current_demangling_style;

/* Define string names for the various demangling styles. */

#define AUTO_DEMANGLING_STYLE_STRING	"auto"
#define GNU_DEMANGLING_STYLE_STRING	"gnu"
#define LUCID_DEMANGLING_STYLE_STRING	"lucid"
#define ARM_DEMANGLING_STYLE_STRING	"arm"

/* Some macros to test what demangling style is active. */

#define CURRENT_DEMANGLING_STYLE current_demangling_style
#define AUTO_DEMANGLING (((int) CURRENT_DEMANGLING_STYLE) & DMGL_AUTO)
#define GNU_DEMANGLING (((int) CURRENT_DEMANGLING_STYLE) & DMGL_GNU)
#define LUCID_DEMANGLING (((int) CURRENT_DEMANGLING_STYLE) & DMGL_LUCID)
#define ARM_DEMANGLING (CURRENT_DEMANGLING_STYLE & DMGL_ARM)

static char *gnu_cplus_demangle PROTO((const char *mangled, int options));

static void wrapup_stf PROTO((stf_t *stf, hf_t **fmap, int mapsize));
static hf_t *lookup_hf PROTO((hf_t *headers, int id));
static void build_symset PROTO((char *aout_symset, char *ofile_symset));
static bool is_fortran_void PROTO((const char *cptr));
static void reverse_stf_funclist PROTO((stf_t *stf));
static void adjust_fil_vars_addr_base PROTO((fil_t *flist, long delta));
static fil_t *add_sol_fil PROTO((stf_t *stf, alloc_pool_t *ap, fil_t *sfiles,
				 fil_t *sofil, const char *name));
static void add_to_fil_funclist PROTO((alloc_pool_t *ap, fil_t *fil, 
                                       func_t *f));
#ifdef AO_ELF
static func_t *note_ofile_function PROTO((alloc_pool_t *ap,
					  alloc_pool_t *tmp_ap, stf_t *stf,
					  const char *path,
					  fil_t *solfil, unsigned sline_offset,
					  const char *namestr, int symno));
#endif

void display_message PROTO((const char *mesg));

const char
bump_str(sr, p_s)
Symrec *sr;
const char **p_s;
{
  char c;

  if (**p_s == '\\')
    *p_s = get_cont_symstring(sr);
  c = **p_s;
  *(*p_s)++;
  if (**p_s == '\\')
    *p_s = get_cont_symstring(sr);
  return c;
}

#ifndef ST_TE
/*  Deduce language of source file from name.
 *  *.c is C, *.f is f77, *.f90 is f90, anything else is unknown.
 *
 *  BUG: dubious way of getting language.
 */
language_t
srctype(name)
const char *name;
{
	char *suf;

	if ((suf = strrchr(name, '.')) == NULL)
		return LANG_UNKNOWN;
	else
		++suf;
	
	if (strcmp(suf, "c") == 0)
		return LANG_C;
	
	if (strcmp(suf, "f") == 0)
		return LANG_F77;

	if (strcmp(suf, "f90") == 0)
		return LANG_F90;
	
	if (strcmp(suf, "cc") == 0 || strcmp(suf, "C") == 0 ||
	    strcmp(suf, "CC") == 0 || strcmp(suf, "c++") == 0 ||
	    strcmp(suf, "cpp") == 0 || strcmp(suf, "cxx") == 0 ||
	    strcmp(suf, "H") == 0 || strcmp(suf, "hh") == 0)
	  return LANG_CC;	

	return LANG_UNKNOWN;
}

/*  Called when we have finished with an stf structure.  Create space
 *  for the map and copy the stuff to it.
 */
static void
wrapup_stf(stf, orig_fmap, mapsize)
stf_t *stf;
hf_t **orig_fmap;
int mapsize;
{
	hf_t **fmap;
	int i;
	
	if (mapsize == 0)
		panic("mapsize 0 in wrapup_stf");
	
	fmap = (hf_t **)alloc(stf->stf_symtab->st_apool,
			      mapsize * sizeof(hf_t *));
	memcpy((char *)fmap, (char *)orig_fmap, mapsize * sizeof(hf_t *));

	for (i = 0; i < mapsize; ++i) {
		stf_t *hstf;

		hstf = fmap[i]->hf_stf;

		if (hstf->stf_fmap == NULL) {
			hstf->stf_fmap = fmap;
			hstf->stf_mapsize = mapsize;
		}
	}
}

static void
reverse_stf_funclist(stf)
stf_t *stf;
{
	funclist_t *fl, *next, *newlist;

	newlist = NULL;
	for (fl = stf->stf_fil->fi_funclist; fl != NULL; fl = next) {
		next = fl->fl_next;
		fl->fl_next = newlist;
		newlist = fl;
	}
	
	stf->stf_fil->fi_funclist = newlist;
}

#ifdef OS_SUNOS
/*  Look up the header file entry with id id in the headers list headers.
 *
 *  Used when we find a N_EXCL symbol meaning a header file excluded
 *  because it has already been encountered.
 */
static hf_t *
lookup_hf(headers, id)
hf_t *headers;
int id;
{
	hf_t *hf;

	for (hf = headers; hf != NULL; hf = hf->hf_next)
		if (hf->hf_id == id)
			return hf;
	panic("id not found in lookup_hf");
	/* NOTREACHED */
	return NULL;	/* to keep gcc happy */

}
#endif /* OS_SUNOS */

/*  Parse a name (NAME in the grammar).
 *  If save is non zero, return a pointer to a saved copy of the name,
 *  otherwise return NULL.  Names are saved via an alloc() on ap.
 */
const char *
parse_name(sr, p_s, ap, func, compiler)
Symrec *sr;
const char **p_s;
alloc_pool_t *ap;
int func;
Compiler_type compiler;
{
	register const char *s, *sub;
	char *name;
	register int dots = 0;
	size_t len;

	s = *p_s;

	 
	/*  Gcc 2.3.3 seems to put out spaces in names, so skip them.
	 */
	while (isspace(*s))
		++s;

	/*  The test for ':' is needed because gcc 2.3.1 puts out symbol
	 *  table entries containing empty names (i.e. nothing before the
	 *  ':') for unnamed enums.  SunOS f77 emits `block.data' as a
	 *  symbol, so we allow dots in names.  I don't remember why `$'
	 *  is allowed.
	 */
	/*  RGA The test for '?' is needed because Pure-link puts out bad */ 
	/*  names apparently */
	if (*s != ':' && *s != '?' && !isalpha(*s) && *s != '_' && *s != '$' && *s != '.')
		panic("bad name in parse_name");

	while (isalnum(*s) || *s == '_' || *s == '$' || *s == '.')
	{
	  if (*s == '.')
	    dots++;
	  ++s;
	}

	if (*s == '\\')
	{
	  char *s1, *s2;
	  int i = 0, new_len;

	  s1 = (char *)get_cont_symstring(sr);
	  new_len = strlen(s1);
	  s2 = alloc(ap, s - *p_s + new_len + 1);
	  memcpy(s2, *p_s, s - *p_s);
	  memcpy(s2 + (s - *p_s), s1, new_len);
	  s2[s - *p_s + new_len] = 0;
	  s = s2;
	  dots = 0;
	  while (isalnum(*s) || *s == '_' || *s == '$' || *s == '.')
	  {
	    if (*s == '.')
	      dots++;
	    ++s, ++i;
	  }
	  len = i;
	  *p_s = s2;
	}
	else
	  len = s - *p_s;

	/* RGA SC4 static data is of the form .damangled_func_name.static_data */
	if (dots > 0)
	  for (sub = s; sub != *p_s; sub--)
	    if (*sub == '.')
	    {
	      sub++;
	      len = s - sub;
	      *p_s = sub;
	      break;
	    }
	
	if (ap != NULL)
	  demangle_name((char *)*p_s, len, ap, &name, func == 1, compiler);
	else 
	  name = NULL;
	
	*p_s = s;
	return name;
}

/*  Return a pointer to a munged saved copy of fname.  Ext is true
 *  if name is an external symbol generated by the linker.  These
 *  symbols are expected to have an underscore prepended - if it
 *  is there it is stripped, otherwise the returned name is
 *  enclosed in brackets (e.g. "[name]").
 *
 *  If modsepc is not NUL, and it occurs in the name, we split the name
 *  into a module name and function name at the seperator character.
 *
 *  We use alloc() to get the space.
 *
 *  NOTE: this routine is not used on the DECstation 3100.
 */
void
parse_fname(ap, name, modsepc, ext, p_modname, p_fname)
alloc_pool_t *ap;
const char *name;
int modsepc;
bool ext;
const char **p_modname, **p_fname;
{
	const char *modname;
	char *fname;
	
#ifdef COFF_SUN386
	/*  On the 386i, external symbols don't seem to
	 *  have underscores prepended, so we force ext
	 *  to false.
	 */
	ext = FALSE;
#endif
	
	if (ext && *name != '_') {
		modname = NULL;
		fname = alloc(ap, strlen(name) + 3);
		sprintf(fname, "[%s]", name);
	}
	else {
		const char *cptr;
		char *pos;
		size_t len;

		if (ext)
			name++;
		
		for (cptr = name; *cptr != '\0' && *cptr != ':'; cptr++)
			;
		
		len = cptr - name;
		fname = alloc(ap, len + 1);
		memcpy(fname, name, len);
		fname[len] = '\0';

		if (modsepc == '\0' || (pos = strchr(fname, modsepc)) == NULL) {
			modname = NULL;
		}
		else {
			modname = fname;
			*pos++ = '\0';
			fname = pos;
		}
	}

	*p_modname = modname;
	*p_fname = fname;
}
#endif /* !ST_TE */

fsyminfo_t *
make_fsyminfo(ap, symno)
alloc_pool_t *ap;
int symno;
{
	fsyminfo_t *fs;

	fs = (fsyminfo_t *)alloc(ap, sizeof(fsyminfo_t));
	fs->fs_initial_lno_fil = NULL;
	fs->fs_initial_sline_offset = 0;
	fs->fs_symno = symno;
	fs->fs_cblist = 0;
#if defined(ARCH_SUN386) && defined(OS_SUNOS)
	fs->fs_coff_lno_start = 0;
	fs->fs_coff_lno_lim = 0;
#endif
	return fs;
}

/*  Allocate a new stf_t structure with alloc() and fill in the fields.
 */
stf_t *
make_stf(ap, name, st, symno, language, addr)
alloc_pool_t *ap;
const char *name;
symtab_t *st;
int symno;
language_t language;
taddr_t addr;
{
#ifdef AO_ELF
	int ar;
#endif
	stf_t *stf;

	stf = (stf_t *)alloc(ap, sizeof(stf_t));
	stf->stf_name = name;
	stf->stf_language = language;
	/*	stf->stf_compiler_type = CT_UNKNOWN;*/
	stf->stf_compiler_type = ao_compiler(NULL, FALSE, CT_UNKNOWN);
	stf->stf_symtab = st;
	stf->stf_symno = symno;
	stf->stf_addr = addr;
	stf->stf_flags = 0;
	stf->stf_fmap = NULL;
	stf->stf_mapsize = 0;
#ifdef ARCH_CLIPPER
	stf->stf_addrlist = NULL;
#endif
#ifndef ST_TE
	stf->stf_snlist = NULL;
	stf->stf_ftypes = NULL;
#endif
#ifdef AO_ELF
	stf->stf_objpath_hint = NULL;
	stf->stf_objname = NULL;
	stf->stf_obj_mtime = 0;
	stf->stf_global_prefix = NULL;
	stf->stf_symio = NULL;

	for (ar = 0; ar < AR_NTYPES; ++ar) {
		stf->stf_range[ar].base = 0;
		stf->stf_range[ar].lim = 0;
	}
#endif
	return stf;
}

/*  Allocate a new fil_t structure with alloc() and fill in the fields.
 */
fil_t *
make_fil(stf, parblock, path_hint, next)
stf_t *stf;
block_t *parblock;
const char *path_hint;
fil_t *next;
{
	fil_t *fil;
	alloc_pool_t *ap;

	ap = stf->stf_symtab->st_apool;

	fil = ci_make_fil(ap, stf->stf_name, (char *)stf,
			  ci_make_block(ap, parblock), next);
	
	fil->fi_path_hint = path_hint;
	fil->fi_source_path = (char *)path_hint;
	fil->fi_language = stf->stf_language;
	fil->fi_symtab = stf->stf_symtab;

	return fil;
}

bool
find_sol_fil(sfiles, path_hint, name, p_fil)
fil_t *sfiles;
const char *path_hint, *name;
fil_t **p_fil;
{
	fil_t *fil;
	bool abspath;

	abspath = *name == '/';
	
	for (fil = sfiles; fil != NULL; fil = fil->fi_next) {
	  if ((abspath || same_string(fil->fi_path_hint, path_hint)) &&
	      strcmp(fil->fi_name, name) == 0) {
			*p_fil = fil;
			return TRUE;
		}
	}

	return FALSE;
}

static fil_t *
add_sol_fil(stf, ap, sfiles, sofil, name)
stf_t *stf;			/* RGA */
alloc_pool_t *ap;
fil_t *sfiles, *sofil;
const char *name;
{
	fil_t *fil;
	const char *path_hint;

	path_hint = (*name != '/') ? sofil->fi_path_hint : NULL;
	
	if (find_sol_fil(sfiles, path_hint, name, &fil))
		return fil;
	
	fil = ci_make_fil(ap, alloc_strdup(ap, name), (char *)stf,
			  ci_make_block(ap, (block_t *)NULL), sofil->fi_next);
		
	fil->fi_path_hint = path_hint;
	fil->fi_source_path = (char *)path_hint;
	fil->fi_language = srctype(fil->fi_name_only);

	/*  The only thing these entries are used for is displaying
	 *  source code, but we need fi_symtab because things like
	 *  open_source_file() go via it to get an alloc pool.
	 */
	fil->fi_symtab = sofil->fi_symtab;

	fil->fi_flags |= FI_DONE_VARS | FI_DONE_TYPES;

	sofil->fi_next = fil;

	return fil;
}

#ifndef ST_TE
#define SYMSET_SIZE	256

/*  Build the map of interesting symbols for scan_symtab.  We could do this
 *  at compile time with some effort, but it's much less hassle to do it at
 *  run time like this.
 */
static void
build_symset(aout_symset, ofile_symset)
char *aout_symset, *ofile_symset;
{
	static int common_syms[] = {
		N_BCOMM, N_STSYM, N_GSYM, N_LCSYM, N_FUN, N_SOL,
#ifdef OS_SUNOS
		N_BINCL, N_EXCL, N_EINCL,
#endif
#ifdef N_XLINE
		N_XLINE,
#endif
#ifdef AO_ELF
		N_UNDF,
#endif
	};
	static int aout_only_syms[] = {
		N_SO,
		N_TEXT, N_TEXT | N_EXT, N_BSS | N_EXT, N_DATA | N_EXT,
#if defined(OS_ULTRIX) || defined(ARCH_CLIPPER)
		N_DATA,
#endif
#ifdef AO_ELF
		N_OPT,
#endif
#ifdef N_MAIN
		N_MAIN
#endif
	};
	int i;

	memset(ofile_symset, '\0', SYMSET_SIZE);
	for (i = 0; i < sizeof(common_syms) / sizeof(common_syms[0]); ++i)
		ofile_symset[common_syms[i]] = TRUE;

	memcpy(aout_symset, ofile_symset, SYMSET_SIZE);
	for (i = 0; i < sizeof(aout_only_syms) / sizeof(aout_only_syms[0]); ++i)
		aout_symset[aout_only_syms[i]] = TRUE;
}

bool
parse_number(stf, sr, p_s, p_val)
stf_t *stf;
Symrec *sr;
const char **p_s;
int *p_val;
{
	const char *s;
	int res;

	s = *p_s;

	if (*s == '\\')
	  s = get_cont_symstring(sr);

	if (!isdigit(*s))
		return FALSE;
	
	res = 0;
	if (stf->stf_compiler_type ==  CT_GNU_CC && s[-1] == '/' && s[-2] == ':')
	  bump_str(sr, &s);	/* RGA - g++ puts out these... */
	if (*s == '0') {
		bump_str(sr, &s);
		if (*s == 'x' || *s == 'X') {
			bump_str(sr, &s);
			for (;;) {
				if (isdigit(*s))
					res = res * 16 + bump_str(sr, &s) - '0';
				else if (*s >= 'A' && *s <= 'F')
					res = res * 16 + bump_str(sr, &s) - 'A' + 10;
				else if (*s >= 'a' && *s <= 'f')
					res = res * 16 + bump_str(sr, &s) - 'a' + 10;
				else
					break;
			}
		}
		else {
		    while (*s >= '0' && *s <= '7')
		      res = res * 8 + bump_str(sr, &s) - '0';
		}
	      }
	else {
		while (isdigit(*s))
		  res = res * 10 + bump_str(sr, &s) - '0';
	}
	if (*s == '\\')
	  s = get_cont_symstring(sr);
	*p_s = s;
	*p_val = res;
	return TRUE;
}
bool
parse_typenum(stf, sr, assume_paren, p_s, p_fnum, p_tnum)
stf_t *stf;
Symrec *sr;
bool assume_paren;
const char **p_s;
int *p_fnum, *p_tnum;
{
	const char *s;
	bool ok;

	s = *p_s;
	if (assume_paren == TRUE || *s == '(') {
	  if (assume_paren == FALSE)
		  bump_str(sr, &s);
		ok = (parse_number(stf, sr, &s, p_fnum) &&
		      bump_str(sr, &s) == ',' &&
		      parse_number(stf, sr, &s, p_tnum) &&
		      bump_str(sr, &s) == ')');
	}
	else {
		*p_fnum = 0;
		ok = parse_number(stf, sr, &s, p_tnum);
	}

	if (ok)
		*p_s = s;
	return ok;
}

bool
char_to_utypecode(c, p_typecode)
int c;
typecode_t *p_typecode;
{
	switch (c) {
	case 's':
		*p_typecode = TY_U_STRUCT;
		break;
	case 'u':
		*p_typecode = TY_U_UNION;
		break;
	case 'e':
		*p_typecode = TY_U_ENUM;
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

/*  BUG: this is gross, and wrong to boot.  The number of basic types
 *       varies between f77 compilers.  See the comment about this
 *       in get_fi_vars().
 */
#define N_FIXED_FORTRAN_TYPES	9

bool
symtab_name_to_sn(snlist, name, p_sn)
snlist_t *snlist;
const char *name;
snlist_t **p_sn;
{
	snlist_t *sn;

	for (sn = snlist; sn != NULL; sn = sn->sn_next) {
		if (strcmp(sn->sn_demangled_symtab_name, name) == 0) {
			*p_sn = sn;
			return TRUE;
		}
	}

	return FALSE;
}

/*  Do a prescan of a symbol table.  We don't load in all of the symbol
 *  table on startup as that costs a lot in startup time for big symbol
 *  tables.  Instead we load enough to get us going (basically function
 *  and global variable names and addresses) and pull other bits in as
 *  needed.  This is a big win in normal use because the average debugger
 *  run touches only a small number of functions and global variables.
 *  Most of the symbol table is never read in at all.
 *
 *  The Sun C compiler has a system for reducing symbol table size by
 *  only including header file symbol table information once.  We have
 *  to do a fair amount of bookkeeping to resolve the header file
 *  references.
 *
 *  For most things that we load in this pass (like functions, global
 *  variables etc) we record the symbol number range of the object.
 *  This is so we can pull in more information when needed (e.g. the
 *  local variables of a function, the globals of a source file).
 */
void
scan_symtab(st, path, stf, p_flist, p_mainfunc_name)
symtab_t *st;
const char *path;
stf_t *stf;
func_t **p_flist;
const char **p_mainfunc_name;
{
	static int first_call = TRUE;
	static char aout_symset[SYMSET_SIZE], ofile_symset[SYMSET_SIZE];
	char *symset;
	snlist_t *sn;
	block_t *rootblock;
	fil_t *solfil;
	symio_t *symio;
	Symrec symrec;
	nlist_t nm;
#ifdef OS_SUNOS
	hf_t *headers, *hf;
#endif
#ifdef AO_ELF
	off_t file_offset, next_file_offset;
#endif
	hf_t **fmap, **istack;
	ao_stdata_t *ast;
	func_t *curfunc, *flist;
	const char *name, *mainfunc_name, *path_hint, *cptr;
	char *cptr2, *end;
	int symno;
	unsigned sline_offset;
	int isp, istack_size, mapsize, max_mapsize, nsyms, doing_header;
	alloc_pool_t *ap, *tmp_ap;
	bool seen_func, doing_ofile;
#ifdef OS_ULTRIX
	int last_data_symno, have_unused_datasym;
	nlist_t data_nm;
	const char *lastname;
#endif

	doing_ofile = stf != NULL;
	
	if (first_call) {
		build_symset(aout_symset, ofile_symset);
		first_call = FALSE;
	}

	ast = AO_STDATA(st);
	flist = NULL;
	curfunc = NULL;
	rootblock = get_rootblock();
	doing_header = 0;

	
#ifdef OS_SUNOS
	headers = NULL;
#endif
	
#ifdef OS_ULTRIX
	/*  FRAGILE CODE
	 *
	 *  The Ultrix C compiler has a charming habit of correcting
	 *  itself over symbol addresses.  A symbol table frequently
	 *  has an N_DATA symbol with one address, followed soon
	 *  after by an N_STSYM for the same symbol name with a slightly
	 *  different address.  The N_DATA address is the correct one.
	 *
	 *  To cope with this, we remember the symbol number of N_DATA
	 *  symbols, and correct the N_STSYM address if necessary.
	 *
	 *  The compiler also tends to put out N_DATA symbols immediately
	 *  after N_STSYM symbols, but in these cases the addresses
	 *  seem to always be the same.
	 */
	have_unused_datasym = FALSE;
	last_data_symno = 0; /* to satisfy gcc */
#endif

#ifdef AO_ELF
	file_offset = next_file_offset = 0;
#endif

	max_mapsize = 32;
	fmap = (hf_t **) e_malloc(max_mapsize * sizeof(hf_t *));

	isp = 0;
	istack_size = 8;
	istack = (hf_t **) e_malloc(istack_size * sizeof(hf_t *));

	symno = -1;
	path_hint = NULL;
	seen_func = 0; /* to satisfy gcc */
	ap = st->st_apool;
	tmp_ap = alloc_create_pool();
	mainfunc_name = NULL;
	solfil = NULL;
	sline_offset = 0;

	if (doing_ofile) {
#ifdef AO_ELF
		symio = stf->stf_symio;
#else
		panic("doing_ofile set for non-ELF file in ss");
		symio = NULL;	/* to satisfy gcc */
#endif
		
		symset = ofile_symset;
		stf->stf_fnum = 0;
		fmap[0] = (hf_t *) alloc(ap, sizeof(hf_t));
		fmap[0]->hf_stf = stf;
		fmap[0]->hf_id = -1;
		fmap[0]->hf_next = NULL;
		mapsize = 1;
	       
	}
	else {
		symset = aout_symset;
		symio = ast->st_text_symio;
		
		mapsize = 0; /* for lint */
	}
	
	nsyms = get_symio_nsyms(symio);
	
	for (symno = findsym(symio, symno + 1, &nm, symset);;
	     symno = next_unmarked_sym(symio, symno + 1)) {
		
		symno = findsym(symio, symno, &nm, symset);

		if (symno >= nsyms)
			break;
		
		cptr2 = 0;
		switch(nm.n_type) {
			const char *symtab_name;
			language_t lang;
#ifdef AO_ELF
			bool has_debug_syms;
#endif
			
#ifdef AO_ELF
		case N_UNDF:
			file_offset = next_file_offset;
			next_file_offset = file_offset + nm.n_value;

			add_extra_string_offset(symio, symno, file_offset);
			
			break;
#endif

#ifdef N_MAIN
		case N_MAIN:
			mainfunc_name = symstring(symio, symno);
			break;
#endif
			
		case N_SO:
			solfil = NULL;
			sline_offset = 0;
			
#ifdef COFF_SUN386
			/*  The Sun 386i C compiler seems to put out
			 *  spurious N_SO symbols for functions.
			 *  The bad ones all seem to have a non zero
			 *  n_dbx_desc field, so skip these.
			 */
			if (nm.n_dbx_desc != 0)
				break;
#endif /* COFF_SUN386 */
			if (curfunc != NULL) {
				AO_FSYMDATA(curfunc)->fs_symlim = symno;
				curfunc = NULL;
			}
			if (stf != NULL)
				stf->stf_symlim = symno;

			name = symstring(symio, symno);

			/*  4.0 cc puts paths ending in '/' just before
			 *  the source files that follow.
			 */
			if (!name || *name == '\0' ||
			    name[strlen(name) - 1] == '/') {
				path_hint = name;
				break;
			}

			if (stf != NULL) {
				wrapup_stf(stf, fmap, mapsize);
				reverse_stf_funclist(stf);
			}

#ifndef AO_ELF
			if (strcmp(name, "libg.s") == 0) {
				stf = NULL;
				break;
			}
#endif
/*			if (!name[0]) */
/*			  break;*/

		
			if (name && *name == '.' && *(name+1) == '/') /* RGA */
			  name += 2;
			lang = srctype(name);
#ifdef OS_SUNOS_4
			/*  F77 files can be compiled with the f90 compiler,
			 *  so try and work out from the symbol table which
			 *  was used.
			 */
			if (IS_FORTRAN(lang)) {
				const char *s;

				s = symstring(symio, symno + 1);

				if (strncmp(s, "byte:", 5) == 0) {
					lang = LANG_F90;
				}
				else if (strncmp(s, "integer*2:", 10) == 0) {
					lang = LANG_F77;
				}
			}
#endif
			
			stf = make_stf(ap, name, st, symno, lang, nm.n_value);
#ifdef AO_ELF
			stf->stf_symio = symio;
#endif
			st->st_sfiles = make_fil(stf, rootblock, path_hint,
						 st->st_sfiles);
#ifndef OS_LINUX
			if (!path_hint && lang == LANG_CC)
			  /* RGA g++ puts out empty names */
			  /*  so hide such files  */
			  st->st_sfiles->fi_flags |= FI_HIDE_ENTRY;
#endif

			doing_header = 0;
			stf->stf_fil = st->st_sfiles;

			if (isp != 0)
			{
			  errf_ofunc_t oldf;

			  /* panic("unmatched N_BINCL");*/

			  oldf = errf_set_ofunc(display_message);
			  errf("unmatched N_BINCL symbol in %s (%s)",
			       path, name);
			  errf_set_ofunc(oldf);
			  isp = 0;
			}

			stf->stf_fnum = 0;
			fmap[0] = (hf_t *) alloc(ap, sizeof(hf_t));
			fmap[0]->hf_stf = stf;
			fmap[0]->hf_id = -1;
			fmap[0]->hf_next = NULL;
			mapsize = 1;
			path_hint = NULL;
			seen_func = FALSE;
			symset[N_SLINE] = TRUE;
#ifdef OS_ULTRIX
			have_unused_datasym = FALSE;
#endif
			break;

		case N_SOL:
			/*  Names are relative to the last N_SO, so we need
			 *  one.
			 */
			if (stf != NULL) {
			  name = symstring(symio, symno);
			  /* RGA added stf for C++ */
			  if (name && *name == '.' && *(name+1) == '/') /* RGA */
			    name += 2;
			  solfil = add_sol_fil(stf, ap, st->st_sfiles,
					       stf->stf_fil, name); 

			  if (solfil == stf->stf_fil)
			    solfil = NULL;

			  if (curfunc != NULL && solfil != NULL &&
			      (solfil->fi_funclist == NULL ||
			       solfil->fi_funclist->fl_func != curfunc)) {
			    add_to_fil_funclist(ap,
						solfil, curfunc);
			  }
				    
			  /* RGA Insight puts out these - ignore them */
			  if (strcmp(name, "NoName") == 0)
			    break;
			  
			  /*  Make ups work better with cfront (c++). */
			  if (solfil && name != NULL &&
			      !strstr(name, ".h") &&
			      !strstr(name, ".H") &&
			      !strstr(name, ".idl") &&
			      !strstr(name, ".icon"))
			  {
			    if (srctype(solfil->fi_name) == LANG_UNKNOWN)
			      solfil->fi_flags |= FI_RENAMED_OTHER;
			    if (!(st->st_sfiles->fi_flags & FI_RENAMED_OTHER) &&
				(!strrchr(name, '/') ||
				 (st->st_sfiles->fi_language !=
				  srctype(stf->stf_name) &&
				  strrchr(stf->stf_name, '.'))))
			      /* ignore bogus files with no extension */
			    {
			      if (st->st_sfiles->fi_language == LANG_C &&
				  srctype(solfil->fi_name) == LANG_CC)
			      { /* use first file renamed by cfront */
				st->st_sfiles->fi_flags |= FI_RENAMED_OTHER;
				st->st_sfiles->fi_name = solfil->fi_name;
				st->st_sfiles->fi_name_only =
				  solfil->fi_name_only;
				st->st_sfiles->fi_language = LANG_CC;
			      }
			      else
			      {
				st->st_sfiles->fi_name = solfil->fi_name;
				st->st_sfiles->fi_name_only =
				  solfil->fi_name_only;
			      }
			      doing_header = 0;
			    }
			    else
			      doing_header = 0;
			  }
			  else
			  {
			    if (solfil != NULL)
			      solfil->fi_flags |= FI_HEADER;
			    doing_header = 1;
			  }
			  if (solfil && name != NULL &&
			      strstr(name, ".hh") || strstr(name, ".H"))
			    solfil->fi_language = LANG_CC;
			}
			break;

#ifdef AO_ELF
		case N_OPT:
			/*  We shouldn't get N_OPT before the first N_SO,
			 *  but lets not core dump if we do.
			 */
			if (stf == NULL)
				break;
			
			cptr = symstring(symio, symno);

			if (cptr== NULL)
			  break;
			
			elf_handle_optsym(ap, cptr, stf->stf_language,
					  &stf->stf_global_prefix,
					  &stf->stf_compiler_type,
					  &has_debug_syms);
			stf->stf_obj_mtime = nm.n_value;
			break;
#endif
			
		case N_SLINE:
			if (!seen_func && stf != NULL)
				stf->stf_flags |= STF_LNOS_PRECEDE_FUNCS;
			symset[N_SLINE] = FALSE;
			break;

#ifdef N_XLINE
		case N_XLINE:
			sline_offset = nm.n_desc << 16;
			break;
#endif
		
#ifdef OS_SUNOS
		case N_BINCL:
		case N_EXCL:
			if (stf == NULL)
				panic("header outside source file in ss");
			
			if (mapsize == max_mapsize) {
				max_mapsize *= 2;
				fmap = (hf_t **) e_realloc((char *)fmap,
						max_mapsize * sizeof(hf_t *));
			}
			
			if (nm.n_type == N_EXCL) {
				hf = lookup_hf(headers, (int)nm.n_value);
				fmap[mapsize++] = hf;
				break;
			}
			
			if (isp == istack_size) {
				istack_size *= 2;
				istack = (hf_t **) e_realloc((char *)istack,
						istack_size * sizeof(hf_t *));
			}
			
			istack[isp] = (hf_t *) alloc(ap, sizeof(hf_t));
			
			istack[isp]->hf_next = headers;
			headers = istack[isp++];

			name = alloc_strdup(ap, symstring(symio, symno));
			headers->hf_stf = make_stf(ap, name, st, symno,
						   st->st_sfiles->fi_language,
						   nm.n_value);

			if (nm.n_type == N_EXCL) {
				headers->hf_stf->stf_fil = NULL;
			}
			else {
				headers->hf_stf->stf_fil =
						make_fil(headers->hf_stf,
							 (block_t *)NULL,
							 (const char *)NULL,
							 (fil_t *)NULL);
			}

			doing_header = 0;
			headers->hf_stf->stf_fnum = mapsize;
			headers->hf_id = nm.n_value;
			fmap[mapsize++] = headers;
			break;

		case N_EINCL:
			if (isp == 0)
				panic("unmatched N_EINCL");
			
			--isp;
			istack[isp]->hf_stf->stf_symlim = symno;
			
			if (!doing_ofile)
				reverse_stf_funclist(stf);
			
			break;
#endif /* OS_SUNOS */
		case N_STSYM:
		case N_GSYM:
		case N_LCSYM:
			/*  We shouldn't get here before stf exists,
			 *  but lets not core dump if we do.
			 */
			if (stf == NULL)
				break;
			
			if (IS_FORTRAN(stf->stf_language) &&
			    symno - stf->stf_symno <= N_FIXED_FORTRAN_TYPES)
				break;

			cptr = symstring(symio, symno);

			/* RGA have seen this with SC4 with templates */
			while(*cptr==';')
			  cptr++;

			/* RGA have seen this with SunOS/centerline */
			if(*cptr == 0 || *cptr  && *cptr==' ')
			  break;

			/* RGA have seen this with SC4.2 with templates */
			if (*cptr != ':' && *cptr != '?' && !isalpha(*cptr) &&
			    *cptr != '_' &&
			    *cptr != '$' && *cptr != '.')
			  break;
			  
			symrec.symio = symio;
			symrec.symno = symno;
		/* RGA SC4.2 folds lines at 400. Can't find a define for */
		/* this, so using an ugly number at present... */
		      {
			size_t len;
			
			len = strlen(cptr);
			end = (char *)cptr + len - 1;
			if (len == 401 && *end == '\\')
			{
			  cptr2 = e_malloc(len + 1);
			  memcpy((void *)cptr2, cptr, len + 1);
			  cptr = cptr2;
			}
			if (len == 401)
			  while (*end == '\\')
			  {
			    char *s1;
			    size_t new_len;
		      
			    s1 = (char *)get_cont_symstring(&symrec);
			    mark_sym(symio, symrec.symno);
			    new_len = strlen(s1);
			    cptr2 = (char *)e_realloc((void *)cptr2, len + new_len);
			    cptr = cptr2;
			    memcpy((void *)(cptr2 + len - 1), s1, new_len);
			    cptr2[len + new_len - 1] = 0;
			    len += new_len - 1;
			    end = (char *)cptr + len - 1;
			    if (new_len != 401)
			      break;
			  }
		      }
			name = parse_name(&symrec, &cptr, tmp_ap, 1,
					  stf->stf_language == LANG_CC &&
					  stf->stf_compiler_type);

#ifdef AO_ELF
			if (doing_ofile) {
				if (symtab_name_to_sn(stf->stf_snlist,
						      name, &sn)) {
					sn->sn_symno = symno;
					break;
				}

				/*  The Sun cc (SPARCompiler 3.0) leaves a
				 *  few symbols out of the .stab.index
				 *  section in the linked file, so don't
				 *  complain about extra symbols - just
				 *  silently add them.  These symbols always
				 *  seem to be local static variables in an
				 *  inner block scope, but I can't reproduce
				 *  this with a small test case.
				 */
			}

			/*  SPARCompiler 3.0 sometimes emit duplicate
			 *  symbols, of which the first seems to be bogus.
			 *  For example, there are two entries for
			 *  file_string in the symbol table for
			 *  test/C/core/f.c.
			 *
			 *  Just ignore the first one.
			 */
			if (stf->stf_snlist != NULL &&
			    strcmp(stf->stf_snlist->sn_symtab_name, name) == 0){
				stf->stf_snlist->sn_symno = symno;
				break;
			}

			symtab_name = name;
			name = elf_name_from_symtab_name
			  (stf->stf_global_prefix, symtab_name);
#else
			symtab_name = name;
#endif

			sn = push_symname(&symrec, ap, &stf->stf_snlist,
					  symtab_name, name, symno,
					  stf->stf_language == LANG_CC &&
					  stf->stf_compiler_type);
			
#ifdef OS_ULTRIX
			sn->sn_addr = 0;
			
			if (have_unused_datasym) {
				lastname = symstring(symio, last_data_symno);
				
				if (*lastname == '_' &&
				    strcmp(lastname + 1, sn->sn_name) == 0) {
					getsym(symio, last_data_symno,
					       &data_nm);
					sn->sn_addr = data_nm.n_value;
				}
				
				have_unused_datasym = FALSE;
						
			}
#endif
			
			/*  The epcf90 emits continuation symbol table lines
			 *  with N_GSYM as the symbol, so skip them.
			 */
			for (;;) {
				size_t len;

				len = strlen(name);
				if (len == 0 || name[len - 1] != '\\')
					break;

				++symno;
				getsym(symio, symno, &nm);
				name = symstring(symio, symno);
			}
			
			break;
#ifdef OS_ULTRIX
		case N_DATA:
			last_data_symno = symno;
			have_unused_datasym = TRUE;
			break;
#endif
		case N_FUN:
			name = symstring(symio, symno);

			/*  Some compilers (e.g. gcc) put read only strings
			 *  in the text segment, and generate N_FUN symbols
			 *  for them.  We're not interested in these here.
			 */
			if ((cptr = strchr(name, ':')) == NULL ||
			    (cptr[1] != 'F' && cptr[1] != 'f'))
				break;

			if (curfunc != NULL) {
				AO_FSYMDATA(curfunc)->fs_symlim = symno;
				curfunc = NULL;
			}

			seen_func = TRUE;

#ifdef AO_ELF
			if (doing_ofile) {
/*				curfunc = note_ofile_function(ap, tmp_ap, stf,*/
				curfunc = note_ofile_function(ap, ap, stf,
							      path, solfil,
							      sline_offset,
							      name, symno);
				break;
			}
#endif
/*#ifdef OS_SUNOS
			add_function_to_symtab
			  (st, &flist, name,
			   (st->st_sfiles->fi_language == LANG_CC && headers &&
			    headers->hf_stf) ? headers->hf_stf->stf_fil:
			   st->st_sfiles, solfil, cptr[1] == 'f', FALSE,
			   symno, nm.n_value);
#else*/
			add_function_to_symtab(st, &flist, name,
					       st->st_sfiles, solfil,
					       cptr[1] == 'f', FALSE,
					       symno, nm.n_value);
/*#endif*/
			AO_FSYMDATA(flist)->fs_initial_sline_offset =
								sline_offset;

			/* RGA */
			if (doing_header && flist->fu_language == LANG_CC)
			  flist->fu_flags |= FU_NOTHEADER;

			curfunc = flist;
			
			if (is_fortran_void(cptr))
				curfunc->fu_type = ci_code_to_type(TY_VOID);
			
			break;
		
		case N_TEXT:
		case N_TEXT | N_EXT:
			name = symstring(symio, symno);
			
			/*  Some compilers put N_TEXT symbols out with object
			 *  file names, often with the same addresses as real
			 *  functions.  We don't want these.
			 *
			 *  We also don't want N_TEXT symbols which immediately
			 *  follow an N_FUN symbol with the same address.
			 */
			if (nm.n_type == N_TEXT) {
				if (*name != '_') {
					/*  A .o file symbol is definitely
					 *  the end of the current function,
					 *  if any.
					 */
					if (curfunc != NULL) {
						AO_FSYMDATA(curfunc)->fs_symlim = symno;
						curfunc = NULL;
					}
					break;
				}
				if (curfunc != NULL && curfunc->fu_addr == nm.n_value)
					break;
			}
			
			add_function_to_symtab(st, &flist, name,
					       (fil_t *)NULL, (fil_t *)NULL,
					       (nm.n_type & N_EXT) == 0, TRUE, 
					       symno, nm.n_value);
			
			/* RGA */
			if (doing_header && flist->fu_language == LANG_CC)
			  flist->fu_flags |= FU_NOTHEADER;

			/*  You'd expect that we'd close down the current
			 *  function here, but some C compilers put out
			 *  N_TEXT symbols in the middle of the N_SLINE
			 *  symbols for a function.
			 *
			 *  Thus we leave curfunc alone, and rely on an N_SO or
			 *  a .o file N_TEXT to terminate the current
			 *  function.
			 */
			
			break;
#ifdef ARCH_CLIPPER
		case N_DATA:
		      {
			/*  The Clipper C compiler puts out an N_LCSYM symbol
			 *  with the wrong address for static globals, then
			 *  later puts out an N_DATA with the right address.
			 *  This we keep a list of N_DATA symbols for each
			 *  file so we can check the addresses later.
			 *
			 *  Note that we can't just stick the symbol in the
			 *  global addresses list, as we may have static
			 *  globals with the same name in different files.
			 */
			char *class_name, *name_start;
			int class_len, name_len;

			name = symstring(symio, symno);
			if (*name != '_')
				break;
			name_start = name+1;
			if (get_mangled_name_info
			    (0, name+1, &name_start, &name_len,
			     &class_name, &class_len))
			  name_start[name_len] = 0;
			else
			  name_start = name+1;

			insert_global_addr(ap, &stf->stf_addrlist,
					   name_start, (taddr_t)nm.n_value);
		      }
			break;
#endif
		case N_BSS | N_EXT:
		case N_DATA | N_EXT:
		      {
			char *class_name, *name_start;
			int class_len, name_len;

			name = symstring(symio, symno);
#ifndef COFF_SUN386
			if (*name != '_')
				break;
			++name;
#endif
			name_start = (char *)name;
			if (get_mangled_name_info
			    (0, (char *)name, &name_start, &name_len,
			     &class_name, &class_len))
			  name_start[name_len] = 0;
			else
			  name_start = (char *)name;

			insert_global_addr(ap, &st->st_addrlist, name_start,
					   ast->st_base_address +
						          (taddr_t)nm.n_value);
		      }
			break;
		case N_BCOMM:
			symno = push_common_block(st, stf, curfunc,
						  symio, symno);
			break;
		default:
			panic("unexpected symbol in init_syms");
		}
		if (cptr2)
		  free((void *)cptr2);
	}

	if (curfunc != NULL) {
		AO_FSYMDATA(curfunc)->fs_symlim = symno;
		curfunc = NULL;
	}
	
	if (stf != NULL) {
		wrapup_stf(stf, fmap, mapsize);

		if (!doing_ofile)
			reverse_stf_funclist(stf);
		
		stf->stf_symlim = symno;
	}

	free((char *)fmap);
	free((char *)istack);
	alloc_free_pool(tmp_ap);

	if (!doing_ofile)
		*p_flist = flist;

	if (p_mainfunc_name != NULL)
		*p_mainfunc_name = mainfunc_name;
}

snlist_t *
push_symname(sr, ap, p_snlist, symtab_name, name, symno, compiler)
Symrec *sr;
alloc_pool_t *ap;
snlist_t **p_snlist;
const char *symtab_name, *name;
int symno;
Compiler_type compiler;
{
	snlist_t *sn;
	const char *cptr;
	
	sn = (snlist_t *)alloc(ap, sizeof(snlist_t));
	sn->sn_symno = symno;
	sn->sn_symtab_name = alloc_strdup(ap, symtab_name);
	cptr = symtab_name;
	/* fix for SC4 templates - only demangle when possible */
	if (*cptr != ':' && *cptr != '?' && !isalpha(*cptr) && *cptr != '_' &&
	    *cptr != '$' && *cptr != '.')
	  sn->sn_demangled_symtab_name = cptr;
	else
	  sn->sn_demangled_symtab_name = parse_name(sr, &cptr,
						    ap, 1, compiler);
	sn->sn_name = sn->sn_symtab_name + (name - symtab_name);
	sn->sn_next = *p_snlist;
	*p_snlist = sn;

	return sn;

}
#endif /* !ST_TE */

#ifdef AO_ELF
static func_t *
note_ofile_function(ap, tmp_ap, stf, path, solfil, sline_offset, namestr, symno)
alloc_pool_t *ap, *tmp_ap;
stf_t *stf;
const char *path;
fil_t *solfil;
unsigned sline_offset;
const char *namestr;
int symno;
{
	func_t *f = NULL;
	fil_t *fil;
	const char *modname, *fname;
	int modsepc;

	fil = stf->stf_fil;
	modsepc = (fil->fi_language == LANG_F90) ? '$' : '\0';
			
	parse_fname(tmp_ap, namestr, modsepc, FALSE, &modname, &fname);

	if (fil->fi_language == LANG_CC)
	{
				/* RGA - loop for overloaded functions */
	  funclist_t *fl;

	  for (fl = fil->fi_funclist; fl != NULL; fl = fl->fl_next)
	  {
	    if (strcmp(fl->fl_func->fu_name, fname) == 0)
	    {
	      fsyminfo_t *fs;
	      
	      f = fl->fl_func;
	      fs = AO_FSYMDATA(f);
	      if (fs->fs_symno == 0)
	      {
		fs->fs_symno = symno;
		fs->fs_initial_lno_fil = (solfil != NULL) ? solfil : fil;
		fs->fs_initial_sline_offset = sline_offset;
		
		if (solfil != NULL)
		  add_to_fil_funclist(ap, solfil, f);
		break;
	      }
	    }
	  }
	}
	else
	{
	  f = name_and_fil_to_func(fname, fil, FALSE);
	  
	  if (f == NULL) {
	    errf("Unexpected function `%s' (symbol #%d) found in %s - %s",
		 fname, symno, path, "ignored");
	  }
	  else {
	    fsyminfo_t *fs;
	    
	    fs = AO_FSYMDATA(f);
	    fs->fs_symno = symno;
	    fs->fs_initial_lno_fil = (solfil != NULL) ? solfil : fil;
	    fs->fs_initial_sline_offset = sline_offset;
	    
	    if (solfil != NULL)
	      add_to_fil_funclist(ap, solfil, f);
	  }
	}
	return f;
}
#endif /* AO_ELF */

static void
add_to_fil_funclist(ap, fil, f)
alloc_pool_t *ap;
fil_t *fil;
func_t *f;
{
	funclist_t *fl;
		
	fl = (funclist_t *)alloc(ap, sizeof(funclist_t));
	fl->fl_func = f;
	fl->fl_next = fil->fi_funclist;
	fil->fi_funclist = fl;
}
	
void
add_function_to_symtab(st, p_flist, namestr, fil, solfil,
		       is_static, is_textsym, symno, addr)
symtab_t *st;
func_t **p_flist;
const char *namestr;
fil_t *fil, *solfil;
bool is_static, is_textsym;
int symno;
taddr_t addr;
{
	fsyminfo_t *fs;
	func_t *f;
	const char *modname, *fname;
	int modsepc;
		
	modsepc = (fil != NULL && fil->fi_language == LANG_F90) ? '$' : '\0';
			
	parse_fname(st->st_apool, namestr, modsepc,
		    is_textsym && !is_static, &modname, &fname);

	fs = make_fsyminfo(st->st_apool, symno);

	/*  I put this in as a workaround for a bug (with a fixme
	 *  comment.  Now I can't remember why it was needed.
	 *
	 *  TODO: find out why it was needed, and maybe do something
	 *        better.
	 */
	fs->fs_symlim = 1;
	
	fs->fs_initial_lno_fil = (solfil != NULL) ? solfil : fil;

	if (fil && fil->fi_language == LANG_CC && solfil &&
	    ao_compiler(fil, FALSE, CT_UNKNOWN) != CT_GNU_CC)
	{
	  f = ci_make_func(st->st_apool, fname, addr, st, solfil, *p_flist);
	  if (!strcmp(f->fu_demangled_name, "__STATIC_CONSTRUCTOR")) 
	  {			 /* static ctor */
	    f->fu_fil = fil;
	    f->fu_language = (fil != NULL) ? fil->fi_language : LANG_UNKNOWN;
	  }
	}
	else
	  f = ci_make_func(st->st_apool, fname, addr, st, fil, *p_flist);
	f->fu_symdata = (char *)fs;
	f->fu_predata = NULL;

	if (fil == NULL) {
		f->fu_flags |= FU_NOSYM | FU_DONE_BLOCKS | FU_DONE_LNOS;
	}
	else {
		add_to_fil_funclist(st->st_apool, fil, f);

		if (solfil != NULL)
			add_to_fil_funclist(st->st_apool, solfil, f);
	}

	if (is_static)
		f->fu_flags |= FU_STATIC;

	if (modname != NULL)
		add_module_function(st, modname, f);

	*p_flist = f;
}
			
			
static bool
is_fortran_void(cptr)
const char *cptr;
{
	/*  BUG: should look at the type properly.
	 *       There is no guarantee that type 11 is always void.
	 */
	return strcmp(cptr, ":F11") == 0;
}

/*  Adjust the addresses of all the global variables associated
 *  with source files in flist.  Called when a shared library
 *  mapping address changes across runs of the target.
 */
static void
adjust_fil_vars_addr_base(flist, delta)
fil_t *flist;
long delta;
{
	fil_t *fil;
	var_t *v;

	for (fil = flist; fil != NULL; fil = fil->fi_next) {
		if (AO_FIDATA(fil) == NULL)
			continue;
		
		AO_FIDATA(fil)->stf_addr += delta;
		
		if (fil->fi_flags & FI_DONE_VARS) {
			for (v = fil->fi_block->bl_vars;
			     v != NULL;
			     v = v->va_next)
				v->va_addr += delta;
		}
	}
}

/*  Deal with a change in the text offset of a symbol table.  This may
 *  be necessary when re-running the target as shared libraries may be
 *  mapped at different addresses.  It's also necessary when we have
 *  preloaded symbol tables with a nominal offset of zero.
 *
 *  We adjust the following:
 *
 *	function and line number addresses
 *	symbol table address to text file offset
 *	addresses of global variables
 *
 *  We don't change breakpoint addresses here - we do that by removing
 *  and recreating all breakpoints just after starting the target.
 */
void
change_base_address(st, new_addr)
symtab_t *st;
taddr_t new_addr;
{
	long delta;
	ao_stdata_t *ast;

	ast = AO_STDATA(st);
	delta = new_addr - ast->st_base_address;

	if (delta != 0) {
		adjust_addrlist_addr_offset(st->st_addrlist, delta);
		ast->st_addr_to_fpos_offset += delta;
		adjust_functab_text_addr_base(st->st_functab,
					      st->st_funclist, delta);
		adjust_fil_vars_addr_base(st->st_sfiles, delta);
		ast->st_base_address = new_addr;
	}
}


#endif /* AO_TARGET */


int
get_mangled_name_info(var, name, name_start, name_len,
		      class_name, class_len)
     int var;
     char *name;
     char **name_start;
     int *name_len;
     char **class_name;
     int *class_len;
{
  int ret = 0;
  char *t, *n, *start, *end;
  char num_buff[10];

  *class_name = NULL;
  *class_len = 0;
  *name_len = 0;
  start = name;

  if (demangling_enabled(0, 0))
  {
    if (var)
    {
      start = *name_start;
      t = strstr(start, "__");
      if (t && isdigit(*(t + 2)) && t == start)
      {
	ret = 1;
	t += 2;		/* skip "__" */
	while(isdigit(*t))
	  t++;
      }
      else
	t = start;
      *name_start = start = t;
    }
    else
      start = name;
    end = start + strlen(start);
    for (t = start; t && t < end;)
    {
      t = strstr(t, "__");
      if (t)
	if (isdigit(*(t + 2)) || (isalpha(*(t + 2)) && isupper(*(t + 2))))
	{
	  for (*name_len = 0, n = start; n < t; (*name_len)++, n++);
	  if (t && isdigit(*(t + 2)))
	  {
	    if (*(t + 2) == '0')
	    {
	      ret = 2;		/* SC3 mangling */
	      if (isupper(*(t + 3)))
	      {
		*name_start = t + 5;
		if ((int)(*(t + 4) - 'A') > 25)
		  *name_len = (int)(*(t + 4) - 'a') + 26;
		else
		  *name_len = (int)(*(t + 4) - 'A');
		t = *name_start + *name_len;
	      }
	      else
	      {
		if (*(t + 3) == 'd' && isdigit(*(t + 4)))
		{
		  while (!isupper(*(t + 4)) && t + 5 < end) t++;
		  if ((int)(*(t + 4) - 'A') > 0)
		    if ((int)(*(t + 4) - 'A') > 25)
		      t += (int)(*(t + 4) - 'a') + 26 + 1;
		    else
		      t += (int)(*(t + 4) - 'A') + 1;
		}
		if ((int)(*(t + 4) - 'A') > 0)
		{
		  *class_name = t + 5;
		  if ((int)(*(t + 4) - 'A') > 25)
		    *class_len = (int)(*(t + 4) - 'a') + 26;
		  else
		    *class_len = (int)(*(t + 4) - 'A');
		  if (*class_len > end - *class_name)
		    *class_len = end - *class_name;
		  t = *class_name + *class_len;
		  while (isdigit(*t)) /* templates...just skip */
		    t++;
		  *name_start = t + 1;
		  if (*name_start > end)
		    *name_start = start;
		  if ((int)(*(t) - 'A') > 25)
		    *name_len = (int)(*(t) - 'a') + 26;
		  else
		    *name_len = (int)(*(t) - 'A');
		  if (*name_len > end - *name_start)
		    *name_len = end - *name_start;
		  if (*name_len < 0)
		    t = *name_start;
		  else
		  t = *name_start + *name_len;
		}
		else
		{
		  ret = 0;
		  break;
		}
	      }
	    }
	    else
	    {
	      for (*name_len = 0, n = start; n < t; (*name_len)++, n++);
	      if (isdigit(*(t + 2)))
	      {
		t += 2;		/* skip "__" */
		for (n = num_buff; (isdigit(*t)); t++, n++)
		  *n = *t;
		*(n++) = 0;
		*class_name = t;
		*class_len = atoi(num_buff);
		if (t + *class_len > end)
		{
		  *class_name = NULL;
		  *class_len = 0;
		  *name_len = end - *name_start;
		}
	      }
	      ret = 1;
	      break;
	    }
	  }
	  if (!ret)
	    ret = 1;
	  break;
	}
	else
	  ++t;
    }
    if (!(*name_len))
      for(*name_len = 0, t = start; isalnum(*t) || *t == '_';
	  (*name_len)++, t++); 
    if (*name_len < 0)
      *name_len = 0;
  }
  return (ret);
}
#define CONSTRUCTOR 0
#define DESTRUCTOR 1
#define OPERATOR 2

static struct 
{
  char *mangle_str;
  char *demangle_str;
  int demangle_str_len;
  int type;
} Mangle_buff[] =
{
  "ct", "", 0, CONSTRUCTOR,
  "dt", "", 0, DESTRUCTOR,
  "op", "operator", 8, OPERATOR,
  "as", "operator=", 9, OPERATOR,
  "eq", "operator==", 10, OPERATOR,
  "pl", "operator+", 9, OPERATOR,
  "mi", "operator-", 9, OPERATOR,
  "ml", "operator*", 9, OPERATOR,
  "dv", "operator/", 9, OPERATOR,
  "md", "operator%", 9, OPERATOR,
  "er", "operator^", 9, OPERATOR,
  "ad", "operator&", 9, OPERATOR,
  "or", "operator|", 9, OPERATOR,
  "co", "operator~", 9, OPERATOR,
  "nt", "operator!", 9, OPERATOR,
  "gt", "operator>", 9, OPERATOR,
  "lt", "operator<", 9, OPERATOR,
  "apl", "operator+=", 10, OPERATOR,
  "ami", "operator-=", 10, OPERATOR,
  "amu", "operator*=", 10, OPERATOR,
  "aml", "operator*=", 10, OPERATOR,
  "adv", "operator/=", 10, OPERATOR,
  "gdv", "operator/=", 10, OPERATOR,
  "amd", "operator%=", 10, OPERATOR,
  "aer", "operator^=", 10, OPERATOR,
  "aad", "operator&=", 10, OPERATOR,
  "gad", "operator&=", 10, OPERATOR,
  "aor", "operator|=", 10, OPERATOR,
  "ls", "operator<<", 10, OPERATOR,
  "rs", "operator>>", 10, OPERATOR,
  "ars", "operator>>=", 11, OPERATOR,
  "als", "operator<<=", 11, OPERATOR,
  "ne", "operator!=", 10, OPERATOR,
  "le", "operator<=", 10, OPERATOR,
  "ge", "operator>=", 10, OPERATOR,
  "rm", "operator->*", 11, OPERATOR,
  "cm", "operator,", 9, OPERATOR,
  "mm", "operator--", 10, OPERATOR,
  "pp", "operator++", 10, OPERATOR,
  "oo", "operator||", 10, OPERATOR,
  "aa", "operator&&", 10, OPERATOR,
  "vc", "operator[]", 10, OPERATOR,
  "cl", "operator()", 10, OPERATOR,
  "rf", "operator->", 10, OPERATOR,
  "nw", "new", 3, OPERATOR,
  "dl", "delete", 3, OPERATOR,
};
#define NMANGLE_STRS (sizeof Mangle_buff / sizeof *Mangle_buff)

void
demangle_name_2(name, len, alloc_id, ptr, func, fil)
     char *name;
     int len;
     alloc_pool_t *alloc_id;
     char **ptr;
     int func;
     fil_t *fil;
{
  demangle_name(name, len, alloc_id, ptr, func,
		ao_compiler(fil, FALSE, CT_UNKNOWN));
}

/* RGA for Solaris, you can use the cplus_demangle() function in libC
   by changing the #ifdefs below. This will demangle templates and
   the like for SC4 and above. You need to extract dem.o cafe_dem.o
   from libC.a. E.g.

   ar x /opt/SUNWspro/SC4.0/lib/libC.a dem.o cafe_dem.o

   and add these files to the link, such as the AOOBJS list in
   ups/Makefile.
   */

void
demangle_name(name, len, alloc_id, ptr, func, compiler)
     char *name;
     int len;
     alloc_pool_t *alloc_id;
     char **ptr;
     int func;
     Compiler_type compiler;
{
  char *class_name, *name_start, tmp = 0, tmp2;
  int class_len, name_len, mangle, index, gnu_destructor = 0;
  
  name_start = name;
  tmp2 = *(name+len);
  *(name+len)= 0;
  *ptr = NULL;

  if (*(name+1) == '$' && *name == '_' && *(name+2) == '_')
  {
    gnu_destructor = 1;
    *(name+1) = '_';
  }
  if (func == 0 
#if 1
      && !(compiler == CT_CC || compiler == CT_CLCC)
#endif
      && demangling_enabled(0, 0)
      )
  {
    char *result;
#if 0
    char buff[512];

    if (compiler == CT_CC)
    {
      cplus_demangle (name, buff, 512);
      result = buff;
    }
    else
#endif
      result = gnu_cplus_demangle (name, DMGL_ANSI);
    *(name+len) = tmp2;
    if (result == NULL)
    {
      *(name+len) = tmp2;
      *ptr = alloc(alloc_id, len + 1);
      memcpy(*ptr, name, len);
      (*ptr)[len] = '\0';
    }
    else
    {
      len = strlen(result);
      *ptr = alloc(alloc_id, len + 1);
      memcpy(*ptr, result, len);
      (*ptr)[len] = '\0';
      free(result);
    }
    return;
  }

  mangle = get_mangled_name_info(func==1, name, &name_start, &name_len,
				 &class_name, &class_len);
  if (gnu_destructor)
    *(name+1) = '$';
  if (mangle)
  {
    if (func == 1)		/* variable */
    {
      *ptr = alloc(alloc_id, name_len + 1);
      (void) strncpy(*ptr, name_start, name_len);
      (*ptr)[name_len] = '\0';
    }
    else
    if (mangle == 2 && !class_len) /* SC 3 mangling*/
    {
      if (!strncmp(name_start, "_STCON_", name_len))
      {
	*ptr = alloc(alloc_id, strlen("__STATIC_CONSTRUCTOR") + 1);
	(void) strcpy(*ptr, "__STATIC_CONSTRUCTOR");
      }
      else
	if (!strncmp(name_start, "_STDES_", name_len))
	{
	  *ptr = alloc(alloc_id, strlen("__STATIC_DESTRUCTOR") + 1);
	  (void) strcpy(*ptr, "__STATIC_DESTRUCTOR");
	}
	else
	{
	  *ptr = alloc(alloc_id, name_len + 1);
	  (void) strncpy(*ptr, name_start, name_len);
	  (*ptr)[name_len] = '\0';
	}
    }
    else
    {				/* function name */
      if (name_len <= len)
	tmp = *(name+name_len);
      if (mangle != 2)
	*(name+name_len) = 0;
      if (gnu_destructor)
      {
	*ptr = alloc(alloc_id, 2*class_len + 4);
	memcpy(*ptr, class_name, class_len);
	memcpy(*ptr+class_len, "::", 2);
	class_len += 2;
	*(*ptr+class_len) = '~';
	memcpy(*ptr+class_len+1, *ptr, class_len-2);
	(*ptr)[class_len+class_len-1] = 0;
      }
      else
	if (class_name && !strncmp(name, "__", 2))
	{
	  if (mangle == 1 && isdigit(*(name + 2)))
	  {			/* g++ constructor */
	    *ptr = alloc(alloc_id, 2*class_len + 3);
	    memcpy(*ptr, class_name, class_len);
	    memcpy(*ptr+class_len, "::", 2);
	    class_len += 2;
	    memcpy(*ptr+class_len, *ptr, class_len - 2);
	    (*ptr)[class_len+class_len-2] = 0;
	  }
	  else
	    for (index = 0; !*ptr && index < NMANGLE_STRS; index++)
	    {
	      if ((mangle == 2 && !strncmp(class_name + class_len,
					   Mangle_buff[index].mangle_str, 2)) ||
		  !strcmp(name + 2, Mangle_buff[index].mangle_str))
	      {
		switch (Mangle_buff[index].type)
		{
		case CONSTRUCTOR:
		  *ptr = alloc(alloc_id, 2*class_len + 3);
		  memcpy(*ptr, class_name, class_len);
		  memcpy(*ptr+class_len, "::", 2);
		  class_len += 2;
		  memcpy(*ptr+class_len, *ptr, class_len - 2);
		  (*ptr)[class_len+class_len-2] = 0;
		  break;
		case DESTRUCTOR:
		  *ptr = alloc(alloc_id, 2*class_len + 4);
		  memcpy(*ptr, class_name, class_len);
		  memcpy(*ptr+class_len, "::", 2);
		  class_len += 2;
		  *(*ptr+class_len) = '~';
		  memcpy(*ptr+class_len+1, *ptr, class_len-2);
		  (*ptr)[class_len+class_len-1] = 0;
		  break;
		case OPERATOR:
		  *ptr = alloc(alloc_id, class_len +
			       Mangle_buff[index].demangle_str_len + 3);
		  memcpy(*ptr, class_name, class_len);
		  memcpy(*ptr+class_len, "::", 2);
		  class_len += 2;
		  memcpy(*ptr+class_len,
			 Mangle_buff[index].demangle_str,
			 Mangle_buff[index].demangle_str_len);
		  (*ptr)[class_len+Mangle_buff[index].demangle_str_len] = 0;
		  break;
		}
	      }
	    }
	}
      if (!*ptr)
      {
	if (mangle == 2)		/* SC 3 mangling*/
	{
	  *ptr = alloc(alloc_id, class_len + name_len + 3);
	  memcpy(*ptr, class_name, class_len);
	  memcpy(*ptr + class_len, "::", 2);
	  class_len += 2;
	  memcpy(*ptr + class_len, name_start, name_len);
	  (*ptr)[class_len+name_len] = 0;
	}
	else
	{
	  if (class_name)
	  {
	    *ptr = alloc(alloc_id, class_len + name_len + 3);
	    memcpy(*ptr, class_name, class_len);
	    memcpy(*ptr+class_len, "::", 2);
	    class_len += 2;
	  }
	  else
	    *ptr = alloc(alloc_id, class_len + name_len + 1);
	  if (name_len <= len)
	    name[name_len] = tmp;
	  memcpy(*ptr+class_len, name, name_len);
	  (*ptr)[class_len+name_len] = 0;
	}
      }
      if (name_len <= len)
	*(name+name_len) = tmp;
    }
    *(name+len) = tmp2;
  }
  else
  {
    if (!strncmp(name, "__sti__", 7))
    {
      *ptr = alloc(alloc_id, strlen("__STATIC_CONSTRUCTOR") + 1);
      (void) strcpy(*ptr, "__STATIC_CONSTRUCTOR");
    }
    else
      if (!strncmp(name, "__std__", 7))
      {
	*ptr = alloc(alloc_id, strlen("__STATIC_DESTRUCTOR") + 1);
	(void) strcpy(*ptr, "__STATIC_DESTRUCTOR");
      }
      else
      {
	*(name+len) = tmp2;
	*ptr = alloc(alloc_id, len + 1);
	memcpy(*ptr, name, len);
	(*ptr)[len] = '\0';
      }
  }
}

int
demangling_enabled(set, reset)
     int set;
     int reset;
{
  static int enabled = 1;

  if (set)
    enabled = 1;
  if (reset)
    enabled = 0;
  return(enabled);
}


/* Demangler for GNU C++ 
   Copyright 1989, 1991, 1994, 1995, 1996 Free Software Foundation, Inc.
   Written by James Clark (jjc@jclark.uucp)
   Rewritten by Fred Fish (fnf@cygnus.com) for ARM and Lucid demangling
   
This file is part of the libiberty library.
Libiberty is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

Libiberty is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with libiberty; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#undef CURRENT_DEMANGLING_STYLE
#define CURRENT_DEMANGLING_STYLE work->options

char *
mystrstr (s1, s2)
  char *s1, *s2;
{
  register char *p = s1;
  register int len = strlen (s2);

  for (; (p = strchr (p, *s2)) != 0; p++)
    {
      if (strncmp (p, s2, len) == 0)
	{
	  return (p);
	}
    }
  return (0);
}

/* In order to allow a single demangler executable to demangle strings
   using various common values of CPLUS_MARKER, as well as any specific
   one set at compile time, we maintain a string containing all the
   commonly used ones, and check to see if the marker we are looking for
   is in that string.  CPLUS_MARKER is usually '$' on systems where the
   assembler can deal with that.  Where the assembler can't, it's usually
   '.' (but on many systems '.' is used for other things).  We put the
   current defined CPLUS_MARKER first (which defaults to '$'), followed
   by the next most common value, followed by an explicit '$' in case
   the value of CPLUS_MARKER is not '$'.

   We could avoid this if we could just get g++ to tell us what the actual
   cplus marker character is as part of the debug information, perhaps by
   ensuring that it is the character that terminates the gcc<n>_compiled
   marker symbol (FIXME). */

#if !defined (CPLUS_MARKER)
#define CPLUS_MARKER '$'
#endif

enum demangling_styles current_demangling_style = gnu_demangling;

static char cplus_markers[] = { CPLUS_MARKER, '.', '$', '\0' };

/* Stuff that is shared between sub-routines.
 * Using a shared structure allows gnu_cplus_demangle to be reentrant. */

struct work_stuff
{
  int options;
  char **typevec;
  int ntypes;
  int typevec_size;
  int constructor;
  int destructor;
  int static_type;	/* A static member function */
  int const_type;	/* A const member function */
};

#define PRINT_ANSI_QUALIFIERS (work -> options & DMGL_ANSI)
#define PRINT_ARG_TYPES       (work -> options & DMGL_PARAMS)

static const struct optable
{
  const char *in;
  const char *out;
  int flags;
} optable[] = {
  {"nw",	  " new",	DMGL_ANSI},	/* new (1.92,	 ansi) */
  {"dl",	  " delete",	DMGL_ANSI},	/* new (1.92,	 ansi) */
  {"new",	  " new",	0},		/* old (1.91,	 and 1.x) */
  {"delete",	  " delete",	0},		/* old (1.91,	 and 1.x) */
  {"vn",	  " new []",	DMGL_ANSI},	/* GNU, pending ansi */
  {"vd",	  " delete []",	DMGL_ANSI},	/* GNU, pending ansi */
  {"as",	  "=",		DMGL_ANSI},	/* ansi */
  {"ne",	  "!=",		DMGL_ANSI},	/* old, ansi */
  {"eq",	  "==",		DMGL_ANSI},	/* old,	ansi */
  {"ge",	  ">=",		DMGL_ANSI},	/* old,	ansi */
  {"gt",	  ">",		DMGL_ANSI},	/* old,	ansi */
  {"le",	  "<=",		DMGL_ANSI},	/* old,	ansi */
  {"lt",	  "<",		DMGL_ANSI},	/* old,	ansi */
  {"plus",	  "+",		0},		/* old */
  {"pl",	  "+",		DMGL_ANSI},	/* ansi */
  {"apl",	  "+=",		DMGL_ANSI},	/* ansi */
  {"minus",	  "-",		0},		/* old */
  {"mi",	  "-",		DMGL_ANSI},	/* ansi */
  {"ami",	  "-=",		DMGL_ANSI},	/* ansi */
  {"mult",	  "*",		0},		/* old */
  {"ml",	  "*",		DMGL_ANSI},	/* ansi */
  {"amu",	  "*=",		DMGL_ANSI},	/* ansi (ARM/Lucid) */
  {"aml",	  "*=",		DMGL_ANSI},	/* ansi (GNU/g++) */
  {"convert",	  "+",		0},		/* old (unary +) */
  {"negate",	  "-",		0},		/* old (unary -) */
  {"trunc_mod",	  "%",		0},		/* old */
  {"md",	  "%",		DMGL_ANSI},	/* ansi */
  {"amd",	  "%=",		DMGL_ANSI},	/* ansi */
  {"trunc_div",	  "/",		0},		/* old */
  {"dv",	  "/",		DMGL_ANSI},	/* ansi */
  {"adv",	  "/=",		DMGL_ANSI},	/* ansi */
  {"truth_andif", "&&",		0},		/* old */
  {"aa",	  "&&",		DMGL_ANSI},	/* ansi */
  {"truth_orif",  "||",		0},		/* old */
  {"oo",	  "||",		DMGL_ANSI},	/* ansi */
  {"truth_not",	  "!",		0},		/* old */
  {"nt",	  "!",		DMGL_ANSI},	/* ansi */
  {"postincrement","++",	0},		/* old */
  {"pp",	  "++",		DMGL_ANSI},	/* ansi */
  {"postdecrement","--",	0},		/* old */
  {"mm",	  "--",		DMGL_ANSI},	/* ansi */
  {"bit_ior",	  "|",		0},		/* old */
  {"or",	  "|",		DMGL_ANSI},	/* ansi */
  {"aor",	  "|=",		DMGL_ANSI},	/* ansi */
  {"bit_xor",	  "^",		0},		/* old */
  {"er",	  "^",		DMGL_ANSI},	/* ansi */
  {"aer",	  "^=",		DMGL_ANSI},	/* ansi */
  {"bit_and",	  "&",		0},		/* old */
  {"ad",	  "&",		DMGL_ANSI},	/* ansi */
  {"aad",	  "&=",		DMGL_ANSI},	/* ansi */
  {"bit_not",	  "~",		0},		/* old */
  {"co",	  "~",		DMGL_ANSI},	/* ansi */
  {"call",	  "()",		0},		/* old */
  {"cl",	  "()",		DMGL_ANSI},	/* ansi */
  {"alshift",	  "<<",		0},		/* old */
  {"ls",	  "<<",		DMGL_ANSI},	/* ansi */
  {"als",	  "<<=",	DMGL_ANSI},	/* ansi */
  {"arshift",	  ">>",		0},		/* old */
  {"rs",	  ">>",		DMGL_ANSI},	/* ansi */
  {"ars",	  ">>=",	DMGL_ANSI},	/* ansi */
  {"component",	  "->",		0},		/* old */
  {"pt",	  "->",		DMGL_ANSI},	/* ansi; Lucid C++ form */
  {"rf",	  "->",		DMGL_ANSI},	/* ansi; ARM/GNU form */
  {"indirect",	  "*",		0},		/* old */
  {"method_call",  "->()",	0},		/* old */
  {"addr",	  "&",		0},		/* old (unary &) */
  {"array",	  "[]",		0},		/* old */
  {"vc",	  "[]",		DMGL_ANSI},	/* ansi */
  {"compound",	  ", ",		0},		/* old */
  {"cm",	  ", ",		DMGL_ANSI},	/* ansi */
  {"cond",	  "?:",		0},		/* old */
  {"cn",	  "?:",		DMGL_ANSI},	/* pseudo-ansi */
  {"max",	  ">?",		0},		/* old */
  {"mx",	  ">?",		DMGL_ANSI},	/* pseudo-ansi */
  {"min",	  "<?",		0},		/* old */
  {"mn",	  "<?",		DMGL_ANSI},	/* pseudo-ansi */
  {"nop",	  "",		0},		/* old (for operator=) */
  {"rm",	  "->*",	DMGL_ANSI}	/* ansi */
};


typedef struct string		/* Beware: these aren't required to be */
{				/*  '\0' terminated. */
  char *b;			/* pointer to start of string */
  char *p;			/* pointer after last character */
  char *e;			/* pointer after end of allocated space */
} string;

#define STRING_EMPTY(str)	((str) -> b == (str) -> p)
#define PREPEND_BLANK(str)	{if (!STRING_EMPTY(str)) \
				   string_prepend(str, " ");}
#define APPEND_BLANK(str)	{if (!STRING_EMPTY(str)) \
				   string_append(str, " ");}

#define ARM_VTABLE_STRING "__vtbl__"	/* Lucid/ARM virtual table prefix */
#define ARM_VTABLE_STRLEN 8		/* strlen (ARM_VTABLE_STRING) */

/* Prototypes for local functions */

static char *
mop_up PROTO ((struct work_stuff *, string *, int));

#if 0
static int
demangle_method_args PROTO ((struct work_stuff *work, const char **, string *));
#endif

static int
demangle_template PROTO ((struct work_stuff *work, const char **, string *,
			   string *));

static int
demangle_qualified PROTO ((struct work_stuff *, const char **, string *,
			    int, int));

static int
demangle_class PROTO ((struct work_stuff *, const char **, string *));

static int
demangle_fund_type PROTO ((struct work_stuff *, const char **, string *));

static int
demangle_signature PROTO ((struct work_stuff *, const char **, string *));

static int
demangle_prefix PROTO ((struct work_stuff *, const char **, string *));

static int
gnu_special PROTO ((struct work_stuff *, const char **, string *));

static int
arm_special PROTO ((struct work_stuff *, const char **, string *));

static void
string_need PROTO ((string *, int));

static void
string_delete PROTO ((string *));

static void
string_init PROTO ((string *));

static void
string_clear PROTO ((string *));

#if 0
static int
string_empty PROTO ((string *));
#endif

static void
string_append PROTO ((string *, const char *));

static void
string_appends PROTO ((string *, string *));

static void
string_appendn PROTO ((string *, const char *, int));

static void
string_prepend PROTO ((string *, const char *));

static void
string_prependn PROTO ((string *, const char *, int));

static int
get_count PROTO ((const char **, int *));

static int
consume_count PROTO ((const char **));

static int
demangle_args PROTO ((struct work_stuff *, const char **, string *));

static int
do_type PROTO ((struct work_stuff *, const char **, string *));

static int
do_arg PROTO ((struct work_stuff *, const char **, string *));

static void
demangle_function_name PROTO ((struct work_stuff *, const char **, string *,
				const char *));

static void
remember_type PROTO ((struct work_stuff *, const char *, int));

static void
forget_types PROTO ((struct work_stuff *));

static void
string_prepends PROTO ((string *, string *));

/*  Translate count to integer, consuming tokens in the process.
    Conversion terminates on the first non-digit character.
    Trying to consume something that isn't a count results in
    no consumption of input and a return of 0. */

static int
consume_count (type)
    const char **type;
{
    int count = 0;

    while (isdigit (**type))
      {
	count *= 10;
	count += **type - '0';
	(*type)++;
      }
    return (count);
}

/* check to see whether MANGLED can match TEXT in the first TEXT_LEN
   characters. */

int cplus_match (mangled, text, text_len)
     const char *mangled;
     char *text;
     int text_len;
{
  if (strncmp (mangled, text, text_len) != 0) {
    return(0); /* cannot match either */
  } else {
    return(1); /* matches mangled, may match demangled */
  }
}

/* char *gnu_cplus_demangle (const char *mangled, int options)

   If MANGLED is a mangled function name produced by GNU C++, then
   a pointer to a malloced string giving a C++ representation
   of the name will be returned; otherwise NULL will be returned.
   It is the caller's responsibility to free the string which
   is returned.

   The OPTIONS arg may contain one or more of the following bits:

   	DMGL_ANSI	ANSI qualifiers such as `const' and `void' are
			included.
	DMGL_PARAMS	Function parameters are included.

   For example,
   
   gnu_cplus_demangle ("foo__1Ai", DMGL_PARAMS)		=> "A::foo(int)"
   gnu_cplus_demangle ("foo__1Ai", DMGL_PARAMS | DMGL_ANSI)	=> "A::foo(int)"
   gnu_cplus_demangle ("foo__1Ai", 0)			=> "A::foo"

   gnu_cplus_demangle ("foo__1Afe", DMGL_PARAMS)		=> "A::foo(float,...)"
   gnu_cplus_demangle ("foo__1Afe", DMGL_PARAMS | DMGL_ANSI)=> "A::foo(float,...)"
   gnu_cplus_demangle ("foo__1Afe", 0)			=> "A::foo"

   Note that any leading underscores, or other such characters prepended by
   the compilation system, are presumed to have already been stripped from
   MANGLED.  */

static char *
gnu_cplus_demangle (mangled, options)
     const char *mangled;
     int options;
{
  string decl;
  int success = 0;
  struct work_stuff work[1];
  char *demangled = NULL;

  if ((mangled != NULL) && (*mangled != '\0'))
    {
      memset ((char *) work, 0, sizeof (work));
      work -> options = options;
      if ((work->options & DMGL_STYLE_MASK) == 0)
	work->options |= (int)current_demangling_style & DMGL_STYLE_MASK;
      
      string_init (&decl);

      /* First check to see if gnu style demangling is active and if the
	 string to be demangled contains a CPLUS_MARKER.  If so, attempt to
	 recognize one of the gnu special forms rather than looking for a
	 standard prefix.  In particular, don't worry about whether there
	 is a "__" string in the mangled string.  Consider "_$_5__foo" for
	 example. */

      if ((AUTO_DEMANGLING || GNU_DEMANGLING))
	{
	  success = gnu_special (work, &mangled, &decl);
	}
      if (!success)
	{
	  success = demangle_prefix (work, &mangled, &decl);
	}
      if (success && (*mangled != '\0'))
	{
	  success = demangle_signature (work, &mangled, &decl);
	}
      if (work->constructor == 2)
        {
          string_prepend(&decl, "global constructors keyed to ");
          work->constructor = 0;
        }
      else if (work->destructor == 2)
        {
          string_prepend(&decl, "global destructors keyed to ");
          work->destructor = 0;
        }
      demangled = mop_up (work, &decl, success);
    }
  return (demangled);
}

static char *
mop_up (work, declp, success)
     struct work_stuff *work;
     string *declp;
     int success;
{
  char *demangled = NULL;

  /* Discard the remembered types, if any. */
  
  forget_types (work);
  if (work -> typevec != NULL)
    {
      free ((char *) work -> typevec);
    }
  
  /* If demangling was successful, ensure that the demangled string is null
     terminated and return it.  Otherwise, free the demangling decl. */
  
  if (!success)
    {
      string_delete (declp);
    }
  else
    {
      string_appendn (declp, "", 1);
      demangled = declp -> b;
    }
  return (demangled);
}

/*

LOCAL FUNCTION

	demangle_signature -- demangle the signature part of a mangled name

SYNOPSIS

	static int
	demangle_signature (struct work_stuff *work, const char **mangled,
			    string *declp);

DESCRIPTION

	Consume and demangle the signature portion of the mangled name.

	DECLP is the string where demangled output is being built.  At
	entry it contains the demangled root name from the mangled name
	prefix.  I.E. either a demangled operator name or the root function
	name.  In some special cases, it may contain nothing.

	*MANGLED points to the current unconsumed location in the mangled
	name.  As tokens are consumed and demangling is performed, the
	pointer is updated to continuously point at the next token to
	be consumed.

	Demangling GNU style mangled names is nasty because there is no
	explicit token that marks the start of the outermost function
	argument list.
*/

static int
demangle_signature (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int success = 1;
  int func_done = 0;
  int expect_func = 0;
  const char *oldmangled = NULL;
  string trawname;
  string tname;

  while (success && (**mangled != '\0'))
    {
      switch (**mangled)
	{
	  case 'Q':
	    oldmangled = *mangled;
	    success = demangle_qualified (work, mangled, declp, 1, 0);
	    if (success)
	      {
		remember_type (work, oldmangled, *mangled - oldmangled);
	      }
	    if (AUTO_DEMANGLING || GNU_DEMANGLING)
	      {
		expect_func = 1;
	      }
	    oldmangled = NULL;
	    break;
	  
	  case 'S':
	    /* Static member function */
	    if (oldmangled == NULL)
	      {
		oldmangled = *mangled;
	      }
	    (*mangled)++;
	    work -> static_type = 1;
	    break;

	  case 'C':
	    /* a const member function */
	    if (oldmangled == NULL)
	      {
		oldmangled = *mangled;
	      }
	    (*mangled)++;
	    work -> const_type = 1;
	    break;
	  
	  case '0': case '1': case '2': case '3': case '4':
	  case '5': case '6': case '7': case '8': case '9':
	    if (oldmangled == NULL)
	      {
		oldmangled = *mangled;
	      }
	    success = demangle_class (work, mangled, declp);
	    if (success)
	      {
		remember_type (work, oldmangled, *mangled - oldmangled);
	      }
	    if (AUTO_DEMANGLING || GNU_DEMANGLING)
	      {
		expect_func = 1;
	      }
	    oldmangled = NULL;
	    break;
	  
	  case 'F':
	    /* Function */
	    /* ARM style demangling includes a specific 'F' character after
	     the class name.  For GNU style, it is just implied.  So we can
	     safely just consume any 'F' at this point and be compatible
	     with either style. */

	    oldmangled = NULL;
	    func_done = 1;
	    (*mangled)++;

	    /* For lucid/ARM style we have to forget any types we might
	       have remembered up to this point, since they were not argument
	       types.  GNU style considers all types seen as available for
	       back references.  See comment in demangle_args() */

	    if (LUCID_DEMANGLING || ARM_DEMANGLING)
	      {
		forget_types (work);
	      }
	    success = demangle_args (work, mangled, declp);
	    break;
	  
	  case 't':
	    /* G++ Template */
	    string_init(&trawname); 
	    string_init(&tname);
            if (oldmangled == NULL)
              {
                oldmangled = *mangled;
              }
	    success = demangle_template (work, mangled, &tname, &trawname);
            if (success)
              {
                remember_type (work, oldmangled, *mangled - oldmangled);
              }
	    string_append(&tname, "::");
	    string_prepends(declp, &tname);
  	    if (work -> destructor & 1)
    	      {
      		string_prepend (&trawname, "~");
      		string_appends (declp, &trawname);
		work->destructor -= 1;
    	      }
  	    if ((work->constructor & 1) || (work->destructor & 1))
    	      {
      		string_appends (declp, &trawname);
		work->constructor -= 1;
              }
	    string_delete(&trawname);
	    string_delete(&tname);
	    oldmangled = NULL;
	    expect_func = 1;
	    break;

	  case '_':
	    /* At the outermost level, we cannot have a return type specified,
	       so if we run into another '_' at this point we are dealing with
	       a mangled name that is either bogus, or has been mangled by
	       some algorithm we don't know how to deal with.  So just
	       reject the entire demangling. */
	    success = 0;
	    break;

	  default:
	    if (AUTO_DEMANGLING || GNU_DEMANGLING)
	      {
		/* Assume we have stumbled onto the first outermost function
		   argument token, and start processing args. */
		func_done = 1;
		success = demangle_args (work, mangled, declp);
	      }
	    else
	      {
		/* Non-GNU demanglers use a specific token to mark the start
		   of the outermost function argument tokens.  Typically 'F',
		   for ARM-demangling, for example.  So if we find something
		   we are not prepared for, it must be an error. */
		success = 0;
	      }
	    break;
	}
/*
      if (AUTO_DEMANGLING || GNU_DEMANGLING)
*/
	{
	  if (success && expect_func)
	    {
	      func_done = 1;
	      success = demangle_args (work, mangled, declp);
	    }
	}
    }
  if (success && !func_done)
    {
      if (AUTO_DEMANGLING || GNU_DEMANGLING)
	{
	  /* With GNU style demangling, bar__3foo is 'foo::bar(void)', and
	     bar__3fooi is 'foo::bar(int)'.  We get here when we find the
	     first case, and need to ensure that the '(void)' gets added to
	     the current declp.  Note that with ARM, the first case
	     represents the name of a static data member 'foo::bar',
	     which is in the current declp, so we leave it alone. */
	  success = demangle_args (work, mangled, declp);
	}
    }
  if (success && work -> static_type && PRINT_ARG_TYPES)
    {
      string_append (declp, " static");
    }
  if (success && work -> const_type && PRINT_ARG_TYPES)
    {
      string_append (declp, " const");
    }
  return (success);
}

#if 0

static int
demangle_method_args (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int success = 0;

  if (work -> static_type)
    {
      string_append (declp, *mangled + 1);
      *mangled += strlen (*mangled);
      success = 1;
    }
  else
    {
      success = demangle_args (work, mangled, declp);
    }
  return (success);
}

#endif

static int
demangle_template (work, mangled, tname, trawname)
     struct work_stuff *work;
     const char **mangled;
     string *tname;
     string *trawname;
{
  int i;
  int is_pointer;
  int is_real;
  int is_integral;
  int is_char;
  int is_bool;
  int r;
  int need_comma = 0;
  int success = 0;
  int done;
  const char *old_p;
  const char *start;
  int symbol_len;
  string temp;

  (*mangled)++;
  start = *mangled;
  /* get template name */
  if ((r = consume_count (mangled)) == 0 || strlen (*mangled) < r)
    {
      return (0);
    }
  if (trawname)
    string_appendn (trawname, *mangled, r);
  string_appendn (tname, *mangled, r);
  *mangled += r;
  string_append (tname, "<");
  /* get size of template parameter list */
  if (!get_count (mangled, &r))
    {
      return (0);
    }
  for (i = 0; i < r; i++)
    {
      if (need_comma)
	{
	  string_append (tname, ", ");
	}
      /* Z for type parameters */
      if (**mangled == 'Z')
	{
	  (*mangled)++;
	  /* temp is initialized in do_type */
	  success = do_type (work, mangled, &temp);
	  if (success)
	    {
	      string_appends (tname, &temp);
	    }
	  string_delete(&temp);
	  if (!success)
	    {
	      break;
	    }
	}
      else
	{
	  /* otherwise, value parameter */
	  old_p  = *mangled;
	  is_pointer = 0;
	  is_real = 0;
	  is_integral = 0;
          is_char = 0;
	  is_bool = 0;
	  done = 0;
	  /* temp is initialized in do_type */
	  success = do_type (work, mangled, &temp);
/*
	  if (success)
	    {
	      string_appends (tname, &temp);
	    }
*/
	  string_delete(&temp);
	  if (!success)
	    {
	      break;
	    }
/*
	  string_append (tname, "=");
*/
	  while (*old_p && !done)
	    {	
	      switch (*old_p)
		{
		  case 'P':
		  case 'p':
		  case 'R':
		    done = is_pointer = 1;
		    break;
		  case 'C':	/* const */
		  case 'S':	/* explicitly signed [char] */
		  case 'U':	/* unsigned */
		  case 'V':	/* volatile */
		  case 'F':	/* function */
		  case 'M':	/* member function */
		  case 'O':	/* ??? */
		    old_p++;
		    continue;
		  case 'Q':	/* qualified name */
                    done = is_integral = 1;
                    break;
		  case 'T':	/* remembered type */
		    abort ();
		    break;
		  case 'v':	/* void */
		    abort ();
		    break;
		  case 'x':	/* long long */
		  case 'l':	/* long */
		  case 'i':	/* int */
		  case 's':	/* short */
		  case 'w':	/* wchar_t */
		    done = is_integral = 1;
		    break;
		  case 'b':	/* bool */
		    done = is_bool = 1;
		    break;
		  case 'c':	/* char */
		    done = is_char = 1;
		    break;
		  case 'r':	/* long double */
		  case 'd':	/* double */
		  case 'f':	/* float */
		    done = is_real = 1;
		    break;
		  default:
		    /* it's probably user defined type, let's assume
		       it's integral, it seems hard to figure out
		       what it really is */
		    done = is_integral = 1;
		}
	    }
	  if (is_integral)
	    {
	      if (**mangled == 'm')
		{
		  string_appendn (tname, "-", 1);
		  (*mangled)++;
		}
	      while (isdigit (**mangled))	
		{
		  string_appendn (tname, *mangled, 1);
		  (*mangled)++;
		}
	    }
	  else if (is_char)
	    {
            char tmp[2];
            int val;
              if (**mangled == 'm')
                {
                  string_appendn (tname, "-", 1);
                  (*mangled)++;
                }
	      string_appendn (tname, "'", 1);
              val = consume_count(mangled);
	      if (val == 0)
		{
		  success = 0;
		  break;
                }
              tmp[0] = (char)val;
              tmp[1] = '\0';
              string_appendn (tname, &tmp[0], 1);
	      string_appendn (tname, "'", 1);
	    }
	  else if (is_bool)
	    {
	      int val = consume_count (mangled);
	      if (val == 0)
		string_appendn (tname, "false", 5);
	      else if (val == 1)
		string_appendn (tname, "true", 4);
	      else
		success = 0;
	    }
	  else if (is_real)
	    {
	      if (**mangled == 'm')
		{
		  string_appendn (tname, "-", 1);
		  (*mangled)++;
		}
	      while (isdigit (**mangled))	
		{
		  string_appendn (tname, *mangled, 1);
		  (*mangled)++;
		}
	      if (**mangled == '.') /* fraction */
		{
		  string_appendn (tname, ".", 1);
		  (*mangled)++;
		  while (isdigit (**mangled))	
		    {
		      string_appendn (tname, *mangled, 1);
		      (*mangled)++;
		    }
		}
	      if (**mangled == 'e') /* exponent */
		{
		  string_appendn (tname, "e", 1);
		  (*mangled)++;
		  while (isdigit (**mangled))	
		    {
		      string_appendn (tname, *mangled, 1);
		      (*mangled)++;
		    }
		}
	    }
	  else if (is_pointer)
	    {
	      if (!get_count (mangled, &symbol_len))
		{
		  success = 0;
		  break;
		}
	      if (symbol_len == 0)
		string_appendn (tname, "0", 1);
	      else
		{
		  char *p = malloc (symbol_len + 1), *q;
		  strncpy (p, *mangled, symbol_len);
		  p [symbol_len] = '\0';
		  q = gnu_cplus_demangle (p, work->options);
		  string_appendn (tname, "&", 1);
		  if (q)
		    {
		      string_append (tname, q);
		      free (q);
		    }
		  else
		    string_append (tname, p);
		  free (p);
		}
	      *mangled += symbol_len;
	    }
	}
      need_comma = 1;
    }
  if (tname->p[-1] == '>')
    string_append (tname, " ");
  string_append (tname, ">");
  
/*
      if (work -> static_type)
	{
	  string_append (declp, *mangled + 1);
	  *mangled += strlen (*mangled);
	  success = 1;
	}
      else
	{
	  success = demangle_args (work, mangled, declp);
	}
    }
*/
  return (success);
}

static int
arm_pt (work, mangled, n, anchor, args)
     struct work_stuff *work;
     const char *mangled;
     int n;
     const char **anchor, **args;
{
  /* ARM template? */
  if (ARM_DEMANGLING)
  {
    *anchor = (const char *)mystrstr ((char *)mangled, "__pt__");
    if (*anchor)
    {
	int len;
        *args = *anchor + 6;
	len = consume_count (args);
        if (*args + len == mangled + n && **args == '_')
	  {
	    ++*args;
	    return 1;
	  }
    }
  }
  return 0;
}

static void
demangle_arm_pt (work, mangled, n, declp)
     struct work_stuff *work;
     const char **mangled;
     int n;
     string *declp;
{
  const char *p;
  const char *args;
  const char *e = *mangled + n;

  /* ARM template? */
  if (arm_pt (work, *mangled, n, &p, &args))
  {
    string arg;
    string_init (&arg);
    string_appendn (declp, *mangled, p - *mangled);
    string_append (declp, "<");
    /* should do error checking here */
    while (args < e) {
      string_clear (&arg);
      do_type (work, &args, &arg);
      string_appends (declp, &arg);
      string_append (declp, ",");
    }
    string_delete (&arg);
    --declp->p;
    string_append (declp, ">");
  }
  else
  {
    string_appendn (declp, *mangled, n);
  }
  *mangled += n;
}

static int
demangle_class_name (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int n;
  int success = 0;

  n = consume_count (mangled);
  if (strlen (*mangled) >= n)
  {
    demangle_arm_pt (work, mangled, n, declp);
    success = 1;
  }

  return (success);
}

/*

LOCAL FUNCTION

	demangle_class -- demangle a mangled class sequence

SYNOPSIS

	static int
	demangle_class (struct work_stuff *work, const char **mangled,
			strint *declp)

DESCRIPTION

	DECLP points to the buffer into which demangling is being done.

	*MANGLED points to the current token to be demangled.  On input,
	it points to a mangled class (I.E. "3foo", "13verylongclass", etc.)
	On exit, it points to the next token after the mangled class on
	success, or the first unconsumed token on failure.

	If the CONSTRUCTOR or DESTRUCTOR flags are set in WORK, then
	we are demangling a constructor or destructor.  In this case
	we prepend "class::class" or "class::~class" to DECLP.

	Otherwise, we prepend "class::" to the current DECLP.

	Reset the constructor/destructor flags once they have been
	"consumed".  This allows demangle_class to be called later during
	the same demangling, to do normal class demangling.

	Returns 1 if demangling is successful, 0 otherwise.

*/

static int
demangle_class (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int success = 0;
  string class_name;

  string_init (&class_name);
  if (demangle_class_name (work, mangled, &class_name))
    {
      if ((work->constructor & 1) || (work->destructor & 1))
	{
	  string_prepends (declp, &class_name);
	  if (work -> destructor & 1)
	    {
	      string_prepend (declp, "~");
              work -> destructor -= 1;
	    }
	  else
	    {
	      work -> constructor -= 1; 
	    }
	}
      string_prepend (declp, "::");
      string_prepends (declp, &class_name);
      success = 1;
    }
  string_delete (&class_name);
  return (success);
}

/*

LOCAL FUNCTION

	demangle_prefix -- consume the mangled name prefix and find signature

SYNOPSIS

	static int
	demangle_prefix (struct work_stuff *work, const char **mangled,
			 string *declp);

DESCRIPTION

	Consume and demangle the prefix of the mangled name.

	DECLP points to the string buffer into which demangled output is
	placed.  On entry, the buffer is empty.  On exit it contains
	the root function name, the demangled operator name, or in some
	special cases either nothing or the completely demangled result.

	MANGLED points to the current pointer into the mangled name.  As each
	token of the mangled name is consumed, it is updated.  Upon entry
	the current mangled name pointer points to the first character of
	the mangled name.  Upon exit, it should point to the first character
	of the signature if demangling was successful, or to the first
	unconsumed character if demangling of the prefix was unsuccessful.
	
	Returns 1 on success, 0 otherwise.
 */

static int
demangle_prefix (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int success = 1;
  const char *scan;
  int i;

  if (strlen(*mangled) >= 11 && strncmp(*mangled, "_GLOBAL_", 8) == 0)
    {
      char *marker = strchr (cplus_markers, (*mangled)[8]);
      if (marker != NULL && *marker == (*mangled)[10])
	{
	  if ((*mangled)[9] == 'D')
	    {
	      /* it's a GNU global destructor to be executed at program exit */
	      (*mangled) += 11;
	      work->destructor = 2;
	      if (gnu_special (work, mangled, declp))
		return success;
	    }
	  else if ((*mangled)[9] == 'I')
	    {
	      /* it's a GNU global constructor to be executed at program init */
	      (*mangled) += 11;
	      work->constructor = 2;
	      if (gnu_special (work, mangled, declp))
		return success;
	    }
	}
    }
  else if (ARM_DEMANGLING && strncmp(*mangled, "__std__", 7) == 0)
    {
      /* it's a ARM global destructor to be executed at program exit */
      (*mangled) += 7;
      work->destructor = 2;
    }
  else if (ARM_DEMANGLING && strncmp(*mangled, "__sti__", 7) == 0)
    {
      /* it's a ARM global constructor to be executed at program initial */
      (*mangled) += 7;
      work->constructor = 2;
    }

/*  This block of code is a reduction in strength time optimization
    of:
    	scan = mystrstr (*mangled, "__"); */

  {
    scan = *mangled;

    do {
      scan = strchr (scan, '_');
    } while (scan != NULL && *++scan != '_');

    if (scan != NULL) --scan;
  }

  if (scan != NULL)
    {
      /* We found a sequence of two or more '_', ensure that we start at
	 the last pair in the sequence. */
      i = strspn (scan, "_");
      if (i > 2)
	{
	  scan += (i - 2); 
	}
    }
 
  if (scan == NULL)
    {
      success = 0;
    }
  else if (work -> static_type)
    {
      if (!isdigit (scan[0]) && (scan[0] != 't'))
	{
	  success = 0;
	}
    }
  else if ((scan == *mangled) &&
	   (isdigit (scan[2]) || (scan[2] == 'Q') || (scan[2] == 't')))
    {
      /* The ARM says nothing about the mangling of local variables.
	 But cfront mangles local variables by prepending __<nesting_level>
	 to them. As an extension to ARM demangling we handle this case.  */
      if ((LUCID_DEMANGLING || ARM_DEMANGLING) && isdigit (scan[2]))
	{
	  *mangled = scan + 2;
	  consume_count (mangled);
	  string_append (declp, *mangled);
	  *mangled += strlen (*mangled);
	  success = 1; 
	}
      else
	{
	  /* A GNU style constructor starts with __[0-9Qt].  But cfront uses
	     names like __Q2_3foo3bar for nested type names.  So don't accept
	     this style of constructor for cfront demangling.  */
	  if (!(LUCID_DEMANGLING || ARM_DEMANGLING))
	    work -> constructor += 1;
	  *mangled = scan + 2;
	}
    }
  else if ((scan == *mangled) && !isdigit (scan[2]) && (scan[2] != 't'))
    {
      /* Mangled name starts with "__".  Skip over any leading '_' characters,
	 then find the next "__" that separates the prefix from the signature.
	 */
      if (!(ARM_DEMANGLING || LUCID_DEMANGLING)
	  || (arm_special (work, mangled, declp) == 0))
	{
	  while (*scan == '_')
	    {
	      scan++;
	    }
	  if ((scan = mystrstr ((char *)scan, "__")) == NULL || (*(scan + 2) == '\0'))
	    {
	      /* No separator (I.E. "__not_mangled"), or empty signature
		 (I.E. "__not_mangled_either__") */
	      success = 0;
	    }
	  else
	    {
	      demangle_function_name (work, mangled, declp, scan);
	    }
	}
    }
  else if (ARM_DEMANGLING && scan[2] == 'p' && scan[3] == 't')
    {
      /* Cfront-style parameterized type.  Handled later as a signature. */
      success = 1;

      /* ARM template? */
      demangle_arm_pt (work, mangled, strlen (*mangled), declp);
    }
  else if (*(scan + 2) != '\0')
    {
      /* Mangled name does not start with "__" but does have one somewhere
	 in there with non empty stuff after it.  Looks like a global
	 function name. */
      demangle_function_name (work, mangled, declp, scan);
    }
  else
    {
      /* Doesn't look like a mangled name */
      success = 0;
    }

  if (!success && (work->constructor == 2 || work->destructor == 2))
    {
      string_append (declp, *mangled);
      *mangled += strlen (*mangled);
      success = 1;
    } 
  return (success);
}

/*

LOCAL FUNCTION

	gnu_special -- special handling of gnu mangled strings

SYNOPSIS

	static int
	gnu_special (struct work_stuff *work, const char **mangled,
		     string *declp);


DESCRIPTION

	Process some special GNU style mangling forms that don't fit
	the normal pattern.  For example:

		_$_3foo		(destructor for class foo)
		_vt$foo		(foo virtual table)
		_vt$foo$bar	(foo::bar virtual table)
		__vt_foo	(foo virtual table, new style with thunks)
		_3foo$varname	(static data member)
		_Q22rs2tu$vw	(static data member)
		__t6vector1Zii	(constructor with template)
		__thunk_4__$_7ostream (virtual function thunk)
 */

static int
gnu_special (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int n;
  int success = 1;
  const char *p;

  if ((*mangled)[0] == '_'
      && strchr (cplus_markers, (*mangled)[1]) != NULL
      && (*mangled)[2] == '_')
    {
      /* Found a GNU style destructor, get past "_<CPLUS_MARKER>_" */
      (*mangled) += 3;
      work -> destructor += 1;
    }
  else if ((*mangled)[0] == '_'
	   && (((*mangled)[1] == '_'
		&& (*mangled)[2] == 'v'
		&& (*mangled)[3] == 't'
		&& (*mangled)[4] == '_')
	     || ((*mangled)[1] == 'v'
		 && (*mangled)[2] == 't'
		 && strchr (cplus_markers, (*mangled)[3]) != NULL)))
    {
      /* Found a GNU style virtual table, get past "_vt<CPLUS_MARKER>"
         and create the decl.  Note that we consume the entire mangled
	 input string, which means that demangle_signature has no work
	 to do. */
      if ((*mangled)[2] == 'v')
	(*mangled) += 5; /* New style, with thunks: "__vt_" */
      else
	(*mangled) += 4; /* Old style, no thunks: "_vt<CPLUS_MARKER>" */
      while (**mangled != '\0')
	{
	  p = strpbrk (*mangled, cplus_markers);
	  switch (**mangled)
	    {
	    case 'Q':
	      success = demangle_qualified (work, mangled, declp, 0, 1);
	      break;
	    case 't':
	      success = demangle_template (work, mangled, declp, 0);
	      break;
	    default:
	      if (isdigit(*mangled[0]))
		{
		  n = consume_count(mangled);
		}
	      else
		{
		  n = strcspn (*mangled, cplus_markers);
		}
	      string_appendn (declp, *mangled, n);
	      (*mangled) += n;
	    }

	  if (success && ((p == NULL) || (p == *mangled)))
	    {
	      if (p != NULL)
		{
		  string_append (declp, "::");
		  (*mangled)++;
		}
	    }
	  else
	    {
	      success = 0;
	      break;
	    }
	}
      if (success)
	string_append (declp, " virtual table");
    }
  else if ((*mangled)[0] == '_'
	   && (strchr("0123456789Qt", (*mangled)[1]) != NULL)
	   && (p = strpbrk (*mangled, cplus_markers)) != NULL)
    {
      /* static data member, "_3foo$varname" for example */
      (*mangled)++;
      switch (**mangled)
	{
	  case 'Q':
	    success = demangle_qualified (work, mangled, declp, 0, 1);
	    break;
	  case 't':
	    success = demangle_template (work, mangled, declp, 0);
	    break;
	  default:
	    n = consume_count (mangled);
	    string_appendn (declp, *mangled, n);
	    (*mangled) += n;
	}
      if (success && (p == *mangled))
	{
	  /* Consumed everything up to the cplus_marker, append the
	     variable name. */
	  (*mangled)++;
	  string_append (declp, "::");
	  n = strlen (*mangled);
	  string_appendn (declp, *mangled, n);
	  (*mangled) += n;
	}
      else
	{
	  success = 0;
	}
    }
  else if (strncmp (*mangled, "__thunk_", 8) == 0)
    {
      int delta = ((*mangled) += 8, consume_count (mangled));
      char *method = gnu_cplus_demangle (++*mangled, work->options);
      if (method)
	{
	  char buf[50];
	  sprintf (buf, "virtual function thunk (delta:%d) for ", -delta);
	  string_append (declp, buf);
	  string_append (declp, method);
	  free (method);
	  n = strlen (*mangled);
	  (*mangled) += n;
	}
      else
	{
	  success = 0;
	}
    }
  else
    {
      success = 0;
    }
  return (success);
}

/*

LOCAL FUNCTION

	arm_special -- special handling of ARM/lucid mangled strings

SYNOPSIS

	static int
	arm_special (struct work_stuff *work, const char **mangled,
			string *declp);


DESCRIPTION

	Process some special ARM style mangling forms that don't fit
	the normal pattern.  For example:

		__vtbl__3foo		(foo virtual table)
		__vtbl__3foo__3bar	(bar::foo virtual table)

 */

static int
arm_special (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  int n;
  int success = 1;
  const char *scan;

  if (strncmp (*mangled, ARM_VTABLE_STRING, ARM_VTABLE_STRLEN) == 0)
    {
      /* Found a ARM style virtual table, get past ARM_VTABLE_STRING
         and create the decl.  Note that we consume the entire mangled
	 input string, which means that demangle_signature has no work
	 to do. */
      scan = *mangled + ARM_VTABLE_STRLEN;
      while (*scan != '\0')        /* first check it can be demangled */
        {
          n = consume_count (&scan);
          if (n==0)
	    {
	      return (0);           /* no good */
	    }
          scan += n;
          if (scan[0] == '_' && scan[1] == '_')
	    {
	      scan += 2;
	    }
        }
      (*mangled) += ARM_VTABLE_STRLEN;
      while (**mangled != '\0')
	{
	  n = consume_count (mangled);
	  string_prependn (declp, *mangled, n);
	  (*mangled) += n;
	  if ((*mangled)[0] == '_' && (*mangled)[1] == '_')
	    {
	      string_prepend (declp, "::");
	      (*mangled) += 2;
	    }
	}
      string_append (declp, " virtual table");
    }
  else
    {
      success = 0;
    }
  return (success);
}

/*

LOCAL FUNCTION

	demangle_qualified -- demangle 'Q' qualified name strings

SYNOPSIS

	static int
	demangle_qualified (struct work_stuff *, const char *mangled,
			    string *result, int isfuncname, int append);

DESCRIPTION

	Demangle a qualified name, such as "Q25Outer5Inner" which is
	the mangled form of "Outer::Inner".  The demangled output is
	prepended or appended to the result string according to the
	state of the append flag.

	If isfuncname is nonzero, then the qualified name we are building
	is going to be used as a member function name, so if it is a
	constructor or destructor function, append an appropriate
	constructor or destructor name.  I.E. for the above example,
	the result for use as a constructor is "Outer::Inner::Inner"
	and the result for use as a destructor is "Outer::Inner::~Inner".

BUGS

	Numeric conversion is ASCII dependent (FIXME).

 */

static int
demangle_qualified (work, mangled, result, isfuncname, append)
     struct work_stuff *work;
     const char **mangled;
     string *result;
     int isfuncname;
     int append;
{
  int qualifiers = 0;
  int namelength = 0;
  int success = 1;
  const char *p;
  char num[2];
  string temp;

  string_init (&temp);
  switch ((*mangled)[1])
    {
    case '_':
      /* GNU mangled name with more than 9 classes.  The count is preceded
	 by an underscore (to distinguish it from the <= 9 case) and followed
	 by an underscore.  */
      p = *mangled + 2;
      qualifiers = atoi (p);
      if (!isdigit (*p) || *p == '0')
	success = 0;

      /* Skip the digits.  */
      while (isdigit (*p))
	++p;

      if (*p != '_')
	success = 0;

      *mangled = p + 1;
      break;

    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      /* The count is in a single digit.  */
      num[0] = (*mangled)[1];
      num[1] = '\0';
      qualifiers = atoi (num);

      /* If there is an underscore after the digit, skip it.  This is
	 said to be for ARM-qualified names, but the ARM makes no
	 mention of such an underscore.  Perhaps cfront uses one.  */
      if ((*mangled)[2] == '_')
	{
	  (*mangled)++;
	}
      (*mangled) += 2;
      break;

    case '0':
    default:
      success = 0;
    }

  if (!success)
    return success;

  /* Pick off the names and collect them in the temp buffer in the order
     in which they are found, separated by '::'. */

  while (qualifiers-- > 0)
    {
      if (*mangled[0] == '_') 
	*mangled = *mangled + 1;
      if (*mangled[0] == 't')
	{
	  success = demangle_template(work, mangled, &temp, 0);
	  if (!success) break;
	}
      else
        {	
	  namelength = consume_count (mangled);
      	  if (strlen (*mangled) < namelength)
	    {
	    /* Simple sanity check failed */
	       success = 0;
	       break;
	    }
      	  string_appendn (&temp, *mangled, namelength);
      	  *mangled += namelength;
	}
      if (qualifiers > 0)
        {
          string_appendn (&temp, "::", 2);
        }
    }

  /* If we are using the result as a function name, we need to append
     the appropriate '::' separated constructor or destructor name.
     We do this here because this is the most convenient place, where
     we already have a pointer to the name and the length of the name. */

  if (isfuncname && (work->constructor & 1 || work->destructor & 1))
    {
      string_appendn (&temp, "::", 2);
      if (work -> destructor & 1)
	{
	  string_append (&temp, "~");
	}
      string_appendn (&temp, (*mangled) - namelength, namelength);
    }

  /* Now either prepend the temp buffer to the result, or append it, 
     depending upon the state of the append flag. */

  if (append)
    {
      string_appends (result, &temp);
    }
  else
    {
      if (!STRING_EMPTY (result))
	{
	  string_appendn (&temp, "::", 2);
	}
      string_prepends (result, &temp);
    }

  string_delete (&temp);
  return (success);
}

/*

LOCAL FUNCTION

	get_count -- convert an ascii count to integer, consuming tokens

SYNOPSIS

	static int
	get_count (const char **type, int *count)

DESCRIPTION

	Return 0 if no conversion is performed, 1 if a string is converted.
*/

static int
get_count (type, count)
     const char **type;
     int *count;
{
  const char *p;
  int n;

  if (!isdigit (**type))
    {
      return (0);
    }
  else
    {
      *count = **type - '0';
      (*type)++;
      if (isdigit (**type))
	{
	  p = *type;
	  n = *count;
	  do 
	    {
	      n *= 10;
	      n += *p - '0';
	      p++;
	    } 
	  while (isdigit (*p));
	  if (*p == '_')
	    {
	      *type = p + 1;
	      *count = n;
	    }
	}
    }
  return (1);
}

/* result will be initialised here; it will be freed on failure */

static int
do_type (work, mangled, result)
     struct work_stuff *work;
     const char **mangled;
     string *result;
{
  int n;
  int done;
  int success;
  string decl;
  const char *remembered_type;
  int constp;
  int volatilep;

  string_init (&decl);
  string_init (result);

  done = 0;
  success = 1;
  while (success && !done)
    {
      int member;
      switch (**mangled)
	{

	/* A pointer type */
	case 'P':
	case 'p':
	  (*mangled)++;
	  string_prepend (&decl, "*");
	  break;

	/* A reference type */
	case 'R':
	  (*mangled)++;
	  string_prepend (&decl, "&");
	  break;

	/* An array */
	case 'A':
	  {
	    const char *p = ++(*mangled);

	    string_prepend (&decl, "(");
	    string_append (&decl, ")[");
	    /* Copy anything up until the next underscore (the size of the
	       array).  */
	    while (**mangled && **mangled != '_')
	      ++(*mangled);
	    if (**mangled == '_')
	      {
		string_appendn (&decl, p, *mangled - p);
		string_append (&decl, "]");             
		*mangled += 1;
	      }
	    else
	      success = 0;
	    break;
	  }

	/* A back reference to a previously seen type */
	case 'T':
	  (*mangled)++;
	  if (!get_count (mangled, &n) || n >= work -> ntypes)
	    {
	      success = 0;
	    }
	  else
	    {
	      remembered_type = work -> typevec[n];
	      mangled = &remembered_type;
	    }
	  break;

	/* A function */
	case 'F':
	  (*mangled)++;
	  if (!STRING_EMPTY (&decl) && decl.b[0] == '*')
	    {
	      string_prepend (&decl, "(");
	      string_append (&decl, ")");
	    }
	  /* After picking off the function args, we expect to either find the
	     function return type (preceded by an '_') or the end of the
	     string. */
	  if (!demangle_args (work, mangled, &decl)
	      || (**mangled != '_' && **mangled != '\0'))
	    {
	      success = 0;
	    }
	  if (success && (**mangled == '_'))
	    {
	      (*mangled)++;
	    }
	  break;

	case 'M':
	case 'O':
	  {
	    constp = 0;
	    volatilep = 0;

	    member = **mangled == 'M';
	    (*mangled)++;
	    if (!isdigit (**mangled))
	      {
		success = 0;
		break;
	      }
	    n = consume_count (mangled);
	    if (strlen (*mangled) < n)
	      {
		success = 0;
		break;
	      }
	    string_append (&decl, ")");
	    string_prepend (&decl, "::");
	    string_prependn (&decl, *mangled, n);
	    string_prepend (&decl, "(");
	    *mangled += n;
	    if (member)
	      {
		if (**mangled == 'C')
		  {
		    (*mangled)++;
		    constp = 1;
		  }
		if (**mangled == 'V')
		  {
		    (*mangled)++;
		    volatilep = 1;
		  }
		if (*(*mangled)++ != 'F')
		  {
		    success = 0;
		    break;
		  }
	      }
	    if ((member && !demangle_args (work, mangled, &decl))
		|| **mangled != '_')
	      {
		success = 0;
		break;
	      }
	    (*mangled)++;
	    if (! PRINT_ANSI_QUALIFIERS)
	      {
		break;
	      }
	    if (constp)
	      {
		APPEND_BLANK (&decl);
		string_append (&decl, "const");
	      }
	    if (volatilep)
	      {
		APPEND_BLANK (&decl);
		string_append (&decl, "volatile");
	      }
	    break;
	  }
        case 'G':
	    (*mangled)++;
	    break;

	case 'C':
	  (*mangled)++;
/*
	  if ((*mangled)[1] == 'P')
	    {
*/
	      if (PRINT_ANSI_QUALIFIERS)
		{
		  if (!STRING_EMPTY (&decl))
		    {
		      string_prepend (&decl, " ");
		    }
		  string_prepend (&decl, "const");
		}
	      break;
/*
	    }
*/

	  /* fall through */
	default:
	  done = 1;
	  break;
	}
    }

  switch (**mangled)
    {
      /* A qualified name, such as "Outer::Inner". */
      case 'Q':
        success = demangle_qualified (work, mangled, result, 0, 1);
	break;

      default:
	success = demangle_fund_type (work, mangled, result);
	break;
    }

  if (success)
    {
      if (!STRING_EMPTY (&decl))
	{
	  string_append (result, " ");
	  string_appends (result, &decl);
	}
    }
  else
    {
      string_delete (result);
    }
  string_delete (&decl);
  return (success);
}

/* Given a pointer to a type string that represents a fundamental type
   argument (int, long, unsigned int, etc) in TYPE, a pointer to the
   string in which the demangled output is being built in RESULT, and
   the WORK structure, decode the types and add them to the result.

   For example:

   	"Ci"	=>	"const int"
	"Sl"	=>	"signed long"
	"CUs"	=>	"const unsigned short"

   */

static int
demangle_fund_type (work, mangled, result)
     struct work_stuff *work;
     const char **mangled;
     string *result;
{
  int done = 0;
  int success = 1;

  /* First pick off any type qualifiers.  There can be more than one. */

  while (!done)
    {
      switch (**mangled)
	{
	  case 'C':
	    (*mangled)++;
	    if (PRINT_ANSI_QUALIFIERS)
	      {
		APPEND_BLANK (result);
		string_append (result, "const");
	      }
	    break;
	  case 'U':
	    (*mangled)++;
	    APPEND_BLANK (result);
	    string_append (result, "unsigned");
	    break;
	  case 'S': /* signed char only */
	    (*mangled)++;
	    APPEND_BLANK (result);
	    string_append (result, "signed");
	    break;
	  case 'V':
	    (*mangled)++;
	    if (PRINT_ANSI_QUALIFIERS)
	      {
		APPEND_BLANK (result);
		string_append (result, "volatile");
	      }
	    break;
	  default:
	    done = 1;
	    break;
	}
    }

  /* Now pick off the fundamental type.  There can be only one. */

  switch (**mangled)
    {
      case '\0':
      case '_':
	break;
      case 'v':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "void");
	break;
      case 'x':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "long long");
	break;
      case 'l':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "long");
	break;
      case 'i':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "int");
	break;
      case 's':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "short");
	break;
      case 'b':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "bool");
	break;
      case 'c':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "char");
	break;
      case 'w':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "wchar_t");
	break;
      case 'r':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "long double");
	break;
      case 'd':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "double");
	break;
      case 'f':
	(*mangled)++;
	APPEND_BLANK (result);
	string_append (result, "float");
	break;
      case 'G':
	(*mangled)++;
	if (!isdigit (**mangled))
	  {
	    success = 0;
	    break;
	  }
	/* fall through */
      /* An explicit type, such as "6mytype" or "7integer" */
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	APPEND_BLANK (result);
	if (!demangle_class_name (work, mangled, result)) {
	  --result->p;
	  success = 0;
	}
	break;
      case 't':
        success = demangle_template(work,mangled, result, 0);
        break;
      default:
	success = 0;
	break;
      }

  return (success);
}

/* `result' will be initialized in do_type; it will be freed on failure */

static int
do_arg (work, mangled, result)
     struct work_stuff *work;
     const char **mangled;
     string *result;
{
  const char *start = *mangled;

  if (!do_type (work, mangled, result))
    {
      return (0);
    }
  else
    {
      remember_type (work, start, *mangled - start);
      return (1);
    }
}

static void
remember_type (work, start, len)
     struct work_stuff *work;
     const char *start;
     int len;
{
  char *tem;

  if (work -> ntypes >= work -> typevec_size)
    {
      if (work -> typevec_size == 0)
	{
	  work -> typevec_size = 3;
	  work -> typevec =
	    (char **) malloc (sizeof (char *) * work -> typevec_size);
	}
      else
	{
	  work -> typevec_size *= 2;
	  work -> typevec =
	    (char **) realloc ((char *)work -> typevec,
				sizeof (char *) * work -> typevec_size);
	}
    }
  tem = malloc (len + 1);
  memcpy (tem, start, len);
  tem[len] = '\0';
  work -> typevec[work -> ntypes++] = tem;
}

/* Forget the remembered types, but not the type vector itself. */

static void
forget_types (work)
     struct work_stuff *work;
{
  int i;

  while (work -> ntypes > 0)
    {
      i = --(work -> ntypes);
      if (work -> typevec[i] != NULL)
	{
	  free (work -> typevec[i]);
	  work -> typevec[i] = NULL;
	}
    }
}

/* Process the argument list part of the signature, after any class spec
   has been consumed, as well as the first 'F' character (if any).  For
   example:

   "__als__3fooRT0"		=>	process "RT0"
   "complexfunc5__FPFPc_PFl_i"	=>	process "PFPc_PFl_i"

   DECLP must be already initialised, usually non-empty.  It won't be freed
   on failure.

   Note that g++ differs significantly from ARM and lucid style mangling
   with regards to references to previously seen types.  For example, given
   the source fragment:

     class foo {
       public:
       foo::foo (int, foo &ia, int, foo &ib, int, foo &ic);
     };

     foo::foo (int, foo &ia, int, foo &ib, int, foo &ic) { ia = ib = ic; }
     void foo (int, foo &ia, int, foo &ib, int, foo &ic) { ia = ib = ic; }

   g++ produces the names:

     __3fooiRT0iT2iT2
     foo__FiR3fooiT1iT1

   while lcc (and presumably other ARM style compilers as well) produces:

     foo__FiR3fooT1T2T1T2
     __ct__3fooFiR3fooT1T2T1T2

   Note that g++ bases it's type numbers starting at zero and counts all
   previously seen types, while lucid/ARM bases it's type numbers starting
   at one and only considers types after it has seen the 'F' character
   indicating the start of the function args.  For lucid/ARM style, we
   account for this difference by discarding any previously seen types when
   we see the 'F' character, and subtracting one from the type number
   reference.

 */

static int
demangle_args (work, mangled, declp)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
{
  string arg;
  int need_comma = 0;
  int r;
  int t;
  const char *tem;
  char temptype;

  if (PRINT_ARG_TYPES)
    {
      string_append (declp, "(");
      if (**mangled == '\0')
	{
	  string_append (declp, "void");
	}
    }

  while (**mangled != '_' && **mangled != '\0' && **mangled != 'e')
    {
      if ((**mangled == 'N') || (**mangled == 'T'))
	{
	  temptype = *(*mangled)++;
	  
	  if (temptype == 'N')
	    {
	      if (!get_count (mangled, &r))
		{
		  return (0);
		}
	    }
	  else
	    {
	      r = 1;
	    }
          if (ARM_DEMANGLING && work -> ntypes >= 10)
            {
              /* If we have 10 or more types we might have more than a 1 digit
                 index so we'll have to consume the whole count here. This
                 will lose if the next thing is a type name preceded by a
                 count but it's impossible to demangle that case properly
                 anyway. Eg if we already have 12 types is T12Pc "(..., type1,
                 Pc, ...)"  or "(..., type12, char *, ...)" */
              if ((t = consume_count(mangled)) == 0)
                {
                  return (0);
                }
            }
          else
	    {
	      if (!get_count (mangled, &t))
	    	{
	          return (0);
	    	}
	    }
	  if (LUCID_DEMANGLING || ARM_DEMANGLING)
	    {
	      t--;
	    }
	  /* Validate the type index.  Protect against illegal indices from
	     malformed type strings. */
	  if ((t < 0) || (t >= work -> ntypes))
	    {
	      return (0);
	    }
	  while (--r >= 0)
	    {
	      tem = work -> typevec[t];
	      if (need_comma && PRINT_ARG_TYPES)
		{
		  string_append (declp, ", ");
		}
	      if (!do_arg (work, &tem, &arg))
		{
		  return (0);
		}
	      if (PRINT_ARG_TYPES)
		{
		  string_appends (declp, &arg);
		}
	      string_delete (&arg);
	      need_comma = 1;
	    }
	}
      else
	{
	  if (need_comma & PRINT_ARG_TYPES)
	    {
	      string_append (declp, ", ");
	    }
	  if (!do_arg (work, mangled, &arg))
	    {
	      return (0);
	    }
	  if (PRINT_ARG_TYPES)
	    {
	      string_appends (declp, &arg);
	    }
	  string_delete (&arg);
	  need_comma = 1;
	}
    }

  if (**mangled == 'e')
    {
      (*mangled)++;
      if (PRINT_ARG_TYPES)
	{
	  if (need_comma)
	    {
	      string_append (declp, ",");
	    }
	  string_append (declp, "...");
	}
    }

  if (PRINT_ARG_TYPES)
    {
      string_append (declp, ")");
    }
  return (1);
}

static void
demangle_function_name (work, mangled, declp, scan)
     struct work_stuff *work;
     const char **mangled;
     string *declp;
     const char *scan;
{
  int i;
  int len;
  string type;
  const char *tem;

  string_appendn (declp, (*mangled), scan - (*mangled));
  string_need (declp, 1);
  *(declp -> p) = '\0';

  /* Consume the function name, including the "__" separating the name
     from the signature.  We are guaranteed that SCAN points to the
     separator. */

  (*mangled) = scan + 2;

  if (LUCID_DEMANGLING || ARM_DEMANGLING)
    {

      /* See if we have an ARM style constructor or destructor operator.
	 If so, then just record it, clear the decl, and return.
	 We can't build the actual constructor/destructor decl until later,
	 when we recover the class name from the signature. */

      if (strcmp (declp -> b, "__ct") == 0)
	{
	  work -> constructor += 1;
	  string_clear (declp);
	  return;
	}
      else if (strcmp (declp -> b, "__dt") == 0)
	{
	  work -> destructor += 1;
	  string_clear (declp);
	  return;
	}
    }

  if (declp->p - declp->b >= 3 
      && declp->b[0] == 'o'
      && declp->b[1] == 'p'
      && strchr (cplus_markers, declp->b[2]) != NULL)
    {
      /* see if it's an assignment expression */
      if (declp->p - declp->b >= 10 /* op$assign_ */
	  && memcmp (declp->b + 3, "assign_", 7) == 0)
	{
	  for (i = 0; i < sizeof (optable) / sizeof (optable[0]); i++)
	    {
	      len = declp->p - declp->b - 10;
	      if (strlen (optable[i].in) == len
		  && memcmp (optable[i].in, declp->b + 10, len) == 0)
		{
		  string_clear (declp);
		  string_append (declp, "operator");
		  string_append (declp, optable[i].out);
		  string_append (declp, "=");
		  break;
		}
	    }
	}
      else
	{
	  for (i = 0; i < sizeof (optable) / sizeof (optable[0]); i++)
	    {
	      int len = declp->p - declp->b - 3;
	      if (strlen (optable[i].in) == len 
		  && memcmp (optable[i].in, declp->b + 3, len) == 0)
		{
		  string_clear (declp);
		  string_append (declp, "operator");
		  string_append (declp, optable[i].out);
		  break;
		}
	    }
	}
    }
  else if (declp->p - declp->b >= 5 && memcmp (declp->b, "type", 4) == 0
	   && strchr (cplus_markers, declp->b[4]) != NULL)
    {
      /* type conversion operator */
      tem = declp->b + 5;
      if (do_type (work, &tem, &type))
	{
	  string_clear (declp);
	  string_append (declp, "operator ");
	  string_appends (declp, &type);
	  string_delete (&type);
	}
    }
  else if (declp->b[0] == '_' && declp->b[1] == '_'
	  && declp->b[2] == 'o' && declp->b[3] == 'p')
    {
      /* ANSI.  */
      /* type conversion operator.  */
      tem = declp->b + 4;
      if (do_type (work, &tem, &type))
	{
	  string_clear (declp);
	  string_append (declp, "operator ");
	  string_appends (declp, &type);
	  string_delete (&type);
	}
    }
  else if (declp->b[0] == '_' && declp->b[1] == '_'
	   && declp->b[2] >= 'a' && declp->b[2] <= 'z'
	   && declp->b[3] >= 'a' && declp->b[3] <= 'z')
    {
      if (declp->b[4] == '\0')
	{
	  /* Operator.  */
	  for (i = 0; i < sizeof (optable) / sizeof (optable[0]); i++)
	    {
	      if (strlen (optable[i].in) == 2
		  && memcmp (optable[i].in, declp->b + 2, 2) == 0)
		{
		  string_clear (declp);
		  string_append (declp, "operator");
		  string_append (declp, optable[i].out);
		  break;
		}
	    }
	}
      else
	{
	  if (declp->b[2] == 'a' && declp->b[5] == '\0')
	    {
	      /* Assignment. */
	      for (i = 0; i < sizeof (optable) / sizeof (optable[0]); i++)
		{
		  if (strlen (optable[i].in) == 3
		      && memcmp (optable[i].in, declp->b + 2, 3) == 0)
		    {
		      string_clear (declp);
		      string_append (declp, "operator");
		      string_append (declp, optable[i].out);
		      break;
		    }		      
		}
	    }
	}
    }
}

/* a mini string-handling package */

static void
string_need (s, n)
     string *s;
     int n;
{
  int tem;

  if (s->b == NULL)
    {
      if (n < 32)
	{
	  n = 32;
	}
      s->p = s->b = malloc (n);
      s->e = s->b + n;
    }
  else if (s->e - s->p < n)
    {
      tem = s->p - s->b;
      n += tem;
      n *= 2;
      s->b = realloc (s->b, n);
      s->p = s->b + tem;
      s->e = s->b + n;
    }
}

static void
string_delete (s)
     string *s;
{
  if (s->b != NULL)
    {
      free (s->b);
      s->b = s->e = s->p = NULL;
    }
}

static void
string_init (s)
     string *s;
{
  s->b = s->p = s->e = NULL;
}

static void 
string_clear (s)
     string *s;
{
  s->p = s->b;
}

#if 0

static int
string_empty (s)
     string *s;
{
  return (s->b == s->p);
}

#endif

static void
string_append (p, s)
     string *p;
     const char *s;
{
  int n;
  if (s == NULL || *s == '\0')
    return;
  n = strlen (s);
  string_need (p, n);
  memcpy (p->p, s, n);
  p->p += n;
}

static void
string_appends (p, s)
     string *p, *s;
{
  int n;

  if (s->b != s->p)
    {
      n = s->p - s->b;
      string_need (p, n);
      memcpy (p->p, s->b, n);
      p->p += n;
    }
}

static void
string_appendn (p, s, n)
     string *p;
     const char *s;
     int n;
{
  if (n != 0)
    {
      string_need (p, n);
      memcpy (p->p, s, n);
      p->p += n;
    }
}

static void
string_prepend (p, s)
     string *p;
     const char *s;
{
  if (s != NULL && *s != '\0')
    {
      string_prependn (p, s, strlen (s));
    }
}

static void
string_prepends (p, s)
     string *p, *s;
{
  if (s->b != s->p)
    {
      string_prependn (p, s->b, s->p - s->b);
    }
}

static void
string_prependn (p, s, n)
     string *p;
     const char *s;
     int n;
{
  char *q;

  if (n != 0)
    {
      string_need (p, n);
      for (q = p->p - 1; q >= p->b; q--)
	{
	  q[n] = q[0];
	}
      memcpy (p->b, s, n);
      p->p += n;
    }
}
