/* display.c - display an Edit_buffer in an abstract rendering area */

/*  Copyright 1994 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char edit_display_c_sccsid[] = "@(#)display.c	1.10 09 Sep 1994 (UKC)";

#include <time.h>
#include <stdlib.h>

#include <local/ukcprog.h>

#include "edit.h"
#include "textbuf.h"
#include "display.h"
#include "props.h"

static void adjust_for_delete PROTO((size_t pos, size_t len, size_t *p_point));
static size_t truncate_point_to_buflen PROTO((Edit_buffer *buffer, size_t pos));
static void clear_display PROTO((Edit_display *d));
static void before_change PROTO((Edit_display *d));
static void after_change PROTO((Edit_display *d, bool track_cursor));


static Edit_display *All_displays = NULL;

void
edit_close_display(d)
Edit_display *d;
{
	Edit_display *prev, *d2;

	prev = NULL;
	
	for (d2 = All_displays; d2 != NULL; d2 = d2->next) {
		if (d2 == d)
			break;
		prev = d2;
	}

	if (d2 == NULL)
		panic("display not found in ecd");

	if (prev == NULL)
		All_displays = d->next;
	else
		prev->next = d->next;

	(*d->display_ops->destroy)(d);

	free((char *)d);
}

static void
clear_display(d)
Edit_display *d;
{
	(*d->render_ops->set_area)(d->render_data, 0, 0,
				   d->display_width, d->display_height, FALSE);
}

void
edit_redraw_display(d)
Edit_display *d;
{
	before_change(d);
	
	if (d->buffer == NULL)
		clear_display(d);
	else
		(*d->display_ops->redraw_display)(d);
	
	after_change(d, FALSE);
}

void
edit_display_from(d, pixel_offset, point)
Edit_display *d;
int pixel_offset;
size_t point;
{
	before_change(d);

	if (d->buffer == NULL) {
		clear_display(d);
	}
	else {
		point = truncate_point_to_buflen(d->buffer, point);
		(*d->display_ops->display_from)(d, pixel_offset, point);
	}

	after_change(d, FALSE);
}

void
edit_get_display_info(d, p_pixel_offset, p_start_point, p_lim_point)
Edit_display *d;
int *p_pixel_offset;
size_t *p_start_point, *p_lim_point;
{
	(*d->display_ops->get_display_info)(d, p_pixel_offset,
					    p_start_point, p_lim_point);
}

bool
edit_point_to_pixel(d, point, p_x, p_y, p_width, p_height, p_baseline)
Edit_display *d;
size_t point;
int *p_x, *p_y, *p_width, *p_height, *p_baseline;
{
	return (*d->display_ops->point_to_pixel)(d, point, p_x, p_y,
						 p_width, p_height,
						 p_baseline);
}

int
edit_scroll_display(d, delta)
Edit_display *d;
int delta;
{
	int res;

	if (delta == 0 || d->display_ops->scroll_display == NULL)
		return 0;
	
	(*d->render_ops->set_updating)(d->render_data, FALSE);
	res = (*d->display_ops->scroll_display)(d, delta);
	(*d->render_ops->set_updating)(d->render_data, TRUE);
	
	return res;
}

bool
edit_pixel_to_point(d, x, y, p_point)
Edit_display *d;
int x, y;
size_t *p_point;
{
	return (*d->display_ops->pixel_to_point)(d, x, y, p_point);
}

bool
edit_ensure_visible(d, point)
Edit_display *d;
size_t point;
{
	point = truncate_point_to_buflen(d->buffer, point);

	if ((*d->display_ops->visible)(d, point))
		return FALSE;
	
	before_change(d);
	(*d->display_ops->redraw_with_point_visible)(d, point);
	after_change(d, FALSE);

	return TRUE;
}

bool
edit_set_display_size(d, width, height)
Edit_display *d;
int width, height;
{
	if (!(*d->display_ops->update_display_size)(d, width, height))
		return FALSE;

	d->display_width = width;
	d->display_height = height;

	return TRUE;
}

void
edit_get_display_size(d, p_width, p_height)
Edit_display *d;
int *p_width, *p_height;
{
	*p_width = d->display_width;
	*p_height = d->display_height;
}

Edit_display *
edit_create_display(want_cursor, keep_cursor_visible, render_data, render_ops)
bool want_cursor, keep_cursor_visible;
char *render_data;
Edit_render_ops *render_ops;
{
	Edit_display *d;

	d = (Edit_display *)e_malloc(sizeof(Edit_display));
	d->keymap = edit_get_default_keymap();
	d->buffer = NULL;
	d->point = 0;
	d->mark = 0;
	d->start = 0;
	d->lim = 0;
	d->goal_column = 0;
	d->render_data = render_data;
	d->render_ops = render_ops;
	d->render_context = NULL;
	d->display_width = 0;
	d->display_height = 0;
	d->display_data = NULL;
	d->display_ops = NULL;
	d->want_cursor = want_cursor;
	d->keep_cursor_visible = keep_cursor_visible;
	d->quit_requested = FALSE;
	d->last_key = 0;
	d->last_modifiers = 0;
	d->user_data = NULL;
	d->next = All_displays;
	All_displays = d;

	return d;
}

void
edit_set_user_data(d, user_data)
Edit_display *d;
char *user_data;
{
	d->user_data = user_data;
}

char *
edit_get_user_data(d)
Edit_display *d;
{
	return d->user_data;
}

void
edit_get_render_data_and_ops(d, p_render_data, p_render_ops)
Edit_display *d;
char **p_render_data;
Edit_render_ops **p_render_ops;
{
	*p_render_data = d->render_data;
	*p_render_ops = d->render_ops;
}

bool
edit_insert(buffer, pos, text, len, update_proplist)
Edit_buffer *buffer;
size_t pos;
const char *text;
size_t len;
bool update_proplist;
{
	Edit_display *d;
	
	if (buffer->textbuf->ops->insert == NULL) {
		errf("Buffer is read-only");
		return FALSE;
	}
	
	for (d = All_displays; d != NULL; d = d->next) {
		if (d->buffer == buffer) {
			before_change(d);
			
			if (d->display_ops->about_to_insert != NULL)
				(*d->display_ops->about_to_insert)(d, pos, len);

			if (pos <= d->point)
				d->point += len;
			if (pos <= d->lim)
				d->lim += len;
			
			if (pos < d->mark)
				d->mark += len;
			if (pos < d->start)
				d->start += len;
		}
	}

	if (update_proplist)
	  edit__update_proplist_points(buffer->proplist, pos, (long)len);

	(*buffer->textbuf->ops->insert)(buffer->textbuf->bufdata,
					pos, text, len);
	
	for (d = All_displays; d != NULL; d = d->next) {
		if (d->buffer == buffer) {
			(*d->display_ops->done_insert)(d, pos, len);
			after_change(d, TRUE);
		}
	}

	return TRUE;
}

static void
before_change(d)
Edit_display *d;
{
	(*d->render_ops->set_updating)(d->render_data, FALSE);

	if (d->buffer != NULL &&
	    d->want_cursor && (*d->display_ops->visible)(d, d->point)) {
		(*d->display_ops->draw_cursor)(d, d->point, FALSE);
	}
}

static void
after_change(d, track_cursor)
Edit_display *d;
bool track_cursor;
{
	if (d->buffer != NULL &&
	    track_cursor &&
	    d->want_cursor &&
	    d->keep_cursor_visible &&
	    !(*d->display_ops->visible)(d, d->point)) {
		(*d->display_ops->redraw_with_point_visible)(d, d->point);
	}

	if (d->buffer != NULL &&
	    d->want_cursor && (*d->display_ops->visible)(d, d->point)) {
		(*d->display_ops->draw_cursor)(d, d->point, TRUE);
	}
	
	(*d->render_ops->set_updating)(d->render_data, TRUE);
}

static void
adjust_for_delete(pos, len, p_point)
size_t pos, len, *p_point;
{
	if (*p_point >= pos) {
		if (*p_point < pos + len)
			*p_point = pos;
		else
			*p_point -= len;
	}
}

bool
edit_delete(buffer, pos, len)
Edit_buffer *buffer;
size_t pos, len;
{
	Edit_display *d;

	if (buffer->textbuf->ops->delete == NULL) {
		errf("Buffer is read-only");
		return FALSE;
	}
	
	for (d = All_displays; d != NULL; d = d->next) {
		if (d->buffer == buffer) {
			before_change(d);
			
			if (d->display_ops->about_to_delete != NULL)
				(*d->display_ops->about_to_delete)(d, pos, len);

			adjust_for_delete(pos, len, &d->point); 
			adjust_for_delete(pos, len, &d->mark); 
			adjust_for_delete(pos, len, &d->start); 
			adjust_for_delete(pos, len, &d->lim); 
		}
	}

	edit__update_proplist_points(buffer->proplist, pos, -(long)len);

	(*buffer->textbuf->ops->delete)(buffer->textbuf->bufdata, pos, len);
	
	for (d = All_displays; d != NULL; d = d->next) {
		if (d->buffer == buffer) {
			(*d->display_ops->done_delete)(d, pos, len);
			after_change(d, TRUE);
		}
	}

	return TRUE;
}

void
edit_set_buffer(d, buffer)
Edit_display *d;
Edit_buffer *buffer;
{
	d->buffer = buffer;
	d->point = 0;
	d->mark = 0;
	d->start = 0;
	d->lim = edit_get_buffer_length(buffer);

	(*d->display_ops->note_new_buffer)(d);
}

Edit_buffer *
edit_get_buffer(d)
Edit_display *d;
{
	return d->buffer;
}

void
edit_set_keymap(d, keymap)
Edit_display *d;
Edit_keymap *keymap;
{
	d->keymap = keymap;
}

Edit_keymap *
edit_get_keymap(d)
Edit_display *d;
{
	return d->keymap;
}

size_t
edit_set_point(d, point)
Edit_display *d;
size_t point;
{
	point = truncate_point_to_buflen(d->buffer, point);
	
	if (d->want_cursor) {
		if ((*d->display_ops->visible)(d, d->point))
			(*d->display_ops->draw_cursor)(d, d->point, FALSE);

		if ((*d->display_ops->visible)(d, point))
			(*d->display_ops->draw_cursor)(d, point, TRUE);
	}
	    
	d->point = point;
	
	return point;
}

void
edit_move_point(d, point)
Edit_display *d;
size_t point;
{
	edit_ensure_visible(d, edit_set_point(d, point));
}

static size_t
truncate_point_to_buflen(buffer, pos)
Edit_buffer *buffer;
size_t pos;
{
	size_t buflen;

	buflen = edit_get_buffer_length(buffer);

	if (pos > buflen)
		pos = buflen;

	return pos;
}

void
edit_set_mark(d, mark)
Edit_display *d;
size_t mark;
{
	d->mark = truncate_point_to_buflen(d->buffer, mark);
}

size_t
edit_get_mark(d)
Edit_display *d;
{
	return d->mark;
}

void
edit_set_point_limits(d, start, lim)
Edit_display *d;
size_t start, lim;
{
	if (start <= lim) {
		d->start = truncate_point_to_buflen(d->buffer, start);
		d->lim = truncate_point_to_buflen(d->buffer, lim);
	}
	else {
		d->start = truncate_point_to_buflen(d->buffer, lim);
		d->lim = truncate_point_to_buflen(d->buffer, start);
	}
}

void
edit_get_point_limits(d, p_start, p_lim)
Edit_display *d;
size_t *p_start, *p_lim;
{
	*p_start = d->start;
	*p_lim = d->lim;
}

bool
edit_set_want_cursor(d, want_cursor)
Edit_display *d;
bool want_cursor;
{
	bool had_cursor;

	had_cursor = d->want_cursor;
	d->want_cursor = want_cursor;

	if (had_cursor != want_cursor &&
	    (*d->display_ops->visible)(d, d->point)) {
		(*d->display_ops->draw_cursor)(d, d->point, want_cursor);
	}

	return had_cursor;
}

size_t
edit_get_point(d)
Edit_display *d;
{
	return d->point;
}

Edit_display *
edit__get_display_list()
{
	return All_displays;
}
