/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "layer.h"
#include "lay_msg.h"

char			*lay_msg_list[] =
{
  "Message 0",					/* 0	*/
  "Get class",					/* 1	*/
  "Name yourself",				/* 2	*/
  "Get offset to sub layer",			/* 3	*/
  "Get sub layer",				/* 4	*/
  "Checksum",					/* 5	*/
  "Get field",					/* 6	*/
  "Set field",					/* 7	*/
  "Get fields",					/* 8	*/
  "Get html template",				/* 9	*/
  "Has option",					/* 10	*/
  "Adapt length",				/* 11	*/
  "Get field typ",				/* 12	*/
  "Get sub layer with id",			/* 13	*/
}; 

/*#define DEBUG_GET_FIELD	1*/
/*#define DEBUG_OFF		1*/


/* calls a "lay" msg_proc.
   Instead of calling msg_proc directly, we use this function as it includes
   useful method debug and trace features.
   Returns 0 if OK. Might return various errors */
t_status			lay_msg(mp,msg,arg1,arg2)
t_msg_proc			mp;	/* Indirected msg_proc */
t_msg				msg;	/* Message number */
VOID_PTR			arg1;	/* Query */
VOID_PTR			arg2;	/* Reply */
{
#ifdef DEBUG
  if (LAYER_VERB(VERB_LAY_MSG))
    {
      t_status			status;
      char			name[STR_BUFSIZ];
      int			len;
      t_bridled_str		namebs;

      namebs.str = name;
      namebs.max_len = sizeof (name);
      name[0] = 0;
      status = mp(LAY_NAME,
		  NULL,
		  &namebs);
      assert(status == 0);
      fprintf(stderr,"lay: %s::%s(%p,%p) ",
	      name,
	      lay_msg_list[msg],
	      arg1,
	      arg2);
#ifdef DEBUG_GET_FIELD
      if (msg == LAY_GET_FIELD)
	{
	  LAY_GET_FIELD_ARGS(gfd,bs);
	  
	  fprintf(stderr,"field=%s ",gfd->field);
	  len = strlen(bs->str);
	}
#endif
      status = mp(msg,arg1,arg2);
      fprintf(stderr,"%s",err_msg(-status));
      if (ERR_SYS(-status))
	fprintf(stderr,": %s",strerror(errno));
      fprintf(stderr,"\n");
      if (status < 0)
	return (status);
#ifdef DEBUG_GET_FIELD
      if (msg == LAY_GET_FIELD)
	{
	  LAY_GET_FIELD_ARGS(gfd,bs);
	  
	  fprintf(stderr,"value=%s\n",bs->str + len);
	}
#endif
#ifdef DEBUG_OFF
      if (msg == LAY_OFF)
	{
	  LAY_OFF_ARGS(b,off);
	  
	  fprintf(stderr,"off=%u\n",*off);
	}
#endif
      return (status);
    }
  else
#endif
    return (mp(msg,arg1,arg2));
}

/* is a t_tmpl_do_proc.
   It transforms %i% to the value of i.
   See also lay_get_itmpl_generic(3).
   Returns 0 if OK. Might return various errors. */
t_status			itmpl_format_do(bs,var,i)
t_bridled_str			*bs;
char				*var;
int				i;
{
  int				varlen;
  long				signedvalue;
  t_status			status;

  varlen = strlen(var);
  if (varlen == 0)
    return (str_cat_char(bs->str,bs->max_len,'%'));
  else
    {
      if (!strcmp(var,"i"))
	{
	  return (long_to_str((signed long)i,
			      10,
			      bs->str,
			      bs->max_len));
	}
    }
  return (0);
}

/* formats an "itmpl" to a "tmpl".
   It is used mostly to allow indexation of layers.
   Returns 0 if OK. Might return various errors. */
t_status			itmpl_format(i,itmpl_str,str,max_len)
int				i;
char				*itmpl_str;
char				*str;
int				max_len;
{
  return (tmpl_str_to_str(itmpl_str,
			  (t_tmpl_do_proc)itmpl_format_do,
			  (VOID_PTR)i,
			  str,
			  max_len));
}

/* gets a field from a buffer according a specification (t_field).
   Returns 0 if OK. -ERR_TRUNC if field offset is greater that buffer len.
   Might return various other errors (according to sub methods). */
t_status			get_field_to_str(buf,
						 len,
						 field,
						 str,
						 max_len)
char				*buf;
int				len;
t_field				*field; /* Field specification */
char				*str;
int				max_len;
{
  t_status			status;
  t_extract_data		ed;
  t_bridled_str			bs;

  if (field->offset >= len)
    return (-ERR_TRUNC);
  ed.b.buf = buf + field->offset;
  ed.b.len = len - field->offset;
  ed.data = field->data;
  bs.str = str;
  bs.max_len = max_len;
  if ((status = typ_msg(field->mp,
			TYP_EXTRACT,
			&ed,
			&bs)) < 0)
    return (status);
  return (0);
}

/* gets a field to a buffer according a specification (t_field).
   Returns 0 if OK. -ERR_TRUNC if field offset is greater that buffer len.
   Might return various other errors (according to sub methods). */
t_status			set_field_from_str(buf,
						   len,
						   field,
						   str)
char				*buf;
int				len;
t_field				*field; /* Field specification */
char				*str; 
{
  t_status			status;
  t_extract_data		ed;

  if (field->offset >= len)
    return (-ERR_TRUNC);
  ed.b.buf = buf + field->offset;
  ed.b.len = len - field->offset;
  ed.data = field->data;
  if ((status = typ_msg(field->mp,
			TYP_INSERT,
			&ed,
			str)) < 0)
    return (status);
  return (0);
}

/* generic MSG_CLASS method for "lay" classes.
   Appends "lay" class name to bridled string.
   Returns 0 if OK. -ERR_BO on buffer overflows. */
t_status		lay_class_generic(bs)
t_bridled_str		*bs;
{
  return (str_cat_str(bs->str,
		      bs->max_len,
		      "lay"));
}

/* generic LAY_NAME method for "lay". 
   It is called when class doesn't manage indexation.
   See lay_name_id_generic(3) for an advanced version.
   Appends layer name to bridled string.
   Returns 0 if OK. -ERR_BO on buffer overflows. */
t_status		lay_name_generic(bs,name)
t_bridled_str		*bs;
char			*name;
{
  return (str_cat_str(bs->str,
		      bs->max_len,
		      name));
}

/* generic advanced LAY_NAME method for "lay". 
   It is called when class manages indexation.
   Appends layer name to bridled string according to the value specified
   in channel. E.g "ip[3]".
   Returns 0 if OK. -ERR_BO on buffer overflows. */
t_status		lay_name_id_generic(optional_id,bs,chan,name)
t_id			*optional_id;	/* Might be NULL in such case
					   name won't be indexed and
					   output will look like "ip" instead
					   of "ip[something]" */
t_bridled_str		*bs;		/* Bridled destination string */
VOID_PTR		chan;		/* Channel, e.g address of
					   layer msg_proc: &lay_ip_msg */
char			*name;		/* Base name of layer, e.g "ip" */
{
  if (optional_id)
    {
      t_hash_elt	*he;
      int		i;
      t_status		status;
      
      if (he = id_get(optional_id,
		      chan))
	i = (int)(he->value);
      else
        i = 0;
      if ((status = str_cat_fmt_va(bs->str,
				   bs->max_len,
				   "%s[%d]",
				   name,
				   i)) < 0)
        return (status);
      if ((status = id_override(optional_id,
				chan,
				(VOID_PTR)(++i))) < 0)
        return (status);
      return (0);
   }
   else
     return (str_cat_str(bs->str,
			 bs->max_len,
			 name));
}

/* generic LAY_GET_FIELD method.
   Retrieves field value to a bridled string.
   Returns 0 if OK. -ERR_NOENT if field doesn't exist. Might return
   various other errors. */
t_status		lay_get_field_generic(gfd,bs,fields) 
t_get_field_data	*gfd;
t_bridled_str		*bs;	 /* Destination bridled string */
t_field			*fields; /* NULL terminated array */
{
  t_field		*field;
  
  field = NULL;
  while (fields->name)
    {
      if (!strcmp(fields->name,gfd->field))
	field = fields; 
      fields++;
    }
  if (!field)
    return (-ERR_NOENT);
  return (get_field_to_str(gfd->b.buf,
			   gfd->b.len,
			   field,
			   bs->str,
			   bs->max_len));
}

/* generic LAY_SET_FIELD method.
   Sets field value from a string.
   Returns 0 if OK. -ERR_NOENT if field doesn't exist. Might return
   various other errors. */
t_status		lay_set_field_generic(gfd,str,fields) 
t_get_field_data	*gfd;
char			*str;	 /* Value string */	
t_field			*fields; /* NULL terminated array */
{
  t_field		*field;
  
  field = NULL;
  while (fields->name)
    {
      if (!strcmp(fields->name,gfd->field))
	field = fields;
      fields++;
    }
  if (!field)
    return (-ERR_NOENT);
  return (set_field_from_str(gfd->b.buf,
			     gfd->b.len,
			     field,
			     str));
}

/* generic LAY_GET_FIELDS method.
   Catenates fields name to a vector of strings.
   Returns 0 if OK. Might return various errors */
t_status		lay_get_fields_generic(b,vec,fields)
t_buf			*b;
t_vec			*vec; /* Vector of strings */
t_field			*fields; /* NULL terminated array */ 
{
  t_status		status;
  
  while (fields->name)
    {
      if ((status = vec_str_add(vec,
				fields->name)) < 0)
	return (status);
      fields++;
    }
  return (0);
}

/* generic LAY_GET_TMPL method.
   Catenates an HTML template to a bridled string.
   Note: it is intended to use with layers which doesn't manage
   indexation. See also lay_get_itmpl_generic(3).
   Returns 0 if OK. -ERR_BO on buffer overflows. */
t_status		lay_get_tmpl_generic(bs,tmpl) 
t_bridled_str		*bs;	/* Destination bridled string */
char			*tmpl;	/* HTML template */
{
  return (str_cat_str(bs->str,
		      bs->max_len,
		      tmpl));
}

/* generic complex LAY_GET_TMPL method.
   This is the advanced version of lay_get_tmpl_generic. Before
   catenating template, it converts it in using itmpl_format(3). This
   allow a pre-substitution of layer indexes.
   Returns 0 if OK. Might return various errors */
t_status		lay_get_itmpl_generic(gtd,bs,chan,itmpl) 
t_get_tmpl_data		*gtd;
t_bridled_str		*bs;
VOID_PTR		chan;		/* Channel e.g &lay_ip_msg */
char			*itmpl;		/* I template */
{
  t_hash_elt		*he;
  int			i;
  t_status		status;

  if (he = id_get(gtd->id,
		  chan))
    i = (int)(he->value);
  else
    i = 0;
  if ((status = itmpl_format(i,
			     itmpl,
			     bs->str,
			     bs->max_len)) < 0)
     return (status);
  if ((status = id_override(gtd->id,
			    chan,
			    (VOID_PTR)(++i))) < 0)
    return (status);
  return (0);
}

/* generic LAY_GET_FIELD_TYP method.
   Retrieves specified field "typ" msg_proc and data.
   It is intended to be used by callers which want to know more about
   layer fields.
   Returns 0 if OK. Might return various errors. */
t_status		lay_get_field_typ_generic(gfd,gftd,fields) 
t_get_field_data	*gfd;
t_get_field_typ_data	*gftd;
t_field			*fields; /* NULL terminated array */
{
  while (fields->name)
    {
      if (!strcmp(gfd->field,fields->name))
	{
	  gftd->mp = fields->mp;
	  gftd->data = fields->data;
	  return (0);
	}
      fields++;
    }
  return (-ERR_NOENT);
}
