/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h> 
#include <sys/socket.h> 
#include <sys/stat.h> 
#include <netinet/in.h>
#include <ctype.h>
#include <fcntl.h>
#include <time.h>
#include "pkt.h"
#include "typ_lay.h"
#include "typ_inaddr.h"
#include "typ_subnet.h"

int			thiszone = 0; 

/* instantiates a new packet structure.
   It allocates the buffer to the specified len with the specified 
   methods. It also allocates the pkt structure and fills correct 
   fields. The timestamp is set to zero. The netlen is set to len.
   Returns a pkt or NULL if it fails (so check status). */ 
t_pkt			*pkt_new(len,mp,alloc_proc,free_proc,status)
int			len;		/* Length of the packet buffer */
t_msg_proc		mp;		/* Lay msg_proc */
t_alloc_proc		alloc_proc;	/* Alloc method */
t_free_proc		free_proc;	/* Free method */
t_status		*status;	/* Filled if NULL */
{ 
  t_pkt			*npkt; 

  if ((npkt = alloc_proc(sizeof (t_pkt), 
			 "pkt", 
 			 "pkt_new:pkt", 
 			 status)) == NULL) 
    return (NULL); 
  if ((npkt->buf = alloc_proc(len * sizeof (char), 
 			      "pkt", 
 			      "pkt_new:buf", 
 			      status)) == NULL) 
    { 
      free_proc(npkt, 
 		"pkt", 
 		"pkt_new:pkt"); 
      return (NULL); 
    } 
  npkt->len = npkt->netlen = len; 
  npkt->mp = mp; 
  npkt->ts.tv_sec = 0; 
  npkt->ts.tv_usec = 0; 
  bzero(npkt->buf,npkt->len); 
  return (npkt); 
} 

/* retrieves the sub pkt of the specified packet (doesn't manage options).
   It performs a call to LAY_SUB to determine the sub layer then performs a 
   call to LAY_OFF to get the offset to sub layer, then fills the sub_pkt  
   structure. If the retrieved offset is equal to pkt->len, it returns 
   an error BUT fills sub_pkt anyway (it means that the packet is consistent 
   and no byte is wasted).
   For an advanced version of this function, check out pkt_sub_id(3).
   Returns 0 if OK. Returns -ERR_LOOP if offset is 0, -ERR_TRUNC if pkt->len  
   is inferior than offset. See the special case for -ERR_ZEROLEN above. It 
   might also return various errors such as -ERR_NOMETHOD (meaning
   that we have reached a final layer such as lay_data_msg(3)). */ 
t_status		pkt_sub(pkt,sub_pkt) 
t_pkt			*pkt;		/* The reference packet */ 
t_pkt			*sub_pkt;	/* The sub packet */ 
{ 
  t_status		status; 
  t_buf			b; 
  t_msg_proc		sub_mp; 
  t_off			off; 

  b.buf = pkt->buf; 
  b.len = pkt->len; 
  if ((status = lay_msg(pkt->mp, 
 			LAY_SUB, 
 			&b, 
 			&sub_mp)) < 0) 
    return (status); 
  sub_pkt->mp = sub_mp; 
  if ((status = lay_msg(pkt->mp, 
 			LAY_OFF, 
 			&b, 
 			&off)) < 0) 
    return (status); 
  if (pkt->len == off) 
    { 
      status = -ERR_ZEROLEN; 
      goto dosub; 
    } 
  if (off == 0) 
    return (-ERR_LOOP); 
  if (pkt->len < off) 
    return (-ERR_TRUNC); 
  status = 0; 
dosub: 
  sub_pkt->buf = pkt->buf + off; 
  sub_pkt->len = pkt->len - off; 
  sub_pkt->netlen = pkt->netlen; 
  sub_pkt->ts = pkt->ts; 
  return (status); 
}  

 /* tries to find the first sub packet matching the desired layer msg_proc. 
    It performs a call to pkt_sub(3) as many times as necessary. 
    If pkt->mp is desired_mp, then it returns immediately with success. 
    Note: advanced sub layer retrieving should be done using 
    pkt_subn_from_name(3). 
    Returns 0 if OK. See pkt_sub(3) for error messages. */ 
t_status		pkt_subn(pkt,subn_pkt,desired_mp) 
t_pkt			*pkt;		/* Reference packet */ 
t_pkt			*subn_pkt;	/* Result packet */ 
t_msg_proc		desired_mp;	/* Desired "lay" msg_proc */ 
{ 
  t_msg_proc		sub_mp; 
  t_status		status; 

  if (pkt->mp == desired_mp) 
    { 
      PKT_COPY(pkt,subn_pkt); 
      return (0); 
    } 
  if ((status = pkt_sub(pkt,subn_pkt)) < 0) 
    return (status); 
  else 
    { 
      t_pkt		tmp_pkt; 

      PKT_COPY(subn_pkt,&tmp_pkt); 
      return (pkt_subn(&tmp_pkt,subn_pkt,desired_mp)); 
    } 
} 

/* checksums the specified packet. 
   The checksum algorithm works as follow: It first tries to retrieve 
   the sub packet in using pkt_sub(3). If the status is -ERR_NOMETHOD or 
   -ERR_ZEROLEN, it returns immediately with success (meaning that the 
   checksum is finnish). If status is 0, it calls pkt_sum(3) recursively 
   whith the sub packet as the reference packet, the previous reference 
   packet becoming the upper packet. 
   When it pops (assuming a -ERR_METHOD or a -ERR_ZEROLEN is found above) 
   it performs a call to LAY_SUM. Checksums are so computed from the  
   last layer to the first one. 
   Returns 0 if OK. Might return various errors. */ 
t_status		pkt_sum(pkt,up_pkt) 
t_pkt			*pkt;		/* Reference packet */ 
t_pkt			*up_pkt;	/* Some layers need the upper 
 					   layer packet to perform their 
 					   checksum, e.g. udp.  
 					   This parameter might be NULL 
 					   (it must be from the top layer) */ 
{ 
  t_pkt			sub_pkt; 
  t_status		status; 

  if ((status = pkt_sub(pkt,&sub_pkt)) < 0) 
    { 
      if (status != -ERR_NOMETHOD && status != -ERR_ZEROLEN) 
 	return (status); 
      else 
 	return (0); 
    } 
  else 
    { 
      t_pkt		tmp_pkt; 
      t_sum_data	sd; 

      PKT_COPY(pkt,&tmp_pkt); 
      if ((status = pkt_sum(&sub_pkt,&tmp_pkt)) < 0) 
 	{ 
 	  if (status != -ERR_NOMETHOD) 
 	    return (status); 
 	} 
      sd.b.buf = pkt->buf; 
      sd.b.len = pkt->len; 
      if (up_pkt) 
 	{ 
 	  sd.up.buf = up_pkt->buf; 
 	  sd.up.len = up_pkt->len; 
 	} 
      else 
 	{ 
 	  sd.up.len = 0; 
 	} 
      if ((status = lay_msg(pkt->mp, 
 			    LAY_SUM, 
 			    &sd, 
 			    NULL)) < 0) 
 	{ 
 	  if (status != -ERR_NOMETHOD) 
 	    return (status); 
 	} 
      return (0); 
    } 
} 

/* duplicates a packet. 
   Return a new packet or NULL if it fails. */
t_pkt			*pkt_dup(pkt,alloc_proc,free_proc,status) 
t_pkt			*pkt;		/* Reference packet */
t_alloc_proc		alloc_proc;	/* Alloc method */ 
t_free_proc		free_proc;	/* Free method */ 
t_status		*status;	/* Filled if NULL */ 
{ 
  t_pkt			*npkt; 

  if ((npkt = alloc_proc(sizeof (t_pkt), 
 			 "pkt", 
 			 "pkt_dup:pkt", 
 			 status)) == NULL) 
    return (NULL); 
  PKT_COPY(pkt,npkt); 
  if ((npkt->buf = alloc_proc(npkt->len * sizeof (char), 
 			      "pkt", 
 			      "pkt_dup:buf", 
 			      status)) == NULL) 
    { 
      free_proc(npkt, 
 		"pkt", 
 		"pkt_dup:pkt"); 
      return (NULL); 
    } 
  bcopy(pkt->buf,npkt->buf,npkt->len * sizeof (char)); 
  return (npkt); 
}

 /* deletes a packet. 
    It frees the packet buffer and the structure. */ 
VOID_FUNC		pkt_delete(pkt,free_proc) 
t_pkt			*pkt;		 
t_free_proc		free_proc;	/* Free method */ 
{ 
  free_proc(pkt->buf, 
 	    "pkt", 
 	    "*:buf"); 
  free_proc(pkt, 
 	    "pkt", 
 	    "*:pkt"); 
} 

/* writes the full name of a packet to a bridled string.
   It is a series of LAY_NAME and pkt_sub(3) calls. 
   The result is in the form lay1.lay2. ... .layn.  
   e.g. ether.ip.udp.rip.ripinfo.ripinfo. 
   It gives an idea about the components of a packet. 
   Returns 0 if OK. Might return various errors */ 
t_status		pkt_full_name_to_str(pkt,str,max_len) 
t_pkt			*pkt;		 
char			*str;		/* A valid string */ 
int			max_len;	/* Maximen length */ 
{ 
  t_status		status; 
  t_pkt			sub_pkt; 
  t_pkt			tmp_pkt; 
  t_boolean		previous; 

  previous = FALSE; 
  PKT_COPY(pkt,&sub_pkt); 
  do  
    { 
      char		name[STR_BUFSIZ]; 
      t_bridled_str	namebs; 
      
      namebs.str = name; 
      namebs.max_len = sizeof (name); 
      name[0] = 0; 
      if ((status = lay_msg(sub_pkt.mp, 
 			    LAY_NAME, 
 			    NULL, 
 			    &namebs)) < 0) 
 	return (status); 
      if (previous) 
 	if ((status = str_cat_char(str, 
 				   max_len, 
 				   '.')) < 0) 
 	  return (status); 
      previous = TRUE; 
      if ((status = str_cat_str(str, 
 				max_len, 
 				name)) < 0) 
 	return (status); 
      PKT_COPY(&sub_pkt,&tmp_pkt); 
    } while (pkt_sub(&tmp_pkt,&sub_pkt) == 0); 
  return (0); 
}

/* writes out a packet buffer in a file name. 
   It is a convenience function. 
   Returns 0 if OK. -ERR_OPEN if open(2) fails, -ERR_WRITE if write(2) 
   returns less bytes than pkt->len */ 
t_status		pkt_save_raw(pkt,fname) 
t_pkt			*pkt; 
char			*fname;		/* File name */ 
{ 
  int			fd; 
  int			cc; 

  if ((fd = open(fname,O_WRONLY|O_CREAT,0600)) < 0) 
    return (-ERR_OPEN); 
  if ((cc = write(fd,pkt->buf,pkt->len)) != pkt->len) 
    return (-ERR_WRITE); 
  close(fd); 
  return (0); 
}

/* loads a binary file into a packet. 
   It is a convenience function. 
   It modifies len (of course), sets netlen to len but doesn't modify 
   timestamp nor the msg_proc. It allocates a new buffer and frees 
   the old one. 
   Returns 0 if OK. -ERR_SYSCALL if stat(2) fails, 
   -ERR_OPEN if open(2) fails, -ERR_READ if read(2) fails, 
   -ERR_TRUNC if read(2) reads less bytes than specified in the 
   stat structure */ 
t_status		pkt_load_raw(pkt,fname,alloc_proc,free_proc) 
t_pkt			*pkt;		/* Valid packet */ 
char			*fname;		/* File name */ 
t_alloc_proc		alloc_proc;	/* Alloc method */ 
t_free_proc		free_proc;	/* Free proc */ 
{ 
  struct stat		st; 
  int			fd; 
  int			cc; 
  t_status		status; 
  char			*buf; 

  if (stat(fname,&st) < 0) 
    return (-ERR_SYSCALL); 
  if ((fd = open(fname,O_RDONLY)) < 0)
    return (-ERR_OPEN);
  if ((buf = alloc_proc(st.st_size,
			"pkt",
			"pkt_load_raw:buf",
			&status)) == NULL)
    {
      close(fd);
      return (status);
    }
  if ((cc = read(fd,buf,st.st_size)) < 0)
    {
      status = -ERR_READ;
      goto bad;
    }
  if (cc != st.st_size)
    {
      status = -ERR_TRUNC;
      goto bad;
    }
  free_proc(pkt->buf,
	    "pkt",
	    "*:buf");
  pkt->buf = buf;
  pkt->len = pkt->netlen = st.st_size;
  status = 0;
  goto end;
bad:
  free_proc(buf,
	    "pkt",
	    "pkt_load_raw:buf");
end:
  close(fd);
  return (status);
}

/* extracts a layer from a packet.
   It tries to find the desired msg_proc using pkt_subn(3), then
   discards the first bytes of the packet buffer.
   It allocates a new buffer, frees the old one and modifies pkt->len.
   An advanced version of this function is pkt_extract_from_name(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_extract(pkt,mp,alloc_proc,free_proc)
t_pkt			*pkt;
t_msg_proc		mp;		/* Extract from this layer*/
t_alloc_proc		alloc_proc;	
t_free_proc		free_proc;
{
  t_status		status;
  char			*nbuf;
  t_pkt			sub_pkt;

  if ((status = pkt_subn(pkt,&sub_pkt,mp)) < 0)
    return (status);
  if ((nbuf = alloc_proc(sub_pkt.len * sizeof (char),
			 "pkt",
			 "pkt_extract:buf",
			 &status)) == NULL)
    return (status);
  free_proc(pkt->buf,
	    "pkt",
	    "*:buf");
  bcopy(sub_pkt.buf,nbuf,sub_pkt.len);
  pkt->buf = nbuf;
  pkt->len = sub_pkt.len;
  pkt->mp = mp;
  return (0);
}

/* truncates a packet to a layer.
   It tries to find the desired msg_proc using pkt_subn(3), then
   discards the last bytes of the packet buffer.
   It allocates a new buffer, frees then old one and modifies pkt->len.
   An advanced version of this function is pkt_trunc_from_name(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_trunc(pkt,mp,alloc_proc,free_proc)
t_pkt			*pkt;
t_msg_proc		mp;		/* Truncate to this layer */
t_alloc_proc		alloc_proc;
t_free_proc		free_proc;
{
  t_status		status;
  t_pkt			sub_pkt;
  t_pkt			sub_sub_pkt;
  char			*nbuf;
  int			nlen;

  if ((status = pkt_subn(pkt,&sub_pkt,mp)) < 0)
    return (status);
  if ((status = pkt_sub(&sub_pkt,&sub_sub_pkt)) < 0)
    return (status);
  nlen = pkt->len - sub_sub_pkt.len;
  if ((nbuf = alloc_proc(nlen * sizeof (char),
			 "pkt",
			 "pkt_trunc:buf",
			 &status)) == NULL)
    return (status);
  bcopy(pkt->buf,nbuf,nlen);
  free_proc(pkt->buf,
	    "pkt",
	    "*:buf");
  pkt->buf = nbuf;
  pkt->len = nlen;
  return (0);
}

/* makes packet valid adapting correct length fields (if so) of each layer
   It walks the various layers of the packet from first to last 
   calling LAY_ADAPT_LEN.
   Returns 0 if OK. Might returns various errors */
t_status		pkt_adapt_len(pkt)
t_pkt			*pkt;
{
  t_pkt			sub_pkt;
  t_status		status;
  t_buf			b;

  if ((status = pkt_sub(pkt,&sub_pkt)) < 0)
    {
      if (status != -ERR_NOMETHOD && status != -ERR_ZEROLEN)
	return (status);
      else
	{
	  b.buf = pkt->buf;
	  b.len = pkt->len;
	  if ((status = lay_msg(pkt->mp,
				LAY_ADAPT_LEN,
				&b,
				NULL)) < 0)
	    {
	      if (status != -ERR_NOMETHOD)
		return (status);
	    }
	  return (0);
	}
    }
  else
    {
      if ((status = pkt_adapt_len(&sub_pkt)) < 0)
	{
	  if (status != -ERR_NOMETHOD)
	    return (status);
	}
      b.buf = pkt->buf;
      b.len = pkt->len;
      if ((status = lay_msg(pkt->mp,
			    LAY_ADAPT_LEN,
			    &b,
			    NULL)) < 0)
	{
	  if (status != -ERR_NOMETHOD)
	    return (status);
	}
      return (0);
    }
}

#ifdef DEBUG
/* shows a packet structure.
   This is a debug function. */
VOID_FUNC		pkt_show(pkt)
t_pkt			*pkt;
{
  fprintf(stderr,"mp=%p\n",pkt->mp);
  fprintf(stderr,"buf=%p\n",pkt->buf);
  fprintf(stderr,"len=%d\n",pkt->len);;
  fprintf(stderr,"netlen=%d\n",pkt->netlen);
  fprintf(stderr,"ts=%d:%d\n",pkt->ts.tv_sec,pkt->ts.tv_usec);
}
#endif

/* advanced retrieve of the sub pkt (doesn't manage options).
   It does exactly the same thing as pkt_sub(3) but tries to call
   LAY_SUB_ID. If there is no method, or if id is NULL,
   it reverts to LAY_SUB. This allow
   from certain layers to store values (as channels)
   in the id parameter which can be used by sub layers.
   Returns 0 if OK, See pkt_sub(3) for errors. */
t_status		pkt_sub_id(pkt,sub_pkt,id)
t_pkt			*pkt;		/* Reference packet */
t_pkt			*sub_pkt;	/* Result */
t_id			*id;		/* Pointers hash table */
{
  t_status		status;
  t_buf			b;
  t_msg_proc		sub_mp;
  t_off			off;
  t_sub_id_data		sid;

  b.buf = pkt->buf;
  b.len = pkt->len;
  sid.b.buf = pkt->buf;
  sid.b.len = pkt->len;
  sid.id = id;
  if (!id)
    goto no_id;
  if ((status = lay_msg(pkt->mp,
			LAY_SUB_ID,
			&sid,
			&sub_mp)) < 0)
    {
      if (status == -ERR_NOMETHOD)
	{
	no_id:
	  if ((status = lay_msg(pkt->mp,
				LAY_SUB,
				&b,
				&sub_mp)) < 0)
	    return (status);
	}
      else
	return (status);
    }
  sub_pkt->mp = sub_mp;
  if ((status = lay_msg(pkt->mp,
			LAY_OFF,
			&b,
			&off)) < 0)
    return (status);
  if (pkt->len == off) /* ACCEPT LEN == OFF == 0 */
    {
      status = -ERR_ZEROLEN;
      goto dosub;
    }
  if (off == 0)
    return (-ERR_LOOP);
  if (pkt->len < off)
    return (-ERR_TRUNC);
  status = 0;
dosub:
  sub_pkt->buf = pkt->buf + off;
  sub_pkt->len = pkt->len - off;
  sub_pkt->netlen = pkt->netlen;
  sub_pkt->ts = pkt->ts;
  return (status);
}

/* retrieves the named sub layer from packet (managing options).
   It does subsequents calls to pkt_sub_id(3) but also checks if 
   layers have options (using LAY_HAS_OPT). If it is the case, it 
   tries to find the named sub layer in the option branch. Once the
   option branch is totaly examined (founding -ERR_ZEROLEN or -ERR_NOMETHOD)
   it reverts to the normal search. If a layer has notified it has
   an option and returned option offset and length don't match the packet
   length, it returns -ERR_BADMATCH. 
   Note that an option may have an option itself.
   Returns 0 if OK. Might return various errors */
t_status		pkt_subn_from_name_i(pkt,subn_pkt,desired_name,id)
t_pkt			*pkt;		/* Reference packet */
t_pkt			*subn_pkt;	/* Result */
char			*desired_name;	/* Name, e.g. "udp[0]" */
t_id			*id;		/* Pointers hash table */
{
  t_msg_proc		sub_mp;
  t_status		status;
  char			name[STR_BUFSIZ];
  t_buf			b;
  t_has_opt_data	hsd;
  t_bridled_str		namebs;

  namebs.str = name;
  namebs.max_len = sizeof (name);
  name[0] = 0;
  if ((status = lay_msg(pkt->mp,
			LAY_NAME,
			id,
			&namebs)) < 0)
    return (status);
  if (!strcmp(name,desired_name))
    {
      PKT_COPY(pkt,subn_pkt);
      return (0);
    }
  b.buf = pkt->buf;
  b.len = pkt->len;
  if ((status = lay_msg(pkt->mp,
			LAY_HAS_OPT,
			&b,
			&hsd)) < 0)
    {
      if (status != -ERR_NOMETHOD)
	return (status);
    }
  else
    {
      if (hsd.has_opt)
	{
	  t_pkt		opt_pkt;

	  if ((hsd.opt_off + hsd.opt_len) > pkt->len)
	    return (-ERR_BADMATCH);
	  PKT_COPY(pkt,&opt_pkt);
	  opt_pkt.mp = hsd.opt_mp;
	  opt_pkt.buf = pkt->buf + hsd.opt_off;
	  opt_pkt.len = hsd.opt_len;
	  if ((status = pkt_subn_from_name_i(&opt_pkt,
					     subn_pkt,
					     desired_name,
					     id)) < 0)
	    {
	      if (status != -ERR_NOMETHOD && status != -ERR_ZEROLEN)
		return (status);
	    }
	  else /* status == 0 */
	    return (0);
	}
    }
  if ((status = pkt_sub_id(pkt,
			   subn_pkt,
			   id)) < 0)
    return (status);
  else
    {
      t_pkt		tmp_pkt;

      PKT_COPY(subn_pkt,&tmp_pkt);
      return (pkt_subn_from_name_i(&tmp_pkt,
				   subn_pkt,
				   desired_name,
				   id));
    }
}

/* retrieves the named sub layer from packet (managing options).
   It is a convenience function. It allocates a id to HASH_TINY_BASE
   value and calls pkt_subn_from_name_i(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_subn_from_name(pkt,subn_pkt,desired_name)
t_pkt			*pkt;		/* Reference packet */
t_pkt			*subn_pkt;	/* Result */
char			*desired_name;	/* E.g "udp[0]" */
{
  t_id			*id;
  t_status		status;

  if ((id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  status = pkt_subn_from_name_i(pkt,
				subn_pkt,
				desired_name,
				id);
  id_delete(id);
  return (status);
}

/* retrieves a packet field value to a bridled string (managing options).
   It first checks for special field names: "Layer", "Len", "Netlen" 
   and "[dD]ate" which
   are substituted respectively by the name of the packet layer, the len,
   netlen and timestamp members. Then examines the fully qualified name
   of the field (in the form layer[index].field) and tries to find it
   (ala pkt_subn_from_name(3)) by subsequent calls of pkt_sub_id(3),
   LAY_NAME (with the optional id set) and LAY_GET_FIELD.
   Note that this function could use pkt_subn_from_name(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_field_to_str_i(pkt,fq_field,str,max_len,id)
t_pkt			*pkt;		/* Reference packet */
char			*fq_field;	/* E.g "udp[0].sport" */
char			*str;		/* Valid string */
int			max_len;	/* Maximimum length */
t_id			*id;		/* Pointers hash table */
{
  t_status		status;
  char			*field;
  int			layerlen;
  t_pkt			sub_pkt;
  t_pkt			tmp_pkt;
  t_bridled_str		bs;
  t_get_field_data	gfd;

  if (!strcmp(fq_field,"Layer"))
    {
      return (lay_to_str(pkt->mp,
			 str,
			 max_len));
    }
  if (!strcmp(fq_field,"Len"))
    {
      return (long_to_str((signed long)(pkt->len),
			  layer_base,
			  str,
			  max_len));
    }
  if (!strcmp(fq_field,"Netlen"))
    {
      return (long_to_str((signed long)(pkt->netlen),
			  layer_base,
			  str,
			  max_len));
    }
  if (!strcmp(fq_field,"date") || !strcmp(fq_field,"Date"))
    {
      return (timeval_to_str(&(pkt->ts),
			     str,
			     max_len,
			     fq_field[0] == 'D'));
    }
  if ((field = index(fq_field,'.')) == NULL)
    return (-ERR_NOSUCHFIELD);
  layerlen = field - fq_field;
  field++;
  PKT_COPY(pkt,&sub_pkt);
  do 
    {
      char		name[STR_BUFSIZ];
      t_buf		b;
      t_has_opt_data	hsd;
      t_bridled_str	namebs;
      
      namebs.str = name;
      namebs.max_len = sizeof (name);
      name[0] = 0;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_NAME,
			    id,
			    &namebs)) < 0)
	return (status);
      if (!strncmp(fq_field,name,layerlen))
	{
	  gfd.optional_id = id;
	  gfd.field = field;
	  gfd.b.buf = sub_pkt.buf;
	  gfd.b.len = sub_pkt.len;
	  bs.str = str;
	  bs.max_len = max_len;
	  return (lay_msg(sub_pkt.mp,
			  LAY_GET_FIELD,
			  &gfd,
			  &bs));
	}
      b.buf = sub_pkt.buf;
      b.len = sub_pkt.len;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_HAS_OPT,
			    &b,
			    &hsd)) < 0)
	{
	  if (status != -ERR_NOMETHOD)
	    return (status);
	}
      else
	{
	  if (hsd.has_opt)
	    {
	      t_pkt         opt_pkt;

	      if ((hsd.opt_off + hsd.opt_len) > sub_pkt.len)
		return (-ERR_BADMATCH);
	      PKT_COPY(pkt,&opt_pkt);
	      opt_pkt.mp = hsd.opt_mp;
	      opt_pkt.buf = sub_pkt.buf + hsd.opt_off;
	      opt_pkt.len = hsd.opt_len;
	      status = pkt_get_field_to_str_i(&opt_pkt,
					      fq_field,
					      str,
					      max_len,
					      id);
	      if (status != -ERR_NOSUCHFIELD)
		return (status);
	    }
	}
      PKT_COPY(&sub_pkt,&tmp_pkt);
    } while (pkt_sub_id(&tmp_pkt,
			&sub_pkt,
			id) == 0);
  return (-ERR_NOSUCHFIELD);
}

/* retrieves a packet field value to a bridled string (managing options).
   It is a convenience function. It allocates a id to HASH_TINY_BASE
   value and calls pkt_get_field_to_str_i(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_field_to_str(pkt,field,str,max_len)
t_pkt			*pkt;		/* Reference packet */
char			*field;		/* E.g "udp[0].sport" */
char			*str;		/* Valid string */
int			max_len;	/* Maximum length */
{
  t_id			*id;
  t_status		status;

  if ((id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  status = pkt_get_field_to_str_i(pkt,
				  field,
				  str,
				  max_len,
				  id);
  id_delete(id);
  return (status);
}

/* retrieves a packet field value to a bridled string (managing options).
   It is a convenience function. It doesn't manage brackets
   but doesn't allocate any memory.
   It calls pkt_get_field_to_str_i(3) with a NULL id.
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_field_to_str_no_id(pkt,field,str,max_len)
t_pkt			*pkt;		/* Reference packet */
char			*field;		/* E.g "udp.sport" */
char			*str;		/* Valid string */
int			max_len;	/* Maximum length */
{
  t_status		status;

  status = pkt_get_field_to_str_i(pkt,
				  field,
				  str,
				  max_len,
				  NULL);
  return (status);
}

/* sets a packet field value from a string (managing options).
   It first checks for special field name "Layer", which acts
   like a cast. Then examines the fully qualified name
   of the field (in the form layer[index].field) and tries to find it
   (ala pkt_subn_from_name(3)) by subsequent calls of pkt_sub_id(3),
   LAY_NAME (with the optional id set) and LAY_SET_FIELD.
   Note that this function could use pkt_subn_from_name(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_set_field_from_str_i(pkt,fq_field,str,id)
t_pkt			*pkt;		/* Reference packet */
char			*fq_field;	/* E.g "udp[0].sport" */
char			*str;		/* E.g "echo" */
t_id			*id;		/* Pointers hash table */
{
  t_status		status;
  char			*field;
  t_pkt			sub_pkt;
  t_pkt			tmp_pkt;
  t_get_field_data	gfd;
  int			layerlen;

  if (!strcmp(fq_field,"Layer"))
    {
      t_msg_proc	mp;

      if ((mp = lay_from_str(str,&status)) == NULL)
	return (status);
      pkt->mp = mp;
      return (0);
    }
  if ((field = index(fq_field,'.')) == NULL)
    return (-ERR_NOSUCHFIELD);
  layerlen = field - fq_field;
  field++;
  PKT_COPY(pkt,&sub_pkt);
  do 
    {
      char		name[STR_BUFSIZ];
      t_buf		b;
      t_has_opt_data	hsd;
      t_bridled_str	namebs;
      
      namebs.str = name;
      namebs.max_len = sizeof (name);
      name[0] = 0;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_NAME,
			    id,
			    &namebs)) < 0)
	return (status);
      if (!strncmp(fq_field,name,layerlen))
	{
	  gfd.optional_id = id;
	  gfd.field = field;
	  gfd.b.buf = sub_pkt.buf;
	  gfd.b.len = sub_pkt.len;
	  return (lay_msg(sub_pkt.mp,
			  LAY_SET_FIELD,
			  &gfd,
			  str));
	}
      b.buf = sub_pkt.buf;
      b.len = sub_pkt.len;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_HAS_OPT,
			    &b,
			    &hsd)) < 0)
	{
	  if (status != -ERR_NOMETHOD)
	    return (status);
	}
      else
	{
	  if (hsd.has_opt)
	    {
	      t_pkt         opt_pkt;

	      if ((hsd.opt_off + hsd.opt_len) > sub_pkt.len)
		return (-ERR_BADMATCH);
	      PKT_COPY(pkt,&opt_pkt);
	      opt_pkt.mp = hsd.opt_mp;
	      opt_pkt.buf = sub_pkt.buf + hsd.opt_off;
	      opt_pkt.len = hsd.opt_len;
	      status = pkt_set_field_from_str_i(&opt_pkt,
						fq_field,
						str,
						id);
	      if (status != -ERR_NOSUCHFIELD)
		return (status);
	    }
	}
      PKT_COPY(&sub_pkt,&tmp_pkt);
    } while (pkt_sub_id(&tmp_pkt,
			&sub_pkt,
			id) == 0);
  return (-ERR_NOSUCHFIELD);
}

/* sets a packet field value from a string (managing options).
   It is a convenience function. It allocates a id to HASH_TINY_BASE
   value and calls pkt_set_field_from_str_i(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_set_field_from_str(pkt,field,str)
t_pkt			*pkt;		/* Reference packet */
char			*field;		/* E.g "udp[0].sport" */
char			*str;		/* E.g "echo" */
{
  t_id			*id;
  t_status		status;

  if ((id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  status = pkt_set_field_from_str_i(pkt,
				    field,
				    str,
				    id);
  id_delete(id);
  return (status);
}

/* retrieves the packet template to a bridled string (managing options).
   It does subsequent calls of LAY_GET_TMPL and pkt_sub_id(3). It
   also checks for options.
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_tmpl_to_str_i(pkt,str,max_len,id)
t_pkt			*pkt;		/* Reference packet */
char			*str;		/* A large valid string! */
int			max_len;	/* Maximum length */
t_id			*id;		/* Pointers hash table */
{
  t_status		status;
  t_get_tmpl_data	gtd;
  t_pkt			sub_pkt;
  t_buf			b;
  t_has_opt_data	hsd;
  t_bridled_str		bs;

  gtd.b.buf = pkt->buf;
  gtd.b.len = pkt->len;
  gtd.id = id;
  bs.str = str;
  bs.max_len = max_len;
  if ((status = lay_msg(pkt->mp,
			LAY_GET_TMPL,
			&gtd,
			&bs)) < 0)
    {
      if (status == -ERR_NOMETHOD)
	  return (0);
      else
	return (status);
    }
  b.buf = pkt->buf;
  b.len = pkt->len;
  if ((status = lay_msg(pkt->mp,
			LAY_HAS_OPT,
			&b,
			&hsd)) < 0)
    {
      if (status != -ERR_NOMETHOD)
	return (status);
    }
  else
    {
      if (hsd.has_opt)
	{
	  t_pkt		opt_pkt;

	  if ((hsd.opt_off + hsd.opt_len) > pkt->len)
	    return (-ERR_BADMATCH);
	  PKT_COPY(pkt,&opt_pkt);
	  opt_pkt.mp = hsd.opt_mp;
	  opt_pkt.buf = pkt->buf + hsd.opt_off;
	  opt_pkt.len = hsd.opt_len;
	  if ((status = pkt_get_tmpl_to_str_i(&opt_pkt,
					      str,
					      max_len,
					      id)) < 0)
	    {
	      if (status == -ERR_NOMETHOD || status == -ERR_TRUNC)
		return (0);
	      else
		return (status);
	    }
	}
    }
  if ((status = pkt_sub_id(pkt,
			   &sub_pkt,
			   id)) < 0)
    {
      if (status == -ERR_NOMETHOD || status == -ERR_ZEROLEN)
	return (0);
      else
	return (status);
    }
  return (pkt_get_tmpl_to_str_i(&sub_pkt,
				str,
				max_len,
				id));
}

/* retrieves the packet template to a bridled string (managing options). 
   It is a convenience function. It allocates a id to HASH_TINY_BASE
   value and calls pkt_get_tmpl_to_str_i(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_tmpl_to_str(pkt,str,max_len)
t_pkt			*pkt;		/* Reference packet */
char			*str;		/* Valid large string! */
int			max_len;	/* Maximum length */	
{
  t_id			*id;
  t_status		status;

  if ((id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  status = pkt_get_tmpl_to_str_i(pkt,
				 str,
				 max_len,
				 id);
  id_delete(id);
  return (status);
}

/* converts a string timeval representation to a timeval structure.
   It understand the following syntax: sec[.usec]. If usec isn't specified,
   it sets it to zero. */
VOID_FUNC		timeval_from_str(str,tv)
char			*str;	/* E.g "42.42" */
struct timeval		*tv;	/* Timeval structure returned */
{
  char			*ptr;

  if (ptr = index(str,'.'))
    *ptr++ = 0;
  else
    ptr = "0";
  tv->tv_sec = atoi(str);
  tv->tv_usec = atoi(ptr);
}

/* converts-and-catenates a timeval struct to a bridled string.
   If resolve is TRUE, the result will be in the format hh:mm:ss.uuuuuu, else
   it will be in the format sec.usec (as parsed by timeval_from_str(3)).
   It is taken from tcpdump. 
   Returns 0 if OK. Returns -ERR_BO on buffer overflows */
t_status		timeval_to_str(tv,str,max_len,resolve)
struct timeval		*tv;		/* Timeval struct */
char			*str;		/* Valid string */
int			max_len;	/* Maximum length */
t_boolean		resolve;	/* See above */
{
  t_status		status;
  
  if (resolve)
    {
      unsigned int	s;
      
      s = (tv->tv_sec + thiszone) % 86400;
      return (str_cat_fmt_va(str,
			     max_len,
			     "%02u:%02u:%02u.%06u",
			     s / 3600,
			     (s % 3600) / 60,
			     s % 60,
			     (unsigned int)(tv->tv_usec)));
    }
  else
    return (str_cat_fmt_va(str,
			   max_len,
			   "%u.%06u",
			   (unsigned int)(tv->tv_sec),
			   (unsigned int)(tv->tv_usec)));
}

t_status		timeval_to_str_init(VOID_DECL)
{
  thiszone = gmt2local(0);
  return (0);
}

/* converts-and-catenates the local address of the packet to a bridled string.
   If remote is FALSE, it will get the local address of the packet,
   knowing that it could be ip.src OR ip.dst. If remote is TRUE, it will
   get the remote address. If both addresses are remote or local, it will
   take the lowest address in using inaddr_cmp(3).
   To do such an operation, it is helped by a vector of local subnets that the
   user must precise.
   Note that this function could be improved by extensing the concept of
   ip.src and ip.dst to other protocols.
   Returns 0 if OK. Might return various errors */
t_status		local_to_str(pkt,
				     vec_prefixes,
				     str,
				     max_len,
				     remote,
				     resolve,
				     get_field_to_str_proc)
t_pkt			*pkt;		/* Reference packet */	
t_vec			*vec_prefixes;	/* Vector of subnets (mandatory) */
char			*str;		/* Valid string */
int			max_len;	/* Maximum length */
t_boolean		remote;		/* Depending on what you want */
t_boolean		resolve;	/* Resolved form or not */
t_pkt_get_field_to_str_proc	get_field_to_str_proc;/* Function used
							 to retrieve "ip.src"
							 and "ip.dst" */
{
  char			src[STR_BUFSIZ];
  char			dst[STR_BUFSIZ];
  struct in_addr	inaddrsrc;
  struct in_addr	inaddrdst;
  t_boolean		srcislocal;
  t_boolean		dstislocal;
  struct in_addr	*wantedinaddr;
  t_status		status;

  src[0] = 0;
  if ((status = get_field_to_str_proc(pkt,
				      "ip.src",
				      src,
				      sizeof (src))) < 0)
      return (status);
  dst[0] = 0;
  if ((status = get_field_to_str_proc(pkt,
				      "ip.dst",
				      dst,
				      sizeof (dst))) < 0)
      return (status);
  if ((status = inaddr_from_str(src,&inaddrsrc,FALSE)) < 0)
    return (status);
  if ((status = inaddr_from_str(dst,&inaddrdst,FALSE)) < 0)
    return (status);
  srcislocal = FALSE;
  VEC_FOR(vec_prefixes,t_subnet *subnet)
    {
      if (is_subnet_member(&inaddrsrc,subnet))
	{
	  srcislocal = TRUE;
	  break ;
	}
    }
  VEC_ENDFOR;
  dstislocal = FALSE;
  VEC_FOR(vec_prefixes,t_subnet *subnet)
    {
      if (is_subnet_member(&inaddrdst,subnet))
	{
	  dstislocal = TRUE;
	  break ;
	}
    }
  VEC_ENDFOR;
#define REMOTESRC	(remote?&inaddrdst:&inaddrsrc)
#define REMOTEDST	(remote?&inaddrsrc:&inaddrdst)
  if (srcislocal != dstislocal)
    wantedinaddr = srcislocal?REMOTESRC:REMOTEDST;
  else
    wantedinaddr = (inaddr_cmp(&inaddrsrc,&inaddrdst) < 0)?REMOTESRC:REMOTEDST;
  return (inaddr_to_str(wantedinaddr,
			str,
			max_len,
			resolve));
}

/* is a t_tmpl_do_proc. 
   It is used to convert a variable from a template to a value.
   It recognizes the following formats (order is important):
   %% is %.
   %0xn% is a char exprimed by an hexadecimal value (using strtol(3)).
   %0n% is a char exprimed by an octal value (using strtol(3)).
   %n% is a char exprimed by a decimal value (using strtol(3)).
   %[Ll]ocal% is the local address of packet (resolved or not).
   %[Rr]emote% is the remote address of packet (resolved or not).
   %layer[index].field% is a packet field.
   %extravar% is a potential variable looked up a user supplied dictionary.
   If nothing matches, it then search for time values using time(2) and
   localtime(3) and recognizes %wday%, %mon%, %mday%, %hour%, %min%, %sec%,
   %year%.
   Returns 0 if OK. Might return various errors including -ERR_TRUNC meaning
   that a field of a layer is truncated. -ERR_CANTHANDLE meaning that field
   contains a value which is not convertible decently to a string (E.g
   compressed dns query names) */
t_status		pkt_format_do(bs,var,pfdd)
t_bridled_str		*bs;		/* Bridled string */
char			*var;		/* Variable to convert */
t_pkt_format_do_data	*pfdd;	       
{
  int			varlen;
  long			signedvalue;
  t_status		status;

  varlen = strlen(var);
  if (varlen == 0)
    return (str_cat_char(bs->str,bs->max_len,'%'));
  else
    if (isdigit(var[0]))
      {
	if (var[0] == '0')
	  {
	    if (var[1] == 'x' || var[1] == 'X')
	      signedvalue = strtol(var,NULL,16);
	    else
	      signedvalue = strtol(var,NULL,8);
	  }
	else
	  signedvalue = strtol(var,NULL,10);
	return (str_cat_char(bs->str,bs->max_len,(int)signedvalue));
      }
    else
      {
	time_t		t;
	struct tm	*tmp;

	if (pfdd->vec_prefixes)
	  {
	    if (!strcmp(var,"local") || !strcmp(var,"Local"))
	      return (local_to_str(pfdd->pkt,
				   pfdd->vec_prefixes,
				   bs->str,
				   bs->max_len,
				   FALSE,
				   var[0] == 'L',
				   pfdd->get_field_to_str_proc));
	    if (!strcmp(var,"remote") || !strcmp(var,"Remote"))
	      return (local_to_str(pfdd->pkt,
				   pfdd->vec_prefixes,
				   bs->str,
				   bs->max_len,
				   TRUE,
				   var[0] == 'R',
				   pfdd->get_field_to_str_proc));
	  }
	if ((status = pfdd->get_field_to_str_proc(pfdd->pkt,
						  var,
						  bs->str,
						  bs->max_len)) < 0)
	  if ((status != -ERR_NOSUCHFIELD &&
	       status != -ERR_NOMETHOD))
	    return (status);
	if (pfdd->extra_vars)
	  {
	    t_hash_elt	*he;

	    if (he = dict_get(pfdd->extra_vars,var))
	      return (str_cat_str(bs->str,
				  bs->max_len,
				  he->value));
	  }
	time(&t);
	tmp = localtime(&t);
	if (!strcmp(var,"wday"))
	  return (long_to_str((signed long)(tmp->tm_wday),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"mon"))
	  return (long_to_str((signed long)(tmp->tm_mon),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"mday"))
	  return (long_to_str((signed long)(tmp->tm_mday),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"hour"))
	  return (long_to_str((signed long)(tmp->tm_hour),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"min"))
	  return (long_to_str((signed long)(tmp->tm_min),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"sec"))
	  return (long_to_str((signed long)(tmp->tm_sec),
			      10,
			      bs->str,
			      bs->max_len));
	if (!strcmp(var,"year"))
	  return (long_to_str((signed long)(tmp->tm_year),
			      10,
			      bs->str,
			      bs->max_len));
      }
  return (0);
}

/* is a t_tmpl_do_proc.
   It is the permissive version of pkt_format_do(3). It calls pkt_format_do(3)
   and catenates a "|" to the bridled string if it has returned -ERR_TRUNC.
   It also catenates a "?" if -ERR_CANTHANDLE is encountered. 
   Returns 0 if OK. Might return various errors. Of course, absorbs 
   -ERR_TRUNC and -ERR_CANTHANDLE errors */
t_status		pkt_format_permissive_do(bs,var,pfdd)
t_bridled_str		*bs;		/* Bridled string */
char			*var;		/* Variable */
t_pkt_format_do_data	*pfdd;
{
  t_status		status;
  
  if ((status = pkt_format_do(bs,var,pfdd)) < 0)
    {
      switch (status)
	{
	case -ERR_TRUNC:
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      "|"));
	case -ERR_CANTHANDLE:
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      "?"));
	}
      return (status);
    }
  return (0);
}

/* is a t_tmpl_do_proc.
   It is inserted before the real t_tmpl_do_proc. Its role is to
   htmlize buffers returned from real templates procedures. E.g when
   it encounters a newline character in the result, it converts it to "<br>".
   It converts "<" and "&" which are HTML special characters respectively
   to "&lt;" and "&amp;". It also converts non-printable and non-ASCII
   characters to a dot ".". 
   For this operation, it needs a user supplied specific (and large) buffer.
   Returns 0 if OK. Might return various errors */
t_status		pkt_format_htmlize_do(bs,var,pfdd)
t_bridled_str		*bs;		/* Bridled string */
char			*var;		/* Variable */
t_pkt_format_do_data	*pfdd;
{
  t_status		status;
  t_bridled_str		htmlize_bs;
  char			*ptr;

  pfdd->htmlize_str[0] = 0;
  htmlize_bs.str = pfdd->htmlize_str;
  htmlize_bs.max_len = pfdd->htmlize_max_len;
  if ((status = pfdd->tmpl_do_proc(&htmlize_bs,var,pfdd)) < 0)
    return (status);
  ptr = htmlize_bs.str;
  while (*ptr)
    {
      switch (*ptr)
	{
	case '\n':
	  if ((status = str_cat_str(bs->str,
				    bs->max_len,
				    "<br>\n")) < 0)
	    return (status);
	  break ;
	case '<':
	  if ((status = str_cat_str(bs->str,
				    bs->max_len,
				    "&lt;")) < 0)
	    return (status);
	  break ;
	case '&':
	  if ((status = str_cat_str(bs->str,
				    bs->max_len,
				    "&amp;")) < 0)
	    return (status);
	  break ;
	default:
	  if ((!isprint(*ptr)) || ((t_u8)(*ptr) > 127)) 
	    {
	      if ((status = str_cat_char(bs->str,
					 bs->max_len,
					 '.')) < 0)
		return (status);
	    }
	  else
	    {
	      if ((status = str_cat_char(bs->str,
					 bs->max_len,
					 *ptr)) < 0)
		return (status);
	    }
	}
      ptr++;
    }
  return (0);
}

/* formats a template to a string.
   It uses pkt_format_do(3) and tmpl_str_to_str(3). It doesn't
   manage brackets.
   Returns 0 if OK. Might return various errors such as -ERR_BO (buffer 
   overflow) */
t_status		pkt_format_simple(pkt,
					  tmpl_str,
					  vec_prefixes,
					  extra_vars,
					  str,
					  max_len)
t_pkt			*pkt;		/* Reference packet */
char			*tmpl_str;	/* Template string */
t_vec			*vec_prefixes;	/* Optional vector of subnets */
t_dict			*extra_vars;	/* Optional dict_str */
char			*str;		/* Destination string */
int			max_len;	/* Dest. string length */
{
  t_pkt_format_do_data	pfdd;

  pfdd.pkt = pkt;
  pfdd.vec_prefixes = vec_prefixes;
  pfdd.get_field_to_str_proc = pkt_get_field_to_str_no_id;
  pfdd.extra_vars = extra_vars;
  pfdd.tmpl_do_proc = NULL;
  pfdd.htmlize_str = NULL;
  pfdd.htmlize_max_len = 0;
  return (tmpl_str_to_str(tmpl_str,
			  (t_tmpl_do_proc)pkt_format_do,
			  &pfdd,
			  str,
			  max_len));
}

/* permissive version of pkt_format_simple(3).
   It does the same thing as pkt_format_simple(3) excepted that it uses
   pkt_format_permissive_do(3) instead of pkt_format_do(3) as t_tmpl_do_proc.
   It doesn't manage brackets.
   Returns 0 if OK. Might return various errors such as -ERR_BO (buffer 
   overflow) */   
t_status		pkt_format_simple_permissive(pkt,
						     tmpl_str,
						     vec_prefixes,
						     extra_vars,
						     str,
						     max_len)
t_pkt			*pkt;		/* Reference packet */
char			*tmpl_str;	/* Template string */
t_vec			*vec_prefixes;	/* Might be NULL */
t_dict			*extra_vars;	/* Might be NULL */
char			*str;		/* Dest. string */
int			max_len;	/* Destr string length */
{
  t_pkt_format_do_data	pfdd;

  pfdd.pkt = pkt;
  pfdd.vec_prefixes = vec_prefixes;
  pfdd.extra_vars = extra_vars;
  pfdd.get_field_to_str_proc = pkt_get_field_to_str_no_id;
  pfdd.tmpl_do_proc = NULL;
  pfdd.htmlize_str = NULL;
  pfdd.htmlize_max_len = 0;
  return (tmpl_str_to_str(tmpl_str,
			  (t_tmpl_do_proc)pkt_format_permissive_do,
			  &pfdd,
			  str,
			  max_len));
}

/* formats a template to a html string.
   It uses pkt_format_htmlize_do(3) and tmpl_str_to_str(3).
   Returns 0 if OK. Might return various errors such as -ERR_BO (buffer 
   overflow) */
t_status		pkt_format_html(pkt,
					tmpl_str,
					vec_prefixes,
					extra_vars,
					str,
					max_len,
					htmlize_str,
					htmlize_max_len)
t_pkt			*pkt;		/* Reference packet */
char			*tmpl_str;	/* Template string */
t_vec			*vec_prefixes;	/* Optional vector of subnets */
t_dict			*extra_vars;	/* Optional dict_str */
char			*str;		/* Destination string */
int			max_len;	/* Dest. string length */
char			*htmlize_str;	/* Htmlization string buffer */
int			htmlize_max_len;/* Htmlization string buffer length */
{
  t_pkt_format_do_data	pfdd;

  pfdd.pkt = pkt;
  pfdd.vec_prefixes = vec_prefixes;
  pfdd.get_field_to_str_proc = pkt_get_field_to_str;
  pfdd.extra_vars = extra_vars;
  pfdd.tmpl_do_proc = (t_tmpl_do_proc)pkt_format_do;
  pfdd.htmlize_str = htmlize_str;
  pfdd.htmlize_max_len = htmlize_max_len;
  return (tmpl_str_to_str(tmpl_str,
			  (t_tmpl_do_proc)pkt_format_htmlize_do,
			  &pfdd,
			  str,
			  max_len));
}

/* permissive version of pkt_format_html(3).
   It does the same thing as pkt_format_html(3) excepted that it uses
   pkt_format_permissive_do(3) instead of pkt_format_do(3) as t_tmpl_do_proc.
   Returns 0 if OK. Might return various errors such as -ERR_BO (buffer 
   overflow) */   
t_status		pkt_format_html_permissive(pkt,
						   tmpl_str,
						   vec_prefixes,
						   extra_vars,
						   str,
						   max_len,
						   htmlize_str,
						   htmlize_max_len)
t_pkt			*pkt;		/* Reference packet */
char			*tmpl_str;	/* Template string */
t_vec			*vec_prefixes;	/* Might be NULL */
t_dict			*extra_vars;	/* Might be NULL */
char			*str;		/* Dest. string */
int			max_len;	/* Destr string length */
char			*htmlize_str;	/* Htmlize string buffer */
int			htmlize_max_len;/* Htmlize string buffer length */
{
  t_pkt_format_do_data	pfdd;

  pfdd.pkt = pkt;
  pfdd.vec_prefixes = vec_prefixes;
  pfdd.extra_vars = extra_vars;
  pfdd.get_field_to_str_proc = pkt_get_field_to_str;
  pfdd.tmpl_do_proc = (t_tmpl_do_proc)pkt_format_permissive_do;
  pfdd.htmlize_str = htmlize_str;
  pfdd.htmlize_max_len = htmlize_max_len;
  return (tmpl_str_to_str(tmpl_str,
			  (t_tmpl_do_proc)pkt_format_htmlize_do,
			  &pfdd,
			  str,
			  max_len));
}

/* gets all the possible choices of a field value (managing options).
   It finds the correct layer then calls LAY_GET_FIELD_TYP to access
   directly the "typ" of field. It then calls TYP_GET_CHOICES to
   retrieve all possible choices. It also manages the special field
   "Layer" (used to cast packet).
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_choices_i(pkt,fq_field,vec_str,id)
t_pkt			*pkt;		/* Reference packet */
char			*fq_field;	/* E.g "ip[0].Tos" */
t_vec			*vec_str;	/* Vector of strings */
t_id			*id;		/* Pointers hash table */
{
  t_status		status;
  char			*field;
  int			layerlen;
  t_pkt			sub_pkt;
  t_pkt			tmp_pkt;
  t_get_field_data	gfd;

  if (!strcmp(fq_field,"Layer"))
    {
      return (lay_get_choices(vec_str));
    }
  if ((field = index(fq_field,'.')) == NULL)
    return (-ERR_NOSUCHFIELD);
  layerlen = field - fq_field;
  field++;
  PKT_COPY(pkt,&sub_pkt);
  do 
    {
      char		name[STR_BUFSIZ];
      t_buf		b;
      t_has_opt_data	hsd;
      t_bridled_str	namebs;
      
      namebs.str = name;
      namebs.max_len = sizeof (name);
      name[0] = 0;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_NAME,
			    id,
			    &namebs)) < 0)
	return (status);
      if (!strncmp(fq_field,name,layerlen))
	{
	  t_get_field_typ_data	gftd;
	  t_extract_data	ed;	

	  gfd.optional_id = id;
	  gfd.field = field;
	  gfd.b.buf = sub_pkt.buf;
	  gfd.b.len = sub_pkt.len;
	  if ((status = lay_msg(sub_pkt.mp,
				LAY_GET_FIELD_TYP,
				&gfd,
				&gftd)) < 0)
	    return (status);
	  ed.b.buf = sub_pkt.buf;
	  ed.b.len = sub_pkt.len;
	  ed.data = gftd.data; 
	  return (typ_msg(gftd.mp,
			  TYP_GET_CHOICES,
			  &ed,
			  vec_str));
	}
      b.buf = sub_pkt.buf;
      b.len = sub_pkt.len;
      if ((status = lay_msg(sub_pkt.mp,
			    LAY_HAS_OPT,
			    &b,
			    &hsd)) < 0)
	{
	  if (status != -ERR_NOMETHOD)
	    return (status);
	}
      else
	{
	  if (hsd.has_opt)
	    {
	      t_pkt         opt_pkt;
	      
	      if ((hsd.opt_off + hsd.opt_len) > sub_pkt.len)
		return (-ERR_BADMATCH);
	      PKT_COPY(pkt,&opt_pkt);
	      opt_pkt.mp = hsd.opt_mp;
	      opt_pkt.buf = sub_pkt.buf + hsd.opt_off;
	      opt_pkt.len = hsd.opt_len;
	      status = pkt_get_choices_i(&opt_pkt,
					 fq_field,
					 vec_str,
					 id);
	      if (status != -ERR_NOSUCHFIELD)
		return (status);
	    }
	}
      PKT_COPY(&sub_pkt,&tmp_pkt);
    } while (pkt_sub_id(&tmp_pkt,
			&sub_pkt,
			id) == 0);
  return (-ERR_NOSUCHFIELD);
}

/* retrieves all the possible choices of a packet field.
   It is a convenience function. It allocates a tiny id (HASH_TINY_BASE) and
   calls pkt_get_choices_i(3).
   Returns 0 if OK. Might return various errors */
t_status		pkt_get_choices(pkt,field,vec_str)
t_pkt			*pkt;		/* Reference packet */
char			*field;		/* E.g "ip[0].Tos" */
t_vec			*vec_str;	/* Vector of strings */
{
  t_id			*id;
  t_status		status;

  if ((id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  status = pkt_get_choices_i(pkt,
			     field,
			     vec_str,
			     id);
  id_delete(id);
  return (status);
}

/* truncates a packet to name.
   It is the advanced version of pkt_trunc(3). It looks up a layer name
   instead of a layer type (assuming it could be many layers with the
   same type e.g an icmp unreachable message which contains an ip header).
   Note that it uses pkt_subn_from_name(3) (with a fixed id size).
   Returns 0 if OK. Might return various errors. */
t_status		pkt_trunc_from_name(pkt,name,alloc_proc,free_proc)
t_pkt			*pkt;		/* Reference packet */
char			*name;		/* E.g "ip[1]". */
t_alloc_proc		alloc_proc;
t_free_proc		free_proc;
{
  t_status		status;
  t_pkt			sub_pkt;
  t_pkt			sub_sub_pkt;
  char			*nbuf;
  int			nlen;

  if ((status = pkt_subn_from_name(pkt,&sub_pkt,name)) < 0)
    return (status);
  if ((status = pkt_sub(&sub_pkt,&sub_sub_pkt)) < 0)
    return (status);
  nlen = pkt->len - sub_sub_pkt.len;
  if ((nbuf = alloc_proc(nlen * sizeof (char),
			 "pkt",
			 "pkt_trunc:buf",
			 &status)) == NULL)
    return (status);
  bcopy(pkt->buf,nbuf,nlen);
  free_proc(pkt->buf,
	    "pkt",
	    "*:buf");
  pkt->buf = nbuf;
  pkt->len = nlen;
  return (0);
}

/* extracts a packet from name.
   It is the advanced version of pkt_extract(3). It looks up a layer name
   instead of a layer type (assuming it could be many layers with the
   same type e.g an icmp unreachable message which contains an ip header).
   Note that it uses pkt_subn_from_name(3) (with a fixed id size).
   Returns 0 if OK. Might return various errors. */
t_status		pkt_extract_from_name(pkt,name,alloc_proc,free_proc)
t_pkt			*pkt;		/* Reference packet */
char			*name;		/* E.g "ip[1]" */
t_alloc_proc		alloc_proc;
t_free_proc		free_proc;
{
  t_status		status;
  char			*nbuf;
  t_pkt			sub_pkt;

  if ((status = pkt_subn_from_name(pkt,&sub_pkt,name)) < 0)
    return (status);
  if ((nbuf = alloc_proc(sub_pkt.len * sizeof (char),
			 "pkt",
			 "pkt_extract:buf",
			 &status)) == NULL)
    return (status);
  free_proc(pkt->buf,
	    "pkt",
	    "*:buf");
  bcopy(sub_pkt.buf,nbuf,sub_pkt.len);
  pkt->buf = nbuf;
  pkt->len = sub_pkt.len;
  pkt->mp = sub_pkt.mp;
  return (0);
}
