/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "layer.h"
#include "typ_msg.h"
#include "typ_16.h"

t_u16			safe_ntohs(p_nu16)
t_u16			*p_nu16;	/* Pointer to network unsigned short */
{
  t_u16			a_u16;
  
  FBCOPY(p_nu16,&a_u16,sizeof (t_u16));
  return (UNSAFE_NTOHS(a_u16));
}

VOID_FUNC		safe_htons(a_u16,p_nu16)
t_u16			a_u16;		/* Aligned host unsigned short */
t_u16			*p_nu16;	/* Pointer to network unsigned short */
{
  a_u16 = UNSAFE_HTONS(a_u16);  
  FBCOPY(&a_u16,p_nu16,sizeof (t_u16));
}

#define TYP_OFF_16_GENERIC \
case TYP_OFF:\
{\
  TYP_OFF_ARGS(ed,off);\
\
  (*off) = 2;\
  return (0);\
}

/* is a t_msg_proc.
   Manages network signed shorts. */
t_status		typ_ns16_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("ns16");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_s16		s16;
	
	if (ed->b.len < sizeof (s16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&s16,sizeof (s16));
	s16 = UNSAFE_NTOHS(s16);
	return (long_to_str((signed long)s16,
			    layer_base,
			    bs->str,
			    bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_s16		s16;
	t_status	status;	
	int		basenum;
	signed long	num;

	if (ed->b.len < sizeof (s16))
	  return (-ERR_TRUNC);
	if ((status = long_from_str(str,
				    &basenum,
				    &num)) < 0)
	  return (status);
	s16 = (t_s16)num;
	s16 = UNSAFE_HTONS(s16);
	FBCOPY(&s16,ed->b.buf,sizeof (s16));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned shorts. */
t_status		typ_nu16_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu16");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u16		u16;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16);
	return (ulong_to_str((unsigned long)u16,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u16		u16;
	t_status	status;	
	int		basenum;
	unsigned long	num;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	if ((status = ulong_from_str(str,
				     &basenum,
				     &num)) < 0)
	  return (status);
	u16 = (t_u16)num;
	u16 = UNSAFE_HTONS(u16);
	FBCOPY(&u16,ed->b.buf,sizeof (u16));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned shorts. */
t_status		typ_u16_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u16");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u16		u16;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	return (ulong_to_str((unsigned long)u16,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u16		u16;
	t_status	status;	
	int		basenum;
	unsigned long	num;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	if ((status = ulong_from_str(str,
				     &basenum,
				     &num)) < 0)
	  return (status);
	u16 = (t_u16)num;
	FBCOPY(&u16,ed->b.buf,sizeof (u16));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned shorts associations.
   Requires a t_assoc array as
   ed->data. Is able to lookup tokens in this array as base for
   extracting or inserting strings. */
t_status		typ_nu16assoc_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu16assoc");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u16		u16;
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16);
	if (assoc = assoc_str_int_from_right(assocs,(int)u16))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)u16,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u16		u16;
	t_status	status;	
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	if (assoc = assoc_str_ptr_from_left(assocs,str))
	  u16 = (t_u16)(t_u32)(assoc->right);
	else
	  {
	    int			basenum;
	    unsigned long	num;

	    if ((status = ulong_from_str(str,
					 &basenum,
					 &num)) < 0)
	      return (status);
	    u16 = (t_u16)num;
	  }
	u16 = UNSAFE_HTONS(u16);
	FBCOPY(&u16,ed->b.buf,sizeof (u16));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_assoc		*assocs;
	t_assoc		*assoc;
	
	assocs = (t_assoc *)(ed->data);
	assoc = assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned shorts bitfields.
   Requires a t_bit_field structure as
   ed->data. Is able to extract or insert bitfield values. */
t_status		typ_nu16bitfield_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu16bitfield");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u16		u16;
	t_bit_field	*bf;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16);
	bf = (t_bit_field *)(ed->data);
	u16 = *bit_field_u16_get(&u16,bf->from,bf->to);
	return (ulong_to_str((unsigned long)u16,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u16		u16;
	t_u16		value;
	t_bit_field	*bf;
	int		basenum;
	unsigned long	num;
	t_status	status;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16); 
	bf = (t_bit_field *)(ed->data);
	if ((status = ulong_from_str(str,
				     &basenum,
				     &num)) < 0)
	  return (status);
	value = (t_u16)num;
	bit_field_u16_set(&u16,bf->from,bf->to,value);
	u16 = UNSAFE_HTONS(u16); 
	FBCOPY(&u16,ed->b.buf,sizeof (u16));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned shorts bitfield associations.
   Requires a t_bit_field_assocs_data structure as
   ed->data. Is able to extract or insert bitfield values taken from
   a t_assoc array. Do not try to generalize more! */
t_status			typ_nu16bitfieldassoc_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu16bitfieldassoc");
      TYP_OFF_16_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u16			u16;
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16);
	bfad = (t_bit_field_assocs_data *)(ed->data);
	u16 = *bit_field_u16_get(&u16,
				 bfad->bit_field->from,
				 bfad->bit_field->to);
	if (assoc = assoc_str_int_from_right(bfad->assocs,(int)u16))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)u16,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u16			u16;
	t_u16			value;
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;

	if (ed->b.len < sizeof (u16))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u16,sizeof (u16));
	u16 = UNSAFE_NTOHS(u16); 
	bfad = (t_bit_field_assocs_data *)(ed->data);
	if (assoc = assoc_str_ptr_from_left(bfad->assocs,str))
	  value = (t_u16)(t_u32)(assoc->right);
	else
	  {
	    int			basenum;
	    unsigned long	num;

	    if ((status = ulong_from_str(str,
					 &basenum,
					 &num)) < 0)
	      return (status);
	    value = (t_u16)num;
	  }
	bit_field_u16_set(&u16,
			  bfad->bit_field->from,
			  bfad->bit_field->to,
			  value);
	u16 = UNSAFE_HTONS(u16); 
	FBCOPY(&u16,ed->b.buf,sizeof (u16));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;
	
	bfad = (t_bit_field_assocs_data *)(ed->data);
	assoc = bfad->assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	
