/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <signal.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xmu/Converters.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Panner.h>
#include <X11/Xaw/Porthole.h>
#include "SmeBSBM.h"
#include "Xws.h"
#include "Xmg.h"
#include "XmgFancy.h"
#include "XmgFileGet.h"
#include "pkt.h"
#include "xt_macro.h"
#include "xipcf.h"
#include "xip.h"
#include "xippkt.h"
#include "xippktvec.h"
#include "xipimg.h"
#include "xipshcut.h"
#include "layer.h"
#include "lay_data.h"
#include "lay_inaddr.h"
#include "lay_ip.h"
#include "lay_tcp.h"
#include "xipicon.xbm"
#include "gray.xbm"
#include "check.xbm"

extern t_vec		*xip_pkt_vec;
extern pcap_t		*pcap;
extern int		user_dropped;

char			*xip_cf = NULL;
t_boolean		verbose = FALSE;
t_boolean		resolve = TRUE;
t_boolean		big_packets = FALSE;
#ifdef DEBUG_MALLOC
t_boolean		debug_malloc = FALSE;
t_boolean		debug_malloc_verbose = FALSE;
#endif
t_boolean		help = FALSE;
char			*pcap_dev = NULL;
char			hard_filter[STR_BUFSIZ];
int			pcap_snaplen = 68;
t_boolean		pcap_promisc = TRUE;
int			pcap_to_ms = 10;
t_boolean		pcap_optimize = TRUE;
char			*pcap_fname = NULL;
int			pcap_cnt = -1;

XtAppContext		app_context;
Widget			toplevel;
Widget			pktbox;
Widget			pannershell;
Widget			pannerbox;
Widget			porthole;    
Widget			aboutshell = NULL;
Widget			aboutfancy;
Widget			xphelpshell;
Widget			xphelpfancy;
Widget			flowbsb = NULL;
Widget			helpmodebsb = NULL;
Widget			pinmodebsb = NULL;
Widget			scrollmodebsb = NULL;
Pixmap			xipicon_pixmap;
Pixmap			gray_pixmap;
Pixmap			check_pixmap;
t_boolean		xphelpvisible;
t_boolean		capflow = TRUE;

t_opt			main_opts[] = 
{
  {"-h",	(t_opt_proc)opt_true,		(t_off)&help,
   NULL,"this help",0},
  {"-v",	(t_opt_proc)opt_true,		(t_off)&verbose,
   NULL,"turn on verbose mode",0},
#ifdef DEBUG_MALLOC
  {"-dm",	(t_opt_proc)opt_true,		(t_off)&debug_malloc,
   NULL,"turn on debug_malloc",0},
  {"-dmv",	(t_opt_proc)opt_true,		(t_off)&debug_malloc_verbose,
   NULL,"turn on debug_malloc_verbose",0},
#endif
  {"-cf",	(t_opt_proc)opt_str,		(t_off)&xip_cf,
   "xip_cf",NULL,0},
  {"-i",	(t_opt_proc)opt_str,		(t_off)&pcap_dev,
   "dev",NULL,0},
  {"-s",	(t_opt_proc)opt_int,		(t_off)&pcap_snaplen,
   "snaplen",NULL,0},
  {"-p",	(t_opt_proc)opt_false,		(t_off)&pcap_promisc,
   NULL,"no promiscuous mode",0},
  {"-t",	(t_opt_proc)opt_int,		(t_off)&pcap_to_ms,
   "to_ms","timeout in milliseconds",0},
  {"-O",	(t_opt_proc)opt_false,		(t_off)&pcap_optimize,
   NULL,"don't optimize",0},
  {"-r",	(t_opt_proc)opt_str,		(t_off)&pcap_fname,
   "tcpdump_file_name",NULL,0},
  {"-c",	(t_opt_proc)opt_int,		(t_off)&pcap_cnt,
   "count","used with -r",0},
  {"-n",	(t_opt_proc)opt_false,		(t_off)&resolve,
   NULL,"turn off resolve mode",0},
  {"-B",	(t_opt_proc)opt_true,		(t_off)&big_packets,
   NULL,"big packets",0},
};

char			*fallback[] = 
{
  "*xmgPromptBox*okCommand.label:	OK",
  "*xmgPromptBox*cancelCommand.label:	Cancel",
  "*xmgPromptBox*Label.borderWidth:	0",
  "*xmgPromptBox*Label.font:-adobe-times-bold-r-normal-*-14-*-*-*-*-*-*-*",

  "*xipCfBox*XmgFancy.margin:		12",
  
  "*xipAboutBox*XmgFancy.margin:	12",
  
  "*flowBSB.rightMargin:		16",
  "*flowBSB.label:			flow",
  "*helpModeBSB.rightMargin:		16",
  "*helpModeBSB.label:			Help mode",
  "*pinModeBSB.rightMargin:		16",
  "*pinModeBSB.label:			Pin mode",
  "*scrollModeBSB.rightMargin:		16",
  "*scrollModeBSB.label:		Scroll mode",

  "*xmgFileGetShell.width:		400",
  "*xmgFileGetShell.height:		400",
  "*FileNominator.filterDirectoryNames:	False",

  "*Panner.rubberBand:			True",
  
  "*XmgFancy.subEltVecBase:		8",
  "*XmgFancy*xmgFancyViewSource.width:	400",
  "*XmgFancy*xmgFancyViewSource.height:	400",
  "*XmgFancy*xmgFancyViewSource*font:	-adobe-courier-medium-r-normal-*-14-*-*-*-*-*-*-*",
  "*XmgFancy*xmgFancyInput*Scrollbar.thickness:	3",
  "*XmgFancy*Toggle.font:		5x8",

  "*smallPktFancy.debugKeepSource:	False",
  
  "*smallPktFancy.normalFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.smallFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy._tinyFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.italicFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.boldFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.header1FontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.header2FontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  "*smallPktFancy.fixedFontName: -*-*-*-*-*-*-2-*-*-*-*-*-*-*",
  
  "*font:	-adobe-times-medium-r-normal-*-14-*-*-*-*-*-*-*",
  
  "*Xip.width:				500",
  "*Xip.height:				300",

  NULL
};

/* gives a short option list then exits. */
VOID_FUNC		usage(VOID_DECL)
{
  opt_usage(stderr,main_opts,ARRAY_COUNT(main_opts),TRUE);
  exit(1);
}

/* check eventual options coherency, prints out a recall of options. */
t_status		check_options(VOID_DECL)
{
  if (verbose)
    {
      fprintf(stderr,"options:\n");
      fprintf(stderr,"\t-v=%s\n",verbose?"verbose":"not verbose");
#ifdef DEBUG_MALLOC
      fprintf(stderr,"\t-dm=%s\n",debug_malloc?"debug_malloc":
	      "no debug_malloc");
      fprintf(stderr,"\t-dmv=%s\n",debug_malloc_verbose?"debug_malloc_verbose":
	      "no debug_malloc_verbose");
#endif
      fprintf(stderr,"\t-cf=%s\n",xip_cf);
      fprintf(stderr,"\t-i=%s\n",pcap_dev?pcap_dev:"");
      fprintf(stderr,"\t-s=%d\n",pcap_snaplen);
      fprintf(stderr,"\t-p=%s\n",pcap_promisc?"promisc":"no promisc");
      fprintf(stderr,"\t-t=%d\n",pcap_to_ms);
      fprintf(stderr,"\t-O=%s\n",pcap_optimize?"optimize":"don't optimize");
      fprintf(stderr,"\t-r=%s\n",pcap_fname?pcap_fname:"");
      fprintf(stderr,"\t-c=%d\n",pcap_cnt);
      fprintf(stderr,"\t-n=%s\n",resolve?"resolve":"don't resolve");
    }
  return (0);
}

/* destroys (mostly) everything then exits.
   Note: XtDestroyWidget(toplevel) and 
   XtDestroyApplicationContext(app_context) don't take effect since there
   are 2 phases of destroy in Xt. It's why we have to "clear" "packet box"
   manually before debugging memory. */
VOID_FUNC		myexit(code)
int			code;	/* exit(2) code */
{
  signal(SIGINT,SIG_DFL);
  vars_destroy();
  destroy_extra_vars();
  pktbar_destroy();
  methods_destroy();
  xippktcut_destroy();
  xippktvec_destroy();
  nbgethost_destroy();
  layer_destroy();
  shortcuts_destroy();
  XmgDestroy();
  xt_macro_destroy();
#ifdef NOTDEF  
  XtDestroyWidget(toplevel);
  XtDestroyApplicationContext(app_context);
#endif
#ifdef DEBUG_MALLOC
  if (debug_malloc)
    {
      gdm_status();
    }
#endif
  exit(code);
}

/* is an XtCallbackProc.
   Does nothing. */
VOID_FUNC			bsb_nop(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  /* NOP */
}

/* is an XtCallbackProc.
   set/unset "check" bitmap to the specified BSB */
VOID_FUNC			XipSetCheck(w,value)
Widget				w;
Boolean				value;
{
  XtVaSetValues(w,
		XtNrightBitmap,	value?check_pixmap:None,
		NULL);
}

/* is an XtCallbackProc.
   Controls flow from GUI. */
VOID_FUNC			bsb_flow(w,closure,ptr)
Widget				w;
VOID_PTR			closure;
XtPointer			ptr;
{
  capflow = !capflow;
  XipSetCheck(flowbsb,capflow);
}

/* is an XtCallbackProc.
   Calls myexit(3) */
VOID_FUNC			bsb_quit(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  myexit(0);
}

/* is an XtCallbackProc.
   Clears the packet box */
VOID_FUNC			bsb_clear(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  WidgetList			children;
  Cardinal			num_children;
  int				i;

  XtVaGetValues(pktbox,
		XtNchildren,	&children,
		XtNnumChildren,	&num_children,
		NULL);
  XtUnmanageChildren(children,
		     num_children);
  i = 0;
  while (i < num_children)
    {
      XtDestroyWidget(children[i]);
      i++;
    }
}

/* is a XtCallbackProc.
   Open a new packet of one byte in a new frame */
VOID_FUNC			bsb_new_pkt(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  t_pkt				*npkt;
  t_status			status;
  
  if ((npkt = xip_pkt_new(1,
			  lay_data_msg,
			  &status)) == NULL)
    {
      XmgErrPrint(-status,"xip_pkt_new");
      return ;
    }
  XipCreatePkt(toplevel,npkt,TRUE);
}

/* is a XtCallbackProc.
   Calls XipShowCfBox(3) */
VOID_FUNC			bsb_edit_config(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  XipShowCfBox(toplevel);
}

/* is an XtCallbackProc.
   Asks for a file name via a file selector then save config */
VOID_FUNC			bsb_save_config(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  t_status			status;
  char				*fname;

  if (fname = XmgFileGet("*.cf",FALSE))
    {
      if ((status = xip_cf_save(fname)) < 0)
	XmgErrPrint(-status,"xip_cf_save %s",fname);
    }
}

/* is an XtCallbackProc.
   Asks for a file name via a file selector then open a packets collection */
VOID_FUNC			bsb_load_collection(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  t_status			status;
  char				*fname;

  if (fname = XmgFileGet("*.pkc",TRUE))
    {
      if ((status = xip_load_collection(fname)) < 0)
	XmgErrPrint(-status,"xip_load_collection %s",fname);
    }
}

/* is an XtCallbackProc.
   Asks for a file name via a file selector then save a packets collection */
VOID_FUNC			bsb_save_collection(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  t_status			status;
  char				*fname;

  if (fname = XmgFileGet("*.pkc",FALSE))
    {
      if ((status = xip_save_collection(fname)) < 0)
	XmgErrPrint(-status,"xip_save_collection %s",fname);
    }
}

/* is a XtCallbackProc.
   Popups the packet box panner */
VOID_FUNC			bsb_remote_control(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  XtPopup(pannershell,XtGrabNone);
}

/* is an XtCallbackProc.
   Make the packet help sticky or not */
VOID_FUNC			bsb_pin_mode(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  if (xt_resources.pinMode)
    {
      xt_resources.pinMode = FALSE;
      xphelpvisible = FALSE;
      XtPopdown(xphelpshell);
    }
  else
    xt_resources.pinMode = TRUE;
  XipSetCheck(pinmodebsb,xt_resources.pinMode);
}

/* is an XtCallbackProc.
   Make the packet help active or not */
VOID_FUNC			bsb_help_mode(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  if (xt_resources.helpMode)
    {
      xt_resources.helpMode = FALSE;
      xphelpvisible = FALSE;
      XtPopdown(xphelpshell);
    }
  else
    {
      xt_resources.helpMode = TRUE;
    }
  XipSetCheck(helpmodebsb,xt_resources.helpMode);
}

/* is an XtCallbackProc.
   Set scroll mode for packet box. */
VOID_FUNC			bsb_scroll_mode(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  if (xt_resources.scrollMode)
    {
      xt_resources.scrollMode = FALSE;
      xphelpvisible = FALSE;
      XtPopdown(xphelpshell);
    }
  else
    {
      xt_resources.scrollMode = TRUE;
    }
  XipSetCheck(scrollmodebsb,xt_resources.scrollMode);
}

char				*about_tmpl = 
"\n\
<html>\n\
<head>\n\
<title>Xip: about</title>\n\
</head>\n\
<body>\n\
<h1>About</h1>\n\
<hr>\n\
<br>\n\
<form>\n\
<table width=100%%>\n\
<tr>\n\
<td align=center>\n\
<img src=\"internal:xipicon.xbm\">\n\
</td>\n\
<td align=center>\n\
<table>\n\
<tr>\n\
<td align=center width=100%%>\n\
<b>\n\
%product% %version% %vendor%\n\
</b>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%>\n\
<small>\n\
Report any problems to \n\
<a href=\"%mail_url%\">\n\
%mail_url%\n\
</a>.<br>\n\
Find additional documentation at <br>\n\
<a href=\"%http_url%\">\n\
%http_url%\n\
</a>.\n\
</small>\n\
</td>\n\
</tr>\n\
</table>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%>\n\
<small>\n\
Packets: %npackets% treated, %user_dropped% dropped.\n\
<br>\n\
Pcap: %ps_recv% received, %ps_drop% dropped, \n\
%ps_ifdrop% dropped by interface.\n\
<br>\n\
Memory: %mem_in_use%b used, %mem_cumulated%b cumulated.\n\
</small>\n\
</td>\n\
</tr>\n\
</table>\n\
<br>\n\
<hr>\n\
<table width=100%%>\n\
<tr>\n\
<td align=center>\n\
<input type=submit value=\"Dismiss\">\n\
</td>\n\
</tr>\n\
</table>\n\
</form>\n\
</body>\n\
</html>\n\
\n\
";

/* is a t_tmpl_do_proc.
   It is used by XipShowAboutBox(3). */ 
t_status			about_format_do(bs,var,i)
t_bridled_str			*bs;
char				*var;
int				i;
{
  int				varlen;
  long				signedvalue;
  t_status			status;

  varlen = strlen(var);
  if (varlen == 0)
    return (str_cat_char(bs->str,bs->max_len,'%'));
  else
    {
      struct pcap_stat		ps;

      if (!strcmp(var,"product"))
	return (str_cat_str(bs->str,bs->max_len,PRODUCT));
      if (!strcmp(var,"version"))
	return (str_cat_str(bs->str,bs->max_len,VERSION));
      if (!strcmp(var,"vendor"))
	return (str_cat_str(bs->str,bs->max_len,VENDOR));
      if (!strcmp(var,"mail_url"))
	return (str_cat_str(bs->str,bs->max_len,MAIL_URL));
      if (!strcmp(var,"http_url"))
	return (str_cat_str(bs->str,bs->max_len,HTTP_URL));
      if (!strcmp(var,"npackets"))
	return (ulong_to_str((unsigned long)VEC_COUNT(xip_pkt_vec),
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (!strcmp(var,"user_dropped"))
	return (ulong_to_str((unsigned long)user_dropped,
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (!strcmp(var,"mem_in_use"))
	return (ulong_to_str((unsigned long)gdm.in_use,
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (!strcmp(var,"mem_cumulated"))
	return (ulong_to_str((unsigned long)gdm.cumulated,
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (pcap_stats(pcap,&ps) < 0)
	return (0);
      if (!strcmp(var,"ps_recv"))
	return (ulong_to_str((unsigned long)ps.ps_recv,
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (!strcmp(var,"ps_drop"))
	return (ulong_to_str((unsigned long)ps.ps_drop,
			     layer_base,
			     bs->str,
			     bs->max_len));
      if (!strcmp(var,"ps_ifdrop"))
	return (ulong_to_str((unsigned long)ps.ps_ifdrop,
			     layer_base,
			     bs->str,
			     bs->max_len));
    }
  return (0);
}

/* is an XtCallbackProc.
   Does "magic" things! */
VOID_FUNC			XipAboutFancyLink(w,xp,cbs)
Widget				w;
t_xip_pkt			*xp;
XmgFancyLinkCallbackStruct	*cbs;
{
  Window			win;

  if (win = XwsFindNetscapeWindow(XtDisplay(w),
				  DefaultRootWindow(XtDisplay(w)),
				  NULL,
				  XwsNetscapeWindow))
     {
       char			buf[STR_BUFSIZ];
       t_status			status;

       buf[0] = 0;
       if ((status = str_cat_fmt_va(buf,
				    sizeof (buf),
				    "OpenURL(%s)",
				    cbs->href)) < 0)
	 return ;
       XwsSendNavigatorCommand(XtDisplay(w),
			       win,
			       buf);
     }
}

/* is an XtCallbackProc.
   Popdowns about shell */
VOID_FUNC			XipAboutFancyForm(w,shell,cbs)
Widget				w;
Widget				shell;
XmgFancyFormCallbackStruct	*cbs;
{
  XtPopdown(shell);
}

/* is an XtCallbackProc.
   Set title of about shell */
VOID_FUNC			XipAboutFancyTitle(w,shell,cbs)
Widget				w;
Widget				shell;
XmgFancyTitleCallbackStruct	*cbs;
{
  XtVaSetValues(shell,
		XtNtitle,	cbs->title,
		XtNiconName,	cbs->title,
		NULL);
}

/* shows about, creates it if it doesn't exist */
VOID_FUNC			XipShowAboutBox(parent)
Widget				parent;
{
  char				html[BUFSIZ];
  t_status			status;

  if (!aboutshell)
    {
      INTERFACE(parent,
		GETCHILDSHELL(aboutshell)
		("xipAboutBox",transientShellWidgetClass,
		 XtNmappedWhenManaged,	False,
		 XtNtransientFor,	parent,
		 XtNallowShellResize,	True,
		 XtNiconPixmap,		xipicon_pixmap,
		 GETCHILD(aboutfancy)
		 ("aboutFancy",xmgFancyWidgetClass,
		  XtNdoCutAndPaste,	True,
		  CALLBACK(XtNlinkCallback,
			   (XtCallbackProc)XipAboutFancyLink,
			   aboutshell),
		  CALLBACK(XtNtitleCallback,
			   (XtCallbackProc)XipAboutFancyTitle,
			   aboutshell),
		  CALLBACK(XtNformCallback,
			   (XtCallbackProc)XipAboutFancyForm,
			   aboutshell),
		  CALLBACK(XtNimgCallback,
			   (XtCallbackProc)XipFancyImg,
			   aboutshell),
		  END),
		 MANAGE,
		 END),
		ENDINTERFACE);
      XmgSetPopdownOnDelete(aboutshell);
    }
  else
    {
      XmgFancyReset(aboutfancy);
    }
  html[0] = 0;
  if ((status = tmpl_str_to_str(about_tmpl,
				(t_tmpl_do_proc)about_format_do,
				NULL,
				html,
				sizeof (html))) < 0)
    return ;
  XmgFancyParseBuf(aboutfancy,
		   html,
		   strlen(html));
  XmgCenterWidget(aboutshell);
  XtPopup(aboutshell,XtGrabExclusive);
}

/* is an XtCallbackProc.
   Shows about */
VOID_FUNC			bsb_about(w,closure,ptr)
Widget				w;
XtPointer			closure;
XtPointer			ptr;
{
  XipShowAboutBox(toplevel);
}

t_boolean			from_panner = FALSE;

/* is an XtCallbackProc.
   Moves the packet box */
VOID_FUNC			report_panner(w,closure,report)
Widget				w;
XtPointer			closure;
XawPannerReport			*report;
{
  from_panner = TRUE;
  XtVaSetValues(pktbox,
		XtNx,		-report->slider_x,
		XtNy,		-report->slider_y,
		NULL);
  from_panner = FALSE;
}

/* performs the action of moving "packet box" to a new value */
VOID_FUNC			move_pktbox(VOID_DECL)
{
  Dimension			width;
  Dimension			height;
  Dimension			port_width;
  Dimension			port_height;
  static t_boolean		selfex = FALSE;

  if (selfex)
    return ;
  XtVaGetValues(pktbox,
		XtNwidth,	&width,
		XtNheight,	&height,
		NULL);
  XtVaGetValues(porthole,
		XtNwidth,	&port_width,
		XtNheight,	&port_height,
		NULL);
  if (xt_resources.smoothScrollMode)
    {
      int			i;
      Position			x;
      Position			y;
      Position			final_x;
      Position			final_y;
      int			offset;

      selfex = TRUE;
      XtVaGetValues(pktbox,
		    XtNx,	&x,
		    XtNy,	&y,
		    NULL);
      
      final_x = -width + port_width;
      final_y = -height + port_height;
      if (x > final_x)
	{
	  i = x;
	  while (i > final_x)
	    {
	      
	      XtVaSetValues(pktbox,
			    XtNx,i,
			    NULL);
	      offset = (i - final_x) / 2;
	      i -= (offset < xt_resources.smoothScrollLimit)?
		xt_resources.smoothScrollOffset:offset;
	    }
	}
      if (y > final_y)
	{
	  i = y;
	  while (i > final_y)
	    {
	      XtVaSetValues(pktbox,
			    XtNy,i,
			    NULL);
	      offset = (i - final_y) / 2;
	      i -= (offset < xt_resources.smoothScrollLimit)?
		xt_resources.smoothScrollOffset:offset;
	    }
	}
      selfex = FALSE;
    }
  else
    {
      XtVaSetValues(pktbox,
		    XtNx,	-width + port_width,
		    XtNy,	-height + port_height,
		    NULL);
    }
}

/* is an XtCallbackProc.
   Resizes the packet box panner */
VOID_FUNC			report_porthole(w,panner,report)
Widget				w;
Widget				panner;
XawPannerReport			*report;
{
  if (!from_panner && xt_resources.scrollMode)
    move_pktbox();
  XtVaSetValues(panner,
		XtNsliderX,	report->slider_x,
		XtNsliderY,	report->slider_y,
		NULL);
  if (report->changed != (XawPRSliderX | XawPRSliderY))
    {
      XtVaSetValues(panner,
		    XtNsliderWidth,	report->slider_width,
		    XtNsliderHeight,	report->slider_height,
		    XtNcanvasWidth,	report->canvas_width,
		    XtNcanvasHeight,	report->canvas_height,
		    NULL);
    }
}

/* is an XtActionProc.
   Maximizes packet */
VOID_FUNC		XipActionMaximizePkt(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  t_xip_pkt		*xp;
  t_status		status;
  Widget		fancy;
  t_xip_method_data	xmd;

  if (!strcmp(XtName(w),"smallPktFancy"))
    fancy = w;
  else
    if (!(fancy = XtNameToWidget(w,"smallPktFancy")))
      {
	err_print(ERR_INTERNAL,"XipActionMaximizePkt");
	return ;
      }
  XtVaGetValues(fancy,
		XtNuserData,	&xp,
		NULL);
  xmd.xp = xp;
  xmd.new_window = TRUE;
  parse_method(&xmd,"extract()");
}

/* is an XtActionProc.
   Minimizes packet */
VOID_FUNC		XipActionMinimizePkt(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  t_xip_pkt		*xp;
  t_status		status;
  Widget		fancy;
  t_xip_method_data	xmd;

  if (!strcmp(XtName(w),"pktFancy"))
    fancy = w;
  else
    if (!(fancy = XtNameToWidget(w,"pktFancy")))
      {
	err_print(ERR_INTERNAL,"XipActionMinimizePkt");
	return ;
      }
  XtVaGetValues(fancy,
		XtNuserData,	&xp,
		NULL);
  xmd.xp = xp;
  xmd.new_window = TRUE;
  parse_method(&xmd,"extracttopktbox()");
}

/* is an XtActionProc.
   Popups packet help */
VOID_FUNC		XipActionPopupPktHelp(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  t_xip_pkt		*xp;
  t_status		status;
  Position		x;
  Position		y;
  Position		xroot;
  Position		yroot;
  Dimension		height;
  char			html[STR_BUFSIZ];

  if (!xt_resources.helpMode)
    return ;
  if (!xt_resources.pinMode)
    if (xphelpvisible)
      return ;
  xphelpvisible = TRUE;
  assert(event->type == EnterNotify);
  XtVaGetValues(w,
		XtNuserData,	&xp,
		NULL);
  XtVaGetValues(w,
		XtNx,		&x,
		XtNy,		&y,
		XtNheight,	&height,
		NULL);
  XtTranslateCoords(pktbox,
		    x,
		    y,
		    &xroot,
		    &yroot);
  XtVaSetValues(xphelpshell,
		XtNx,	xroot + xt_resources.pktHelpOffset,
		XtNy,	yroot + height + xt_resources.pktHelpOffset,
		NULL);
  if ((status = shortcuts_do(xp->pkt,
			     html,
			     sizeof (html))) < 0)
    ;
  XmgFancyReset(xphelpfancy);
  XmgFancyParseBuf(xphelpfancy,html,strlen(html));
  XmgTallyWidgetWithScreen(xphelpshell);
  XtPopup(xphelpshell,XtGrabNone);
}

/* is an XtActionProc.
   Popdowns packet help */
VOID_FUNC		XipActionPopdownPktHelp(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  if (xt_resources.pinMode)
    return ;
  xphelpvisible = FALSE;
  XtPopdown(xphelpshell);
}

XtActionsRec            XipActions[] =
{
  {"XipMaximizePkt",	XipActionMaximizePkt},
  {"XipMinimizePkt",	XipActionMinimizePkt},
  {"XipPopupPktHelp",	XipActionPopupPktHelp},
  {"XipPopdownPktHelp",	XipActionPopdownPktHelp},
};

/* is a signal handler.
   Controls the flow and re-arms */
RETSIGTYPE		sigint(unused)
int			unused;
{
  capflow = !capflow;
  if (flowbsb)
    XipSetCheck(flowbsb,capflow);
  fprintf(stderr,"flow %s\n",capflow?"on":"off");
  signal(SIGINT,sigint);
}

/* sets the flow-control procedure.
   
   Warning: It modifies the SIGINT signal handler for the process. */
VOID_FUNC		handle_signals(VOID_DECL)
{
  signal(SIGINT,sigint);
}

t_status		make_hard_filter_from_trailing_args(argc,argv)
int			argc;
char			**argv;
{
  int			i;
  t_status		status;

  hard_filter[0] = 0;
  i = 1;
  while (i < argc)
    {
      if ((status = str_cat_str(hard_filter,
				sizeof (hard_filter),
				argv[i])) < 0)
	return (status);
      if ((status = str_cat_char(hard_filter,
				 sizeof (hard_filter),
				 ' ')) < 0)
	return (status);
      i++;
    }
  return (0);
}

VOID_FUNC		doit(argc,argv)
int			argc;
char			**argv;
{
  t_status		status;
  char			title[STR_BUFSIZ];
  Widget		panner;
    
  if (verbose)
    {
      fprintf(stderr,"\t%s %s %s\n",PRODUCT,VERSION,VENDOR);
#ifdef DEBUG
      fprintf(stderr,"\tcompiled with DEBUG\n");
#endif
#ifdef DEBUG_MALLOC
      fprintf(stderr,"\tcompiled with DEBUG_MALLOC\n");
#endif
    }
  if ((status = check_options()) < 0)
    {
      err_print(-status,"check_options");
      exit(1);
    }
  handle_signals();
  if ((status = layer_init()) < 0)
    {
      err_print(-status,"layer_init");
      exit(1);
    }
  if ((status = nbgethost_init()) < 0)
    {
      err_print(-status,"nbgethost_init");
      exit(1);
    }
  if ((status = vars_init()) < 0)
    {
      err_print(-status,"vars_init");
      exit(1);
    }
  if ((status = xippktvec_init()) < 0)
    {
      err_print(-status,"xippktvec_init");
      exit(1);
    }
  if ((status = xippktcut_init()) < 0)
    {
      err_print(-status,"xippktcut_init");
      exit(1);
    }
  if ((status = methods_init()) < 0)
    {
      err_print(-status,"methods_init");
      exit(1);
    }
  if ((status = pktbar_init()) < 0)
    {
      err_print(-status,"pktbar_init");
      exit(1);
    }
  if (xip_cf)
    {
      if ((status = xip_cf_load(xip_cf)) < 0)
	{
	  err_print(-status,"loading %s",xip_cf);
	  exit(1);
	}
    }
  if ((status = xt_macro_init()) < 0)
    {
      err_print(-status,"xt_macro_init");
      exit(1);
    }
  toplevel = xt_macro_app_initialize(&app_context,
				     "Xip",
				     NULL,
				     0,
				     &argc,
				     argv,
				     fallback,
				     NULL);
  XmgInit();
  XtGetApplicationResources(toplevel,
			    &xt_resources,
			    resources,
			    num_resources,
			    NULL,
			    0);
  xipicon_pixmap = 
    XCreateBitmapFromData(XtDisplay(toplevel),
			  DefaultRootWindow(XtDisplay(toplevel)),
			  (char *)xipicon_bits,
			  xipicon_width,
			  xipicon_height);
  if ((status = XmgPixmapDictAdd("internal:xipicon.xbm",
				 1,
				 xipicon_pixmap)) < 0)
    {
      err_print(-status,"XmgPixmapDictAdd");
      exit(1);
    }
  gray_pixmap = 
    XCreateBitmapFromData(XtDisplay(toplevel),
			  DefaultRootWindow(XtDisplay(toplevel)),
			  gray_bits,
			  gray_width,
			  gray_height);
  if ((status = XmgPixmapDictAdd("internal:gray.xbm",
				 1,
				 gray_pixmap)) < 0)
    {
      err_print(-status,"XmgPixmapDictAdd");
      exit(1);
    }
  check_pixmap = 
    XCreateBitmapFromData(XtDisplay(toplevel),
			  DefaultRootWindow(XtDisplay(toplevel)),
			  check_bits,
			  check_width,
			  check_height);
  if ((status = XmgPixmapDictAdd("internal:check.xbm",
				 1,
				 check_pixmap)) < 0)
    {
      err_print(-status,"XmgPixmapDictAdd");
      exit(1);
    }
  if ((status = get_extra_vars_from_xrm()) < 0)
    {
      err_print(-status,"get_extra_vars_from_xrm");
      exit(1);
    }
#ifdef DEBUG
  if (verbose)
    show_xt_resources();
#endif
  INTERFACE(toplevel,
	    CHILD("paned",panedWidgetClass,
		  CHILD("box",boxWidgetClass,
			XtNshowGrip,	False,
			XtNorientation,	XtorientHorizontal,
			CHILD("file",menuButtonWidgetClass,
			      CHILDSHELL("menu",simpleMenuWidgetClass,
					 XtNtranslations,
					 XtParseTranslationTable(
"<EnterWindow>:     XmgPopdownAllMenusAfterMe() highlight()  \n\
<LeaveWindow>:     unhighlight()           \n\
<BtnMotion>:       highlight()             \n\
<BtnUp>:           XmgPopdownAllMenus() notify() unhighlight()\n"),
					 CHILD("new_pkt",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_new_pkt,
							NULL),
					       END),
					 CHILD("smeline",smeLineObjectClass,
					       END),
					 CHILD("edit_config",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_edit_config,
							NULL),
					       END),
					 CHILD("save_config",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_save_config,
							NULL),
					       END),
					 CHILD("smeline",smeLineObjectClass,
					       END),
				    CHILD("load_collection",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_load_collection,
							NULL),
					       END),
				   CHILD("save_collection",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_save_collection,
							NULL),
					       END),
					 CHILD("smeline",smeLineObjectClass,
					       END),
					 CHILD("clear",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_clear,
							NULL),
					       END),
					 CHILD("smeline",smeLineObjectClass,
					       END),
					 CHILD("quit",smeBSBMObjectClass,
					       XtNmenuName,	"quitmenu",
					       END),
					 END),
			      CHILDSHELL("quitmenu",simpleMenuWidgetClass,
					 XtNtranslations,
					 XtParseTranslationTable(
"<EnterWindow>:     XmgPopdownAllMenusAfterMe() highlight()  \n\
<LeaveWindow>:     unhighlight()           \n\
<BtnMotion>:       highlight()             \n\
<BtnUp>:	   notify() unhighlight()\n"),
					 CHILD("are_you_sure?",
					       smeBSBMObjectClass,
					       XtNmenuName,"quityesnomenu",
					       END),
					 END),
			      CHILDSHELL("quityesnomenu",simpleMenuWidgetClass,
					 XtNtranslations,
					 XtParseTranslationTable(
"<EnterWindow>:     XmgPopdownAllMenusAfterMe() highlight()  \n\
<LeaveWindow>:     unhighlight()           \n\
<BtnMotion>:       highlight()             \n\
<BtnUp>:	   notify() unhighlight()\n"),
					 CHILD("no",
					       smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_nop,
							NULL),
					       END),
					 CHILD("yes",
					       smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_quit,
							NULL),
					       END),
					 END),
			      END),
			CHILD("options",menuButtonWidgetClass,
			      CHILDSHELL("menu",simpleMenuWidgetClass,
				     GETCHILD(flowbsb)
					 ("flowBSB",smeBSBObjectClass,
					  CALLBACK(XtNcallback,
						   bsb_flow,
						   NULL),
					  END),
					 CHILD("line",smeLineObjectClass,
					       END),
				     CHILD("remote_control",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_remote_control,
							NULL),
					   END),
					 CHILD("line",smeLineObjectClass,
					       END),
					 GETCHILD(helpmodebsb)
					 ("helpModeBSB",smeBSBObjectClass,
					  CALLBACK(XtNcallback,
						   bsb_help_mode,
						   NULL),
					   END),
					 GETCHILD(pinmodebsb)
					 ("pinModeBSB",smeBSBObjectClass,
					  CALLBACK(XtNcallback,
						   bsb_pin_mode,
						   NULL),
					   END),
					 CHILD("line",smeLineObjectClass,
					       END),
					 GETCHILD(scrollmodebsb)
					 ("scrollModeBSB",smeBSBObjectClass,
					  CALLBACK(XtNcallback,
						   bsb_scroll_mode,
						   NULL),
					   END),
					 END),
			      END),
			CHILD("help",menuButtonWidgetClass,
			      CHILDSHELL("menu",simpleMenuWidgetClass,
				     CHILD("about",smeBSBObjectClass,
					       CALLBACK(XtNcallback,
							bsb_about,
							NULL),
					       END),
					 END),
			      END),
			END),
		  GETCHILD(porthole)
		  ("porthole",portholeWidgetClass,
		   GETCHILD(pktbox)
		   ("pktBox",boxWidgetClass,
		    END),
		   END),
		  END),
	    REALIZE,
	    ENDINTERFACE);
  INTERFACE(toplevel,
	    GETCHILDSHELL(pannershell)
	    ("xipPannerShell",transientShellWidgetClass,
	     XtNmappedWhenManaged,	False,
	     XtNallowShellResize,	True,
	     XtNtransientFor,		toplevel,
	     XtNiconPixmap,		xipicon_pixmap,
	     GETCHILD(panner)
	     ("panner",pannerWidgetClass,
	      END),
	     MANAGE,
	     END),
	    ENDINTERFACE);
  XmgSetPopdownOnDelete(pannershell);
  XtAddCallback(porthole,
		XtNreportCallback,
		(XtCallbackProc)report_porthole,
		panner);
  XtAddCallback(panner,
		XtNreportCallback,
		(XtCallbackProc)report_panner,
		porthole);
  INTERFACE(toplevel,
	    GETCHILDSHELL(xphelpshell)
	    ("xipPktHelpShell",overrideShellWidgetClass,
	     XtNmappedWhenManaged,	False,
	     XtNallowShellResize,	True,
	     XtNtransientFor,		toplevel,
	     GETCHILD(xphelpfancy)
	     ("xipPktHelpFancy",xmgFancyWidgetClass,
	      END),
	     MANAGE,
	     END),
	    ENDINTERFACE);
  XtAppAddActions(app_context,XipActions,XtNumber(XipActions));
  if ((status = make_hard_filter_from_trailing_args(argc,argv)) < 0)
    {
      err_print(-status,"make_hard_filter_from_trailing_args");
      exit(1);
    }
  title[0] = 0;
  if ((status = str_cat_fmt_va(title,
			       sizeof (title),
			       "xip: \"%s\"",
			       hard_filter)) < 0)
    if (status != -ERR_BO)
      {
	err_print(-status,"str_cat_fmt_va");
	exit(1);
      }
  XipSetCheck(flowbsb,capflow);
  XipSetCheck(pinmodebsb,xt_resources.pinMode);
  XipSetCheck(helpmodebsb,xt_resources.helpMode);
  XipSetCheck(scrollmodebsb,xt_resources.scrollMode);
  XtVaSetValues(toplevel,
		XtNtitle,	title,
		XtNiconPixmap,	xipicon_pixmap,
		NULL);
  xipcap_init();
  if ((status = shortcuts_init()) < 0)
    {
      err_print(-status,"shortcuts_init");
      exit(1);
    }
  XtAppMainLoop(app_context);
}

int			main(argc,argv)
int			argc;
char			**argv;
{
  t_opt_context		oc;
  t_boolean		trargsbuf[256];
  t_boolean		troptsbuf[256];
  t_status              status;

  assert(opt_check(main_opts,ARRAY_COUNT(main_opts)) == 0);
  oc.tr_args = trargsbuf;
  oc.nb_tr_args = sizeof (trargsbuf);
  oc.tr_opts = troptsbuf;
  oc.nb_tr_opts = sizeof (troptsbuf);
  oc.argc = &argc;
  oc.argv = argv;
  oc.opts = main_opts;
  oc.nb_opts = ARRAY_COUNT(main_opts);
  oc.base_addr = NULL;
  if ((status = opt_get(&oc)) < 0)
    {
      err_print(-status,"couldn't get options");
      usage();
    }
  if (help)
    usage();
  gdm_init();
#ifdef DEBUG_MALLOC
  if (debug_malloc)
    {
      global_debug_malloc = TRUE;
      gdm.verbose = debug_malloc_verbose;
    }
#endif
#ifdef DEBUG
  a_debug_init();
  layer_debug_init();
  xmg_debug_init();
#endif
  layer_resolve = (t_32)resolve;
  doit(argc,argv);
}
