/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <fcntl.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Viewport.h>
#include "xt_macro.h"
#include "XmgFancy.h"
#include "lay_data.h"
#include "lay_ip.h"
#include "lay_udp.h"
#include "lay_pad.h"
#include "typ_lay.h"
#include "typ_ethaddr.h"
#include "xip.h"
#include "xipcf.h"
#include "layer.h"
#include "nbgethost.h"
#ifdef HAVE_DLOPEN
#include <dlfcn.h>
#endif

extern Widget		toplevel;
extern t_boolean	verbose;
extern Pixmap		xipicon_pixmap;

t_u32			tmpl_bufsiz = 2 * BUFSIZ;
t_u32			html_bufsiz = 8 * BUFSIZ;
t_u32			htmlize_bufsiz = 8 * BUFSIZ;
struct in_addr		ghost_ip = {-2};
t_ether_addr		ghost_ether = {0x42,0x42,0x42,0x42,0x42,0x43};
t_u16			ip_id = 1;

t_assoc			boolean_assocs[] = 
{
  {"true",		(VOID_PTR)TRUE},
  {"false",		(VOID_PTR)FALSE},
  {NULL,		NULL},
};

char			*tmpl_buf = NULL;
char			*html_buf = NULL;
char			*htmlize_buf = NULL;

/* is a t_msg_proc.
   manages memory zones. Ptr is passed as ed->data and size is passed as
   an t_u32 in ed->b.buf and ed->b.len. */
t_status		typ_memoryzone_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("memoryzone");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	
	return (typ_msg(typ_u32_msg,
			msg,
			arg1,
			arg2));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	char		**zone;
	t_u32		siz;	

	if ((status = typ_msg(typ_u32_msg,
			      msg,
			      arg1,
			      arg2)) < 0)
	  return (status);
	assert(ed->b.len == sizeof (siz));
	siz = *((t_u32 *)(ed->b.buf));
	zone = (char **)(ed->data);
	if (*zone)
	  {
	    if (((*zone) = XIP_REALLOC_PROC(*zone,
					    siz,
					    "xipcf",
					    "typ_memoryzone_msg:ptr",
					    &status)) == NULL)
	      {
		err_print(-status,"XIP_REALLOC_PROC");
		exit(1);
	      }
	  }
	else
	  {
	    if (((*zone) = XIP_ALLOC_PROC(siz,
					  "xipcf",
					  "typ_memoryzone_msg:ptr",
					  &status)) == NULL)
	      {
		err_print(-status,"XIP_ALLOC_PROC");
		exit(1);
	      }
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}

#ifdef HAVE_DLOPEN
/* is a t_msg_proc.
   Manages plugins. It detects new plugins inserted. */
t_status		typ_plugins_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;
  static t_vec		*plugins = NULL;

  if (!plugins)
    {
      if ((plugins = XIP_VEC_NEW(VEC_BASE,
				 &status)) == NULL)
	return (status);
    }
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("plugins");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);

	VEC_FOR(plugins,char *path)
	  {
	    if ((status = str_cat_str(bs->str,
				      bs->max_len,
				      path)) < 0)
	      return (status);
	    if (VEC_IDX != (VEC_COUNT(plugins) - 1))
	      if ((status = str_cat_char(bs->str,
					 bs->max_len,
					 ',')) < 0)
		return (status);
	  }
	VEC_ENDFOR;
	return (0);
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_vec		*vec;

	if ((vec = XIP_VEC_NEW(VEC_BASE,
			       &status)) == NULL)
	  return (status);
	if ((status = vec_str_split(vec,str,',')) < 0)
	  {
	    vec_str_delete(vec);
	    return (status);
	  }
	VEC_FOR(vec,char *path)
	  {
	    VOID_PTR	plugin;
	    t_status	(*xip_plugin_init)();
	    
	    if ((status = vec_str_index(plugins,path)) >= 0)
	      {
		continue ;
	      }
	    if ((plugin = dlopen(path,RTLD_NOW)) == NULL)
	      {
		err_print(ERR_SUBROUTINE,"%s: %s",dlerror());
	      }
	    else
	      {
		if ((xip_plugin_init =
		     dlsym(plugin,"xip_plugin_init")) == NULL)
		  {
		    err_print(ERR_SUBROUTINE,"%s: %s",dlerror());
		    dlclose(plugin);
		  }
		else
		  {
		    if ((status = xip_plugin_init()) < 0)
		      return (status);
		    if ((status = vec_str_add(plugins,path)) < 0)
		      return (status);
		  }
	      }
	  }
	VEC_ENDFOR;
	vec_str_delete(vec);
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}
#endif

t_var			vars[] = 
{
#define NOT_A_REAL_OFFSET	((t_off)(NULL))
#define NOT_A_REAL_SIZE		0

  {"tmpl_bufsiz",
   (t_off)(&tmpl_bufsiz),
   sizeof (tmpl_bufsiz),
   typ_memoryzone_msg,
   &tmpl_buf},
  {"html_bufsiz",
   (t_off)(&html_bufsiz),
   sizeof (html_bufsiz),
   typ_memoryzone_msg,
   &html_buf},
  {"htmlize_bufsiz",
   (t_off)(&htmlize_bufsiz),
   sizeof (htmlize_bufsiz),
   typ_memoryzone_msg,	
   &htmlize_buf},
  {"lay_data_bytes_per_line",
   (t_off)(&lay_data_bytes_per_line),
   sizeof (lay_data_bytes_per_line),
   typ_u32_msg,
   NULL},
  {"lay_udp_perform_sum",
   (t_off)(&lay_udp_perform_sum),
   sizeof (lay_udp_perform_sum),
   typ_u32assoc_msg,
   boolean_assocs},
  {"lay_udp_guess_mode",
   (t_off)(&lay_udp_guess_mode),
   sizeof (lay_udp_guess_mode),
   typ_u32assoc_msg,
   boolean_assocs},
  {"layer_verb_level",
   (t_off)(&layer_verb_level),
   sizeof (layer_verb_level),
   typ_u32mask_msg,
   layer_verb_level_mask_defs},
  {"layer_base",(t_off)(&layer_base),
   sizeof (layer_base),
   typ_u32_msg,
   NULL},
  {"ghost_ip",
   (t_off)(&ghost_ip),
   sizeof (ghost_ip),
   typ_inaddr_resolved_msg,
   NULL},
  {"ghost_ether",
   (t_off)(&ghost_ether),
   sizeof (ghost_ether),
   typ_etheraddr_msg,
   NULL},
  {"ip_id",
   (t_off)(&ip_id),
   sizeof (ip_id),
   typ_u16_msg,
   NULL},
  {"nbgethost_timeout",
   (t_off)(&nbgethost_timeout),
   sizeof (nbgethost_timeout),
   typ_u32_msg,
   NULL},
  {"lay_pad_nbytes",
   (t_off)(&lay_pad_nbytes),
   sizeof (lay_pad_nbytes),
   typ_u32_msg,
   NULL},
  {"lay_pad_sub_lay",
   (t_off)(&lay_pad_sub_lay),
   sizeof (lay_pad_sub_lay),
   typ_lay_msg,
   NULL},
  {"etheraddr_dec_way",
   (t_off)(&etheraddr_dec_way),
   sizeof (etheraddr_dec_way),
   typ_u32assoc_msg,
   boolean_assocs},
#ifdef HAVE_DLOPEN
  {PLUGINS_STR,
   NOT_A_REAL_OFFSET,
   NOT_A_REAL_SIZE,
   typ_plugins_msg,
   NULL},
#endif
  {NULL,	
   0,
   0,
   NULL},
};

t_xt_resources		xt_resources;

XtResource		resources[] = 
{
#define RESSTRING(Name,Class,Field,Def) \
  {Name,		Class,		XtRString,\
   sizeof (xt_resources.Field),\
   XtOffset(t_xt_resources *,Field),	XtRString,	(Def)}

#define RESINT(Name,Class,Field,Def) \
  {Name,		Class,		XtRInt,\
   sizeof (xt_resources.Field),\
   XtOffset(t_xt_resources *,Field),	XtRImmediate,	(XtPointer)(Def)}

#define RESGC(Name,Class,Field,Def) \
  {Name,		Class,		XtRGC,\
   sizeof (xt_resources.Field),\
   XtOffset(t_xt_resources *,Field),	XtRString,	(Def)}

#define RESBOOLEAN(Name,Class,Field,Def) \
  {Name,		Class,		XtRBoolean,\
   sizeof (xt_resources.Field),\
   XtOffset(t_xt_resources *,Field),	XtRImmediate,	(XtPointer)(Def)}

  RESSTRING("binEditorPath","BinEditorPath",binEditorPath,"emacs"),
  RESSTRING("defaultColorName","DefaultColorName",
	    defaultColorName,"white"),
  RESSTRING("bodyColorName","BodyColorName",bodyColorName,"white"),
  RESSTRING("bodyBackgroundName","BodyBackgroundName",
	    bodyBackgroundName,"internal:gray.xbm"),
  RESSTRING("controlsColorName","ControlsColorName",
	    controlsColorName,"white"),
  RESINT("pktHelpOffset","PktHelpOffset",pktHelpOffset,8),
  RESBOOLEAN("pinMode","PinMode",pinMode,FALSE),
  RESBOOLEAN("helpMode","HelpMode",helpMode,TRUE),
  RESBOOLEAN("scrollMode","ScrollMode",scrollMode,TRUE),
  RESBOOLEAN("smoothScrollMode","SmoothScrollMode",smoothScrollMode,TRUE),
  RESINT("smoothScrollLimit","SmoothScrollLimit",smoothScrollLimit,8),
  RESINT("smoothScrollOffset","SmoothScrollOffset",smoothScrollOffset,1)
};

Cardinal		num_resources = XtNumber(resources);

#ifdef DEBUG
VOID_FUNC		show_xt_resources(VOID_DECL)
{
  fprintf(stderr,"resources:\n");
  fprintf(stderr,"\tbinEditorPath=%s\n",xt_resources.binEditorPath);
  fprintf(stderr,"\tdefaultColorName=%s\n",xt_resources.defaultColorName);
  fprintf(stderr,"\tbodyColorName=%s\n",xt_resources.bodyColorName);
  fprintf(stderr,"\tbodyBackgroundName=%s\n",xt_resources.bodyBackgroundName);
  fprintf(stderr,"\tcontrolsColorName=%s\n",xt_resources.controlsColorName);
  fprintf(stderr,"\tpktHelpOffset=%d\n",xt_resources.pktHelpOffset);
  fprintf(stderr,"\tpinMode=%s\n",xt_resources.pinMode?"True":"False");
  fprintf(stderr,"\thelpMode=%s\n",xt_resources.helpMode?"True":"False");
  fprintf(stderr,"\tscrollMode=%s\n",xt_resources.scrollMode?"True":"False");
  fprintf(stderr,"\tsmoothScrollMode=%s\n",
	  xt_resources.smoothScrollMode?"True":"False");
  fprintf(stderr,"\tsmoothScrollLimit=%d\n",
	  xt_resources.smoothScrollLimit);
  fprintf(stderr,"\tsmoothScrollOffset=%d\n",
	  xt_resources.smoothScrollOffset);
}
#endif

t_dict			*extravars = NULL;
char			*progname;
char			*progclass;

/* is a t_dict_walk_proc.
   Walks the "lay" dictionary to lookup associated Xrm resources. */
t_status		get_extra_vars_from_xrm_walk(he,db)
t_hash_elt		*he;
XrmDatabase		db;
{
  char			str_name[STR_BUFSIZ];
  char			str_class[STR_BUFSIZ];
  char			*str_type;
  XrmValue		xrmvalue;
  t_status		status;
  char			key[STR_BUFSIZ];

  str_name[0] = 0;
  if ((status = str_cat_fmt_va(str_name,
			       sizeof (str_name),
			       "%s.%sColor",
			       progname,
			       he->key)) < 0)
    return (status);
  str_class[0] = 0;
  if ((status = str_cat_fmt_va(str_class,
			       sizeof (str_class),
			       "%s.%sColor",
			       progclass,
			       he->key)) < 0)
    return (status);
  key[0] = 0;
  if ((status = str_cat_fmt_va(key,
			       sizeof (key),
			       "%sColor",
			       he->key)) < 0)
    return (status);
  if (!XrmGetResource(db,
		      str_name,
		      str_class,
		      &str_type,
		      &xrmvalue))
    {
      if (verbose)
	err_print(ERR_MISSINGVAR,"%s",str_name);
      return (dict_str_override(extravars,
				key,
				xt_resources.defaultColorName));
    }
  if (strcmp(str_type,XtRString))
    {
      err_print(ERR_CANTHANDLE,"Unknown Xrm type %s",str_type);
      return (0);
    }
  return (dict_str_override_with_len(extravars,
				     key,
				     xrmvalue.addr,
				     xrmvalue.size));
}

/* is a t_dict_walk_proc.
   Shows extra vars */
t_status		extra_vars_walk(he,unused)
t_hash_elt		*he;
VOID_PTR		unused;
{
  fprintf(stderr,"\t%s=%s\n",he->key,he->value);
  return (0);
}

/* Extracts variables from Xrm which will be used as "extravars" for
   packet templates.
   Returns 0 if OK, might return and print various errors. */
t_status		get_extra_vars_from_xrm(VOID_DECL)
{
  XrmDatabase		db;
  t_status		status;

  if ((extravars = XIP_DICT_NEW(HASH_SMALL_BASE,
				VEC_BASE,
				&status)) == NULL)
    return (status);
  db = XtDatabase(XtDisplay(toplevel));
  XtGetApplicationNameAndClass(XtDisplay(toplevel),
			       &progname,
			       &progclass);
  if ((status = dict_walk(lay_dict,
			  (t_dict_walk_proc)get_extra_vars_from_xrm_walk,
			  db)) < 0)
    return (status);
  if ((status = dict_str_add(extravars,
			     "bodyColorName",
			     xt_resources.bodyColorName)) < 0)
    {
      err_print(-status,"dict_str_add bodyColorName");
      return (0);
    }
  if ((status = dict_str_add(extravars,
			     "bodyBackgroundName",
			     xt_resources.bodyBackgroundName)) < 0)
    {
      err_print(-status,"dict_str_add bodyBackgroundName");
      return (0);
    }
  if ((status = dict_str_add(extravars,
			     "controlsColorName",
			     xt_resources.controlsColorName)) < 0)
    {
      err_print(-status,"dict_str_add controlsColorName");
      return (0);
    }
  if (verbose)
    {
      fprintf(stderr,"extravars:\n");
      dict_walk_sorted(extravars,
		       (t_dict_walk_proc)extra_vars_walk,
		       NULL);
    }
  return (0);
}

/* deletes "extravars" */
VOID_FUNC		destroy_extra_vars(VOID_DECL)
{
  dict_str_delete(extravars);
}

/* gets t_var structure associated with name.
   Returns NULL of not found. */
t_var			*var_from_vars(name)
char			*name;
{
  t_var			*var;

  var = vars;
  while (var->name)
    {
      if (!strcmp(name,var->name))
	return (var);
      var++;
    }
  return (NULL);
}

/* sets a configuration variable.
   Note: it prints an error message and returns 0 if var doesn't exist.
   Returns 0 if OK. Might return various errors. */
t_status		var_set(name,value)
char			*name;
char			*value;
{
  t_status		status;
  t_extract_data	ed;
  t_var			*var;

  if (!(var = var_from_vars(name)))
    {
      err_print(ERR_NOSUCHVAR,"no such var %s",name);
      return (0);
    }
  ed.b.buf = (char *)(var->offset);
  ed.b.len = var->size;
  ed.data = var->data;
  return (typ_msg(var->mp,
		  TYP_INSERT,
		  &ed,
		  value));
}

/* gets a variable value to a string.
   Returns 0 if OK. -ERR_NOSUCHVAR if var doesn't exist. Might
   return various errors. */
t_status		var_get(name,str,max_len)
char			*name;
char			*str;
int			max_len;
{
  t_status		status;
  t_extract_data	ed;
  t_var			*var;
  t_bridled_str		bs;

  if (!(var = var_from_vars(name)))
    return (-ERR_NOSUCHVAR);
  ed.b.buf = (char *)(var->offset);
  ed.b.len = var->size;
  ed.data = var->data;
  bs.str = str;
  bs.max_len = max_len;
  return (typ_msg(var->mp,
		  TYP_EXTRACT,
		  &ed,
		  &bs));
}

/* destroys vars context.
   Currently, it destroys html_buf, tmpl_buf and htmlize_buf. */
VOID_FUNC		vars_destroy(VOID_DECL)
{
  XIP_FREE_PROC(html_buf,
		"xipcf",
		"*:ptr");
  XIP_FREE_PROC(tmpl_buf,
		"xipcf",
		"*:ptr");
  XIP_FREE_PROC(htmlize_buf,
		"xipcf",
		"*:ptr");
}

/* inits vars context.
   Currently, it allocates html_buf, tmpl_buf and htmlize_buf.
   Returns 0 if OK. Might return various errors */
t_status		vars_init(VOID_DECL)
{
  t_status		status;
  
  if ((html_buf = XIP_ALLOC_PROC(html_bufsiz,
				 "xipcf",
				 "vars_init:ptr",
				 &status)) == NULL)
    return (status);
  if ((tmpl_buf = XIP_ALLOC_PROC(tmpl_bufsiz,
				 "xipcf",
				 "vars_init:ptr",
				 &status)) == NULL)
    return (status);
  if ((htmlize_buf = XIP_ALLOC_PROC(htmlize_bufsiz,
				    "xipcf",
				    "vars_init:ptr",
				    &status)) == NULL)
    return (status);
  return (0);
}

/* saves configuration to a file.
   Returns 0 if OK. -ERR_OPEN if fopen(3) fails. Might return various
   errors as it calls typ methods. */
t_status		xip_cf_save(fname)
char			*fname;
{
  FILE			*f;
  t_status		status;
  t_var			*var;

  if ((f = fopen(fname,"w+")) == NULL)
    return (-ERR_OPEN);
  fprintf(f,"#XIPCF\n");
  var = vars;
  while (var->name)
    {
      char		value[STR_BUFSIZ];
      t_extract_data	ed;
      t_bridled_str	bs;
      
      value[0] = 0;
      ed.b.buf = (char *)(var->offset);
      ed.b.len = var->size;
      ed.data = var->data;
      bs.str = value;
      bs.max_len = sizeof (value);
      if ((status = typ_msg(var->mp,
			    TYP_EXTRACT,
			    &ed,
			    &bs)) < 0)
	{
	  fclose(f);
	  return (status);
	}
      fprintf(f,"%s = %s\n",var->name,value);
      var++;
    }
  fclose(f);
  return (status);
}

/* is a t_vec_cmp_proc.
   Sorts vars with "plugins" first.
   Note: this is used to allow plugins to get configuration values by
   adding t_var to a vector of var (this feature is not done yet). */
int			xip_cf_load_cmp(p1,p2)
VOID_PTR		*p1;
VOID_PTR		*p2;
{
  t_hash_elt		*he1;
  t_hash_elt		*he2;
  
  he1 = (t_hash_elt *)(*p1);
  he2 = (t_hash_elt *)(*p2);
  if (!strcmp(he1->key,PLUGINS_STR))
    {
      if (!strcmp(he2->key,PLUGINS_STR))
	return (0);
      else
	return (-1);
    }
  else
    {
      if (!strcmp(he2->key,PLUGINS_STR))
	return (1);
      else
	return (0);
    }
}

/* is a t_dict_walk_proc.
   Assign value to key in using var_set(3).
   If global verbose is TRUE, then it prints a shortcut of the operation.
   Anyway this function will print an error message if there is a problem
   with var_set(3).
   Returns 0 if OK. Might return various errors. */
t_status		xip_cf_load_walk(he,unused)
t_hash_elt		*he;
VOID_PTR		unused;
{
  t_status		status;
   
  if (verbose)
    fprintf(stderr,"\t%s=%s\n",he->key,he->value);
  if ((status = var_set(he->key,he->value)) < 0)
    err_print(-status,"var_set %s",he->key);
  return (status);
}

/* loads a configuration file.
   Note: it might print various error and verbose messages as it calls
   xip_cf_load_walk(3).
   Returns 0 if OK. -ERR_OPEN if open(2) fails. Might return various errors. */
t_status		xip_cf_load(fname)
char			*fname;
{
  int			fd;
  t_status		status;
  t_dict		*vars;
  t_hash_elt		*he;

  if ((fd = open(fname,O_RDONLY)) < 0)
    return (-ERR_OPEN);
  vars = NULL;
  if ((vars = XIP_DICT_NEW(HASH_SMALL_BASE,
			   VEC_BASE,
			   &status)) == NULL)
    goto end;
  if ((status = parse_vars(fd,
			   '=',
			   vars,
			   dict_str_override)) < 0)
    goto end;
  if (verbose)
    fprintf(stderr,"vars:\n");
  if ((status = dict_walk_sorted_cmp(vars,
				     (t_dict_walk_proc)xip_cf_load_walk,
				     (t_vec_cmp_proc)xip_cf_load_cmp,
				     NULL)) < 0)
    goto end;
  status = 0;
end:
  close(fd);
  if (vars)
    dict_str_delete(vars);
  return (status);
}

/* is a t_dict_walk_proc.
   It sets value to key in using var_set(3).
   It prints an error message if there is a problem with var_set(3).
   Returns 0 if OK. Might return various errors. */
t_status			XipCfFancyFormWalk(he,unused)
t_hash_elt			*he;
VOID_PTR			unused;
{
  t_status			status;

  if ((status = var_set(he->key,he->value)) < 0)
    err_print(-status,"var_set: %s",he->key);
  return (status);
}

/* is an XtCallbackProc.
   It is the "form" action of the configuration panel.
   It calls dict_walk(3) for all the variables with XipCfFancyFormWalk(3)
   as t_dict_walk_proc. 
   It shows a popup if there is an error. */
VOID_FUNC			XipCfFancyForm(w,shell,cbs)
Widget				w;
Widget				shell;
XmgFancyFormCallbackStruct	*cbs;
{
  t_status			status;
  
  if ((status = dict_walk(cbs->vars,
			  (t_dict_walk_proc)XipCfFancyFormWalk,
			  NULL)) < 0)
    {
      XmgErrPrint(-status,"dict_walk");
      goto end;
    }
end:
  XtDestroyWidget(shell);
}

/* is an XtCallbackProc.
   Destroys configuration panel if a link is activated (currently "cancel") */
VOID_FUNC			XipCfFancyLink(w,shell,cbs)
Widget				w;
Widget				shell;
XmgFancyLinkCallbackStruct	*cbs;
{
  XtDestroyWidget(shell);
}

/* is an XtCallbackProc.
   Sets shell title. */
VOID_FUNC			XipCfFancyTitle(w,shell,cbs)
Widget				w;
Widget				shell;
XmgFancyTitleCallbackStruct	*cbs;
{
  XtVaSetValues(shell,
		XtNtitle,	cbs->title,
		NULL);
}

char			*xip_cf_html_top = "\
<html>\n\
<head>\n\
<title>Xip: Configuration</title>\n\
</head>\n\
<body>\n\
<h1>Configuration</h1>\n\
<hr>\n\
<form>\n\
<table width=100%%>\n\
<th>\n\
<td width=33% align=center>\n\
<i>Name</i>\n\
</td>\n\
<td align=center>\n\
<i>value</i>\n\
</td>\n\
</th>\n\
";

char			*xip_cf_html_bottom = "\
</table>\n\
<br>\n\
<hr>\n\
<table width=100%%>\n\
<tr>\n\
<td width=50%% align=center>\n\
<input type=reset value=\"Reset values\">\n\
</td>\n\
<td width=25%% align=center>\n\
<input type=submit value=\"Ok\">\n\
</td>\n\
<td width=25%% align=center>\n\
<a href=\"cancel\">Cancel</a>\n\
</td>\n\
</tr>\n\
</table>\n\
</form>\n\
</body>\n\
</html>\n\
"; 

/* creates a configuration panel from variables */
VOID_FUNC		XipShowCfBox(parent)
Widget			parent;
{
  Widget		shell;
  Widget		fancywidget;
  char			html[BUFSIZ];
  t_status		status;
  t_var			*var;
  Widget		viewport;
  Dimension		width;
  Dimension		height;

  html[0] = 0;
  if ((status = str_cat_str(html,
			    sizeof (html),
			    xip_cf_html_top)) < 0)
    {
      XmgErrPrint(-status,"str_cat_str");
      return ;
    }
  var = vars;
  while (var->name)
    {
      char		value[STR_BUFSIZ];

      value[0] = 0;
      if ((status = var_get(var->name,
			    value,
			    sizeof (value))) < 0)
	{
	  XmgErrPrint(-status,"var_get %s",var->name);
	  return ;
	}
      if ((status = str_cat_fmt_va(html,
				   sizeof (html),
				   "<tr>\n\
<td colspan=1 align=center>\n\
<b>%s: </b>\n\
</td>\n\
<td align=center>\n\
<input width=25 name=\"%s\" value=\"%s\">\n\
</td>\n\
</tr>\n",
				   var->name,
				   var->name,
				   value)) < 0)
	{
	  XmgErrPrint(-status,"str_cat_fmt_va");
	  return ;
	}
      var++;
    }
  if ((status = str_cat_str(html,
			    sizeof (html),
			    xip_cf_html_bottom)) < 0)
    {
      XmgErrPrint(-status,"str_cat_str");
      return ;
    }
  INTERFACE(parent,
            GETCHILDSHELL(shell)
            ("xipCfBox",transientShellWidgetClass,
             XtNmappedWhenManaged,	False,
	     XtNtransientFor,		parent,
	     XtNallowShellResize,	True,
	     XtNiconPixmap,		xipicon_pixmap,
	     GETCHILD(viewport)
	     ("viewport",viewportWidgetClass,
	      XtNallowVert,		True,
	      XtNallowHoriz,		True,
	      GETCHILD(fancywidget)
	      ("fancy",xmgFancyWidgetClass,
	       CALLBACK(XtNformCallback,
			(XtCallbackProc)XipCfFancyForm,
			shell),
	       CALLBACK(XtNlinkCallback,
			(XtCallbackProc)XipCfFancyLink,
			shell),
	       CALLBACK(XtNtitleCallback,
			(XtCallbackProc)XipCfFancyTitle,
			shell),
	       END),
	      END),
	     MANAGE,
	     END),
	    ENDINTERFACE);
  XmgFancyParseBuf(fancywidget,
		   html,
		   strlen(html));
  XtVaGetValues(fancywidget,
		XtNwidth,	&width,
		XtNheight,	&height,
		NULL);
  XtVaSetValues(viewport,
		XtNwidth,	MIN(width,
				    DisplayWidth(XtDisplay(toplevel),
					 DefaultScreen(XtDisplay(toplevel)))),
		XtNheight,	MIN(height,
				    DisplayHeight(XtDisplay(toplevel),
					 DefaultScreen(XtDisplay(toplevel)))),
		NULL);
  XmgCenterWidget(shell);
  XmgSetDestroyOnDelete(shell);
  XtPopup(shell,XtGrabExclusive);
}
