/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/wait.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xmu/Converters.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Tree.h>
#include <X11/Xaw/Viewport.h>
#include "XmgFancy.h"
#include "XmgFileGet.h"
#include "xt_macro.h"
#include "pkt.h"
#include "xippkt.h"
#include "typ_lay.h"
#include "lay_data.h"
#include "lay_ether.h"
#include "lay_ip.h"
#include "lay_arp.h"
#include "pkt_sav.h"
#include "xippktvec.h"
#include "xippktcut.h"
#include "xipcf.h"
#include "xipimg.h"

extern XtAppContext		app_context;	
extern Widget			toplevel;
extern Widget			pktbox;
extern t_boolean		stop;
extern Pixmap			xipicon_pixmap;

/* deletes a xip packet context (t_xip_pkt).
   Note: this operation removes packet (t_pkt) from main packet vector.
   It also destroy the widgets.
   Returns 0 if OK. -ERR_IN_USE if packet is in use (e.g a setfield prompt
   is still popped up). */
t_status			xp_delete(xp)
t_xip_pkt			*xp;
{
  if (xp->in_delete)
    return (0);
  xp->in_delete = TRUE;
  if (VEC_COUNT(xp->users) == 0)
    {
      if (xp->own_pkt)
	xip_pkt_delete(xp->pkt);
      vec_delete(xp->users);
      if (xp->shell)
	XtDestroyWidget(xp->shell);
      XIP_FREE_PROC(xp,
		    "xippkt",
		    "xp");
      return (0);
    }
  else
    return (-ERR_INUSE);
}

/* is a t_xip_method_proc.
   Usage: "delete()".
   It deletes packet in using xp_delete(3). 
   Returns 0 if OK. Might return various errors. */
t_status			method_delete(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  return (xp_delete(xmd->xp));
}

/* is an XmgMCallbackProc.
   It retrieves textfield widget value and sets it to packet. 
   It might recreate a packet if needed. 
   It also removes itself from packet users. */
VOID_FUNC			setfield_prompt_ok(w,vec,cbs)
Widget				w;
t_vec				*vec;
XtPointer			cbs;
{
  Widget			shell;			
  Widget			text;
  t_xip_method_data		*own_xmd;
  char				*value;
  t_status			status;
  t_pkt				*pkt;

  assert(VEC_COUNT(vec) == 3);
  shell = (Widget)VEC_AT(vec,0);
  text = (Widget)VEC_AT(vec,1);
  own_xmd = (t_xip_method_data *)VEC_AT(vec,2);
  XtVaGetValues(text,
		XtNstring,	&value,
		NULL);
  if (own_xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(own_xmd->xp->pkt,
			     &status)) == NULL)
	{
	  
	  XmgErrPrint(-status,"xip_pkt_dup");
	  goto end;
	}
    }
  else
    pkt = own_xmd->xp->pkt;
  if ((status = pkt_set_field_from_str(pkt,
				       own_xmd->data,
				       value)) < 0)
    {
      XmgErrPrint(-status,"pkt_set_field_from_str");
      if (own_xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      goto end;
    }
  if (own_xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(own_xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  XmgErrPrint(-status,"xip_pkt_html");
	  goto end;
	}
      XmgFancyParseBuf(own_xmd->xp->fancy,html_buf,strlen(html_buf));
    }
end:
  XtDestroyWidget(shell);
  VEC_RM_ELT(own_xmd->xp->users,own_xmd);
  XIP_FREE_PROC(own_xmd->data,
		"xippkt",
		"own_xmd->data");
  XIP_FREE_PROC(own_xmd,
		"xippkt",
		"own_xmd");
}

/* is an XmgMCallbackProc.
   It retrieves pulldown widget value and sets it to packet. 
   It might recreate a packet if needed. 
   It also removes itself from packet users. */
VOID_FUNC			setfield_prompt_pulldown_ok(w,vec,cbs)
Widget				w;
t_vec				*vec;
XtPointer			cbs;
{
  Widget			shell;			
  Widget			pulldown;
  t_xip_method_data		*own_xmd;
  char				*value;
  t_status			status;
  t_pkt				*pkt;

  assert(VEC_COUNT(vec) == 3);
  shell = (Widget)VEC_AT(vec,0);
  pulldown = (Widget)VEC_AT(vec,1);
  own_xmd = (t_xip_method_data *)VEC_AT(vec,2);
  XtVaGetValues(pulldown,
		XtNlabel,	&value,
		NULL);
  if (own_xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(own_xmd->xp->pkt,
			     &status)) == NULL)
	{
	  XmgErrPrint(-status,"xip_pkt_dup");
	  goto end;
	}
    }
  else
    pkt = own_xmd->xp->pkt;
  if ((status = pkt_set_field_from_str(pkt,
				       own_xmd->data,
				       value)) < 0)
    {
      XmgErrPrint(-status,"pkt_set_field_from_str");
      if (own_xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      goto end;
    }
  if (own_xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(own_xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  XmgErrPrint(-status,"xip_pkt_html");
	  goto end;
	}
      XmgFancyParseBuf(own_xmd->xp->fancy,html_buf,strlen(html_buf));
    }
end:
  XtDestroyWidget(shell);
  VEC_RM_ELT(own_xmd->xp->users,own_xmd);
  XIP_FREE_PROC(own_xmd->data,
		"xippkt",
		"own_xmd->data");
  XIP_FREE_PROC(own_xmd,
		"xippkt",
		"own_xmd");
}

/* is an XmgMCallbackProc.
   It destroys prompt and removes itself from packet users. */
VOID_FUNC			setfield_prompt_cancel(w,vec,cbs)
Widget				w;
t_vec				*vec;
XtPointer			cbs;
{
  Widget			shell;
  t_xip_method_data		*own_xmd;
  
  assert(VEC_COUNT(vec) == 2);
  shell = (Widget)VEC_AT(vec,0);
  own_xmd = (t_xip_method_data *)VEC_AT(vec,1);
  XtDestroyWidget(shell);
  VEC_RM_ELT(own_xmd->xp->users,own_xmd);
  XIP_FREE_PROC(own_xmd->data,
		"xippkt",
		"own_xmd->data");
  XIP_FREE_PROC(own_xmd,
		"xippkt",
		"own_xmd");
}

/* is a t_xip_method_proc.
   Usage: "setfield(layer[idx].field)" e.g "setfield(ip[0].Src)".
   It needs one argument used as field name.
   It tries first to retrieve field value, then it tries to see if
   they are limited possible choices by using pkt_get_choices(3).
   If it is the case it calls XmgShowPromptComplexPulldownBox(3) instead of 
   XmgShowPromptBox(3). 
   Returns 0 if OK. Might return various errors. */
t_status			method_setfield(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  char				value[STR_BUFSIZ];
  t_status			status;
  t_xip_method_data		*own_xmd;
  t_vec				*choices;

  if (VEC_COUNT(args) != 2)
    return (-ERR_SYNTAX);
  value[0] = 0;
  if ((status = pkt_get_field_to_str(xmd->xp->pkt,
				     VEC_AT(args,1),
				     value,
				     sizeof (value))) < 0)
    {
      return (status);
    }
  if ((own_xmd = XIP_ALLOC_PROC(sizeof (t_xip_method_data),
				"xippkt",
				"own_xmd",
				&status)) == NULL)
    return (status);
  bcopy((char *)xmd,(char *)own_xmd,sizeof (t_xip_method_data));
  if ((own_xmd->data = strdup_alloc(VEC_AT(args,1),
				    XIP_ALLOC_PROC,
				    "xippkt",
				    "own_xmd->data",
				    &status)) == NULL)
    {
      XIP_FREE_PROC(own_xmd,
		    "xippkt",
		    "own_xmd");
      return (status);
    }
  if ((status = vec_add(own_xmd->xp->users,own_xmd)) < 0)
    {
      XIP_FREE_PROC(own_xmd->data,
		    "xippkt",
		    "own_xmd->data");
      XIP_FREE_PROC(own_xmd,
		    "xippkt",
		    "own_xmd");
      return (status);
    }
  if ((choices = XIP_VEC_NEW(VEC_BASE,
			     &status)) == NULL)
    return (status);
  if ((status = pkt_get_choices(xmd->xp->pkt,
				VEC_AT(args,1),
				choices)) == 0)
    {
      vec_str_sort(choices);
      XmgShowPromptComplexPulldownBox(toplevel,
				      VEC_AT(args,1),
				      choices,
				      value,
				      setfield_prompt_pulldown_ok,
				      setfield_prompt_cancel,
				      own_xmd,
				      XtGrabNone);
    }
  else
    {
      XmgShowPromptBox(toplevel,
		       VEC_AT(args,1),
		       value,
		       setfield_prompt_ok,
		       setfield_prompt_cancel,
		       own_xmd,
		       XtGrabNone);
    }
  vec_str_delete(choices);
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "extract(layer[idx])"|"extract()" e.g "extract(udp[0])" 
   Extracts the specified layer to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_extract(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  char				value[STR_BUFSIZ];
  t_status			status;
  t_pkt				*pkt;
  
  if (VEC_COUNT(args) != 1 && VEC_COUNT(args) != 2)
    return (-ERR_SYNTAX);
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	{
	  return (status);
	}
    }
  else
    pkt = xmd->xp->pkt;
  if (VEC_COUNT(args) == 2)
    {
      if ((status = pkt_extract_from_name(pkt,
					  VEC_AT(args,1),
					  XIP_ALLOC_PROC,
					  XIP_FREE_PROC)) < 0)
	{
	  if (xmd->new_window)
	    xip_pkt_delete(pkt);
	  return (status);
	}
    }
  else
    {
      if ((status = pkt_extract(pkt,
				pkt->mp,
				XIP_ALLOC_PROC,
				XIP_FREE_PROC)) < 0)
	{
	  if (xmd->new_window)
	    xip_pkt_delete(pkt);
	  return (status);
	}
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  return (status);
	}
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "extracttopktbox(layer[idx])"|"extracttopktbox()"
   e.g "extracttopktbox()" 
   Extracts the specified layer to packet box.
   Currently it is called by XipActionMinimizePkt(3).
   Returns 0 if OK. Might return various errors. */
t_status			method_extracttopktbox(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  char				value[STR_BUFSIZ];
  t_status			status;
  t_pkt				*pkt;
  
  if (VEC_COUNT(args) != 1 && VEC_COUNT(args) != 2)
    return (-ERR_SYNTAX);
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	{
	  return (status);
	}
    }
  else
    pkt = xmd->xp->pkt;
  if (VEC_COUNT(args) == 2)
    {
      if ((status = pkt_extract_from_name(pkt,
					  VEC_AT(args,1),
					  XIP_ALLOC_PROC,
					  XIP_FREE_PROC)) < 0)
	{
	  if (xmd->new_window)
	    xip_pkt_delete(pkt);
	  return (status);
	}
    }
  else
    {
      if ((status = pkt_extract(pkt,
				pkt->mp,
				XIP_ALLOC_PROC,
				XIP_FREE_PROC)) < 0)
	{
	  if (xmd->new_window)
	    xip_pkt_delete(pkt);
	  return (status);
	}
    }
  if (xmd->new_window)
    {
      XipCreatePkt(pktbox,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  return (status);
	}
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "trunc(layer[idx])"|"trunc()" e.g "trunc(udp[0])" 
   Truncates the specified layer to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_trunc(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  char				value[STR_BUFSIZ];
  t_status			status;
  t_pkt				*pkt;
  
  if (VEC_COUNT(args) != 2)
    return (-ERR_SYNTAX);
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	{
	  return (status);
	}
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = pkt_trunc_from_name(pkt,
				    VEC_AT(args,1),
				    XIP_ALLOC_PROC,
				    XIP_FREE_PROC)) < 0)
    {
      if (xmd->new_window)
	xip_pkt_delete(pkt);
      return (status);
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  return (status);
	}
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "save()"
   Asks for a file name then saves the packet to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_save(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  char				*fname;

  if (fname = XmgFileGet("*.pkt",FALSE))
    {
      if ((status = pkt_save(xmd->xp->pkt,fname)) < 0)
	return (status);
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "load()"
   Asks for a file name then loads the packet to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_load(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  char				*fname;

  if (fname = XmgFileGet("*.pkt",FALSE))
    {
      t_pkt			*pkt;

      if (xmd->new_window)
	{
	  if ((pkt = xip_pkt_dup(xmd->xp->pkt,
				 &status)) == NULL)
	    return (status);
	}
      else
	pkt = xmd->xp->pkt;
      if ((status = pkt_load(pkt,
			     fname,
			     XIP_ALLOC_PROC,
			     XIP_FREE_PROC)) < 0)
	{
	  if (xmd->new_window)
	    {
	      xip_pkt_delete(pkt);
	    }
	  return (status);
	}
        if (xmd->new_window)
	  {
	    XipCreatePkt(toplevel,pkt,TRUE);
	  }
	else
	  {
	    XmgFancyReset(xmd->xp->fancy);
	    html_buf[0] = 0;
	    if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	      return (status);
	    XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
	  }
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "sum()"
   Checksums packet to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_sum(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  t_pkt				*pkt;
  
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	return (status);
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = pkt_sum(xmd->xp->pkt,NULL)) < 0)
    {
      if (xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      return (status);
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	return (status);
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "adaptlen()"
   Adapts packet len to a possible new frame.
   Returns 0 if OK. Might return various errors. */
t_status			method_adaptlen(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  t_pkt				*pkt;
  
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	return (status);
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = pkt_adapt_len(xmd->xp->pkt)) < 0)
    {
      if (xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      return (status);
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	return (status);
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "cut()"
   Cuts packet to an internal cut buffer.
   This method will one day accept one layer name parameter and will use
   an X atom to store values to allow communication between multiple xip(8).
   Returns 0 if OK. Might return various errors. */
t_status			method_cut(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  return (xip_pkt_cut(xmd->xp->pkt));
}

/* is a t_xip_method_proc.
   Usage: "paste()"
   Pastes packet from an internal cut buffer. See method_cut(3).
   Returns 0 if OK. Might return various errors. */
t_status			method_paste(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  t_pkt				*pkt;
  
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	return (status);
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = xip_pkt_paste(xmd->xp->pkt)) < 0)
    {
      if (xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      return (status);
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	{
	  return (status);
	}
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

/* is a t_xip_method_proc.
   Usage: "edit()"
   Calls an external binary editor such as emacs(1) or beav(1). It saves
   packet to a file using pkt_save_raw(3).
   Returns 0 if OK. Might return various errors. */
t_status			method_edit(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  char				fnamebuf[MAXPATHLEN];
  char				*fname;
  pid_t				pid;
  t_pkt				*pkt;

#define TMP_TEMPLATE		"/tmp/xipXXXXXX"

  strcpy(fnamebuf,TMP_TEMPLATE);
  if ((fname = mktemp(fnamebuf)) == NULL)
    return (-ERR_SUBROUTINE);
  if ((status = pkt_save_raw(xmd->xp->pkt,
			     fname)) < 0)
    return (status);
  if ((pid = fork()) < 0)
    {
      status = -ERR_SYSCALL;
      goto end;
    }
  if (pid == 0)
    {
      execlp(xt_resources.binEditorPath,
	     xt_resources.binEditorPath,
	     fname,
	     NULL);
      err_print(ERR_SYSCALL,"execl");
      exit(1);
    }
  waitpid(pid,&status,0);
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	goto end;
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = pkt_load_raw(pkt,
			     fname,
			     XIP_ALLOC_PROC,
			     XIP_FREE_PROC)) < 0)
    {
      if (xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      goto end;
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	goto end;
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  status = 0;
end:
  unlink(fname);
  return (status);
}

/* is a t_xip_method_proc.
   Usage: "reply()"
   Make an automatic reply according to protocol used by packet (see
   xip_pkt_reply(3)).
   Returns 0 if OK. Might return various errors. */
t_status			method_reply(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  t_pkt				*pkt;
  
  if (xmd->new_window)
    {
      if ((pkt = xip_pkt_dup(xmd->xp->pkt,
			     &status)) == NULL)
	return (status);
    }
  else
    pkt = xmd->xp->pkt;
  if ((status = xip_pkt_reply(pkt)) < 0)
    {
      if (xmd->new_window)
	{
	  xip_pkt_delete(pkt);
	}
      return (status);
    }
  if (xmd->new_window)
    {
      XipCreatePkt(toplevel,pkt,TRUE);
    }
  else
    {
      XmgFancyReset(xmd->xp->fancy);
      html_buf[0] = 0;
      if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
	return (status);
      XmgFancyParseBuf(xmd->xp->fancy,html_buf,strlen(html_buf));
    }
  return (0);
}

t_assoc				methods_assocs[] =
{
  {"delete",			method_delete},
  {"setfield",			method_setfield},
  {"extract",			method_extract},
  {"extracttopktbox",		method_extracttopktbox},
  {"trunc",			method_trunc},
  {"save",			method_save},
  {"load",			method_load},
  {"sum",			method_sum},
  {"adaptlen",			method_adaptlen},
  {"cut",			method_cut},
  {"paste",			method_paste},
  {"edit",			method_edit},
  {"reply",			method_reply},
  {NULL,			NULL},
};

t_vec				*methods_vec = NULL;

/* adds a method to methods vector.
   It is intended to use by plugins which adds new methods dynamically.
   Returns 0 if OK. Might return various errors */
t_status			method_add(assoc)
t_assoc				*assoc;
{
  t_status			status;
  t_xip_method			*xm;
  
  if ((xm = XIP_ALLOC_PROC(sizeof (t_xip_method),
			   "xip",
			   "method_add:ptr",
			   &status)) == NULL)
    return (status);
  xm->name = assoc->left;
  xm->proc = (t_xip_method_proc)(assoc->right);
  if ((status = vec_add(methods_vec,
			xm)) < 0)
    {
      XIP_FREE_PROC(xm,
		    "xip",
		    "*:ptr");
      return (status);
    }
  return (0);
}

/* initializes dynamical methods context.
   Returns 0 if OK. Might return various errors */
t_status			methods_init(VOID_DECL)
{
  t_status			status;
  t_assoc			*assoc;

  if ((methods_vec = XIP_VEC_NEW(VEC_ONE_BASE,
				 &status)) == NULL)
    return (status);
  assoc = methods_assocs;
  while (assoc->left)
    {
      if ((status = method_add(assoc)) < 0)
	return (status);
      assoc++;
    }
  return (0);
}

/* destroys dynamical methods context */
VOID_FUNC			methods_destroy(VOID_DECL)
{
  vec_ptr_delete(methods_vec);
}

/* applies a method.
   If method fails, it shows an error popup. */
VOID_FUNC			method_do(xmd,args)
t_xip_method_data		*xmd;
t_vec				*args;
{
  t_status			status;
  char				buf[STR_BUFSIZ];

  status = -ERR_NOMETHOD;
  VEC_FOR(methods_vec,t_xip_method *xm)
    {
      if (!strcmp(xm->name,VEC_AT(args,0)))
	{
	  if ((status = xm->proc(xmd,args)) < 0)
	    {
	      break ;
	    }
	  return ;
	}
    }
  VEC_ENDFOR;
  buf[0] = 0;
  VEC_FOR_FROM(args,1,char *str)
    {
      t_status			lstatus;

      if ((lstatus = str_cat_str(buf,
				 sizeof (buf),
				 str)) < 0)
	break ;
      if (VEC_IDX != (VEC_COUNT(args) - 1))
	if ((lstatus = str_cat_char(buf,
				    sizeof (buf),
				    ',')) < 0)
	  break ;
    }
  VEC_ENDFOR;
  XmgErrPrint(-status,"%s(%s)",VEC_AT(args,0),buf);
  return ;
}

/* parses a method.
   Might show an error popup if there is a syntax error. If OK,
   it calls method_do(3). */
VOID_FUNC			parse_method(xmd,str)
t_xip_method_data		*xmd;
char				*str;
{
  char				safe_str[BUFSIZ];
  t_status			status;
  char				*ptr;
  char				*arg;
  t_vec				*args;
  t_boolean			found_lp;

  safe_str[0] = 0;
  if ((status = str_cat_str(safe_str,
			    sizeof (safe_str),
			    str)) < 0)
    {
      XmgErrPrint(-status,"str_cat_str");
      return ;
    }
  if ((ptr = index(safe_str,'(')) == NULL)
    {
      XmgErrPrint(ERR_SYNTAX,"missing '(' in %s",str);
      return ;
    }
  if ((args = XIP_VEC_NEW(VEC_BASE,
			  &status)) == NULL)
    {
      XmgErrPrint(-status,"XIP_VEC_NEW");
      return ;
    }
  *ptr++ = 0;
  if ((status = vec_str_add(args,safe_str)) < 0)
    {
      XmgErrPrint(-status,"vec_str_add");
      vec_str_delete(args);
      return ;
    }
  found_lp = FALSE;
  arg = ptr;
  while (*ptr)
    {
      if (isspace(*ptr))
	{
	  ptr++;
	  continue ;
	}
      if ((*ptr) == ',' || (*ptr) == ')')
	{
	  if ((*ptr) == ')')
	    found_lp = TRUE;
	  *ptr++ = 0;
	  if (strcmp(arg,""))
	    if ((status = vec_str_add(args,arg)) < 0)
	      {
		XmgErrPrint(-status,"vec_str_add");
		vec_str_delete(args);
		return ;
	      }
	  arg = ptr;
	}
      if (found_lp)
	break ;
      ptr++;
    }
  if (found_lp)
    method_do(xmd,args);
  else
    XmgErrPrint(ERR_SYNTAX,"missing ')' in %s",str);
  vec_str_delete(args);
}

/* is an XtCallbackProc.
   It parses method pointed by link. */
VOID_FUNC			XipPktFancyLink(w,xp,cbs)
Widget				w;
t_xip_pkt			*xp;
XmgFancyLinkCallbackStruct	*cbs;
{
  t_xip_method_data		xmd;

  xmd.xp = xp;
  xmd.new_window = cbs->new_window;
  parse_method(&xmd,cbs->href);
}

/* is an XtCallbackProc.
   It sets shell title if packet is "big" */
VOID_FUNC			XipPktFancyTitle(w,xp,cbs)
Widget				w;
t_xip_pkt			*xp;
XmgFancyTitleCallbackStruct	*cbs;
{
  if (xp->shell)
    {
      XtVaSetValues(xp->shell,
		    XtNtitle,	cbs->title,
		    XtNiconName,cbs->title,
		    NULL);
    }
}

/* is an XtCallbackProc.
   Destroys packet in using xp_delete(3) */
VOID_FUNC			XipPktDestroy(w,xp,cbs)
Widget				w;
t_xip_pkt			*xp;
VOID_PTR			cbs;
{
  t_status			status;
  
  if ((status = xp_delete(xp)) < 0)
    {
      XmgErrPrint(-status,"xp_delete");
    }
}

char				*top_tmpl = "\
<html>\n\
<head>\n\
<title>%Layer% %Date%</title>\n\
</head>\n\
<body link=black bgcolor=%bodyColorName% background=%bodyBackgroundName%>\n\
<table width=100%% bgcolor=%controlsColorName%>\n\
<tr>\n\
<td align=center>\n\
<b><a href=\"setfield(Layer)\">%Layer%</a></b>\n\
</td>\n\
<td align=center>\n\
<b>%Date%</b> (%date%)\n\
</td>\n\
<td align=center>\n\
%Len% (was %Netlen%)\n\
</td>\n\
</tr>\n\
<tr>\n\
<td>\n\
<small>\n\
";

char				*afterpktbar_tmpl = "\
</small>\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_assoc				pktbar_assocs[] = 
{
  {"Delete",			"delete()"},
  {"Load",			"load()"},
  {"Save",			"save()"},
  {"Ad.len",			"adaptlen()"},
  {"Sum",			"sum()"},
  {"Cut",			"cut()"},
  {"Paste",			"paste()"},
  {"Edit",			"edit()"},
  {"Reply",			"reply()"},
  {NULL,			NULL},
};

t_vec				*pktbar_vec = NULL;

/* adds a title and a method to packet bar.
   It is intended to use by plugins which adds new methods dynamically.
   Returns 0 if OK. Might return various errors */
t_status			pktbar_add(assoc)
t_assoc				*assoc;
{
  t_status			status;
  t_xip_pktbar			*xpb;
  
  if ((xpb = XIP_ALLOC_PROC(sizeof (t_xip_pktbar),
			   "xip",
			   "pktbar_add:ptr",
			   &status)) == NULL)
    return (status);
  xpb->label = assoc->left;
  xpb->methodstr = assoc->right;
  if ((status = vec_add(pktbar_vec,
			xpb)) < 0)
    {
      XIP_FREE_PROC(xpb,
		    "xip",
		    "*:ptr");
      return (status);
    }
  return (0);
}

/* initializes dynamical packet bar context.
   Returns 0 if OK. Might return various errors */
t_status			pktbar_init(VOID_DECL)
{
  t_status			status;
  t_assoc			*assoc;

  if ((pktbar_vec = XIP_VEC_NEW(VEC_ONE_BASE,
				 &status)) == NULL)
    return (status);
  assoc = pktbar_assocs;
  while (assoc->left)
    {
      if ((status = pktbar_add(assoc)) < 0)
	return (status);
      assoc++;
    }
  return (0);
}

/* destroys dynamical packet bar context. */
VOID_FUNC			pktbar_destroy(VOID_DECL)
{
  vec_ptr_delete(pktbar_vec);
}

char				*bottom_tmpl = "\n\
</body>\n\
</html>\n\
";

/* generate an html page from a packet.
   It accepts -ERR_TRUNC error message from subroutines.
   Returns 0 if OK. Might return various errors. */
t_status			xip_pkt_html(pkt,str,max_len)
t_pkt				*pkt;
char				*str;
int				max_len;
{
  t_status			status;

  tmpl_buf[0] = 0;
  if ((status = str_cat_str(tmpl_buf,
			    tmpl_bufsiz,
			    top_tmpl)) < 0)
    {
      return (status);
    }
  VEC_FOR(pktbar_vec,t_xip_pktbar *xpb)
    {
      if ((status = str_cat_fmt_va(tmpl_buf,
				   tmpl_bufsiz,
				   "<a href=\"%s\">[%s]</a>",
				   xpb->methodstr,
				   xpb->label)) < 0)
	return (status);
    }
  VEC_ENDFOR;
  if ((status = str_cat_str(tmpl_buf,
			    tmpl_bufsiz,
			    afterpktbar_tmpl)) < 0)
    {
      return (status);
    }
  if ((status = pkt_get_tmpl_to_str(pkt,
				    tmpl_buf,
				    tmpl_bufsiz)) < 0)
    {
      if (status != -ERR_TRUNC)
	XmgErrPrint(-status,"pkt_get_tmpl_to_str");
    }
  if ((status = str_cat_str(tmpl_buf,
			    tmpl_bufsiz,
			    bottom_tmpl)) < 0)
    {
      return (status);
    }
  if ((status = pkt_format_html_permissive(pkt,
					   tmpl_buf,
					   NULL,
					   extravars,
					   str,
					   max_len,
					   htmlize_buf,
					   htmlize_bufsiz)) < 0)
    {
      return (status);
    }
  return (0);
}

/* creates a graphical packet.
   If parent is toplevel then it creates a shell widget (it becomes
   then a "big" packet). If parent is not toplevel, it becomes 
   a "small" packet.
   It also defines specific action for "big" and "small" packets. 
   It might popup various error messages. */
VOID_FUNC			XipCreatePkt(parent,pkt,own_pkt)
Widget				parent;
t_pkt				*pkt;
t_boolean			own_pkt;	/* If TRUE, considers
						   that t_pkt is
						   deletable */
{
  t_status			status;
  t_xip_pkt			*xp;
  Atom				atom;

  html_buf[0] = 0;
  if ((status = xip_pkt_html(pkt,html_buf,html_bufsiz)) < 0)
    {
      if (parent == toplevel)
	XmgErrPrint(-status,"xip_pkt_html");
      return ;
    }
  if ((xp = XIP_ALLOC_PROC(sizeof (t_xip_pkt),
			   "xippkt",
			   "xp",
			   &status)) == NULL)
    {
      XmgErrPrint(-status,"XIP_ALLOC_PROC");
      return ;
    }
  if ((xp->users = XIP_VEC_NEW(VEC_BASE,
			       &status)) == NULL)
    {
      XmgErrPrint(-status,"XIP_VEC_NEW");
      XIP_FREE_PROC(xp,
		    "xippkt",
		    "xp");
      return ;
    }
  xp->pkt = pkt;
  xp->own_pkt = own_pkt;
  xp->in_delete = FALSE;
  if (parent == toplevel)
    {
      INTERFACE(toplevel,
		GETCHILDSHELL(xp->shell)
		("pktShell",wmShellWidgetClass,
		 XtNmappedWhenManaged,	False,
		 XtNallowShellResize,	True,
		 XtNiconPixmap,		xipicon_pixmap,
		 GETCHILD(xp->fancy)
		 ("pktFancy",xmgFancyWidgetClass,
		  XtNdoCutAndPaste,	True,
		  XtNuserData,		xp,
		  CALLBACK(XtNlinkCallback,
			   (XtCallbackProc)XipPktFancyLink,
			   xp),
		  CALLBACK(XtNtitleCallback,
			   (XtCallbackProc)XipPktFancyTitle,
			   xp),
		  CALLBACK(XtNimgCallback,
			   (XtCallbackProc)XipFancyImg,
			   xp),
		  CALLBACK(XtNdestroyCallback,
			   (XtCallbackProc)XipPktDestroy,
			   xp),
		  END),
		 MANAGE,
		 END),
		ENDINTERFACE);
    }
  else
    {
      xp->shell = NULL;
      INTERFACE(parent,
		GETCHILD(xp->fancy)
		 ("smallPktFancy",xmgFancyWidgetClass,
		  XtNuserData,		xp,
		  CALLBACK(XtNimgCallback,
			   (XtCallbackProc)XipFancyImg,
			   xp),
		  CALLBACK(XtNdestroyCallback,
			   (XtCallbackProc)XipPktDestroy,
			   xp),
		  END),
		ENDINTERFACE);
    }
  XmgFancyParseBuf(xp->fancy,html_buf,strlen(html_buf));
  if (xp->shell)
    {
      XmgSetDestroyOnDelete(xp->shell);
      XtOverrideTranslations(xp->fancy,
	       XtParseTranslationTable("<Btn3Down>: XipMinimizePkt()"));
      XtPopup(xp->shell,XtGrabNone);
    }
  else
    {
      XtOverrideTranslations(xp->fancy,
	       XtParseTranslationTable("<Btn3Down>: XipMaximizePkt()\n\
<EnterWindow>: XipPopupPktHelp()\n\
<LeaveWindow>: XipPopdownPktHelp()"));
    }
}
