/* image_tcl.c: TCL interface to image routines.
**
** written and copyright (C) 1996 by Michael J. Gourlay
**
** PROVIDED AS IS.  NO WARRENTIES, EXPRESS OR IMPLIED.
*/

#include <stdio.h>
#include <stdlib.h>

#include <tcl.h>

#include "my_malloc.h"
#include "rgba_image.h"

#include "hash.h"
#include "image_tcl.h"




#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE (!FALSE)
#endif




/* Remember that my goal here is to store images that are
** immediately useful for using the digital image warping algorithms,
** which means that I will not necessarily be using the smallest
** amount of memory possible to store these images.
** In fact, I'll always store these images as 4-channel (RGBA),
** 32-bits per pixel images.
*/




/* img_table: hash table of images for mapping an image tag to an
** image data structure.
*/
Tcl_HashTable img_table;


/* img_table_initF: flag which indicates whether the hash table has
** been initialized yet.
*/
int img_table_initF = FALSE;




/* createImageTag: create a unique "tag" (i.e. TCL name) for the RGBA image
** imgP: (in) pointer to the RGBA image
** tag: (out) tag of the RGBA image
*/
void
createImageTag(RgbaImageT *imgP, char *tag)
{
  createTag(imgP, tag, "img", &img_table_initF, &img_table);
}




/* imageLoadCmd: TCL command to load an RGBA image.
**
** This code is somewhat redundant with the Tk "photo" image type.
** But I need alpha channels, and I want the data stored
** with non-interleaved channels.
*/
int
imageLoadCmd(ClientData client_data, Tcl_Interp *interp, int argc, char *argv[])
{
  char *filename;
  RgbaImageT *imgP;

  /* ---- Allocate and Initialize image */
  if((imgP = MY_CALLOC(1, RgbaImageT, "imageLoadCmd")) == NULL) {
    return TCL_ERROR;
  }
  rgbaImageInit(imgP);


  /* ---- Parse the arguments to get the file name. */
  if(argc != 2) {
    interp->result = "Usage: imageLoad ?filename?";
    return TCL_ERROR;
  }

  filename = argv[1];

  /* ---- Load the image using rgbaImageLoad. */
  if(rgbaImageLoad(filename, imgP)) {
    sprintf(interp->result, "imageLoad: could not load file %s\n", filename);
    return TCL_ERROR;
  }

  /* ---- Create a new image tag for this image. */
  /* ---- Create a new hash table entry. */
  /* ---- Register the image in the hash table. */
  createImageTag(imgP, interp->result);

  return TCL_OK;
}




void
rgbaToPhoto(RgbaImageT *rgbaP, Tk_PhotoHandle photoH)
{
  /* Give the image block to Tk */
  Tk_PhotoImageBlock block;

  block.pixelPtr = rgbaP->ri;
  block.width = rgbaP->ncols;
  block.height = rgbaP->nrows;
  block.pitch = rgbaP->ncols;
  block.pixelSize = sizeof(unsigned char);
  block.offset[0] = 0;
  block.offset[1] = rgbaP->ncols * rgbaP->nrows;
  block.offset[2] = 2 * block.offset[1];

  /* Set the photo image size */
  Tk_PhotoSetSize(photoH, block.width, block.height);

  Tk_PhotoPutBlock(photoH, &block, 0, 0, block.width, block.height);
}




/* imageConvertCmd: TCL/Tk command to convert an image between RGBA and Photo
*/
int imageConvertCmd(ClientData client_data, Tcl_Interp *interp,
                    int argc, char *argv[])
{
  char *photoImageTag;
  char *rgbaImageTag;

  if(argc != 3) {
    interp->result = "Usage: imageConvert ?photo? ?rgba?";
    return TCL_ERROR;
  }

  photoImageTag = argv[1];
  rgbaImageTag = argv[2];

  {
    Tk_PhotoHandle photoH;
    RgbaImageT *rgbaP;

    /* Find the handle for the photo image */
    photoH = Tk_FindPhoto(photoImageTag);

    HASH_FIND(img_table, rgbaImageTag, rgbaP, RgbaImageT, "imageConvertCmd");

    rgbaToPhoto(rgbaP, photoH);
  }

  return TCL_OK;
}
