/* -------------------------------- body.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handle body descriptions.
*/

#include "fly.h"


extern BODY	BoGround, BoBox, BoPlane, BoRunway, BoM61, BoTarget,
		BoBroken, BoViewer, BoChute, BoHouse, BoTower, BoLow,
		BoGtarget, BoMK82, BoCrater, BoSmoke;

static int NEAR
init_internals (void)
{
	st.bodies[O_GROUND]  = &BoGround;
	st.bodies[O_BOX]     = &BoBox;
	st.bodies[O_PLANE]   = &BoPlane;
	st.bodies[O_RUNWAY]  = &BoRunway;
	st.bodies[O_M61]     = &BoM61;
	st.bodies[O_TARGET]  = &BoTarget;
	st.bodies[O_BROKEN]  = &BoBroken;
	st.bodies[O_VIEWER]  = &BoViewer;
	st.bodies[O_CHUTE]   = &BoChute;
	st.bodies[O_HOUSE]   = &BoHouse;
	st.bodies[O_TOWER]   = &BoTower;
	st.bodies[O_LOW]     = &BoLow;
	st.bodies[O_GTARGET] = &BoGtarget;
	st.bodies[O_MK82]    = &BoMK82;
	st.bodies[O_CRATER]  = &BoCrater;
	st.bodies[O_SMOKE]   = &BoSmoke;

	return (0);
}

extern BODY * FAR
bodies_new (ONAME n)
{
	BODY	*b;

	if (n >= 0) {
		if (n < 0 || n >= O_INT+O_EXT || st.bodies[n])
			return (0);
	} else {
		for (n = 0; st.bodies[n]; ++n)
			if (n >= O_INT+O_EXT)
				return (0);
	}

	if (!NEW (b))
		return (0);

	st.bodies[n] = b;
	b->name = n;

	return (b);
}

extern void FAR
bodies_extent (ONAME name)
{
	Uint	extent;
	VERTEX	*v;

	extent = 0;
	for (v = st.bodies[name]->shape->v; v->flags; ++v) {
		if (extent < iabs(v->V[X]))
			extent = iabs (v->V[X]);
		if (extent < iabs(v->V[Y]))
			extent = iabs (v->V[Y]);
		if (extent < iabs(v->V[Z]))
			extent = iabs (v->V[Z]);
	}
	if (st.bodies[name]->shape->flags & SH_FINE)
		extent /= VONE;
	if (0 == extent)
		extent = 1;
	st.bodies[name]->shape->extent = (Ushort)extent;
}

extern int FAR
bodies_init (void)
{
	ONAME	i;

	if (!(st.bodies = (BODY **)xcalloc (O_INT+O_EXT+1, sizeof (BODY *))))
		return (1);

	if (init_internals ())
		return (1);

	for (i = 0; st.bodies[i]; ++i) {
		st.bodies[i]->name = i;
		if (st.bodies[i]->init && st.bodies[i]->init(st.bodies[i]))
			return (1);
		bodies_extent (i);
	}
	return (0);
}

extern BODY * FAR
bodies_del (ONAME name)
{
	if (name < 0 || name >= O_INT+O_EXT || !st.bodies || !st.bodies[name])
		return (0);

	if (st.bodies[name]->term)
		st.bodies[name]->term(st.bodies[name]);

	if (name >= O_INT)
		DEL0 (st.bodies[name]);
	else
		st.bodies[name] = 0;
	return (0);
}

extern void FAR
bodies_term (void)
{
	ONAME	name;

	if (!st.bodies)
		return;

	for (name = 0; name < O_INT+O_EXT; ++name) {
		if (!st.bodies[name])
			continue;
		bodies_del (name);
	}

	st.bodies = xfree (st.bodies);
}

extern ONAME FAR
body_name (char *title)
{
	int	i;

	if (!st.bodies)
		return (-1);

	for (i = 0; st.bodies[i]; ++i) {
		if (!strcmp (st.bodies[i]->title, title))
			return (i);
	}

	return (-1);
}
