/* --------------------------------- udp.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for server based UDP level networking (low level).
 * An address here is IP address (4 bytes) + UDP port num (2 bytes).
 *
 * It needs the server fly8srv for operation, it will NOT talk directly
 * to another fly8.
*/

#include "fly.h"

#ifdef HAVE_UDP

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>


/* The following definitions are shared with the server programs fly8srv.c
 * and udpcli.c.
*/
#define SOCK_FLY8	SOCK_DGRAM
#define IPPORT_FLY8	0xf8f8

#define PHDATA		(pack->data+PACKHEADLEN)
#define PHLEN		(PHDATA-2)
#define PHFROM		(PHLEN-LADDRESS)
#define PHTO		(PHFROM-LADDRESS)
#define PHEAD		PHTO
#define PHSIZE		(PHDATA-PHEAD)

typedef struct port PORT;
struct port {
	int	flags;
#define POF_ON		0x0001
	int		fd;
	struct sockaddr_in	svr, cli;
	int		netport;		/* back-pointer */
	char		*server;		/* server name */
	Uchar		address[LADDRESS];	/* my address */
};

static PORT	ports[] = {
	{0, -1},
	{0, -1},
	{0, -1},
	{0, -1},
};
#define	NDEV	rangeof(ports)

static int	nports = 0;		/* number of active ports */

static int NEAR
get_options (PORT *port, char *options)
{
	char	*p, *q;
	int	i;
	long	l;

	if (T(p = get_iarg (options, 0))) {
		q = strchr (p, ':');
		if (q) {
			i = q - p;
			if (F(q = xmalloc (i)))
				return (1);
			memcpy (q, p, i);
			q[i] = '\0';
		} else
			q = strdup (p);
		port->server = q;
	} else
		return (1);

	if (get_narg (options, "port=", &l))
		l = IPPORT_FLY8;
	port->svr.sin_port = htons ((Ushort)l);

	return (0);
}

static int FAR
udp_init (NETPORT *np, char *options)
{
	int		portno;
	PORT		*port;
	char		*protoname;
	struct protoent	*proto;
	struct hostent	*hostptr;
	Ulong		srv_addr;		/* server address */
	int		n;

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV) {
		MsgEPrintf (-100, "%s.%c: bad port",
			np->NetDriver->name, np->unit);
		return (1);
	}
	port = &ports[portno];
	if (port->flags & POF_ON) {
		MsgEPrintf (-100, "%s.%c: already on",
			np->NetDriver->name, np->unit);
		return (1);
	}

	memset (&port->svr, 0, sizeof (port->svr));
	memset (&port->cli, 0, sizeof (port->cli));

	if (get_options (port, options))
		return (1);

	protoname = "udp";
	if ((proto = getprotobyname (protoname)) == NULL) {
		MsgEPrintf (-100, "%s.%c: getprotobyname(%s) failed: %s",
			np->NetDriver->name, np->unit,
			protoname, strerror (errno));
		return (1);
	}

	if ((port->fd = socket (AF_INET, SOCK_FLY8, proto->p_proto)) < 0) {
		MsgEPrintf (-100, "%s.%c: socket() failed: %s",
			np->NetDriver->name, np->unit, strerror (errno));
		return (1);
	}

	n = fcntl (port->fd, F_GETFL);
	if (fcntl (port->fd, F_SETFL, n|FLY8_NONBLOCK) < 0) {
		MsgEPrintf (-100, "%s.%c: fcntl() failed: %s",
			np->NetDriver->name, np->unit, strerror (errno));
		return (1);
	}

/* Set up server (our output) address.
*/
	if ((hostptr = gethostbyname (port->server)) == NULL) {
		MsgEPrintf (-100, "%s.%c: gethostbyname(%s) failed: %s",
			np->NetDriver->name, np->unit,
			port->server, strerror (errno));
		return (1);
	}

	if (hostptr->h_addrtype != AF_INET) {
		MsgEPrintf (-100, "%s.%c: not AF_INET address",
			np->NetDriver->name, np->unit);
		return (1);
	}

	srv_addr = ((struct in_addr *)hostptr->h_addr_list[0])->s_addr;
	MsgPrintf (-100, "%s.%c: server is %08lx",
			np->NetDriver->name, np->unit, ntohl (srv_addr));

	port->svr.sin_family      = AF_INET;
	port->svr.sin_addr.s_addr = srv_addr;

/* Set up client (our input) address.
*/
	port->cli.sin_family      = AF_INET;
	port->cli.sin_addr.s_addr = htonl (INADDR_ANY);
	port->cli.sin_port        = htons (0);

	if (bind (port->fd, (struct sockaddr *) &port->cli,
						sizeof (port->cli)) < 0) {
		MsgEPrintf (-100, "%s.%c: bind() failed: %s",
			np->NetDriver->name, np->unit, 	strerror (errno));
		return (1);
	}
	memcpy (port->address,   (char *)&port->cli.sin_addr.s_addr, 4);
	memcpy (port->address+4, (char *)&port->cli.sin_port,        2);

	port->flags |= POF_ON;
	port->netport = np->netport;
	++nports;

	return (0);
}

static void FAR
udp_term (NETPORT *np)
{
	int	portno;
	PORT	*port;

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return;
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return;

	port->server = xfree (port->server);
	if (port->fd >= 0) {
		close (port->fd);
		port->fd = -1;
	}
	port->flags = 0;
	--nports;
}

static int FAR
udp_send (NETPORT *np, PACKET *pack)
{
	int	portno;
	PORT	*port;
	int	n;

	if (0 == pack)
		return (0);

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return (1);
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return (1);

	if (pack->address) {
		memcpy (PHTO, pack->address, LADDRESS);	/* to */
		pack->address = 0;			/* private */
	} else
		memset (PHTO, 0xff, LADDRESS);		/* broadcast */
	memcpy (PHFROM, port->address, LADDRESS);	/* from */
	PHLEN[0] = (Uchar)(0x0ff&(pack->length >> 8));	/* length */
	PHLEN[1] = (Uchar)(0x0ff&(pack->length));

	n = PHSIZE + pack->length;
	if (n != sendto (port->fd, PHEAD, n, 0, (struct sockaddr *)&port->svr,
							sizeof (port->svr))) {
		if (EWOULDBLOCK == errno)
			return (1);	/* busy */
		else
			return (1);	/* error */
	}

	return (0);
}

static int FAR
udp_receive (NETPORT *np)
{
	int	portno;
	PORT	*port;
	PACKET	*pack;
	int	n, len, limit;
	char	msg[512];	/* message buffer */

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return (1);
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return (1);

	for (limit = 256; limit-- > 0;) {
		len = sizeof (port->cli);
		n = recvfrom (port->fd, msg, sizeof (msg), 0,
				(struct sockaddr *)&port->cli, (int *)len);
		if (n < 0) {
			if (EWOULDBLOCK != errno) {
				MsgEPrintf (-100, 
					"%s.%c: recvfrom() failed: %s",
					np->NetDriver->name, np->unit,
					strerror (errno));
				return (1);
			}
		} else {
			len  = PHLEN-PHEAD;
			len  = ((msg+len)[0] << 8) + (msg+len)[1];
			if (len < 3 || len+PHSIZE != n) {
				++st.stats[5];
				continue;
			}
			if (F(pack = packet_new (len))) {
				++st.stats[5];
				continue;
			}
			memcpy (PHEAD, msg, n);
			pack->netport = port->netport;
			pack->length = len;
			pack->address = PHFROM;		/* from */

			if (packet_deliver (pack))
				packet_del (pack);
			continue;
		}
		break;
	}

	return (0);
}

struct NetDriver NEAR NetUdp = {
	"UDP",
	0,
	udp_init,
	udp_term,
	udp_send,
	udp_receive
};
#endif /* ifdef HAVE_UDP */
